# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Helper file for performing File System operations

ScanType , IndexEnabled, CommitCondition and FSHelper are 4 classes defined in this file.

ScanType: Enum constants for all the scan types currently supported for file system subclients

IndexEnabled: With Blocklevel we check Meta Data Collection on or not.

CommitCondition: Enum constants for all supported commit conditions.

FSHelper: Helper class to perform file system operations

FSHelper:
    __init__()                        --  initializes file system helper object

    create_backupset()                --  creates new backupset

    create_instance()                 --  creates new instance for filesystem big data apps

    create_subclient()                --  creates new subclient

    update_subclient()                --  updates properties of existing subclient

    run_backup()                      --  Run backup on the testcase subclient

    find_items_subclient()            --  Find items by browse
    on the testcase subclient

    compare_lists()                   --  Compare source and destination
    list and find differennces

    restore_in_place()                --  Run in place restore

    restore_out_of_place()            --  Run out of place restore

    run_backup_verify()               --  Initiates backup and
    performs the applicable verifications

    run_restore_verify()              --  Initiates restore for data
    backed up in the given job and performs the applicable verifications

    run_find_verify()                 --  Finds items backed up in the given job
    and compares it with items from machine

    add_new_data_incr()               --  Adds new data for incremental
    with provisions for modification after this function execution,
    call mod_data_incr for modifications

    mod_data_incr()                   --  Modifies the data added by add_new_data_incr
    call add_new_data_incr before calling this.

    check_pseudo_client()             --  Checks for the existence of pseudo-client on CS
    for Usercentric laptop clients

    check_scheduled_backup()          --  Checks for the status of the scheduled backup job of
    pseudo-client for Usercentric laptop clients

    pseudo_client_cleanup()           --  Removes pseudo-client from CS for Usercentric
    laptop clients

    get_subclient_content()           --  Generates the subclient content list

    volume_level_restore()            --  To perform blocklevel volume level restore on client

    backup_copy()                     --  Run backup copy from Subclient level

    get_backup_stream_count()         --  Returns the number of streams used by a backup job

    get_restore_stream_count()        --  Returns the number of streams used by a restore job

    verify_dc()                       --  Verifies whether the job used Scan optimization

    validate_trueup()                 --  Checks if trueup ran or not;
    based on conditional variable it also checks
    if trueup picked up files for backup.

    run_systemstate_backup()          --  Checks for the system state property and runs a backup

    run_onetouch_backup()             --  Checking if onetouch option is enabled and running a backup

    commit_job()                      --  Commits the backup job if the threshold condition is met or exceeded.

    parse_restart_string()            --  Parses a backup job restart string and
    prepares a dictionary of collect file ID to offset value.

    create_expected_commit_files()    --  Prepares the expected committed files for the committed backup job.

    get_actual_commit_files()         --  Retrieves the actual commit files that were generated by scan.

    validate_backup()                 -- Function checks whether the files are backedup or not.

    validate_download_files()         -- Function checks whether file/folder downloaded with accurate data

    validate_preview_file()           -- Function verifies the preview of file is successful or not.

    validate_cross_machine_restore()   -- Function checks files are restored properly or not.

    validate_multiple_versions_of_file()  -- Function whether old version of file is correctly restored or not

    validate_restore_for_selected_files() -- Function checks selected/deleted/hidden files are restored or not.

    validate_collect_files()          -- Function checks backup files with collect files

    validate_commit_files()           --  Function checks whether the commit collect file and
    commit dirchange file were created correctly.

    _update_ref_time_dict()           --  Updates the reference time dictionary self.job_ref_time_dict.

    extent_level_validation()         --  Function attempts to check whether large files were backed up as extents.
    It also, optionally, validates the extent eligible collect file.

    generate_testdata()                -- To generate the test data

    validate_filters()                 -- To validate the filters set at client group level

    validate_failed_files()             -- To validate the failed files from collects

    is_snapshot_deleted()               -- To check if snapshot has been deleted

    get_snapshot_id()                   -- To get the snapshot ID created for that job

    validate_pre_post_commands()        --  To validate the pre post commands are set correctly

    validate_rfc_files()                -- To validate Remote File cache Files

    get_index_cache()                   -- To get index Cache Path on Indexing MA.

    get_cluster_nodes()                 -- To get list of cluster nodes

    active_cluster_node()               -- To get active cluster node

    do_failover()                       -- To do cluster failover

    identify_master()                   --  Identifies the master node in multi-node backup

    verify_scan_type_used()             -- Returns Scan type for a given job


FSHelper Instance Attributes:
=============================

    **subclient_job_results_directory**            --  Returns the subclient job  results directory path.

"""
import os
import itertools
import re
from base64 import b64encode
from math import floor
from copy import deepcopy
from datetime import datetime
from difflib import Differ
from enum import Enum
from time import sleep, time
from zipfile import ZipFile
from past.builtins import basestring
from AutomationUtils import logger
from AutomationUtils.constants import DistributedClusterType
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.windows_machine import WindowsMachine
from AutomationUtils.unix_machine import UnixMachine
from AutomationUtils.database_helper import get_csdb


class ScanType(Enum):
    """Constants for all the scan types
        currently supported for file system subclients"""
    RECURSIVE = 1
    OPTIMIZED = 2
    CHANGEJOURNAL = 3


class IndexEnabled(Enum):
    """With Blocklevel we check Meta Data Collection on or not"""

    NOINDEX = 1
    INDEX = 2


class CommitCondition(Enum):
    """Constants for all supported commit conditions"""
    FILES = 1
    SECONDS = 2


class SoftwareCompression(Enum):
    """Constants for all the supported software compression values."""

    ON_CLIENT = 1
    ON_MEDIA_AGENT = 2
    USE_STORAGE_POLICY_SETTINGS = 3
    OFF = 4


class FSHelper(object):
    """Helper class to perform file system operations"""

    def __init__(self, testcase):
        """Initialize instance of the FSHelper class."""
        self.testcase = testcase
        self.log = logger.get_log()
        self._csdb = get_csdb()
        self.pseudo_client_name = None
        self._job_ref_time_dict = dict()
        self.indexpath = None
        self.backupset_guid = None
        self.testcase.nfs_client_mount_dir = None

    @property
    def job_ref_time_dict(self):
        """Returns a dictionary of jobs and their reference times."""
        return self._job_ref_time_dict

    @job_ref_time_dict.setter
    def job_ref_time_dict(self, value):
        """Updates the job to reference time dictionary."""
        self._job_ref_time_dict.update(value)

    @staticmethod
    def get_storage_policy(commcell_object):
        """Returns the name of first online storage policy in the given commcell

            Args:
                commcell_object     (object)  --  instance of the Commcell class

            Returns:
                str - name of the selected storage policy

            Raises:
                Exception:
                    if there is no online storage policy in the given commcell
         """
        storage_polices_obj = commcell_object.storage_policies
        disk_libraries_obj = commcell_object.disk_libraries
        selected_storage_policy = None
        for storage_policy in storage_polices_obj.all_storage_policies:
            if str(storage_policy).lower() == "commservedr":
                continue
            library = disk_libraries_obj.get(storage_polices_obj.get(str(storage_policy)).library_name)
            mag_lib_summary = library.library_properties.get('magLibSummary', None)
            if mag_lib_summary is not None:
                lib_online_status = mag_lib_summary.get('isOnline', None)
                if (lib_online_status is not None) and (lib_online_status == 'Ready'):
                    selected_storage_policy = str(storage_policy)
                    break
        if selected_storage_policy is None:
            raise Exception("No online storage policies found")
        return selected_storage_policy

    @staticmethod
    def populate_tc_inputs(cv_testcase, mandatory=True):
        """Initializes all the test case inputs after validation

        Args:
            cv_testcase    (obj)    --    Object of CVTestCase
            mandatory      (bool)   --    whether to check for mandatory inputs
            and raise exception if not found

        Returns:
            None

        Raises:
            Exception:
                if a valid CVTestCase object is not passed.
                if CVTestCase object doesn't have agent initialized

        """
        if not isinstance(cv_testcase, CVTestCase):
            raise Exception(
                "Valid test case object must be passed as argument"
            )
        cv_testcase.client_name = cv_testcase.tcinputs.get('ClientName', None)
        cv_testcase.test_path = cv_testcase.tcinputs.get('TestPath', None)
        cv_testcase.test_path2 = cv_testcase.tcinputs.get('TestPath2', None)
        cv_testcase.test_path3 = cv_testcase.tcinputs.get('TestPath3', None)
        cv_testcase.test_path_list = cv_testcase.tcinputs.get("TestPathList", None)
        cv_testcase.restore_path = cv_testcase.tcinputs.get('RestorePath', None)
        cv_testcase.storage_policy = cv_testcase.tcinputs.get('StoragePolicyName', None)
        cv_testcase.snap_engine = cv_testcase.tcinputs.get('snapengine', None)
        cv_testcase.storage_policy = cv_testcase.tcinputs.get('StoragePolicyName', None)
        cv_testcase.unicode = cv_testcase.tcinputs.get('Unicode', False)
        cv_testcase.acls = cv_testcase.tcinputs.get('ACLS', False)
        cv_testcase.xattr = cv_testcase.tcinputs.get('XATTR', False)
        cv_testcase.hlinks = cv_testcase.tcinputs.get('hlinks', True)
        cv_testcase.slinks = cv_testcase.tcinputs.get('slinks', True)
        cv_testcase.sparse = cv_testcase.tcinputs.get('sparse', True)
        cv_testcase.file_rename = cv_testcase.tcinputs.get('FileRename', True)
        cv_testcase.find_verify = cv_testcase.tcinputs.get('FindVerify', True)
        cv_testcase.long_path = cv_testcase.tcinputs.get('LongPath', False)
        cv_testcase.verify_dc = cv_testcase.tcinputs.get('VerifyDC', False)
        cv_testcase.problematic = cv_testcase.tcinputs.get('Problematic', False)
        cv_testcase.dirtime = cv_testcase.tcinputs.get('FolderTimeStamp', False)
        cv_testcase.skiplink = cv_testcase.tcinputs.get('SkipLink', False)
        cv_testcase.skip_classic = cv_testcase.tcinputs.get('SkipClassic', False)
        cv_testcase.hostname = cv_testcase.tcinputs.get('HostName', None)
        cv_testcase.username = cv_testcase.tcinputs.get('UserName', None)
        cv_testcase.password = cv_testcase.tcinputs.get('Password', None)
        cv_testcase.proxy_client = cv_testcase.tcinputs.get('ProxyClient', None)
        cv_testcase.iscsi_server = cv_testcase.tcinputs.get('IscsiServer', None)
        cv_testcase.client_machine = None
        cv_testcase.index_ma = cv_testcase.tcinputs.get('IndexMediaAgent', None)
        cv_testcase.ma_machine = None
        cv_testcase.runid = int(time())
        cv_testcase.is_client_big_data_apps = None
        cv_testcase.is_client_network_share = None
        cv_testcase.master_node = cv_testcase.tcinputs.get('MasterNode', None)
        cv_testcase.data_access_nodes = None
        cv_testcase.no_of_streams = None
        cv_testcase.instance_name = None
        cv_testcase.cleanup_run = cv_testcase.tcinputs.get('CleanupRun', True)
        cv_testcase.hdfs_user = 'hdfs'
        cv_testcase.cluster_type = None
        cv_testcase.client_node = None
        cv_testcase.sdkPath = cv_testcase.tcinputs.get('sdkPath', None)
        cv_testcase.whichPython = cv_testcase.tcinputs.get('whichPython', None)

        # wait time after populating the data for journals to catch up
        cv_testcase.WAIT_TIME = cv_testcase.tcinputs.get('WaitTime', 30)

        # no of days to retain test data on the machine.
        cv_testcase.RETAIN_DAYS = 10

        cv_testcase.should_wait = True

        if (cv_testcase.test_path is None and cv_testcase.test_path_list is None and mandatory):
            raise Exception(
                "TestPath is a mandatory string input for the test case."
                "TestPathList is a mandatory list input for the test case."
                "At least one of these MUST be provided."
            )

        if (cv_testcase.storage_policy is None and mandatory):
            raise Exception(
                "StoragePolicyName is a"
                " mandatory string input for the test case"
            )

        if isinstance(cv_testcase.tcinputs.get('DataAccessNodes', None), list):
            cv_testcase.data_access_nodes = cv_testcase.tcinputs['DataAccessNodes']

        # Auto Center doesn't accept traditional lists, we need to convert a CSV string input to list in those cases
        elif isinstance(cv_testcase.tcinputs.get('DataAccessNodes', None), str):
            cv_testcase.data_access_nodes = cv_testcase.tcinputs['DataAccessNodes'].split(',')

        if isinstance(cv_testcase.tcinputs.get('NoOfStreams', None), int):
            cv_testcase.no_of_streams = cv_testcase.tcinputs['NoOfStreams']

        if cv_testcase.agent is not None:

            # Agent Name == File system
            if cv_testcase.agent.agent_name.lower() in ["file system", "windows file System", "linux file system"]:
                cv_testcase.instance = cv_testcase.agent.instances.get(
                    "DefaultInstanceName"
                )

                # For Network share client, the data access nodes will be set in input JSON
                if cv_testcase.data_access_nodes is not None:
                    cv_testcase.is_client_network_share = True
                    cv_testcase.skiplink = True
                    # select the first node in data access nodes as master node
                    if cv_testcase.data_access_nodes:
                        cv_testcase.client_node = cv_testcase.data_access_nodes[0]

                    if cv_testcase.client_node is None:
                        raise Exception("The data access node list cannot be empty")

                    # use cvd for connecting with the master node
                    if cv_testcase.client_machine is None:
                        cv_testcase.client_machine = Machine(
                            cv_testcase.client_node,
                            cv_testcase.commcell
                        )

                    # for network share clients, set number of streams to be 2* number of data
                    # access nodes
                    if cv_testcase.no_of_streams is None:
                        cv_testcase.no_of_streams = 2 * len(cv_testcase.data_access_nodes)

                # for filesystem clients set number of streams to be 1
                if cv_testcase.no_of_streams is None:
                    cv_testcase.no_of_streams = 1

                if cv_testcase.index_ma:
                    cv_testcase.ma_machine = Machine(
                        cv_testcase.index_ma,
                        cv_testcase.commcell
                    )

            # Agent Name = Big Data Apps
            if cv_testcase.agent.agent_name.lower() == "big data apps":
                cv_testcase.is_client_big_data_apps = True
                cv_testcase.skiplink = True
                cv_testcase.client_node = None

                if cv_testcase.master_node is not None and cv_testcase.master_node != '':
                    cv_testcase.client_node = cv_testcase.master_node
                elif cv_testcase.data_access_nodes is not None and cv_testcase.data_access_nodes:
                    cv_testcase.client_node = cv_testcase.data_access_nodes[0]

                if cv_testcase.client_node is None:
                    raise Exception(
                        "The Master node and the data access nodes both cannot be empty")

                if cv_testcase.master_node is None:
                    cv_testcase.master_node = ''

                first_instance = None
                instances_obj = cv_testcase.agent.instances
                first_instance = instances_obj.get(next(iter(instances_obj.all_instances)))
                if first_instance is not None:
                    distributed_inst_prop = first_instance.properties.get('distributedClusterInstance')
                    if distributed_inst_prop is not None:
                        cv_testcase.cluster_type = distributed_inst_prop.get('clusterType')
                        if (
                                (cv_testcase.cluster_type is not None)
                                and
                                (cv_testcase.cluster_type == DistributedClusterType.HADOOP.value)
                        ):
                            cv_testcase.hdfs_user = (
                                distributed_inst_prop['clusterConfig']['hadoopConfig']['hadoopSites'][0]['hdfsUser'])

                # use cvd for all communications with the master node
                if cv_testcase.client_machine is None:
                    if (
                            (cv_testcase.cluster_type is not None)
                            and
                            (cv_testcase.cluster_type == DistributedClusterType.HADOOP.value)
                    ):
                        cv_testcase.client_machine = Machine(
                            machine_name=cv_testcase.client_node,
                            commcell_object=cv_testcase.commcell,
                            hdfs_user=cv_testcase.hdfs_user
                        )
                    else:
                        cv_testcase.client_machine = Machine(
                            cv_testcase.client_node,
                            cv_testcase.commcell
                        )

                # If no of streams is not given in input. Configure 2 * no of data access
                # nodes for big data apps
                if cv_testcase.no_of_streams is None:
                    cv_testcase.no_of_streams = 2 * len(cv_testcase.data_access_nodes)

        else:
            raise Exception(
                "Agent object is not initialized for test case. "
                "Valid ClientName and AgentName must be provided "
                "in the inputs.")

        # If credentials are given, directly login using credentials
        if cv_testcase.hostname is not None:
            if ((cv_testcase.username is not None) and (cv_testcase.password is not None)):
                cv_testcase.client_machine = Machine(
                    cv_testcase.hostname,
                    cv_testcase.commcell,
                    username=cv_testcase.username,
                    password=cv_testcase.password
                )

        # If no credentials are given, use cvd for all communications
        if cv_testcase.client_machine is None:
            cv_testcase.client_machine = Machine(
                cv_testcase.client.client_name,
                cv_testcase.commcell
            )

        if isinstance(cv_testcase.client_machine, WindowsMachine):
            cv_testcase.applicable_os = cv_testcase.os_list.WINDOWS
            cv_testcase.slash_format = '\\'
            from FileSystem.FSUtils.winfshelper import WinFSHelper
            cv_testcase.helper = WinFSHelper(cv_testcase)
        else:
            cv_testcase.slash_format = '/'
            cv_testcase.applicable_os = cv_testcase.os_list.UNIX
            if cv_testcase.client_machine.os_flavour != 'OS400':
                from FileSystem.FSUtils.unixfshelper import UnixFSHelper
                cv_testcase.helper = UnixFSHelper(cv_testcase)
            else:
                from FileSystem.FSUtils.ibmihelper import IBMiHelper
                cv_testcase.helper = IBMiHelper(cv_testcase)
                if cv_testcase.sdkPath is not None:
                    from FileSystem.FSUtils.ibmicvc import IBMiCVC
                    cv_testcase.cvc = IBMiCVC(cv_testcase)

        if cv_testcase.test_path is None:
            if 'SubclientName' in cv_testcase.tcinputs and \
                len(cv_testcase.subclient.content) > 0 and \
                cv_testcase.subclient.content[0] != cv_testcase.slash_format:
                cv_testcase.test_path = "{0}{1}cv_fs_automation_{2}".format(
                    cv_testcase.subclient.content[0].rstrip(cv_testcase.slash_format),
                    cv_testcase.slash_format,
                    cv_testcase.id)
            else:
                cv_testcase.test_path = "{0}{1}cv_fs_automation_{2}".format(
                    cv_testcase.client_machine.client_object.install_directory.rstrip(cv_testcase.slash_format),
                    cv_testcase.slash_format,
                    cv_testcase.id
                    )

        if cv_testcase.storage_policy is None:
            if 'SubclientName' in cv_testcase.tcinputs and cv_testcase.subclient.storage_policy is not None:
                cv_testcase.storage_policy = cv_testcase.subclient.storage_policy
            else:
                cv_testcase.storage_policy = FSHelper.get_storage_policy(cv_testcase.commcell)

        if cv_testcase.client_machine.os_flavour == 'OpenVMS':
            cv_testcase.helper.mount_openvms_testpath(cv_testcase.test_path)
            cv_testcase.client_machine.test_path = cv_testcase.test_path

        # Hard links and Symbolic links are disabled by default for windows
        if cv_testcase.client_machine.os_info == "WINDOWS":
            cv_testcase.hlinks = cv_testcase.tcinputs.get('hlinks', False)
            cv_testcase.slinks = cv_testcase.tcinputs.get('slinks', False)

        # nfs_client_mount_dir IS WHERE THE SHARE WILL BE MOUNTED FOR AUTOMATION'S USAGE.
        if isinstance(cv_testcase.client_machine, UnixMachine) and cv_testcase.client_machine.client_object:
            cv_testcase.nfs_client_mount_dir = cv_testcase.slash_format.join(
                (cv_testcase.client_machine.client_object.job_results_directory, 'tmp_mnt'))

    # The subclient_job_results_directory is less of an input to the test case
    # and more of a constant for the test case, hence it has been defined as a property.
    @property
    def subclient_job_results_directory(self):
        """Returns the subclient job  results directory path."""

        common_sc_jr_path_items = [str(self.testcase.commcell.commcell_id), str(self.testcase.subclient.subclient_id)]
        if self.testcase.client_machine.os_info == "WINDOWS":
            common_sc_jr_path_items = ["CV_JobResults", "iDataAgent", "FileSystemAgent"] + common_sc_jr_path_items
        if self.testcase.data_access_nodes:
            sc_jr_path_on_node = {node: self.testcase.client_machine.join_path(Machine(node, self.testcase.commcell).client_object.job_results_directory, *common_sc_jr_path_items) for node in self.testcase.data_access_nodes}
        else:
            sc_jr_path_on_node = {self.testcase.client_machine.machine_name: self.testcase.client_machine.join_path(self.testcase.client_machine.client_object.job_results_directory, *common_sc_jr_path_items)}
        return sc_jr_path_on_node

    def verify_dc(self, jobid, log_dir=None):
        """Verifies whether the job used Scan optimization

             Args:
                jobid    (int)  -- jobid to be checked

                log_dir  (str)  -- path of the log directory
                    default:None

            Returns:
                bool   -   Returns True if dc scan was successful or else false

            Raises:
                Exception:
                    if any error occurred while verifying the scan status

        """
        self.log.error("Arguments %s %s", jobid, str(log_dir))
        self.log.error("Has to be implemented in child class before calling")
        raise NotImplementedError
        # Below code is unreachable, but adding it to avoid pylint error
        # as we are calling this function in parent code using child object and using the return value from child
        return False

    def get_restore_stream_count(self, job, log_dir=None):
        """Returns the number of streams used by a restore job

             Args:
                job    (object)  -- job object of the restore job to be checked

                log_dir  (str)  -- path of the log directory
                    default:None

            Returns:
                int   -   the number of streams used by a restore job

            Raises:
                Exception:
                    if any error occurred while getting the restore stream count

        """
        self.log.error("Arguments %s %s", job.job_id, str(log_dir))
        self.log.error("Has to be implemented in child class before calling")
        raise NotImplementedError
        # Below code is unreachable, but adding it to avoid pylint error
        # as we are calling this function in parent code using child object and using the return value from child
        return -1

    def create_backupset(self, name, delete=False, is_nas_turbo_backupset=False):
        """Creates new backupset with specified parameters
             under the current testcase Instance.

            Checks if the backupset exists or not.
            If the backupset exists, deletes the existing backupset
            and creates new one with the testcase id.

            Args:
                name   (str)  -- name of the backupset

                delete (bool) -- indicates whether existing backupset should be deleted
                    default: False

                is_nas_turbo_backupset  (bool): True for NAS based client.

            Returns:
                None

            Raises:
                Exception - Any error occured during backupset creation

        """
        try:
            self.log.info("Checking if backupset %s exists.", name)
            backupsets_object = self.testcase.instance.backupsets
            if backupsets_object.has_backupset(name):
                if delete:
                    self.log.info("Backupset exists, deleting backupset %s", name)
                    backupsets_object.delete(name)
                    self.log.info("Creating backupset %s", name)
                else:
                    self.log.info("Backupset exists, using existing backupset %s", name)
                    self.testcase.backupset = backupsets_object.get(name)
                    return
            else:
                self.log.info("Backupset doesn't exist, creating backupset %s", name)
            self.testcase.backupset = (
                backupsets_object.add(name,
                                      is_nas_turbo_backupset=is_nas_turbo_backupset)
            )
        except Exception as excp:
            self.log.error('Backupset Creation Failed with error: %s', excp)
            raise Exception('Backupset Creation Failed with error: {0}'.format(str(excp)))

    def create_instance(self, name, delete=False):
        """Creates new Instance with specified parameters
             under the current Client.

            Checks if the Instance exists or not.
            If the Instance exists, deletes the existing Instance
            and creates new one with the testcase id.

            Args:
                name   (str)  -- name of the Instance

                delete (bool) -- indicates whether existing Instance should be deleted
                    default: False

            Returns:
                None

            Raises:
                Exception - Any error occured during Instance creation

        """
        try:
            self.log.info("Checking if Instance %s exists.", name)
            instances_object = self.testcase.agent.instances
            if instances_object.has_instance(name):
                if delete:
                    self.log.error("Instance exists, deleting Instance %s", name)
                    instances_object.delete(name)
                    self.log.info("Creating Instance %s", name)
                else:
                    self.log.info("Instance exists, using existing Instance %s", name)
                    self.testcase.instance = instances_object.get(name)
                    if (
                            (self.testcase.cluster_type is not None)
                            and
                            (self.testcase.cluster_type == DistributedClusterType.HADOOP.value)
                    ):
                        self.testcase.backupset = self.testcase.instance.backupsets.get("HDFS")
                    else:
                        self.testcase.backupset = self.testcase.instance.backupsets.get("DefaultBackupSet")
                    return
            else:
                self.log.info("Instance doesn't exist, creating Instance %s", name)

            data_access_nodes_json = []
            for access_node in self.testcase.data_access_nodes:
                data_access_nodes_json.append({"clientName": access_node})

            json = {
                "instanceName": name,
                "MasterNode": self.testcase.master_node,
                "dataAccessNodes": data_access_nodes_json
            }
            self.testcase.instance = (
                instances_object.add_big_data_apps_instance(json)
            )
            if (
                    (self.testcase.cluster_type is not None)
                    and
                    (self.testcase.cluster_type == DistributedClusterType.HADOOP.value)
            ):
                self.testcase.backupset = self.testcase.instance.backupsets.get("HDFS")
            else:
                self.testcase.backupset = self.testcase.instance.backupsets.get("DefaultBackupSet")
        except Exception as excp:
            self.log.error('Instance Creation Failed with error: %s', str(excp))
            raise Exception(
                'Instance Creation Failed with error: {0}'.format(str(excp)))

    def create_subclient(
            self,
            name,
            storage_policy,
            content,
            filter_content=None,
            exception_content=None,
            trueup_option=True,
            trueup_days=30,
            scan_type=ScanType.RECURSIVE,
            data_readers=2,
            allow_multiple_readers=False,
            read_buffer_size=512,
            block_level_backup=None,
            delete=False,
            data_access_nodes=None,
            **kwargs
    ):
        """Creates subclient under current testcase backupset
            with specified parameters

            Args:
                name (str)                       -- subclient name

                storage_policy (str)             -- storage policy to assign to subclient

                content (list)                   -- content list

                filter_content (list)            -- filter list
                    default: None

                exception_conent (list)          -- exception list
                    default: None

                trueup_option (bool)             -- enable / disable true up
                    default: True

                trueup_days (int)                -- trueup after n days value of the subclient
                    default: 30

                data_readers (int)               -- number of data readers
                    default: 2

                allow_multiple_readers (bool)    -- enable / disable allow multiple readers
                    default: False

                read_buffer_size (int)           -- read buffer size in KB
                    default: 512

                delete (bool)                    -- indicates whether existing subclient should be deleted
                    default: False

                scan_type(ScanType(Enum))        --  scan type (RECURSIVE/OPTIMIZED/CHANGEJOURNAL)
                    default: ScanType.RECURSIVE

                data_access_nodes(list)          -- Data Access nodes for NFS share or big data apps
                    default : None

                block_level_backup (str)         -- blocklevel backup data switch
                    default: None


                \*\*kwargs  (dict)              --  Optional arguments.

                Available kwargs Options:

                    software_compression    (SoftwareCompression(Enum)) --  Software compression property for a subclient.
                    (ON_CLIENT/ON_MEDIA_AGENT/USE_STORAGE_POLICY_SETTINGS/OFF)

                    catalog_acl             (bool)                      --  Collects file permissions
                    default: False

                    use_vss                 (dict)                      --  Enable or Disable Use VSS, Applicable to Windows FS only.
                    Dictionary contains the keys 'useVSS', 'vssOptions' and 'useVssForAllFilesOptions'.

                        useVSS:
                            True    -   ENABLED
                            False   -   DISABLED

                        vssOptions:
                            1   -   For all files
                            2   -   For locked files only

                        useVssForAllFilesOptions:
                            1   -   Fail the job
                            2   -   Continue and reset access time
                            3   -   Continue and do not reset access time

                    impersonate_user        (dict)                      --  Details for impersonation, Applicable to Windows FS only.
                    Dictionary values are 'username', 'password'.

            Returns:
                None

            Raises:
                Exception - Any error occurred during Subclient creation

        """
        try:
            self.log.info("Checking if subclient %s exists.", name)
            subclients_object = self.testcase.backupset.subclients

            if subclients_object.has_subclient(name):
                if delete:
                    self.log.info("Subclient exists, deleting subclient %s", name)
                    subclients_object.delete(name)
                    self.log.info("Creating subclient %s", name)
                else:
                    self.log.info("Subclient exists, use existing subclient %s", name)
                    self.testcase.subclient = subclients_object.get(name)
                    return
            else:
                self.log.info("Subclient doesn't exist, creating subclient %s", name)

            self.testcase.subclient = (
                subclients_object.add(name, storage_policy))
            self.update_subclient(
                content=content,
                filter_content=filter_content,
                exception_content=exception_content,
                trueup_option=trueup_option,
                trueup_days=trueup_days,
                scan_type=scan_type,
                data_readers=data_readers,
                allow_multiple_readers=allow_multiple_readers,
                block_level_backup=block_level_backup,
                read_buffer_size=read_buffer_size,
                data_access_nodes=data_access_nodes,
                software_compression=kwargs.get('software_compression', None),
                catalog_acl=kwargs.get('catalog_acl', None),
                backup_onetouch=kwargs.get('backup_onetouch', None),
                onetouch_server=kwargs.get('onetouch_server', None),
                onetouch_server_directory=kwargs.get('onetouch_server_directory', None),
                description=kwargs.get('description', None),
                use_vss={'useVSS': True, 'useVssForAllFilesOptions': 1, 'vssOptions': 2},
                impersonate_user=kwargs.get('impersonate_user', None)
            )

        except Exception as excp:
            self.log.error('Subclient Creation Failed with error: %s', str(excp))
            raise Exception('Subclient Creation Failed with error: {0}'.format(str(excp)))

    def update_subclient(self,
                         storage_policy=None,
                         content=None,
                         filter_content=None,
                         exception_content=None,
                         trueup_option=None,
                         trueup_days=None,
                         scan_type=None,
                         data_readers=None,
                         allow_multiple_readers=None,
                         read_buffer_size=None,
                         block_level_backup=None,
                         createFileLevelIndex=False,
                         data_access_nodes=None,
                         backup_system_state=None,
                         **kwargs):
        """Updates subclient property of current
            testcase subclient with specified parameters

            Args:
                storage_policy (str)            -- storage policy to assign to subclient
                    default: None

                content (list)                  -- content list
                    default: None

                filter_content (list)           -- filter list
                    default: None

                exception_conent (list)         -- exception list
                    default: None

                trueup_option (bool)            -- enable / disable true up
                    default: None

                trueup_days (int)               -- trueup after n days value of the subclient
                    default: None

                scan_type(ScanType(Enum))       -- scan type(RECURSIVE/OPTIMIZED/CHANGEJOURNAL)
                    default: None

                data_readers (int)              -- number of data readers
                    default: None

                allow_multiple_readers (bool)   -- enable / disable allow multiple readers
                    default: None

                read_buffer_size (int)          -- read buffer size in KB
                    default: None

                data_access_nodes (list)        -- sets the list passed as data access nodes
                or backup nodes for this subclient

                blockLevelBackup                -- enable/Disable Blocklevel Option
                    default:None

                createFileLevelIndex            -- Enable/Diable Metadata option
                     default:False

                backup_system_state             --  Enable/Disable system state option
                     default:None

                backup_onetouch                 --  Enable/Disable 1-Touch Recovery option
                    default:None

                onetouch_server                 -- provides onetouch server name
                    default:None

                onetouch_server_directory       -- Provides onetouch server directory
                    default:None

                \*\*kwargs  (dict)              --  Optional arguments.

                    software_compression    (SoftwareCompression(Enum)) --  Software compression property for a subclient.
                    (ON_CLIENT/ON_MEDIA_AGENT/USE_STORAGE_POLICY_SETTINGS/OFF)

                    catalog_acl             (bool)                      --  Collects file permissions
                    default: False

                    use_vss                 (dict)                      --  Enable or Disable Use VSS, Applicable to Windows FS only.
                    Dictionary contains the keys 'useVSS', 'vssOptions' and 'useVssForAllFilesOptions'.

                        useVSS:
                            True    -   ENABLED
                            False   -   DISABLED

                        vssOptions:
                            1   -   For all files
                            2   -   For locked files only

                        useVssForAllFilesOptions:
                            1   -   Fail the job
                            2   -   Continue and reset access time
                            3   -   Continue and do not reset access time

                    impersonate_user        (dict)                      --  Details for impersonation, Applicable to Windows FS only.
                    Dictionary values are 'username', 'password'.

            Returns:
                None

            Raises:
                Exception - Any error occurred during Subclient Property update

        """
        try:
            if storage_policy is not None:
                self.testcase.subclient.storage_policy = storage_policy
            if content is not None:
                self.testcase.subclient.content = content
                for content in content:
                    if re.match(r'(?P<server>[\w\d\.]*):/(?P<share>[\w\d\.\/]*)', content):
                        # file_server:/path CONTENT MEANS WE NEED TO SET FOLLOWING PROPERTY SO -amr IS SENT TO IFIND.
                        self.testcase.subclient.network_share_auto_mount = True
            if filter_content is not None:
                self.testcase.subclient.filter_content = filter_content
            if exception_content is not None:
                self.testcase.subclient.exception_content = exception_content
            if scan_type is not None:
                self.testcase.subclient.scan_type = scan_type.value
            if trueup_option is not None:
                self.testcase.subclient.trueup_option = trueup_option
            if trueup_days is not None:
                self.testcase.subclient.trueup_days = trueup_days
            if data_readers is not None:
                if data_access_nodes is not None:
                    self.testcase.subclient.data_readers = 2 * len(data_access_nodes)
                else:
                    self.testcase.subclient.data_readers = data_readers
            if allow_multiple_readers is not None:
                if data_access_nodes is not None:
                    self.testcase.subclient.allow_multiple_readers = True
                else:
                    self.testcase.subclient.allow_multiple_readers = (
                        allow_multiple_readers)
            if read_buffer_size is not None:
                self.testcase.subclient.read_buffer_size = read_buffer_size

            if self.testcase.is_client_big_data_apps:
                self.testcase.subclient.set_data_access_nodes(data_access_nodes)
            elif self.testcase.is_client_network_share:
                if data_access_nodes is not None:
                    self.testcase.subclient.set_backup_nodes(data_access_nodes)

            if block_level_backup is not None:
                self.testcase.subclient.block_level_backup_option = block_level_backup
            if createFileLevelIndex is not False:
                self.testcase.subclient.create_file_level_index_option = createFileLevelIndex
            if backup_system_state is not None:
                self.testcase.subclient.system_state_option = backup_system_state
            if kwargs.get('catalog_acl', None):
                self.testcase.subclient.catalog_acl = kwargs.get('catalog_acl')
            if kwargs.get('backup_onetouch', None):
                self.testcase.subclient.onetouch_option = kwargs.get('backup_onetouch')

            if kwargs.get('onetouch_server', None):
                self.testcase.subclient.onetouch_server = kwargs.get('onetouch_server')

            if kwargs.get('onetouch_server_directory', None):
                self.testcase.subclient.onetouch_server_directory = kwargs.get(
                    'onetouch_server_directory')

            if kwargs.get('software_compression', None):
                self.testcase.subclient.software_compression = kwargs.get('software_compression')

            if kwargs.get('use_vss', None) and self.testcase.applicable_os.upper() == 'WINDOWS':
                self.testcase.subclient.use_vss = kwargs.get('use_vss')

            if kwargs.get('description', None):
                self.testcase.subclient.description = kwargs.get('description')

            if kwargs.get('impersonate_user', None):
                self.testcase.subclient.impersonate_user = kwargs.get('impersonate_user')

        except Exception as excp:
            self.log.error('Subclient Update Failed with error: %s', str(excp))
            raise Exception('Subclient Update Failed with error: {0}'.format(str(excp)))

    def run_backup(self,
                   backup_level="Incremental",
                   incremental_backup=False,
                   incremental_level='BEFORE_SYNTH',
                   wait_to_complete=True):
        """Initiates backup job with specified options
            on the current testcase subclient object
            and waits for completion.

            Args:
                backup_level            (str)   --  level of backup
                (Full/Incremental/Differential/Synthetic_full)

                    default: Incremental

                incremental_backup      (bool)  --  run incremental backup
                only applicable in case of Synthetic_full backup

                    default: False

                incremental_level       (str)   --  run incremental backup
                before/after synthetic full (BEFORE_SYNTH / AFTER_SYNTH)
                only applicable in case of Synthetic_full backup

                    default: BEFORE_SYNTH

                wait_to_complete        (bool)  --  Specifies whether to wait until job finishes.

            Returns:
                list    - list of instances of the Job class for the backup jobs ran

            Raises:
                Exception - Any error occurred while running backup or
                backup didn't complete successfully.

        """
        try:
            if (backup_level == "Synthetic_full" and incremental_backup):
                backup_type = "".join((backup_level + " with incremental " + incremental_level))
            else:
                backup_type = backup_level

            self.log.info("Starting %s Backup ", backup_type)
            job = self.testcase.subclient.backup(
                backup_level, incremental_backup, incremental_level)

            if job.backup_level is None:
                job_type = job.job_type
            else:
                job_type = job.backup_level

            if wait_to_complete:
                self.log.info(
                    "Waiting for completion of %s backup with Job ID: %s", job_type, str(
                        job.job_id))

                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run {0} backup {1} with error: {2}".format(
                            job_type, str(job.job_id), job.delay_reason
                        )
                    )

                if not job.status.lower() == "completed":
                    raise Exception(
                        "{0} job {1}status is not Completed, job has status: {2}".format(
                            job_type, str(
                                job.job_id), job.status))

                self.log.info("Successfully finished %s job %s", job_type, str(job.job_id))

                # Wait for second job to complete correctly if applicable
                # and return both of them
                if (backup_level == "Synthetic_full" and incremental_backup):
                    if incremental_level == 'BEFORE_SYNTH':
                        expected_job2_type = 'synthetic full'
                    else:
                        expected_job2_type = 'incremental'
                    attempts = 1
                    job2 = None
                    while attempts < 6:
                        latest_job = self.testcase.subclient.find_latest_job()
                        if latest_job.backup_level is None:
                            latest_job_type = latest_job.job_type
                        else:
                            latest_job_type = latest_job.backup_level
                        if int(latest_job.job_id) > int(job.job_id):
                            if latest_job_type.lower() == expected_job2_type:
                                job2 = latest_job
                                break
                            else:
                                raise Exception(
                                    'Expected {0} but job {1} is {2}'.format(
                                        expected_job2_type, str(latest_job.job_id), latest_job_type
                                    )
                                )
                        attempts = attempts + 1
                        sleep(10)
                    if job2 is None:
                        raise Exception('Expected {0} job not started within timeout'.format(
                            expected_job2_type))

                    if job2.backup_level is None:
                        job2_type = job2.job_type
                    else:
                        job2_type = job2.backup_level
                    self.log.info(
                        "Waiting for completion of %s backup with Job ID: %s",
                        job2_type,
                        str(job2.job_id)
                    )
                    if not job2.wait_for_completion():
                        raise Exception(
                            "Failed to run {0} backup {1} with error: {2}".format(
                                job2_type, str(job2.job_id), job2.delay_reason
                            )
                        )

                    if not job2.status.lower() == "completed":
                        raise Exception(
                            "{0} job {1} status is not Completed, job has status: {2}".format(
                                job2_type, str(job2.job_id), job2.status
                            )
                        )

                    self.log.info("Successfully finished %s job %s", job2_type, str(job2.job_id))
                    return [job, job2]
                else:
                    self._update_ref_time_dict(job)
                    return [job]

            else:
                self.log.info("Started %s backup with Job ID: %s", job_type, str(job.job_id))
                return [job]

        except Exception as excp:
            self.log.error('Job error : %s', excp)
            raise Exception('Job error : {0}'.format(str(excp)))

    def find_items_subclient(
            self, path='\\**\\*', filters=None, from_time=None,
            to_time=None, copy_precedence=0, media_agent='',
            page_size=100000, sorted_output=True
    ):
        """Searches a file/folder in the subclient backup content,
            and returns all the files matching the filters given.

             Args:
                path   (str)          --  directory path to get the items list
                    default: '\\**\\*'

                filters (list)        -- list of filters
                                            to exclude from the given path

                from_time (int)       --  time to find the contents after
                    format: timestamp
                    default: None

                to_time (int)         --  time to find the contents before
                    format: timestamp
                    default: None

                copy_precedence (int) -- copy precedence for the find operation
                    default: 0

                media_agent (str)     -- mediaagent to be used for the find operation
                    default: ''

                page_size (int)       -- page size for the find
                    default: 100000

                sorted_output (bool)  -- to specify whether the list should be sorted.
                    default : True

            Returns:
                list    -   list of the items
                             from the find operation on the subclient

                dict    -   Dictionary of all the paths
                             with additional metadata retrieved from browse

            Raises:
                Exception:
                    if any error occurred while doing the find operation.

        """
        try:
            find_options = {}
            find_options['path'] = path
            if from_time is not None:
                find_options['from_time'] = from_time
            if to_time is not None:
                find_options['to_time'] = to_time
            if filters is None:
                filters = []

            find_options['copy_precedence'] = copy_precedence
            find_options['media_agent'] = media_agent
            find_options['page_size'] = page_size
            find_options['filters'] = filters

            find_items, meta_data = self.testcase.subclient.find(find_options)

            if sorted_output:
                find_items.sort()

            return find_items, meta_data

        except Exception as excp:
            self.log.error('Error occurred while doing the find operation: %s', str(excp))
            raise Exception('Error occurred while doing the find operation: {0}'.format(
                str(excp)))

    def compare_lists(self, source_list, destination_list, sort_list=False):
        """Compares the source list with destination list
            and checks if they are same.

             Args:
                source_list      (list)

                destination_list (list)

                sort_list        (bool)    -- Indicates whether the list
                should be sorted before comparison

                    default: False

            Returns:
                bool   -   Returns True if lists are same or returns False

            Raises:
                Exception:
                    if any error occurred while comparing the lists.

        """

        try:
            if sort_list:
                source_list.sort()
                destination_list.sort()

            if source_list == destination_list:
                return True

            diff = Differ().compare(source_list, destination_list)
            diff_output = '\n'.join(
                [x for x in list(diff) if not x.startswith('  ')])
            self.log.info("Source and destination lists differ")
            self.log.info("Diff Output : \n%s", str(diff_output))
            return False

        except Exception as excp:
            self.log.error('Error occurred while comparing the lists %s', str(excp))
            raise Exception('Error occurred while comparing the lists {0}'.format(str(excp)))

    def restore_in_place(
            self,
            paths,
            **kwargs
    ):
        """Restores the files/folders specified
            in the input paths list to the same location.

            Args:
                paths     (list)  --  list of full paths of files/folders to restore

                Kwargs:
                Keyword argument of restore options:

                    Example:
                        {

                        overwrite                (bool)  -- unconditional overwrite files
                            default: True

                        restore_data_and_acl     (bool)  -- restore data and ACL files

                            default: True

                        copy_precedence          (int)   -- copy precedence value
                            default: None

                        from_time                (str)   -- time from(restore the contents after)
                            format: YYYY-MM-DD HH:MM:SS

                            default: None

                        to_time                  (str)   -- time to(restore the contents before)
                            format: YYYY-MM-DD HH:MM:SS

                            default: None

                        restoreDataInsteadOfStub (bool)  -- to restore data instead of stub
                            default: True

                        restoreOnlyStubExists    (bool)  -- to restore data only if stub exists
                            default: False

                        overwriteFiles           (bool)  -- to overwrite if data on media is newer
                            default: True

                        doNotOverwriteFileOnDisk (bool)  -- to not overwrite files on disk
                            default: False

                        disableStubRestore       (bool)  -- will not allow stub restore
                        in case dangle stub existed

                            default True

                        _schedule_pattern        (dict)  -- schedule create job option

                            default :None

                        amr                      (bool)  -- If the restore is an auto mount restore (amr), certain
                        modifications will be made to compare_source and compare_destination so verification doesn't fail on account of invalid paths.

                        restore_nodes            (str/list) --  Client name or list of client names for the restore operation.
                        }

            Returns:
                object - instance of the Job class for this restore job

            Raises:
                Exception - Any error occurred while running restore
                or restore didn't complete successfully.

        """
        try:
            log = self.log
            log.info("Run Restore in place ")
            fs_options = {}
            overwrite = kwargs.get("overwrite", True)
            restore_data_and_acl = kwargs.get("restore_data_and_acl", True)
            copy_precedence = kwargs.get("copy_precedence", None)
            from_time = kwargs.get("from_time", None)
            to_time = kwargs.get("to_time", None)
            fs_options['restoreDataInsteadOfStub'] = kwargs.get("restoreDataInsteadOfStub", True)
            fs_options['restoreOnlyStubExists'] = kwargs.get("restoreOnlyStubExists", False)
            fs_options['overwriteFiles'] = kwargs.get("overwriteFiles", True)
            fs_options['doNotOverwriteFileOnDisk'] = kwargs.get("doNotOverwriteFileOnDisk", False)
            fs_options['disableStubRestore'] = kwargs.get("disableStubRestore", True)
            schedule_pattern = kwargs.get("schedule_pattern", None)

            # IF AUTO MOUNT RESTORE, CLIENT NODE NEEDS TO BE SET AS THE DESTINATION CLIENT FOR RESTORE
            if kwargs.get('amr', False):
                fs_options['client'] = self.testcase.client_node

            # MULTI NODE RESTORES IF restore_nodes IS SET
            if kwargs.get('restore_nodes', False):
                restore_nodes_json = []
                restore_nodes = [kwargs['restore_nodes']] if isinstance(kwargs['restore_nodes'], str) else kwargs['restore_nodes']
                for access_node in restore_nodes:
                    restore_nodes_json.append({"clientName": access_node})
                    fs_options['data_access_nodes'] = restore_nodes_json
                client = self.testcase.commcell.clients.get(restore_nodes[0])
                fs_options['multinode_restore'] = True
                fs_options['client'] = client.client_name
                fs_options['destination_appTypeId'] = 33 if "windows" in client.os_info.lower() else 29

            job = self.testcase.subclient.restore_in_place(
                paths=paths,
                overwrite=overwrite,
                restore_data_and_acl=restore_data_and_acl,
                copy_precedence=copy_precedence,
                from_time=from_time,
                to_time=to_time,
                fs_options=fs_options,
                schedule_pattern=schedule_pattern
            )

            self.log.info("Started restore in place job with job id: %s", (job.job_id))

            if kwargs.get('wait_to_complete', True):

                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run restore in place job with error: {0}".format(
                            job.delay_reason))

                if not job.status.lower() == "completed":
                    raise Exception(
                        "Job status is not Completed, job has status: {0}".format(
                            job.status))

                self.log.info("Successfully finished restore in place job")
                return job
            return job

        except Exception as excp:
            self.log.error('Job error : %s', str(excp))
            raise Exception('Job error : {0}'.format(str(excp)))

    def restore_out_of_place(
            self,
            destination_path,
            paths,
            client=None,
            overwrite=True,
            restore_data_and_acl=True,
            restore_ACL=True,
            copy_precedence=None,
            from_time=None,
            to_time=None,
            preserve_level=1,
            proxy_client=None,
            impersonate_user=None,
            impersonate_password=None,
            is_vlr_restore=False,
            is_client_big_data_apps=None,
            destination_instance=None,
            no_of_streams=1,
            restore_nodes=None,
            iscsi_server=None,
            **kwargs
    ):
        """Restores the files/folders specified
            in the input paths list to the input client,
            at the specified destination location.

            Args:
                client                (str/object) --  either the name or the instance of Client
                    default: None

                destination_path      (str)        --  full path of the restore location on client

                paths                 (list)       --  list of full paths of
                files/folders to restore

                overwrite             (bool)       --  unconditional overwrite files during restore
                    default: True

                restore_data_and_acl  (bool)       --  restore data and ACL of files
                    default: True

                restore_ACL           (bool)       --  restore ACLs for Unix
                    default: True

                copy_precedence       (int)        --  copy precedence value
                    default: None

                from_time             (str)        --  time to restore the contents after
                    format: YYYY-MM-DD HH:MM:SS

                    default: None

                to_time               (str)        --  time to retore the contents before
                    format: YYYY-MM-DD HH:MM:SS

                    default: None

                preserve_level        (int)        --  preserve level option to set in restore
                    default: 0

                proxy_client          (str)        --  proxy that needed to be used for restore
                    default: None

                impersonate_user      (str)        --  Impersonate user options for restore
                    default: None

                impersonate_password  (str)        --  Impersonate password option for restore
                in base64 encoded form

                    default: None

                is_vlr_restore        (bool)       --  Blocklevel Switch to run restore of Volume
                    default: False


                is_client_big_data_apps(bool)      --  to indicate if the destintation client
                is of type DISTRIBUTED_IDA

                    default : None

                destination_instance  (str)        --  destination instance if the destination
                client is of type DISTRIBUTED_IDA

                    default : None

                no_of_streams         (int)        --  number of streams used for restore
                    default : 1

                restore_nodes         (list)       --  the data access nodes used for restore
                in case of multi-node restore.

                    default : None

            Returns:
                object - instance of the Job class for this restore job

            Raises:
                Exception - Any error occurred while running restore
                            or restore didn't complete successfully.

        """
        try:

            self.log.info("Run Restore out of place ")
            multinode_restore = None

            # set fs_options
            fs_options = {}
            fs_options['preserve_level'] = preserve_level
            fs_options['restore_ACL'] = restore_ACL
            fs_options['no_of_streams'] = no_of_streams
            fs_options['fs_options'] = kwargs.get(
                "fs_options", {})

            if proxy_client is not None:
                fs_options['proxy_client'] = proxy_client

            if iscsi_server is not None:
                fs_options['iscsi_server'] = iscsi_server

            if is_vlr_restore is not False:
                fs_options['is_vlr_restore'] = is_vlr_restore

            if impersonate_user is not None:
                fs_options['impersonate_user'] = impersonate_user

            if impersonate_password is not None:
                fs_options['impersonate_password'] = impersonate_password

            # Multi-node restores if restore_nodes is set
            if restore_nodes is not None:
                fs_options['multinode_restore'] = True
                multinode_restore = True

            if multinode_restore:

                restore_nodes_json = []
                for access_node in restore_nodes:
                    restore_nodes_json.append({"clientName": access_node})

                fs_options['data_access_nodes'] = restore_nodes_json

                if is_client_big_data_apps:
                    # use the current test case client as default
                    if client is None:
                        client = self.testcase.client
                    elif isinstance(client, basestring):
                        client = self.testcase.commcell.clients.get(client)
                    fs_options['destination_instance'] = destination_instance
                    fs_options['destination_appTypeId'] = 64
                    fs_options['client_type'] = 29
                    if (
                            (destination_instance is not None)
                            and
                            (self.testcase.cluster_type is not None)
                            and
                            (self.testcase.cluster_type == DistributedClusterType.HADOOP.value)
                    ):
                        destination_agent = client.agents.get('Big Data Apps')
                        destination_instance_id = destination_agent.instances.get(destination_instance).instance_id
                        fs_options['destination_instance_id'] = destination_instance_id
                else:
                    # use the client_node as default for multi-node
                    if client is None:
                        client = self.testcase.commcell.clients.get(self.testcase.client_node)
                    elif isinstance(client, basestring):
                        client = self.testcase.commcell.clients.get(client)
                    fs_options['destination_appTypeId'] = 33 if "windows" in client.os_info.lower() else 29

            elif no_of_streams > 1:
                # use the current test case client as default
                if client is None:
                    client = self.testcase.client
                elif isinstance(client, basestring):
                    client = self.testcase.commcell.clients.get(client)
                fs_options['destination_appTypeId'] = 33 if "windows" in client.os_info.lower() else 29

            # use the current test case client as default
            if client is None:
                client = self.testcase.client_node if self.testcase.is_client_network_share else self.testcase.client

            if impersonate_user is None:
                fs_options.update({"impersonate_user": self.testcase.username, "impersonate_password": b64encode(
                    self.testcase.password.encode()).decode()}) if destination_path.startswith(r"\\") else None

            job = self.testcase.subclient.restore_out_of_place(
                client=client,
                destination_path=destination_path,
                paths=paths,
                overwrite=overwrite,
                restore_data_and_acl=restore_data_and_acl,
                copy_precedence=copy_precedence,
                from_time=from_time,
                to_time=to_time,
                fs_options=fs_options
            )

            self.log.info("Started restore out of place job with job id: %s", str(job.job_id))

            if kwargs.get('wait_to_complete', True):

                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run restore out of place job with error: {0}".format(
                            job.delay_reason))

                if not job.status.lower() == "completed":
                    raise Exception(
                        "Job status is not Completed, job has status: {0}".format(
                            job.status))

                self.log.info("Successfully finished restore out of place job")
                return job
            return job

        except Exception as excp:
            self.log.error('Job error : %s', str(excp))
            raise Exception('Job error : {0}'.format(str(excp)))

    def _verify_job_streams(self, job, no_of_streams, streams_used):
        """ Verifies if the job used expected number of streams

            Args:
                job (obj)                --  instance of the job class

                no_of_streams (int)      --  expected number of streams to be used

                streams_used (int)       --  actual number of streams used

            Raises:
                Exception - if the job didn't use expected streams

        """
        if streams_used == no_of_streams:
            self.log.info("Job %s used %s streams as expected", str(job.job_id), str(streams_used))
        else:
            self.log.error("Job:%s used incorrect streams ExpectedStreams:%s ActualStreams:%s",
                           str(job.job_id), str(no_of_streams), str(streams_used))
            raise Exception("Job:{0} used incorrect streams ExpectedStreams:{1} ActualStreams:{2}".format(
                job.job_id, no_of_streams, streams_used))

    def run_backup_verify(self,
                          scan_type=ScanType.RECURSIVE,
                          backup_level="Incremental",
                          incremental_backup=False,
                          incremental_level='BEFORE_SYNTH',
                          scan_marking=False):
        """ Initiates backup and performs the applicable verifications
             on current test case subclient

            Args:
                scan_type         (ScanType(Enum)) -- scan type(RECURSIVE/OPTIMIZED/CHANGEJOURNAL)
                    default: ScanType.RECURSIVE

                backup_level      (str)            -- level of backup
                (Full/Incremental/Differential/Synthetic_full)

                    default: Incremental

                incremental_backup(bool)           -- run incremental backup
                only applicable in case of Synthetic_full backup

                    default: False

                incremental_level (str)            -- run incremental backup
                before/after synthetic full(BEFORE_SYNTH / AFTER_SYNTH)
                only applicable in case of Synthetic_full backup

                    default: BEFORE_SYNTH

                scan_marking      (bool)           --  to check whether
                the job ran as Scan Marking Job

                    default: False

            Returns:
                list    - instances of the Job class for the backup job ran

            Raises:
                Exception - Any error occurred while
                running backup or during verification.

        """
        log = logger.get_log()
        testcase = self.testcase
        jobs = self.run_backup(
            backup_level=backup_level,
            incremental_backup=incremental_backup,
            incremental_level=incremental_level)
        for job in jobs:
            if (job.summary['totalFailedFiles'] != 0) or (job.summary['totalFailedFolders'] != 0):
                log.error("Job %s has failed items", str(job.job_id))
                raise Exception("Job {0} has failed items".format(str(job.job_id)))

            if (testcase.applicable_os == 'UNIX' and testcase.verify_dc) and (
                    backup_level.lower() != 'synthetic_full'):
                is_dc_used = self.verify_dc(job.job_id)
                if scan_type.value == ScanType.OPTIMIZED.value and not is_dc_used:
                    log.error("Job %s did not use scan optimization", str(job.job_id))
                    raise Exception("Job {0} did not use scan optimization".format(
                        str(job.job_id)))
                elif scan_type.value == ScanType.RECURSIVE.value and is_dc_used:
                    log.error("Job %s incorrectly used scan optimization", str(job.job_id))
                    raise Exception("Job {0} incorrectly used scan optimization".format(
                        str(job.job_id)))
            if scan_marking:
                result = self.verify_scan_marking(str(job.job_id))
                if not result:
                    raise Exception("Job {0} didn't complete in scan phase as expected.".format(
                        str(job.job_id)))
            elif backup_level.lower() != 'synthetic_full' and self.testcase.no_of_streams > 1:
                if not (self.testcase.is_client_big_data_apps or self.testcase.is_client_network_share):
                    self._verify_job_streams(job, self.testcase.no_of_streams, self.get_backup_stream_count(job))

        log.info("Job completed successfully without any failures")
        return jobs

    def run_restore_verify(self,
                           slash_format,
                           data_path,
                           tmp_path,
                           data_path_leaf,
                           job=None,
                           cleanup=True,
                           is_client_big_data_apps=None,
                           destination_instance=None,
                           no_of_streams=1,
                           proxy_client=None,
                           restore_nodes=None,
                           iscsi_server=None,
                           **kwargs):
        """ Initiates out of place restore for data backed up in the given job
            and performs the applicable verifications

            Args:
                slash_format    (str)       -- OS path separator format

                data_path       (str/list)  -- Source data path, a single source path can be provided as a string or
                multiple source paths can be provided as a list of strings.

                tmp_path        (str)       -- temporary path for restoring the data

                data_path_leaf  (str)       -- leaf level of the data_path

                job             (obj)       -- instance of the job class
                whose data needs to be restored

                    default : None

                cleanup (bool)           -- to indicate if restored data should be cleaned up
                    default : True

                is_client_big_data_apps(bool) -- to indicate if the destintation client
                is of type DISTRIBUTED_IDA

                    default : None

                destination_instance (str) -- destination instance if the destination
                client is of type DISTRIBUTED_IDA

                    default : None

                no_of_streams (int)      -- number of streams used for restore

                    default : 1

                proxy_client(str)        -- Proxy for restore

                    default : None

                restore_nodes (list)     -- the data access nodes used for restore in case of
                multi-node restore.

                    default : None

                iSCSIServer (str)         -- iscsi server option for restore

                    default: None

                kwargs (dict)                    -- dictionary of optional arguments
                Available kwargs Options:

                    in_place    (bool)  --  If restore needs to be done in-place.

                    amr         (bool)  --  If restore is an auto mount restore (amr), certain modifications will be
                    made to compare_source and compare_destination so verification doesn't fail on account of invalid paths.

            Returns:
               object - instance of the Job class for this restore job

            Raises:
                Exception - Any error occurred while
                            running restore or during verification.

        """
        testcase = self.testcase
        log = logger.get_log()
        client_machine = testcase.client_machine
        paths = []
        if isinstance(data_path, list):
            paths.extend(list(path for path in data_path))
        else:
            paths.append(data_path)

        # CONVERT \\ TO UNC-NT_ IF SELECTED ITEM IS A UNC PATH
        paths = (lambda paths: [str(path).replace("\\\\", "UNC-NT_") if path.startswith("\\\\") else path for path in
                                paths])(paths)

        dest_path = tmp_path + slash_format + data_path_leaf + "_restore"

        restore_from_time = None
        restore_to_time = None
        if job is not None:
            restore_from_time = str(
                datetime.utcfromtimestamp(job.summary['jobStartTime'])
            )
            restore_to_time = str(
                datetime.utcfromtimestamp(job.summary['jobEndTime'])
            )

        # clean up destination directory before starting restore
        testcase.client_machine.remove_directory(dest_path)

        log.info('Starting restore with source:%s destination:%s from_time:%s to_time:%s',
                 str(paths), dest_path, str(restore_from_time), str(restore_to_time))

        if kwargs.get('amr', False):
            assert self.testcase.nfs_client_mount_dir, 'Ensure self.testcase.nfs_client_mount_dir value is specified or exists.'

        if kwargs.get('in_place', False):
            source_suffix = "source"
            paths_copy = deepcopy(paths)

            if kwargs.get('amr', False):
                paths_copy = (
                    lambda paths_copy: [slash_format.join((self.testcase.nfs_client_mount_dir, path.split('/', 2)[-1]))
                                        for path in paths_copy])(paths_copy)

            for path in paths_copy:
                client_machine.rename_file_or_folder(path.replace("UNC-NT_", "\\\\"), "_".join((path.replace("UNC-NT_", "\\\\"), source_suffix)))

            if restore_nodes:
                kwargs['restore_nodes'] = restore_nodes

            jobs = self.restore_in_place(paths, **kwargs)

            compare_destination = data_path
            compare_source = "_".join((data_path, source_suffix))

        else:
            jobs = self.restore_out_of_place(
                destination_path=dest_path,
                paths=paths,
                from_time=restore_from_time,
                to_time=restore_to_time,
                is_client_big_data_apps=is_client_big_data_apps,
                destination_instance=destination_instance,
                no_of_streams=no_of_streams,
                proxy_client=proxy_client,
                restore_nodes=restore_nodes,
                iscsi_server=iscsi_server)

            compare_source = data_path
            compare_destination = dest_path + slash_format + data_path_leaf

        if kwargs.get('amr', False):
            compare_source = slash_format.join(
                (self.testcase.nfs_client_mount_dir, compare_source.split(slash_format, 2)[-1]))
            compare_destination = slash_format.join(
                (self.testcase.nfs_client_mount_dir, compare_destination.split(slash_format, 2)[-1]))

        log.info("Comparing source:%s destination:%s", compare_source, compare_destination)

        if restore_nodes is None and no_of_streams > 1:
            self._verify_job_streams(jobs, no_of_streams, self.get_restore_stream_count(jobs))

        result, diff_output = client_machine.compare_meta_data(
            compare_source,
            compare_destination,
            dirtime=testcase.dirtime,
            skiplink=testcase.skiplink
        )
        if result:
            log.info("Meta data comparison successful")
        else:
            log.error("Meta data comparison failed")
            log.info("Diff output: \n%s", diff_output)
            raise Exception("Meta data comparison failed")

        result, diff_output = client_machine.compare_checksum(
            compare_source, compare_destination
        )
        if result:
            log.info("Checksum comparison successful")
        else:
            log.error("Checksum comparison failed")
            log.info("Diff output: \n%s", diff_output)
            raise Exception("Checksum comparison failed")

        if testcase.applicable_os == 'UNIX':
            if testcase.acls:
                result, diff_output = client_machine.compare_acl(
                    compare_source,
                    compare_destination
                )
                if result:
                    log.info("ACL comparison successful")
                else:
                    log.error("ACL comparison failed")
                    log.info("Diff output: \n%s", diff_output)
                    raise Exception("ACL comparison failed")
            if testcase.xattr:
                result, diff_output = client_machine.compare_xattr(
                    compare_source,
                    compare_destination
                )
                if result:
                    log.info("XATTR comparison successful")
                else:
                    log.error("XATTR comparison failed")
                    log.info("Diff output: \n%s", diff_output)
                    raise Exception("XATTR comparison failed")

        if cleanup:
            if kwargs.get('in_place', False):
                for path in paths_copy:
                    testcase.client_machine.remove_directory("_".join((path, source_suffix)))
            elif kwargs.get('amr', False):
                testcase.client_machine.remove_directory(
                    slash_format.join((self.testcase.nfs_client_mount_dir, dest_path.split(slash_format, 2)[-1])))
            else:
                testcase.client_machine.remove_directory(dest_path)

        return jobs

    def run_find_verify(self, machine_path, job=None):
        """ Finds items backed up in the given job
            and compares it with items from machine

            Args:
                machine_path (str)       -- Source data path on the machine

                job (obj)                -- instance of the job class whose data needs to be found
                    default : None

            Returns:
                None

            Raises:
                Exception - Any error occurred while finding items or during verification.

        """
        log = logger.get_log()
        testcase = self.testcase

        if not testcase.find_verify:
            log.info("Skipping find verify as per input")
            return

        from_time = None
        to_time = None
        if job is not None:
            from_time = job.summary['jobStartTime']
            to_time = job.summary['jobEndTime']

        index_items, _ = self.find_items_subclient(
            from_time=from_time, to_time=to_time
        )

        # CONVERT UNC-NT_ TO \\ IF SELECTED ITEM IS A UNC PATH
        if machine_path.startswith('\\\\'):
            index_items = (lambda index_items: [str(item).replace("UNC-NT_", "\\") for item in index_items])(
                index_items)

        if self.testcase.client_machine.os_flavour == 'OpenVMS':
            for i, index_item in enumerate(index_items):
                index_item = index_item.replace(".;", ";")
                pos = index_item.find(";")
                if pos is not -1:
                    index_item = index_item[:pos]
                index_items[i] = index_item

        machine_items = testcase.client_machine.get_items_list(
            machine_path,
            include_parents=True
        )

        if self.compare_lists(index_items, machine_items):
            log.info(
                "Items from find operation matches with items on the machine"
            )
        else:
            raise Exception(
                "Items from find operation doesn't match"
                " with items on the machine")

    def add_new_data_incr(self,
                          incr_path,
                          slash_format,
                          scan_type=ScanType.RECURSIVE,
                          **kwargs):
        """ Adds new data for incremental with provisions for modification
            after this function execution call mod_data_incr for modifications

            Args:

                incr_path       (str)              -- path for adding data

                slash_format    (str)              -- OS path separator format

                scan_type       (ScanType(Enum))   -- scan type(RECURSIVE/OPTIMIZED/CHANGEJOURNAL)
                    default: ScanType.RECURSIVE

                kwargs (dict)                    -- dictionary of optional arguments
                Available kwargs Options:

                    dirs                (int)   --  number of directories in each level
                        default: 3

                    files               (int)   --  number of files in each directory
                        default: 5

                    file_size           (int)   --  Size of the files in KB
                        default: 20

                    levels              (int)   --  number of levels to be created
                        default: 1

                    hlinks              (bool)  --  whether to create hardlink files
                        default: True

                    slinks              (bool)  --  whether to create symbolic link files
                        default: True

                    sparse              (bool)  --  whether to create sparse files
                        default: True

                    sparse_hole_size    (int)   --  Size of the holes in sparse files in KB
                        default: 1024

                    options             (str)   --  to specify any other additional parameters
                        default: ""

                    username            (str)   --  Username of the account that needs to be impersonated when
                    creating a dataset on a UNC path (Windows ONLY).

                    password            (str)   --  Password of the account that needs to be impersonated when
                    creating a dataset on a UNC path (Windows ONLY).

            Returns:
                None

            Raises:
                Exception - Any error occurred while
                            populating the data.

        """
        log = logger.get_log()
        testcase = self.testcase

        dirs = kwargs.get('dirs', 3)
        files = kwargs.get('files', 5)
        file_size = kwargs.get('file_size', 20)
        levels = kwargs.get('levels', 1)
        hlinks = kwargs.get('hlinks', True)
        slinks = kwargs.get('slinks', True)
        sparse = kwargs.get('sparse', True)
        sparse_hole_size = kwargs.get('sparse_hole_size', 1024)
        options = kwargs.get('options', "")
        username = kwargs.get('username', "")
        password = kwargs.get('password', "")

        data_options = {}
        data_options['acls'] = testcase.acls
        data_options['unicode'] = testcase.unicode
        data_options['xattr'] = testcase.xattr
        data_options['long_path'] = testcase.long_path
        data_options['problematic'] = testcase.problematic
        testcase.incr_modify = incr_path + slash_format + "modify"
        testcase.incr_file_rename = incr_path + slash_format + "filerename"
        testcase.incr_dir_rename = incr_path + slash_format + "dirrename"
        testcase.incr_permissions_change = (
                incr_path + slash_format + "permissionschange")
        testcase.incr_delete_data = incr_path + slash_format + "deleted"
        testcase.incr_hlinks_change = None
        testcase.incr_acl_change = None
        testcase.incr_xattr_change = None

        # IBMi and OpenVMS do not support unix style user permissions
        if testcase.client_machine.os_flavour in ['OS400', 'OpenVMS']:
            testcase.incr_permissions_change = None

        # OS400 and OpenVMS do not support ACL, XATTR and doesn't preserve hardlink counts
        if testcase.applicable_os == 'UNIX' and testcase.client_machine.os_flavour not in [
            'OS400', 'OpenVMS']:
            if not testcase.is_client_big_data_apps:
                testcase.incr_hlinks_change = "{0}{1}hlinkschange".format(incr_path, slash_format)
            if testcase.acls:
                testcase.incr_acl_change = "{0}{1}aclchange".format(
                    incr_path, slash_format)
            if testcase.xattr:
                testcase.incr_xattr_change = "{0}{1}xattrchange".format(
                    incr_path, slash_format)

        else:
            # Recursive Scan on Windows cannot identify changes to ACLs
            if testcase.acls and scan_type.value != ScanType.RECURSIVE.value:
                testcase.incr_acl_change = "{0}{1}aclchange".format(
                    incr_path, slash_format)

            hlinks = testcase.hlinks if testcase.hlinks is False else hlinks
            slinks = testcase.slinks if testcase.slinks is False else slinks
            sparse = testcase.sparse if testcase.sparse is False else sparse

        # Disabling acl change for Lustre as it's not supported by Lustre Scan
        # Disabling permissions, xattr and acl change for Hadoop Scan as it is not supported
        if testcase.cluster_type is not None:
            if testcase.cluster_type == DistributedClusterType.LUSTREFS.value:
                testcase.incr_acl_change = None
            elif testcase.cluster_type == DistributedClusterType.HADOOP.value:
                testcase.incr_acl_change = None
                testcase.incr_xattr_change = None
                testcase.incr_permissions_change = None

        log.info("Adding data under path: %s", incr_path)
        # SKIP FILE RENAMES IF OS IS WINDOWS AND SCAN TYPE IS RECURSIVE
        if testcase.file_rename and not ((testcase.applicable_os == 'WINDOWS') and (
                scan_type.value == ScanType.RECURSIVE.value)):
            testcase.client_machine.generate_test_data(
                testcase.incr_file_rename,
                dirs=dirs,
                files=files,
                file_size=file_size,
                levels=levels,
                hlinks=hlinks,
                slinks=slinks,
                sparse=sparse,
                sparse_hole_size=sparse_hole_size,
                acls=data_options['acls'],
                unicode=data_options['unicode'],
                xattr=data_options['xattr'],
                long_path=data_options['long_path'],
                problematic=data_options['problematic'],
                options=options,
                username=username,
                password=password)
        testcase.client_machine.generate_test_data(
            testcase.incr_dir_rename,
            dirs=dirs,
            files=files,
            file_size=file_size,
            levels=levels,
            hlinks=hlinks,
            slinks=slinks,
            sparse=sparse,
            sparse_hole_size=sparse_hole_size,
            acls=data_options['acls'],
            unicode=data_options['unicode'],
            xattr=data_options['xattr'],
            long_path=data_options['long_path'],
            problematic=data_options['problematic'],
            options=options,
            username=username,
            password=password)
        testcase.client_machine.generate_test_data(
            testcase.incr_delete_data,
            dirs=dirs,
            files=files,
            file_size=file_size,
            levels=levels,
            hlinks=hlinks,
            slinks=slinks,
            sparse=sparse,
            sparse_hole_size=sparse_hole_size,
            acls=data_options['acls'],
            unicode=data_options['unicode'],
            xattr=data_options['xattr'],
            long_path=data_options['long_path'],
            problematic=data_options['problematic'],
            options=options,
            username=username,
            password=password)

        # We shouldn't add soft links for below modifications
        # as the soft links won't change
        change_data_options = deepcopy(data_options)
        change_data_options['slinks'] = slinks
        # ADDING WINDOWS AS WELL
        if (
                testcase.applicable_os == 'UNIX' or
                testcase.applicable_os == 'WINDOWS' or
                testcase.client_machine.os_flavour == 'OS400' or
                testcase.client_machine.os_flavour == 'OpenVMS'):
            change_data_options['slinks'] = False
        testcase.client_machine.generate_test_data(
            testcase.incr_modify,
            dirs=dirs,
            files=files,
            file_size=file_size,
            levels=levels,
            hlinks=hlinks,
            sparse=sparse,
            sparse_hole_size=sparse_hole_size,
            slinks=change_data_options['slinks'],
            acls=change_data_options['acls'],
            unicode=change_data_options['unicode'],
            xattr=change_data_options['xattr'],
            long_path=data_options['long_path'],
            problematic=data_options['problematic'],
            options=options,
            username=username,
            password=password)
        if testcase.incr_permissions_change is not None:
            testcase.client_machine.generate_test_data(
                testcase.incr_permissions_change,
                dirs=dirs,
                files=files,
                file_size=file_size,
                levels=levels,
                hlinks=hlinks,
                sparse=sparse,
                sparse_hole_size=sparse_hole_size,
                slinks=change_data_options['slinks'],
                acls=change_data_options['acls'],
                unicode=change_data_options['unicode'],
                xattr=change_data_options['xattr'],
                long_path=data_options['long_path'],
                problematic=data_options['problematic'],
                options=options,
                **kwargs)
        if testcase.incr_hlinks_change is not None:
            testcase.client_machine.generate_test_data(
                testcase.incr_hlinks_change,
                dirs=dirs,
                files=files,
                file_size=file_size,
                levels=levels,
                hlinks=hlinks,
                sparse=sparse,
                sparse_hole_size=sparse_hole_size,
                slinks=change_data_options['slinks'],
                acls=change_data_options['acls'],
                unicode=change_data_options['unicode'],
                xattr=change_data_options['xattr'],
                long_path=data_options['long_path'],
                problematic=data_options['problematic'],
                options=options,
                username=username,
                password=password)
        if testcase.incr_acl_change is not None:
            testcase.client_machine.generate_test_data(
                testcase.incr_acl_change,
                dirs=dirs,
                files=files,
                file_size=file_size,
                levels=levels,
                hlinks=hlinks,
                sparse=sparse,
                sparse_hole_size=sparse_hole_size,
                slinks=change_data_options['slinks'],
                acls=change_data_options['acls'],
                unicode=change_data_options['unicode'],
                xattr=change_data_options['xattr'],
                long_path=data_options['long_path'],
                problematic=data_options['problematic'],
                options=options,
                username=username,
                password=password)
        if testcase.incr_xattr_change is not None:
            testcase.client_machine.generate_test_data(
                testcase.incr_xattr_change,
                dirs=dirs,
                files=files,
                file_size=file_size,
                levels=levels,
                hlinks=hlinks,
                sparse=sparse,
                sparse_hole_size=sparse_hole_size,
                slinks=change_data_options['slinks'],
                acls=change_data_options['acls'],
                unicode=change_data_options['unicode'],
                xattr=change_data_options['xattr'],
                long_path=data_options['long_path'],
                problematic=data_options['problematic'],
                options=options,
                username=username,
                password=password)

        # wait for for journals to get flushed
        if testcase.should_wait:
            log.info("Waiting for journals to get flushed")
            sleep(testcase.WAIT_TIME)

    def mod_data_incr(self, scan_type=ScanType.RECURSIVE):
        """ Modifies the data added by add_new_data_incr
            call add_new_data_incr before calling this.

            Args:
                scan_type(ScanType(Enum))   -- scan type(RECURSIVE/OPTIMIZED/CHANGEJOURNAL)
                    default: ScanType.RECURSIVE

            Returns:
                None

            Raises:
                Exception - Any error occurred while modifying the data.

        """
        log = logger.get_log()
        testcase = self.testcase
        log.info("Deleting path: %s", testcase.incr_delete_data)
        testcase.client_machine.remove_directory(testcase.incr_delete_data)
        log.info("Renaming directory: %s", testcase.incr_dir_rename)
        if ((testcase.cluster_type is not None) and (testcase.cluster_type == DistributedClusterType.HADOOP.value)):
            testcase.client_machine.modify_test_data(testcase.incr_dir_rename, rename_dir=True)
        else:
            testcase.client_machine.rename_file_or_folder(
                testcase.incr_dir_rename, testcase.incr_dir_rename + "_renamed")
        log.info("Modifying files on path: %s", testcase.incr_modify)
        testcase.client_machine.modify_test_data(
            testcase.incr_modify, modify=True)
        if testcase.file_rename:
            log.info("Renaming files on path:%s", testcase.incr_file_rename)
            testcase.client_machine.modify_test_data(testcase.incr_file_rename, rename=True)
        if testcase.incr_permissions_change is not None:
            log.info("Changing permissions of files on path: %s",
                     testcase.incr_permissions_change)
            testcase.client_machine.modify_test_data(
                testcase.incr_permissions_change, permissions=True)
        if testcase.incr_hlinks_change is not None:
            log.info("Adding hardlinks to files on path: %s",
                     testcase.incr_hlinks_change)
            testcase.client_machine.modify_test_data(
                testcase.incr_hlinks_change, hlinks=True)
        if testcase.incr_acl_change is not None:
            log.info("Changing acls of files on path: %s",
                     testcase.incr_acl_change)
            testcase.client_machine.modify_test_data(
                testcase.incr_acl_change, acls=True)
        if testcase.incr_xattr_change is not None:
            log.info("Changing xattrs of files on path: %s",
                     testcase.incr_xattr_change)
            testcase.client_machine.modify_test_data(
                testcase.incr_xattr_change, xattr=True)

        # wait for for journals to get flushed
        if testcase.should_wait:
            log.info("Waiting for journals to get flushed")
            sleep(testcase.WAIT_TIME)

    def check_pseudo_client(self, pseudo_client_name):
        """ Check if pseudo-client is created on CS for a usercentric
            laptop client and check if it belongs to
            'Usercentric clients' client group.

            Args:
                pseudo_client_name  (str) - Name of the pseudo-client

            Raises:
                Exception - If pseudo-client doesn't exist
                            If pseudo-client does not belong to "Usercentric clients" client group

        """
        commcell_obj = self.testcase.commcell
        self.pseudo_client_name = pseudo_client_name
        commcell_obj.clients.refresh()  # refreshing the clients associated with the commcell
        if commcell_obj.clients.has_client(self.pseudo_client_name):
            self.log.info("Pseudo-client with name %s exists on CS", self.pseudo_client_name)
            client_group_obj = commcell_obj.client_groups.get("Usercentric clients")
            if client_group_obj:
                associated_clients_list = [associated_client.lower() for associated_client in
                                           client_group_obj.associated_clients]
                if self.pseudo_client_name.lower() in associated_clients_list:
                    self.log.info("Pseudo-client is part of 'Usercentric clients' group")
                else:
                    raise Exception(
                        "Pseudo-client is not a part of 'Usercentric clients' client group"
                    )
            else:
                raise Exception("Failed to create client group object")
        else:
            raise Exception(
                "Pseudo-client with name {0} does not exist on CS".format(self.pseudo_client_name))

    def check_scheduled_backup(self):
        """ To check if, for a user-centric laptop client, scheduled
            backup was launched on the pseudo-client and
            whether it was successful

            Raises:
                Exception - If retreiving Job ID fails
                            If first backup job was not scheduled
                            If the scheduled job did not complete successfully

        """
        commcell_obj = self.testcase.commcell
        job_controller_obj = commcell_obj.job_controller
        if job_controller_obj:
            first_job = None
            all_jobs_dict = job_controller_obj.all_jobs(
                client_name=self.pseudo_client_name, lookup_time=2, job_filter='Backup'
            )
            if all_jobs_dict:
                first_job = job_controller_obj.get(list(all_jobs_dict.keys())[-1])
                self.testcase.first_job = first_job
                self.log.info(
                    "Job ID of first job on client %s is %s",
                    self.pseudo_client_name,
                    first_job.job_id)
                if first_job.wait_for_completion():
                    self.log.info("Backup job finished successfully")
                    job_details = first_job.details
                    details = (
                        str(job_details['jobDetail']['generalInfo']['jobStartedFrom'])
                    ).lower()
                    if details == 'scheduled':
                        self.log.info("Backup job was scheduled")
                    else:
                        raise Exception("Backup job was not scheduled")
                else:
                    raise Exception("Backup job failed to finish successfully")
            else:
                raise Exception("Failed to  retreive jobs")
        else:
            raise Exception("Failed to create job controller object")

    def pseudo_client_cleanup(self, pseudo_client_name):
        """ To clean up pseudo-client from CS for
            usercentric laptop client.

            Raises:
                Exception - if pseudo-client could not be removed

        """
        commcell_obj = self.testcase.commcell
        commcell_obj.refresh()
        if commcell_obj.clients.has_client(pseudo_client_name):
            job_controller_obj = commcell_obj.job_controller
            if job_controller_obj:
                active_jobs_dict = job_controller_obj.active_jobs(
                    client_name=pseudo_client_name, lookup_time=6
                )
                if active_jobs_dict:
                    for key in active_jobs_dict:
                        job_obj = job_controller_obj.get(key)
                        if job_obj:
                            job_obj.kill(wait_for_job_to_kill=True)
                    self.log.info("Jobs running on pseudo-client killed")
            commcell_obj.clients.delete(pseudo_client_name)
            self.log.info("Pseudo-client cleaned up")

    def get_subclient_content(self, test_path, slash_format, subclient_name):
        """Generates the subclient content list.

        Args:
            test_path (str) : The testpath provided for the test case

            slash_format (str) : The slash format for the platform

            subclient_name (str) : The name of the subcllient for which data is generated.

        Returns:
            list : list of subclient contents for this test case

        """
        temp = list()
        content = self.testcase.client_machine.join_path(test_path, subclient_name)
        temp.append(content)
        return temp

    def verify_scan_marking(self, job_id):
        """"Verifies whether the job ran onle Scan phase i.e., Scan Marking Job
        Args :
            jobId   -- to verify the given job ran only Scan phase

        Return:
            True        -- if ran only Scan phases and the job is in Compeleted State

            False       -- if didn't ran Scan Marking Job

        """
        # query to get the number of phases ran
        str_query = (" select count(distinct(phase)) from JMBkpAtmptStats"
                     " with (NOLOCK) where status=1 and jobID={0}".format(job_id))
        self.testcase.csdb.execute(str_query)
        cur = self.testcase.csdb.fetch_one_row()
        # if count is one it means only one phase was ran else multiple phases were ran
        if cur[0] == '1':
            # query to get which phase was ran
            str_query = (" select distinct(phase) from JMBkpAtmptStats"
                         " with (NOLOCK) where status=1 and jobID={0}".format(job_id))
            self.testcase.csdb.execute(str_query)
            cur = self.testcase.csdb.fetch_one_row()
            if cur[0] == '4':  # 4 is number for Scan phase
                return True
        return False

    def backup_copy(self):
        """ Runs Offline backup copy for the given subclient
        """

        job = self.testcase.subclient.run_backup_copy()
        self.log.info("Backup copy workflow job id is : %s", job.job_id)
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run  Workflow job {0} for backup copy with error: {1}".format(
                    job.job_id, job.delay_reason)
            )
        self.log.info("Successfully finished backup copy workflow Job :%s", job.job_id)

        if job.status != 'Completed':
            raise Exception(
                "job: {0} for Backup copy operation is completed with errors, Reason: {1}".format(
                    job.job_id, job.delay_reason)
            )

    def volume_level_restore(
            self,
            source_path,
            destination_path,
            client_name,
            job=None,
            cleanup=True,
            from_time=None,
            to_time=None):
        """ Function to perform volume level restore

            Args:
                source_path(str)       -- Source Path for the restore

                destination_path(str)  -- Destination Path you want to restore

                client_name(str)       -- On client which you want to perform Volume level Restore


                from_time (str)        -- From time for Restore

                    default : None

                to_time (str)          -- to time for Restore

                    default : None

            Returns:
                None

            Raises:
                Exception - Any error occurred while doing volume level restore.

        """
        testcase = self.testcase
        log = logger.get_log()
        client_machine = testcase.client_machine

        paths, properties = self.testcase.subclient.browse(browse_view_name='VOLUMEVIEW')

        log.info("browse paths Recieved  %s", paths)
        log.info("browse paths Recieved  %s", properties)
        (vlrdrive, _) = os.path.split(destination_path)
        log.info("Drive used to restore volume level %s", vlrdrive)

        self.restore_out_of_place(
            client=client_name,
            destination_path=vlrdrive,
            paths=paths,
            from_time=from_time,
            to_time=to_time,
            is_vlr_restore=True
        )
        log.info(" Volume level Restore Done")

        log.info("Comparing source:%s destination:%s", source_path, destination_path)

        result, diff_output = client_machine.compare_meta_data(
            source_path, destination_path
        )
        if result:
            log.info("Meta data comparison successful")
        else:
            log.error("Meta data comparison failed")
            log.info("Diff output: \n %s", str(diff_output))
            raise Exception("Meta data comparison failed")

        result, diff_output = client_machine.compare_checksum(
            source_path, destination_path
        )
        if result:
            log.info("Checksum comparison successful")
        else:
            log.error("Checksum comparison failed")
            log.info("Diff output: \n %s", str(diff_output))
            raise Exception("Checksum comparison failed")

        if testcase.applicable_os == 'UNIX':
            if testcase.acls:
                result, diff_output = client_machine.compare_acl(
                    source_path,
                    destination_path
                )
                if result:
                    log.info("ACL comparison successful")
                else:
                    log.error("ACL comparison failed")
                    log.info("Diff output: \n%s", str(diff_output))
                    raise Exception("ACL comparison failed")
            if testcase.xattr:
                result, diff_output = client_machine.compare_xattr(
                    source_path,
                    destination_path
                )
                if result:
                    log.info("XATTR comparison successful")
                else:
                    log.error("XATTR comparison failed")
                    log.info("Diff output: \n%s", str(diff_output))
                    raise Exception("XATTR comparison failed")

        if cleanup:
            testcase.client_machine.remove_directory(destination_path)

    def get_backup_stream_count(self, job):
        """Returns the number of streams used by a backup job

            Args:
                job    (object)  -- job object of the backup job to be checked

            Returns:
                int   -   the number of streams used by a backup job

        """
        self.log.info("Getting the number of streams from job details")
        return int(job.details['jobDetail']['detailInfo']['numOfStreams'])

    def validate_trueup(self, job):
        """Given job object, it runs a query in JMMisc table and
            checks if itemtype 38 is present in query result.
            ItemType - 38 - Confirms if true up ran or not.

        Args:
            job (object)    -- instance of the Job class for the backup job to query.

        Returns:
            bool     -  Returns True if trueup Ran
                        Returns False if trueup didn't Run

        Raises:
            Exception:
                if any error occurred during SQL query execution.
        """
        trueup_ran_query = ("select count(*) from JMMisc WITH (NOLOCK) where jobid"
                            " = {0} and itemType = {1}".format(job.job_id, "38"))
        try:
            self._csdb.execute(trueup_ran_query)
            trueup_ran = self._csdb.rows
            if trueup_ran[0][0] == '1':
                return True
            return False
        except Exception as err:
            self.log.error(err)
            raise Exception(
                "Failed to execute Trueup SQL Query for "
                "validating trueup {0}".format(err)
            )

    def run_systemstate_backup(self, backup_type, wait_to_complete=None):
        """Checking if system state is enabled and running a backup
        Args:

            backup_type		     (str)				-- The type of backup job to be run.

            wait_to_complete     (bool)  			-- Specifies whether to wait until job finishes.

        Returns:
            list    - list of instances of the Job class for the backup jobs ran

            """
        self.update_subclient(backup_system_state=True)
        job = self.run_backup(backup_type, wait_to_complete=wait_to_complete)
        return job

    def run_onetouch_backup(self, backup_type):
        """Checking if onetouch option is enabled and running a backup"""
        self.update_subclient(backup_onetouch=True)
        self.run_backup(backup_type)

    def commit_job(self, job, threshold, commit_condition, timeout=300, pending=False):
        """Commits the backup job if the threshold condition is met or exceeded.

        Args:
            job                 (object)                --  Instance of Job object, the job that has to be committed.

            threshold           (int)                   --  Threshold value for the condition
            based on which job will be committed.

            commit_condition    (CommitCondition(Enum)) --  Commit condition can take one of the following values.
                -   FILES
                -   SECONDS

            timeout             (int)                   --  Specifies how long we must wait (in seconds)
            for the job to be committed before we give up.

                default:    300

            pending           (bool)                   --  Whether the job should be in pending state before commit.

                default:    False


        Example:
            job_1 = self.helper.run_backup(backup_level="Full", wait_to_complete=False)
            self.helper.commit_job(job_1[0], 3, CommitCondition.FILES)

        Returns:
           None

        Raises:
            Exception:
            Any error occurred while trying to commit the job.

        """

        try:
            # STEP 1 - WAIT UNTIL JOB REACHES BACKUP PHASE
            self.log.info("Waiting for backup phase of job %s to begin.", job.job_id)
            while True:
                if (job.phase).upper() == "BACKUP" and (job.status).upper() == "RUNNING":
                    break

            # STEP 2 - WAIT UNTIL COMMIT CONDITION IS MET
            if commit_condition == CommitCondition.FILES:
                self.log.info(
                    "Job will be committed after %s files have been transferred.",
                    threshold)
                while job.num_of_files_transferred <= threshold:
                    if job.num_of_files_transferred >= 0:
                        self.log.info("%s files transferred so far.", job.num_of_files_transferred)

            elif commit_condition == CommitCondition.SECONDS:
                self.log.info("Job will be committed in %s seconds.", threshold)
                sleep(threshold)

            if pending:
                self.testcase.client_machine.kill_process("clBackup")
                state = job.state
                while (state).upper() != "PENDING":
                    state = job.state
                    self.log.info("Job state is %s", (state).upper())

            # STEP 3 - TRY TO COMMIT THE JOB
            job.kill(wait_for_job_to_kill=False)
            start_time = time()
            state = job.state
            while (state).upper() != "COMMITTED":
                state = job.state
                if time() - start_time > timeout:
                    break

            # STEP 4 - LOG THAT WE'VE COMMITTED THE JOB
            if (state).upper() == "COMMITTED":
                self.log.info("Job %s was committed", job.job_id)
            else:
                excp = " Error occurred, job {} couldn't be committed, it's state is {}".format(
                    job.job_id, (state).upper()
                )
                raise Exception(excp)

        except Exception as excp:
            error_message = ("Error occurred when attempting to commit the job : {}").format(excp)
            raise Exception(error_message)

    def verify_collect_extent_acl(self, source_list, job_type, acl=1):
        """To verify the extent/acls only files
            Args:
                source_list (list): List of files from source data
                job_type  (str): Full/incremental
                acl (int): 1- extent/container check
                           2 - only acl check
                           3 - extent/container/acl check
                :return(boolean): True - Successful
                                  False - Failed
        """

        file_list = []
        # Get the job results directory
        collect_path = "{0}/{1}/{2}".format(self.testcase.client.job_results_directory,
                                            str(self.testcase.commcell.commcell_id),
                                            str(self.testcase.subclient.subclient_id))
        self.log.info(collect_path)

        # Check the collect files based on job type
        if (job_type == "FULL" and acl == 1):
            # Check for Extents and Container flag in CollectTot
            cmd = "cat  {0}/CollectTot* | grep %EXTENT%".format(collect_path)
            extent_output = self.testcase.client_machine.execute(cmd)
            cmd_container = "cat {0}/CollectTot* | grep %CONTAINER%".format(collect_path)
            container_output = self.testcase.client_machine.execute(cmd_container)
            extent_output_loop = extent_output.formatted_output
            container_loop = container_output.formatted_output
            self.log.info("Container output:")
            self.log.info(container_loop)

        elif (job_type == "INCREMENTAL" and acl == 3):
            # Check for Extents, Container flag and ContainerACL flag in CollectInc
            cmd = "cat  {0}/CollectInc* | grep %EXTENT%".format(collect_path)
            cmd_container = "cat {0}/CollectInc* | grep %CONTAINER%".format(collect_path)
            cmd_container_acl = "cat {0}/CollectInc* | grep %CONTAINERACL%".format(collect_path)

            extent_output = self.testcase.client_machine.execute(cmd)
            container_output = self.testcase.client_machine.execute(cmd_container)
            container_acl_output = self.testcase.client_machine.execute(cmd_container_acl)
            extent_output_loop = extent_output.formatted_output
            container_acl_output_loop = container_acl_output.formatted_output
            container_loop = container_output.formatted_output

        elif (job_type == "INCREMENTAL" and acl == 1):
            cmd = "cat  {0}/CollectInc* | grep %EXTENT%".format(collect_path)
            cmd_container = "cat {0}/CollectInc* | grep %CONTAINER%".format(collect_path)
            extent_output = self.testcase.client_machine.execute(cmd)
            container_output = self.testcase.client_machine.execute(cmd_container)
            extent_output_loop = extent_output.formatted_output
            container_loop = container_output.formatted_output
            self.log.info("Container output:")
            self.log.info(container_loop)

        else:
            # Check for ContainerACL flag in CollectInc
            cmd = "cat {0}/CollectInc* | grep %CONTAINERACL%".format(collect_path)
            container_acl_output = self.testcase.client_machine.execute(cmd)
            container_acl_output_loop = container_acl_output.formatted_output

        if acl == 1:
            container_acl_output_loop = []

        elif acl == 2:
            container_loop = []
            extent_output_loop = []

        # Maintain container only list and extent only list to compare flags are sent correctly
        container_file_list = []
        extent_file_list = []
        # Splitting the collect file items to get the path list for extents or ACLs
        for list1 in itertools.chain(
                extent_output_loop,
                container_acl_output_loop,
                container_loop):
            for innerlist in list1:
                subpaths = innerlist.split("/")
                self.log.info(subpaths)
                length = len(subpaths)
                if subpaths[0] == "%CONTAINERACL%":
                    lastword = subpaths[length - 1]
                    container_acl_file_split = lastword.split(":")
                    file_list.append("/" + "/".join(subpaths[1:len(subpaths) - 1])
                                     + "/" + str(container_acl_file_split[0]))
                    self.log.info(file_list)
                elif subpaths[0] == "%CONTAINER%":
                    lastword = subpaths[length - 1]
                    container_file_split = lastword.split(":")
                    container_file_list.append("/" + "/".join(subpaths[1:len(subpaths) - 1])
                                               + "/" + str(container_file_split[0]))
                    self.log.info(container_file_list)
                else:
                    file_list.append("/" + "/".join(subpaths[1:len(subpaths) - 1]))
                    extent_file_list.append("/" + "/".join(subpaths[1:len(subpaths) - 1]))
                    self.log.info(file_list)
                    self.log.info(extent_file_list)

        collect_list = list(set(file_list))
        container_cmp_list = list(set(container_file_list))
        extent_cmp_list = list(set(extent_file_list))
        if len(container_cmp_list) != 0:
            self.log.info("Comparing extent files against container list:")
            compare_result = self.compare_lists(extent_cmp_list, container_cmp_list, True)
            if not compare_result:
                self.log.info("Extent back up file may not have associated container")
            else:
                self.log.info("Containers are sent correctly for extent backed up files!")
        self.log.info("collect list:")
        self.log.info(collect_list)
        self.log.info("source list:")
        self.log.info(source_list)
        result = self.compare_lists(source_list, collect_list, True)

        if result:
            self.log.info("Collect file comparison is successful")
            return True
        else:
            self.log.info("Failed to compare the collect files")
            return False

    def parse_backup_restart_string(self, job):
        """Parses the restart map file and fetches the restart string from JM
        to prepare a dictionary of collect file ID to offset value.

        Args:
            job  (object)   --  Instance of Job object, the job for which a restart string has to be parsed.

        Returns:
           dict -   Returns a dictionary of collect file ID to offset value.

        Raises:
            None

        """
        # STEP 1 - FETCH THE RESTART STRING FOR THE JOB
        rst_string_query = "SELECT restartString FROM JMBkpAtmptStats WITH (NOLOCK) WHERE jobId={}".format(
            job.job_id)
        self._csdb.execute(rst_string_query)
        restart_string = self._csdb.rows[0]

        # STEP 2 - PARSE THE RESTART STRING
        cvf_offset_dict = dict()
        stream_restart_strings = str(restart_string).split("-o")
        if restart_string == "-1":
            cvf_offset_dict["1"] = "-1"  # NOTICED THIS TO BE THE RESTART STRING IN A CORNER CASE
            return cvf_offset_dict
        for stream_restart_string in stream_restart_strings:
            if stream_restart_string != '':
                cvf_offsets = str(stream_restart_string).split(
                    "@")  # GET RESTART STRING PER STREAM
                for cvf_offset in cvf_offsets:
                    cvf_offset = cvf_offset.strip()
                    match_ptn = re.match(
                        r"(?P<cvf_id>[0-9]*)\|(?P<offset>[0-9-]*)",
                        cvf_offset.strip("\n"))
                    if match_ptn is not None:
                        cvf_id = int(match_ptn.group('cvf_id'))
                        # IF OFFSET IN RESTART STRING IS -1, IT MEANS CVF IS COMPLETELY PROCESSED
                        # IF CVF IS COMPLETELY PROCESSED, CORRESPONDING CLIENT SIDE OFFSET IS -999
                        offset = -999 if int(match_ptn.group('offset')) == - \
                            1 else int(match_ptn.group('offset'))
                        cvf_offset_dict[cvf_id] = offset
        return cvf_offset_dict

    def create_expected_commit_files(self, committed_job):
        """Prepares the expected committed files for the committed backup job.
        These files will be compared with the actual collect files that are prepared by scan.

        Args:
            committed_job   (object)    --  Instance of Job, of the committed job.

        Returns:
           dict -   Returns a dictionary of lists containing entries in the commit collect and commit dirchange.

        Raises:
            None

        """
        machine_obj = self.testcase.client_machine

        # STEP 1 - PREPARE EXPECTED COMMIT COLLECT
        # STEP 1.1 - INITIALIZE BACKUP TYPE DEPENDENT CONSTANTS
        if committed_job.backup_level.upper() == "FULL":
            cvf_name = "NumColTot{}.cvf" if machine_obj.os_info == "WINDOWS" else "CollectTot{}.cvf"
            restart_map = "CollectMap_Tot.cvf" if machine_obj.os_info == "WINDOWS" else "restartMapTot"
            dc_name = "DCTot.cvf"

            # RESTART STRING PATTERN IS DIFFERENT ON UNIX COMPARED TO WINDOWS
            if machine_obj.os_info == "WINDOWS":
                pattern = re.compile(r"(.*)NumColTot(?P<cvf_id>[0-9]*)\.cvf\\\/\\\/(?P<offset>.*)")
            elif machine_obj.os_info == "UNIX":
                pattern = re.compile(r"(?P<cvf_id>[0-9]+)\s*(?P<offset>[0-9]+)")

        elif committed_job.backup_level.upper() in ("INCREMENTAL", "DIFFERENTIAL"):
            cvf_name = "NumColInc{}.cvf" if machine_obj.os_info == "WINDOWS" else "CollectInc{}.cvf"
            restart_map = "CollectMap_lnc.cvf" if machine_obj.os_info == "WINDOWS" else "restartMapInc"
            dc_name = "DCInc.cvf"

            if machine_obj.os_info == "WINDOWS":
                pattern = re.compile(r"(.*)NumColInc(?P<cvf_id>[0-9]*)\.cvf\\\/\\\/(?P<offset>.*)")
            elif machine_obj.os_info == "UNIX":
                pattern = re.compile(r"(?P<cvf_id>[0-9]+)\s*(?P<offset>[0-9]+)")

        split_char = "\r\n" if machine_obj.os_info == "WINDOWS" else "\n"
        # FETCHING AND STORING VALUE OF self.subclient_job_results_directory  IN
        # sc_jr FOR FURTHER USAGE
        sc_jr = self.subclient_job_results_directory

        dc_path = machine_obj.join_path(sc_jr, dc_name)
        cvf_path = machine_obj.join_path(sc_jr, cvf_name)
        restart_map_file_path = machine_obj.join_path(sc_jr, restart_map)

        # STEP 1.2 - PARSE THE RESTART MAP FILE AND GET THE COLLECT FILE TO OFFSET VALUES
        cvf_offset_dict = dict()
        output = machine_obj.read_file(restart_map_file_path).split(split_char)
        for line in output:
            match_ptn = pattern.match(line.strip("\n"))
            if match_ptn is not None:
                cvf_offset_dict[int(match_ptn.group('cvf_id'))] = int(match_ptn.group('offset'))

        # STEP 1.3 - PARSE RESTART STRING AND UPDATE DICTIONARY WITH OFFSET VALUES
        cvf_offset_dict.update(self.parse_backup_restart_string(committed_job))
        self.log.info(" cvf_offset_dict = %s", cvf_offset_dict)

        # STEP 1.4 - PREPARE THE COMMIT COLLECT FILE LIST
        commit_cvf = list()
        for cvf_id, offset in cvf_offset_dict.items():
            if offset == -999:  # CVF IS COMPLETELY PROCESSED
                continue
            elif offset == -1:  # CVF IS YET TO BE PROCESSED
                offset = 0
            entries = machine_obj.read_file(
                cvf_path.format(cvf_id),
                offset=offset).split(split_char)
            commit_cvf.extend(entries)

        # STEP 2 - PREPARE EXPECTED COMMIT DIRCHANGE
        commit_dc = machine_obj.read_file(dc_path).split(split_char)

        expected_commit_files = {'cvf': commit_cvf, 'dc': commit_dc}
        return expected_commit_files

    def get_actual_commit_files(self, committed_job):
        """Retrieves the actual commit files generated by scan.
        These files will be compared with the expected collect files that we created based on the restart string.

        Args:
            committed_job   (object)    --  Instance of Job, of the committed job.

        Returns:
           dict -   Returns a dictionary of lists containing entries in the commit collect and commit dirchange.

        Raises:
            None

        """
        machine_obj = self.testcase.client_machine

        commit_cvf_name = "".join(("Commit_J", committed_job.job_id, "_Collect.cvf"))
        commit_dc_name = "".join(("Commit_J", committed_job.job_id, "_DirChange.cvf"))

        split_char = "\r\n" if machine_obj.os_info == "WINDOWS" else "\n"
        # FETCHING AND STORING VALUE OF self.subclient_job_results_directory  IN
        # sc_jr FOR FURTHER USAGE
        sc_jr = list(self.subclient_job_results_directory.values())[0]

        # STEP 1 - GET ACTUAL COMMIT COLLECT FILE
        commit_cvf_file_path = machine_obj.join_path(sc_jr, commit_cvf_name)
        commit_cvf = machine_obj.read_file(commit_cvf_file_path).split(split_char)

        # STEP 2 - GET ACTUAL COMMIT DIR CHANGE
        commit_dc_file_path = machine_obj.join_path(sc_jr, commit_dc_name)
        commit_dc = machine_obj.read_file(commit_dc_file_path).split(split_char)

        actual_commit_files = {'cvf': commit_cvf, 'dc': commit_dc}
        return actual_commit_files

    def validate_commit_files(self, expected_commit_files, actual_commit_files, only_cvf=False):
        """Function checks whether the commit collect file and commit dirchange file were created correctly.

        Args:
            expected_commit_files   (dict)  --  Dictionary of commit files created by create_expected_commit_files.

            actual_commit_files     (dict)  --  Dictionary of commit files created by get_actual_commit_files.

            only_cvf                (bool)  --  If true, skips validation of the commit dirchange.

        Returns:
            bool    -   Returns True if the lists are identical.
            Returns False if the contents of the lists are different.

        Raises:
            None

        """

        # REMOVING ALL EMPTY ELEMENTS FROM LIST PRIOR TO COMPARISON
        ignore_items = ('', '/', '\\')

        expected_cvf_list = [
            item for item in expected_commit_files['cvf'] if item not in ignore_items]
        actual_cvf_list = [item for item in actual_commit_files['cvf'] if item not in ignore_items]

        expected_dc_list = [
            item for item in expected_commit_files['dc'] if item not in ignore_items]
        actual_dc_list = [item for item in actual_commit_files['dc'] if item not in ignore_items]

        if self.compare_lists(expected_cvf_list, actual_cvf_list):
            self.log.info("Commit collect file passed validation.")
        else:
            self.log.info("Commit collect file failed validation.")
            return False

        if not only_cvf:
            if self.compare_lists(expected_dc_list, actual_dc_list):
                self.log.info("Commit dirchange passed validation.")
            else:
                self.log.info("Commit dirchange failed validation.")
                return False
        else:
            self.log.info("Skipping commit dirchange validation.")

        return True

    def _update_ref_time_dict(self, job):
        """Updates the reference time dictionary self.job_ref_time_dict.

        Args:
            job             (object)    --  Instance of backup Job.

        Returns:
            None

        Raises:
            None

        """

        query = "SELECT attrVal FROM APP_SubclientProp WITH (NOLOCK) WHERE attrName = '{}' AND componentNameId={}"
        self._csdb.execute(query.format('sys:last full', self.testcase.subclient.subclient_id))
        last_full = self._csdb.rows[0][0]

        self._csdb.execute(query.format('sys:last incr', self.testcase.subclient.subclient_id))
        last_incr = self._csdb.rows[0][0]

        if job.backup_level.upper() == "INCREMENTAL":
            self.job_ref_time_dict = {
                job.job_id: self.job_ref_time_dict.get(
                    'next_incr_ref_time', -1)}
            self.job_ref_time_dict = {'next_incr_ref_time': last_incr}

        elif job.backup_level.upper() == "FULL":
            self.job_ref_time_dict = {job.job_id: 0}
            self.job_ref_time_dict = {'next_incr_ref_time': last_full}

        else:
            self.job_ref_time_dict = {job.job_id: last_full}
            self.job_ref_time_dict = {'next_incr_ref_time': last_incr}

        if last_full > last_incr:
            self.job_ref_time_dict = {'next_incr_ref_time': last_full}
        else:
            self.job_ref_time_dict = {'next_incr_ref_time': last_incr}

    def extent_level_validation(self, job, cvf_validation=False, **kwargs):
        """Attempts to check whether large files were backed up as extents.
        It also, optionally, validates the extent eligible collect file.

        Args:
            job             (object)    --  Instance of backup Job.

            cvf_validation  (bool)      --  If true, validates entries in the extent eligible collect file.

            \*\*kwargs  (dict)              --  Optional arguments.

            Available kwargs Options:

                    node    (str) --  Name of the node/client whose job results directory should be used for validation.

        Returns:
            bool    -   Returns True if the lists are identical.
            Returns False if the contents of the lists are different.

        Raises:
            None

        """

        def get_query_op(col, table, cond):
            self._csdb.execute(f"SELECT {col} FROM {table} WITH (NOLOCK) WHERE {cond}")
            if self._csdb.rows[0][0] == "":
                return 0  # 0 FILES HAVE BEEN TRANSFERRED SO FAR
            return int(self._csdb.rows[0][0])

        def get_ref_time_and_patterns():
            if job.backup_level.upper() == "FULL":
                if self.testcase.data_access_nodes:
                    return 0, "FileExtentNumColTot*", re.compile(r"(?!.*\\\d{8}\|.*)[*?][*?][*?](?P<file_path>.*?)\|(?P<flags>.*)")
                return 0, "*ExtentEligibleNumColTot*", re.compile(r"[*?][*?][*?](?P<file_path>.*?)(?P<flags>\|.*)")
            if self.testcase.data_access_nodes:
                return int(self.job_ref_time_dict[job.job_id]), "FileExtentNumColInc*", re.compile(r"(?!.*\\\d{8}\|.*)[*?][*?][*?](?P<file_path>.*?)\|(?P<flags>.*)")
            return int(self.job_ref_time_dict[job.job_id]), "ExtentEligibleNumColInc*", re.compile(r"[*?][*?][*?](?P<file_path>.*?)(?P<flags>\|.*)")

        # STEP 1 - GET NUMBER OF FILES TRANSFERRED AND NUMBER OF STREAMS USED BY THE JOB.
        num_files = get_query_op("totalNumOfFiles", "JMBkpStats", f"jobId ={job.job_id}")
        num_streams = get_query_op("COUNT(*)", "JMJobResourceHistory", f"jobId={job.job_id} AND phaseNum=7")
        result = False

        if num_files == 1 and num_streams > 1:
            self.log.info(f"{num_streams} streams used to transfer one file, it was backed up as extents")
            result = True
        else:
            self.log.info("Unable to perform validation for this job based merely on the number of streams used.")

        if cvf_validation:
            self.log.info("Collect file validation will be performed.")
            machine_obj = self.testcase.client_machine
            items_on_setup, items_in_collect, split_char = list(), list(), "\r\n"
            ref_time, ptrn, cvf_entry_ptrn = get_ref_time_and_patterns()

            # STEP 1 - GET LIST OF ALL FILES ON SETUP THAT SHOULD QUALIFY FOR EXTENT LEVEL BACKUP
            list_of_items = machine_obj.get_test_data_info(self.testcase.content[0], custom_meta_list="'FilesOnly','Hidden','FullName','LastWriteTimeUtc','Size'").split(split_char)
            for item in list_of_items:
                if item not in ("", "\n"):
                    item_prop = item.rsplit(maxsplit=2)
                    path, lwt, size = item_prop[0], floor(float(item_prop[1])), int(item_prop[2])
                    items_on_setup.append(path) if size > self.testcase.threshold and lwt > ref_time else None

            # STEP 2 - GET LIST OF ALL ENTRIES FROM ALL FileExtentEligibleNumCol*.cvf OR ExtentEligibleNumCol*.cvf
            node = Machine(kwargs.get('node', list(self.subclient_job_results_directory.keys())[0]), self.testcase.commcell)
            sc_jr_path = self.subclient_job_results_directory[kwargs.get('node', node.machine_name)]
            cmd = f"(Get-ChildItem \"{sc_jr_path}\" -Filter \"{ptrn}\").FullName"
            extent_cvfs = [i for i in (node.execute_command(cmd)).output.split(split_char) if i != ""]

            # PROCESS EVERY SINGLE EXTENT ELIGIBLE COLLECT FILE
            for extent_cvf in extent_cvfs:
                for entry in (node.read_file(extent_cvf)).split(split_char):
                    match_ptrn = cvf_entry_ptrn.match(entry)
                    items_in_collect.append(match_ptrn.group('file_path').replace('UNC-NT_', '\\\\')) if match_ptrn else None

            items_in_collect = list(set(items_in_collect))

            if self.compare_lists(items_on_setup, items_in_collect, sort_list=True):
                self.log.info("Collect file validation was successful")
                result = True
            else:
                result = False
        else:
            self.log.info("Skipping extent collect file validation by default")

        return result

    def generate_testdata(self, file_extensions, path=None, no_of_files=5):
        """Creates files with the specified extensions

        Args:
            file_extensions   (list)    --  List of all the extensions
            path               (str)    --  Source path to create files on
            no.of files        (int)    --  File count to create in destination

        """
        if path is None:
            self.log.info("Populating test data")
            install_dir = self.testcase.client.install_directory
            content_path = self.testcase.client_machine.join_path(install_dir, "Test")
        else:
            content_path = path

        if self.testcase.client_machine.check_directory_exists(content_path):
            self.log.info("Deleting the existing folder")
            self.testcase.client_machine.remove_directory(content_path)

        self.testcase.client_machine.create_directory(content_path)
        for each_extension in file_extensions:
            for i in range(1, no_of_files):
                file_name = "{0}{1}{2}{3}".format(content_path, self.testcase.slash_format,
                                                  str(i), each_extension)
                data_to_file = str("My File Data {0}".format(file_name))
                self.testcase.client_machine.create_file(file_name, data_to_file)

    def rem_common_files(self, list_1, list_2):
        """ Function to remove common file with b in a
                Args:
                    list_1 (list) : Entity in which common files to be removed
                    list_2 (list) : Entity which is compared to get common files
                Returns:
                    list of files excluding common files
        """
        self.log.info("%s Removes common files b/w lists %s", "*" * 8, "*" * 8)
        for file in list_2:
            if file in list_1:
                idx = list_1.index(file)  # index of common_filename
                list_1.pop(idx)  # removing common_files
        return list_1

    def extract_filename(self, dir_paths):
        """ Function to get filenames from their path
                Args:
                    dir_paths (list) : Entity to access it's files
                Returns:
                    list of files names (base names)
        """
        self.log.info("%s Gets file name from path %s",
                      "*" * 8, "*" * 8)
        file_names = []
        for file in dir_paths:
            file_names.append(os.path.basename(file))
        return file_names

    def validate_backup(self, content_paths, restore_path, dest_client=None, add_exclusions=None,
                        rem_exclusions=None, exceptions_list=None):
        """ Vadilates whether content is backedup or not
                Args:
                    content_paths       (list(paths)): Folder path to which the data is compared
                    restore_path        (str): Destination path of restored files
                    dest_client         (str): Name of the destination client
                    add_exclusions      (list(paths)): Exclusions of a subclient
                    rem_exclusions      (list(paths)): Removed Exclusions of subclient
                    exceptions_list     (list(paths)): Exceptions of a subclient
                Raises:
                    Exception :
                        -- If exception files are not backedup
                        -- If removed Exclusion files are not backedup
                        -- The restored files doesn't match with backedup content
        """
        self.log.info("%s Validating the backup files %s", "*" * 8, "*" * 8)
        if dest_client is None:
            dest_client = Machine(self.testcase.client)
        else:
            dest_client = Machine(dest_client, self.testcase.commcell)

        if self.testcase.client_machine.os_info == "WINDOWS":
            concat = 2
        else:
            concat = 0
        self.log.info("%s Comparing the backup folders %s", "*" * 8, "*" * 8)
        file_paths = []
        flag = True
        for idx, key in enumerate(content_paths):
            path = restore_path + content_paths[idx][concat:]
            file = []
            if self.testcase.client_machine.is_directory(path):
                file = self.testcase.client_machine.compare_folders(dest_client,
                                                                    source_path=key,
                                                                    destination_path=path,
                                                                    algorithm="MD5")

            if self.testcase.client_machine.is_file(path):
                check_status = self.testcase.client_machine.compare_files(dest_client,
                                                                          source_file_path=key,
                                                                          destination_file_path=path,
                                                                          algorithm="MD5")
                if check_status:
                    file = []
                else:
                    file = [path]

            if len(file) > 0:
                flag = False
                break
            file_paths.append(file)
        files = [file for lst in file_paths for file in lst]
        files = self.extract_filename(files)

        if flag:
            files = [file for lst in file_paths for file in lst]
            files = self.extract_filename(files)

            if exceptions_list:
                exception_list = self.extract_filename(exceptions_list)
                for file in exception_list:
                    if file in files:
                        raise Exception("Exception files are not backed up")

            if rem_exclusions:
                rem_exclu_list = self.extract_filename(rem_exclusions)
                for file in rem_exclu_list:
                    if file in files:
                        raise Exception("Removed Exclusion files are not backed up")

            if add_exclusions:
                excls_n = add_exclusions[:]
                excls_n = self.rem_common_files(excls_n, exceptions_list)
                excls_n = self.extract_filename(excls_n)
                files = self.rem_common_files(files, excls_n)
                if len(files) > 0:
                    raise Exception("Added Exclusion files are backed up")

        if not flag:
            raise Exception("The restored files doesn't match with backed up content")

    def validate_download_files(self, backup_files, download_path, folder=False):
        """ Validates downloaded files
                Args:
                    backup_files (dictionary(file_name,content)): files to be validated
                                                                  (eg: {'test.txt': 'data is added'})
                    download_path (str): Destination path of restored files
                    folder (bool): True if the folder is included in the backup
        """
        dest_client = Machine()
        if self.testcase.client_machine.os_info == "WINDOWS":
            delimiter = "\\"
        else:
            delimiter = "/"
        if len(backup_files) > 1:
            path = os.path.dirname(download_path) + delimiter + \
                   os.path.splitext(os.path.basename(download_path))[0]
            if self.testcase.client_machine.check_directory_exists(path):
                self.testcase.client_machine.remove_directory(path)
            with ZipFile(download_path, "r") as zip_ref:
                zip_ref.extractall(path)
            file_path = dest_client.get_files_in_path(path)
        else:
            file_path = [download_path]
        flag = False
        file_names = list(backup_files.keys())
        if folder:
            for file in file_path:
                if '.' not in file:
                    files = dest_client.get_files_in_path(file)
                    file_path.append(files)
            file_path = [file for lst in file_path for file in lst]

        if len(file_path) == len(backup_files):
            flag = True
        if flag:
            for file in file_path:
                file_name = os.path.basename(file)
                if file_name in file_names:
                    content = dest_client.read_file(file_path=file).rstrip()
                    if content != backup_files[file_name]:
                        raise Exception("Files are not downloaded properly")
        else:
            raise Exception("few files are not downloaded")

    def validate_preview_file(self, log_directory, file_type):
        """Verifies the file is viewed successfully or not
                Args:
                    log_directory     (str): location of log directory
                                             on the cs machine
                    file_type         (str): Type of the file
                                             (eg: text, audio, video)
                Raises:
                    Exception :
                        -- File is not previewed successfully
        """
        cs_machine_obj = Machine(self.testcase.commcell.commserv_client)
        if '/' in log_directory:
            delimiter = '/'
        else:
            delimiter = '\\'
        log_file_path = log_directory + delimiter + "WebServer.log"
        flag = True

        if file_type.lower() == 'audio' or file_type.lower() == 'video':
            file_content = cs_machine_obj.read_file(
                file_path=log_file_path,
                search_term="GET / Stream / Recall: HTTP code 'OK'")
            if len(file_content) == 0:
                flag = False

        else:
            file_content = cs_machine_obj.read_file(
                file_path=log_file_path,
                search_term="/GetDocPreviewWithFields : HTTP code 'OK'")
            if len(file_content) == 0:
                flag = False

        if not flag:
            raise Exception("File is not previewed successfully")

    def validate_cross_machine_restore(self, content_paths, restore_path, dest_client=None):
        """ Vadilates whether content is backedup or not
                Args:
                    content_paths       (list(paths)): Folder path to which the data is compared
                    restore_path        (str): Destination path of restored files
                    dest_client         (str): Name of the destination client
                Raises:
                    Exception :
                        -- The restored files doesn't match with backedup content
        """
        self.log.info("%s Validating the backup files %s", "*" * 8, "*" * 8)
        content_files = []
        for path in content_paths:
            content_files.append(self.testcase.client_machine.get_files_in_path(path))
        content_files = [file for lst in content_files for file in lst]
        content_files = self.extract_filename(content_files)
        if dest_client is None:
            dest_client = Machine(self.testcase.client)
        else:
            dest_client = Machine(dest_client, self.testcase.commcell)
        dest_files = dest_client.get_files_in_path(restore_path)
        dest_files = self.extract_filename(dest_files)

        flag = True
        if len(content_files) == len(dest_files) and \
                len(set(content_files).intersection(dest_files)) == len(dest_files):
            flag = False
        if flag:
            raise Exception("The restored files doesn't match with backedup content")

    def validate_multiple_versions_of_file(self, restore_path, older_version_file_content,
                                           dest_client=None, version_no=1):
        """ Validates whether old version of file is correctly restored or not
                Args:
                    restore_path  (str)    --  Destination path of restored files.
                    older_version_file_content  (str)    --   The content of the older version
                                                              file.
                    dest_client   (str): Name of the destination client
                    version_no    (int): Compare with the version number file backed up
                Raises:
                    Exception :
                        -- If the old version file is not restored properly
        """
        self.log.info("%s Validating the restore file %s", "*" * 8, "*" * 8)
        if dest_client is None:
            dest_client = Machine(self.testcase.client)
        else:
            dest_client = Machine(dest_client, self.testcase.commcell)
        file_path = dest_client.get_files_in_path(restore_path)[version_no - 1]
        restore_file_content = dest_client.read_file(file_path)
        restore_file_content = restore_file_content.rstrip()
        if older_version_file_content != restore_file_content:
            raise Exception("The older version of file is not correctly restored")

    def validate_restore_for_selected_files(self, backup_files, restore_path,
                                            dest_client=None, hidden_files=False):
        """ Validates whether selected/deleted/hidden files are backedup or not
                Args:
                    backup_files   (list(filepaths))    --  List of select/delete/hidden
                                                          file paths.
                    restore_path  (str)    --  Destination path of restored files.
                    dest_client    (str) -- Name of the destination client
                    hidden_files   (bool) -- if selected files are hidden files True
                                             else False
                Raises:
                    Exception :
                        -- If the selected/deleted/hidden files are not restored
        """
        self.log.info("%s Validating the restore files %s", "*" * 8, "*" * 8)
        if dest_client is None:
            dest_client = Machine(self.testcase.client)
        else:
            dest_client = Machine(dest_client, self.testcase.commcell)

        if '\\' in backup_files[0]:
            delimiter = '\\'
        else:
            delimiter = '/'

        backup_files = self.extract_filename(backup_files)
        if hidden_files:
            for file in backup_files:
                dest_client.unhide_path(path=restore_path + delimiter + file)

        files = dest_client.get_files_in_path(restore_path)
        files = self.extract_filename(files)
        flag = True
        for file in backup_files:
            if file not in files:
                flag = False

        if not flag:
            raise Exception("The selected/deleted/hidden files are not correctly restored")

    def get_collect_file_list(self, collect_file_path, collect_file_level, match_string):
        """ Gets the list of collect files
                Args:
                    collect_file_path   (str)    --  Path of Collect files location.
                    collect_file_level  (str)    --  Level_type to collect files.
                    match_string         (str)    --  Determines os_type.

                Returns:
                    list    -   List of collect files in the specified path

                Raises:
                    None
        """
        files_list = self.testcase.client_machine.get_files_in_path(collect_file_path)
        res_files_list = []
        if collect_file_level.upper() == "FULL":
            string = match_string + "Tot"
            res_files_list = [file for file in files_list if string in file]

        elif collect_file_level.upper() == "INCR":
            string = match_string + "Inc"
            res_files_list = [file for file in files_list if string in file]
        return res_files_list

    def validate_collect_files(self, subclient_id, collect_file_level,
                               new_path=None, deleted_path=None):
        """" Validates the modified files/folders are backed up or not
                Args:
                    subclient_id    (str) :        --  To determine the subclient.
                    collect_file_level  (str) :    --  Level_type to collect files.
                    new_path   (list(paths)) :       --  List of new paths added
                                                       to the subclient content.
                    deleted_path   (list(paths)) :   --  List of paths deleted from
                                                       the subclient content.
                Raises:
                    Exception:
                        -- if new files are not backed up.
                        -- if deleted files are not marked as delete.
        """
        collect_file_list = []
        if self.testcase.client_machine.os_info == "WINDOWS":
            collect_file_path = self.testcase.client_machine.join_path(
                self.testcase.client.job_results_directory,
                r"CV_JobResults\iDataAgent\FileSystemAgent", "2",
                subclient_id)
            collect_file_list = self.get_collect_file_list(collect_file_path,
                                                           collect_file_level, 'NumCol')

        elif self.testcase.client_machine.os_info == "UNIX":
            collect_file_path = self.testcase.client_machine.join_path(
                self.testcase.client.job_results_directory, "2",
                subclient_id)
            collect_file_list = self.get_collect_file_list(collect_file_path,
                                                           collect_file_level, 'Collect')

        files_content = []
        for file in collect_file_list:
            files_content.append(self.testcase.client_machine.read_file(file))
        if new_path:
            new_list_files = []
            for idx, key in enumerate(new_path):
                if '.' in os.path.basename(key):
                    new_list_files.append([key])
                else:
                    path = self.testcase.client_machine.get_files_in_path(key)
                    new_list_files.append(path)
            new_list_files = [file for lst in new_list_files for file in lst]
            for file in new_list_files:
                flag = False
                for string in files_content:
                    if file.lower() in string.lower():
                        flag = True
                        break
                if not flag:
                    raise Exception("New files are not backed up")

        if deleted_path:
            content = []
            for string in files_content:
                idx = 0
                for i, key in enumerate(string):
                    if key == "\n":
                        content.append(string[idx:i])
                        idx = i + 1
            for path in deleted_path:
                flag = False
                for line in content:
                    if path in line and '??D' in line:
                        flag = True
                        break
                if not flag:
                    raise Exception("Deleted files are not marked")

    def validate_filters(self, win_filters, unix_filters):
        """Function to validate whether the filters set are being honoured or not

        Args:
            win_filters         (list)  --  windows filters list.

            unix_filters        (list)  --  unix filters list.


        Returns:
            bool    -   Returns True if the filters are being honoured correctly
                        Returns False if the filters are not being honoured

        Raises:
            None

        """
        try:
            filters = []
            if self.testcase.client_machine.os_info == "WINDOWS":

                collect_file_path = self.testcase.client_machine.join_path(
                    self.testcase.client.job_results_directory,
                    r"CV_JobResults\iDataAgent\FileSystemAgent", "2",
                    self.testcase.subclient.subclient_id,
                    "NumColTot1.cvf")

                filters = win_filters

            elif self.testcase.client_machine.os_info == "UNIX":

                collect_file_path = self.testcase.client_machine.join_path(
                    self.testcase.client.job_results_directory, "2",
                    self.testcase.subclient.subclient_id,
                    "CollectTot1.cvf")

                filters = unix_filters

            for filter in filters:
                if filter.replace(
                        '*', '') in self.testcase.client_machine.read_file(collect_file_path):
                    self.log.error("The filter %s is not being honoured ", filter)
                    raise Exception("The filter %s is not being honoured")

                self.log.info("The filters %s is  honoured ", filter)

        except Exception as excp:
            raise Exception(str(excp))

    def validate_failed_files(self, source_list=[], job=None):
        """Function to validate failed files

        Args:
            source_list         (list)  --  list of source paths to be locked.

            Job                 (str)  --  Job type Failed or not .


        Raises:
            None

        """

        machine_obj = self.testcase.client_machine
        self.log.info(" Failures Collect file validation will be performed.")

        if self.testcase.client_machine.os_info == "WINDOWS":
            failures_file_path = self.testcase.client_machine.join_path(
                self.testcase.client.job_results_directory,
                r"CV_JobResults\iDataAgent\FileSystemAgent", "2",
                self.testcase.subclient.subclient_id,
                "Failures.cvf")

        for failed_item in source_list:
            if failed_item in self.testcase.client_machine.read_file(failures_file_path):
                if job == "Failed":
                    self.log.error("The Failed item  %s is failures.cvf ", failed_item)
                else:
                    raise Exception("The Failed item  %s is failures.cvf", failed_item)
            else:
                self.log.info("File %s is not in Failures collect ", failed_item)

    def get_snapshot_id(self, job):
        """Function to get eh snapshotID created for that job

        Args:
            Job         (obj)  --  Job object

        Returns:
            String    -   Returns the snapshot ID

        Raises:
            None

        """
        while True:
            if job.phase is not None and job.status is not None and (job.phase).upper() == "BACKUP" and (
            job.status).upper() == "RUNNING":
                job.pause(True)
                break
        query = "Select shadowCopyId from JMBkpJobInfo where jobId = {0}".format(
            job.job_id)
        self.testcase.csdb.execute(query)
        row = self.testcase.csdb.fetch_one_row()
        snapshot_id = row[0].split('|')[2][4:]
        job.resume(True)
        return snapshot_id

    def is_snapshot_deleted(self, snapshot_id):
        """Function to check if snapshot has been deleted post backup

        Args:
            snapshot_id         (String)  --  snapshot ID

        Raises:
            Exception if snapshot has not been deleted

        """
        output = self.testcase.client_machine.execute_command(r"vssadmin list shadows")
        if output.exit_code != 0:
            raise Exception("List shadows command failed to get executed")
        else:
            self.log.info("Got the shadow list succesfully")
        if snapshot_id not in output.output:
            self.log.info("The Snapshot %s has been deleted successfully ", snapshot_id)
        else:
            raise Exception("The Snapshot %s has not been deleted ", snapshot_id)

    def validate_pre_post_commands(self, commands_from_inputs):
        """Function to validate if pre_post_commands are set

        Args:
            commands_from_inputs         (dict)  --  Pre/Post commands set by user

        Raises:
            Exception if the commands set in the DB and commands specified by user are not same

        """
        commands_modified = {}
        if self.testcase.client_machine.os_info == "WINDOWS":
            for key in commands_from_inputs.keys():
                if commands_from_inputs[key] != '':
                    commands_modified[key] = commands_from_inputs[key].replace('/', '\\')
                else:
                    commands_modified[key] = commands_from_inputs[key]

        elif self.testcase.client_machine.os_info == "UNIX":
            for key in commands_from_inputs.keys():
                if commands_from_inputs[key] != '':
                    commands_modified[key] = '/' + commands_from_inputs[key]
                else:
                    commands_modified[key] = commands_from_inputs[key]

        commands_set = self.testcase.subclient.pre_post_commands
        if commands_modified == commands_set:
            self.log.info("The pre/post commands have been set correctly")
        else:
            self.log.error("The pre/post commands have not been set correctly")
            raise Exception("check if pre.post commands are getting set correctly or not")

    def validate_rfc_files(self, jobid, source_list, delete_rfc=False):
        """Function to validate Remote file cache with backup and restore

        Args:
            jobid                (str)  --  Job ID of backup  .

            source_list         (list)  --  list of RFC files to be validated .

            delete_rfc           (bool) --  True: Deletes RFC path
                                            Default :False

        Raises:
            Exception if file not available in media agent RFC paths
        """
        ma_machine = self.testcase.ma_machine
        self.log.info(" Failures Collect file validation will be performed.")
        self.indexpath = self.get_index_cache(ma_machine)
        subclient_guid = self.testcase.subclient.subclient_guid
        rfc_path = self.testcase.ma_machine.join_path(
            self.indexpath,
            r"RemoteFileCache", "2",
            subclient_guid,
            jobid)
        rfc_list = self.testcase.ma_machine.get_files_in_path(rfc_path)
        self.log.info(rfc_list)
        rfc_files = []
        for file in rfc_list:
            rfc_list_fol, rfc_list_file = os.path.split(file)
            rfc_files.append(rfc_list_file.lower())
        self.log.info(rfc_files)
        for rfc_item in source_list:
            if rfc_item.lower() in rfc_files:
                self.log.info("File %s Exists  in RFC Index Cache  ", rfc_item)
            else:
                regexp = [x for x in rfc_files if re.search(rfc_item, x)]
                if len(regexp) != 0:
                    self.log.info("File %s Exists  in RFC Index Cache  ", rfc_item)
                else:
                    self.log.info("File %s is not in RFC Index Cache  ", rfc_item)
                    raise Exception("File %s is not in RFC Index Cache  ", rfc_item)
        self.log.info("All files validated in RFC")
        self.log.info("Validate RFC Arch file in DB ")
        self._csdb.execute(
            """
            select count(*) from archfile where jobId={0} and  name ='RFC_AFILE'            
            """.format(jobid))
        self.log.info("No.of RFC Arch Files Created Per job  %s", self._csdb.fetch_one_row()[0])
        if (int(self._csdb.fetch_one_row()[0])) >= 1:
            self.log.info("RFC Arch file got created with backup job %s ", str(jobid))
        else:
            self.log.info("RFC Arch file failed to  create with backup%s ", str(jobid))
            raise Exception("RFC Arch file failed to  create with backup job %s ", str(jobid))

        if delete_rfc:
            self.log.info("Deleting  RFC Folder for JOB ID %s", rfc_path)
            self.testcase.ma_machine.remove_directory(rfc_path)
            self.log.info("RFC Folder %s deleted ", rfc_path)

    def get_index_cache(self, media_agent=None):
        """Gets the index cache for the given media agent

            Args:
                media_agent     (obj)   --      The media agent CvPySDK client object

            Returns:
                (str)       -       The index cache path of the media agent

        """
        if media_agent is None:
            media_agent = self.testcase.client_machine
            ma_id = media_agent.client_object.client_id
        else:
            ma_id = media_agent.client_object.client_id

        self._csdb.execute(
            """
                select attrVal from app_clientprop where
                componentNameId = {0} and
                attrName = 'Idx: cache path'
            """.format(ma_id)
        )

        if not self._csdb.rows:
            raise Exception('Unable to get IndexCache directory for the MA [{0}]'.format(
                media_agent.client_name
            ))

        return self._csdb.fetch_one_row()[0]

    def get_logs_for_job_from_file(self, job_id, log_file_name, search_term=None):
        """From a log file object only return those log lines for a particular job ID.

        Args:
            job_id          (str)   --  Job ID for which log lines need to be fetched.

            log_file_name   (bool)  --  Name of the log file.

            search_term     (str)   --  Only capture those log lines containing the search term.

        Returns:
            str     -   \\r\\n separated string containing the requested log lines.

            None    -   If no log lines were found for the given job ID or containing the given search term.

        Raises:
            None

        """

        # GET ONLY LOG LINES FOR A PARTICULAR JOB ID
        raise NotImplementedError('Method Not Implemented by the Child Class')

    def get_cluster_nodes(self, cluster_name=None):

        """Gets the list of cluster nodes

            Args:
                cluster_name     (str)   --      cluster name

            Returns:
                nodes           (list)    --       list of cluster nodes

        """

        if cluster_name is None:
            cmd = ("Get-ClusterNode -Cluster {0}".format(self.testcase.client_name))
            output = self.testcase.client_machine.execute_command(cmd)
        else:
            cmd = ("Get-ClusterNode -Cluster {0}".format(cluster_name))
            output = cluster_name.execute_command(cmd)
        if output.exit_code != 0:
            raise Exception("Get Cluster Node call failed  ")
        else:
            self.log.info("Get Cluster Node call successful ")
        nodes = []
        for element in range(len(output.formatted_output)):
            nodes.append((output.formatted_output[element][0]).lower())
        return nodes

    def active_cluster_node(self, cluster_name):

        """Gets the active cluster node.

            Args:
                cluster_name     (str)   --      cluster name

            Returns:
                active_node      (str)  --      active cluster node

        """
        cmd = ("Get-WMIObject Win32_ComputerSystem -ComputerName {0}".format(cluster_name))
        output = self.testcase.client_machine.execute_command(cmd)
        if output.exit_code != 0:
            raise Exception("Get Active cluster Node call failed  ")
        else:
            self.log.info("Get Active cluster Node call successful ")
        temp1 = output.formatted_output.split("\r\n")
        temp2 = temp1[3].split(":")
        active_node = temp2[1].strip().lower()

        return active_node

    def do_failover(self, cluster_name, failover_node):

        """Gets the list of cluster nodes

            Args:
                cluster_name     (str)   --      cluster name

                failover_node    (str)   --      node to make as active node

        """
        cmd = ("Move-ClusterGroup -Name {0} -Node {1} -Wait 0".format(cluster_name, failover_node))
        self.log.info("Running failover cmd {0}".format(cmd))
        output = self.testcase.client_machine.execute_command(cmd)
        if output.exit_code != 0:
            raise Exception("failover failed to node {0}".format(failover_node))
        else:
            self.log.info("Running failover to node {0}".format(failover_node))

    @staticmethod
    def identify_master(job, all_machines):
        """
            Identifies Master node using clBackup.log of nodes
            Args:
                job (Job Instance) - job for which master node to be identified

                all_machines (dict) - dictionary of machines instacnes with their name as key

            Returns:
                node name/None - Node if master node found else None
        """
        for node in all_machines:
            temp_machine = all_machines[node]
            logs = temp_machine.get_logs_for_job_from_file(log_file_name="clBackup.log",
                                                           job_id=job.job_id,
                                                           search_term="DistributedIDA::CMaster")
            if logs:
                return node

        return None

    def verify_scan_type_used(self, job_id):
        """
            Function to verify job type used in the given job
            Args:
                job_id (str) - job_id of the job for which scan method needs to be verified

            Returns:
                scan_type (str) - if able to detect scan method from logs
                                else "" (empty string).
        """
        logs = self.testcase.client_machine.get_logs_for_job_from_file(job_id=job_id,
                                                                       log_file_name="FileScan.log",
                                                                       search_term="CumulativeScanModeAndReason")

        if logs:
            scan_type = re.search(r"CumulativeScanModeAndReason=[A-Z+]*", logs)
            if scan_type:
                return scan_type[0].split("=")[1]

        return ""
