import os
import json
import shutil
import sqlite3
import subprocess
import random
import time
from datetime import timedelta
import mimetypes

from datetime import datetime
from subprocess import Popen, PIPE
import win32security
import win32api
import win32file
import ntsecuritycon as con
from AutomationUtils import logger
from AutomationUtils.constants import AUTOMATION_BIN_PATH
from AutomationUtils.database_helper import CommServDatabase
import dynamicindex.utils.constants as cs


class ActivateUtils:
    def __init__(self):
        """
        Initializes ActivateUtils object
        """
        self.log = logger.get_log()

    @staticmethod
    def db_get_entities_dict_from_sqllite(sqllite_db_path):
        """
        Returns all the entities present in the db

            Args:
                sqllite_db_path -- If True, entities which are null or blank
                                        are removed

        """
        connection = sqlite3.connect(sqllite_db_path)
        connection.row_factory = sqlite3.Row
        cursor = connection.cursor()
        query = 'select * from entity'
        try:
            column_names = []
            return_list = []
            result = cursor.execute(query)
            for row in result.description:
                column_names.append(row[0])
            print(column_names)
            result = result.fetchall()
            for res in range(len(result)):
                temp_dict = {}
                for col in range(len(column_names)):
                    temp_dict.update({column_names[col]: result[res][col]})
                return_list.append(temp_dict)
            dict1 = {'result': return_list}
            return dict1

        except Exception as ex:
            print("Error {0}".format(ex))
        finally:
            connection.close()

    @staticmethod
    def get_workflow_interaction(commcell, workflow_id):
        """
        Get workflow interaction detail
        :param commcell: commcell object
        :param workflow_id: workflow job
        :return: interactionid
        """
        query = """select interactionGuid from WF_Interaction where jobId ={}""".format(
            workflow_id)
        csdb = CommServDatabase(commcell)
        csdb.execute(query)
        cur = csdb.fetch_one_row()
        value = cur.pop(0)
        if value is "":
            raise Exception("Could not find interaction details")
        return value

    def sensitive_data_generation(self, database_path, number_files=5):
        """
        Generate sensitive data with PII
        :param self:
        :param database_path: dump database path
        :param number_files: number of files to generate
        :return:
        """
        javahome = os.environ.get('JAVA_HOME')
        if javahome is None:
            print("Java_Home is not set")
            return 2
        filename = AUTOMATION_BIN_PATH
        self.log.info(filename)
        self.delete_old_data(database_path)
        # Form the command to run
        runcommand = r'"{0}\bin\java.exe" ' \
                     r'-jar "{1}\GDPR.jar" -NUM {2} -PATH {3}'.format(javahome, filename,
                                                                      number_files, database_path)
        self.log.info("Running [%s]." % (str(runcommand)))
        process = Popen(runcommand, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, shell=False,
                        cwd=filename)
        stdout, stderr = process.communicate()
        self.log.info(stderr)
        self.log.info(stdout)

    @staticmethod
    def delete_old_directory(_directory):
        """
        :param _directory: directory to be deleted
        """
        shutil.rmtree(_directory, ignore_errors=True)

    def delete_old_data(self, _directory):
        """
        :param _directory:
        """
        try:
            if len(os.listdir(_directory)) == 0:
                self.log.info("Directory [%s] is empty", _directory)
            else:
                self.log.info(
                    "Directory [%s] is not empty, deleting..", _directory)
                self.delete_old_directory(_directory)
        except Exception as exception:
            self.log.error(exception)

    def run_backup(self, subclient_obj):
        """
        Run Backup JOB
        Args:
            subclient_obj : Subclient Object To Run Backup
        """
        self.log.info(
            '*' * 5 + "Starting Backup JOB for Mentioned Subclient" + '*' * 5)
        backup_job_obj = subclient_obj.backup()
        if not backup_job_obj.wait_for_completion():
            raise Exception("Subclient Backup JOB Failed")
        self.log.info("Job Completed Successfull!!")

    @staticmethod
    def get_sensitive_content_details(data_source_type, entity_type,
                                      db_path, entity_separator):
        """
        Get Sensitive content with linked entity value for for creating
        GDPR (DELETE/EXPORT) request.
        Args:
            data_source_type (str): Type of Data Source
            entity_type (str): Entity Type to paired with sensitive file name
            entity_separator: Separator between Entities
            db_path: Path of DB to use
        Returns:
            Tuple containing sensitive (file_name, entity_value)
        """
        __entity = ""
        __file = ""
        entities_list = list()
        entities_list.append(entity_type)
        __sensitive_files = ActivateUtils.db_get_sensitive_columns_list(
            data_source_type,
            entities_list,
            db_path
        )
        for filepath in __sensitive_files:
            __file = filepath
            db_entities_dict = ActivateUtils.db_get_sensitive_entities(
                data_source_type,
                filepath,
                entities_list,
                entity_separator,
                db_path
            )
            if entity_type.lower() in db_entities_dict.keys():
                __entity = db_entities_dict[entity_type.lower()].pop(0)
                break
        return __file, __entity

    @staticmethod
    def db_get_sensitive_columns_list(data_source, entities_list, db_path, flag=0):
        """
        Get Sensative Files From SQL DB for Matching Entities
        Args:
            data_source: Data Source Type
            entities_list: List of Entities
            db_path: Path to Database Object
            flag: Value of FLag Column in Database

        Returns: List of Sensitive Columns
        """
        column_dict = {
            cs.EXCHANGE: "Subject",
            cs.ONE_DRIVE: "FilePath",
            cs.DATABASE: "Subject",
            cs.GOOGLE_DRIVE: "FilePath",
            cs.SHAREPOINT: "FilePath"
        }
        if data_source == cs.EXCHANGE or data_source == cs.DATABASE:
            flag = 1
        column_name = column_dict[data_source]
        sensitive_columns_list = []
        db_table = ActivateUtils.db_get_entities_dict_from_sqllite(db_path)
        for rows in db_table['result']:
            for entity in entities_list:
                if rows[entity] is not None and rows['Flag'] == flag:
                    sensitive_columns_list.append(rows[column_name])
                    break
        return sensitive_columns_list

    @staticmethod
    def db_get_sensitive_entities(
            data_source, column_data,
            entities_list, entity_separator, db_path):
        """
        Get Sensisitve Entities for Row which matches column data
        Args:
            data_source: Data Source Type
            column_data: Column Data for filtering
            entities_list: List of Entities to return
            entity_separator: Separator between Entities
            db_path: Path of DB to use

        Returns: Entites Dictionary for matching column_data

        """
        columns_dict = {
            cs.EXCHANGE: "Subject",
            cs.ONE_DRIVE: "FilePath",
            cs.DATABASE: "Subject",
            cs.GOOGLE_DRIVE: "FilePath",
            cs.SHAREPOINT: "FilePath"
        }
        column_name = columns_dict[data_source]
        db_entities_dict = {}
        db_table = ActivateUtils.db_get_entities_dict_from_sqllite(db_path)
        for row in db_table['result']:
            if column_data == row[column_name]:
                for entity in entities_list:
                    if row[entity] is not None:
                        db_entities_dict[entity.lower()] = \
                            sorted(row[entity].split(
                                entity_separator), key=str.lower)
                break
        return db_entities_dict

    def run_data_generator(self, data_api_path, data_endpoint):
        """
        Run Data Generator Utility TO Populate Endpoint.
        Args:
            data_api_path: Full Path To Utility
            data_endpoint: Data Endpoint Like Exchange,OneDrive,Database

        Returns: Boolean response to JOB Success ot Failure

        """
        self.log.info(
            f"Log File for Data Generation Utility {os.path.join(data_api_path, f'DataGenerator_{data_endpoint}.log')}")
        data_gen = \
            os.path.join(data_api_path, f'DataGenerator.exe "{data_endpoint}"') + f' 1> "DataGenerator_{data_endpoint}.log" 2>&1'
        status = Popen(data_gen, cwd=data_api_path,
                       stdout=PIPE, stderr=PIPE, shell=True)
        if status.wait() != 0:
            raise Exception("Data Generation JOB Failed .Please Check LOG")
        self.log.info("Completed Data Generation Succesfully!!")
        return True

    def mail_generate_and_send(self, recepients, mail_api_path):
        """
        Generate Random Mails And Send
        Args:
            recepients: List of Reciepients
            mail_api_path: Path of Mail API root Dir

        Returns:

        """
        mailboxes = recepients.strip().replace(",", ";")
        params_file = os.path.join(mail_api_path, "params.json")
        self.log.info(f"Updating {params_file} with {mailboxes} ")
        if not os.path.exists(params_file):
            raise Exception(f"File Not Found {params_file}")

        data = {}
        with open(params_file, "r") as file:
            data = file.read()
            data = json.loads(data)
            data[cs.EXCHANGE]["recepients"] = mailboxes
            file.close()
        with open(params_file, "w") as file:
            json.dump(data, file)
            file.close()
        if self.run_data_generator(mail_api_path, cs.EXCHANGE):
            self.log.info("Mailing Completed Successfull!!")

    def run_backup_mailbox_job(self, subclient_obj, backupset_obj,
                               subclient_name, exchange_mailbox_alias):
        """
        Run Backup JOB from Comcell
        Args:
            subclient_obj: Subclient Object
            backupset_obj: BackupSet Object
            subclient_name: Name of Subclient
            exchange_mailbox_alias: Mailbox Aliases

        Returns: SMTP of Mailboxes

        """
        associated_users = [elem["alias_name"] for elem in subclient_obj.users]

        input_users = str(exchange_mailbox_alias).split(",")

        mailboxes = []
        if str(exchange_mailbox_alias).lower() == "all mailboxes":
            mailboxes = ["all mailboxes"]
        else:
            new_user_mailbox = list(
                set(input_users).difference(set(associated_users)))
            self.log.info('*' * 5 + f"Selected Mailboxes for Association {str(new_user_mailbox)}" + '*' * 5)
            if len(new_user_mailbox) > 0:
                subclient_obj.set_user_assocaition(
                    {
                        'mailboxNames': new_user_mailbox,
                        'archive_policy': "Archving Policy",
                        'cleanup_policy': "cleanup_policy",
                        'retention_policy': "retention_policy"
                    }
                )
            subclient_obj = backupset_obj.subclients.get(subclient_name)
            mailboxes = [item["smtp_address"] for mail in input_users for item in subclient_obj.users if
                         item["alias_name"] == mail]

        self.log.info(
            '*' * 5 + "Starting Backup JOB for Associated Mailboxes" + '*' * 5)
        job_obj = subclient_obj.backup()
        if not job_obj.wait_for_completion():
            raise Exception("Backup JOB of Associated Subclients Could\
            not be Complted . JOB Status is " + str(job_obj.status))
        self.log.info("Job Completed Successfully!!")
        return mailboxes

    @staticmethod
    def get_user_to_sid_mapping(user_list, target_machine_name=None):
        """
        Get Username to SID mapping dictionary for given user list
        Args:
            user_list (list): List of users
            target_machine_name (str): Host name of remote machine, None for local
        Returns:
            {"username": "sid"} (dictionary)
        """
        user_to_sid = {}
        for user in user_list:
            user_to_sid[user] = win32security.LookupAccountName(
                target_machine_name, user)[0]
        return user_to_sid

    @staticmethod
    def get_access_control_list(path, target_machine_name=None):
        """
        Get Access Control List for given file or directory
        Args:
            path (str): Path to file or folder
            target_machine_name (str): Name of target machine
        Return:
            {"username": set(permissions)} dict: Return permissions dictionary
        """
        permission_dict = {}
        for line in os.popen("icacls \"%s\"" % path).read().splitlines()[:-1]:
            if not line.__eq__(""):
                line = line.strip()
                if len(line.split(path)) > 1:
                    line = line.split(path)[1].strip()
                user = line.split(":")[0]
                if user.startswith('S-1'):
                    user, domain, user_type = win32security.LookupAccountSid(
                        target_machine_name,
                        win32security.ConvertStringSidToSid(user)
                    )
                    user = f"{domain}\\{user}"
                permission = set(line.split(
                    ":")[1][1:-1].replace(')(', ',').split(','))
                if user in permission_dict.keys():
                    permission_dict[user] = permission_dict[user] | permission
                else:
                    permission_dict[user] = permission
        return permission_dict

    @staticmethod
    def get_user_info(file_path, target_machine_name=None):
        """
        Get File Owner for given file path
        Args:
            file_path (str): Path of file
            target_machine_name (str): Hostname of target machine.
                                        None implies current machine
        Returns:
            "DOMAIN\\OWNER" (str)
        """
        sd = win32security.GetFileSecurity(
            file_path, win32security.OWNER_SECURITY_INFORMATION)
        owner_sid = sd.GetSecurityDescriptorOwner()
        username = "ORPHAN"
        domain = "ORPHAN"
        if not win32security.ConvertSidToStringSid(owner_sid).__eq__('S-1-0-0-0'):
            username, domain, user_type = win32security.LookupAccountSid(
                target_machine_name, owner_sid
            )
        return f"{domain}\\{username}"

    @staticmethod
    def set_file_time_info(file_path):
        """
        Set Access Time, Creation Time, Modified Time randomly for
        given file (TimeZone Set GMT standard time)
            Args:
                file_path (str): Full path of file
        """
        create_time_offset = random.randint(3, 8) * 367
        access_time_offset = random.randint(1, 2) * 365
        modified_time_offset = random.randint(1, 2) * 365
        fh = win32file.CreateFile(
            file_path, win32file.GENERIC_READ | win32file.GENERIC_WRITE, 0, None, win32file.OPEN_EXISTING, 0, 0)
        old_create_time, old_access_time, old_modify_time = win32file.GetFileTime(
            fh)
        new_create_time = old_create_time - timedelta(create_time_offset)
        new_access_time = new_create_time + timedelta(access_time_offset)
        new_modified_time = new_create_time + timedelta(modified_time_offset)
        win32file.SetFileTime(fh, new_create_time,
                              new_access_time, new_modified_time)
        win32file.CloseHandle(fh)

    @staticmethod
    def change_file_owner(file_path_list, owner_sid=None,
                          target_machine_name=None, force=True):
        """
        Make the ownership of a given NTFS file/directories to  given sid
            Args:
                file_path_list (list): List of files to make orphan
                owner_sid (PySID): Change ownership of file to this sid.
                                If None is passed makes files orphan
                target_machine_name (str): Hostname of machine on which files reside.
                                        None for local Machine
                force (bool):
        """
        if owner_sid is None:
            owner = win32security.ConvertStringSidToSid('S-1-0-0-0')
        else:
            owner = owner_sid
        try:
            htoken = win32security.OpenThreadToken(
                win32api.GetCurrentThread(),
                win32security.TOKEN_ALL_ACCESS, True)
        except win32security.error:
            htoken = win32security.OpenProcessToken(
                win32api.GetCurrentProcess(),
                win32security.TOKEN_ALL_ACCESS)
        prev_state = ()
        if force:
            new_state = [(
                win32security.LookupPrivilegeValue(target_machine_name, name),
                win32security.SE_PRIVILEGE_ENABLED
            )for name in (win32security.SE_TAKE_OWNERSHIP_NAME, win32security.SE_RESTORE_NAME)]
            prev_state = win32security.AdjustTokenPrivileges(
                htoken, False, new_state)
        file_path = ""
        try:
            sd = win32security.SECURITY_DESCRIPTOR()
            sd.SetSecurityDescriptorOwner(owner, False)
            for file_path in file_path_list:
                win32security.SetFileSecurity(
                    file_path, win32security.OWNER_SECURITY_INFORMATION, sd)
        except win32security.error as e:
            raise OSError(
                'Cannot take ownership of file: {0}. {1}.'.format(file_path, e))
        finally:
            if prev_state:
                win32security.AdjustTokenPrivileges(htoken, False, prev_state)

    def create_fso_metadata_db(self, target_data_path, target_data_db,
                               target_machine_name=None):
        """
        Create fso metadata database5
        Args:
            target_data_path (str): Fso data path
            target_data_db (str): Target data path
            target_machine_name (str): Host name of target machine, None for local
        """

        if os.path.exists(target_data_db):
            os.remove(target_data_db)
        dir_list = list()
        file_meta_data = list()
        for root, dirs, files in os.walk(target_data_path):
            for name in files:
                file_os_obj = os.stat(os.path.join(root, name))
                m_timestamp = datetime.fromtimestamp(file_os_obj.st_mtime)
                c_timestamp = datetime.fromtimestamp(file_os_obj.st_ctime)
                a_timestamp = datetime.fromtimestamp(file_os_obj.st_atime)
                temp_list = [
                    os.path.join(root, name)[
                        0:os.path.join(root, name).rindex('\\')],
                    name, os.path.join(root, name),
                    file_os_obj.st_size,
                    mimetypes.guess_type(os.path.join(root, name))[0],
                    f"{m_timestamp.date().strftime('%B %d, %Y')} {m_timestamp.time().strftime('%r')}",
                    f"{c_timestamp.date().strftime('%B %d, %Y')} {c_timestamp.time().strftime('%r')}",
                    f"{a_timestamp.date().strftime('%B %d, %Y')} {a_timestamp.time().strftime('%r')}",
                    self.get_user_info(os.path.join(
                        root, name), target_machine_name),
                    str(ActivateUtils.get_access_control_list(
                        os.path.join(root, name)[
                            0:os.path.join(root, name).rindex('\\')],
                        target_machine_name=target_machine_name)),
                    str(ActivateUtils.get_access_control_list(
                        os.path.join(root, name),
                        target_machine_name=target_machine_name))
                ]
                file_meta_data.append(temp_list)
                del temp_list
                del m_timestamp
                del c_timestamp
                del a_timestamp
            for name in dirs:
                dir_list.append(os.path.join(root, name))
        connection = sqlite3.connect(target_data_db)
        create_table_query = '''
        CREATE TABLE fso_metadata (
        ID INTEGER  PRIMARY KEY AUTOINCREMENT,
        PARENT_DIR TEXT NOT NULL,
        FILENAME TEXT NOT NULL,
        FILEPATH TEXT NOT NULL,
        FILE_SIZE BIGINT NOT NULL,
        MIME_TYPE TEXT NOT NULL,
        MODIFIED_TIME VARCHAR(30) NOT NULL,
        CREATED_TIME VARCHAR(30) NOT NULL,
        ACCESS_TIME VARCHAR(30) NOT  NULL,
        FILE_OWNER VARCHAR(50) NOT NULL,
        TOTAL_DIR_COUNT INT NOT NULL,
        PARENT_DIR_PERMISSION TEXT NOT NULL,
        FILE_PERMISSION TEXT NOT NULL
        );
        '''
        connection.execute(create_table_query)
        for data_row in file_meta_data:
            insert_query = f'''
            INSERT INTO fso_metadata 
            (PARENT_DIR, FILENAME, FILEPATH, FILE_SIZE,MIME_TYPE,
            MODIFIED_TIME, CREATED_TIME, ACCESS_TIME, 
            FILE_OWNER, TOTAL_DIR_COUNT,
            PARENT_DIR_PERMISSION, FILE_PERMISSION) VALUES(
            "{data_row[0]}",
            "{data_row[1]}",
            "{data_row[2]}",
            "{data_row[3]}",
            "{data_row[4]}",
            "{data_row[5]}",
            "{data_row[6]}",
            "{data_row[7]}",
            "{data_row[8]}",
            "{len(dir_list)}",
            "{data_row[9]}",
            "{data_row[10]}"
            );
            '''
            connection.execute(insert_query)
        connection.commit()
        connection.close()

    def create_fso_data(self, target_data_path, file_count,
                        target_machine_name=None):
        """
        Create SQL Database to store FSO entities for Admin Console
        Review
        Args:
            target_data_path (str): Target UNC or local path to dump data
            file_count (atr): Total no of files to be generated
            target_machine_name (str): Hostname of target machine, None for local
        """
        temp_file_count = file_count
        if os.path.exists(target_data_path):
            self.delete_old_data(target_data_path)
        if not target_data_path.startswith("\\") and not os.path.exists(target_data_path):
            os.mkdir(target_data_path)
        duplicate_files_count = 0
        orphan_files_count = 0
        files_path_list = list()
        if 2 < file_count < 5:
            duplicate_files_count = 1
            orphan_files_count = 1
        if file_count >= 5:
            duplicate_files_count = int(
                (random.randint(20, 25) / 100) * file_count)
            orphan_files_count = int(
                (random.randint(20, 25) / 100) * file_count)
        file_count = file_count - duplicate_files_count
        self.log.info(
            "Total File Count %d Duplicate file count %d orphan files count %d"
            % (file_count, duplicate_files_count, orphan_files_count)
        )
        file_per_folder = file_count
        if file_count > 200:
            file_per_folder = int((random.randint(5, 10) / 100) * file_count)
        self.log.info("Generating %d Files per Folder" % file_per_folder)
        while file_count != 0:
            path = os.path.join(target_data_path, f"fso_data_{str(int(time.time()))}")
            os.mkdir(path)
            self.log.info("%s created successfully" % path)
            self.log.info("Generating %d files at %s" %
                          (min(file_count, file_per_folder), path))
            self.sensitive_data_generation(
                path, min(file_count, file_per_folder))
            if file_count < file_per_folder:
                file_count = 0
            else:
                file_count = file_count - file_per_folder

        duplicate_list = []
        orphan_list = []
        dir_list = []
        self.log.info("Crawling %s" % target_data_path)
        for root, dirs, files in os.walk(target_data_path):
            for file in files:
                files_path_list.append(os.path.join(root, file))
            for d in dirs:
                dir_list.append(os.path.join(root, d))
        duplicate_file_path = os.path.join(target_data_path, 'Duplicate_Files')
        if duplicate_files_count > 0:
            self.log.info("Generating Duplicate Files")
            duplicate_list = files_path_list[:duplicate_files_count]
            os.mkdir(duplicate_file_path)
            if os.path.exists(duplicate_file_path):
                self.log.info(f"{duplicate_file_path} created successfully")
            for file in duplicate_list:
                shutil.copy2(file, duplicate_file_path)
        if orphan_files_count > 0:
            self.log.info("Generating Orphan Files")
            orphan_list = files_path_list[file_count - orphan_files_count:]
            ActivateUtils.change_file_owner(
                orphan_list, target_machine_name=target_machine_name)
        duplicate_list_temp = list()
        for i in range(len(duplicate_list)):
            duplicate_list_temp.append(duplicate_list[i])
            temp = duplicate_list[i].rindex("\\")
            duplicate_list[i] = os.path.join(
                duplicate_file_path,
                duplicate_list[i][temp + 1:]
            )
            duplicate_list_temp.append(duplicate_list[i])
        dir_list.append(duplicate_file_path)
        user_list = ['system', 'administrator', 'administrators', 'everyone']
        user_sid_dict = ActivateUtils.get_user_to_sid_mapping(
            user_list, target_machine_name=target_machine_name)
        files_path_list = files_path_list + duplicate_list
        temp_list = list(set(files_path_list) - set(orphan_list))
        i = 0
        while i < len(temp_list):
            for username in user_list[:-1]:
                if i < len(temp_list):
                    self.change_file_owner(
                        [temp_list[i]], owner_sid=user_sid_dict[username],
                        target_machine_name=target_machine_name
                    )
                i = i + 1
        for d in dir_list:
            owner_name = ActivateUtils.get_user_info(
                d, target_machine_name).split("\\")[-1].lower()
            owner_sid = user_sid_dict[owner_name]
            sd = win32security.GetFileSecurity(
                d, win32security.DACL_SECURITY_INFORMATION)
            dacl = win32security.ACL()
            dacl.AddAccessAllowedAce(
                win32security.ACL_REVISION, con.FILE_ALL_ACCESS, owner_sid)
            d_user_list = list(set(user_list[:-1]) - {owner_name})
            dacl.AddAccessAllowedAce(
                win32security.ACL_REVISION,
                con.FILE_GENERIC_READ | con.FILE_GENERIC_WRITE | con.FILE_GENERIC_EXECUTE,
                user_sid_dict[d_user_list[0]])
            dacl.AddAccessAllowedAce(
                win32security.ACL_REVISION,
                con.FILE_GENERIC_WRITE,
                user_sid_dict[d_user_list[1]])
            everyone_sid = user_sid_dict["everyone"]
            dacl.AddAccessAllowedAce(
                win32security.ACL_REVISION,
                con.FILE_GENERIC_READ | con.FILE_LIST_DIRECTORY,
                everyone_sid)
            sd.SetSecurityDescriptorDacl(1, dacl, 0)
            win32security.SetFileSecurity(
                d, win32security.DACL_SECURITY_INFORMATION, sd)
        for file in files_path_list:
            owner_name = ActivateUtils.get_user_info(
                file, target_machine_name).split("\\")[-1].lower()
            if owner_name.__eq__("orphan"):
                owner_name = 'administrators'
            owner_sid = user_sid_dict[owner_name]
            sd = win32security.GetFileSecurity(
                file, win32security.DACL_SECURITY_INFORMATION)
            dacl = win32security.ACL()
            dacl.AddAccessAllowedAce(
                win32security.ACL_REVISION, con.FILE_ALL_ACCESS, owner_sid)
            f_user_list = list(set(user_list[:-1]) - {owner_name})
            dacl.AddAccessAllowedAce(win32security.ACL_REVISION,
                                     con.FILE_GENERIC_READ | con.FILE_GENERIC_WRITE | con.FILE_GENERIC_EXECUTE,
                                     user_sid_dict[f_user_list[0]])
            dacl.AddAccessAllowedAce(
                win32security.ACL_REVISION,
                con.FILE_GENERIC_WRITE,
                user_sid_dict[f_user_list[1]])
            everyone_sid = user_sid_dict['everyone']
            dacl.AddAccessAllowedAce(
                win32security.ACL_REVISION,
                con.FILE_GENERIC_READ, everyone_sid)
            sd.SetSecurityDescriptorDacl(1, dacl, 0)
            win32security.SetFileSecurity(
                file, win32security.DACL_SECURITY_INFORMATION, sd)
            if file not in duplicate_list_temp:
                ActivateUtils.set_file_time_info(file)
        self.log.info(f"Generated {temp_file_count} files at  {target_data_path}")
