# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""helper class for index server related operations

    IndexServerHelper:

        __init__()                              --      initialize the IndexServerHelper class

        get_cores_for_role()                    --      returns the core details for the given role

        validate_backup_size_with_src           --      validates folder size matches between browse response
                                                            and source index directory

        validate_restore_data_with_browse       --      validates restored data size matches the browse response size

        update_roles                            --      configures required roles on index server

        run_full_backup                         --      Enables client/subclient backup & runs full backup on
                                                            default subclient on index server

        monitor_restore_job                     --      monitors the restore job

        init_subclient                          --      initialise the objects for client/backupset/subclient

        validate_data_in_core                   --      validates the file/folder count present in data source core

		delete_index_server                     --      Deletes the Index Server and removes Index directory

		check_solr_doc_count                    --      checks whether solr document count is matching or not with input

"""
from past.builtins import basestring
from cvpysdk.datacube.constants import IndexServerConstants as index_server_const
from dynamicindex.utils import constants as dynamic_constants
from AutomationUtils import logger
from AutomationUtils.machine import Machine


class IndexServerHelper():
    """Helper class for index server operations"""

    def __init__(self, commcell_object, index_server_name):
        """Initialize the IndexServerHelper object"""
        self.log = logger.get_log()
        self.commcell = commcell_object
        self.index_server_obj = self.commcell.index_servers.get(index_server_name)
        self.index_server_name = index_server_name
        self.sub_client = "default"
        self.is_sub_client_initialized = False
        self.client_obj = None
        self.agent_obj = None
        self.instance_obj = None
        self.subclient_obj = None

    def check_solr_doc_count(self, solr_response, doc_count):
        """checks whether solr document count is matching or not with input

                Args:

                    solr_response       (dict)      --  Solr response JSON

                    doc_count           (int)       --  Document count to verify
                                                            if -1, then we check for condition solr_doc_count!=0

                Returns:

                    None

                Raises:

                    Exception:

                            if input data type is not valid

                            if document count mismatches
        """
        if not isinstance(solr_response, dict) or not isinstance(doc_count, int):
            raise Exception("Input data type is not valid")
        solr_doc_count = int(solr_response['response']['numFound'])
        self.log.info(f"Solr Response document count : {solr_doc_count}")
        if doc_count not in (-1, solr_doc_count):
            raise Exception(f"Document count mismatched. Expected : {doc_count} Actual : {solr_doc_count}")
        elif doc_count == -1 and solr_doc_count == 0:
            raise Exception(f"Document count mismatched. Expected doc count > zero but Actual : {solr_doc_count}")
        self.log.info(f"Document count matched as expected : {solr_doc_count}")

    def init_subclient(self):
        """Initialise the client/backupset/subclient object"""
        if not self.is_sub_client_initialized:
            self.client_obj = self.commcell.clients.get(self.index_server_name)
            self.agent_obj = self.client_obj.agents.get(index_server_const.INDEX_SERVER_IDA_NAME)
            self.instance_obj = self.agent_obj.instances.get(index_server_const.INDEX_SERVER_INSTANCE_NAME)
            self.subclient_obj = self.instance_obj.subclients.get(self.sub_client)
            self.is_sub_client_initialized = True
            self.log.info("Client/Agent/Instance/Subclient objects initialized")

    def validate_data_in_core(self, data_source_obj, file_count, folder_count, file_criteria=None):
        """validates the file/folder count present in data source core matches with given count

                Args:

                    data_source_obj (object)    --  DataSource class object

                    file_count      (int)        --  total file count

                    folder_count    (int)        --  total folder count

                    file_criteria   (dict)       --  Dictionary containing search criteria and value for files
                                                            Acts as 'q' field in solr query

                Returns:

                    None

                Raises:

                    Exception:

                        if input is not valid

                        if there is mismatch in file/folder count

        """
        if not isinstance(file_count, int) or not isinstance(folder_count, int):
            raise Exception("Input data is not valid")
        self.log.info(f"File/Folder count to verify is : {file_count}/{folder_count}")
        if file_criteria is None:
            file_criteria = dynamic_constants.QUERY_FILE_CRITERIA
        else:
            file_criteria.update(dynamic_constants.QUERY_FILE_CRITERIA)
        self.log.info(f"File criteria formed : {file_criteria}")
        resp = self.index_server_obj.execute_solr_query(core_name=data_source_obj.computed_core_name,
                                                        select_dict=file_criteria)
        solr_files = int(resp['response']['numFound'])
        if solr_files != file_count:
            raise Exception(f"File count mismatched Expected : {file_count} Actual : {solr_files}")
        self.log.info("File count Matched!!! Total crawl count from data source : %s", solr_files)

        if folder_count != dynamic_constants.SKIP_FOLDER_CHECK:
            resp = self.index_server_obj.execute_solr_query(core_name=data_source_obj.computed_core_name,
                                                            select_dict=dynamic_constants.QUERY_FOLDER_CRITERIA)
            solr_folder = int(resp['response']['numFound'])
            if solr_folder != folder_count:
                raise Exception(f"Folder count mismatched Expected : {folder_count} Actual : {solr_folder}")
            self.log.info("Folder count Matched!!! Total crawl count from data source : %s", solr_folder)

    def monitor_restore_job(self, job_obj):
        """Monitors the restore job till it completes

                Args:

                    job_obj     (object)        --  instance of job class

                Returns:

                    None

                Raises:

                    Exception:

                        if job fails or threshold time reaches for job completion


        """

        self.log.info("Going to Monitor this restore job for completion : %s", job_obj.job_id)
        if not job_obj.wait_for_completion(timeout=90):
            self.log.info("Index server restore failed. Please check logs")
            raise Exception("Index server restore failed. Please check logs")
        if job_obj.status.lower() != 'completed':
            raise Exception("Index server restore completed with error status. Please check logs")
        self.log.info("Index server restore job is finished")

    def run_full_backup(self):
        """ Enables client/subclient backup & runs full backup on default subclient

                Args:
                    None

                Returns:

                    str     --  job id

                Raises:
                    None
        """
        self.init_subclient()
        self.log.info("Enable backup at client and subclient level")
        if not self.client_obj.is_backup_enabled:
            self.client_obj.enable_backup()
        if not self.subclient_obj.is_backup_enabled:
            self.subclient_obj.enable_backup()

        self.log.info("Going to start FULL backup job for index server")
        job_obj = self.subclient_obj.run_backup()
        self.log.info("Going to Monitor this backup job for completion : %s", job_obj.job_id)
        backup_job_id = job_obj.job_id
        if not job_obj.wait_for_completion(timeout=90):
            self.log.info("Index server backup failed. Please check logs")
            raise Exception("Index server backup failed. Please check logs")
        if job_obj.status.lower() != 'completed':
            raise Exception("Index server backup completed with error status. Please check logs")
        self.log.info("Index server backup job is finished")
        return backup_job_id

    def update_roles(self, index_server_roles):
        """ updates the index server roles with specified role list

                Args:

                    index_server_roles      (list)      --  list of index server roles

                Returns:
                    None

                Raises:
                    Exception:

                            if input data type is not valid

                            if failed to update roles

        """
        if not isinstance(index_server_roles, list):
            raise Exception("Input data type is not valid")

        update_roles = []
        update_required = False
        self.log.info("Setting up index server")
        self.log.info("Current roles defined in index server : %s", self.index_server_obj.role_display_name)
        for role in index_server_roles:
            if role not in self.index_server_obj.role_display_name:
                temp_role = index_server_const.UPDATE_ADD_ROLE
                temp_role['roleName'] = role
                update_roles.append(temp_role)
                update_required = True

        if update_required:
            self.log.info("Required roles is not present. Calling role update on index server")
            self.log.info("Request Json : %s", update_roles)
            self.index_server_obj.update_role(update_roles)

    def validate_restore_data_with_browse(
            self,
            role_name,
            client_name,
            restore_path,
            backup_job_id,
            index_server_node=None):
        """ validate restored core folder size and browse core sizer matches or not

            Args:

                role_name                   (str)       --  role name whose cores size needs to be verified

                client_name                 (str)       --  client name where data has been restored

                restore_path                (str)       --  folder path in client where restored data is present

                backup_job_id               (str)       --  backup job id from where data has been restored

                index_server_node           (str)       --  index server client node name
                                                    if none, then first node from index server will be considered.

            Returns:

                True if size matches between restored data and browse
                False if size differs

            Raises:

                Exception:

                    if input data is not valid

                    if unable to verify size for cores

        """
        if index_server_node is None:
            index_server_node = self.index_server_obj.client_name[0]
        self.init_subclient()
        if not (isinstance(backup_job_id, basestring) or
                isinstance(role_name, basestring) or
                isinstance(client_name, basestring) or
                isinstance(restore_path, basestring)):
            raise Exception("Input data type is not valid")
        folder_list, data_from_index_server = self.subclient_obj.get_file_details_from_backup(
            roles=[role_name], include_files=False,
            job_id=backup_job_id)
        self.log.info("Browse response from index server : %s", data_from_index_server)
        src_machine_obj = Machine(commcell_object=self.commcell,
                                  machine_name=client_name)
        self.log.info("Created machine object for client : %s", client_name)
        self.log.info("Restored path : %s", restore_path)
        for folder in folder_list:
            if folder == f"\\{role_name}\\{index_server_node}":
                self.log.info(f"Ignore root folder : {folder}")
                continue
            if not (
                    folder.endswith(
                        dynamic_constants.BACKUP_CONF_DIR) or
                    folder.endswith(dynamic_constants.BACKUP_INDEX_DIR)):
                self.log.info(f"Ignore root folder : {folder}")
                continue
            # split folder and take only last two value which gives corename\<config/index folder>
            folder_split = folder.split('\\')
            modified_folder = f"{folder_split[-2]}\\{folder_split[-1]}"
            self.log.info("Validating folder : %s", folder)
            self.log.info("Modified folder : %s", modified_folder)
            restored_folder = str(f"{restore_path}\\{modified_folder}")
            restored_folder = restored_folder.replace("\\\\", "\\")
            self.log.info("Restored folder path : %s", restored_folder)
            restored_folder_size = src_machine_obj.get_folder_size(folder_path=restored_folder,
                                                                   in_bytes=True)
            browse_folder_size = data_from_index_server[folder]['size']
            self.log.info("Restore folder size : %s", restored_folder_size)
            self.log.info("Browse folder size : %s", browse_folder_size)
            if browse_folder_size != restored_folder_size:
                msg = f"Restore data size not matched with browse for folder : {folder}"
                self.log.info(msg)
                return False
            self.log.info("Folder size matched : %s", browse_folder_size)

        return True

    def validate_backup_size_with_src(self, role_name, job_id=0, backup_all_prop=False, index_server_node=None):
        """ validate core folder size in backup and source core size in index server matches or not

            Args:

                job_id                      (int)       --  job id on which browse will be done

                role_name                   (str)       --  role name whose core size needs to be verified

                backup_all_prop             (bool)      --  true if core.properties files are backed up and
                                                                need to be included during config size calculation
                    default : False
                  *** we are not backing up core.properties file for default cores.change this once it is backed up ***

                index_server_node           (str)       --  index server client node name
                                                    if none, then first node from index server will be considered.

            Returns:

                True if size matches between source and browse
                False if size differs

            Raises:

                Exception:

                    if input data is not valid

                    if unable to verify size for cores

        """
        if index_server_node is None:
            index_server_node = self.index_server_obj.client_name[0]
        self.init_subclient()
        if not isinstance(role_name, basestring):
            raise Exception("Input data type is not valid")
        folder_list, data_from_index_server = self.subclient_obj.get_file_details_from_backup(
            roles=[role_name], include_files=False,
            job_id=job_id)
        self.log.info("Browse response from index server : %s", data_from_index_server)
        self.log.info(f"Browse folder list : {folder_list}")
        src_machine_obj = None
        self.log.info("Creating machine object for client : %s", index_server_node)
        src_machine_obj = Machine(machine_name=index_server_node,
                                  commcell_object=self.commcell)

        analytics_dir = src_machine_obj.get_registry_value(commvault_key=dynamic_constants.ANALYTICS_REG_KEY,
                                                           value=dynamic_constants.ANALYTICS_DIR_REG_KEY)
        self.log.info("Index server Index directory is : %s", analytics_dir)
        core_list, core_details = self.get_cores_for_role(role_name=role_name)
        self.log.info("Cores got for validation from index server : %s", core_list)
        # from folder list, reduce one count for dummy root folder
        if len(core_list) != (len(folder_list) - 1) / \
                4:  # for each core, there will 4 folder.. core folder, parent folder,index folder,config folder
            msg = f"Core list from browse and index server not matched. " \
                  f"Browse core count {(len(folder_list)-1)/4} index server core count {len(core_list)}"
            self.log.info(msg)
            return False
        self.log.info("Cores from browse and index server matched. No of cores : %s", len(core_list))
        for core in core_list:
            conf_size = 0
            index_size = 0
            core_info = None
            browse_conf_dir = None
            browse_index_dir = None
            self.log.info("Validating core : %s", core)
            for details in core_details:
                if core in details:
                    if details[core]['name'] == core:
                        core_info = details[core]
            instance_dir = core_info['instanceDir']
            data_dir = f"{core_info['dataDir']}\\{dynamic_constants.BACKUP_INDEX_DIR}"
            self.log.info("Instance dir : %s", instance_dir)
            self.log.info("Data dir : %s", data_dir)
            index_size = src_machine_obj.get_folder_size(folder_path=data_dir, in_bytes=True)
            self.log.info("Source index size : %s", index_size)
            if f"\\{dynamic_constants.BACKUP_CONF_HOME}\\" in instance_dir:
                self.log.info("Size will be calculated based on ConfHome folder alone for : %s", core)
                conf_size = src_machine_obj.get_folder_size(folder_path=instance_dir,
                                                            in_bytes=True)
            elif f"\\{dynamic_constants.BACKUP_CONF_SETS}\\" in instance_dir:
                instance_dir = f"{instance_dir}\\conf"
                self.log.info("Modified Instance dir for configsets: %s", instance_dir)
                self.log.info("Size will be calculated based on configset for : %s", core)
                conf_size = src_machine_obj.get_folder_size(folder_path=instance_dir,
                                                            in_bytes=True)
                if backup_all_prop:
                    core_prop = f"{analytics_dir}\\{dynamic_constants.BACKUP_CONF_HOME}\\{core}\\core.properties"
                    self.log.info("Core properties location : %s", core_prop)
                    core_prop_size = src_machine_obj.get_file_size(file_path=core_prop, in_bytes=True)
                    self.log.info("Core properties size : %s", core_prop_size)
                    conf_size = conf_size + core_prop_size
            self.log.info("Source Config size : %s", conf_size)
            browse_conf_dir = f"\\{role_name}\\{index_server_node}\\{core}" \
                              f"\\{core}\\{dynamic_constants.BACKUP_CONF_DIR}"
            browse_index_dir = f"\\{role_name}\\{index_server_node}\\{core}" \
                               f"\\{core}\\{dynamic_constants.BACKUP_INDEX_DIR}"
            self.log.info("Browse config dir : %s", browse_conf_dir)
            self.log.info("Browse index dir : %s", browse_index_dir)
            if browse_conf_dir not in data_from_index_server or browse_index_dir not in data_from_index_server:
                raise Exception(f"Browse don't have required folder structure")
            browse_conf_size = data_from_index_server[browse_conf_dir]['size']
            browse_index_size = data_from_index_server[browse_index_dir]['size']
            if conf_size != browse_conf_size:
                msg = f"Config size not matched. Source {conf_size} Browse {browse_conf_size}"
                self.log.info(msg)
                return False
            self.log.info("Config size matched. Size = %s", browse_conf_size)
            if index_size != browse_index_size:
                msg = f"Index size not matched. Source {index_size} Browse {browse_index_size}"
                self.log.info(msg)
                return False
            self.log.info("Index size matched. Size = %s", browse_index_size)
        return True

    def get_cores_for_role(self, role_name, client_name=None):
        """ Gets core names for the given role from index server

            Args:

                role_name                   (str)       --  role name

                client_name                 (str)       --  client node name
                    ***Applicable only for solr cloud mode***

            Returns:

                (list,dict)             -- list containing core names
                                        -- dict containing details about cores

            Raises:

                Exception:

                    if input data is not valid

                    if unable to get core details

        """
        if not isinstance(role_name, basestring):
            raise Exception("Input data type is not valid")
        self.log.info("Going to get core details from index server : %s", self.index_server_name)
        core_list, core_details = self.index_server_obj.get_all_cores(client_name=client_name)
        self.log.info("Core details : %s", core_list)
        self.log.info("Filtering out cores for role : %s", role_name)
        default_core = None
        dynamic_core = None
        filtered_list = []
        filtered_details = []
        if role_name == index_server_const.ROLE_DATA_ANALYTICS:
            default_core = dynamic_constants.DATA_ANALYTICS_DEFAULT_CORES
            dynamic_core = dynamic_constants.DATA_ANALYTICS_DYNAMIC_CORES
        elif role_name == index_server_const.ROLE_SYSTEM_DEFAULT:
            default_core = dynamic_constants.SYSTEM_DEFAULT_CORES
            dynamic_core = []
        else:
            raise Exception("Unable to determine default/dynamic cores for given role")
        self.log.info(f"Default core list : {default_core} Dynamic core list : {dynamic_core}")
        # filter logic
        for core in core_list:
            if core in default_core:
                filtered_list.append(core)
                filtered_details.append({core: core_details[core]})
                self.log.info("Adding default core : %s", core)
            else:
                for prefix in dynamic_core:
                    if str(core).startswith(prefix):
                        filtered_list.append(core)
                        filtered_details.append({core: core_details[core]})
                        self.log.info("Adding dynamic core : %s", core)
        self.log.info("Filtered cores : %s", filtered_list)
        self.log.info("Filtered cores details : %s", filtered_details)
        return filtered_list, filtered_details

    def delete_index_server(self):
        """Deletes the Index Server and removes Index directory from the Index node clients"""
        node_objects = []
        for node_name in self.index_server_obj.client_name:
            node_objects.append(self.index_server_obj.get_index_node(node_name))
        self.log.info("Deleting index server")
        self.commcell.index_servers.delete(self.index_server_name)
        self.log.info("Index server deleted")
        for node in node_objects:
            self.log.info(f"Deleting Index directory: {node.index_location} on client: {node.node_name}")
            node_machine = Machine(node.node_name, self.commcell)
            node_machine.remove_directory(node.index_location, 0)
            self.log.info("Index directory deleted")
