# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""helper class for exchange usermailbox client operations like
    backup/restore

    ExchangeClientHelper:

        __init__()                                  --  initialize the ExchangeClientHelper class

        create_exchange_mailbox_client()            --  creates a new exchange mailbox client

        create_exchange_policy()                    --  creates a exchange configuration policy

        create_exchange_configuration_policies()    --  creates all 3 exchange policies

        restore_exchange_mailbox_client()           --  perform a disk restore job on the exchange
                                                        mailbox client

"""

import json
from past.builtins import basestring
from AutomationUtils import logger
from AutomationUtils.machine import Machine
from dynamicindex.Datacube.crawl_job_helper import CrawlJobHelper
from Application.Exchange.ExchangeMailbox.exchange_mailbox import ExchangeMailbox
from Application.Exchange.ExchangeMailbox.solr_helper import SolrHelper
from Application.Exchange.ExchangeMailbox.constants import \
    CLEANUP_POLICY_DEFAULT, RETENTION_POLICY_DEFAULT, ARCHIVE_POLICY_DEFAULT


class ExchangeClientHelper():
    """Helper class for Exchange client operations like create/backup/restore"""

    def __init__(self, commcell_object):
        """Initialize the class with commcell object"""
        self.log = logger.get_log()
        self.commcell = commcell_object

    def create_exchange_mailbox_client(self, tc_object, storage_policy, index_server_name):
        """Creates a new exchange pseudoclient to the commcell

        Args:
            tc_object           (object)    --  testcase object

            storage_policy      (str)       --  storage policy name for exchange
                                                pseudo client

            index_server_name   (str)       --  index server name where the exchange
                                                pseudoclient will be pointing

        Returns:
            object      (ExchangeMailbox)   --  Exchange mailbox client instance

        """
        tc_object.tcinputs['EnvironmentType'] = 1
        tc_object.tcinputs['IndexServer'] = index_server_name
        tc_object.tcinputs['SubclientName'] = "usermailbox"
        tc_object.tcinputs['BackupsetName'] = "User Mailbox"
        tc_object.tcinputs['StoragePolicyName'] = storage_policy
        tc_object.tcinputs['RecallService'] = f'http://{self.commcell.webconsole_hostname}/webconsole'
        exchange_client = ExchangeMailbox(tc_object)
        self.log.info('Creating a exchange client')
        exchange_client.tc_object.client = exchange_client.cvoperations.add_exchange_client()
        self.log.info('Exchange client created with id: %s', exchange_client.tc_object.client.client_id)
        self.log.info("Creating an instance of subclient")
        exchange_client.tc_object.subclient = exchange_client.cvoperations.subclient
        self.log.info("Subclient instance created.")
        return exchange_client

    def create_exchange_policy(self, policy_name, policy_type):
        """Create an exchange configuration policy

        Args:
            policy_name     (str)       --  Config policy name

            policy_type     (str)       --  Policy type to be created
                  Valid values:
                        1.  "Cleanup"
                        2.  "Retention"
                        3.  "Archive"

        Returns:
            Configuration policy class instance

        """
        configuration_policies = self.commcell.policies.configuration_policies
        policy_object = configuration_policies.get_policy_object(
            policy_type, policy_name)
        self.log.info('Creating Exchange Policy %s ', policy_name)
        if configuration_policies.has_policy(policy_name):
            self.log.info('Policy exists. Deleting it and creating')
            configuration_policies.delete(policy_name)
        policy = configuration_policies.add_policy(policy_object)
        return policy

    def create_exchange_configuration_policies(self, custom_str):
        """Creates all 3 default types of configuration policies

        Args:
            custom_str      (str)   --  prefix string for the policy names

        Returns:
            None

        """
        policy_names = [ARCHIVE_POLICY_DEFAULT % custom_str,
                        CLEANUP_POLICY_DEFAULT % custom_str,
                        RETENTION_POLICY_DEFAULT % custom_str]
        policy_types = ['Archive', 'Cleanup', 'Retention']
        for index in range(len(policy_names)):
            self.create_exchange_policy(policy_names[index], policy_types[index])

    def restore_exchange_mailbox_client(self, exchange_mailbox_client, restore_machine, restore_path):
        """
        Invokes disk restore job for a given exchange client and verifies

        Args:
            exchange_mailbox_client     (object)    -   ExchangeMailbox object

            restore_machine             (str)       -   Restore client name where backed-up mails
                                                        will be restored

            restore_path                (str)       -   Restore path present on restore client
                                                        where backed-up mails will be restored

        Returns:
            None

        Raises:

            if input data is not valid

            if restored mails count was not same as expected

            if restored mail file size is invalid

        """
        if not (isinstance(exchange_mailbox_client, ExchangeMailbox) and
                isinstance(restore_machine, basestring) and isinstance(restore_path, basestring)):
            raise Exception("Input data is not of valid datatype")
        tc_object = exchange_mailbox_client.tc_object
        index_server_obj = self.commcell.index_servers.get(exchange_mailbox_client.index_server)
        restore_client_machine = Machine(restore_machine, self.commcell)
        restore_client_machine.remove_directory(restore_path, 0)
        crawl_job_helper = CrawlJobHelper(tc_object)
        self.log.info("Now starting the restore process")
        job = exchange_mailbox_client.cvoperations.subclient. \
            disk_restore(paths=[r"\MB"],
                         destination_client=restore_machine,
                         destination_path=restore_path,
                         overwrite=True,
                         journal_report=False)
        exchange_mailbox_client.cvoperations.check_job_status(job)
        restore_mails_count = crawl_job_helper.get_docs_count(
            folder_path=restore_path,
            machine_name=restore_machine,
            include_folders=False
        )
        solr_helper = SolrHelper(exchange_mailbox_client)
        select_dict = {
            "datatype": 2,
            "apid": tc_object.subclient.subclient_id
        }
        attr_params = ['szkb', 'conv']
        if index_server_obj.is_cloud:
            select_dict = {
                "DocumentType": 2,
                "ApplicationId": tc_object.subclient.subclient_id
            }
            attr_params = ['Size', 'Subject']
        solr_resp = solr_helper.create_url_and_get_response(
            select_dict=select_dict
        )
        total_count = solr_helper.get_count_from_json(solr_resp.content)
        self.log.info("Expected restore mails count : %s", total_count)
        if not restore_mails_count == total_count:
            self.log.error("Restored files count doesn't match with expected value")
            self.log.error("Expected value: %s", total_count)
            self.log.error("Actual value: %s", restore_mails_count)
            raise Exception("Restored mails count doesn't match with expected value")
        self.log.info("All mails Restored successfully")
        files_list = restore_client_machine.get_files_in_path(restore_path)
        file_name_map = {}
        for mail_file in files_list:
            file_name_map["_".join((mail_file.split('\\')[-1]).split('_')[:-3])] = mail_file
        op_params = {
            "rows": restore_mails_count
        }
        solr_resp = json.loads(solr_helper.create_url_and_get_response(
            select_dict=select_dict,
            attr_list={attr for attr in attr_params},
            op_params=op_params
        ).content)
        for doc in solr_resp['response']['docs']:
            illegal_entries = '\t\"*/:<>?\\|'
            for illegal_entry in illegal_entries:
                doc[attr_params[1]] = str(doc[attr_params[1]]).replace(illegal_entry, '_')
            disk_size = restore_client_machine.get_file_size(file_name_map[doc[attr_params[1]]], True)
            if int(disk_size) < int(doc[attr_params[0]]):
                self.log.error("Invalid file size found for %s.msg" % doc[attr_params[1]])
                self.log.error("Expected file size >=%s" % doc[attr_params[0]])
                raise Exception("Invalid file size found")
        self.log.info("No invalid file size found for any email")