# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""helper class for datacube sync

    SyncHelper:
        __init__(testcase)          --  Initialize the DcubesyncHelper object

        setdcubesync_registry(current_time,client_id,totalhours,enablesync)
                                    -- Set sync related registry on CS


        getsyncxml_bycsrestart(datasource_id) -- Get sync xml for the given datasource id

        getcoremaxstats(self,old_syncxml,new_stat,o_synctime,n_synctime)
                                    -- Get the max stats by comparing previous stats

        verifysyncxml(self,sync_xml,corestats,lastsynctime,facetstats=None,maxstats=None)
                                            -- verify the sync xml for the datasource


"""
import datetime
import time
import xmltodict

from AutomationUtils.machine import Machine


class SyncHelper():

    """ contains helper class for datacube sync
        """

    def __init__(self, tc_object):
        self.commcell = tc_object.commcell
        self.csdb = tc_object.csdb
        self.log = tc_object.log
        self.dssync_regname = "nLastDCubeSyncTime"
        self.dssync_disablereg = "nDCubeSyncDisable"
        self.dssync_regtype = "DWord"

    def set_dcube_sync_registry(self, current_time, client_id, totalhours, enablesync):
        """Set datacube sync related registry keys on given client id (CS)

                Args:
                    current_time(datetime) --  Now time
                    client_id(int)         --  id of client.
                    totalhours(int)        --  Total hours which needs to set behind
                    enablesync(boolean)    -- to enable sync or not

                Raises
                    Exception on failure

                """
        self.log.info("Current Epoch time : " + str(int(current_time.timestamp())))
        oneday_ago = current_time - datetime.timedelta(hours=totalhours)
        oneday_ago_epoch = int(oneday_ago.timestamp())
        self.log.info("Going to set nLastDCubeSyncTime as 24 hrs behind : %s",
                      str(oneday_ago_epoch))
        self.cs_machineobj = Machine(self.commcell.clients.get(client_id), self.commcell)
        reg_key = "CommServe"
        update_success = self.cs_machineobj.update_registry(
            reg_key, self.dssync_regname, oneday_ago_epoch, self.dssync_regtype)
        if not update_success:
            self.log.info("Unable to set datacube sync registry. Abort")
            raise Exception("Unable to set datacube sync registry. Abort")

        # Make sure below registry is not set
        if enablesync:
            update_success = self.cs_machineobj.create_registry(
                reg_key, self.dssync_disablereg, "0", self.dssync_regtype)
        else:
            update_success = self.cs_machineobj.create_registry(
                reg_key, self.dssync_disablereg, "1", self.dssync_regtype)

        if not update_success:
            self.log.info("Unable to set datacube disable sync registry. Abort")
            raise Exception("Unable to set datacube disable sync registry. Abort")

    def get_syncxml_by_csrestart(self, datasource_id):
        """Returns the sync xml for the given datasource id

                Args:
                    datasource_id(str)   --  id of the datasource

                Returns

                    str              -- status xml of the datasource

                Raises

                    Exception on failure


                """
        self.dsprop_query = "select PropertyValue from SEDataSourceproperty " \
                            "where PropertyId=107 and DataSourceId= %s"
        self.log.info("Going to restart all services on CS")
        self.cs_clientobj = self.commcell.commserv_client
        self.cs_clientobj.restart_services(True)
        self.log.info("Waiting for 8Mins")
        time.sleep(500)
        _query = (self.dsprop_query % datasource_id)
        self.log.info("Querying CS DB to get status info properties")
        self.csdb.execute(_query)
        statusxml = self.csdb.fetch_one_row()
        statusxml = ''.join(str(x) for x in statusxml)
        self.log.info("Status xml from CS db : " + str(statusxml))
        if statusxml is None:
            raise Exception("Status xml is empty. Datacube sync didnt happen within stipulated time!!!")
        return statusxml

    def get_core_maxstats(self, old_syncxml, new_stat, old_synctime, new_synctime):
        """Returns the max stats by comparing given status xml and core stats

                        Args:
                            old_syncxml(str) --  older sync xml before last sync
                            new_stat(dict)   -- core stats of data source after sync
                            old_synctime(str)  -- last Sync time
                            new_synctime(str)  -- Latest sync time

                        Returns

                            dict              -- containing all max stats

                        Raises

                            Exception on failure


        """
        maxstats = {}
        self.log.info("Comparing sync stats")
        self.log.info("Old Sync time : %s", str(old_synctime))
        self.log.info("Latest sync time : %s", str(new_synctime))
        self.log.info("Old sync xml : %s", str(old_syncxml))
        self.log.info("Latest stats : %s", str(new_stat))
        sync_jobj = xmltodict.parse(old_syncxml)
        old_val = int(sync_jobj['StatusInfo']['@MaxTotalDocs'])
        new_val = int(new_stat['index']['numDocs'])
        if new_val >= old_val:
            self.log.info("New stats is max for field TotalDocs")
            maxstats['MaxTotalDocs'] = new_val
            maxstats['MaxTotalSyncTime'] = new_synctime
        else:
            self.log.info("Old stats is max for field TotalDocs")
            maxstats['MaxTotalDocs'] = old_val
            maxstats['MaxTotalSyncTime'] = str(
                sync_jobj['StatusInfo']['@MaxTotalSyncTime'])[:-3]
            maxstats['totaldocs_oldsynctime'] = old_synctime

        old_val = int(sync_jobj['StatusInfo']['@MaxTotalSizeInBytes'])
        new_val = int(new_stat['index']['sizeInBytes'])
        if new_val >= old_val:
            self.log.info("New stats is max for field Total Size")
            maxstats['MaxTotalSizeInBytes'] = new_val
            maxstats['MaxSizeSyncTime'] = new_synctime
        else:
            self.log.info("Old stats is max for field Total Size")
            maxstats['MaxTotalSizeInBytes'] = old_val
            maxstats['MaxSizeSyncTime'] = str(
                sync_jobj['StatusInfo']['@MaxSizeSyncTime'])[:-3]
            maxstats['totalsize_oldsynctime'] = old_synctime

        return maxstats

    def verify_sync_xml(self, sync_xml, corestats, last_synctime, facetstats=None, maxstats=None):
        """verifies sync xml for the given corestats

                Args:
                    syncxml(str)      --  sync xml of datasource
                    corestats(dict)   -- core stats of data source
                    last_synctime(str) -- latest sync time
                    facetstats(dict)  -- FS facet stats
                    maxstats(dict)    -- max stats details for datasource

                Returns:
                    True if success

                Raises

                    Exception on mismatch of synx xml data
        """
        self.log.info("Going to validate sync xml with core stats")
        sync_jobj = xmltodict.parse(sync_xml)
        actual_val = int(sync_jobj['StatusInfo']['@TotalDocs'])
        expected_val = int(corestats['index']['numDocs'])
        if actual_val == expected_val:
            self.log.info("Total Docs matched : " + str(actual_val))
        else:
            raise Exception("Total Docs mismatched Actaul<{0}> Expected<{1}> "
                            .format(actual_val, expected_val))

        actual_val = int(sync_jobj['StatusInfo']['@TotalSizeInBytes'])
        expected_val = int(corestats['index']['sizeInBytes'])
        if actual_val == expected_val:
            self.log.info("TotalSizeInBytes matched : " + str(actual_val))
        else:
            raise Exception("TotalSizeInBytes mismatched Actaul<{0}> Expected<{1}> "
                            .format(actual_val, expected_val))

        actual_val = int(sync_jobj['StatusInfo']['@LastModified'])
        expected_val = int(corestats['index']['userData'].get('commitTimeMSec', "0"))
        if actual_val == expected_val:
            self.log.info("LastModified matched : " + str(actual_val))
        else:
            raise Exception("LastModified mismatched Actaul<{0}> Expected<{1}> "
                            .format(actual_val, expected_val))

        statusxml_synctime = sync_jobj['StatusInfo']['@LastSyncTime']
        statusxml_synctime = statusxml_synctime[:-3]  # remove last three millisecs from it
        actual_val = int(statusxml_synctime)
        expected_val = int(last_synctime)
        if actual_val > expected_val:
            self.log.info("LastSyncTime criteria matched : " + str(actual_val))
        else:
            raise Exception("LastSyncTime criteria mismatched Actaul<{0}> Expected<{1}> "
                            .format(actual_val, expected_val))

        if maxstats is None:
            self.log.info("MaxStats is none. consider this as fresh data source")
            actual_val = int(sync_jobj['StatusInfo']['@MaxTotalDocs'])
            expected_val = int(corestats['index']['numDocs'])
            if actual_val == expected_val:
                self.log.info("MaxTotal Docs matched : " + str(actual_val))
            else:
                raise Exception("MaxTotal Docs mismatched Actaul<{0}> Expected<{1}> "
                                .format(actual_val, expected_val))

            actual_val = int(sync_jobj['StatusInfo']['@MaxTotalSizeInBytes'])
            expected_val = int(corestats['index']['sizeInBytes'])
            if actual_val == expected_val:
                self.log.info("MaxTotalSizeInBytes matched : " + str(actual_val))
            else:
                raise Exception("MaxTotalSizeInBytes mismatched Actaul<{0}> Expected<{1}> "
                                .format(actual_val, expected_val))

            statusxml_synctime = sync_jobj['StatusInfo']['@MaxTotalSyncTime']
            statusxml_synctime = statusxml_synctime[:-3]
            actual_val = int(statusxml_synctime)
            expected_val = int(last_synctime)
            if actual_val > expected_val:
                self.log.info("MaxTotalSyncTime criteria matched : " + str(actual_val))
            else:
                raise Exception(
                    "MaxTotalSyncTime criteria mismatched Actaul<{0}> Expected<{1}> "
                    .format(actual_val, expected_val))

            statusxml_synctime = sync_jobj['StatusInfo']['@MaxSizeSyncTime']
            statusxml_synctime = statusxml_synctime[:-3]
            actual_val = int(statusxml_synctime)
            expected_val = int(last_synctime)
            if actual_val > expected_val:
                self.log.info("MaxSizeSyncTime criteria matched : " + str(actual_val))
            else:
                raise Exception(
                    "MaxSizeSyncTime criteria mismatched Actaul<{0}> Expected<{1}> "
                    .format(actual_val, expected_val))

        else:
            self.log.info("Max stats response found")
            actual_val = int(sync_jobj['StatusInfo']['@MaxTotalDocs'])
            expected_val = int(maxstats['MaxTotalDocs'])
            if actual_val == expected_val:
                self.log.info("MaxTotal Docs matched : " + str(actual_val))
            else:
                raise Exception("MaxTotal Docs mismatched Actaul<{0}> Expected<{1}> "
                                .format(actual_val, expected_val))

            actual_val = int(sync_jobj['StatusInfo']['@MaxTotalSizeInBytes'])
            expected_val = int(maxstats['MaxTotalSizeInBytes'])
            if actual_val == expected_val:
                self.log.info("MaxTotalSizeInBytes matched : " + str(actual_val))
            else:
                raise Exception(
                    "MaxTotalSizeInBytes mismatched Actaul<{0}> Expected<{1}> "
                    .format(actual_val, expected_val))

            statusxml_synctime = sync_jobj['StatusInfo']['@MaxTotalSyncTime']
            statusxml_synctime = statusxml_synctime[:-3]
            actual_val = int(statusxml_synctime)
            expected_val = int(maxstats['MaxTotalSyncTime'])
            if actual_val >= expected_val:
                self.log.info("MaxTotalSyncTime criteria matched : " + str(actual_val))
            else:
                raise Exception(
                    "MaxTotalSyncTime criteria mismatched Actaul<{0}> Expected<{1}> "
                    .format(actual_val, expected_val))

            statusxml_synctime = sync_jobj['StatusInfo']['@MaxSizeSyncTime']
            statusxml_synctime = statusxml_synctime[:-3]
            actual_val = int(statusxml_synctime)
            expected_val = int(maxstats['MaxSizeSyncTime'])
            if actual_val >= expected_val:
                self.log.info("MaxSizeSyncTime criteria matched : " + str(actual_val))
            else:
                raise Exception(
                    "MaxSizeSyncTime criteria mismatched Actaul<{0}> Expected<{1}> "
                    .format(actual_val, expected_val))

        if facetstats is not None:
            self.log.info("Facetstats is not None. consider it as FS Datasource")
            actual_val = int(
                sync_jobj['StatusInfo']['TypeStatusInfo']['FileInfo']['@SourceCount'])
            expected_val = int(facetstats['Source'])
            if actual_val == expected_val:
                self.log.info("SourceCount matched : " + str(actual_val))
            else:
                raise Exception("SourceCount mismatched Actaul<{0}> Expected<{1}> "
                                .format(actual_val, expected_val))

            actual_val = int(
                sync_jobj['StatusInfo']['TypeStatusInfo']['FileInfo']['@UsersCount'])
            expected_val = int(facetstats['Users'])
            if actual_val == expected_val:
                self.log.info("UsersCount matched : " + str(actual_val))
            else:
                raise Exception("UsersCount mismatched Actaul<{0}> Expected<{1}> "
                                .format(actual_val, expected_val))

            actual_val = int(
                sync_jobj['StatusInfo']['TypeStatusInfo']['FileInfo']['@SizeAnalyzed'])
            expected_val = int(facetstats['Size'])
            if actual_val == expected_val:
                self.log.info("SizeAnalyzed matched : " + str(actual_val))
            else:
                raise Exception("SizeAnalyzed mismatched Actaul<{0}> Expected<{1}> "
                                .format(actual_val, expected_val))

            actual_val = int(
                sync_jobj['StatusInfo']['TypeStatusInfo']['FileInfo']['@FilesCount'])
            expected_val = int(facetstats['FilesCount']['count'])
            if actual_val == expected_val:
                self.log.info("FilesCount matched : " + str(actual_val))
            else:
                raise Exception("FilesCount mismatched Actaul<{0}> Expected<{1}> "
                                .format(actual_val, expected_val))

            actual_val = int(
                sync_jobj['StatusInfo']['TypeStatusInfo']['@EntitiesCount'])
            expected_val = int(facetstats['EntitiesCount']['count'])
            if actual_val == expected_val:
                self.log.info("EntitiesCount matched : " + str(actual_val))
            else:
                raise Exception("EntitiesCount mismatched Actaul<{0}> Expected<{1}> "
                                .format(actual_val, expected_val))

        return True
