#-*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for performing database related operations.

DbHelper is the only class defined in this file.

DbHelper: Class for performing operations generic to all the database iDAs


DbHelper:
    __init__()                          --  initialise object of DbHelper object

    run_backup()                        --  initiates the backup job for the specified subclient

    prepare_aux_copy_restore()          --  method to create a secondary copy in the name
    'automation_copy' for the storage policy, if it is not already existing. Once the storage
    policy is created the aux copy is run for the created secondary copy

    run_backup_copy()                   --  method to run backup copy job for given storage policy
    and return the copy precedence associated with the primary snap copy once the job is completed

    get_snap_log_backup_job()           --  returns the log backup job object if provided
    the snap backup job id

    get_backup_copy_job()               --   waits for the backup copy job to get completed and
    returns the job object associated with inline backup copy job of snap backup

    check_if_restore_from_snap_copy     -- checks if the restore job is run from snap copy or not

    check_chunk_commited()              --  checks if atleast one chunk is committed for the
    initiated job

    get_index_cache_path()              --  checks the MA logs and fetches Index cache
    location for the job specified

    get_volume_id()                     --  method to fetch the volume id associated
    with the snap backup

    synthfull_validation()              --  method to run synthfull backup job and
    validate the synthfull job

"""
import time
from cvpysdk.exception import SDKException
from AutomationUtils import logger
from AutomationUtils import database_helper
from AutomationUtils import machine
from AutomationUtils import idautils



class DbHelper(object):
    """Class for performing operations generic to all the database iDAs"""

    def __init__(self, commcell):
        """Initialize the DbHelper object.

            Args:
                commcell             (obj)  --  Commcell object

            Returns:
                object - instance of DbHelper class

        """
        self._commcell = commcell
        self._csdb = database_helper.get_csdb()
        self.log = logger.get_log()

    def run_backup(
            self,
            subclient,
            backup_type,
            inc_with_data=False):
        """Initiates the backup job for the specified subclient

            Args:
                subclient      (obj)   -- Subclient object

                backup_type    (str)   -- Backup type to perform on subclient
                                            Either FULL or INCREMENTAL

                inc_with_data  (bool)  --  flag to determine if the incremental backup
                includes data or not

            Returns:
                job            (obj)   -- Returns Job object

            Raises:
                Exception:
                    if unable to run backup job

        """
        self.log.info("#####Starting Subclient %s Backup#####", backup_type)
        if inc_with_data:
            job = subclient.backup(backup_type, inc_with_data=inc_with_data)
        else:
            job = subclient.backup(backup_type)
        self.log.info(
            "Started %s backup with Job ID: %s", backup_type, job.job_id)
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}".format(
                    backup_type, job.delay_reason
                )
            )

        self.log.info("Successfully finished %s backup job", backup_type)
        return job

    def prepare_aux_copy_restore(self, storage_policy):
        """ Method to create a secondary copy in the name 'automation_copy'
        for the storage policy, if it is not already existing.
        Once the storage policy is created the aux copy is run for the created
        secondary copy.

            Args:
                storage_policy   (str)   -- Name of the storage policy

            Returns:
                copy_precendence (str)  -- Copy precedence associated with the
                new copy created

            Raises:
                SDKException:
                    if Aux copy Job fails to run

                    if failed to get copy details from policy

        """
        storage_policy_object = self._commcell.storage_policies.get(storage_policy)
        library_name = storage_policy_object.library_name
        disk_library_object = self._commcell.disk_libraries.get(library_name)
        media_agent_name = disk_library_object.media_agents_associated[0]
        if not storage_policy_object.has_copy("automation_copy"):
            storage_policy_object.create_secondary_copy(
                "automation_copy",
                library_name,
                media_agent_name)

        job = storage_policy_object.run_aux_copy(
            "automation_copy",
            media_agent_name)

        self.log.info(
            "Started aux copy job with Job ID: %s", str(job.job_id))

        if not job.wait_for_completion():
            raise SDKException(
                'Job',
                '102',
                'Failed to run aux copy job with error: {1}'.format(
                    job.delay_reason))

        policy_copies = storage_policy_object.copies
        if policy_copies.get('automation_copy'):
            if policy_copies['automation_copy'].get('copyPrecedence'):
                return policy_copies['automation_copy']['copyPrecedence']
        raise SDKException(
            'Storage',
            '102',
            'Failed to get copy precedence from policy')

    def run_backup_copy(self, storage_policy):
        """ Method to run backup copy job for given storage policy
        and return the copy precedence associated with the primary snap
        copy once the job is completed.

            Args:
                storage_policy   (str)   -- Name of the storage policy

            Returns:
                copy_precendence (str)  -- Copy precedence associated with the
                primary snap copy created

            Raises:
                SDKException:
                    if backup copy Job fails to run

                    if failed to get copy details from policy

        """
        storage_policy_object = self._commcell.storage_policies.get(storage_policy)

        job = storage_policy_object.run_backup_copy()

        self.log.info(
            "Started backup copy job with Job ID: %s", job.job_id)

        if not job.wait_for_completion():
            raise SDKException(
                'Job',
                '102',
                'Failed to run backup copy job with error: {1}'.format(
                    job.delay_reason))

        return storage_policy_object.get_copy_precedence("primary snap")

    def get_snap_log_backup_job(self, job_id):
        """ Returns the log backup job object if provided
        the snap backup job id

            Args:
                job_id          (str)  -- snap backup job id

            Returns:
                job             (obj)  -- Job object associated with log backup

        """
        query = (
            "select jobId from JMBkpStats where subTaskId=(select subTaskId "
            "from JMBkpStats where jobid={0}) and bkpLevel='2' and appId=(select "
            "appId from JMBkpStats where jobid={0}) and jobId!={0}").format(job_id)
        while True:
            self._csdb.execute(query)
            cur = self._csdb.fetch_one_row()
            if cur[0] != '':
                return self._commcell.job_controller.get(cur[0])
            self.log.info(
                "Log backup is not completed yet. Sleeping for 30 seconds, before trying again")
            time.sleep(30)

    def get_backup_copy_job(self, job_id):
        """waits for the backup copy job to get completed and returns the job
        object associated with inline backup copy job of snap backup

            Args:
                job_id          (str)  -- snap backup job id

            Returns:
                job             (obj)  -- Job object associated with backup copy job

            Raises:
                Exception:
                    if backup copy job fails to run

        """
        common_utils = idautils.CommonUtils(self._commcell)
        backup_copy_job_id = ''
        while backup_copy_job_id == '' or backup_copy_job_id == '0':
            backup_copy_job_id = common_utils.get_backup_copy_job_id(job_id)
        self.log.info("backup copy job is started with Job ID: %s", backup_copy_job_id)
        self.log.info("Waiting for backup copy job to finish")
        backup_copy_job = self._commcell.job_controller.get(backup_copy_job_id)
        if not backup_copy_job.wait_for_completion():
            raise Exception(
                "Failed to run backup copy job with error: {0}".format(
                    backup_copy_job.delay_reason
                )
            )
        return backup_copy_job

    def check_if_restore_from_snap_copy(self, job_object, client_object=None, machine_object=None):
        """checks if the restore job is run from snap copy or not

            Args:
                job_object      (obj)       --      Job object associated with restore job

                client_object   (obj)       --      client object of client associated with the job

                    default: None

                machine_object  (obj)       --      machine object of client associated with the job

                    default: None

            Returns:
                True - if the restore job is from snap copy

                False - if restore job is not from snap copy

        """
        if client_object is None:
            client_object = self._commcell.clients.get(job_object.client_name)
        client_log_directory = client_object.log_directory
        if machine_object is None:
            machine_object = machine.Machine(client_object)
        client_log_directory = machine_object.join_path(client_log_directory, "CVMA.log")
        command = (
            "sed -n \"/%s.*SERVICE.*Received.*CVMA_VOL_SNAP_OPERATION_REQ.*/s/://gp\" %s") % (
                job_object.job_id, client_log_directory)
        if "windows" in machine_object.os_info.lower():
            command = (
                "(Select-String -Path \"%s\" -Pattern"
                " \"%s.*SERVICE.*Received.*CVMA_VOL_SNAP_OPERATION_REQ.*\")") % (
                    client_log_directory, job_object.job_id)
        output = machine_object.execute_command(command).formatted_output
        if output != '':
            return True
        return False




    def check_chunk_commited(self, job_id):
        """ Checks if atleast one chunk is committed
        for the initiated job

        Args:

            job_id               (str)  -- Job ID

        Returns:

            (bool)    --  Returns true on success/False on failure

        """
        query = (
            "select * FROM archChunkMapping where jobId = {0} and physicalSize > '0'".format(
                job_id))
        self._csdb.execute(query)
        if len(self._csdb.rows[0]) == 1 and self._csdb.rows[0][0] == '':
            return False
        return True

    def get_index_cache_path(
            self,
            job_id,
            ma_machine_object,
            backupset=None,
            version=1):
        """ Checks the MA logs and fetches Index cache
        location for the job specified

            Args:
                job_id              (str)   --  Job ID

                ma_machine_object   (obj)   --  Machine object of media agent

                backupset           (obj)   --  Backupset Object

                    default:    None

                version             (int)   -- Indexing version

                    Accepted values: 1 and 2

                    default:    1

            Returns:
                (str)     --  Returns Index cache location for the Job

            Raises:
                Exception:
                    if backupset is None when index version is 2

        """
        index_cache_location = ma_machine_object.get_registry_value(
            ma_machine_object.join_path("Machines", ma_machine_object.client_object.client_name),
            "dFSINDEXCACHE")
        self.log.info("Index cache location of MA: %s", index_cache_location)
        if version == 1:
            command = None
            log_dir_path = ma_machine_object.client_object.log_directory
            if "windows" in ma_machine_object.os_info.lower():
                command = (
                    "(Select-String -Path \"%s\CreateIndex.log\" -Pattern"
                    " \"%s.*INDEXCACHEDIR:.*job.*%s.*creating\")") % (log_dir_path, job_id, job_id)
            else:
                command = (
                    "sed -n \"/%s.*INDEXCACHEDIR.*job.*%s.*creating/s/:/ "
                    "/gp\" %s") % (job_id, job_id, log_dir_path)
            new_directory_created = None
            while True:
                new_directory_created = ma_machine_object.execute_command(command).formatted_output
                if new_directory_created != '':
                    break
            new_directory_created = new_directory_created.split("'")[1]
            self.log.info(
                "Index cache directory to be backed-up for this Job: %s",
                new_directory_created)
            return new_directory_created

        if backupset is None:
            raise Exception(
                'backup set object needs to be passed for index verison 2')
        indexing_directory_path = ma_machine_object.join_path(
            index_cache_location,
            "CVIdxLogs",
            str(self._commcell.commcell_id),
            backupset.guid,
            "J{0}".format(job_id))
        self.log.info(
            "Index cache directory to be backed-up for this Job: %s",
            indexing_directory_path)
        return indexing_directory_path

    def get_volume_id(self, snap_backup_jobid):
        """ method to fetch the volume id associated with the snap backup
            Args:
                snap_backup_jobid   (int)  -- snap backup job id

            Returns:
                volume_id           (str)  -- volume id associated with
                the snap backup

            Raise:
                Exception:
                    if unable to find volume id

        """
        query = "select SMVolumeId from SMVolume where jobid='{0}'".format(
            snap_backup_jobid)
        self._csdb.execute(query)
        cur = self._csdb.fetch_one_row()
        if cur[0] != '':
            return cur[0].strip()
        self.log.error(
            "Unable to find the volume id for given backup job")
        raise Exception("Unable to find the volume id for given backup job")

    def synthfull_backup_validation(
            self, client_object, machine_object, subclient_object, is_synthfull_loop=False):
        """ method to run synthfull backup job and validate the synthfull job

        Args:

                client_object       (obj)   --  Client object

                machine_object      (obj)   --  Machine class object

                subclient_object    (obj)   --  Subclient Object

                is_synthfull_loop   (bool)  --  flag to determine if this is the
                synthfull loop testcase

                    default: False

        Raise:
            Exception:
                if failed to run mount job

                if unable to verify the mounted snap

                if failed to run unmount job

        """
        ##### run synthfull backup jobs ######
        self.log.info("Starting synthetic full backup.")
        synth_job = self.run_backup(subclient_object, "synthetic_full")
        self.log.info("Synthetic full backup %s is finished", synth_job.job_id)

        self.log.info(
            ("Running data aging on storage policy:%s copy:primary to "
             "make sure the restore is triggered from Synthfull backup"),
            subclient_object.storage_policy)

        common_utils = idautils.CommonUtils(self._commcell)
        data_aging_job = common_utils.data_aging(
            subclient_object.storage_policy, "primary", False)
        if not data_aging_job.wait_for_completion():
            raise Exception(
                "Failed to run data aging job with error: {0}".format(
                    data_aging_job.delay_reason
                )
            )
        self.log.info("Dataaging job run is:%s", data_aging_job.job_id)

        if is_synthfull_loop:
            return
        ######### Synth full validation #########
        self.log.info("Mounting the snap in client at:/tmp/testcase_mount")
        volume_id = int(self.get_volume_id(synth_job.job_id))
        self.log.info("Volume ID: %s", volume_id)
        # mount the snap in the client
        array_mgmt_object = self._commcell.array_management
        machine_object.execute_command("mkdir -p /tmp/testcase_mount")
        mount_job = array_mgmt_object.mount(
            [[volume_id]],
            client_object.client_name,
            "/tmp/testcase_mount")
        self.log.info(
            "Mounting the snapshot in the client with job:%s",
            mount_job.job_id)
        if not mount_job.wait_for_completion():
            raise Exception(
                "Failed to run mount job with error: {0}".format(
                    mount_job.delay_reason
                )
            )
        self.log.info("Succesfully mounted the snapshot in the client")

        # Validate the data
        # Run find command to check if the data is corrupted
        command = ("find /tmp/testcase_mount -type f -print "
                   "-exec cp -r {} /dev/null \; > readsnapdata")
        output = machine_object.execute_command(command)
        if output.exception_message != '':
            raise Exception("Unable to verify the mounted snap from synhfull job")
        self.log.info("Synthfull job is verified")

        # Unmount the snap
        self.log.info("Unmounting snapshot")
        unmount_job = array_mgmt_object.unmount(
            [[volume_id]])
        self.log.info(
            "UnMounting the snapshot in the client with job:%s",
            unmount_job.job_id)
        if not unmount_job.wait_for_completion():
            raise Exception(
                "Failed to run unmount job with error: {0}".format(
                    unmount_job.delay_reason
                )
            )
        self.log.info("Snapshot is unmounted")
