# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Main Module for setting input variables and creating objects of all other modules.
This module is imported in any test case.
You need to create an object of this module in the test case.

Splunk: Class for initializing input variables and other module objects.

Splunk
======

    __init__()                          --  initialize object of Splunk class

    populate_tc_inputs()                --  populates test case inputs in splunk object

    validate_bigdata_app_list()         --  validates if the client exists in Big Data Apps List

    validate_client_parameters()        --  validates if the client parameters such as
    master uri, master_node,user_name and plan are set as required

    make_request()                      --  common module to make splunk related REST API calls

    splunk_bundle_push()                --   pushes the latest Splunk bundle to slave nodes

    splunk_nodes_health_check()         --  checks if all the slaves are up

    splunk_rolling_restart()            --  initiates a rolling restart over the Splunk cluster
    and waits until all the slave nodes restart

    add_splunk_index()                  --  creates a new splunk index

    edit_splunk_index()                 --  edits the events associated with the index_obj

    delete_index()                      --  deletes Splunk index by changing indexes.conf
    on master node and performing bundle push and rolling restart

    cleanup_index()                     --  cleans up the created index for the test-case
    as a part of cleanup job

    make_after_restore_configuration()  --  writes appropriate content into indexes.conf file
    of master node after successful restore job

"""

from __future__ import unicode_literals
import time
import calendar
from http import HTTPStatus
import requests
import xmltodict
import splunklib.client as splunk_client
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.windows_machine import WindowsMachine
from AutomationUtils.unix_machine import UnixMachine
from AutomationUtils.machine import Machine
from Database.SplunkApplication.operations import CvOperation


class Splunk():
    """Class for initializing input variables"""

    def __init__(self, tc_object):
        """
            Initializes the input variables,logging and creates object from other modules.

            Args:
                tc_object   --  instance of testcase class

            Returns:
                object  --  instance of splunk class object
        """
        self.tc_object = tc_object
        self.log = self.tc_object.log
        self.app_name = self.__class__.__name__
        self.csdb = self.tc_object.csdb
        self.tcinputs = tc_object.tcinputs
        self.commcell = tc_object.commcell
        self.cvpysdk_object = self.commcell._cvpysdk_object
        self.new_client_name = ""
        self.master_node = ""
        self.master_uri = ""
        self.user_name = ""
        self.password = ""
        self.plan = ""
        self.populate_tc_inputs(tc_object)
        self.cvoperations = CvOperation(self)

    def populate_tc_inputs(self, tc_object):
        """
        Initializes all the test case inputs after validation

        Args:
            tc_object (obj)    --    Object of testcase

        Raises:
            Exception:
                if a valid CVTestCase object is not passed.

                if CVTestCase object doesn't have agent initialized
        """
        if not isinstance(tc_object, CVTestCase):
            raise Exception("Valid test case object must be passed as argument")
        self.new_client_name = tc_object.tcinputs.get("NewClientName")
        self.master_node = tc_object.tcinputs.get("MasterNode")
        self.master_uri = tc_object.tcinputs.get("MasterUri")
        self.user_name = tc_object.tcinputs.get("UserName")
        self.password = tc_object.tcinputs.get("Password")
        self.plan = tc_object.tcinputs.get("Plan")
        self.log = tc_object.log

    def validate_bigdata_app_list(self, client_obj):
        """
        Validates if the client exists in Big Data Apps List

        Args:
            client_obj              (Object)    --      splunk client object

        Returns:
            instance ID
        """
        agent_obj = client_obj.agents
        req_agent = agent_obj.get("big data apps")
        instance_obj = req_agent.instances
        all_instances = instance_obj.all_instances

        if self.new_client_name.lower() in all_instances.keys():
            instance_id = all_instances[self.new_client_name.lower()]
            return instance_id

        raise Exception("Failed to find required client in big data entities")

    def validate_client_parameters(self, instance_id, client_object):
        """
        Validates if the client parameters such as master uri, master_node,user_name
        and plan are set as required.

        Args:
           instance_id    (int)  --  instance ID of an instance associated with an agent

           client_object  (obj)  --  client object of the newly created splunk client

        """
        instance_id = int(instance_id)
        agent_obj = client_object.agents
        req_agent = agent_obj.get("big data apps")
        instance_obj = req_agent.instances
        req_instance = instance_obj.get(instance_id)
        instance_prop = req_instance.properties
        master_uri = instance_prop["distributedClusterInstance"]["clusterConfig"] \
            ["splunkConfig"]["url"]
        master_node = instance_prop["distributedClusterInstance"]["clusterConfig"] \
            ["splunkConfig"]["primaryNode"]["entity"]["clientName"]
        user_name = instance_prop["distributedClusterInstance"]["clusterConfig"] \
                     ["splunkConfig"]["splunkUser"]["userName"]

        if(master_uri != self.master_uri or \
                master_node != self.master_node or \
                user_name != self.user_name):
            raise Exception("Client Parameter Validation Failed")

        backupset_obj = req_agent.backupsets
        req_backupset = backupset_obj.get("defaultbackupset")
        subclients_obj = req_backupset.subclients
        req_subclient = subclients_obj.get("default")
        subclient_prop = req_subclient.properties
        if subclient_prop["planEntity"]["planName"] != self.plan:
            raise Exception("Client Parameter Validation Failed")

    def make_request(self, url, method):
        """
        Common module to make splunk related REST API calls

        Args:
            url             (str)   -- url of the REST API

            method          (str)   --  method type of the request

        Returns:
            returned_resp   (obj)   --  response object associated with the request
        """

        if method == "POST":
            returned_resp = requests.post(
                url, auth=(self.user_name, self.password), verify=False
            )

        elif method == "GET":
            returned_resp = requests.get(
                url, auth=(self.user_name, self.password), verify=False
            )

        else:
            raise Exception("Request Method Not supported")

        if returned_resp.status_code == HTTPStatus.UNAUTHORIZED:
            raise Exception("Unauthorized Splunk Request")

        if returned_resp.status_code == HTTPStatus.OK and returned_resp.ok:
            return returned_resp

        raise Exception("Splunk Request Failed")

    def splunk_bundle_push(self):
        """
        Pushes the latest Splunk bundle to slave nodes

        Args:
            Nothing

        """

        self.log.info("Starting Splunk Bundle Push")
        returned_response = self.make_request(
            self.master_uri+"/services/cluster/master/control/default/apply",
            "POST"
        )
        if returned_response.status_code != 200:
            raise Exception("Splunk Bundle Push Failed")

        time.sleep(60)
        self.log.info("Splunk Bundle Push Successful")

    def splunk_nodes_health_check(self):
        """
        Checks if all the slave nodes are up.

        Args:
            Nothing

        Exception:
            Raises exception if slave nodes do not come up within 300 secs
        """
        health_uri = self.master_uri + "/services/cluster/master/health"

        self.log.info("Waiting For All Slave Nodes To Be Up")

        returned_response = self.make_request(health_uri, "GET")
        if returned_response.status_code != 200:
            raise Exception("Failed To Check Splunk Health")

        all_peers_up = False
        total_time_elapsed = 0

        up_result_count = 0

        while (not all_peers_up and total_time_elapsed <= 300):
            returned_response = self.make_request(health_uri, "GET")
            parse_data = xmltodict.parse(returned_response.text)
            key_value_list = parse_data["feed"]["entry"]["content"]["s:dict"]["s:key"]
            for i in range(len(key_value_list)):
                req_dict = dict(parse_data["feed"]["entry"]["content"]["s:dict"]["s:key"][i])
                if (req_dict["@name"] == "all_peers_are_up" and req_dict["#text"] == "1"):
                    up_result_count = up_result_count + 1
                    if up_result_count >= 3:
                        all_peers_up = True
                        break

            if not all_peers_up:
                self.log.info("Some Nodes Are Down/Restarting")
                time.sleep(15)
                total_time_elapsed = total_time_elapsed + 15

        if not all_peers_up:
            raise Exception("Failed To Restart Slave Nodes")

        self.log.info("All Slave Nodes Are Up")

    def splunk_rolling_restart(self):
        """
        Initiates a rolling restart over the Splunk cluster and
        waits until all the slave nodes restart

        Args:
             Nothing

        """

        self.log.info("Starting Rolling Restart")
        returned_response = self.make_request(
            self.master_uri+"/services/cluster/master/control/control/restart",
            "POST"
        )

        if returned_response.status_code != 200:
            raise Exception("Splunk Rolling Restart Failed")

        time.sleep(15)
        self.splunk_nodes_health_check()

    def add_splunk_index(self):
        """
        Creates new Splunk index

        Args:
            Nothing

        Returns:
              Splunk index object
        """
        self.log.info("Starting To Add New Splunk Index")
        time_stamp = str(calendar.timegm(time.gmtime()))
        index_name = "testindex_" + time_stamp
        slave_username = self.tcinputs.get("Slave1SplunkUsername")
        slave_password = self.tcinputs.get("Slave1SplunkPassword")
        slave_ip = self.tcinputs.get("Slave1Ip")
        slave_port = self.tcinputs.get("Slave1Port")
        master_name = self.tcinputs.get("MasterNode")
        machine_obj = Machine(machine_name=master_name, commcell_object=self.commcell)
        splunk_home_path = self.tcinputs.get("SplunkHomePath")

        if machine_obj.os_info.lower() == "windows":
            windows_machine_obj = WindowsMachine(machine_name=master_name,
                                                 commcell_object=self.commcell)
            file_path = splunk_home_path + "\\etc\\master-apps\\_cluster\\local\\indexes.conf"
            index_stanza = "[" + index_name + "]\r\n"
            index_stanza = index_stanza + "coldPath = `$" + "SPLUNK_DB\\" + \
                           index_name + "\\colddb\r\n"
            index_stanza = index_stanza + "homePath = `$" + "SPLUNK_DB\\" + \
                           index_name + "\\db\r\n"
            index_stanza = index_stanza + "thawedPath = `$" + "SPLUNK_DB\\" + \
                           index_name + "\\thaweddb\r\n"
            index_stanza = index_stanza + "repFactor = auto\r\n"
            windows_machine_obj.append_to_file(file_path, index_stanza)

        else:
            unix_machine_obj = UnixMachine(machine_name=master_name,
                                           commcell_object=self.commcell)
            file_path = splunk_home_path + "/etc/master-apps/_cluster/local/indexes.conf"
            index_stanza_dict = {
                "index_stanza1": "[" + index_name + "]",
                "index_stanza2": "homePath = \\$" + "SPLUNK_DB/" + index_name + "/db",
                "index_stanza3": "coldPath = \\$" + "SPLUNK_DB/" + index_name + "/colddb",
                "index_stanza4": "thawedPath = \\$" + "SPLUNK_DB/" + index_name + "/thaweddb",
                "index_stanza5": "repFactor = auto"
            }
            for i in index_stanza_dict:
                unix_machine_obj.append_to_file(file_path, index_stanza_dict[i])

        self.splunk_bundle_push()
        time.sleep(15)
        self.splunk_nodes_health_check()
        self.log.info("New Index Added Successfully With Name %s", index_name)
        self.log.info("Populating Index %s", index_name)
        service = splunk_client.connect(
            host=slave_ip, port=slave_port,
            username=slave_username, password=slave_password
        )
        index_obj = service.indexes[index_name]

        for i in range(100):
            index_obj.submit("This event from automation" + \
                             str(i), sourcetype="custom", host="remote_host")

        time.sleep(30)
        self.log.info("Populating Index %s Completed Successfully", index_name)
        index_obj = service.indexes[index_name]
        return index_obj

    def edit_splunk_index(self, index_name):
        """
        Edits the events associated with the index_obj

        Args:
            index_name   (obj)   --  Splunk index object

        """
        self.log.info("Starting To Edit Splunk Index")
        slave_username = self.tcinputs.get("Slave1SplunkUsername")
        slave_password = self.tcinputs.get("Slave1SplunkPassword")
        slave_ip = self.tcinputs.get("Slave1Ip")
        slave_port = self.tcinputs.get("Slave1Port")
        service = splunk_client.connect(host=slave_ip, port=slave_port,
                                        username=slave_username, password=slave_password)
        index_obj = service.indexes[index_name]
        self.log.info("Adding New Events To The Index")
        for i in range(10):
            index_obj.submit("This is a modified event" + \
                             str(i), sourcetype="custom", host="remote_host")
        time.sleep(180)
        self.log.info("Successfully Completed Editing Splunk Index")

    def delete_index(self):
        """
        Deletes Splunk index by changing indexes.conf on master node
        and performing bundle push and rolling restart

        Args:
            Nothing

        """

        self.log.info("Starting Index Deletion")
        master_name = self.tcinputs.get("MasterNode")
        machine_obj = Machine(machine_name=master_name, commcell_object=self.commcell)
        splunk_home_path = self.tcinputs.get("SplunkHomePath")

        if machine_obj.os_info.lower() == "windows":
            windows_machine_obj = WindowsMachine(machine_name=master_name,
                                                 commcell_object=self.commcell)
            file_path = splunk_home_path + "\\etc\\master-apps\\_cluster\\local\\indexes.conf"
            windows_machine_obj.append_to_file(file_path, "\ndeleted=true")

        else:
            unix_machine_obj = UnixMachine(machine_name=master_name,
                                           commcell_object=self.commcell)
            file_path = splunk_home_path + "/etc/master-apps/_cluster/local/indexes.conf"
            unix_machine_obj.append_to_file(file_path, "deleted=true")

        self.splunk_bundle_push()
        time.sleep(15)
        self.splunk_nodes_health_check()
        self.splunk_rolling_restart()
        self.log.info("Index Deletion Successful")

    def cleanup_index(self, index_name):
        """
         Cleans up the created index for the testcase as a part of cleanup job

         Args:
             index_name     (str)   --  Name associated with the index which
             we want to cleanup
        """
        self.log.info("Starting Index Cleanup")
        master_name = self.tcinputs.get("MasterNode")
        machine_obj = Machine(machine_name=master_name, commcell_object=self.commcell)
        splunk_home_path = self.tcinputs.get("SplunkHomePath")

        if machine_obj.os_info.lower() == "windows":
            windows_machine_obj = WindowsMachine(machine_name=master_name,
                                                 commcell_object=self.commcell)
            file_path = splunk_home_path + "\\etc\\master-apps\\_cluster\\local\\indexes.conf"
            index_stanza = "[" + index_name + "]\r\n"
            index_stanza = index_stanza + "coldPath = `$" + "SPLUNK_DB\\" + \
                           index_name + "\\colddb\r\n"
            index_stanza = index_stanza + "homePath = `$" + "SPLUNK_DB\\" + \
                           index_name + "\\db\r\n"
            index_stanza = index_stanza + "thawedPath = `$" + "SPLUNK_DB\\" + \
                           index_name + "\\thaweddb\r\n"
            index_stanza = index_stanza + "repFactor = auto\r\n"
            index_stanza = index_stanza + "deleted = true\r\n"
            windows_machine_obj.append_to_file(file_path, index_stanza)

        else:
            unix_machine_obj = UnixMachine(machine_name=master_name,
                                           commcell_object=self.commcell)
            file_path = splunk_home_path + "/etc/master-apps/_cluster/local/indexes.conf"
            index_stanza_dict = {
                "index_stanza1": "[" + index_name + "]",
                "index_stanza2": "homePath = \\$" + "SPLUNK_DB/" + index_name + "/db",
                "index_stanza3": "coldPath = \\$" + "SPLUNK_DB/" + index_name + "/colddb",
                "index_stanza4": "thawedPath = \\$" + "SPLUNK_DB/" + index_name + "/thaweddb",
                "index_stanza5": "repFactor = auto",
                "index_stanza6": "deleted = true"
            }
            for i in index_stanza_dict:
                unix_machine_obj.append_to_file(file_path, index_stanza_dict[i])

        self.splunk_bundle_push()
        time.sleep(15)
        self.splunk_nodes_health_check()
        self.splunk_rolling_restart()
        self.log.info("Index Cleanup Successful")


    def make_after_restore_configuration(self):
        """
        Writes appropriate content into indexes.conf file of master node
        after successful restore job

        Args:
            Nothing
        """

        self.log.info("Starting After Restore Configuration")
        master_name = self.tcinputs.get("MasterNode")
        machine_obj = Machine(machine_name=master_name, commcell_object=self.commcell)
        splunk_home_path = self.tcinputs.get("SplunkHomePath")

        if machine_obj.os_info.lower() == "windows":
            windows_machine_obj = WindowsMachine(machine_name=master_name,
                                                 commcell_object=self.commcell)
            file_path = splunk_home_path + "\\etc\\master-apps\\_cluster\\local\\indexes.conf"
            windows_machine_obj.append_to_file(file_path, "\r\ndeleted=false")

        else:
            unix_machine_obj = UnixMachine(machine_name=master_name,
                                           commcell_object=self.commcell)
            file_path = splunk_home_path + "/etc/master-apps/_cluster/local/indexes.conf"
            unix_machine_obj.append_to_file(file_path, "\r\n")
            unix_machine_obj.append_to_file(file_path, "deleted=false")

        self.splunk_bundle_push()
        time.sleep(60)
        self.splunk_nodes_health_check()
        self.splunk_rolling_restart()
        time.sleep(30)
        self.log.info("After Restore Configuration Successful")
