# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright  Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""SAPOraclehelper file for performing SAP Oracle operations

SAPOraclehelper is the only class defined in this file

SAPOraclehelper: Helper class to perform SAP Oracle operations

SAPOraclehelper:
    __init__()                            --  initializes SAP Oracle helper object

    get_saporacle_db_connectpassword()    --  gets the oracle connect password from cs db

    db_connect()                          --  Connect to Oracle db using cx_Oracle using
                                              SAPOracle class object

    findtablespace()                      --  Connects to Oracle db and finds the
                                              the specified tablespace exists or not.

    findtablename()                       --  Connects to Oracle db and finds the
                                              the specified table name exists or not.

    GetDatabaseState()                    --  gets the database state from the database

    droptablespace()                      --  Connects to Oracle db and finds the
                                              the specified tablespace exists or not.
                                              if finds drops the tablespace.useful for
                                              validation after restore and also for data
                                              creation while running TC

    getjobbackupsize()                    --  Gets the backup size of the completed job status
                                                from jmbkpstats table totalUncompBytes
                                                is the application size
    getjobdatabasephasebackupsize()       --  Gets the backup size of the completed job status
                                              from jmbkatmppstats table totalUncompBytes
                                              is the application size

    switchlogfile()                       --  Creates some archive logs

    getdatafile()                         --  gets the data file path for creating
                                              tablespaces from database

    create_test_tables()                  --  creates test tablespace and tables
                                              in the database

    createStopSqlFile()                   --  creates stop.sql needed for automation
                                                 test cases

    gracefulDBShutDown()                  --  creates shutdown.sql needed for automation
                                              test cases

    test_tables_validation()              --  validates the test tables that were restored

    touchfilesonclient()                  --  keeps hook files on clients whick are
                                              useful for making job goes to pending
                                              at differemt phases for Sap oracle
                                              
    run_backup()                          -- Runs backup function
    
    run_restart_backup()                  -- suspendes and resumes backup job

    run_pending_backup()                  -- run make job go to pending function

    run_client_service_restart_backup()   -- run client restart function

    run_kill_process_backup()             -- kills client process for running backup job

    run_random_kill_process()             -- randomly kills running client process
    
    run_kill_process_backupcopy()         -- randomly kills client process for backupcopy job
    
    restart_backupcopy()                  -- restarts running backup copy job
    
    run_client_service_restart_backupcopy -- restarts running backup copy client process
    
    

"""
import os
from random import randint
import time
from past.builtins import basestring
from AutomationUtils import logger, constants, idautils
from AutomationUtils import database_helper
from AutomationUtils.database_helper import get_csdb
from AutomationUtils.database_helper import SAPOracle
from AutomationUtils import cvhelper
from AutomationUtils.machine import Machine
from AutomationUtils.interruption import Interruption



class SAPOraclehelper(object):
    """Helper class to perform SAPOracle operations"""

    def __init__(self, commcell, instance):
        """Initializes SAPOraclehelper object
        Args:
            commcell    (obj)   -- commcell object to connect to

            instance    (obj)   -- instance object to connect to

        """

        #self.testcase = test_case_obj
        self.log = logger.get_log()
        self.log.info('  Initializing SAPOracle Helper ...')
        self._commcell = commcell
        self._instance = instance
        self._csdb = get_csdb()
        self._saporacle_client = instance._agent_object._client_object.client_name
        self.log.info("Client is "+self._saporacle_client)
        self._saporacle_clientid = instance._agent_object._client_object._get_client_id()
        self._jobresultsdir = instance._agent_object._client_object.job_results_directory
        self.log.info("client is "+self._saporacle_clientid)
        self.log.info("jobresults directlty is "+self._jobresultsdir)
        self._clienthostname = instance._agent_object._client_object.client_hostname
        self.log.info("client hostname  is "+self._clienthostname)
        self._saporacle_instanceid = instance.saporacle_instanceid
        self._storage_policy_name1 = self._instance.log_sp
        self._saporacle_db_user = instance.saporacle_db_user
        self.log.info("sap oracle db user name is "+self._saporacle_db_user)
        self._saporacle_db_connectstring = instance.saporacle_db_connectstring
        self.log.info("sap oracle db connect string  is "+self._saporacle_db_connectstring)
        self._saporacle_instanceid = instance.saporacle_instanceid
        self._saporadb = None
        self._connection = None
        self._cursor = None
        self.storage_policy_name = commcell.storage_policies.get(self._storage_policy_name1)
        


    @property
    def saporacle_db_connectpassword(self):
        """Gets the sql connect password of the instance"""
        return self._saporacle_db_connectpassword

    def getsaporacle_db_connectpassword(self):
        """Gets the sql connect password of the instance
        Args:

            saporacle_instanceid   (str)         --  Gets the clientid from SAPOracle instance class
                                                          and passit using Testcase calling
        Raises:
                Exception:

                    if failed to get Oracle db connect password from Commserver database

        """
        try:

            query = "Select attrVal from app_instanceprop where componentNameId = {0}\
            and attrName = 'SQL Connect Password'".format(str(self._saporacle_instanceid))
            self.log.info("Cs db query for getting sql connect password is "+query)
            self._csdb.execute(query)
            cur = self._csdb.fetch_one_row()
            if cur:
                password = cur[0]
                self._saporacle_db_connectpassword = cvhelper.format_string(self._commcell,\
                                                                            password)
                #self.log.info("connect password we got is "+str(self._saporacle_db_connectpassword))
                return self._saporacle_db_connectpassword
                #self.log.info("connect password we got is "+str(self._saporacle_db_connectpassword))
            else:
                self.log.info("there is some issue while getting connect password from cs db")
                return None

        except Exception as exp:
            raise Exception("failed to get sqlplus connect password from cs db "+str(exp))

    def db_connect(self):
        """TODO: Doc String for db_connect"""
        self._saporadb = database_helper.SAPOracle(self._saporacle_db_user,\
                                  self._saporacle_db_connectpassword,\
                                  self._saporacle_db_connectstring)

    def findtablespace(self, tablespace):
        """connects to oracledb and finds the tablespaces
        Args:

            tablespace   (str)         --  Specify tablespace name
                                             and passit using Testcase calling
        Raises:
                Exception:

                    Failed to find tablespace\nError
        """
        try:
            query = "select tablespace_name from dba_tablespaces"
            self.log.info("query running is "+query)
            response = self._saporadb.execute(query)
            self.log.info("tablespaces names we got is "+str(response))
            tablespaces = response.rows
            if str(tablespaces).find(tablespace) >= 0:
                self.log.info("tablespace name exists: {0} tablespace ".\
                              format(tablespace))
                return 0
            else:
                return 1
        except Exception as excp:
            raise Exception('Failed to find tablespace\nError: {0}'.format(excp))

    def findtablename(self, tablename):
        """connects to oracledb and finds the table name
         Args:

            tablename   (str)         --  Specify tablename and
                                            passit using Testcase calling
        Raises:
                Exception:

                    Failed to find tablename\nError
        """
        try:
            query = "Select table_name from dba_tables where table_name='{0}'"\
                                                        .format(tablename)
            self.log.info("query running is "+query)
            response = self._saporadb.execute(query)
            self.log.info("tablespaces names we got is "+str(response))
            tablespaces = response.rows
            if str(tablespaces).find(tablename) >= 0:
                self.log.info("table name exists: {0} tablename ".\
                              format(tablename))
                return 0
            else:
                return 1
        except Exception as excp:
            raise Exception('Failed to find tablename\nError: {0}'.format(excp))

    def getdatabasestate(self):
        """connects to oracledb and gets the database status

        """
        try:
            query = "Select status from v$instance"
            self.log.info("query running is "+query)
            response = self._saporadb.execute(query)
            self.log.info("response we got is "+str(response))
            row = response.rows
            self.log.info(row[0])
            return str(row[0])
        except Exception as excp:
            raise Exception("failed to get database status: {0} ".format(excp))

    def droptablespace(self, tablespace):
        """connects to oracledb and finds the tablespaces.
        if the tablespace specified exists drops the tablespace
        Args:

            tablespace   (str)         --  Specify tablespace name
                                             and passit using Testcase calling
        Raises:
                Exception:

                    Failed to drop tablespace\nError
        """
        try:
            status = self.findtablespace(tablespace)
            if status == 0:
                self.log.info("tablespace exists we are going to drop tablespace ")
                query = "drop tablespace " + tablespace + \
                " including contents and datafiles"
                self.log.info("query running is "+query)
                response = self._saporadb.execute(query)
                return 0
            else:
                return 1

        except Exception as excp:
            raise Exception('Failed to drop tablespace\nError: {0}'.format(excp))


    def getjobbackupsize(self, jobid):
        """Gets the backup size of the completed job status
        from jmbkpstats table totalUncompBytes is the application size
        Args:

            jobid        (str)         --  Gets the jobid from job class
                                             and pass it using Testcase calling
        Raises:
                Exception:

                    if failed to get currentphase of the running job from Commserver database

        """
        try:
            query = "select sum(UncompBytes)/1024/1024  from JMBkpAtmptStats \
            where jobid={0} and phase=4".format(str(jobid))
            self.log.info("Cs db query for getting application size job is "+query)
            self._csdb.execute(query)
            cur = self._csdb.fetch_one_row()
            if cur:
                self.log.info(cur[0])
                return cur[0]
            else:
                self.log.info("there is some issue while getting application size from cs db")
                return None

        except Exception as exp:
            raise Exception("failed to get currentphase of the running job from cs db "+str(exp))

    def getjobdatabasephasebackupsize(self, jobid):
        """Gets the backup size of the completed job status
        from jmbkpstats table totalUncompBytes is the application size
        Args:

            jobid        (str)         --  Gets the jobid from job class
                                             and pass it using Testcase calling
        Raises:
                Exception:

                    if failed to get currentphase of the running job from Commserver database
        """
        try:
            query = "select sum(UncompBytes)/1024/1024  from JMBkpAtmptStats where jobid={0}\
            and phase=4 and status=1".format(str(jobid))
            self.log.info("Cs db query for getting application size for \
            database backup phase "+query)
            self._csdb.execute(query)
            cur = self._csdb.fetch_one_row()
            if cur:
                self.log.info(cur[0])
                return cur[0]
            else:
                self.log.info("there is some issue while getting application size from cs db")
                return None

        except Exception as exp:
            raise Exception("failed to get for database backup phase job from cs db "+str(exp))

    def switchlogfile(self, lognumber):
        """Creates some archivelogs before backup.
        Args:

            lognumber(int)             -- Specify the number of logs to be created in
                                                          TC calling

        Raises:
                Exception:

                    if Could not delete the test tables
        """
        try:

            self.log.info("##generating archive logs starts here##")
            for count in range(0, int(lognumber)):
                query = ("alter system switch logfile")
                self.log.info(query)
                response = self._saporadb.execute(query)
                self.log.info("switch log file is sucessfull")
            return 0
        except Exception as exp:
            raise Exception("Could not delete the test tables. "+ str(exp))

    def getdatafile(self, tocreate):
        """gets the dtafilepath by connecting to oracle database
        Args:

            tocreate               (str)            --  pass the tablespace Name.
                                                       If none datafile is not created

        Raises:
                Exception:

                    if failed to get datafile path
        """
        try:
            #self._db_connect = self._saporadb
            query = "select name from v$datafile"
            self.log.info(query)
            response = self._saporadb.execute(query)
            row = response.rows
            self.log.info(row)
            firstrow = row[0]
            firstrow = str(firstrow).lstrip("('")
            self.log.info(firstrow)
            i = len(firstrow) -1
            self.log.info(i)
            while i >= 0:
                if firstrow[i] != "/" and firstrow[i] != "\\":
                    dbfile = firstrow[0:i]
                    self.log.info("firstrow we got is: {0} dbfile ".format(str(dbfile)))
                else:
                    break
                i = i-1

            if tocreate == None:
                dbfile = str(dbfile)
                return dbfile
            else:

                dbfile = (str(dbfile)+tocreate)
                self.log.info("Datafile path we got is:  {0} dbfile ".format(str(dbfile)))
            return dbfile
        except Exception as exp:
            raise Exception("Could not get datafile path: {0} dbfile ".format(str(exp)))

    def create_test_tables(self, dbfile, tablespace, tablename, flagcreatetableSpace):
        """ Creates the test tablespace,tables in the source database
        Args:

            dbfile                        (str)     -- Specify the Datafile path we got from
                                                         getdatafile function

            flagcreatetableSpace        (bool)    -- Takes True or False.
                                                        if True Tablespace will be created
                                                        if False Tablespace will not be created

        Raises:
                Exception:

                    if not able to create test tables

        """
        try:
            status = self.findtablespace(tablespace)
            if status == 0:
                self.log.info("Tablespace name exists..so dropping tablespace")
                status = self.droptablespace(tablespace)
            query = "create tablespace " + tablespace + " datafile '"+\
            dbfile + "' size 10M reuse"
            self.log.info("query running is "+query)
            response = self._saporadb.execute(query)
            self.log.info(response)
            self.log.info("query running is "+query)
            status = self.findtablename(tablename)
            self.log.info("query running is "+query)
            query = "create table " + tablename + " (name varchar2(30), ID number)" +\
                                                     " tablespace " + tablespace

            response = self._saporadb.execute(query)
            self.log.info("create table response we got is"+str(response))
            for count in range(0, 1000):
                #log.info("query running is "+query)
                query = ("insert into " +tablename + " values('" + tablename+str(count)+ "'," \
                                                               +  str(count) + ")")
                response = self._saporadb.execute(query, commit=True)
            return 0
        except Exception as exp:
            raise Exception("failed to create tablespace and table: {0} exp ".format(str(exp)))

    def createstopsqlfile(self):
        """
        creates startup no script for oracle database
        """
        try:
            #(CommonDirPath) = SAPOraclehelper.get_CommonDirPath()
            (commondirpath) = constants.TEMP_DIR
            self.log.info(commondirpath)
            fd = os.path.join(commondirpath, 'stop.sql')
            fd = open('stop.sql', 'w')
            fd.write('startup nomount;\n')
            fd.write('exit;\n')
            fd.close()
            self.log.info("Created stopsql file successfully")
            return(0, True)
        except Exception as exp:
            raise Exception("failed to createStopSqlFile: {0} exp ".format(str(exp)))

    def gracefuldbshutdown(self):
        """
        script to shutdown the oracle database
        """
        try:
            (commondirpath) = constants.TEMP_DIR
            fd = os.path.join(commondirpath, 'DBShutDown.sql')
            fd = open('DBShutDown.sql', 'w')
            self.log.info(fd)
            fd.write('shutdown immediate;\n')
            self.log.info(fd)
            fd.write('exit;\n')
            fd.close()
            self.log.info("Created DBShutDown file successfully")
            return(0, True)
        except Exception as exp:
            raise Exception("failed to gracefulDBShutDown:  {0} exp ".format(str(exp)))


    def test_tables_validation(self, tablespace, tablename):
        """ Validates the test tables that were created before the backup

        Args:
            tablespace_name         (str)     -- Tablespace name to be created specified in
                                                          TC calling

            tablename               (str)     -- Table name to be created specified in
                                                          TC calling
        Raises:
                Exception:

                    if Failed to validate the source and restored test tables
        """
        try:
            self.table_count = 1000
            status = self.findtablespace(tablespace)
            if status == 0:
                self.log.info("tablespace name exists.Restore tablespace is sucessfull")
            else:
                self.log.error("there is some issue with restoring tablespace")
            status = self.findtablename(tablename)
            if status == 0:
                self.log.info("tablen name exists.Restore table is sucessfull")
            else:
                self.log.error("there is some issue with restoring table")
            query = "select  * from "+(tablename)+" order by "+(tablename)+".ID ASC"
            self.log.info(query)
            response = self._saporadb.execute(query)
            row_count_of_tables = response.rows
            self.log.info(row_count_of_tables)
            self.log.info(row_count_of_tables[0])
            if len(row_count_of_tables) == 0:
                raise Exception("Could not obtain the row count of all the tables in the schema")
            else:
                self.log.info(len(row_count_of_tables))
            if len(row_count_of_tables) != self.table_count:
                raise Exception("The restored table does not contain all the rows")
            else:
                self.log.info(len(row_count_of_tables))
                return 0

        except Exception as exp:
            raise Exception("Failed to validate the source and restored test tables: {0} exp ".\
                            format(str(exp)))

    def getarchfileisvalid(self, jobid, filetype):
        """Gets the archfile is valid from Cs db
        Args:

            jobid       (str)           --  Gets the job id from jobclass
                                            pass it using TC calling

            filetype     (str)          --  Based on the archive type verifies
                                            cs db whether file is marked invalid or not
                                            Filetype 1 means data
                                            filetype 4 means log

        Raises:
                Exception:

                    if Failed to run query from cs db
        """

        try:
            query = "Select isvalid from archfile where jobid="+(str(jobid))+" \
            and isvalid=-1 and filetype="+filetype+""
            self.log.info("Cs db query for getting archivefile "+query)
            self._csdb.execute(query)
            cur = self._csdb.fetch_all_rows()
            if cur:
                cur1 = str(cur[0])
                self.log.info(cur1)
                if str(cur1) >= '-1':
                    self.log.info("Archive files are invalidated for the restarted job")
                    return 0
                else:
                    return -1
        except Exception as exp:
            raise Exception("failed to run query from cs db: {0} exp ".format(str(exp)))
            
    def run_backup(self, subclient, backup_type):
        """Starts backup job
        Args:

             subclient(str)     -- Specify the subclient object name where backups
                                             needs to be run
            backup_type(str)    --  specify the backup type needs to be run
                                   ex:FULL or INCREMENTAL
        Raises:
                Exception:

                    if failed to run backup

        """

        self.log.info("*" * 10 + " Starting Subclient {0} Backup "\
                 .format(backup_type) + "*" * 10)
        job = subclient.backup(backup_type)
        self.log.info("Started {0} backup with Job ID: {1}".\
                 format(backup_type, str(job.job_id)))

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}"\
                               .format(backup_type, job.delay_reason))
        self.log.info("Successfully finished {0} backup job".format(backup_type))
        return job
    
    def run_restart_backup(self, subclient, backup_type):
        """Starts backup job
         Args:

             subclient(str)     -- Specify the subclient object name where backups
                                             needs to be run
            backup_type(str)    --  specify the backup type needs to be run
                                   ex:FULL or INCREMENTAL
        Raises:
                Exception:

                    if failed to run backup
        """

        self.log.info("*" * 10 + " Starting Subclient {0} Backup "\
                 .format(backup_type) + "*" * 10)
        job = subclient.backup(backup_type)
        self.log.info("Started {0} backup with Job ID: {1}".\
                 format(backup_type, str(job.job_id)))
        self.interruption = Interruption(job.job_id, self._commcell)
        self.interruption.suspend_resume_job()
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}"\
                               .format(backup_type, job.delay_reason))
        self.log.info("Successfully finished {0} backup job".format(backup_type))
        return job
    
    def run_pending_backup(self, subclient, backup_type, machine, pathname,\
                           pathname1, pathname2, pathname3):
        """Starts backup job
         Args:

             subclient(str)     -- Specify the subclient object name where backups
                                             needs to be run
            backup_type(str)    --  specify the backup type needs to be run
                                   ex:FULL or INCREMENTAL
            machine(str)        -- specify the machine object
            pathname(str)      -- specify the name of saphookfile
            pathname1(str)      -- specify the name of saphookfile
            pathname2(str)      -- specify the name of saphookfile
            pathname3(str)      -- specify the name of saphookfile

        Raises:
                Exception:

                    if failed to run backup
        """

        self.log.info("*" * 10 + " Starting Subclient {0} Backup "\
                 .format(backup_type) + "*" * 10)
        job = subclient.backup(backup_type)
        self.log.info("Started {0} backup with Job ID: {1}".\
                 format(backup_type, str(job.job_id)))
        jobid = job.job_id
        job_status_flag = 1
        while job_status_flag:
            time.sleep(60)
            self.log.info("Job Status:%s", job.status)
            if(job.status.lower() == "pending" or
               "completed" in job.status.lower()):
                job_status_flag = 0
                if job.status.lower() == "pending":
                    self.log.info("Status of  {0} backup with Job ID: {1}")
                    machine.remove_directory(pathname)
                    self.log.info("hookfile failSAPDataBackupB4Intimate are deleted sucessfully")

        self.log.info(job.resume(True))
        self.log.info(job._wait_for_status("Running"))
        if job.status == "Running":
            recode = self.getarchfileisvalid(str(jobid), '1')
            if recode == 0:
                self.log.info("archivefiles are invalidated sucessfully in Cs db for restarted job")
        job_status_flag = 1
        while job_status_flag:
            time.sleep(60)
            self.log.info("Job Status:%s", job.status)
            if(job.status.lower() == "pending" or
               "completed" in job.status.lower()):
                job_status_flag = 0
                if job.status.lower() == "pending":
                    self.log.info("Status of  {0} backup with Job ID: {1}")
                    machine.remove_directory(pathname1)
                    self.log.info("hookfile failSAPDataConfigBackupB4Intimate are deleted sucessfully")

        self.log.info(job.resume(True))
        self.log.info(job._wait_for_status("Running"))
        self.log.info(job._wait_for_status("Running"))
        if job.status == "Running":
            recode = self.getarchfileisvalid(str(jobid), '1')
            if recode == 0:
                self.log.info("archivefiles are invalidated sucessfully in Cs db for restarted job")
        job_status_flag = 1
        while job_status_flag:
            time.sleep(60)
            self.log.info("Job Status:%s", job.status)
            if(job.status.lower() == "pending" or
               "completed" in job.status.lower()):
                job_status_flag = 0
                if job.status.lower() == "pending":
                    self.log.info("Status of  {0} backup with Job ID: {1}")
                    machine.remove_directory(pathname2)
                    self.log.info("hookfile failSAPLogBackupB4Intimate are deleted sucessfully")
        self.log.info(job.resume(True))
        self.log.info(job._wait_for_status("Running"))
        if job.status == "Running":
            recode = self.getarchfileisvalid(str(jobid), '4')
            if recode != '0':
                self.log.info("archivefiles are not invalidated sucessfully\
                in Cs db for restarted job")
        job_status_flag = 1
        while job_status_flag:
            time.sleep(60)
            self.log.info("Job Status:%s", job.status)
            if(job.status.lower() == "pending" or
               "completed" in job.status.lower()):
                job_status_flag = 0
                if job.status.lower() == "pending":
                    self.log.info("Status of  {0} backup with Job ID: {1}")
                    machine.remove_directory(pathname3)
                    self.log.info("hookfile failSAPLogConfigBackupB4Intimate are deleted sucessfully")
        self.log.info(job.resume(True))
        self.log.info(job._wait_for_status("Running"))
        if job.status == "Running":
            recode = self.getarchfileisvalid(str(jobid), '4')
            if recode != '0':
                self.log.info("archivefiles are not invalidated sucessfully\
                in Cs db for restarted job")

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}"\
                               .format(backup_type, job.delay_reason))
        self.log.info("Successfully finished {0} backup job".format(backup_type))
        return job
    
    def run_client_service_restart_backup(self, subclient, backup_type):
        """Starts backup job
         Args:

             subclient(str)         -- Specify the subclient object name where backups
                                             needs to be run
            backup_type(str)        --  specify the backup type needs to be run
                                       ex:FULL or INCREMENTAL
        Raises:
                Exception:

                    if failed to run backup
        """

        self.log.info("*" * 10 + " Starting Subclient {0} Backup "\
                 .format(backup_type) + "*" * 10)
        job = subclient.backup(backup_type)
        self.log.info("Started {0} backup with Job ID: {1}".\
                 format(backup_type, str(job.job_id)))
        self.interruption = Interruption(job.job_id, self._commcell)
        self.interruption.restart_client_services()
        self.interruption.wait_and_resume()

        self.log.info(job._wait_for_status("running"))
        if not job.wait_for_completion():
            raise Exception
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}"\
                               .format(backup_type, job.delay_reason))
        self.log.info("Successfully finished {0} backup job".format(backup_type))
        return job

    def run_kill_process_backup(self, subclient, backup_type, machine):
        """Starts backup job
         Args:

             subclient(str)         -- Specify the subclient object name where backups
                                             needs to be run
            backup_type(str)        --  specify the backup type needs to be run
                                       ex:FULL or INCREMENTAL

            machine(str)            -- specify the machine object

        Raises:
                Exception:

                    if failed to run backup
        """

        self.log.info("*" * 10 + " Starting Subclient {0} Backup "\
                 .format(backup_type) + "*" * 10)
        job = subclient.backup(backup_type)
        self.log.info("Started {0} backup with Job ID: {1}".\
                 format(backup_type, str(job.job_id)))
        time.sleep(40)
        self.log.info(job._wait_for_status("running"))
        machine.kill_process('ClSapAgent')
        self.log.info("ClSapAgent process is killed sucessfully")
        time.sleep(40)
        self.log.info(job.status)
        if job.status == "Pending":
            self.log.info("Status of  {0} backup with Job ID: {1}".\
                     format(backup_type, str(job.job_id)))
            self.log.info(job.resume(True))
            self.log.info(job._wait_for_status("Running"))
            if job.status == "Running":
                self.log.info("Status of  {0} backup with Job ID: {1}")

        self.log.info(job._wait_for_status("running"))
        if not job.wait_for_completion():
            raise Exception
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}"\
                               .format(backup_type, job.delay_reason))
        self.log.info("Successfully finished {0} backup job".format(backup_type))
        return job

    def run_random_kill_process(self, subclient, backup_type):
        """Starts backup job
         Args:

             subclient(str)         -- Specify the subclient object name where backups
                                             needs to be run
            backup_type(str)        --  specify the backup type needs to be run
                                       ex:FULL or INCREMENTAL

        Raises:
                Exception:

                    if failed to run backup
        """

        self.log.info("*" * 10 + " Starting Subclient {0} Backup "\
                 .format(backup_type) + "*" * 10)
        job = subclient.backup(backup_type)
        self.log.info("Started {0} backup with Job ID: {1}".\
                 format(backup_type, str(job.job_id)))
        time.sleep(30)
        self.interruption = Interruption(job.job_id, self._commcell)
        self.interruption.random_process_kill()
        self.interruption.wait_and_resume()

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}"\
                               .format(backup_type, job.delay_reason))
        self.log.info("Successfully finished {0} backup job".format(backup_type))
        return job
    
    def run_kill_process_backupcopy(self, subclient, snap_job_id):
        """Starts backup job
         Args:

           subclient(str)        -- specify the subclient name
                                       ex:default

           snap_job_id(int)      -- specify the snap backup jobid

        Raises:
                Exception:

                    if failed to run backup
        """

        self.log.info("*" * 10 + " Starting backup copy ""*" * 10)
        job = self.storage_policy_name.run_backup_copy()
        self.log.info("Backup copy workflow job id is : {0}".\
                 format(str(job.job_id)))
        time.sleep(30)
        backupcopyjob = idautils.get_backup_copy_job_id(self._commcell, subclient, snap_job_id)
        self.log.info(backupcopyjob)
        self.interruption = Interruption(job.job_id, self._commcell)
        self.interruption.random_process_kill()
        self.interruption.wait_and_resume()
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}"\
                               .format(str(job.job_id)), job.delay_reason)
        self.log.info("Successfully finished  backupcopy job {0} ".format(str(job.job_id)))
        return backupcopyjob

    def restart_backupcopy(self, subclient, snap_job_id):
        """Starts backup copy workflow job and gets the backupcopy job
         Args:

            subclient(str)        -- specify the subclient name
                                       ex:default

            snap_job_id(int)      -- specify the snap backup jobid

        Raises:
                Exception:

                    if failed to run backup copy job
        """

        self.log.info("*" * 10 + " Starting backup copy ""*" * 10)
        job = self.storage_policy_name.run_backup_copy()
        self.log.info("Backup copy workflow job id is : {0}".\
                 format(str(job.job_id)))
        time.sleep(30)
        backupcopyjob = idautils.get_backup_copy_job_id(self._commcell, subclient, snap_job_id)
        self.log.info(backupcopyjob)
        self.interruption = Interruption(backupcopyjob, self._commcell)
        self.interruption.suspend_resume_job()

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}"\
                               .format(str(job.job_id)), job.delay_reason)
        self.log.info("Successfully finished  backupcopy job {0} ".format(str(job.job_id)))
        return backupcopyjob

    def run_client_service_restart_backupcopy(self, subclient, snap_job_id):
        """Starts backup job
         Args:

            subclient(str)        -- specify the subclient name
                                       ex:default

            snap_job_id(int)      -- specify the snap backup jobid
        Raises:
                Exception:

                    if failed to run backupcopy
        """

        self.log.info("*" * 10 + " Starting Subclient {0} Backupcopy " "*" * 10)
        job = self.storage_policy_name.run_backup_copy()
        self.log.info("Started backup with Job ID: {0}".\
                 format(str(job.job_id)))
        time.sleep(30)
        backupcopyjob = idautils.get_backup_copy_job_id(self._commcell, subclient, snap_job_id)
        self.log.info(backupcopyjob)
        self.interruption = Interruption(backupcopyjob, self._commcell)
        self.interruption.restart_client_services()
        self.interruption.wait_and_resume()
        self.log.info(job._wait_for_status("running"))
        if not job.wait_for_completion():
            raise Exception
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}"\
                               .format(str(job.job_id)), job.delay_reason)
        self.log.info("Successfully finished  backupcopy job {0} ".format(str(job.job_id)))
        return backupcopyjob

