# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Helper file for performing sap hana operations

HANAHelper is the only class defined in this file

HANAHelper: Helper class to perform sap hana operations

HANAHelper:
    __init__()                      --  initializes sap hana helper object

    backup_prefix()                 --  sets the back up prefix to be used during backup

    get_hana_server_details()       --  gets the SAP HANA server details like the hostname of the
                                            HANA server, port

    hana_backup()                   --  runs the sap hana backup with the given backup type

    hana_restore()                  --  runs the given SAP HANA restore

    cleanup_test_data()             --  cleans up the test data created during automation

    create_test_tables()            --  creates test tables in the schema of the database user
                                        provided

    test_tables_validation()        --  validates the test tables that were restored

"""

import datetime
from random import randint
from pytz import timezone

from AutomationUtils import cvhelper
from AutomationUtils.database_helper import SAPHANA


class HANAHelper:
    """Helper class to perform sap hana operations"""

    def __init__(self, testcase):
        """
        Initializes hanahelper object

        Args:
            testcase    (object):   Object of the testcase class to get all the details

        """
        self.log = testcase.log
        self.commcell = testcase.commcell
        self.csdb = testcase.csdb
        self.hana_client = testcase.client.client_name
        self.hana_instance = testcase.instance
        self.hana_subclient = testcase.subclient
        self.hana_backupset = self.hana_subclient._backupset_object

        if self.hana_backupset:
            self.backupset_name = self.hana_backupset.backupset_name
        else:
            self.backupset_name = "default"

        self.hana_server = self.hana_instance.db_instance_client['clientName']
        self.hana_server_hostname = None
        self._hana_port = None
        self._hana_db_user_name = self.hana_instance.instance_db_username
        self._hana_db_password = None
        self._total_tables = 1002
        self._total_rows = 300
        self._table_number = 1
        self._test_data = "SAP HANA test automation:SAP HANA test automation:SAP HANA test \
        automation:SAP HANA test automation:SAP HANA test automation:SAP HANA test automation:\
        SAP HANA test automation:SAP HANA test automation: SAP HANA test automation:"

        self._connection = None
        self._cursor = None
        self._backup_prefix = None

        self.get_hana_server_details()
        self._db_connect = None

        self.restore_data_prefix = None
        self.incremental_job_time = None

        self.cleanup_test_data()

    @property
    def backup_prefix(self):
        """ getter for the backup prefix 
		Returns:
			(str)	--	Returns backup prefix
		"""
        return self._backup_prefix

    @backup_prefix.setter
    def backup_prefix(self, value):
        """ Setter for the backup prefix 
		Args:
			value	(str)	-- backup prefix of job
		"""
        self._backup_prefix = str(value)

    def get_hana_server_details(self):
        """ Gets the SAP HANA server details like hostname, password and port

        Raises:
            Exception:
                if the hostname of the HANA server could not be obtained or
                if the password for the HANA instance could not be obtained or
                if there is an issue with getting the port
        """
        try:
            # Gets the SAP HANA server client's hostname
            query = "select net_hostname from App_Client where id = {0}".format(int(
                self.hana_instance.db_instance_client['clientId']
            ))
            self.csdb.execute(query)
            cur = self.csdb.fetch_one_row()
            if cur:
                self.hana_server_hostname = cur[0]
            else:
                raise Exception("Failed to get the HANA server host name from database")

            # Gets the password of the SAP HANA server
            query = "Select attrVal from app_instanceprop where componentNameId = {0} and \
                        attrName = 'DB User Password'".format(str(self.hana_instance.instance_id))

            self.csdb.execute(query)
            cur = self.csdb.fetch_one_row()
            if cur:
                password = cur[0]
                self._hana_db_password = cvhelper.format_string(self.commcell, password)
            else:
                raise Exception("Failed to get the HANA client name from database")

            # Gets the port number to connect to the SAP HANA server
            instance_id = self.hana_instance.instance_number
            self._hana_port = int("3{0}15".format(instance_id))

        except Exception as exp:
            raise Exception(str(exp))

    def hana_backup(self, backup_type):
        """
        Submits a SAP HANA backup for the subclient with the given backup type

        Args:
            backup_type (basestring):  the type of backup to be submitted

        Raises:
            Exception:
                if there is an error while submitting SAP HANA backup

        """
        try:
            self.log.info("*" * 10 + " Starting Subclient %s Backup ", backup_type + "*" * 10)
            backup_job = self.hana_subclient.backup(backup_type, self.backup_prefix)
            self.log.info("Started %s backup with Job ID: %s", backup_type, backup_job.job_id)
            if not backup_job.wait_for_completion():
                raise Exception(
                    "Failed to run {0} backup job with error: {1}".format(
                        backup_type, backup_job.delay_reason
                    )
                )
            if backup_type.lower() == "full":
                if self.backup_prefix:
                    self.restore_data_prefix = "{0}_{1}".format(backup_job.job_id, self.backup_prefix)
                else:
                    self.restore_data_prefix = "{0}_COMPLETE_DATA_BACKUP".format(backup_job.job_id)
            elif backup_type.lower() == "incremental":
                self.incremental_job_time = str(backup_job.end_time)
                self.log.info("Incremental End time -- %s", self.incremental_job_time)
                time_stamp = datetime.datetime.strptime(self.incremental_job_time,
                                                        '%Y-%m-%d %H:%M:%S')
                time_stamp = time_stamp.replace(tzinfo=timezone('UTC'))
                time_stamp = str(int(time_stamp.timestamp()))
                self.log.info("Incremental timestamp -- %s", time_stamp)
                self.incremental_job_time = time_stamp
        except Exception as exp:
            raise Exception("Exception occurred while trying to run hana backup. {0}".format(
                str(exp)))

    def hana_restore(self, point_in_time=False, data_only=False):
        """
        Submits a SAP HANA restore of the given type

        Args:
            point_in_time   (bool): if the restore has to be point in time
			
				default: False

            data_only       (bool): if the restore has to be recover data only
			
				default: False

        Raises:
            Exception:
                if there is an error while submitting a restore

        """
        try:
            self.log.info("***** Starting Subclient Restore *****")
            if all([point_in_time, data_only]):
                raise Exception("Both point in time and data only restores are selected "
                                "for a single restore")

            if not point_in_time and not data_only:
                self.log.info("Starting a current time restore")
                restore_job = self.hana_instance.restore(pseudo_client=self.hana_client.lower(),
                                                         instance=self.hana_instance.instance_name,
                                                         backupset_name=self.backupset_name,
                                                         ignore_delta_backups=False,
                                                         check_access=True)
                self.log.info("Current time restore job started. Job ID is %s", restore_job.job_id)
                if not restore_job.wait_for_completion():
                    raise Exception("Failed to run current time restore job with error: " + str(
                        restore_job.delay_reason))

                self.test_tables_validation()
            elif point_in_time:
                restore_job = self.hana_instance.restore(pseudo_client=self.hana_client.lower(),
                                                         instance=self.hana_instance.instance_name,
                                                         backupset_name=self.backupset_name,
                                                         point_in_time=self.incremental_job_time,
                                                         check_access=True,
                                                         ignore_delta_backups=False)
                self.log.info("Point in time restore job started. Job ID is %s",
                              restore_job.job_id)
                if not restore_job.wait_for_completion():
                    raise Exception("Failed to run point in time restore job with error: " + str(
                        restore_job.delay_reason))

                self.test_tables_validation(point_in_time=True)
            elif data_only:
                self.log.info("Starting a recover data only restore")
                restore_job = self.hana_instance.restore(pseudo_client=self.hana_client.lower(),
                                                         instance=self.hana_instance.instance_name,
                                                         backupset_name=self.backupset_name,
                                                         backup_prefix=self.restore_data_prefix,
                                                         check_access=True)
                self.log.info("Recover data only restore job started. Job ID is %s",
                              restore_job.job_id)
                if not restore_job.wait_for_completion():
                    raise Exception("Failed to run recover data only restore job with "
                                    "error: " + str(restore_job.delay_reason))

                self.test_tables_validation(recover_data=True)
            self.log.info("Successfully validated restored content")
        except Exception as exp:
            raise Exception("Sap Hana restore or validation failed. {0}".format(str(exp)))

    def cleanup_test_data(self):
        """
        Cleans up the testdata from the previous cycle

        Raises:
            Exception:
                if the testdata could not be cleaned up properly
                or if connection could not be established to the HANA client

        """
        try:
            try:
                self._db_connect = SAPHANA(self.hana_server_hostname,
                                           self._hana_port,
                                           self._hana_db_user_name,
                                           self._hana_db_password)
            except Exception:
                self._hana_port = "3{0}13".format(self.hana_instance.instance_number)
                self._db_connect = SAPHANA(self.hana_server_hostname,
                                           int(self._hana_port),
                                           self._hana_db_user_name,
                                           self._hana_db_password)

            query = "Select TABLE_NAME from TABLES where SCHEMA_NAME='{0}' and TABLE_NAME like\
            'AUTOMATIONTABLE%'".format(self._hana_db_user_name)
            response = self._db_connect.execute(query, commit=False)

            all_tables = response.rows

            if all_tables:
                for table in all_tables:
                    query = "Drop table {0}.{1} CASCADE;".format(self._hana_db_user_name,
                                                                 str(table[0]))
                    self._db_connect.execute(query)
        except Exception as exp:
            raise Exception("Could not delete the test tables. " + str(exp))

    def create_test_tables(self):
        """
        Creates the test tables in the source database

        Raises:
            Exception:
                if not able to create test tables

        """
        try:
            try:
                self._db_connect = SAPHANA(self.hana_server_hostname,
                                           self._hana_port,
                                           self._hana_db_user_name,
                                           self._hana_db_password)
            except Exception:
                self._hana_port = "3{0}13".format(self.hana_instance.instance_number)
                self._db_connect = SAPHANA(self.hana_server_hostname,
                                           int(self._hana_port),
                                           self._hana_db_user_name,
                                           self._hana_db_password)

            table_name = "AutomationTable" + str(self._table_number)

            table_count = int(self._total_tables / 3)

            query = "create table {0} (Id int,String1 varchar(256),String2 varchar(256),String3 \
            varchar(256),String4 varchar(256),String5 varchar(256),String6 varchar(256),String7 \
            varchar(256),String8 varchar(256),String9 varchar(256),String10 varchar(256)); \
            ".format(table_name)

            self._db_connect.execute(query)

            query = "Insert into {0}.{1} values('1','{key}','{key}','{key}','{key}','{key}',\
            '{key}','{key}','{key}','{key}','{key}');".format(self._hana_db_user_name, table_name,
                                                              key=self._test_data)
            self._db_connect.execute(query)

            for row in range(2, self._total_rows+1):
                query = "Insert into {0}.{1} (Id,String1,String2,String3,String4,String5,\
                String6,String7,String8,String9,String10) select {2},String1,String2,String3,\
                String4,String5,String6,String7,String8,String9,String10 from {0}.{1}\
                where Id=1".format(self._hana_db_user_name, table_name, str(row))
                self._db_connect.execute(query)
            self._table_number += 1

            for table_no in range(2, table_count + 1):
                table_name = "AutomationTable" + str(self._table_number)
                query = "Create table {0} (Id int,String1 varchar(256),String2 varchar(256),\
                String3 varchar(256),String4 varchar(256),String5 varchar(256),String6 \
                varchar(256),String7 varchar(256),String8 varchar(256),String9 varchar(256),\
                String10 varchar(256));".format(table_name)
                self._db_connect.execute(query)

                query = "Insert into {0}.{1} (Id,String1,String2,String3,String4,String5,\
                String6,String7,String8,String9,String10) select Id,String1,String2,String3,\
                String4,String5,String6,String7,String8,String9,String10 from {0}.{2}".format(
                    self._hana_db_user_name, table_name, "AutomationTable" + "1")
                self._db_connect.execute(query)

                self._table_number += 1

        except Exception as exp:
            raise Exception(str(exp))

    def test_tables_validation(self, point_in_time=False, recover_data=False):
        """
        Validates the test tables that were created before the backup

        Args:
            point_in_time   (boolean):  True / False based on the restore job to be
                                        validated is Point In Time Restore or not

                default:    False

            recover_data    (boolean):  True / False based on the restore job to be
                                        validated is Recover Data only restore or not

                default:    False

        Raises:
            
			Exception:
				if database connection fails
				
				if not table is restored by the job

                if the number of source tables and the restored tables do not match

                if the number of table is greater than the number of source table

				if we are unable to get row count for a table

				if the row count doesn't match

				if the values in table doesn't match

				if restore validation fails

        """
        try:
            try:
                self._db_connect = SAPHANA(self.hana_server_hostname,
                                           self._hana_port,
                                           self._hana_db_user_name,
                                           self._hana_db_password)
            except Exception:
                self._hana_port = "3{0}13".format(self.hana_instance.instance_number)
                self._db_connect = SAPHANA(self.hana_server_hostname,
                                           self._hana_port,
                                           self._hana_db_user_name,
                                           self._hana_db_password)

            query = "Select TABLE_NAME from TABLES where SCHEMA_NAME='{0}'".format(
                self._hana_db_user_name)
            response = self._db_connect.execute(query, commit=False)

            all_tables = response.rows
            if not all_tables:
                raise Exception("There are no restored tables in the restored database.")

            restored_tables = []
            for table in all_tables:
                restored_tables.append(str(table[0]))

            if point_in_time:
                source_table_count = (self._total_tables/3)*2
            else:
                if recover_data:
                    source_table_count = self._total_tables / 3
                else:
                    source_table_count = self._total_tables
            print(source_table_count)

            restored_table_count = 0
            prefixed_tables = []
            restored_table_number = []
            for table_name in restored_tables:
                if "AUTOMATIONTABLE" in table_name:
                    restored_table_count += 1
                    prefixed_tables.append(table_name)
                    restored_table_number.append(int(table_name.split("AUTOMATIONTABLE")
                                                     [-1].strip()))

            if not restored_table_count == source_table_count:
                raise Exception("The number of restored tables does not match the number of "
                                "source tables.")

            if any(table_number > source_table_count for table_number in restored_table_number):
                raise Exception("The restored table names do not match with the source"
                                "table names")

            query = "SELECT \"RECORD_COUNT\" FROM \"SYS\".\"M_TABLES\" WHERE SCHEMA_NAME='{0}';\
            ".format(self._hana_db_user_name)
            response = self._db_connect.execute(query, commit=False)

            row_count_of_tables = response.rows
            if not row_count_of_tables:
                raise Exception("Could not obtain the row count of all the tables in the schema")

            for each_tables_row in row_count_of_tables:
                if int(each_tables_row[0]) != self._total_rows:
                    raise Exception("The restored table does not contain all the rows")

            random_table_number = [randint(1, (self._total_tables/3)+1) for table in range(0, 6)]
            random_rows = [randint(1, (self._total_rows/3)+1) for rows in range(0, 5)]
            for table in random_table_number:
                query = "Select String3 from {0}.{1} where id in {2}".format(
                    self._hana_db_user_name, "AUTOMATIONTABLE"+str(table), tuple(random_rows))
                response = self._db_connect.execute(query, commit=False)
                table_values = response.rows
                for values in table_values:
                    if str(values[0]) != self._test_data:
                        raise Exception("Test data does not match the cell in the table")

        except Exception as exp:
            raise Exception("Failed to validate the source and restored test tables. " + exp)
