# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright  Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""Db2helper file for performing DB2 operations

DB2 is the only class defined in this file

DB2: Helper class to perform SAP Oracle operations

DB2:
====
    __init__()                              -- initializes DB2 Helper object

    get_db2_password()                      -- Gets the db2 password of the instance

    reconnect()                             -- reconnects to given database name

    run_backup()                            -- triggers backup job

    run_restore()                           -- Starts restore job

    get_datafile_location()                 -- retrieves tablespace datafile location

    prepare_data()                          -- create sample data in test tablespace

    create_tablespace()                     -- creates sample tablespace

    drop_tablespace()                       -- method to drop tablespace

    create_table2()                         -- creates sample data table in sample tablespace

    get_active_logfile()                    -- Fetches active logfile number by connecting
    to db2 database

    backup_validation()                     -- validates backup image timestamp with db2
    history file

    restore_validation()                    -- validates restore job by checking if data restored
    is accessible or not

    log_backup_validation()                 -- validates db2 log backup by connecting
    to commserve database

    get_backup_time_stamp_and_streams()     -- records the db2 image time stamp and streams of
    given backup job id

    get_database_state()                    -- checks if database is in active state or backup
    pending state

    disconnect_applications()               -- disconnects all the database connections

    exec_immediate_method()                 -- executes given db2 command

    update_db2_database_configuration1()    -- updates db2 db configurations

    get_db2_version()                       -- returns db2 application version

    get_db2_information()                   -- gets database list, db2 home directory for the
    given db2 instance

    close_db2_connection()                  -- closes ibm_db db2 connection

    list_database_directory()               -- returns list of databases for the given db2 instance

    get_database_aliases_helper()           -- returns database alias names

    third_party_command_backup()            -- used to trigger command line backup job for database

    third_party_command_restore()           -- triggers command line restore job for database

    third_party_command_rollforward()       -- triggers command line rollforward for database

    third_party_command_recover()           -- triggers command line recover database

    db2_archive_log()                       -- uses command line to archive db2 logs

    third_party_command()                   -- method to execute third party commands

    db2_cold_backup()                       -- triggers cold backup

    restore_from_cold_backup()              -- triggers restore from cold backup

    compare_db2_versions()                  -- compare db2 version

"""

import ibm_db
from AutomationUtils import logger
from AutomationUtils import database_helper
from AutomationUtils import cvhelper
from AutomationUtils.database_helper import Db2
from AutomationUtils import machine


class DB2(object):
    """Helper class to perform DB2 operations"""

    def __init__(self, commcell, client, instance, backupset, port):
        """Initializes DB2helper object

            Args:
                commcell             (obj)  --  Commcell object

                client               (obj)  --  Client object

                instance             (obj)  --  db2 instance object

                backupset            (obj)  --  backupset object

                port                 (int)  --  db2 database port

        """

        self.log = logger.get_log()
        self.log.info('  Initializing db2 Helper ...')
        self.commcell = commcell
        self.client = client
        self.instance = instance
        self.backupset = backupset

        self._database = self.backupset.backupset_name
        self._hostname = self.client.client_hostname
        self._port = str(port)
        self._csdb = database_helper.get_csdb()
        self._protocol = "TCPIP"
        self._db_user = self.instance.user_name
        self._db_password = None
        self.get_db2_password()
        self.log.info(" db2 user name is %s", self._db_user)
        self._db2_home_path = self.instance.home_directory
        self.db2 = Db2(
            self._database,
            self._hostname,
            self._port,
            self._protocol,
            self._db_user,
            self._db_password)
        self._connection = self.db2._connection
        self.machine_object = machine.Machine(self.client)
        self.machine_db2object = machine.Machine(
            self.client.client_name,
            username=self.instance.user_name,
            password=self._db_password)
        self.platform = self.machine_object.os_info
        self.db2_profile = ""
        self.simpana_instance = self.client.instance
        self.simpana_base_path = self.machine_object.get_registry_value(
            "Base", "dBASEHOME")
        if self.platform.upper() == 'WINDOWS':
            self.db2cmd = " set-item -path env:DB2CLP -value **$$** ; db2 -o  "
            self.load_path = r"{0}\Db2Sbt.dll".format(self.simpana_base_path)
        else:
            self.db2cmd = ""
            self.load_path = "{0}/libDb2Sbt.so".format(self.simpana_base_path)
        self.get_db2_information()

    def get_db2_password(self):
        """Gets the db2 password of the instance

                Raises:
                    Exception:
                        if failed to get the db password of the instance """
        query = (
            "Select attrVal from app_instanceprop where componentNameId = {0} and "
            "attrName = 'DB2 Domain Password'".format(
                str(
                    self.instance.instance_id)))
        self._csdb.execute(query)
        cur = self._csdb.fetch_one_row()

        if cur:
            self._db_password = cvhelper.format_string(
                self.commcell, cur[0])
            self.log.info("Password for db2 database is fetched Successfully")
        else:
            raise Exception("Failed to get the db2 database password")

    def reconnect(self):
        """
        Reconnects to the db2 database

        Raises:
            Exception:
                if connection to db fails

        """
        try:
            self.db2._connect()
            self._connection = self.db2._connection
            self.log.info("Reconnecting to db2 database")
        except Exception as excp:
            raise Exception("Exception in reconnect: {0}".format(excp))

    def run_backup(self, subclient, backup_type):
        """Starts backup job

        Args:

            subclient   (obj)       --  Specify the subclient object name where backups
                                             needs to be run
            backup_type (str)       --  specify the backup type needs to be run say
                                        FULL or INCREMENTAL or DELTA
        Returns:
                        (obj)       --  job object will be returned
        Raises:
                Exception:

                    if failed to run backup

        """
        self.log.info("Starting subclient %s Backup ", backup_type)
        job = subclient.backup(backup_type)
        self.log.info("Started: backup with Job ID: %s job_id ",
                      job.job_id)
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run: {0} backup_type backup job with error: {1} delay_reason ".
                format(backup_type, job.delay_reason))
        self.log.info("Successfully finished backup_type %s backup_type ",
                      backup_type)

        return job

    def run_restore(self, backupset, recover_db=True):
        """Starts restore job
        Args:

            backupset_name  (obj)     --  Specify the backupset name where restores
            needs to be run

            recover_db      (bool)    -- boolean value to specify if db needs to be recovered

                    default: True


        Returns:
            (obj)   --  job object

        Raises:
                Exception:

                    if failed to run restore

        """
        job = backupset.restore_entire_database(
            recover_db=recover_db, restore_incremental=recover_db)
        self.log.info(
            "Started: FULL restore with Job ID: %s", job.job_id)

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run: {0} restore job with error: {1} delay_reason ".
                format(job.job_id, job.delay_reason))
        self.log.info("Successfully finished restore ")

        return job

    def get_datafile_location(self):
        """
        Retrieves datafile location.

        Returns:
            (str)       --      tablespace location.

        Raises:
            Exception: if tablespace is not found

        """
        try:
            cmd = "select path from sysibmadm.dbpaths where type = 'DBPATH'"
            self.log.info(cmd)
            try:
                stmt = ibm_db.exec_immediate(self._connection, cmd)
            except Exception as excp:
                raise Exception(
                    "Failed to get datafile with error: {0}".format(excp))
            datafile_location = ibm_db.fetch_assoc(stmt)
            self.log.info(datafile_location)
            for value in datafile_location.values():
                val = value
                self.log.info("Data file Location is: %s", value)
            return val
        except Exception as excp:
            raise Exception(
                "Exception in get_datafile_location: {0}".format(excp))

    def prepare_data(self, table_name):
        """
        Method for getting table content,tablespace list and count.

        Args:

             table_name  (str) - name of the table

        Returns:
            table_content       (str) - table contents like number of rows
            tablespaces_list    (list) - list of tablespaces
            tablespace_count    (int) - number of tablespaces

        Raises:
              Exception:
                If unable to retrieve any of these information

        """
        try:
            table_content = self.db2.get_table_content(table_name)
            (tablespaces_list, tablespace_count) = self.db2.get_tablespaces()
            return table_content, tablespaces_list, tablespace_count

        except Exception as excp:
            raise Exception("Exception in prepare_data : {0}".format(excp))

    def create_tablespace(self, datafile, tblspace_name,
                          flag_create_tablespace):
        """
        Method to create tablespace

        Args:
            datafile (str)                  -- datafile location where tablespace can
            hold the physical file

            tblspace_name (str)             -- name of the tablespace

            flag_create_tablespace (bool)   -- flag to create tablespace or not

        Returns:

            (bool)      -   method return false if drop tablespace fails
            (bool)      -   tablespace creation fails


        """
        if flag_create_tablespace:
            cmd = "select tbsp_name from sysibmadm.tbsp_utilization where tbsp_name = '{0}'".format(
                tblspace_name)
            stmt = ibm_db.exec_immediate(self._connection, cmd)
            tablespaces = ibm_db.fetch_assoc(stmt)
            if tablespaces:
                tablespaces = str(tablespaces.items())
                self.log.info("tablespace exists : %s", tablespaces)
                cmd = "drop tablespace {0}".format(tblspace_name)
                output = ibm_db.exec_immediate(self._connection, cmd)
                if output:
                    self.log.info("tablespace dropped successfully")
                else:
                    self.log.info("tablespace not dropped successfully")
                    return False

            datafile1 = "{0}{1}_Full.dbf".format(datafile, tblspace_name)

            cmd = (
                "CREATE TABLESPACE {0} MANAGED BY DATABASE "
                "USING (FILE '{1}' 100M ) AUTORESIZE NO ".format(tblspace_name, datafile1))

            output = ibm_db.exec_immediate(self._connection, cmd)
            if output:
                self.log.info("Created tablespace successfully")
            else:
                self.log.info(
                    "tablespace is not created successfully :%s", output)
        return False

    def drop_tablespace(self, tblspace_name):
        """
        Method to drop tablespace

        Args:

            tblspace_name (str)             -- name of the tablespace

        Returns:

            (bool)      -   method return false if drop tablespace fails

        """

        cmd = "select tbsp_name from sysibmadm.tbsp_utilization where tbsp_name = '{0}'".format(
            tblspace_name)
        stmt = ibm_db.exec_immediate(self._connection, cmd)
        tablespaces = ibm_db.fetch_assoc(stmt)
        if tablespaces:
            tablespaces = str(tablespaces.items())
            self.log.info("tablespace exists : %s", tablespaces)
            cmd = "drop tablespace {0}".format(tblspace_name)
            output = ibm_db.exec_immediate(self._connection, cmd)
            if output:
                self.log.info("tablespace dropped successfully")
                return True
            self.log.info("tablespace not dropped successfully")
            return False
        else:
            self.log.info("Tablespace doesn't exists")
            return True

    def create_table2(self, datafile, tablespace_name,
                      table_name, flag_create_tablespace):
        """ creates table in the given tablespace

            Args:

                datafile                (str)       -- datafile location

                tablespace_name         (str)       -- name of the tablespace

                table_name              (str)       -- name of the table

                flag_create_tablespace  (bool)      -- set flag to create or not the tablespace

            Returns:

                (bool)  - returns false if table creation fails

        """
        self.create_tablespace(
            datafile, tablespace_name, flag_create_tablespace)

        cmd = "create table {0} (name varchar(30), ID decimal) in {1} ".format(
            table_name, tablespace_name)

        output = ibm_db.exec_immediate(self._connection, cmd)
        if output:
            self.log.info(
                "Created table successfully with table name %s ",
                table_name)
        else:
            self.log.info("table is not created successfully")
            return False

        cmd = "insert into {0} values('commvault', 1)".format(table_name)
        for _ in range(1, 10):
            output = ibm_db.exec_immediate(self._connection, cmd)

        self.log.info("Inserted rows into table successfully")
        return True

    def get_active_logfile(self):
        """
        Fetches active logfile number by connecting to db2 database

        Raises:
            Exception:
                If unable to retrieve active log file number from db2 database

        """
        try:
            cmd = (
                "select MEMBER, CURRENT_ACTIVE_LOG from "
                "table(mon_get_transaction_log(-1)) as t order by member asc")
            self.log.info(cmd)
            stmt = ibm_db.exec_immediate(self._connection, cmd)
            output = ibm_db.fetch_tuple(stmt)
            self.log.info(output)
        except Exception as excp:
            raise Exception(
                "exception in get_active_logfile: {0}".format(excp))

    def backup_validation(self, operation_type,
                          tablespaces_count, backup_time_stamp):
        """
        Validates if backup job is successful

        Args:

            operation_type      (str)   -- type backup job like full/incremental/delta

            tablespaces_count   (int)   -- tablespace count

            backup_time_stamp   (str)   -- backup image timestamp

        Raises:

            Exception:
                If job type ran is not correct.
                If there is a mismatch in number of tablespaces backedup.
                If tablespace backup is not successfull.

        """
        try:
            cmd = (
                "select operationtype, TBSPNAMES from sysibmadm.db_history "
                "where start_time =  '{0}' and "
                "operationtype in('F','N','I','O','D','E')".format(backup_time_stamp))
            self.log.info(cmd)
            stmt = ibm_db.exec_immediate(self._connection, cmd)
            output = ibm_db.fetch_tuple(stmt)
            self.log.info(output)
            tble_line = output[1]
            self.log.info("table spaces backed up: '%s'", tble_line)
            self.log.info(
                "total table spaces: '%s'", tablespaces_count)
            if str(output[0]) == operation_type:
                self.log.info(
                    "Correct backup type job ran  F - Offline   N - Online   I - "
                    "Incremental offline  O - Incremental online D - Delta offline "
                    "E - Delta online  Actual job type :%s , Ran job type: %s",
                    operation_type,
                    output[0])

            else:
                raise Exception("Correct backup type job is not ran. Actual job type:{0} Ran "
                                "Job Type: {1}".format(operation_type, output[0]))
            for db_tablespace in tablespaces_count:
                if str(tble_line).find(db_tablespace) >= 0:
                    self.log.info(
                        "Table space backed up successfully : '%s'", db_tablespace)
                else:
                    raise Exception(
                        "Table space was not able to back up successfully : '{0}'".format(
                            db_tablespace))
                self.log.info(
                    " All Table space backed up successfully : '%s'", db_tablespace)

        except Exception as excp:
            raise Exception("exception in backup_validation: {0}".format(excp))

    def restore_validation(
            self,
            table_space,
            table_name,
            tablecount_full=None,
            tablecount_incr=None,
            tablecount_delta=None):
        """
        After restore it will check whether table space is accessible
        and checks the restore table data with original table data

        Args:

            table_space         (str)   --  name of the tablespace
            table_name          (str)   --  table name

            tablecount_full     (int)   --  number of rows in table during full backup
                default:    None

            tablecount_incr     (int)   --  number of rows in table during incremental backup
                default:    None

            tablecount_delta    (int)   --  number of rows in table during delta backup
                default:    None

        Raises:

            Exception:
                if any table or tablespace is not restored successfully

        """
        try:
            cmd = "SELECT SERVICE_LEVEL FROM TABLE(SYSPROC.ENV_GET_INST_INFO())"
            self.log.info(cmd)
            stmt = ibm_db.exec_immediate(self._connection, cmd)
            output = ibm_db.fetch_assoc(stmt)
            self.log.info(output)

            self.log.info("DB2 version: %s", output['SERVICE_LEVEL'])
            if str(output['SERVICE_LEVEL']).find("v9.") >= 0:
                cmd = ("select ACCESSIBLE from  table(sysproc.snapshot_container('{0}',0)) tbl "
                       "where TABLESPACE_NAME = '{1}'".format(self._database, table_space))
                self.log.info(cmd)
            else:
                cmd = ("select ACCESSIBLE from  table(sysproc.SNAP_GET_CONTAINER_V91('{0}',0)) tbl"
                       " where TBSP_NAME = '{1}'".format(self._database, table_space))

                self.log.info(cmd)

            stmt = ibm_db.exec_immediate(self._connection, cmd)
            output = ibm_db.fetch_assoc(stmt)
            self.log.info(output)

            self.log.info("Accessible value %s", output['ACCESSIBLE'])
            if output['ACCESSIBLE'] == 0:
                self.log.info("Data is not restored")

            output = self.db2.get_table_content("{0}_FULL".format(table_name))

            if output != tablecount_full:
                output = self.db2.get_table_content(
                    "{0}_PARTIAL".format(table_name))
                self.log.info(output)
                if output != tablecount_full:
                    self.log.info(
                        "table %s_FULL or %s_PARTIAL is not restored correctly",
                        table_name, table_name)

            if tablecount_incr is not None:
                output = self.db2.get_table_content(
                    "{0}_INCR".format(table_name))
                self.log.info(output)
                if output != tablecount_incr:
                    self.log.info(
                        "table : %s_INCR is not restored correctly", table_name)

            if tablecount_delta is not None:
                output = self.db2.get_table_content(
                    "{0}_DELTA".format(table_name))
                self.log.info(output)
                if output != tablecount_delta:
                    self.log.info(
                        "table %s_DELTA is not restored correctly", table_name)

                self.log.info(
                    "All tables are restored correctly from "
                    "all backup images(full, incremental, delta)")

        except Exception as excp:
            self.log.exception(
                "Exception while Running restore_validation function %s", excp)

    def log_backup_validation(self, jobid):
        """
        Validates db2 log backup by connecting to commserve database

        Args:

            jobid(str)  -- jobid of db2 log backup

        Returns:

            bool -- returns boolean based on the fileType output

        Raises:
            Exception:
                If unable to retrieve fileType value from csdb

        """
        try:
            self.log.info(
                "Get fileType from archFile table for a given backup job : %s", jobid)
            query = (
                "select fileType from archFile where archFile.jobId = {0} "
                "and commCellId = '2'".format(jobid))
            self.log.info(query)
            self._csdb.execute(query)
            file_type = self._csdb.fetch_all_rows()
            self.log.info("fileType for log backup is %s", file_type)
            if str(file_type).find("4"):
                return True
            else:
                raise Exception("log backup validation failed")
        except Exception as excp:
            raise Exception(
                "Exception raised in get_backup_time_stamp_and_streams()"
                "\nError: '{0}'".format(excp))

    def get_backup_time_stamp_and_streams(self, jobid):
        """
        Gets Backup Time Stamp for a given backup job

        Args:

            jobid (str) -- jobid of db2 backup

        Returns:

              str - backup image timestamp and number of streams used for that backup

        Raises:
              Exception:
                if unable to get values from archfile table of csdb

        """
        try:
            self.log.info(
                "Get Backup Time Stamp for a given backup job : %s", jobid)
            query = (
                "select archFile.name from archFile where archFile.jobId = {0} "
                "and commCellId = '2'".format(jobid))
            self.log.info(query)
            self._csdb.execute(query)
            backup_time_stamp = self._csdb.fetch_all_rows()
            self.log.info("backuptimestamp %s", backup_time_stamp)
            backup_time_stamp0 = str(backup_time_stamp[0][0]).rsplit("_", 2)
            i = 1
            for i in range(1, len(backup_time_stamp)):
                if str(backup_time_stamp[i][0]).find(
                        backup_time_stamp0[1]) < 0:
                    flag = 1
                    break

            if int(flag) != 1 and i == (len(backup_time_stamp) - 1):
                backup_time_stamp = str(backup_time_stamp[i][0]).rsplit("_", 2)
            else:
                backup_time_stamp = str(
                    backup_time_stamp[i - 1][0]).rsplit("_", 2)
            self.log.info(
                "backuptimestamp: %s , streams: %s",
                backup_time_stamp[1],
                backup_time_stamp[2])

            return (backup_time_stamp[1], backup_time_stamp[2])
        except Exception as excp:
            raise Exception(
                "Exception raised in get_backup_time_stamp_and_streams()"
                "\nError: '{0}'".format(excp))

    def get_database_state(self):
        """
        Checks whether database is in active state

        Raises:
            Exception:
                if unable to get database state

        """

        try:
            cmd = "CALL SYSPROC.ADMIN_CMD('connect to {0}')".format(
                self._database)
            self.log.info(cmd)
            stmt = ibm_db.exec_immediate(self._connection, cmd)
            output = ibm_db.fetch_assoc(stmt)
            self.log.info(output)
            if str(output).find("Database Connection Information") >= 0:
                self.log.info(
                    "Database: %s is in active state. ",
                    self._database)
            if str(output).upper().find("BACKUP PENDING") >= 0:
                cmd = "{0} backup database {1}".format(
                    self.db2, self._database)
                output = ibm_db.exec_immediate(self._connection, cmd)
                if str(output).find("Backup successful") >= 0:
                    self.log.info(
                        "first time backup after database creation is successful :%s", output)
                else:
                    self.log.info(
                        "first time backup after database creation is not successful :%s", output)
            else:
                self.log.info(
                    "Database is not active state. Please check database :%s",
                    self._database)
        except Exception as excp:
            self.log.exception(
                "Exception raised at get_database_state '%s'", excp)

    def disconnect_applications(self, database):
        """
        It will disconnect all applications of DB2 so that offline restore will work

        Args:

            database (str)  -- database name
        Returns:
                    (bool)  --  Returns true if command executes successfully
        Raises:
            Exception:
                If deactivate database command fails

        """

        cmd = "{0} force applications all".format(self.db2cmd)
        self.log.info(cmd)
        cmd_output = self.machine_db2object.execute_command(cmd)
        output = cmd_output.formatted_output
        self.log.info("output: %s", output)
        cmd1 = "{0} deactivate database {1}".format(self.db2cmd, database)
        self.log.info(cmd1)
        cmd1_output = self.machine_db2object.execute_command(cmd1)
        output = cmd1_output.formatted_output
        self.log.info("%s", output)

    def exec_immediate_method(self, cmd):
        """
        Execute the given command using exec_immediate in ibm_db

        Args:

            cmd (str) -- db2 command that need to be executed

        """

        stmt = ibm_db.exec_immediate(self._connection, cmd)

        self.log.info(stmt)
        output = ibm_db.stmt_error(stmt)
        self.log.info(output)

    def update_db2_database_configuration1(self):
        """
        updates DB2 db configurations LOGARCHMETH1, LOGARCHOPT1, VENDOROPT, TRACKMOD parameters

        Returns:
            (bool)  --  returns true or false based on success or failure of update cfg command

        Raises:
            Exception:
                If  db update cfg command fails.

        """
        self.log.info(self)
        try:
            client_name = self.client.client_name
            base_path = self.simpana_base_path

            if self.platform.lower() == "windows":

                cmd = ("CALL SYSPROC.ADMIN_CMD( 'update db cfg for  {0} using LOGARCHMETH1 "
                       "''VENDOR:{1}\DB2Sbt.dll''')".format(self._database, base_path))
                self.log.info("Set LOGRETAIN ON :%s", cmd)
                self.exec_immediate_method(cmd)

            else:

                cmd = (
                    "CALL SYSPROC.ADMIN_CMD( 'update db cfg for {0} using LOGARCHMETH1 "
                    "''VENDOR:{1}/libDb2Sbt.so''')".format(
                        self._database, self.simpana_base_path))
                self.log.info("Set LOGARCHMETH1 to CV %s", cmd)
                self.exec_immediate_method(cmd)

            cmd = (
                "CALL SYSPROC.ADMIN_CMD('update db cfg for {0} using LOGARCHOPT1 ''"
                "CvClientName={1},CvInstanceName={2}''')".format(
                    self._database,
                    client_name,
                    self.simpana_instance))
            self.log.info("Set LOGARCHOPT1 options %s", cmd)
            self.exec_immediate_method(cmd)

            cmd = (
                "CALL SYSPROC.ADMIN_CMD('update db cfg for {0} using VENDOROPT ''CvClientName={1},"
                "CvInstanceName={2}''')".format(
                    self._database,
                    client_name,
                    self.simpana_instance))
            self.log.info("Set VENDOROPT options %s", cmd)
            self.exec_immediate_method(cmd)

            cmd = "CALL SYSPROC.ADMIN_CMD( 'update db cfg for {0} using LOGARCHMETH2 OFF')".format(
                self._database)
            self.log.info("Set LOGARCHMETH2 OFF :%s", cmd)
            self.exec_immediate_method(cmd)

            cmd = "CALL SYSPROC.ADMIN_CMD( 'update db cfg for {0} using TRACKMOD ON')".format(
                self._database)
            self.log.info("Set TRACKMOD ON %s", cmd)
            self.exec_immediate_method(cmd)

            cmd = "{0} connect reset".format(self.db2cmd)
            self.log.info(cmd)
            cmd_output = self.machine_object.execute_command(cmd)
            output = cmd_output.formatted_output
            self.log.info("output: %s", output)
            retcode = self.disconnect_applications(self._database)
            self.reconnect()
            if retcode != 0:
                return False

            return True

        except Exception as excp:
            raise Exception(
                "Exception raised at update_db2_database_configuration:  '{0}'".format(excp))

    def get_db2_version(self):
        """
        Returns the DB2 application version level

        Raises:
            Exception:
                if unable to get db2 version

        """
        try:
            cmd = "SELECT SERVICE_LEVEL FROM TABLE(SYSPROC.ENV_GET_INST_INFO())"
            self.log.info(cmd)
            stmt = ibm_db.exec_immediate(self._connection, cmd)
            output = ibm_db.fetch_assoc(stmt)

            self.log.info("DB2 version: %s", (output['SERVICE_LEVEL']))
            return output['SERVICE_LEVEL']
        except Exception as excp:
            raise Exception(
                "Exception raised at get_db2_version :  '{0}'".format(excp))

    def close_db2_connection(self):
        """
        Closes db2 ibm_db connection

        Raises:
            Exception:
                if unable to close db2 connection

        """
        try:
            self.log.info("closing db2 ibm_db connection")
            ibm_db.close(self._connection)
        except Exception as excp:
            raise Exception(
                "Exception raised at close_db2_connection :  '{0}'".format(excp))

    def get_db2_information(self):
        """
        gets information about db2home, db2 databases, simpana instance name, simpana base path

        Raises:
            Exception:
                if unable to get db2 information

        """

        try:
            self.log.info("Instance name is : %s", self.simpana_instance)

            self.log.info("Simpana Base Path : %s", self.simpana_base_path)
            db2_home_path = self._db2_home_path
            self.log.info("DB2 Home Path : %s", db2_home_path)
            if self.platform.lower() != "windows":
                self.db2_profile = "{0}/sqllib/db2profile".format(
                    db2_home_path)
                self.db2cmd = "{0};db2".format(self.db2_profile)

            db2_database_list = self.list_database_directory()
            self.log.info("DB2 Database List is : %s", db2_database_list)

            return db2_home_path, db2_database_list, self.simpana_instance, self.simpana_base_path
        except Exception as excp:
            raise Exception(
                "Exception raised at get_db2_information:  '{0}'".format(excp))

    def list_database_directory(self):
        """
        method to list all database directories

        Returns: (list) - list of database aliases

        """
        cmd = "db2 list database directory"
        self.log.info(cmd)
        cmd_output = self.machine_db2object.execute_command(cmd)
        output = cmd_output.formatted_output
        self.log.info("%s", output)
        return output

    def get_database_aliases_helper(self, alias_helper):
        """
        Helper method that retrieves a list of the database aliases from the database directory
        Args:
            alias_helper    (str)   -- database name need to ba passed

        Returns:

            (list) - List of databases

        """
        database_list = []
        list_db = alias_helper.split()
        count = 0

        while True:
            if len(list_db) > count:
                if list_db[count].lower() == 'alias':
                    alias = list_db[count + 2]
                    database_list.append(alias)
                    count = count + 1
                else:
                    count = count + 1
            else:
                break
        self.log.info(database_list)

        return database_list

    def third_party_command_backup(self, db_name, backup_type):
        """
        Uses command line to submit backup job on database.

        Args:

            db_name        (str)   -- database name

            backup_type    (str)   -- backup job type like FULL, INCREMENTAL , DELTA

        Returns:

            str - backup image timestamp

        Raises:
            Exception:
                if incorrect backup type is given or if any issue in running the backup job

        """
        try:
            self.log.info(self.load_path)

            if backup_type.upper() == "FULL":
                cmd = "{0} backup db {1} online load \"'{2}'\"".format(
                    self.db2cmd, db_name, self.load_path)
            elif backup_type.upper() == "DELTA":
                cmd = "{0} backup db {1} online incremental delta load \"'{2}'\"".format(
                    self.db2cmd, db_name, self.load_path)
            elif backup_type.upper() == "INCREMENTAL":
                cmd = "{0} backup db {1} online incremental load \"'{2}'\"".format(
                    self.db2cmd, db_name, self.load_path)
            else:
                raise Exception("Incorrect backup type entered")

            output = self.third_party_command(cmd)

            output = output.replace(" ", "")
            output = output.replace("\n", "")
            output = output.split(":")
            backup_time_stamp1 = output[1]

            return backup_time_stamp1
        except Exception as excp:
            raise Exception(
                "Exception raised at third_party_command_backup: '{0}'".format(excp))

    def third_party_command_restore(
            self, db_name, backup_time, version, restore_cycle=False):
        """
        Uses command line to submit restore job on database

        Args:

            db_name         (str)       -- database name

            backup_time     (str)       -- backup image timestamp

            version         (Str)       -- db2 application version

            restore_cycle   (bool)      -- whether to restore entire cycle of backup images or not
                    default: False

        Raises:

            Exception:
                if any issue occurs in triggering the cli restore

        """
        try:
            is_greater = self.compare_db2_versions(version)
            if is_greater:
                self.log.info(self.load_path)
                if restore_cycle:
                    cmd = (
                        "{0} restore db {1} incremental automatic load \"'{2}'\" open 2 sessions "
                        "taken at {3} without prompting".format(
                            self.db2cmd,
                            db_name,
                            self.load_path,
                            backup_time))
                else:
                    cmd = (
                        "{0} restore db {1} load \"'{2}'\" open 2 sessions taken at {3} without "
                        "prompting".format(
                            self.db2cmd,
                            db_name,
                            self.load_path,
                            backup_time))
                output = self.third_party_command(cmd)
            else:
                if restore_cycle:
                    cmd = (
                        "{0} restore db {1} incremental automatic load \"'{2}'\" taken at {3} "
                        "without prompting".format(
                            self.db2cmd,
                            db_name,
                            self.load_path,
                            backup_time))
                else:
                    cmd = ("{0} restore db {1} load \"'{2}'\" taken at {3} without "
                           "prompting".format(self.db2cmd, db_name, self.load_path, backup_time))
                output = self.third_party_command(cmd)

            if str(output).find("'Restore', 'is', 'successful") or str(
                    output).find("Restore is successful") >= 0:
                self.log.info("CLI restore is successful :%s", output)
            else:
                raise Exception(
                    "CLI restore is not successful :{0}".format(output))

            self.third_party_command_rollforward(db_name)
        except Exception as excp:
            raise Exception(
                "Exception raised at third_party_command_restore: '{0}'".format(excp))

    def third_party_command_rollforward(self, db_name):
        """
        Uses command line to Rollforward database

        Args:
            db_name (str) -- database name

        Raises:
            Exception:
                if commandline rollforward fails

        """
        try:
            cmd = "{0} rollforward db {1} to end of logs and complete".format(
                self.db2cmd, db_name)
            output = self.third_party_command(cmd)
            if str(output).find("'not', 'pending'") or str(
                    output).find("not pending") >= 0:
                self.log.info("Rollforward was successfull: %s", output)
            else:
                raise Exception(
                    "Rollforward is not successfull: {0}".format(output))
        except Exception as excp:
            raise Exception(
                "Exception raised at third_party_command_rollover: '{0}'".format(excp))

    def third_party_command_recover(self, db_name):
        """
        Uses command line to recover database.

        Args:
            db_name (str) -- database name

        Raises:
            Exception:
                If command line recover fails

        """
        try:
            self.log.info(self.load_path)
            cmd = "{0} recover database {1}".format(self.db2cmd, db_name)
            output = self.third_party_command(cmd)
            self.log.info("%s", output)
            if str(output).find("DB20000I") >= 0:
                self.log.info("Recover was successful: %s", output)
            else:
                raise Exception(
                    "Recover is not successful: {0}".format(output))

        except Exception as excp:
            raise Exception(
                "Exception raised at third_party_command_recover: '{0}'".format(excp))

    def db2_archive_log(self, db_name, archive_number_of_times):
        """
        Uses command line to archive db2 logs.

        Args:
            db_name (str) -- database name
            archive_number_of_times (int)   -- log archival count

        Raises:
            Exception:
                If command line archive log fails

        """
        try:
            count = 1
            while count < archive_number_of_times:
                archive_cmd = "{0} archive log for db {1}".format(
                    self.db2cmd, db_name)
                self.third_party_command(archive_cmd)
                count += 1

        except Exception as excp:
            raise Exception(
                "Exception raised at third_party_command_recover: '{0}'".format(excp))

    def third_party_command(self, cmd):
        """
        Module to execute third party commands

        Args:
            cmd (str) -- db2 application command

        Returns:
            str -- output of the command that is executed in db2 cli interface

        Raises:
            Exception:
              if third party command fails to execute

        """
        try:
            self.log.info(cmd)
            cmd_output = self.machine_db2object.execute_command(cmd)
            self.log.info(cmd_output)
            output = cmd_output.formatted_output
            self.log.info(output)
            return output
        except Exception as excp:
            raise Exception(
                "Exception raised at third_party_command: '{0}'".format(excp))

    def db2_cold_backup(self, cold_backup_path, db_name):
        """
        Takes DB2 cold backup

        Args:
            cold_backup_path (str) -- path on disk to backup db2 database

            db_name (str) -- database name

        Returns:
            str     -- cold backup image timestamp
            bool    -- true if cold backup is already taken

        Raises:
            Exception:
                if any issue occurs with cold backup

        """
        try:

            self.log.info("Check if backup is already exists")

            if self.platform.upper() != 'WINDOWS':
                cmd = "touch {0} timestamp.txt".format(cold_backup_path)
                self.log.info(cmd)
                cmd_output = self.machine_object.execute_command(cmd)
                self.log.info(cmd_output)
                output = cmd_output.formatted_output
                self.log.info(output)

                if str(output).find("Backup successful") >= 0:
                    self.log.info("Cold backup is already taken")
                    return True

            self.disconnect_applications(self._database)
            cmd = "{0} -z '{1}'  backup database {2} to '{1}'".format(
                self.db2cmd, cold_backup_path, db_name)
            self.log.info(cmd)
            cmd_output = self.machine_object.execute_command(cmd)
            self.log.info(cmd_output)
            output = cmd_output.formatted_output
            self.log.info(output)
            output = self.third_party_command(cmd)

            if str(output).find("Backup successful") >= 0:
                self.log.info("Cold backup was successful")
            else:
                raise Exception("Cold backup was not successful. Please check")

            output = output.replace(" ", "")
            output = output.replace("\n", "")
            output = output.split(":")
            backup_time_stamp1 = output[1]

            return backup_time_stamp1
        except Exception as excp:
            raise Exception(
                "Exception raised at db2_cold_backup : '{0}'".format(excp))

    def restore_from_cold_backup(
            self, db_name, cold_backup_path, backup_time_stamp):
        """
        If database is down, restores from cold backup

        Args:
            db_name             (str)   -- database name
            cold_backup_path    (str)   -- disk path where cold backup image exists
            backup_time_stamp   (str)   -- backup image timestamp

        Raises:
            Exception :
                If any issue occurs while running restore from cold backup
                If restore command fails

        """

        try:

            self.log.info(
                "if any incremental restore is in progress abort that restore")
            cmd = "{0} restore db {1} incremental abort ".format(
                self.db2cmd, db_name)
            self.log.info(cmd)
            cmd_output = self.machine_object.execute_command(cmd)
            self.log.info(cmd_output)
            output = cmd_output.formatted_output
            self.log.info(output)
            self.log.info("Restoring from cold backup")
            cmd = ("{0} restore db {1} from '{2}' taken at  {3} without rolling forward without "
                   "prompting ".format(self.db2cmd, db_name, cold_backup_path, backup_time_stamp))
            self.log.info(cmd)
            cmd_output = self.machine_object.execute_command(cmd)
            self.log.info(cmd_output)
            output = cmd_output.formatted_output
            self.log.info(output)

            if str(output).find("Restore is successful") >= 0:
                self.log.info("Restore from Cold backup is successful")
            else:
                raise Exception(
                    "Restore from Cold backup is not successful. Please check")

        except Exception as excp:
            raise Exception(
                "Exception raised at restore_from_db2_cold_backup : '{0}'".format(excp))

    def compare_db2_versions(self, vers1):
        """
        Helper method that compares versions

        Args:
            vers1 (str) -- db2 application version for the given db2 instance

        Returns:

              (bool)    --  be returns if db2 version is less than 10.5.0.8

        """
        vers1 = vers1[5:]
        vers1 = vers1.split(".")
        vers1 = int(vers1[0] + vers1[1] + vers1[2] + vers1[3])
        vers2 = "10.5.0.8"
        vers2 = vers2.split(".")
        vers2 = int(vers2[0] + vers2[1] + vers2[2] + vers2[3])

        if vers2 <= vers1:
            self.log.info(
                "----------db2 version is greater than or equal to v10.5fp8------------")
            return True
        return False

    def create_database(self, db_name):
        """
        It will create database with given name and takes a base backup.

        Args:
            db_name (str) -- database name

        Raises:
            Exception:
                If issue occurs while creating database
                If disk backup after database creation fails
                If any other issue occurs in try block

        """
        try:
            cmd = (
                "{0} create database {1} automatic storage yes alias {1} using codeset "
                "utf-8 territory us collate using identity pagesize 4 K".format(
                    self.db2cmd, db_name))
            self.log.info(cmd)
            cmd_output = self.machine_object.execute_command(cmd)
            self.log.info(cmd_output)
            output = cmd_output.formatted_output
            self.log.info(output)

            self.log.info(output)
            if str(output).find("DB20000I") >= 0:
                self.log.info("Database Created %s", db_name)
                cmd = "{0} backup database {1}".format(self.db2cmd, db_name)
                cmd_output = self.machine_object.execute_command(cmd)
                self.log.info(cmd_output)
                output = cmd_output.formatted_output
                self.log.info(output)

                if str(output).find("Backup successful") >= 0:
                    self.log.info(
                        "first time backup after database creation is successful: %s",
                        output)

                else:
                    raise Exception(
                        "first time backup after database creation is not successful :{0}".format(
                            output))

            else:
                raise Exception("Database is not created {0}".format(db_name))

        except Exception as excp:
            raise Exception(
                "Exception raised at create_database  : '{0}'".format(excp))
