# Python imports
import os
import time

# Project imports
import cvmanager_task_step
import cvmanager_task
from cvmanager_task_arg import TaskArg
import cvupgrade_common
import cvmanager_defines
import task_defines
from HsObject import hs_node
import wrappers
import common


class Task(cvmanager_task.TaskObject):
    """ This Task() is strictly for upgrading the current node.   In parent Upgrade(Task) operations, this
    Upgrade_Node task is what will be launched on all remote nodes and itself.  Therefore, this task needs to be
    isolated.  It is assumed that when it's executed, it will upgrading itself.  No calls should be remote.
    """
    task_args = {
            # This hostname will be specified if launched from parent task.
            'hostname': TaskArg('hostname', str),

            # log_dir = /ws/ddb/cvmanager/share/<hostname>  <- This would be set when launched as part of parent task.
            'log_dir': TaskArg('log_dir', str,
                               default_value=os.path.join(common.get_cvlt_logdir(), cvupgrade_common.debug_logdir))
        }

    @cvmanager_task_step.OptionStep.run_always
    def copy_repo_from_share(self, *args, **kwargs):
        cvupgrade_common.copy_repo_file_v2(cvmanager_defines.ACT_REPO_FILE)
        self.log.info("Copied repo file [{0}] to /etc/yum.repos.d".format(cvmanager_defines.ACT_REPO_FILE))
        return True

    @cvmanager_task_step.OptionStep.run_always
    def init(self, *args, **kwargs):
        # Setup node of self; node_type is irrelevant for this Task, so don't worry about setting it.
        node = hs_node.HyperScaleNode(self.kwargs.get('hostname', wrappers.get_hostname()))
        setattr(self, 'NODE', node)

        self.log.info("=================================================")
        self.log.info("Initializing Node Upgrade")
        self.log.info("=================================================")

        # Checking if lvscan is stuck on any of the nodes, if so do not proceed with upgrade
        if not node.check_lvscan():
            self.log.error("It looks the process lvscan is stuck on [{0}]...Please make sure it runs successfully on "
                           "all nodes and rerun OS upgrade.")
            return False

        if not node.disable_multipath():
            self.log.error("Failed to disable multipath on node [{0}].".format(node.hostname))
            return False

        # Do not proceed if other repos are available on the machine
        if not cvupgrade_common.check_for_repos(cvmanager_defines.REPO_CLEANUP):
            return False

        cvupgrade_common.save_prev_rhel_ver()

        return True

    @cvmanager_task_step.TaskStep
    def uninstall_conflicting_packages(self, *args, **kwargs):
        node = getattr(self, 'NODE')
        self.log.info("Checking and removing conflicting packages.")
        if not node.uninstall_conflicting_packages():
            return False
        return True

    @cvmanager_task_step.TaskStep
    def yum_upgrade(self, *args, **kwargs):
        """
        cvmanager_defines.TaskDir.share is the working directory for this task.
        Args:
            *args:
            **kwargs:

        Returns:

        """
        log_dir = self.kwargs.get('log_dir')

        wrappers.mkdir(log_dir)
        stdout_file = open(log_dir + "/yum.out.log", "w")
        stderr_file = open(log_dir + "/yum.err.log", "w")

        self.log.info("Installing rpms...this will take several minutes...Please wait")
        self.log.info("Please see [" + log_dir + "/yum.out.log] for detailed progress")
        cmd = "yum -y update"

        from subprocess import Popen, PIPE, STDOUT
        p = Popen(cmd, shell=True, stdin=PIPE, stdout=stdout_file, stderr=stderr_file)
        ret = p.wait()
        stdout_file.close()
        stderr_file.close()

        # Always cleanup the repo regardless or pass or fail....re-running upgrade always re-copies the repo.
        cvupgrade_common.delete_repo_file_v2(cvmanager_defines.ACT_REPO_FILE)

        # Always restore the initramfs; never trust for another step to do it, thats why its part of this step.
        if not self.restore_initramfs():
            return False

        if not ret == 0:
            self.log.error("yum update failed ...")
            return False

        # Append the yum logs; why don't we do this in all cases, not just success?
        local_log_dir = os.path.join(common.get_cvlt_logdir(), cvupgrade_common.debug_logdir)
        yum_logfile = os.path.join(local_log_dir, 'upgrade_yum.log')

        wrappers.waitsystem("echo `date` >> " + yum_logfile)
        wrappers.waitsystem("cat " + log_dir + "/yum.out.log >> " + yum_logfile)

        return True

    @cvmanager_task_step.TaskStep
    def set_grub_default(self, *args, **kwargs):
        node = getattr(self, 'NODE')

        self.log.info("Set grub default with new kernel if not already set ...")
        if not node.set_grub_default():
            return False

        return True

    @cvmanager_task_step.TaskStep
    def check_upgrade_and_reboot(self, *args, **kwargs):
        """ Validates if the upgrade was successful and requires reboot based on the yum.out.log.
        If this exits with REBOOT_AND_RESUME code, the task manager framework will handle this, setting the reboot
        registry key, pointing to the <task input>.yaml file.  When the node reboots, the hyperscale service will
        automatically re-launch cvmanager.py with this input file to resume.

        :param args:
        :param kwargs:
        :return:
        """
        with open(self.kwargs.get('log_dir') + "/yum.out.log", "r") as stdout_file:
            if any("No packages marked for update" in line for line in stdout_file):
                self.log.info("It looks machine is already upgraded...No packages marked for update by yum.")

                # Skip reboot only if node is already running latest kernel
                if cvupgrade_common.isrunning_latest_kernel():
                    self.log.info("Skipping reboot as the machine is already running latest kernel.")
                    return True
            else:
                self.log.info("Installed rpms successfully...node will reboot and resume.")
                return task_defines.StepStatus.REBOOT_AND_RESUME

        # If we've reached this point, then something went wrong, like missing yum.out.log, etc.  Fail upgrade.
        return False

    @cvmanager_task_step.TaskStep
    def yum_clean(self, *args, **kwargs):
        command = "/usr/sbin/subscription-manager unregister"
        ret = wrappers.waitsystem_nostdout(command)
        command = "yum clean all"
        ret = wrappers.waitsystem_nostdout(command)
        if not ret == 0:
            self.log.error("yum clean failed ...")
            return False
        return True

    @cvmanager_task_step.TaskStep
    def post_reboot(self, *args, **kwargs):
        # Post upgrade post reboot operations on this node.
        self.log.info("Node post-upgrade.")
        node = hs_node.HyperScaleNode(self.kwargs.get('hostname', wrappers.get_hostname()))

        node.enable_multipath()

        # Setting this, but history file is used to determine upgrade eligibility now anyway.
        common.setregistryentry(cvupgrade_common.maregistry, "nHyperScaleLastUpgradeTime", str(int(time.time())))
        return True

    @cvmanager_task_step.TaskStep
    def backup_initramfs(self, *args, **kwargs):
        if not cvupgrade_common.backup_initramfs():
            return False
        return True

    def restore_initramfs(self):
        if not cvupgrade_common.restore_initramfs():
            return False
        return True

    def set_process(self, process):
        process.pre_process = [
            self.init,
            self.copy_repo_from_share
        ]
        process.main_process = [
            self.uninstall_conflicting_packages,
            self.yum_clean,
            self.backup_initramfs,
            self.yum_upgrade
        ]
        process.post_process = [
            self.set_grub_default,
            self.check_upgrade_and_reboot,
            self.post_reboot
        ]
