# Python imports
import time
import os

# Project imports
import cvmanager_task_step
import cvupgrade_common
from cvmanager_task_arg import TaskArg
import Upgrade
from HsObject import hs_gluster_block
from HsObject import hs_ovirt

_GLUSTER_BLOCK_KEY = 'GLUSTER_BLOCKS'


# Extend the main upgrade task, use this model so we can reuse some of the same code and anything handled in Upgrade.
class Task(Upgrade.Task):
    """ Gluster upgrade splits the nodes into blocks and upgrade all blocks in parallel, but the nodes in the block,
    sequentially.

    Everything we do for the gluster upgrade should focus on blocks, not nodes.  So any tasks are performed at the block
    level, and a block is made up of nodes.

    """

    def set_process(self, process):
        process.pre_process = [
            self.get_gluster_blocks,
            self.virtual_machine_checks,
            self.restart_gluster_all_nodes,
            self.check_bricks,
            self.stop_commvault_services,
            self.stop_gluster_volumes,
            self.stop_gluster_all_nodes
        ]

        process.main_process = [
            self.upgrade_blocks,
            self.upgrade_self
        ]

        process.post_process = [
            self.gluster_post_upgrade,
            self.start_gluster_volumes,
            self.start_services,
            self.virtual_machine_post_upgrade
        ]

    @cvmanager_task_step.OptionStep.run_always
    def get_gluster_blocks(self, *args, **kwargs):
        """get hs_gluster_block.GlusterBlock() objects for each detected block
        This should always run because many steps in this Task are dependent on gluster block config.
        """
        gluster_blocks = hs_gluster_block.get_gluster_blocks(self.kwargs.get('StoragePoolName', None))

        setattr(self, _GLUSTER_BLOCK_KEY, gluster_blocks)

        return True

    @cvmanager_task_step.TaskStep
    def restart_gluster_all_nodes(self, *args, **kwargs):
        for block in getattr(self, _GLUSTER_BLOCK_KEY, []):
            if not block.restart_cluster_all_nodes():
                return False
        return True

    @cvmanager_task_step.OptionStep.run_always
    def stop_gluster_all_nodes(self, *args, **kwargs):
        # Always do this because upgrade will fail in if glusterd is running.
        # We also always restart glusterd in failed upgrade cases, so we should always make sure this is down.
        for block in getattr(self, _GLUSTER_BLOCK_KEY, []):
            if not block.stop_gluster_on_all_nodes():
                return False
        return True

    @cvmanager_task_step.TaskStep
    def check_bricks(self, *args, **kwargs):
        attempts = 3
        bricks_online = False
        for i in range(attempts):
            if cvupgrade_common.checkif_bricks_online():
                bricks_online = True
                break
            self.log.warning("Re-checking brick status in 10 seconds.")
            time.sleep(10)
        else:
            self.log.error("Reached maximum attempts waiting for bricks to come online. Exiting.")
        return bricks_online

    @cvmanager_task_step.TaskStep
    def stop_gluster_volumes(self, *args, **kwargs):
        # This call fails if volume is already down, only do it one time per upgrade.
        if not cvupgrade_common.stop_all_gluster_volumes():
            return False
        return True

    @cvmanager_task_step.OptionStep.run_always
    def stop_commvault_services(self, *args, **kwargs):
        """ Stop commvault services on all node.
        """
        for block in getattr(self, _GLUSTER_BLOCK_KEY, []):
            if not block.stop_services_on_all_nodes():
                return False
        return True

    @cvmanager_task_step.TaskStep
    def upgrade_blocks(self, *args, **kwargs):
        """For gluster upgrade, all blocks are upgraded in parallel, and nodes within the block are upgraded
        sequentially."""
        upgrade_tasks = []
        for block in getattr(self, _GLUSTER_BLOCK_KEY, []):
            self.log.info("Beginning upgrade of gluster block: {0}".format(block))
            self.kwargs['block'] = block
            upgrade_tasks.append(self.create_child_task('Upgrade_Block', launch=False, **self.kwargs))

        # Launch and wait for the tasks to complete.
        self.wait_for_child_tasks(upgrade_tasks, launch=True)
        return_code = self.check_and_return_all_child_task_statuses(upgrade_tasks)
        if not return_code:
            self.log.error("Failed upgrading the block [{}].".format(block))

            # When upgrade fails on any gluster block, ensure glusterd is started on all nodes.
            for block in getattr(self, _GLUSTER_BLOCK_KEY, []):
                block.start_cluster_all_nodes()
            return return_code

        return return_code

    @cvmanager_task_step.TaskStep
    def upgrade_self(self, *args, **kwargs):
        self.log.info("Upgrading local gluster node.")  

        # Figure out which node this is
        local_node = None
        for block in getattr(self, _GLUSTER_BLOCK_KEY, []):
            for node in block.nodes:
                if node.local_node:
                    local_node = node
                    break
            else:
                self.log.info("No local node found in block [{0}].".format(block))
            if local_node is not None:
                break
        else:
            self.log.error("Unable to find the local node in any blocks.  Exiting!")
            return False

        # FYI this task will reboot the machine if completed, meaning it will fail, but when rebooted and resumed
        # and this step is re-run, we will then proceed.
        upgrade_task = self.create_child_task('Upgrade_Node', launch=True, **self.kwargs)

        upgrade_task.wait_for_child_task()

        # In the case of reboot being needed, remember and return this return code.
        ret_code = upgrade_task.check_and_return_child_task_status()
        if not ret_code:
            # Upgrade failed on this node, perform failure upgrade clean up operations.
            self.log.error("Failed to upgrade local node; restarting gluster.")

            # When upgrade fails on local node, ensure glusterd is left running.
            local_node.start_glusterd()

        return ret_code

    @cvmanager_task_step.TaskStep
    def virtual_machine_checks(self, *args, **kwargs):
        """ These checks are for the VMs only.  Any stand alone nodes need not be included.
        We do not want to re-run this step if it's already been executed as part of this upgrade Task(), as the VMs
        may already be down and we don't want to unintentionally perform vm management.

        """
        he_deployed = self.kwargs.get('he_deployed', False)
        manage_vms = self.kwargs.get('manage_vms', False)

        if not cvupgrade_common.is_he_deployed():
            self.log.info("It looks hosted engine is not deployed. Assuming that CommServe is not running on the VM "
                          "on MediaAgent nodes ... skipping VM management.")
        else:
            he_deployed = True
            self.log.info("It appears hosted engine is deployed.")
            if manage_vms:
                self.log.info("CommServe and Hosted Engine VMs will now be shutdown/powered off and started after "
                              "upgrade is complete.")
            else:
                self.log.warning("Please make sure both CommServe and Hosted-Engine VMs are shutdown and powered off "
                                 "before starting upgrade.")

        if not manage_vms or not he_deployed:
            self.log.info("No virtual machine checks pre-upgrade checks required.")
            return True
        else:
            ovirt_host = hs_ovirt.HSoVirt()
            if not ovirt_host.is_initialized():
                return False

        if not cvupgrade_common.check_he_status(cvupgrade_common.START_VM):
            self.log.error("Unable to get status for Hosted engine:\n"
                           "Below are possible reasons, please correct and rerun the upgrade.\n"
                           "\t1. Please make sure the password is correct.\n"
                           "\t2. Please make sure the VM is up and running.\n"
                           "\t3. If the VM is down and you do not want us to manage, rerun the program with option "
                           "-no_vm_management.")
            return False

        # Find which node is running the hosted engine
        he_node = None
        for block in getattr(self, _GLUSTER_BLOCK_KEY, []):
            for node in block.nodes:
                if node.is_running_he(ovirt_host):
                    self.log.info("Hosted engine VM is running on node [{0}].".format(node))
                    he_node = node

        if he_node is None:
            self.log.error("Could not determine the node on which Hosted engine VM is running")
            return False

        # Stop the CS VM
        if not cvupgrade_common.stop_cs_vm(ovirt_host.host, ovirt_host.user, cvupgrade_common.decp(ovirt_host.ename)):
            self.log.error("There was an error while stopping CS VM. Try stopping it manually and rerun upgrade.")
            return False

        self.log.info("Commserve virtual machine stopped successfully ...")

        # Stop the HE VM
        if not he_node.stop_he_vm():
            self.log.error("There was an error while stopping Hosted Engine VM on node [{0}]. ".format(he_node))
            return False

        self.log.info("Hosted engine virtual machine stopped successfully ...")
        return True

    @cvmanager_task_step.TaskStep
    def gluster_post_upgrade(self, *args, **kwargs):
        # There is a chance that glusterd won't be running if reboot was not required, so make sure its started before
        # running gluster commands.
        # Figure out which node this is
        local_node = None
        for block in getattr(self, _GLUSTER_BLOCK_KEY, []):
            for node in block.nodes:
                if node.local_node:
                    local_node = node
                    break
            else:
                self.log.info("Local node not found in block [{0}].".format(block))

            if local_node is not None:
                break
        else:
            self.log.error("Unable to find the local node in any blocks.  Exiting!")
            return False

        for block in getattr(self, _GLUSTER_BLOCK_KEY, []):
            if not block.start_cluster_all_nodes():
                return False

        # Upgrade was successful on this gluster node; perform gluster post op for this node.
        # Fixing gluster volume size display, some times is not correctly displayed
        if not local_node.fix_gluster_vol_size():
            return False

        ret = cvupgrade_common.reflect_volsize()
        if not ret == 0:
            return False

        if not cvupgrade_common.set_cluster_opversion():
            return False

        return True

    @cvmanager_task_step.TaskStep
    def start_services(self, *args, **kwargs):
        for block in getattr(self, _GLUSTER_BLOCK_KEY, []):
            if not block.start_services_on_all_nodes():
                return False
        return True

    @cvmanager_task_step.TaskStep
    def start_gluster_volumes(self, *args, **kwargs):
        if not cvupgrade_common.start_all_gluster_volumes():
            return False
        return True

    @cvmanager_task_step.TaskStep
    def virtual_machine_post_upgrade(self, *args, **kwargs):
        he_deployed = self.kwargs.get('he_deployed', False)
        manage_vms = self.kwargs.get('manage_vms', False)

        if cvupgrade_common.is_he_deployed():
            he_deployed = True

        if he_deployed and manage_vms:
            # Initialize the ovirt object
            ovirt_host = hs_ovirt.HSoVirt()
            if not ovirt_host.is_initialized():
                return False

            if not cvupgrade_common.start_he_vm():
                self.log.error("Failed re-starting hosted engine virtual machine.")
                return False
            self.log.info("Successfully re-started hosted engine virtual machine.")

            if not cvupgrade_common.start_cs_vm(ovirt_host.host, ovirt_host.user,
                                                cvupgrade_common.decp(ovirt_host.ename)):
                self.log.error("Failed re-starting CommServe virtual machine.")
                return False
            self.log.info("Successfully re-started CommServe virtual machine.")

        return True
