# Python imports

# Project imports
import cvmanager_task_step
import cvmanager_task
from cvmanager_task_arg import TaskArg
import cvmanager_defines
from HsObject import hs_gluster_block


class Task(cvmanager_task.TaskObject):
    """ Task() designed to take a formed GlusterBlock as input, and upgrade the nodes in the block sequentially.

    """
    task_args = {
        'block': TaskArg('nodes', hs_gluster_block.GlusterBlock, required=True)
    }

    @cvmanager_task_step.OptionStep.with_options({'display_name': 'Upgrading Remote Nodes of the Block.'})
    def upgrade_block_nodes(self, *args, **kwargs):
        block = self.kwargs['block']

        for node in block.nodes:
            if node.local_node:
                self.log.warning("Skipping upgrade on local node [{0}] because it's the control node and will be "
                                 "upgraded after all other nodes in all blocks are complete.".format(node))
                continue

            self.log.info("Upgrading node [{0}] in block [{1}].".format(node, block))
            self.kwargs['log_dir'] = cvmanager_defines.TaskDir.share_string.format(node.hostname)
            proc = self.create_remote_child_task('Upgrade_Node', node, wait=False, launch=True, **self.kwargs)

            self.wait_for_remote_process(proc)

            # If any node in the block fails to upgrade, stop upgrading immediately, and do not upgrade any other nodes.
            if not self.check_and_return_remote_process_status(proc):
                # Upgrade failed on this node, perform failure upgrade clean up operations.
                self.log.error("Failed to upgrade remote node [{0}].".format(node))

                # If a node fails, bring the glusterd back up on that node.
                node.start_cluster()
                return False

            # Upgrade was successful on this node.  Fix the gluster block size
            if not node.fix_gluster_vol_size():
                return False

        return True

    def set_process(self, process_object):
        process_object.main_process = [
            self.upgrade_block_nodes
        ]
