# Python imports
import os
import time
import datetime

# Project imports
import cvmanager_task_step
import cvmanager_task
from HsObject import hs_node
import cvmanager_defines
import common
import cvupgrade_common
import cvupgradeos
from HsObject import hs_defines
from HsObject import hs_utils
import cvmanager_catalog
from cvmanager_task_arg import TaskArg
from collections import defaultdict


_UPGRADE_HISTORY_FILE = 'upgrade_history.json'
_REMOTE_NODES_KEY = 'REMOTE_NODES'


class Task(cvmanager_task.TaskObject):
    task_args = {
        'remote_nodes': TaskArg('remote_nodes', list),
        'excluded_nodes': TaskArg('excluded_nodes', list),
        'manage_vms': TaskArg('manage_vms', bool, default_value=True, valid_values=['True', 'False']),
        'silent': TaskArg('silent', bool, default_value=False, valid_values=['True', 'False'],
                          description="If True, non-interactive."),
        'force_upgrade_local_node': TaskArg('force_upgrade_local_node', bool, valid_values=['True', 'False'],
                                            default_value=True, description="Not currently implemented.")  # Not implemented
    }

    def set_process(self, process_object):
        process_object.pre_process = [
            self.collect_nodes,
            self.pre_upgrade_requirements,
            self.create_registry,
            self.create_repo
        ]

        process_object.main_process = [
            self.launch_node_upgrade_tasks
        ]

        process_object.post_process = [
            self.save_upgrade_history_for_nodes,
            self.cleanup
        ]

    def check_if_upgrade_required(self, nodes):
        """ Using the Catalog() framework, get the history file for this task and see if these nodes are required
        to be upgraded.

        Args:
            nodes: list - hs_node.HSNode() object.  This list of objects should be HyperScale nodes.

        Returns:

        """
        global _UPGRADE_HISTORY_FILE

        # Include the control node in history file checks; in case its different control node than previously
        local_node = hs_node.HyperScaleNode()

        # Initialize the task catalog for the history file.
        task_catalog = cvmanager_catalog.Catalog()
        history_file = task_catalog.get_file(_UPGRADE_HISTORY_FILE)
        self.log.info("Current upgrade history file: {0}".format(history_file.file_path))

        # Get and check the cache sync time.
        cache_sync_time = hs_utils.get_last_sw_cache_sync_time()
        if cache_sync_time == 0:
            raise Exception("Software cache has not been synchronized! Upgrade cannot proceed.  Please be sure to "
                            "setup remote software cache and run download software job prior to attempting upgrade.")
        else:
            self.log.info("Software cache last sync time is [{0}].".format(datetime.datetime.fromtimestamp(
                cache_sync_time).strftime(cvmanager_defines.READABLE_TIME_FORMAT)))

        # Check if the history file has any data; if not initialize it to 0
        if len(history_file.lines) == 0:
            # This is a new history file, so assume ALL nodes need upgrade
            self.log.info("No existing upgrade history is found, assuming all nodes require upgrade.")

            # Create the empty history file, in case anything goes south during upgrade.
            history_json = {}
            for node in nodes:
                node.upgrade_required = True
                history_json[node.hostname] = {'last_upgrade_time': '0'}

            history_file.write(history_json, cvmanager_catalog.FileType.JSON)

        # Existing history file.  Serialize the json.
        history_json = history_file.serialize_json()

        nodes.append(local_node)
        for node in nodes:
            # Check for the node in the JSON, if it's NOT there...assume upgrade required, otherwise compare time.
            if history_json.get(node.hostname, False):
                # Node history exists.
                upgrade_timestamp = int(history_json[node.hostname].get('last_upgrade_time', time.time()))
                upgrade_time = datetime.datetime.fromtimestamp(upgrade_timestamp).strftime(
                    cvmanager_defines.READABLE_TIME_FORMAT)
                self.log.debug("Node [{0}] upgrade time [{1}].".format(node, upgrade_time))

                if self.kwargs.get('force_upgrade_local_node', False) and node.local_node:
                    # Check the upgrade history on the control node.
                    self.log.info("Forcing upgrade of the control node [{0}].".format(node))
                    node.upgrade_required = True
                elif upgrade_timestamp >= cache_sync_time:
                    self.log.warning("Upgrade not required for node [{0}].  Previous upgrade time [{1}].".format(
                        node, upgrade_time))
                    node.upgrade_required = False
                else:
                    self.log.debug("Proceeding with node [{0}] upgrade.".format(node))
                    node.upgrade_required = True
            else:
                # Node does not exist in the history file, upgrade it!
                self.log.debug("No upgrade history for node [{0}], proceeding with node upgrade.".format(node))
                node.upgrade_required = True

        return nodes

    @cvmanager_task_step.TaskStep.with_options({'always_run': True,
                                                'display_name': 'Detecting nodes to upgrade.'})
    def collect_nodes(self, *args, **kwargs):
        """Get the nodes for upgrade.  See Node.collect_remote_nodes for detailed information.
        """
        global _REMOTE_NODES_KEY

        nodes = hs_node.collect_remote_nodes(self)
        setattr(self, _REMOTE_NODES_KEY, nodes)

        return True

    @cvmanager_task_step.TaskStep.with_options({'always_run': True,
                                                'display_name': 'Upgrade Pre-Requisites.'})
    def pre_upgrade_requirements(self, *args, **kwargs):
        """
        """
        global _REMOTE_NODES_KEY

        # Check for the repo directory which SHOULD be generated by the cvmanager Query_RPM task.
        if not os.path.exists(cvmanager_defines.NFS_SHARE):
            self.log.error("Directory {0} does not exist. It appears download job was not run to get "
                           "required RPMs for upgrade. Please re-run after downloadsoftware completes successfully "
                           "from Commserve GUI.".format(cvmanager_defines.NFS_SHARE))
            return False

        # Check if the upgrade is needed for any node regardless of the type.  It doesn't matter at this point.
        if any(node.upgrade_required for node in self.check_if_upgrade_required(getattr(self, _REMOTE_NODES_KEY, []))):
            self.log.info("New packages are present in SW cache, going ahead with upgrading required HyperScale nodes.")
        else:
            self.log.warning("\n\nUpgrade is not needed for any nodes at this time ... exiting.\n")
            self.log.warning("You can force upgrade by removing the upgrade history file noted above and re-running.")
            return False

        # Purge the nodes where upgrade is not needed.
        node_upgrade_list = []
        for node in getattr(self, _REMOTE_NODES_KEY, []):
            if node.upgrade_required:
                node_upgrade_list.append(node)
            else:
                self.log.debug("Removing node [{0}] from list of required nodes to upgrade.".format(node.hostname))
        setattr(self, _REMOTE_NODES_KEY, node_upgrade_list)

        # Check if SW cache is getting updated
        if cvupgradeos.checkif_swcache_busy():
            self.log.error("Upgrade cannot proceed at this time because SWcache is being updated with latest packages.")
            return False

        # Check if all remote nodes are reachable
        self.log.info("\nPerforming some pre-upgrade checks ... please wait...")
        if not cvupgradeos.checkif_allnodes_reachable_v2(getattr(self, _REMOTE_NODES_KEY)):
            self.log.error("\nIt appears all remote nodes are not accessible at this time. Please check... exiting\n")
            return False

        # Get confirmation from user
        self.log.info(" ")
        self.log.info("You are attempting to perform OS upgrade of HyperScale nodes.")
        self.log.info("This will bring down gluster services/volume and reboot the nodes one after the other. Each "
                      "configured block will be upgraded in parallel, while the nodes in the block will be upgraded "
                      "sequentially.  All non-gluster nodes will be upgraded in parallel.")

        self.log.info("Following nodes will be upgraded.")
        for node in getattr(self, _REMOTE_NODES_KEY, []):
            self.log.info("\t\t{0}".format(node.hostname))

        if not self.kwargs.get("silent", False):
            try:
                question = 'Please confirm if you want to upgrade at this time (y/n): '
                if not common.yes_no(question):
                    self.log.error("User opted not to upgrade at this time...exiting")
                    return False
            except EOFError, eof:
                # If we resume in silent mode, assume yes, just continue upgrade.
                pass
            except Exception:
                raise
        else:
            self.log.info("silent=True command line option specified, proceeding with upgrade.")

        self.log.info("Proceeding with upgrade.")

        return True

    @cvmanager_task_step.TaskStep
    def create_registry(self, *args, **kwargs):
        self.log.info("Creating registry keys before starting main upgrade ...")

        common.setregistryentry(cvmanager_defines.REG_MA_KEY, "sHyperScaleUpgradeInProgress", "Yes")
        if self.kwargs.get('is_rolling', False):
            common.setregistryentry(cvmanager_defines.REG_MA_KEY, "sHyperScaleIsRollingUpgrade", "Yes")
        if self.kwargs.get('manage_vms', False):
            common.setregistryentry(cvmanager_defines.REG_MA_KEY, "sHyperScaleUpgradeManageVMs", "No")

        return True

    @cvmanager_task_step.TaskStep
    def create_repo(self, *args, **kwargs):
        self.log.info("Creating rpm repository [{0}].".format(cvmanager_defines.NFS_SHARE_REPO_DIR))
        if not cvupgrade_common.create_repo_v2(cvmanager_defines.NFS_SHARE_REPO_DIR,
                                               cvmanager_defines.NFS_SHARE_TMP_REPO_DIR,
                                               cvmanager_defines.UPGRADE_XML):
            return False
        else:
            if not os.path.exists(cvmanager_defines.NFS_SHARE_REPO_DIR):
                self.log.error("Repository [{0}] is not present, it appears there are no packges available for "
                               "upgrade. Nothing to be done.".format(cvmanager_defines.NFS_SHARE_REPO_DIR))

                # TODO: Leave this upgrade task partially complete, it won't matter....decide what to do later.
                # conclude_upgrade()
                # cleanup()
                return False

        self.log.info("Created cvrepo successfully.")
        return True

    @cvmanager_task_step.TaskStep
    def launch_node_upgrade_tasks(self, *args, **kwargs):
        """ This is the main upgrade for all node types.  Divide and group the nodes into the known types, and launch
        the appropriate child tasks.

        For GLUSTER nodes, they're upgraded block by block.
        For STAND ALONE nodes, they're upgraded in parallel.
        For HS 2.0, they're upgraded .......unsure.

        But we should be able to upgrade ALL Gluster Blocks and all standalone nodes in parallel, which is why we start
        both child tasks immediately.

        Returns:

        """
        # nodes will be every node detected, gluster, standalone, hv, etc.  In most environments, they will all fall
        # into a single type, but lets not limit them.
        nodes = getattr(self, _REMOTE_NODES_KEY, [])
        for node in nodes:
            cur_nodes = getattr(self, 'UPGRADE_TYPES', defaultdict(list))
            cur_nodes[node.node_type.name].append(node)

            setattr(self, 'UPGRADE_TYPES', cur_nodes)

        upgrade_tasks = []
        for node_type, nodes in getattr(self, 'UPGRADE_TYPES', defaultdict(list)).items():
            # Spin up the child task for each upgrade type; these are child tasks on the controller node.
            if node_type == hs_defines.NodeTypes.STAND_ALONE.name:
                self.kwargs['nodes'] = nodes
                upgrade_tasks.append(self.create_child_task('Upgrade_StandAlone', launch=False, **self.kwargs))
            elif node_type == hs_defines.NodeTypes.GLUSTER.name:
                upgrade_tasks.append(self.create_child_task('Upgrade_Gluster', launch=False, **self.kwargs))
            elif node_type == hs_defines.NodeTypes.HEDVIG.name:
                upgrade_tasks.append(self.create_child_task('Upgrade_HV', launch=False, **self.kwargs))
            else:
                raise Exception("Unknown node type [{0}]. Cannot run upgrade".format(node_type))

        # Launch and wait for the tasks to complete.
        self.wait_for_child_tasks(upgrade_tasks, launch=True)
        return_code = self.check_and_return_all_child_task_statuses(upgrade_tasks)
        if not return_code:
            # Upgrade task failed, perform failed upgrade cleanup operations
            self.log.error("Failed upgrade node(s).  Please check logs.")

            cvupgradeos.cleanup()
            return return_code

        return return_code

    def cleanup_registry(self):
        common.deleteregistryentry(cvupgrade_common.maregistry, "sHyperScaleUpgradeInProgress")
        common.deleteregistryentry(cvupgrade_common.maregistry, "nHyperScaleUpgradePid")
        common.deleteregistryentry(cvupgrade_common.maregistry, "sHyperScaleUpgradeEname")
        common.deleteregistryentry(cvupgrade_common.maregistry, "sHyperScaleUpgradeManageVMs")
        common.deleteregistryentry(cvupgrade_common.maregistry, "sHyperScaleUpgradeHEName")
        common.deleteregistryentry(cvupgrade_common.maregistry, "sHyperScaleIsRollingUpgrade")
        common.deleteregistryentry(cvupgrade_common.maregistry, "sHyperScaleUpgradePending")

    @cvmanager_task_step.TaskStep
    def cleanup(self, *args, **kwargs):
        self.log.info("Cleanup - Deleting registry entries and repo file etc ...")
        self.cleanup_registry()

        if not cvupgrade_common.move_repos_back():
            return False

        return True

    @cvmanager_task_step.TaskStep
    def save_upgrade_history_for_nodes(self, *args, **kwargs):
        global _REMOTE_NODES_KEY, _UPGRADE_HISTORY_FILE

        remote_nodes = getattr(self, _REMOTE_NODES_KEY, [])
        self.log.info("Saving upgrade history for node(s).")

        task_catalog = cvmanager_catalog.Catalog()
        history_file = task_catalog.get_file(_UPGRADE_HISTORY_FILE)
        self.log.info("Current upgrade history file: {0}".format(history_file.file_path))

        upgrade_timestamp = str(int(time.time()))

        # History file is json, so read it
        if len(history_file.lines) == 0:
            # We should technically never get here; but in case, handle it.
            history_json = {}
        else:
            history_json = history_file.serialize_json()

        for node in remote_nodes:
            history_json[node.hostname] = {'last_upgrade_time': upgrade_timestamp}

        # Don't forget this current node, as remote_nodes are strictly the remote_nodes
        local_node = hs_node.HyperScaleNode()
        history_json[local_node.hostname] = {'last_upgrade_time': upgrade_timestamp}

        # Write the history file.
        history_file.write(history_json, cvmanager_catalog.FileType.JSON)

        return True
