# Python imports
import os

# Project imports
import cvmanager_task_step
import cvmanager_task
import cvmanager_defines
import cvupgrade_common
import wrappers
import cvmanager_task_arg
import common


class Task(cvmanager_task.TaskObject):
    """
    This task is designed to connect to a remote NFS shared repo, and compute the missing RPMs.
    It will create a LOCAL (to the node) mount point to connect to the NFS share.

    It should NOT operate on the main REPO, it must use that as the source, and operate in it's own node space.
    """
    task_args = {
        'nfs_mount': cvmanager_task_arg.TaskArg('nfs_mount', str, required=True),
        'hostname': cvmanager_task_arg.TaskArg('hostname', str, required=True)
    }

    def set_process(self, process):
        process.pre_process = [
            self.check_for_repos,
            self.copy_repo_from_share,
            self.disable_unused_20_repo
        ]
        process.main_process = [
            self.compute_missing_rpm
        ]

        process.post_process = [
            self.cleanup
        ]

    @cvmanager_task_step.OptionStep.run_always
    def disable_unused_20_repo(self, *args, **kwargs):
        if common.is_hs2dot0():
            ret = cvupgrade_common.disable_repos(cvupgrade_common.repolist_to_disable,
                                                 cvmanager_defines.TaskDir.share)
            if not ret == 0:
                return False
        return True

    @cvmanager_task_step.OptionStep.run_always
    def check_for_repos(self, *args, **kwargs):
        if not cvupgrade_common.check_for_repos(cvmanager_defines.REPO_CLEANUP):
            return False
        return True

    @cvmanager_task_step.TaskStep
    def compute_missing_rpm(self, *args, **kwargs):
        # compute_missing_rpms will act upon the master RPM list on the master node.
        wrappers.rm(os.path.join(cvmanager_defines.TaskDir.share, 'rpmlist.txt'))
        if not cvupgrade_common.compute_missing_rpms_v2(cvmanager_defines.TaskDir.share):
            return False
        self.log.info("Successfully created list of missing local node rpm(s).")

        cvupgrade_common.delete_repo_file(cvmanager_defines.TMP_REPONAME)

        return True

    @cvmanager_task_step.OptionStep.run_always
    def copy_repo_from_share(self, *args, **kwargs):
        # Share REPO file; this is where we get the repo from the nfs share and copy to local machine.
        # It must be created by a master node
        cvupgrade_common.copy_repo_file_v2(cvmanager_defines.TMP_REPO_FILE)
        self.log.info("Copied repo file [{0}] to /etc/yum.repos.d ...".format(cvmanager_defines.TMP_REPO_FILE))

        return True

    @cvmanager_task_step.TaskStep
    def cleanup(self, *args, **kwargs):
        cvupgrade_common.move_repos_back()
        cvupgrade_common.delete_repo_file(cvmanager_defines.TMP_REPONAME)

        return True
