# Python imports
import time
import os
import shutil

# Project imports
import cvmanager_task_step
import cvmanager_task
import cvmanager_defines
import cvupgrade_common
import common
import wrappers
from HsObject import hs_node
import cvmanager_task_arg
import cvmanager_error_codes


class Task(cvmanager_task.TaskObject):
    """Designed from the existing cvcreate_rpmxml.py file.

    """
    task_args = {
        'payload-loc': cvmanager_task_arg.TaskArg('payload-loc', str),
        'nodes': cvmanager_task_arg.TaskArg('nodes', list),
        'excluded_nodes': cvmanager_task_arg.TaskArg('excluded_nodes', list),
        'exit_only_true': cvmanager_task_arg.CommonArg.EXIT_ONLY_TRUE
    }

    def set_process(self, process):
        process.pre_process = [
            self.init,
            self.collect_nodes,
            self.check_for_repos,
            self.init_registry,
            self.build_repo,
            self.disable_unused_20_repo,
        ]

        process.main_process = [
            self.find_local_missing_rpm,
            self.find_missing_rpm_remote_nodes,
            self.update_master_rpm_list,
            self.generate_xml
        ]

        process.post_process = [
            self.move_xml_for_download,
            self.cleanup
        ]

    @cvmanager_task_step.TaskStep.with_options({'display_name': 'Detecting missing local RPMs.'})
    def find_local_missing_rpm(self, *args, **kwargs):
        # Detects missing RPMs on local node.
        wrappers.rm(os.path.join(cvmanager_defines.NFS_SHARE_TMP_REPO_DIR, 'rpmlist.txt'))
        if not cvupgrade_common.compute_missing_rpms_v2(cvmanager_defines.TaskDir.share):
            return False
        self.log.info("Successfully created list of missing local rpm(s).")

        cvupgrade_common.delete_repo_file(cvmanager_defines.TMP_REPONAME)

        return True

    @cvmanager_task_step.TaskStep.with_options({'display_name': 'Detecting missing remote RPMs.'})
    def find_missing_rpm_remote_nodes(self, *args, **kwargs):
        # Spawn child task for each node to run in parallel
        remote_processes = []
        for node in getattr(self, 'QUERY_NODES', []):
            remote_processes.append(self.create_remote_child_task('Query_RPM_Node', node, wait=False,
                                                                  launch=False, **self.kwargs))

        # This call will launch the task threads and wait for them.  We don't launch them during init because we don't
        # want to affect the NFS share, as we may cycle services many times.
        self.wait_for_remote_processes(remote_processes, launch=True)

        # This is a short-cut to return True/False.
        return self.check_and_return_remote_processes_status(remote_processes)

    @cvmanager_task_step.TaskStep
    def update_master_rpm_list(self, *args, **kwargs):
        """ Take all of the rpmlist.txt files from all remote nodes, and build 1 master list."""
        master_rpm_list = os.path.join(cvmanager_defines.NFS_SHARE, cvmanager_defines.UPGRADE_REPO_MASTER_RPM_LIST)

        for node in getattr(self, 'QUERY_NODES', []):
            # Each node will have created the rpmlist.txt in it's working directory.
            node_rpm_list = os.path.join(cvmanager_defines.NFS_SHARE, node.hostname,
                                         cvmanager_defines.UPGRADE_REPO_NODE_RPM_LIST_FILE)
            if not os.path.exists(node_rpm_list):
                # For some reason, this node did not create an rpmlist.txt file.
                self.log.error("File not found: [{}].  Cannot build master RPM list.".format(node_rpm_list))
                return False
            cvupgrade_common.append_rpmlist(node_rpm_list, master_rpm_list)

        # There are no nodes, use local rpm list only
        node_rpm_list = os.path.join(cvmanager_defines.NFS_SHARE, wrappers.get_hostname(),
                                     cvmanager_defines.UPGRADE_REPO_NODE_RPM_LIST_FILE)
        cvupgrade_common.append_rpmlist(node_rpm_list, master_rpm_list)

        return True

    @cvmanager_task_step.OptionStep.run_always
    def disable_unused_20_repo(self, *args, **kwargs):
        if common.is_hs2dot0():
            ret = cvupgrade_common.disable_repos(cvupgrade_common.repolist_to_disable,
                                                 cvmanager_defines.TaskDir.share)
            if not ret == 0:
                return False
        return True

    @cvmanager_task_step.OptionStep.run_always
    def build_repo(self, *args, **kwargs):
        # copy repodata downloaded by CS from remote cache to share
        cvupgrade_common.copy_repodata_to_share(cvmanager_defines.NFS_SHARE_TMP_REPO_DIR)
        self.log.info("Successfully copied repodata from swcache to nfs share.")

        # create repofile on nfs share
        cvupgrade_common.create_repo_file_v2(cvmanager_defines.NFS_SHARE_TMP_REPO_DIR)
        self.log.info("Successfully created repo file.")

        return True

    @cvmanager_task_step.TaskStep
    def generate_xml(self, *args, **kwargs):
        generated_xml = cvmanager_defines.UPGRADE_XML
        master_rpm_list = os.path.join(cvmanager_defines.NFS_SHARE, 'master_rpm_list.txt')

        if not cvupgrade_common.generate_xml_v2(master_rpm_list, generated_xml):
            return False

        self.log.info("Successfully created xml file with list of rpms required to download.")
        return True

    @cvmanager_task_step.TaskStep
    def move_xml_for_download(self, *args, **kwargs):
        generated_xml = cvmanager_defines.UPGRADE_XML
        download_xml = cvmanager_defines.UPGRADE_REPO_XML_FOR_DOWNLOAD

        try:
            shutil.copy(generated_xml, download_xml)
        except IOError as io_err:
            os.makedirs(os.path.dirname(download_xml))
            shutil.copy(generated_xml, download_xml)

        self.log.info("Successfully moved xml to download location.")
        return True

    @cvmanager_task_step.OptionStep.run_always
    def check_for_repos(self, *args, **kwargs):
        return cvupgrade_common.check_for_repos(cvmanager_defines.REPO_CLEANUP)

    @cvmanager_task_step.OptionStep.run_always
    def init(self, *args, **kwargs):
        # Set reg key that RPM query is not complete
        cvupgrade_common.set_rpm_query_key('No')

        # Create an empty XML file so that download software doesn't fail.  This is needed in case query fails,
        # we dont want CS download software job to fail.  It should just proceed.
        cvupgrade_common.generate_empty_xml(cvmanager_defines.NFS_SHARE + r'/')  # need this trailing slash for old fm

        # Check to ensure the payload exists
        payload_path = self.kwargs.get('payload-loc', cvmanager_defines.DEFAULT_SW_CACHE)
        if not os.path.exists(payload_path):
            self.log.error(cvmanager_error_codes.MISSING_DOWNLOAD_JOB)
            return False

        return True

    @cvmanager_task_step.OptionStep.run_always
    def cleanup(self, *args, **kwargs):
        # Move the repos back.
        cvupgrade_common.move_repos_back()

        # This step is only reached when all previous steps are completed, safe to assume RPM Query was successful.
        cvupgrade_common.set_rpm_query_key('Yes')

        return True

    @cvmanager_task_step.TaskStep.with_options({'always_run': True,
                                                'display_name': 'Detecting nodes to query.'})
    def collect_nodes(self, *args, **kwargs):
        """Here we will check cluster nodes, the YAML input, and the registry, and concatenate that list of nodes
        for querying to get the RPM packages.

        This step has run_always as we should always collect the nodes to run on.
        """
        nodes = hs_node.collect_remote_nodes(self)
        setattr(self, 'QUERY_NODES', nodes)

        return True

    @cvmanager_task_step.TaskStep.with_options({'always_run': True,
                                                'display_name': 'Initializing Registry.'})
    def init_registry(self, *args, **kwargs):
        common.deleteregistryentry(cvmanager_defines.REG_MA_KEY, "sHyperScalePayloadDir")

        # Set the default repository directory if NOT specified.
        payload_path = self.kwargs.get('payload-loc', cvmanager_defines.DEFAULT_SW_CACHE)
        self.log.info("Payload path: {0}".format(payload_path))

        self.log.info("Creating regkey sHyperScalePayloadDir with value [{0}]".format(payload_path))
        common.setregistryentry(cvmanager_defines.REG_MA_KEY, "sHyperScalePayloadDir", payload_path)
        unix_time = int(time.time())

        self.log.info("Creating regkey nSWCacheLastSyncTime with value [{0}]".format(unix_time))
        common.setregistryentry(cvmanager_defines.REG_MA_KEY, "nSWCacheLastSyncTime", str(unix_time))

        self.log.info("Adding regkey sHyperScaleRemoteCache with value Yes")
        common.setregistryentry(cvmanager_defines.REG_MA_KEY, "sHyperScaleRemoteCache", "Yes")

        return True
