# Python imports
import ConfigParser
import sys
import copy

# Project imports
from HsObject import hs_node
from HsObject import hs_utils
from HsObject import hs_ssh
from HsObject import hs_defines
import cvmanager_task_step
import cvmanager_task
import cvmanager_task_arg
import common
import cvmanager_catalog
import cvmanager_defines

SKIP_VAL = False
DYNAMIC_DEFINES = sys.modules[cvmanager_defines.DYNAMIC_DEFINE_NAME]
python_path = None  # This is for debugging purposes only; use with caution. python_path = None


class ConfigFile(object):
    AVAILABLE_NODES = 'Available_Nodes.json'
    NETWORK_CONFIG = '{0}_network_config.json'
    SILENT_INSTALL_CONFIG = '{0}_silentinstall.cfg'


class Task(cvmanager_task.TaskObject):
    """
    This task will configure the current node where it's executing on and join an existing HyperScale X cluster.
    It will configure the network, add itself to the cluster, and register with CommServe.
    """
    # Task input metadata
    task_args = {
        'remote_node': cvmanager_task_arg.TaskArg('remote_node', str, required=True, description="Node serial number")
    }

    def set_process(self, process):
        process.pre_process = [
            self.init,
            self.validate_user_input,
            self.create_remote_node_json,
            self.create_remote_node_config,
            self.copy_configs_to_node
        ]

        process.main_process = [
            self.configure_remote_node
        ]

        process.post_process = [
            self.wait_for_network_config,
            self.set_hostname_and_password
        ]

    @cvmanager_task_step.TaskStep
    def set_hostname_and_password(self, *args, **kwargs):
        """Sets the hostname and password on the remote node.
        """
        remote_node = self.kwargs.get('remote_node')
        remote_info = getattr(self, 'REMOTE_NODE_INFO', {}).get('discovered_file_data')

        # Get the hostname of the remote node from the data_protection IP.  This is what's being set.
        node_hostname = common.gethostname_fromip(remote_info.get('data_protection_ip'))
        if len(node_hostname) == 0:
            self.log.error("Reverse lookup of data protection IP [{0}] failed for node [{1}].".format(
                remote_info, remote_node
            ))
            return False

        # If there's a password in the input, set it using ssh; this will cause ssh to drop.
        existing_password = remote_info.get('existing_password', cvmanager_defines.DEFAULT_IMAGE_PASSWORD)
        existing_password = hs_utils.valid_password(existing_password)

        # Open an SSH connection to the remote node using default credentials; and change to user supplied credentials.
        ssh_conn = hs_ssh.RemoteSSH(remote_info.get('data_protection_ip'),
                                    username=cvmanager_defines.DEFAULT_IMAGE_USER, password=existing_password)
        if not ssh_conn.set_hostname_remote_node(node_hostname) == 0:
            self.log.error("Failed setting hostname over ssh on remote node [{0}].".format(node_hostname))
            return False
        self.log.info("Successfully set hostname of node [{0}] to [{1}].".format(remote_node, node_hostname))

        if remote_info.get('new_password', False):
            new_password = hs_utils.valid_password(remote_info.get('new_password'), return_password_on_failure=False)
            if not new_password:
                self.log.error("Unable to determine validity of user input password.  Please ensure the proper root"
                               " password was entered and try again.")
                return False

            if not ssh_conn.set_password(password=new_password) == 0:
                self.log.error("Failed setting root password over ssh on remote node [{0}].".format(node_hostname))
                return False
            self.log.info("Successfully set root password on node [{0}].".format(node_hostname))

        return True

    @cvmanager_task_step.TaskStep
    def wait_for_network_config(self, *args, **kwargs):
        """Waits for the remote node to come online; using avahi.
        """
        remote_node = self.kwargs.get('remote_node')
        return hs_utils.wait_for_network_config(remote_node)

    @cvmanager_task_step.OptionStep.run_always
    def init(self, *args, **kwargs):
        """Make sure the local node config is populated.  If this task was called from Add_Nodes_To_Cluster, it will be,
        otherwise, if this is being called externally for a single node, get the local config.
        """
        # Read the discovered node information from the avahi output file.
        task_catalog = cvmanager_catalog.Catalog('Discover_Nodes')
        discovered_nodes = task_catalog.get_file(ConfigFile.AVAILABLE_NODES).serialize_json()

        # Form the discovered information with the user input information; link them together
        node_serial_number = self.kwargs.get('remote_node')

        # If this is silent mode,
        remote_node_info = {'discovered_file_data': discovered_nodes[node_serial_number]}
        remote_node_info.update(self.kwargs.get('node_info', {}).get(node_serial_number, {}))

        # This will be used by EVERY other step.  It's the JSON information for this node; BLOCK_DEVICE, AVAHI, ARCH
        setattr(self, 'REMOTE_NODE_INFO', remote_node_info)

        local_node = hs_node.HyperScaleNode()

        local_node_network_json = self.kwargs.get('local_node_network_json', False)
        if not local_node_network_json:
            local_node_network_json = local_node.get_node_network_json()

        setattr(self, 'LOCAL_NODE', local_node)
        setattr(self, 'LOCAL_NODE_NETWORK_JSON', local_node_network_json)

        return True

    @cvmanager_task_step.TaskStep
    def configure_remote_node(self, *args, **kwargs):
        """
        Using SSH and default credentials, launch the network config.
        :param args:
        :param kwargs:
        :return:
        """
        remote_info = getattr(self, 'REMOTE_NODE_INFO').get('discovered_file_data')
        remote_node_ip = remote_info['AVAHI']['ip']

        ssh_conn = hs_ssh.RemoteSSH(remote_node_ip, username=cvmanager_defines.DEFAULT_IMAGE_USER,
                                    password=cvmanager_defines.DEFAULT_IMAGE_PASSWORD,
                                    timeout=hs_defines.SSH_TIMEOUT, banner_timeout=hs_defines.SSH_BANNER_TIMEOUT)

        if not ssh_conn.config_network(r'/tmp/nw_config.json', python_path=python_path) == 0:
            self.log.error("Failed to configure network on node [{0}].".format(remote_node_ip))
            return False

        # Successfully launched network config on the remote node.
        return True

    @cvmanager_task_step.TaskStep
    def validate_user_input(self, *args, **kwargs):
        """
        Validate user input(s) for adding nodes is correct.  At this point in time we only need to validate the
        network settings.  If we need additional input validations, add that code here.

        :param args:
        :param kwargs:
        :return:
        """
        node = self.kwargs.get('remote_node')
        node_info = getattr(self, 'REMOTE_NODE_INFO').get('discovered_file_data')

        backend_json = copy.deepcopy(getattr(self, 'LOCAL_NODE_NETWORK_JSON'))

        # In interactive mode, the user would have already entered the information and had it validated.
        # So we only need to validate in non-interactive mode.
        if not self.kwargs.get('interactive', False):
            # Non-interactive mode; user input MUST be specified.
            if not hs_utils.validate_network_for_node(node, node_info, backend_json=backend_json):
                self.log.error("Please correct all IP validation errors and re-submit.")
                if not SKIP_VAL:
                    return False
        else:
            # Interactive mode, pass.  Input was validated as it was entered.
            pass

        return True

    @cvmanager_task_step.TaskStep
    def create_remote_node_config(self, *args, **kwargs):
        remote_node = self.kwargs.get('remote_node')
        remote_info = getattr(self, 'REMOTE_NODE_INFO').get('discovered_file_data')

        task_catalog = cvmanager_catalog.Catalog()

        # Setup the config file for the remote node.
        config = ConfigParser.ConfigParser()

        # We only need 2 sections.
        config.add_section('data_protection_ips')
        config.add_section('network')

        # We are not setting management network for this.
        config.set('network', 'config_mgmt', False)

        # Add the data protection IP for this node.
        config.set('data_protection_ips', self.kwargs.get('remote_node'), remote_info['data_protection_ip'])

        # Add the dns servers
        for i, name_server in enumerate(remote_info['network_config'].get('dnsList', []), 1):
            config.set('network', 'nameserver{0}'.format(i), name_server)

        node_silent_install = task_catalog.get_file(ConfigFile.SILENT_INSTALL_CONFIG.format(remote_node), True)
        node_silent_install.write(config, cvmanager_catalog.FileType.CONFIG)

        return True

    @cvmanager_task_step.TaskStep
    def create_remote_node_json(self, *args, **kwargs):
        """
        Takes the existing node network config and performs the following actions for each node in user input.
        The local_network_json is in the front-end (webtool.py) format.
        1) Locates the commServeRegistration network interface and updates it with user input remote node IP address.
        2) Locates the storagePool network interface and updates it with user input remote node IP address.
        3) Removes the mac ("hwaddr") from the interface for all interfaces.
        :param args:
        :param kwargs:
        :return:
        """
        # local_network_json is the [cvnwconfigmgr.py get out.json] of the current local node.
        local_node = getattr(self, 'LOCAL_NODE')
        local_network_json = copy.deepcopy(getattr(self, 'LOCAL_NODE_NETWORK_JSON'))

        # Check the user input nodes in [node_info] and get the user supplied IP addresses.
        remote_node = self.kwargs.get('remote_node')
        remote_info = getattr(self, 'REMOTE_NODE_INFO').get('discovered_file_data')

        updated_interface_list = []
        # Iterate through all local machine interfaces, replace IP addresses as necessary for appropriate type.
        for interface in local_network_json['interfaces']:
            if interface.get('hyperscale_nwtype', False) == 'commServeRegistration':
                interface['ipaddr'] = remote_info.get('data_protection_ip')
            elif interface.get('hyperscale_nwtype', False) == 'storagePool':
                interface['ipaddr'] = remote_info.get('storage_pool_ip')
            elif interface.get('hyperscale_nwtype', False) == 'ipmi':
                # Don't carry over the ipmi config for now.
                continue

            # For all interfaces, remove the hwaddr.  It's not needed on remote node to configure.
            interface.pop('hwaddr', None)
            updated_interface_list.append(interface)

        # Now that we have an updated list of all interfaces, with new IP address and no IPMI, swap it in
        # the local node config.  The local node config is driving remote node config....keeping them exact.
        local_network_json['interfaces'] = updated_interface_list

        # Convert this to front-end json
        frontend_json = local_node.convert_backend_json_to_front_end(remote_node, local_network_json)

        # Update the user input for the remote node with the newly formed, modified JSON.
        remote_info['network_config'] = frontend_json

        # Create the network config json file for the remote node.  Get a clean copy.
        task_catalog = cvmanager_catalog.Catalog()
        remote_node_json = task_catalog.get_file(ConfigFile.NETWORK_CONFIG.format(remote_node), True)
        self.log.debug("Saving network config to: {0}".format(str(remote_node_json)))
        remote_node_json.write(frontend_json, cvmanager_catalog.FileType.JSON)

        return True

    @cvmanager_task_step.TaskStep
    def copy_configs_to_node(self, *args, **kwargs):
        """
        Copies the configuration files to the remote node using arch.  The JSON network config, and
        the silentinstall.cfg
        :param args:
        :param kwargs:
        :return:
        """
        task_catalog = cvmanager_catalog.Catalog()

        remote_node = self.kwargs.get('remote_node')
        remote_info = getattr(self, 'REMOTE_NODE_INFO').get('discovered_file_data')

        remote_node_ip = remote_info['AVAHI']['ip']

        # Each list item pair is local file, and remote file.
        files_to_copy = [
            [
                str(task_catalog.get_file(ConfigFile.NETWORK_CONFIG.format(remote_node))),
                r'/tmp/nw_config.json'
            ],
            [
                str(task_catalog.get_file(ConfigFile.SILENT_INSTALL_CONFIG.format(remote_node))),
                    r'/var/www/my_flask_prod/config/silentinstall.cfg'
            ]
        ]

        ssh_conn = hs_ssh.RemoteSSH(remote_node_ip, username=cvmanager_defines.DEFAULT_IMAGE_USER,
                                    password=cvmanager_defines.DEFAULT_IMAGE_PASSWORD,
                                    timeout=hs_defines.SSH_TIMEOUT, banner_timeout=hs_defines.SSH_BANNER_TIMEOUT)

        # Copy these file using arch.
        for remote_file in files_to_copy:
            if not ssh_conn.copy_file_to_remote(remote_file[0], remote_file[1]):
                self.log.error("Failed writing file [{0}] on remote node [{1}].".format(
                    str(remote_file), remote_node_ip
                ))
                break
        else:
            return True

        self.log.error("Failed copying config files to remote nodes for performing network config.")
        return False
