# Python imports
import os
import copy
import wrappers
import getpass
from Queue import Queue
from threading import Thread
import sys

# Project imports
import common
import cvmanager_task_step
import cvmanager_task
import cvmanager_catalog
from HsObject import hs_node
from HsObject import hs_utils
from HsObject import hs_hv_node
from HsObject import hs_ssh
from HsObject import hs_hv_block
import cvmanager_task_arg
import cvupgradeos
import cvmanager_defines
import cvupgrade_common
import cvmanager_xml_request

SKIP_VAL = False
SKIP_REPO_CHECK = False
DYNAMIC_DEFINES = sys.modules[cvmanager_defines.DYNAMIC_DEFINE_NAME]
python_path = None  # This is for debugging purposes only; use with caution. python_path = None


class ConfigFile(object):
    AVAILABLE_NODES = 'Available_Nodes.json'
    NETWORK_CONFIG = '{0}_network_config.json'
    SILENT_INSTALL_CONFIG = '{0}_silentinstall.cfg'
    CLUSTER_ANSIBLE = '{0}.ansi'


class Task(cvmanager_task.TaskObject):
    """This task can be executed on an existing node of the cluster, used for adding new remote nodes into
    this same cluster.  It can be used for adding 1 or many new nodes into any existing cluster for HyperScale X.

    You must run node discovery before running this task, using the 'Discover_Nodes' task.  After running node
    discovery, a file containing all available nodes will exist in the task catalog;
    /ws/ddb/cvmanager/catalog/task_storage/Discover_Nodes/Available_Nodes.json.

    This task should be used for running multi-node operations; configuration changes that we do on ALL nodes.
    -- configure network - all new nodes
    -- passwordless ssh - all nodes in the cluster
    -- etc/hosts - all nodes in cluster
    -- etc

    """
    # Task input metadata
    task_args = {
        'node_info': cvmanager_task_arg.TaskArg('node_info', dict, required=False,
                                                description="""
                                                Dictionary of node information, in the format of:
                                                { <serial_number> :
                                                    { 'data_protection_ip' : <IP> ,
                                                      'storage_pool_ip':<IP>
                                                    }
                                                }
                                                """),
        'storage_pool_name': cvmanager_task_arg.TaskArg('storage_pool_name', str, required=False,
                                                        description="Storage pool to add the node to."),
        # Interactive Mode will ask users for nodes, and IP addresses.
        'interactive': cvmanager_task_arg.TaskArg('interactive', bool, required=False, default_value=True,
                                                  valid_values=[True, False],
                                                  description="Interactive mode.  Does not require input file."),
        'use_avahi': cvmanager_task_arg.CommonArg.USE_AVAHI,
        'run_discovery': cvmanager_task_arg.TaskArg('run_discovery', bool, required=False, default_value=False,
                                                    description="Explicitly run discovery in this task.")
    }

    def set_process(self, process):
        process.pre_process = [
            self.init,   # run every execution, even resume
            self.check_and_collect_input
        ]

        process.main_process = [
            self.configure_remote_node_network,     # Only run until successful, do not re-run this.
            self.setup_passwordless_ssh,
            self.add_etc_hosts,
            self.backup_network_data,
            self.register_nodes_to_cs,
            self.setup_repo_on_remote_nodes,
            self.search_and_update_ansible,
            self.add_nodes_to_cluster,
            self.expand_storage_pool,
            self.resize_virtual_disk
        ]

        process.post_process = [
            self.mark_node_configured,
            # self.set_node_parameters,
            self.cleanup
        ]

    @cvmanager_task_step.TaskStep
    def setup_passwordless_ssh(self, *args, **kwargs):
        """  Setup the passwordless SSH connection between this node & all existing nodes in the cluster.

        This is 2 step here:
            1) On all NEW nodes, generate the ssh keys in standard location.  Existing nodes should have keys in that
            locaation as well.
            2) On every node, copy its public key to every other node.

        :param args:
        :param kwargs:
        :return:
        """
        local_node = getattr(self, 'LOCAL_NODE')
        cluster_nodes = copy.copy(getattr(self, 'CLUSTER_NODES', []))

        if local_node not in cluster_nodes:
            cluster_nodes.append(local_node)

        # This is the first step, of generating SSH key on remote node(s)
        node_password = ''
        for add_node, node_info in getattr(self, 'ADD_NODES', {}).items():
            # Here we loop through ALL NEW nodes being added to the cluster.  add_node = new node being added.
            data_protection_ip = node_info.get('data_protection_ip', None)
            storage_pool_ip = node_info.get('storage_pool_ip', None)
            node_hostname = common.gethostname_fromip(data_protection_ip)

            node_password = hs_utils.valid_password(node_info.get('new_password', False))

            cluster_nodes.insert(len(cluster_nodes), hs_hv_node.HVNode(
                node_hostname, **{'node_ips': [data_protection_ip, storage_pool_ip]}))

        cluster = hs_hv_block.HVBlock(cluster_nodes)
        if not cluster.configure_passwordless_ssh(user_name=cvmanager_defines.DEFAULT_IMAGE_USER,
                                                  password=node_password):
            self.log.error("Failed configuring passwordless ssh for [{0}].".format(cluster))
            return False
        else:
            self.log.info("Successfully configured root passwordless ssh for [{0}].".format(cluster))

        if not cluster.configure_passwordless_ssh(user_name=cvmanager_defines.DEFAULT_IMAGE_ADMIN_USER):
            self.log.error("Failed configuring passwordless ssh for [{0}].".format(cluster))
            return False
        else:
            self.log.info("Successfully configured admin passwordless ssh for [{0}].".format(cluster))

        return True

    @cvmanager_task_step.TaskStep
    def add_etc_hosts(self, *args, **kwargs):
        """

        :param args:
        :param kwargs:
        :return:
        """
        local_node = getattr(self, 'LOCAL_NODE')
        cluster_nodes = copy.copy(getattr(self, 'CLUSTER_NODES'))

        if local_node not in cluster_nodes:
            cluster_nodes.append(local_node)

        add_nodes = getattr(self, 'ADD_NODES')

        # Iterate through all nodes being added, and set the private interface resolvable.  This must be done FIRST
        # on all new nodes being added, otherwise existing nodes will not resolve them.
        for node, node_info in add_nodes.items():
            data_protection_ip = node_info.get('data_protection_ip', None)
            storage_pool_ip = node_info.get('storage_pool_ip', None)
            node_hostname = common.gethostname_fromip(data_protection_ip)

            node_object = hs_node.HyperScaleNode(node_hostname)
            if not node_object.set_private_interface_resolution(storage_pool_ip):
                self.log.error("Failed setting storage pool resolution on node [{0}].".format(node_object))
                return False

            # Add this node to the cluster_nodes.
            cluster_nodes.append(node_object)

        # Go through all cluster nodes and set the same resolution for all the NEW nodes.
        for cluster_node in cluster_nodes:
            # For each node in the cluster, make sure ALL cluster nodes are resolvable in the /etc/hosts of that node.
            if not cluster_node.set_private_interface_resolution([n.hostname for n in cluster_nodes], resolve=True):
                self.log.error("Failed setting storage pool resolution on node [{0}].".format(cluster_node))
                return False

        self.log.info("Successfully set storage pool /etc/hosts entries across the cluster.")
        return True

    @cvmanager_task_step.TaskStep
    def backup_network_data(self, *args, **kwargs):
        """
        cvavahi.py backup_network

        :param args:
        :param kwargs:
        :return:
        """
        local_node = getattr(self, 'LOCAL_NODE')
        cluster_nodes = copy.copy(getattr(self, 'CLUSTER_NODES'))

        if local_node not in cluster_nodes:
            cluster_nodes.append(local_node)

        # Create node objects for entire cluster.
        for add_node, node_info in getattr(self, 'ADD_NODES', {}).items():
            data_protection_ip = node_info.get('data_protection_ip', None)
            node_hostname = common.gethostname_fromip(data_protection_ip)
            cluster_nodes.append(hs_node.HyperScaleNode(node_hostname))

        for node in cluster_nodes:
            if not node.backup_network(python_path=python_path):
                self.log.error("Failed backing up network on node: [{0}]".format(node))
                break
        else:
            self.log.info("Successfully backed up network for all nodes.")
            return True

        return False

    @cvmanager_task_step.OptionStep.run_always
    def init(self, *args, **kwargs):
        """
        Checks if the avahi node discovery has been recently run, if not, run the discovery to ensure that the user
        supplied nodes are discoverable.  When this task is executed, the discovery should have happened within the
        last 60 minutes (TODO - change time?).
        This is because the workflow will launch discovery task separately, ask users to select nodes, collect the
        ip addresses, then launch this task.   We only check here in cases this task is being launched interactively,
        and not from the workflow.  Basically ensure we have a "fresh" list of avahi nodes discovered.

        Check to ensure this is a software cache node, if not, do not run.

        nAddNodesReady is set to 1 on the software cache node when the cache is populated by download software with
        the proper cv-hedvig rpms.

        :param args:
        :param kwargs:
        :return:
        """
        # This object is used in a few places; the local node where python is executed.
        local_node = hs_node.get_local_node()

        # Create the REPO for the nodes.
        # if not self.create_hedvig_repo():
        #    return False

        # Check to ensure software cache has required cv-hedvig rpms.
        # ############################## SOFTWARE CACHE CHECK#######################
        # In here, ensure that the software cache is populated with cv-hedvig RPMs.
        temp_cv_repo = cvmanager_defines.NFS_SHARE_TMP_REPO_DIR
        if not os.path.exists(temp_cv_repo):
            # fall back to pre-cvmanager mode.
            temp_cv_repo = r'/ws/ddb/upgos/tmpcvrepo'
        cv_hv_repo = os.path.join(temp_cv_repo, 'cv-hedvig')
        if not os.path.exists(cv_hv_repo):
            self.log.error("Repository [{0}] is not present, please be sure to run download software after setting "
                           "the proper registry key on remote software cache.  Refer to Add Nodes documentation."
                           "".format(cv_hv_repo))
            return False

        # REPO is present.  Ensure that the repo version matches the installed version.
        if not self.repo_version_check(local_node):
            if not SKIP_REPO_CHECK:
                return False
        # ############################## SOFTWARE CACHE CHECK COMPLETE##############

        # ############################## DISCOVERY CHECK ##############################
        # Check and ensure node discovery was completed and we have the Available_Nodes.json file in the catalog.
        # Give the name of the discovery task, as it will be the one writing the file into it's catalog.
        task_catalog = cvmanager_catalog.Catalog('Discover_Nodes')
        discovered_nodes_file = os.path.join(task_catalog.catalog_dir, ConfigFile.AVAILABLE_NODES)

        # By default, we do not run discovery in this task.
        run_discovery = False
        if not os.path.exists(discovered_nodes_file):
            # Did not find discovered information file, run the discovery because we need something in all modes
            # regardless of what user wants or doesn't want.
            self.log.debug("Did not find discovered node information; running discovery.")
            run_discovery = True
        elif self.kwargs.get('run_discovery', False):
            # Discovery is explicitly asked for on command line.
            self.log.debug("Command line [run_discovery=True] specified, running discovery.")
            run_discovery = True
        elif self.kwargs.get('interactive', False) and not self.resumed:
            # Run discovery only when interactive and not resumed
            run_discovery = True

        if run_discovery:
            # The discovery was not run, so run it.  Never run discovery when resuming unfinished task, as we could lose
            # information like custom user input.
            discovery_task = self.create_child_task('Discover_Nodes', **self.kwargs)
            discovery_task.wait_for_child_task()
            if not discovery_task.check_and_return_child_task_status():
                # The node discovery failed.
                return False
            self.log.info("Successfully discovered remote nodes.")
        # ###############################DISCOVERY CHECK COMPLETE###################################

        # Ensure that all nodes from the user input, are available in the discovered nodes.
        # If this is interactive mode, nodes were interactively discovered, so the list will be finite to just the ones
        # the user typed in as input.
        missing_nodes = []

        # Read the available nodes file that discovery generated.
        discovered_nodes = task_catalog.get_file(ConfigFile.AVAILABLE_NODES)
        discovered_nodes_file_json = discovered_nodes.serialize_json()

        # Make sure that any nodes given as input, have been discovered, and had their hardware validated.
        input_nodes = self.kwargs.get('node_info', {})  # The user input information, like IP addresses.
        for node in input_nodes:
            if node not in discovered_nodes_file_json:
                missing_nodes.append(node)
                continue

        if len(missing_nodes) > 0:
            self.log.error("Cannot add unknown nodes to the cluster.  Remove the following missing nodes from your "
                           "input or correct avahi discovery issue for following nodes.\nNodes: {0}".
                           format(','.join(missing_nodes)))
            return False

        # The ADD_NODES attribute on self will be the FINAL list of nodes to configure.
        if self.kwargs.get('interactive', False):
            # Interactive mode, add all the discovered nodes because discovery gives a finite list when its run
            # interactively.
            setattr(self, 'ADD_NODES', discovered_nodes_file_json)
        else:
            # For non-interactive mode, user should have supplied node_info input for all nodes to configure.
            # If no nodes; fail.
            if len(input_nodes) == 0:
                self.log.error("You did not supply the required input for which nodes to configure.  Please see help."
                               "\nYou must give the node_info JSON data."
                               "\nnode_info:{<serial_num>:{data_protection_ip:'<IP>',storage_pool_ip:'<IP>'}}")
                return False

            # Because the nodes are explicitly provided in non-interactive mode, and them all as 'ADD_NODES' which is
            # the final list of nodes being configured.
            for node, node_info in input_nodes.items():
                # Take all the user input and add it into the JSON for storage and restartability.
                discovered_nodes_file_json[node].update(node_info)

                ds = getattr(self, 'ADD_NODES', {})
                ds[node] = discovered_nodes_file_json[node]
                setattr(self, 'ADD_NODES', ds)

            # Re-write the file with user inputs.
            discovered_nodes.write(discovered_nodes_file_json, cvmanager_catalog.FileType.JSON, True)

        # All input nodes have been discovered.  Create the local node object.
        local_node_network_json = local_node.get_node_network_json()
        cluster_nodes = hs_node.collect_remote_nodes(self, active_nodes_only=True)

        setattr(self, "LOCAL_NODE", local_node)
        setattr(self, "LOCAL_NODE_NETWORK_JSON", local_node_network_json)
        setattr(self, "CLUSTER_NODES", cluster_nodes)

        return True

    @cvmanager_task_step.TaskStep
    def check_and_collect_input(self, *args, **kwargs):
        """
        At this point, we have all the nodes to add discovered.  We already know they have matching hardware, the
        discovery does that checking.  The discovery also asks for user to select the nodes, so we're operating on
        the final list of nodes in this function.

        Check and make sure we have valid data protection and storage pool IPs for these node(s); as well as passwords
        if its not an interactive mode.  Normally, we can set [required=True] on the task and the framework will
        validate the input...however, because this task is supporting 2 modes (silent and interactive) we will do
        the validation here instead.

        This writes all of the collected information into a user input file; for restartability.

        :return:
        """
        task_catalog = cvmanager_catalog.Catalog('Discover_Nodes')
        discovered_nodes_file = task_catalog.get_file(ConfigFile.AVAILABLE_NODES)
        discovered_nodes_file_json = discovered_nodes_file.serialize_json()

        # Get the list of node we're working with
        add_nodes = getattr(self, "ADD_NODES", {})

        # If this is interactive mode; ask for the DP & SP IPs for all selected nodes.
        if self.kwargs.get('interactive', False):
            for node, node_info in add_nodes.items():
                remote_data_protection_ip = ''
                remote_storage_pool_ip = ''
                while True:
                    # This loop will run until user gives valid network credentials.
                    remote_data_protection_ip = raw_input("Data protection IP [{0}] for node [{1}]: ".
                                                          format(remote_data_protection_ip,
                                                                 node)) or remote_data_protection_ip
                    remote_storage_pool_ip = raw_input("Storage pool IP [{0}] for node [{1}]: ".format(
                        remote_storage_pool_ip, node)) or remote_storage_pool_ip

                    # Save these IPs to the node info JSON
                    node_info['data_protection_ip'] = remote_data_protection_ip
                    node_info['storage_pool_ip'] = remote_storage_pool_ip

                    # if self.validate_network_for_node(node, node_info):
                    if hs_utils.validate_network_for_node(node, node_info):
                        break

                    if not SKIP_VAL:
                        # Re-try.
                        raw_input("Press Enter to continue...")
                        wrappers.clear_screen()
                    else:
                        break

                # Now get the root credentials from user. TODO: validate current password.
                # If there's a password in the input, set it using ssh; this will cause ssh to drop.
                if not node_info.get('existing_password', False):
                    # This is interactive mode with NO password given in user input; ask for it.
                    root_password = raw_input("Current root password for node {0}: ".format(node))

                    # Save the password to input for this node.
                    node_info['existing_password'] = cvupgrade_common.encp(root_password)

                if not node_info.get('new_password', False):
                    # This is interactive mode with NO password given in user input; ask for it.
                    pprompt = lambda: (getpass.getpass("New root password for node {0}: ".format(node)),
                                       getpass.getpass("Confirm root password for node {0}: ".format(node)))

                    p1, p2 = pprompt()
                    while p1 != p2:
                        print('Passwords do not match. Try again')
                        p1, p2 = pprompt()

                    # Save the password to input for this node.
                    node_info['new_password'] = cvupgrade_common.encp(p1)  # It doesn't matter, both are same.

                # We have all the new information collected for this node; update the discovered information.
                discovered_nodes_file_json[node] = node_info

        else:
            # In non-interactive mode, pass, and let the network config task do the network validation.
            # In this mode ALL inputs must be supplied and valid.
            all_valid = True
            for node, node_info in self.kwargs.get('node_info', {}).items():
                if not node_info.get('data_protection_ip', False):
                    self.log.error("Required input [{0}] missing for node [{1}].".format('data_protection_ip', node))
                    all_valid = False

                if not node_info.get('storage_pool_ip', False):
                    self.log.error("Required input [{0}] missing for node [{1}].".format('storage_pool_ip', node))
                    all_valid = False

            if not all_valid:
                return False

            # All information valid, write to the file to save what input was given.
            # This takes the user input for this specific node and links with the discovered information.
            discovered_nodes_file_json[node].update(node_info)

        # Re-write the file with user inputs.
        discovered_nodes_file.write(discovered_nodes_file_json, cvmanager_catalog.FileType.JSON, True)
        self.log.info("All required inputs are present.")
        return True

    @cvmanager_task_step.TaskStep
    def resize_virtual_disk(self, *args, **kwargs):
        local_node = getattr(self, 'LOCAL_NODE')

        if not local_node.resize_v_disk():
            return False

        return True

    @cvmanager_task_step.TaskStep
    def set_node_parameters(self, *args, **kwargs):
        """
        Add any extended attributes we need at this point.  These should be added on all data drives.
        trusted.hedvigclusternodes

        remote_block_device = node_info['BLOCK_DEVICE']

        :param args:
        :param kwargs:
        :return:
        """
        local_node = getattr(self, 'LOCAL_NODE')
        cluster_nodes = copy.copy(getattr(self, 'CLUSTER_NODES', []))

        if local_node not in cluster_nodes:
            cluster_nodes.append(local_node)

        for node_obj in cluster_nodes:
            if not node_obj.set_xattr_cluster_nodes(cluster_nodes=cluster_nodes, python_path=python_path):
                self.log.error("Failed setting extended attribute [trusted.hedvigclusternodes].")
                if not node_obj.local_node:
                    # Remote node failure is a soft failure.
                    self.log.warning("Non-fatal error for remote node xattr.")
                    continue
                # Only fail if the local node failed.
                return False

        return True

    @cvmanager_task_step.TaskStep
    def mark_node_configured(self, *args, **kwargs):
        for add_node, node_info in getattr(self, 'ADD_NODES', {}).items():
            data_protection_ip = node_info.get('data_protection_ip', None)
            node_hostname = common.gethostname_fromip(data_protection_ip)
            node_obj = hs_hv_node.HVNode(node_hostname)

            if not node_obj.mark_node_configured(python_path=python_path):
                self.log.error("Failed setting avahi status=HyperScaleConfigured.")
                return False

        return True

    @cvmanager_task_step.OptionStep.run_always
    def cleanup(self, *args, **kwargs):
        self.log.info("All nodes added!")
        return True

    @cvmanager_task_step.TaskStep
    def search_and_update_ansible(self, *args, **kwargs):
        """
        Searches through the cluster for the ansible file; once found, ensures the proper sections are present.
        :param args:
        :param kwargs:
        :return:
        """
        local_node = getattr(self, 'LOCAL_NODE')
        cluster_nodes = copy.copy(getattr(self, 'CLUSTER_NODES'))
        cluster_nodes.append(local_node)
        task_catalog = cvmanager_catalog.Catalog()

        cluster_name = None
        remote_cluster_file = None
        for node in cluster_nodes:
            # Open SSH connection to the node.  Passwordless is setup by this point, its required.
            ssh_conn = hs_ssh.RemoteSSH(node.hostname)

            if node.cluster_name is not None:
                cluster_name = node.cluster_name
            try:
                remote_cluster_file = r'/home/admin/{0}.ansi'.format(cluster_name)
                local_cluster_ansible = task_catalog.get_file(ConfigFile.CLUSTER_ANSIBLE.format(cluster_name))
                ssh_conn.get_remote_file(remote_cluster_file, str(local_cluster_ansible))

                # Read this file; its a config file.
                self.log.info("Found [{0}] on node [{1}].".format(remote_cluster_file, node.hostname))
                ansi_config = local_cluster_ansible.read_config_file(allow_no_value=True)

                # Check if this has the [commvault] section; if not add it.  Duplicate the cvms section
                if not ansi_config.has_section("commvault"):
                    self.log.info("Addming [commvault] section to ansible file.")
                    ansi_config.add_section("commvault")
                    for item in ansi_config.items("cvms"):
                        ansi_config.set("commvault", item[0])

                    # save this new file.
                    local_cluster_ansible.write_config_file(ansi_config)

                    # Write the file back to the deploy node.
                    ssh_conn.copy_file_to_remote(str(local_cluster_ansible), remote_cluster_file)
                    self.log.info("Successfully updated cluster ansible file.")
                else:
                    # [commvault] section already exists. do nothing?
                    pass

                return True

            except IOError, ioe:
                if ioe.errno == 2:
                    self.log.debug("Cluster file [{0}] not available on node [{1}].".format(remote_cluster_file,
                                                                                            node.hostname))
                else:
                    raise
            except Exception, err:
                raise

        # Didn't find and update the ansible
        self.log.error("Unable to locate ansible file [{0}] on any cluster node.".format(remote_cluster_file))
        return False

    @cvmanager_task_step.TaskStep
    def setup_repo_on_remote_nodes(self, *args, **kwargs):
        """
        Before adding the node to hedvig cluster, upgrade the remote OS.
        :param args:
        :param kwargs:
        :return:
        """
        # Check if SW cache is getting updated
        if cvupgradeos.checkif_swcache_busy():
            self.log.error("Upgrade cannot proceed at this time because SWcache is being updated with latest packages.")
            return False

        # Create the REPO for the nodes.
        if not self.create_hedvig_repo():
            return False

        # Go to each node and setup the repo back to this node; mount the NFS share and copy
        for add_node, node_info in getattr(self, 'ADD_NODES', {}).items():
            data_protection_ip = node_info.get('data_protection_ip', None)
            node_hostname = common.gethostname_fromip(data_protection_ip)

            # Copy the cvrepo to the remote node.
            ssh_conn = hs_ssh.RemoteSSH(node_hostname)
            if not ssh_conn.copy_file_to_remote(r'/etc/yum.repos.d/cvrepo.repo', r'/etc/yum.repos.d/cvrepo.repo'):
                self.log.error("Failed copying repo to [{0}].".format(node_hostname))
                return False

            self.log.info("Copying required RPMs to node [{0}].".format(node_hostname))
            if os.path.exists(cvmanager_defines.NFS_SHARE_REPO_DIR):
                cv_repo_dir = cvmanager_defines.NFS_SHARE_REPO_DIR
            else:
                # Pre-cvmanager style.
                cv_repo_dir = r'/ws/ddb/upgos/cvrepo'
            if not ssh_conn.copy_dir_to_remote(cv_repo_dir, cv_repo_dir, mode=0511, ignore_existing=True):
                self.log.error("Failed copying repo to [{0}].".format(node_hostname))
                return False

        self.log.info("Successfully setup repo and mounted for all nodes.")
        return True

    @cvmanager_task_step.TaskStep
    def register_nodes_to_cs(self, *args, **kwargs):
        """

        :param args:
        :param kwargs:
        :return:
        """
        if self.kwargs.get('interactive', False):
            # If this is interactive mode, and no cs credentials, ask for them.
            # This is interactive, check and ask for missing credentials.
            if not self.kwargs.get('cs_username', False):
                cs_username = raw_input("Please enter user for registering nodes to CommServe: ")
            else:
                cs_username = self.kwargs.get('cs_username')
            self.kwargs['cs_username'] = cs_username

            if not self.kwargs.get('cs_password', False):
                cs_password = getpass.getpass("Please enter password for registering nodes to CommServe: ")
            else:
                cs_password = hs_utils.valid_password(self.kwargs.get('cs_password'))

            # Save this password in the object.
            self.kwargs['cs_password'] = cvupgrade_common.encp(cs_password)

        elif self.kwargs.get('auth_code', False):
            # This is auth code based registration.
            self.log.info("Performing auth code based node registration for node(s).")
        elif not (self.kwargs.get('cs_username', False) and self.kwargs.get('cs_password', False)) or not(
                self.kwargs.get('auth_code', False)):
            # No username or Password, and not a workflow
            self.log.error("Please provide (cs_username and cs_password) or (auth_code) input parameters for "
                           "registering nodes to the CommServe.")
            return False

        # Create node objects for entire cluster.
        for add_node, node_info in getattr(self, 'ADD_NODES', {}).items():
            data_protection_ip = node_info.get('data_protection_ip', None)
            node_hostname = common.gethostname_fromip(data_protection_ip)
            remote_node = hs_node.HyperScaleNode(node_hostname)

            if not remote_node.register_to_cs(self.kwargs.get('cs_username', False),
                                              hs_utils.valid_password(self.kwargs.get('cs_password', False)),
                                              self.kwargs.get('auth_code', False), python_path=python_path):
                self.log.error("Failed registering node [{0}] to CommServe.".format(remote_node))
                break
        else:
            self.log.info("Successfully registered all nodes to CommServe.")
            return True

        return False

    @cvmanager_task_step.OptionStep.run_always
    def join_mount_path(self, *args, **kwargs):
        return True

    @cvmanager_task_step.TaskStep
    def expand_storage_pool(self, *args, **kwargs):
        """
        Perform the REST API call for adding expanding the storage pool.

        If the input yaml has credentials, use those for expanding.
        If we don't have in input yaml, and its interactive, ask for them.
        If we don't have in input yaml, and its non-interactive & workflow token is present, use that.
        If we don't have in input yaml, non-interactive, no workflow token; fail.

        :param args:
        :param kwargs:
        :return:
        """
        local_node = getattr(self, 'LOCAL_NODE')

        # Figure out credentials; TODO - refactor down to common method.
        cs_username = None
        cs_password = None
        cs_token = None
        if self.kwargs.get('cs_username', False) and self.kwargs.get('cs_password', False):
            cs_username = self.kwargs.get('cs_username')
            cs_password = hs_utils.valid_password(self.kwargs.get('cs_password'))
        elif self.kwargs.get('interactive', False):
            # If this is interactive mode, and no cs credentials, ask for them.
            # This is interactive, check and ask for missing credentials.
            if not self.kwargs.get('cs_username', False):
                cs_username = raw_input("CommServe login user name for expanding storage pool: ")
            else:
                cs_username = self.kwargs.get('cs_username')
            self.kwargs['cs_username'] = cs_username

            if not self.kwargs.get('cs_password', False):
                cs_password = getpass.getpass("CommServe login password for expanding storage pool: ")
            else:
                cs_password = cvupgrade_common.encp(self.kwargs.get('cs_password'))
            self.kwargs['cs_password'] = cs_password

        elif self.kwargs.get('no_hash', {}).get('cs_token', False):
            # Workflow_data is present, use this key.
            cs_token = self.kwargs['no_hash']['cs_token']  # leave this so we get an exception for malformed token dict.
            self.log.info("Performing token based storage pool expansion.")
        elif not (self.kwargs.get('cs_username', False) and self.kwargs.get('cs_password', False)) or not (
                self.kwargs.get('workflow_data', False)):
            # No username or Password, and not a workflow
            self.log.error("Please provide (cs_username and cs_password) input parameters for expanding the storage "
                           "pool.")
            return False

        # Get the CS hostname
        cs_host_name = hs_utils.get_cs_host_name()
        if not cs_host_name:
            return False

        # TODO: Figure this out automatically.
        storage_pool_name = self.kwargs.get('storage_pool_name', False)
        if not storage_pool_name and self.kwargs.get('interactive', False):
            # No storage pool name, but interactive mode.
            storage_pool_name = common.getregistryentry(cvmanager_defines.REG_MA_KEY,
                                                        cvmanager_defines.REG_STORAGE_POOL_NAME)
            storage_pool_name = raw_input("HyperScale storage pool to add this node in ({0}): ".format(
                storage_pool_name)) or storage_pool_name

        # Get the existing HV cluster password.
        existing_cid = common.getregistryentry(cvmanager_defines.REG_MA_KEY, cvmanager_defines.REG_HEDVIG_CID)
        hedvig_cluster_pw = cvupgrade_common.decp(existing_cid)

        auth_mode = {}
        if cs_token is not None:
            auth_mode = {
                'tk': cs_token
            }
        else:
            # Need to perform qlogin if non-token based attempt.
            if not local_node.login_to_cs(cs_username, cs_password, cs_host_name):
                return False

        v_disk_name = common.getregistryentry(cvmanager_defines.REG_MA_KEY, cvmanager_defines.REG_V_DISK)

        # We have the credentials; run the command.
        nodes_to_validate = []
        for add_node, node_info in getattr(self, 'ADD_NODES', {}).items():
            data_protection_ip = node_info.get('data_protection_ip', None)
            node_hostname = common.gethostname_fromip(data_protection_ip)
            node_obj = hs_hv_node.HVNode(node_hostname)

            if not node_obj.prepare_for_addition_to_cluster(v_disk_name, hedvig_cluster_pw, python_path):
                self.log.error("Failed setting up new node [{0}] for cluster addition.".format(node_obj))
                return False

            xml_params = {
                'storage/mediaAgent/mediaAgentName': node_hostname,
                'StoragePolicy/storagePolicyName': storage_pool_name,
                'scaleoutConfiguration/configurationType': 1,
            }

            xml_params.update(auth_mode)
            if not cvmanager_xml_request.XMLRequest('add_single_node_to_storage_pool', **xml_params).execute():
                self.log.error("Failed expanding the storage pool")
                return False

            nodes_to_validate.append(node_obj)

        if not self.wait_for_node_storage_pool_expansion(nodes_to_validate, storage_pool_name, v_disk_name):
            self.log.error("Not all nodes successfully mounted storage pool [{0}] in the allowed time.".format(
                storage_pool_name
            ))
            return False

        self.log.info("All nodes successfully added to the storage pool [{0}].".format(storage_pool_name))
        return True

    @cvmanager_task_step.TaskStep
    def add_nodes_to_cluster(self, *args, **kwargs):
        """
        Call the Hedvig ansible to add all these nodes into the existing cluster.

        If it hangs use this: ps wwwaux | grep ansible

        """
        cluster_nodes = copy.copy(getattr(self, 'CLUSTER_NODES'))

        # Create node objects for entire cluster.
        nodes_to_add = []
        for add_node, node_info in getattr(self, 'ADD_NODES', {}).items():
            data_protection_ip = node_info.get('data_protection_ip', None)
            node_hostname = common.gethostname_fromip(data_protection_ip)
            nodes_to_add.append(node_hostname)

        if all(node in [cnode.hostname for cnode in cluster_nodes] for node in nodes_to_add):
            self.log.info("All nodes selected for addition are already part of cluster.  Skipping addition, moving to "
                          "registration or expansion.")
            return True

        self.log.info("Adding node(s) [{0}] to cluster. This can take upwards of 60 minutes.  Do not attempt to "
                      "manually interrupt this job.".format(",".join(nodes_to_add)))
        if not hs_hv_node.add_nodes_to_cluster(nodes_to_add):
            self.log.error("Failed adding nodes to cluster.")
            return False

        self.log.info("Successfully added node(s) to cluster.")
        return True

    @cvmanager_task_step.TaskStep
    def configure_remote_node_network(self, *args, **kwargs):
        """
        Launch child tasks for all nodes in parallel.
        Be extremely careful here to NEVER add changing information into self.kwargs.  If you add information that
        can change between runs (resume cases) then we could start this configure network all over instead of resuming
        failed attempts.

        The Configure_Remote_Network task will re-read the config file and use the discovered information as necessary,
        during its init() call.

        node_info is the input dictionary from yaml.

        """
        child_tasks = []
        for node, node_info in getattr(self, 'ADD_NODES', {}).items():
            self.kwargs['local_node_network_json'] = getattr(self, 'LOCAL_NODE_NETWORK_JSON')
            self.kwargs['remote_node'] = str(node)
            configure_node_task = self.create_child_task('Configure_Remote_Network', launch=False, **self.kwargs)
            child_tasks.append(configure_node_task)

        # Launch and wait for the tasks to complete.
        self.wait_for_child_tasks(child_tasks, launch=True)
        return self.check_and_return_all_child_task_statuses(child_tasks)

    def create_hedvig_repo(self):
        # LOCAL node is the node we're running from. TODO: is this is the deploy node also?
        if os.path.exists(cvmanager_defines.NFS_SHARE_TMP_REPO_DIR):
            cv_repo_dir = cvmanager_defines.NFS_SHARE_REPO_DIR
            temp_cv_repo = cvmanager_defines.NFS_SHARE_TMP_REPO_DIR
            xml_file = cvmanager_defines.UPGRADE_XML
        else:
            # Pre-cvmanager style.
            cv_repo_dir = r'/ws/ddb/upgos/cvrepo'
            temp_cv_repo = r'/ws/ddb/upgos/tmpcvrepo'
            xml_file = cvmanager_defines.UPGRADE_REPO_XML_FOR_DOWNLOAD

        self.log.info("Creating rpm repository [{0}].".format(cv_repo_dir))
        if not cvupgrade_common.create_repo_v2(cv_repo_dir,
                                               temp_cv_repo,
                                               xml_file,
                                               repo_name_filter=[r'/cv-hedvig/']):
            return False
        else:
            if not os.path.exists(cv_repo_dir):
                self.log.error("Repository [{0}] is not present, it appears there are no packges available for "
                               "upgrade. Nothing to be done.".format(cv_repo_dir))
                return False

        self.log.info("Created cvrepo successfully.")
        return True

    def repo_version_check(self, local_node):
        cluster_version = local_node.cluster_version
        software_cache = cvupgrade_common.get_remotecache_dir()

        version = hs_utils.get_rpm_version(os.path.join(software_cache, 'CVAppliance', 'Unix', '*'),
                                           r'hedvig-cluster*.rpm')

        if not cluster_version == version:
            self.log.error("Cache [{0}] rpm versions [{1}] do not match the current cluster version [{2}].\n"
                           "Please be sure to run download software after setting the proper registry key on remote\n "
                           "software cache, and running [cvupgradeos.py -upgrade_hedvig_only].\n"
                           "Refer to Add Nodes documentation for additional information."
                           "".format(software_cache, version, local_node.cluster_version))
            return False

        return True

    def wait_for_node_storage_pool_expansion(self, nodes, storage_pool_name, v_disk_name):
        node_queue = Queue()
        new_node_list = []

        def worker(valid_node_dict):
            while True:
                try:
                    node_obj = node_queue.get()
                    if node_obj.wait_for_v_disk_online(storage_pool_name, v_disk_name):
                        new_node_list.append(node_obj)
                except Exception, err:
                    pass
                finally:
                    node_queue.task_done()

        for node in nodes:
            node_queue.put(node)

        for i in range(cvmanager_defines.MAX_REMOTE_NODE_SSH_CONNECTIONS):
            t = Thread(target=worker, args=(new_node_list,))
            t.daemon = True
            t.start()

        # block until all tasks are done
        node_queue.join()

        return new_node_list
