# Python imports
import yaml
from exceptions import OSError

# Project imports
import task_defines
import cvmanager_defines
import cvmanager_logging
import cvmanager_catalog

log = cvmanager_logging.get_log()


class YAMLTask(yaml.YAMLObject):
    """This is the task object from he input yaml based on the tag indicating its a Task to run.
    We pre-process the yaml like this before creating the base class TaskObject.
    """
    yaml_loader = yaml.Loader
    yaml_tag = '!Task'

    def __str__(self):
        return self.task_type.name

    def __init__(self, *args, **kwargs):
        try:
            self.task_type = task_defines.TaskType[kwargs.get('type')]
        except KeyError, err:
            print("There's no task implementation found for task input {0}".format(err))
            raise

        for k, v in kwargs.items():
            if k == 'type':
                continue
            setattr(self, k, v)

    @classmethod
    def from_yaml(cls, loader, node):
        # Load the yaml input; don't worry about objects....ignore them.
        arg_dict = loader.construct_mapping(node, deep=True)
        return cls(**arg_dict)

    @classmethod
    def to_yaml(cls, dumper, node):
        # This is where we dump the yaml.  DO NOT DUMP PYTHON OBJECTS!!!!!
        dict_representation = {
            'type': node.task_type.name,
            'kwargs': node.kwargs
        }
        return dumper.represent_mapping(cls.yaml_tag, dict_representation)


def load_yaml(yaml_file, additional_args={}):
    """Converts the YAML into YAMLTask() objects.

    Args:
        yaml_file: file - An open file handle, this would be if argparse is using the type=argparse.FileType('r')
        parameter to load and validate the file.
        additional_args: dict - Dictionary of command line arguments to add or replace specific kwargs.

    Returns: dict - {tasks: [list of YAMLTask() objects], task:[List of individual string tasks]}

    """
    f_path = None
    if isinstance(yaml_file, file) or isinstance(yaml_file, cvmanager_catalog.CatalogFile):

        # In some reboot scenarios where we reboot with yaml file on command line, make sure the file is open for use.
        if getattr(yaml_file, 'closed', False):
            yaml_file = cvmanager_catalog.CatalogFile(yaml_file.name)

        with yaml_file as in_file:
            f_path = in_file.name
            dictionary = yaml.load(in_file)

    elif isinstance(yaml_file, str):
        f_path = yaml_file
        with open(yaml_file, 'r') as in_file:
            dictionary = yaml.load(in_file)
    else:
        raise IOError(2, "No input file found for processing!")

    # This updates the kwargs in the yaml, with anything specified on command line OR additional..
    for idx, v in enumerate(dictionary.get('tasks', [])):
        if dictionary['tasks'][idx].kwargs is not None:
            dictionary['tasks'][idx].kwargs.update(**additional_args)
        else:
            dictionary['tasks'][idx].kwargs = additional_args

    # If the additional args are > 0 it means these arguments came from somewhere that isn't the input file.  Therefore
    # we should re-rewrite the input file with these arguments.
    # 3/12/2020: EF - Removed this because its polluting the template yaml file.  We shouldn't modify the template.
    """
    if len(additional_args) > 0:
        with open(f_path, 'w') as tmp:
            tmp.write(yaml.dump(dictionary, default_flow_style=False))
    """

    return dictionary


def get_template(task_type, **kwargs):
    """
    Gets the template for the specified task.  This is used for running remote tasks on nodes.
    Args:
        task_type (str): Name of the task.  This task must exist in Tasks repository.
        **kwargs:

    Returns:

    """
    # Create the template file in the catalog dir.  Setup the catalog.
    task_catalog = cvmanager_catalog.Catalog()
    temp_file = task_catalog.get_temp_file(**{'suffix': cvmanager_defines.YAML_SUFFIX,
                                              'prefix': cvmanager_defines.YAML_PREFIX})

    # open the remote task template yaml.
    with open(cvmanager_defines.REMOTE_TASK_YAML, 'r') as in_yaml:
        loaded_yaml = load_yaml(in_yaml, {})

    loaded_yaml['tasks'][0].task_type = task_defines.TaskType[task_type]

    # These kwargs will be from the template YAML file.  I believe it's always None, but handle ones we may add.
    if loaded_yaml['tasks'][0].kwargs is None:
        loaded_yaml['tasks'][0].kwargs = kwargs
    else:
        loaded_yaml['tasks'][0].kwargs.update(**kwargs)

    with temp_file as tmp:
        tmp.write(yaml.dump(loaded_yaml, default_flow_style=False))

    return temp_file

