# Python imports
import errno
import json
import datetime
import os

# Project imports
import cvmanager_logging
import cvmanager_xml_request
import cvmanager_defines
import cvmanager_yaml


class Workflow(object):
    def __init__(self, workflow_data):
        self.name = workflow_data.get('name', 'default')
        self.tk = workflow_data.get('token', 'default')
        self.job_id = workflow_data.get('job_id', 'default')
        self.current_status = None
        self.log = cvmanager_logging.get_log()
        self.initialized = False

    def initialize(self):
        # Do not suspend if the workflow doesn't have that parameter.
        self.log.info("Initializing a workflow based task execution.")

        xml_params = {
            'tk': self.tk,
            'jobId': self.job_id,
            'operationType': cvmanager_defines.JOB_OP_TYPE_SUSPEND
        }
        if not cvmanager_xml_request.XMLRequest('job_control', **xml_params).execute():
            self.log.error("Failed making xml request to suspend job.")
            return False

        self.initialized = True
        return True

    def tear_down(self):
        self.log.info("Resuming workflow job.")

        xml_params = {
            'tk': self.tk,
            'jobId': self.job_id,
            'operationType': cvmanager_defines.JOB_OP_TYPE_RESUME
        }

        if not cvmanager_xml_request.XMLRequest('job_control', **xml_params).execute():
            self.log.error("Failed making xml request to resume job.")
            return False

        return True

    def write_job_status(self, exit_code):
        json_data = {}
        if not os.path.exists(os.path.dirname(cvmanager_defines.CVMANAGER_EXIT_CODE_STATUS)):
            os.makedirs(os.path.dirname(cvmanager_defines.CVMANAGER_EXIT_CODE_STATUS))

        if os.path.exists(cvmanager_defines.CVMANAGER_EXIT_CODE_STATUS):
            with open(cvmanager_defines.CVMANAGER_EXIT_CODE_STATUS, 'r') as in_file:
                json_data = json.load(in_file)

        # Update this job id.
        json_data[self.job_id] = {
            'pid': os.getpid(),
            'latest_timestamp': datetime.datetime.today().isoformat(),
            'exit_code': exit_code
        }

        with open(cvmanager_defines.CVMANAGER_EXIT_CODE_STATUS, 'w') as out_file:
            json.dump(json_data, out_file)

        return


def workflow_factory(input_file, command_line_args):
    workflow_list = []
    if cvmanager_defines.WORKFLOW_DATA_INPUT_KEY in command_line_args.keys():
        workflow = Workflow(command_line_args[cvmanager_defines.WORKFLOW_DATA_INPUT_KEY])
        workflow_list.append(workflow)

    # Not on command line, check input yaml file if this is a workflow
    try:
        loaded_input = cvmanager_yaml.load_yaml(input_file, {})
    except IOError as e:
        if e.errno == errno.ENOENT:
            # do your FileNotFoundError code here
            return workflow_list
        else:
            raise
    except Exception, ex:
        raise

    # check the loaded yaml file if there's any workflow data.
    for t in loaded_input.get('tasks', {}):
        if cvmanager_defines.WORKFLOW_DATA_INPUT_KEY in t.kwargs.keys():
            workflow = Workflow(t.kwargs[cvmanager_defines.WORKFLOW_DATA_INPUT_KEY])
            workflow_list.append(workflow)

    return workflow_list
