# Python imports
import socket
import os
import re
import glob
import time
import psutil
import netifaces
import json
from ConfigParser import NoOptionError, NoSectionError

# Project imports
import wrappers


def ip4_addresses():
    ip_list = []
    for interface in netifaces.interfaces():
        for link in netifaces.ifaddresses(interface).get(netifaces.AF_INET, ()):
            ip_list.append(link['addr'])
    return ip_list


def get_local_ip_for_remote_ip(remote_ip):
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    s.connect((remote_ip, 80))
    ip_addr = s.getsockname()[0]
    s.close()

    return ip_addr


def purge(directory, pattern):
    for f in os.listdir(directory):
        if re.search(pattern, f):
            os.remove(os.path.join(directory, f))


def get_latest_file_by_pattern(search_path, file_name_pattern, recursive=False):
    """
    Searches the [search_path] for files matching [file_name_pattern] and returns the MOST RECENT matches.
    Args:
        search_path: (str) - The directory path to search for files.
        file_name_pattern: (str) - The file name pattern to search for.
        recursive: (bool) - True to search recursively.

    Returns: (str) - Path to the latest file matching the pattern.

    """
    latest = None
    matches = get_all_file_by_pattern(search_path, file_name_pattern)
    if len(matches) > 0:
        matches.sort(key=os.path.getmtime)
        latest = matches[-1]

    return latest


def get_all_file_by_pattern(search_path, file_name_pattern):
    """
    Searches the [search_path] for files matching [file_name_pattern] and returns those matches.
    Args:
        search_path: (str) - The directory path to search for files.
        file_name_pattern: (str) - The file name pattern to search for.

    Returns: (str) - Path to the latest file matching the pattern.

    """
    return glob.glob(os.path.join(search_path, file_name_pattern))


class BaseClass(object):
    def __init__(self, class_type):
        self._type = class_type


def factory(name, args, base_class=BaseClass):
    def __init__(self, **kwargs):
        for key, value in kwargs.items():
            # here, the args variable is the one passed to the
            # ClassFactory call
            if key not in args:
                raise TypeError("Argument {0} not valid for {1}".format(key, self.__class__.__name__))
            setattr(self, key, value)
        BaseClass.__init__(self, name[:-len("Class")])
    new_class = type(name, (base_class,), {"__init__": __init__})
    return new_class


def seconds_elapsed():
    # How long has the system been up.
    return time.time() - boot_time()


def boot_time():
    return psutil.boot_time()


def cvmanager_pid_running(pid):
    if psutil.pid_exists(int(pid)):
        proc_name = "/proc/{0}/cmdline".format(pid)
        if os.path.exists(proc_name):
            with open(proc_name, "r") as f:
                if any("cvmanager.py" in line for line in f):
                    return True
                else:
                    return False
    else:
        return False


def timed_join_all(threads, timeout):
    start = cur_time = time.time()
    while cur_time <= (start + timeout) and any(thread.is_alive() for thread in threads):
        for thread in threads:
            if not thread.is_alive():
                thread.join()
        time.sleep(1)
        cur_time = time.time()
    else:
        if any(thread.is_alive() for thread in threads):
            print("Timed out waiting for threads to finish!")


def wait_for_arch(remote_node, timeout=1200, sleep_time=10):
    if timeout is None:
        # Set some default timeout, never let it be None
        timeout = 1200

    num_retries = (timeout/sleep_time)
    count = 0
    cmd = "/usr/local/bin/arch ping "+remote_node
    while count < num_retries:
        ret = wrappers.waitsystem_nostdout(cmd)
        if (ret):
            wrappers.sleep(sleep_time)
            count += 1
        else:
            return True
    return False


def get_from_config_file(config, section, key, boolean=False):
    try:
        return config.get(section, key)
    except NoSectionError, nse:
        return ""
    except NoOptionError, noe:
        return ""
    except Exception, err:
        raise


def get_json_from_config_section(cfg_file, section):
    # Take the (name, value) pair from options in this section and convert to JSON.
    option_map = {}
    for option in cfg_file.items(section):
        option_map[option[0]] = option[1]
    return json.loads(json.dumps(option_map))


def adjust_directory_permissions(path=None, recursive=False, mode=None):
    """
    Navigates through cvmanager, adjusting permissions on directories only, not modifying files.

    :param recursive: bool - recursively set permissions on all dirs.
    :param path: str - base path to adjust permissions on.
    :param mode: int - Permissions to set on directory.
    :return:
    """
    import cvmanager_defines
    if path is None:
        path = cvmanager_defines.CV_MGR_ROOT

    if mode is None:
        mode = cvmanager_defines.DEFAULT_PERMISSIONS

    if os.path.exists(path):
        os.chmod(path, mode)

        if recursive:
            for root, dirs, _ in os.walk(path):
                for d in dirs:
                    os.chmod(os.path.join(root, d), mode)
