# Python imports
import time
from functools import partial
import inspect
import threading

# Project imports
import task_defines
import cvmanager_logging

# We may run lots of test steps, don't constantly re-init the log
log = cvmanager_logging.get_log()


class TaskStep(object):
    @property
    def attribs(self):
        # Dont make a setter.  It's handled differently
        return self._attrib_dict

    @property
    def step_obj(self):
        return self

    @property
    def status(self):
        return self._status

    @status.setter
    def status(self, value):
        # Any time a step status is changed, write the file.
        self._status = value

    @property
    def start_time(self):
        return self._start_time

    @start_time.setter
    def start_time(self, value):
        self._start_time = value

    @property
    def end_time(self):
        return self._end_time

    @end_time.setter
    def end_time(self, value):
        self._end_time = value

    @property
    def local_child_task(self):
        return self._local_child_task

    @local_child_task.setter
    def local_child_task(self, value):
        self._local_child_task = value

    @classmethod
    def with_options(cls, arg_dict):
        def inner(func):
            new_func = cls(func)
            for k, v in arg_dict.items():
                new_func.add_attrib(k, v)
            return new_func
        return inner

    def add_attrib(self, k, v):
        self.attribs[k] = v

    def __init__(self, f, *args, **kwargs):
        self.func = f
        self.name = self.func.func_name

        # properties
        self._status = task_defines.StepStatus.NOT_ATTEMPTED
        self._start_time = ''
        self._end_time = ''
        self._attrib_dict = {}
        self._local_child_task = False
        self.__return_code = False

        # check function args.
        func_insp = inspect.getargspec(self.func)
        assert func_insp.keywords is not None, 'Your test step definition must include (....., *args, **kwargs)'
        assert func_insp.varargs is not None, 'Your test step definition must include (....., *args, **kwargs)'

    def __iter__(self):
        """ This is critical....anything here is used throughout the manager when accessing the TaskStep()
        Example: These are the ONLY keys and values written to and read from the status file.

        Returns: Tuple - (property, property value)

        """
        yield 'name', self.name
        yield 'status', self.status.name
        yield 'start_time', self.start_time
        yield 'end_time', self.end_time
        yield 'local_child_task', self.local_child_task
        yield 'return_codes', self.__return_code
        for k, v in self.attribs.items():
            yield k, v

    def __get__(self, instance, owner):
        return partial(self.__call__, instance, **dict(self))

    def __call__(self, *args, **kwargs):
        global log
        # Before function
        self.start_time = time.time()

        try:
            # __return_code captures the actual return value from every step function.
            self.__return_code = self.func(*args, **kwargs)
        except Exception, err:
            log.debugv("{0}".format(err))
            self.__return_code = task_defines.StepStatus.EXCEPTION  # False

        self.end_time = time.time()

        # Translate those return codes into status enumerations.  It's possible an enum was directly returned.
        if self.__return_code == task_defines.StepStatus.PASSED or self.__return_code is True:
            self.status = task_defines.StepStatus.PASSED
        elif self.__return_code == task_defines.StepStatus.EXCEPTION:
            self.status = task_defines.StepStatus.EXCEPTION
        else:
            if self.__return_code == task_defines.StepStatus.REBOOT_AND_RESUME:
                self.status = task_defines.StepStatus.REBOOT_AND_RESUME
            else:
                self.status = task_defines.StepStatus.FAILED

        # We return self because the obj has been modified.  So the updated obj is sent back.
        return self


class StepThread(threading.Thread):
    """This is a python 2.7 workaround.  Will not work for Python 3.
    See here:
        https://stackoverflow.com/questions/6893968/how-to-get-the-return-value-from-a-thread-in-python
    """
    def __init__(self, group=None, target=None, name=None,
                 args=(), kwargs={}):
        threading.Thread.__init__(self, group, target, name, args, kwargs)
        self._return = None

    def run(self):
        if self._Thread__target is not None:
            self._return = self._Thread__target(*self._Thread__args, **self._Thread__kwargs)

    def my_join(self, *args):
        threading.Thread.join(self)
        return self._return


class OptionStep(TaskStep):
    # Method definition(s) MUST NOT match the attribute being set, otherwise you'll see unexpected results.
    @classmethod
    def run_always(cls, func):
        new_func = cls(func)
        new_func.add_attrib('always_run', True)
        return new_func

    @classmethod
    def run_once(cls, func):
        # Any step decorated with this, will only run 1 time per any given task\resume\etc.
        new_func = cls(func)
        new_func.add_attrib('once', True)
        return new_func

    def __init__(self,  f, *args, **kwargs):
        """These must ALL be decorated class methods.  You can then decorate your steps with any one of these.
            Example: In your task file, you can do the following:
                @OptionStep.run_always
                def check_for_nodes(self, *args, **kwargs): pass

        They are basically simple helper methods instead of users having to add the dictionary
        """
        super(OptionStep, self).__init__(f, *args, **kwargs)
