# Python imports
import curses
import fnmatch
import json
import os
import errno
from collections import OrderedDict
import threading
import time
import datetime
import shutil
import glob

# Project imports
import task_defines
import cvmanager_defines as define
import cvmanager_logging
import cvmanager_yaml
import cvmanager_task
import cvmanager_utils
import wrappers


_NO_WRITE = False    # Useful if debugging and not wanting to save any status files.


class StatusFile(object):
    @property
    def report(self):
        return self._status_report

    @report.setter
    def report(self, value):
        self._status_report = value

    @property
    def path(self):
        return os.path.join(self.status_dir, self.status_file)

    @property
    def status_file(self):
        return self._status_file

    @status_file.setter
    def status_file(self, value):
        self._status_file = value

    def __init__(self, status_dir, status_file, *args, **kwargs):
        """ A basic status file, this is already proven to be an existing file, so do a basic parse.
        Args:
            *args:
            **kwargs:
        """
        self.log = cvmanager_logging.get_log()
        self.status_file = status_file
        self.status_dir = status_dir
        self.json = self.parse_status_file()
        self.new_file = False
        self._status_report = []
        self.archive_file = True if 'archive' in status_dir else False

    def write_status_file(self, data):
        """Writes the current status to file."""
        if _NO_WRITE:
            return

        with open(self.path, 'w') as sf:
            json.dump(data, sf)

    def delete_status_file(self):
        if os.path.exists(self.path):
            os.remove(self.path)

    def parse_status_file(self):
        status_info = {}
        if not os.path.exists(self.path):
            return status_info

        try:
            with open(self.path, 'r') as sf:
                status_info = json.load(sf, object_pairs_hook=OrderedDict)
        except ValueError, ve:
            if ve.message == 'No JSON object could be decoded':
                # Couldn't load, just assume new run
                pass
        return status_info

    def create_status_file(self):
        if _NO_WRITE:
            return

        self.check_path()
        with open(self.path, 'w') as sf:
            json.dump(self.json, sf)

        # Remember that we created this file!
        self.new_file = True

    def check_path(self, path=None):
        cur_path = path or os.path.dirname(self.path)
        if not os.path.exists(cur_path):
            try:
                os.makedirs(cur_path)
            except OSError as err:
                if err.errno != errno.EEXIST:
                    raise

    def build_report(self):
        # Always reload the status file.  TODO: Race conditions and locking here?
        table = []
        json_data = self.parse_status_file()
        if len(json_data) == 0:
            self.report = (table, json_data.get('status', 'N\\A'))
            return

        # Forms the file into an output report.
        # Task - Phase - Step - Status
        task_name = json_data['display_tree'] + ' -> ' + self.status_file.split('.')[0]

        def process_section(heading, section):
            for step_name, step_params in json_data[section].items():
                if step_params.get('local_child_task', False):
                    status = 'Child Task: {0} - {1}'.format(step_params.get('local_child_task'),
                                                            step_params['status'])
                else:
                    status = step_params['status']
                table_row = (task_name, heading, step_params.get('display_name', step_name), status)
                table.append(table_row)

        process_section('Pre Process', 'pre_process')
        process_section('Main Process', 'main_process')
        process_section('Post Process', 'post_process')

        self.report = (table, json_data.get('status', 'N\\A'))


class ProcessStatus(StatusFile):
    """UpgradeStatus is the main class which handles all node upgrade tracking.  On the NFS share (define.NFS_SHARE)
    there will be a directory structure as follows:
    defines.NFS_SHARE
        |__ define.Task_<uid> directory
                -- <task_name>.status       <- Status file holding information for this specific task
                |_ Task_<uid> dir           <- This is a child task spawned from main task.
                    -- <task_name>.status
                |_ Task_<uid> dir           <- This is a child task spawned from main task.
                    -- <task_name>.status

    All files are JSON format serialization of the TaskProcess and its TaskStep
    """
    def __init__(self, task, *args, **kwargs):
        """

        Args:
            task: cvmanager_task.TaskObject() - The task for this specific status to be tracked.
            *args:
            **kwargs:
        """
        # Setup the task directory tree.
        task_base_dir = define.TaskDir.base.format(task.uid)   # Task_<UID>
        
        if task.tree == '':
            root_dir = os.path.join(define.TaskDir.status, task_base_dir)      # /ws/ddb/cvmanager/catalog/status
            self.status_dir = os.path.join(root_dir, task.tree)
        else:
            root_dir = os.path.join(task.tree, task_base_dir)
            self.status_dir = root_dir

        self.current_tree = root_dir
        self.display_tree = task.display_tree

        # Initialize the file.
        super(ProcessStatus, self).__init__(self.status_dir, define.TaskDir.status_file.format(task.task_type.name))

        # Dump the process to file.
        if len(self.json) == 0:
            # We didn't find an existing file, so create a new one based off this task process.
            self.json = self.create_status_file(task.process)

        # At this point the TaskProcess() has been defined, and the TaskStatus has been created OR loaded.
        self.init(task)

    def init(self, task):
        """ Go through the status file & task process to determine if we need to run this step again or not.
        The process object will be treated as golden standard, if there's no status for this operation or step in the
        JSON, then assume we need to run this step, its new or added.  If this is first run, it will always run
        everything.

        Because a TaskStep is a partial object, its not initialized so we dont have access to self.  Use the keywords
        set here.  If you need more attributes, then add more keywords in the TaskStep.def __iter__()
            def __iter__(self):
                yield 'name', self.name
                yield 'status', self.status.name

            def __get__(self, instance, owner):
                return partial(self.__call__, instance, **dict(self))
        """
        # If this is already successful, delete it and start fresh!
        # Only do this for parent tasks....child tasks, if already completed, should not be re-executed.
        if self.json['status'] == task_defines.ProcessStatusCode.SUCCESS.name and self.display_tree == 'Main':
            self.log.warning("Task [{0}] already completed, deleting the existing status file and re-starting!".format(
                task))
            self.delete_status_file()
            self.json = self.create_status_file(task.process)

        # This section of code will synchronize the JSON status file TO the internal TaskProcess and TaskStep() objects.
        for input_phase, task_process_steps in task.process:
            # We should NEVER Have a phase in the json status file that is NOT part of the process.  This means
            # something went really wrong.  TODO: ABORT OR Start over?
            if input_phase in self.json.keys():
                # Sync the status file to the task.process.  task.process steps are the official list of steps.
                for step in task_process_steps:
                    # step -> partial task_process_step function
                    if step.keywords['name'] in self.json[input_phase].keys():
                        # This step is part of the current task AND in the saved status file.  So we will & replace.
                        for k, input_v in self.json[input_phase][step.keywords['name']].items():
                            step.keywords[k] = input_v

        # Decide what to set the task process status_code as.  This means init is done and correct.
        # It needs to be a task_defines.StatusStates.READY option...This is the ONLY place to set the task status.S
        if self.new_file:
            task.process.status_code = task_defines.ProcessStatusCode.INITIALIZED
        else:
            if not self.json['status'] == task_defines.ProcessStatusCode.SUCCESS.name:
                task.process.status_code = task_defines.ProcessStatusCode.RESUMING
            else:
                self.log.info("Task [{0}]->[{1} - {2}] is already completed; not resuming.".format(self.display_tree,
                                                                                                   task, task.uid))

    def setup_remote_child_task_status(self, remote_task_hash, task_name):
        # This is on the CONTROL node, tracking remote task status.
        task_base_dir = 'Remote_' + define.TaskDir.base.format(remote_task_hash)
        child_status_dir = os.path.join(self.status_dir, task_base_dir)

        if not os.path.exists(child_status_dir):
            # First attempt running this child task.
            os.makedirs(child_status_dir)

            # By default just set the file to FAILED status.  Remote tasks are ONLY pass\fail
            open(os.path.join(child_status_dir, task_name+'.FAILED'), 'w').close()
        else:
            # The path exists, check file.
            if not bool(glob.glob(os.path.join(child_status_dir, task_name+'.*'))):
                # There are no status files for this remote child.  Create a failed one.
                open(os.path.join(child_status_dir, task_name+'.FAILED'), 'w').close()

        # Do not touch any existing files, passed or failed.
        return child_status_dir

    def create_status_file(self, process_template):
        if _NO_WRITE:
            return

        self.check_path()
        status_info = process_template.serialize_process()
        with open(self.path, 'w') as sf:
            json.dump(status_info, sf)

        # First task to create this file, mark it as NEW!
        self.new_file = True
        return status_info

    def fcount(self, path):
        count1 = 0
        for root, dirs, files in os.walk(path):
            count1 += len(dirs)

        return count1

    def archive_tree(self):
        # Create a path for today and move the file there.
        # TODO: Limit the number of archive files!!!!!
        archive_day_path = os.path.join(define.TaskDir.archive, datetime.date.today().isoformat())
        archive_time_path = os.path.join(archive_day_path, datetime.datetime.now().strftime("%s"))

        self.check_path(archive_time_path)
        self.log.info("Archiving status file [{0}] to [{1}].".format(self.status_dir, archive_time_path))
        shutil.move(self.status_dir, archive_time_path)
        return


class TailStatus(StatusFile):
    def __init__(self, input_task=None, status_file=None, by_task_name=None, *args, **kwargs):
        """This will tail all active status files and show output on the screen in a friendly user format.

        We have to try to find the proper status directory.  It changes for hostname or ip
        If the cvmanager is run with -t and -s; taskname and status, this will display only the most recent status for
        this task from anywhere including archive.

            EXAMPLE: cvmanager.py -t Upgrade -s
            - Will search for, and display the most recent, including currently running statuses.
            - If its currently running, it will be tailed, if its in the archive, only displayed.

        status_file = Full path to a <task_name>.status file.  Displays just that.

        Args:
            *args:
            **kwargs:
        """
        self.threads = []
        self.current_files = []

        if by_task_name is not None and input_task is None and status_file is None:
            # This case we're going to find the latest status file for named task.
            self.task_base_dir = define.TaskDir.status.replace('{0}', '*')
            self.find_all_status_files(True, by_task_name)

            if not len(self.current_files) > 0:
                raise Exception("Unable to find any status for the task [{0}].  It looks like it has never run.".format(
                    by_task_name
                ))

            def mod_time(sf):
                return os.path.getmtime(sf.path)

            latest_file = max(self.current_files, key=mod_time)

            latest_file.log.info("Loading status file: {0}".format(latest_file.path))
            status_file = latest_file.path
            self.current_files = [latest_file]

        if status_file is not None:
            # User supplied a specific file; exact path to file.
            include_archive = True
            if not os.path.exists(status_file):
                raise Exception("Did not locate specified status file [{0}].".format(status_file))
            local_cvmanager_root = os.path.dirname(status_file)
        else:
            include_archive = False
            local_cvmanager_root = define.TaskDir.status.replace('{0}', '*')

        status_dirs = [os.path.dirname(d) for d in glob.glob(local_cvmanager_root)]
        hostname = wrappers.get_hostname()
        for idx, dir_name in enumerate([os.path.basename(d) for d in status_dirs]):
            if dir_name == hostname:
                # We found a status dir for this hostname, use it.
                local_cvmanager_root = os.path.join(status_dirs[idx], 'status')
                break
        else:
            # We did not find a match, so try with IP address.
            ip_addr = cvmanager_utils.get_local_ip_for_remote_ip('8.8.8.8')
            for idx, dir_name in enumerate([os.path.basename(d) for d in status_dirs]):
                if dir_name == ip_addr:
                    # We found a status dir for this hostname, use it.
                    local_cvmanager_root = os.path.join(status_dirs[idx], 'status')
                    break

        # Setup the task directory tree.
        matches = []
        if input_task is not None:
            # This is for the -cs SPECIFIC TASK STATUS option
            # Find this directory anywhere in the tree of status directories.
            for root, dirs, files in os.walk(local_cvmanager_root):
                for d in dirs:
                    if d == input_task:
                        matches.append(os.path.join(root, d))

            if len(matches) > 1:
                # We found this task in many places. Not sure this is possible.
                raise Exception("Too many status files found for this task.  Re-run cvmanager.py with -s option only.")
            elif len(matches) == 0:
                return
            else:
                self.task_base_dir = matches[0]
        else:
            self.task_base_dir = local_cvmanager_root

        curses.wrapper(self.print_table, include_archive)

    def print_table(self, screen, include_archive, *args, **kwargs):
        try:
            while True:
                height, width = screen.getmaxyx()

                # This is every loop of updating the display for the user; this gets all of the child tasks too.
                self.update_files_and_launch_threads(include_archive)

                if len(self.current_files) == 0:
                    break

                def add_new_row(current_row, inner_line, *inner_args):
                    if current_row >= height:
                        return current_row

                    screen.addnstr(current_row, 0, inner_line, width - 1, *inner_args)
                    return current_row+1

                added_rows = 0
                for main_row, status_file in enumerate(self.current_files):
                    if len(status_file.report) != 2:
                        continue
                    col_width = [max(len(x) for x in col) for col in zip(*status_file.report[0])]

                    # Check if we have a new status.
                    heading_line = "-----Task: {0} - {1}".format(
                        status_file.status_file.replace('.status', ''),
                        status_file.report[1])
                    added_rows = add_new_row(added_rows, heading_line, *(curses.A_BOLD,))

                    for row, line in enumerate(status_file.report[0], added_rows):
                        formatted_line = "| " + " | ".join("{0:{1}}".format(x, col_width[i]) for i, x in
                                                           enumerate(line)) + " |"
                        added_rows = add_new_row(added_rows, formatted_line)

                    added_rows = add_new_row(added_rows, "")

                screen.scrollok(True)
                screen.refresh()
                time.sleep(task_defines.STATUS_CONSOLE_REFRESH)
        except Exception as err:
            print(str(err))

    def update_files_and_launch_threads(self, include_archive=False):
        """ This will get a StatusFile() object per status file located in the non-archive repository.  It will
        put each one of those files into a thread which will call the build_report() method.  This is thread will
        essentially be formatting the status file into a screen viewable report in a thread.

        Returns:

        """
        self.find_all_status_files(include_archive)

        # Initialize the file & monitoring thread.
        for i, status_file in enumerate(self.current_files):
            # Only start a thread if its not already running.
            if id(status_file) not in self.threads:
                status_thread = StatusThread(name=i, kwargs={'status_file': status_file})
                self.threads.append(id(status_file))
                status_thread.start()

    def find_all_status_files(self, include_archive=False, task_name_filter=False):
        """ This just searches the status file repo and looks for files.  As it finds files, it will create new
        StatusFile() objects and keep track of them.
        TODO: Remove ones or handle completed files.
        """
        for root, dir_names, filenames in os.walk(self.task_base_dir):
            if 'archive' in root and not include_archive:
                # Do not include archived tasks in the status tail.  Tailing is designed for live running tasks.
                continue
            if task_name_filter:
                f_name_filter = task_name_filter + '.status'
            else:
                f_name_filter = '*.status'

            for filename in fnmatch.filter(filenames, f_name_filter):
                temp_status = StatusFile(root, filename)
                if temp_status.path not in (x.path for x in self.current_files):
                    self.current_files.append(temp_status)


class StatusThread(threading.Thread):
    def __init__(self, group=None, target=None, name=None,
                 args=(), kwargs=None, verbose=None):
        super(StatusThread, self).__init__()
        self.daemon = True

        self.status_file = kwargs['status_file']

    def run(self):
        while True:
            # Only reload the report every so often.  Don't tight loop this.
            self.status_file.build_report()
            time.sleep(task_defines.STATUS_FILE_RELOAD_THREAD)


class LatestStatus(StatusFile):
    def __init__(self, input_yaml, mgr, *args, **kwargs):
        """
        Load the supplied input yaml file.  THis file will have a task type and kwargs.
        Args:
            input_yaml:
            *args:
            **kwargs:
        """
        self.task_base_dir = define.TaskDir.status
        self.input_path = os.path.abspath(input_yaml)

        tasks = cvmanager_yaml.load_yaml(self.input_path, {})

        # Init the task.
        for task in tasks['tasks']:
            task_hash = cvmanager_task.TaskObject.hasher(task.task_type.name, task.kwargs)
            task_dir = os.path.join('Task_{0}'.format(task_hash))
            task_file = os.path.join(task_dir, task.task_type.name + '.status')

            # Look for this pattern anywhere in the tree.
            TailStatus(task_dir)
