# Python imports
from functools import partial
from collections import OrderedDict
import time
from os import getpid

# Project imports
import task_defines
import cvmanager_task_step
import cvmanager_logging


class TaskProcess(object):
    """This is nothing more than a sequence of steps required for upgrade.  The individual task class must update the
    sequence, as only they will know."""
    @property
    def pre_process(self):
        return self._pre_process

    @pre_process.setter
    def pre_process(self, value):
        self._pre_process = value

    @property
    def main_process(self):
        return self._main_process

    @main_process.setter
    def main_process(self, value):
        self._main_process = value

    @property
    def post_process(self):
        return self._post_process

    @post_process.setter
    def post_process(self, value):
        self._post_process = value

    def __iter__(self):
        # return tuple of the process phase,actual instances of the steps
        yield task_defines.PROC_PRE, self.pre_process
        yield task_defines.PROC_MAIN, self.main_process
        yield task_defines.PROC_POST, self.post_process

    def __init__(self, task):
        """TaskProcess is the steps or code or "process" that must be completed for this task.  There are 3
        distinct phases of the TaskProcess; pre-process, main-process, and post-process, run in that order.  It is the
        consumer (person writing the Task()) to set the steps required for each phase.  It is not required to utilize
        all phases.  They are a convenience provided only as a logical grouping if needed.

        Args:
            task: Task() - Any task object which has been created.
        """
        self._pre_process = []
        self._main_process = []
        self._post_process = []
        self.status_code = task_defines.ProcessStatusCode.NOT_STARTED
        self.status = None
        self.pid = getpid()
        self.log = cvmanager_logging.get_log()
        self.display_tree = task.display_tree
        self.active_step = 'None'

        task.set_process(self)

    def __call__(self, phase):
        serialized_proc = self.serialize_process()
        return serialized_proc[phase]

    def __format_process(self, proc_type):
        process = OrderedDict()
        process[proc_type] = OrderedDict()
        for step in getattr(self, proc_type, []):
            if isinstance(step, partial):
                params = step.keywords
            elif isinstance(step, cvmanager_task_step.TaskStep):
                params = dict(step)

            process[proc_type][params.get('name')] = OrderedDict()
            for k, v in params.items():
                if isinstance(v, task_defines.StepStatus):
                    v = v.name
                process[proc_type][params.get('name')][k] = v
        return process

    def serialize_phase(self, phase):
        return self.__format_process(phase)

    def serialize_process(self):
        # This takes the TaskProcess object and serializes it into OrderedDict so it may be saved to file.
        process = OrderedDict()
        process['status'] = self.status_code.name
        process['pid'] = self.pid
        process['display_tree'] = self.display_tree

        process.update(self.__format_process(task_defines.PROC_PRE))
        process.update(self.__format_process(task_defines.PROC_MAIN))
        process.update(self.__format_process(task_defines.PROC_POST))

        return process

    def save(self):
        serialized_json = self.serialize_process()
        if self.status:
            self.status.write_status_file(serialized_json)

    def run_step(self, step, current_phase, step_index, *args, **kwargs):
        # Run the actual step
        self.active_step = step

        step.keywords['status'] = task_defines.StepStatus.INITIALIZED

        thread = cvmanager_task_step.StepThread(target=step, args=args, kwargs=kwargs)
        thread.start()

        # Mark the step RUNNING.
        step.keywords['status'] = task_defines.StepStatus.RUNNING

        while thread.isAlive():
            # This loop continually updates the status file as needed.
            # TODO: only update if changed!  Right now its constantly updating.
            self.save()

            # Update the status file every 10 seconds.
            time.sleep(task_defines.TASK_POLL_INTERVAL)

        updated_step = thread.my_join()

        # Don't remember why this is here; leave for now, but check on any issues with child tasks.
        updated_step.local_child_task = self.active_step.keywords.get('local_child_task', False)

        # current_phase is the Phase.pre_process\main_process\post_process object.  Update with the returned info.
        current_phase[step_index] = updated_step

        # The step has completed running.  Save the status a final time.
        self.save()

        del thread

        return updated_step.status or task_defines.StepStatus.FAILED
