import threading
import subprocess
import traceback
import shlex
import os
from enum import Enum

import cvmanager_utils
import cvmanager_defines
import cvmanager_logging
import common
import wrappers
from HsObject import hs_ssh

LOG = cvmanager_logging.get_log()


class ProcessCode(Enum):
    SUCCESS = 0
    FAILED = 1
    EXCEPTION = -1
    FAILED_INIT = 2
    NOT_STARTED = 999


class RemoteCommand(object):
    """ Run subprocess in separate thread so we can have timeout.

    """
    command = None
    process = None
    status = None
    output, error = '', ''

    def __repr__(self):
        return getattr(self, 'status', False)

    def __init__(self, remote_host, local_status_dir, task_name='Remote_Task', wait=True):

        self.remote_hostname = remote_host
        self.command = None
        self.wait = wait
        self.log = LOG

        # All remote tasks need a local working directory as they will write into the share.
        # This is the local reference to the share the remote task will write things to: cvmanager_defines.TaskDir.share
        self.local_remote_task_working_dir = os.path.join(cvmanager_defines.NFS_SHARE, self.remote_hostname)
        if not os.path.exists(self.local_remote_task_working_dir):
            os.makedirs(self.local_remote_task_working_dir)

        # The local directory for status files of this remote task.
        self.status_dir = local_status_dir

        self.task_name = task_name

    def is_ready(self):
        raise NotImplementedError("is_ready must be implemented for this type.")

    def update_status_file(self):
        # Purge any stale files, just write the current status.  We don't care about previous attempts.
        cvmanager_utils.purge(self.status_dir, self.task_name)

        if self.status == 0 or self.status is True or self.status == ProcessCode.SUCCESS:
            status_file = os.path.join(self.status_dir, self.task_name + '.PASSED')
        else:
            status_file = os.path.join(self.status_dir, self.task_name + '.FAILED')
        open(status_file, 'w').close()

    def run(self, timeout=None, **kwargs):
        """ Run a command then return: (status, output, error).
        The OpenArchive framework only returns True or False (0, 1)

        timeout should always = None so that thread default is NO timeout.

        """

        def target(**keywords):
            try:
                # Reset the outputs in case the process is re-launched, like post reboot.
                self.process = None
                self.output = None
                self.error = None
                self.status = ProcessCode.NOT_STARTED

                if isinstance(self.command, basestring):
                    self.command = shlex.split(self.command)

                self.process = subprocess.Popen(self.command, **keywords)
                self.output, self.error = self.process.communicate()
                self.status = self.process.returncode
            except Exception, err:
                self.error = traceback.format_exc()
                self.status = ProcessCode.EXCEPTION
            finally:
                self.update_status_file()

        # default stdout and stderr; TODO: Ensure we don't hang here.
        if 'stdout' not in kwargs:
            kwargs['stdout'] = subprocess.PIPE
        if 'stderr' not in kwargs:
            kwargs['stderr'] = subprocess.PIPE

        # If this is an arch command, re-copy
        if isinstance(self, ArchCommand):
            if not self.initialize_command(timeout):
                self.log.error("Failed to initialize the command for execution.")
                self.status = ProcessCode.NOT_STARTED
                return

        # thread
        thread = threading.Thread(target=target, kwargs=kwargs)
        thread.start()

        if self.wait:
            # Wait for the thread to complete OR timeout.
            thread.join(timeout)
            if thread.is_alive():
                # We timed out waiting for status.
                self.process.terminate()
                thread.join()

        return

    def time_out_tear_down(self):
        # Default tear down, do nothing, just return
        return

    def wait_for_reboot(self, status):
        import cvupgrade_common

        local_remote_node_path = self.local_remote_task_working_dir
        reboot_file = cvmanager_defines.RTC_REBOOT_AND_RESUME.format(self.uid)
        remote_task_reboot_file = os.path.join(local_remote_node_path, reboot_file)

        if not os.path.exists(remote_task_reboot_file):
            # reboot file is gone?
            self.log.error("The reboot file is gone?")
            status[self.remote_hostname] = False
            return

        self.log.info("Waiting for node [{0}] to shutdown.".format(self.remote_hostname))
        if not cvupgrade_common.wait_for_shutdown(self.remote_hostname):
            self.log.error("Remote host has problem shutting down....please check")
            return
        self.log.info("Node [{0}] has shutdown.".format(self.remote_hostname))

        self.log.info("Waiting for node [{0}] to restart.".format(self.remote_hostname))
        if not cvupgrade_common.wait_for_reboot(self.remote_hostname) == 0:
            self.log.error("Remote host does not seem to be up after reboot....please check")
            return

        status[self.remote_hostname] = True
        return


class SSHCommand(RemoteCommand):
    def __init__(self, remote_host, local_status_dir, task_name='Remote_Task', wait=True, uid=None, yaml_file=None):
        super(SSHCommand, self).__init__(remote_host, local_status_dir, task_name, wait)

        self.uid = uid
        self.yaml_file = yaml_file

    def test_connection(self):
        ssh_conn = hs_ssh.RemoteSSH(self.remote_hostname)
        return ssh_conn.test_connection()

    def time_out_tear_down(self):
        # Close any and all SSH connections we might have on the instance.
        self.ssh_conn.close_connections()

    def run(self, timeout=None, **kwargs):
        self.status = ProcessCode.NOT_STARTED

        def target(**keywords):
            try:
                # Reset the outputs in case the process is re-launched, like post reboot.
                self.ssh_conn = None
                self.output = None
                self.error = None
                self.status = ProcessCode.NOT_STARTED

                self.ssh_conn = hs_ssh.RemoteSSH(self.remote_hostname)
                self.ssh_conn.execute_remote(self.command, debug=False)

                self.output = self.ssh_conn.output
                self.error = self.ssh_conn.error
                self.status = self.ssh_conn.status

            except Exception, err:
                self.error = traceback.format_exc()
                self.status = ProcessCode.EXCEPTION
            finally:
                self.update_status_file()

        if not self.initialize_command():
            self.log.error("Failed to initialize cvmanager; passwordless ssh not available.")
            self.status = ProcessCode.FAILED_INIT
            self.update_status_file()
            return

        # thread
        thread = threading.Thread(target=target, kwargs=kwargs)
        thread.start()

        if self.wait:
            # Wait for the thread to complete OR timeout.
            thread.join(timeout)
            if thread.is_alive():
                # We timed out waiting for status.
                self.log.error("Timed out waiting in remote SSH command; aborting and closing the connection.")
                self.time_out_tear_down()
                thread.join()
        return

    def initialize_command(self):
        # Copy the yaml file to remote node for execution. Copy to /tmp as arch only works there?
        remote_file_path = os.path.join(os.sep, 'tmp', os.path.basename(str(self.yaml_file)))

        try:
            ssh_conn = hs_ssh.RemoteSSH(self.remote_hostname)
            with ssh_conn.ssh_cli() as ssh_cli:
                sftp = ssh_cli.open_sftp()
                sftp.put(str(self.yaml_file), remote_file_path, confirm=True)

            # Setup the command for the remote task manager exec.
            self.command = "/opt/commvault/MediaAgent/task_manager/cvmanager.py {0}".format(remote_file_path)

            return True
        except Exception, err:
            self.log.debug(str(err))
            return False


class ArchCommand(RemoteCommand):
    def __init__(self, remote_host, local_status_dir, task_name='Remote_Task', wait=True, uid=None, yaml_file=None):
        super(ArchCommand, self).__init__(remote_host, local_status_dir, task_name, wait)

        self.uid = uid
        self.yaml_file = yaml_file

    def initialize_command(self, timeout):
        # This will init the command to run remotely, including re-copying any files needed on remote node.
        if not self.is_ready(timeout):
            return False

        # Copy the yaml file to remote node for execution. Copy to /tmp as arch only works there?
        remote_file_path = os.path.join(os.sep, 'tmp', os.path.basename(str(self.yaml_file)))
        cmd = "/usr/local/bin/arch write-file {0} {1} {2}".format(self.remote_hostname, self.yaml_file,
                                                                  remote_file_path)
        ret = wrappers.waitsystem(cmd)
        if ret:
            # TODO: Check if we should not do an exception, could exit caller which is bad.
            self.log.error("Failed copying task file to host.\n{0}\n{1}".format(cmd, self.remote_hostname))
            return False

        # Launch cvmanager on the remote node using arch, with this yaml file.
        self.command = "/usr/local/bin/arch manage commvault execute {0} task_manager/cvmanager.py {1}".format(
            self.remote_hostname, remote_file_path)

        return True

    def is_ready(self, timeout):
        # Using cv open archive (arch), lets copy this file to the remote node.
        # Wait up to 5 minutes for arch.
        self.log.debug("Checking arch connection to node [{0}] is available.".format(self.remote_hostname))
        if not cvmanager_utils.wait_for_arch(self.remote_hostname, timeout=timeout):
            self.log.error("Timed out connecting to remote node [{0}] using arch.".format(self.remote_hostname))
            return False

        return True

