# Python imports
import os
import shutil
import re
from threading import Lock

# Project imports
import cvmanager_defines
import cvmanager_logging
import wrappers
import common

LOG = cvmanager_logging.get_log()
MUTEX = Lock()


def mutex_lock(f):
    global MUTEX

    def inner(*args, **kwargs):
        if not kwargs.get('locked', False):
            # locked not present assume we must take the lock.
            with MUTEX:
                return f(*args, **kwargs)
        else:
            # Ok to proceed without locking, because we already have it and know about it.
            return f(*args, **kwargs)
    return inner


class NFS(object):
    def __init__(self, exported_fs, clean_export=False):
        """
        This is not a universal library for all exports yet, restricted to just the selected exported_fs, to avoid
        possible conflicts or unknowns with customer parameters.
        Once we are more confident we can remove the restriction of the exported_fs!
        Args:
            exported_fs (str): Path of the file system to be exported.  EXAMPLE: /ws/ddb/cvmanager
            clean_export (bool): If TRUE, we will not carry-over any exports for the specified exported_fs.  The host
            for the exported_fs will be cleaned, and re-written with add_host_to_export().  If you don't add any hosts
            the the export will not work.
        """
        # Make sure the path we're going to export actually exists.
        if not os.path.exists(exported_fs):
            wrappers.mkdirs(exported_fs)

        self.exports_path = cvmanager_defines.EXPORTS_PATH
        self.exported_fs = exported_fs
        self.exports = {}
        self.clean_exports = clean_export
        self.init_exports(clean_export)

    @mutex_lock
    def initialize_nfs_server(self):
        # Check nfs-server.service is running.  If it's not, start it, and set reg key to stop on tear down.
        if not wrappers.is_service_up_nostdout('nfs-server.service'):
            self.kill_3dnfsd()
            ret = wrappers.start_service_nostdout('nfs-server.service', False)
            if ret:
                LOG.error("Failed to start nfs-server.service, exiting.")
                return False
            LOG.debug("Started nfs-server.service ...")
            common.setregistryentry(cvmanager_defines.REG_MA_KEY, cvmanager_defines.REG_SHUTDOWN_NFS_VALUE, "Yes")
        else:
            LOG.info("NFS server is already running on this node.")

    def kill_3dnfsd(self):
        # First stop ganesha.nfs, followed by 3dnfs.exe; waiting for the ports to be available
        if not wrappers.ProcessHandler.kill_ganesha():
            LOG.error("Failed to stop ganesha.nfs service.")
            return False

        if not wrappers.ProcessHandler.kill_3dnfs():
            LOG.error("Failed to stop 3dnfs.exe service.")
            return False
        return True

    @mutex_lock
    def stop(self, locked=False):
        # Read the reg key if we should stop nfs-server.service.
        value = common.getregistryentry(cvmanager_defines.REG_MA_KEY, cvmanager_defines.REG_SHUTDOWN_NFS_VALUE)

        # Always remove the reg key
        common.deleteregistryentry(cvmanager_defines.REG_MA_KEY, cvmanager_defines.REG_SHUTDOWN_NFS_VALUE)

        if value.lower() == 'yes':
            # Stop the NFS server.
            LOG.info("Stopping nfs-server.service")
            wrappers.waitsystem_nostdout("systemctl stop nfs-server.service")

        return True

    @mutex_lock
    def export_without_restart(self, locked=False):
        ret = wrappers.waitsystem("exportfs -ra")
        if not ret == 0:
            return False
        return True

    def backup_exports(self):
        # This backup is original before we started doing anything.  Preserve the original /etc/exports and don't touch
        # ever again.  This is the export before we changed anything.
        if os.path.exists(self.exports_path):
            if not os.path.exists(cvmanager_defines.EXPORTS_BACKUP):
                # Only create the backup if one does not exist.
                shutil.copy(self.exports_path, cvmanager_defines.EXPORTS_BACKUP)

    @mutex_lock
    def init_exports(self, clean_exports, locked=False):
        """Ensure that the exports file is present and loaded.
        Will create the file if it does not exist, or load everything in it.

        We ONLY care about the export that cvmanager will be using.

        With export files, the space is critical.
        https://access.redhat.com/documentation/en-us/red_hat_enterprise_linux/5/html/deployment_guide/s1-nfs-server-config-exports
        The /etc/exports file controls which file systems are exported to remote hosts and specifies options. Blank
        lines are ignored, comments can be made by starting a line with the hash mark (#), and long lines can be
        wrapped with a backslash (\). Each exported file system should be on its own individual line, and any lists of
        authorized hosts placed after an exported file system must be separated by space characters. Options for each
        of the hosts must be placed in parentheses directly after the host identifier, without any spaces separating
        the host and the first parenthesis.

        <export> <host1>(<options>) <hostN>(<options>)...

        Returns:

        """
        # Create a backup of exports so we don't destroy it.  Never alter the backup after the inital time.
        # Only create a new backup if someone deletes the .bak file.
        self.backup_exports()

        if os.path.exists(self.exports_path):
            with open(self.exports_path, 'r') as exports:
                for line in exports.readlines(): #non_blank_lines(exports):
                    if line.startswith('{0} '.format(self.exported_fs)):
                        # The file system we're exporting is already included in /etc/exports.
                        if not clean_exports:
                            # Load up the existing exports because we are not cleaning it.
                            self.exports[self.exported_fs] = Export(self.exported_fs, line)
                    else:
                        # This is not what we're exporting, just make sure it exists or exportfs command will fail
                        # because of a missing path.
                        line = line.rstrip()
                        if line and line.startswith('#'):
                            self.exports[line] = ExportComment(line)
                            continue

                        export_line = line.split(' ')
                        if os.path.exists(export_line[0]):
                            self.exports[export_line[0]] = Export(export_line[0], line)

    @mutex_lock
    def add_host_to_export(self, host, exported_fs=None):
        """Adds a specified hostname to an exported file system.

        Args:
            host (str): The hostname of the server this filesystem will be exported to.
            exported_fs (str): The filesystem to export
        Returns:  True if successful.

        """
        if exported_fs is None:
            exported_fs = self.exported_fs

        write_file = False

        # Every time we add a host, we shuold re-init, the exports, in case any other task(s) modified it.
        # Use the initialized clean
        self.init_exports(self.clean_exports, locked=True)

        if self.exports.get(exported_fs, False):
            # This export exists, so get it and add this node if missing.
            if host not in self.exports[self.exported_fs].nodes.keys():
                self.exports[self.exported_fs].add_node(host)
                write_file = True
            else:
                # This host is already specified, do nothing
                pass
        else:
            # This is the first host for this exported file system; so lets add it. OR we aren't cleaning it, overwrite
            new_export = Export(self.exported_fs)
            new_export.add_node(host)
            self.exports[self.exported_fs] = new_export
            write_file = True

        if write_file:
            self.__write_exports_file()
            return self.export_without_restart(locked=True)

        return True

    def remove_all_hosts_from_export(self, exported_fs=None):
        # Only call this 1 time per process.
        if exported_fs is None:
            exported_fs = self.exported_fs

        write_file = False

        # init the current exports but without clean option; meaning read it all.
        self.init_exports(False)
        if self.exports.get(exported_fs, False):
            # This export exists, so get it and remove this host
            for host in self.exports[self.exported_fs].nodes.keys():
                self.exports[self.exported_fs].remove_node(host)
                write_file = True
        else:
            # This export doesn't exist.  Just ignore it.
            LOG.debug("Export [{0}] not found in /etc/exports.".format(exported_fs))

        # If there are no more nodes for this export, blow it away.
        if self.exports.has_key(self.exported_fs):
            if len(self.exports[self.exported_fs].nodes) == 0:
                self.exports.pop(self.exported_fs)

        if write_file:
            self.__write_exports_file()

        self.export_without_restart(locked=False)

    def remove_host_from_export(self, host, exported_fs=None):
        if exported_fs is None:
            exported_fs = self.exported_fs

        write_file = False
        if self.exports.get(exported_fs, False):
            # This export exists, so get it and remove this host
            if host in self.exports[self.exported_fs].nodes.keys():
                self.exports[self.exported_fs].remove_node(host)
                write_file = True
            else:
                # This host is not there. do nothing.
                pass
        else:
            # This export doesn't exist.  Just ignore it.
            pass

        if write_file:
            self.__write_exports_file()

    def __write_exports_file(self):
        # Do not let anyone write directly into the file.  They should use the add\remove methods only
        with open(self.exports_path, "w") as f:
            for export in self.exports.values():
                f.write(str(export))


class Export(object):
    def __repr__(self):
        # Handle writing the line output format.  This formats it for writing.
        export_line = self.fs + ' '
        for node, options in self.nodes.items():
            export_line += '{0}{1} '.format(node, options)
        export_line += '\n'
        return export_line

    def __init__(self, exported_fs, line=''):
        self.fs = exported_fs
        self.nodes = {}
        for node in line.replace(exported_fs, '').split():
            opt_match = re.search(r'\((.*?)\)', node)
            if opt_match:
                node_options = opt_match.group()
            else:
                node_options = ''
            self.nodes[node.replace(node_options, '')] = node_options

    def add_node(self, host):
        self.nodes[host] = '(rw,no_root_squash,sync,insecure)'

    def remove_node(self, host):
        self.nodes.pop(host)


class ExportComment(object):
    def __repr__(self):
        line = self.comment
        line += '\n'
        return line

    def __init__(self, comment):
        self.comment = comment


def non_blank_lines(f):
    for l in f:
        line = l.rstrip()
        if line and not line.startswith('#'):
            yield line
