# Python imports
import os
import inspect
from enum import Enum
import json
import tempfile
import ConfigParser
import xml.etree.ElementTree as ET
import errno

# Project imports
import cvmanager_defines
import cvmanager_logging

LOG = cvmanager_logging.get_log(cvmanager_defines.LOG_FILE_NAME)


class FileType(Enum):
    DEFAULT = 0
    JSON = 1
    TEMPORARY = 2
    MULTILINE = 3
    SCRIPT = 4
    CONFIG = 5


class CatalogFile(object):
    def __init__(self, file_path):
        self.file_path = file_path
        self.f = None
        self.lines = []
        self.file_type = FileType.DEFAULT
        self.name = None
        self.clean_write = False

    def __str__(self):
        return self.file_path

    def __enter__(self):
        mode = 'r+'
        if self.clean_write:
            mode = 'w+'

        if self.f is None or self.f.closed:
            self.f = open(self.file_path, mode)
        return self.f

    def __exit__(self, exc_type, exc_value, traceback):
        if not self.f.closed:
            self.f.close()

    def read_file(self):
        # Attempt to figure out the type.
        with self as in_file:
            self.lines = in_file.readlines()

    def write(self, file_data, file_type, new_file=False):
        # If this is a new_file, blow it away.
        if new_file:
            self.clean_write = True

        if file_type == FileType.JSON:
            with self as out_file:
                json.dump(file_data, out_file, indent=4)
        elif file_type == FileType.MULTILINE or file_type == FileType.SCRIPT:
            with self as out_file:
                out_file.writelines(file_data)
        elif file_type == FileType.CONFIG:
            self.write_config_file(file_data)
        else:
            with self as out_file:
                out_file.write(file_data)

    def write_config_file(self, config):
        with self as out_file:
            config.write(out_file)

    def serialize_json(self):
        with self as out_file:
            return json.load(out_file)

    def read_config_file(self, **kwargs):
        with self as in_file:
            config_file = ConfigParser.ConfigParser(**kwargs)
            config_file.readfp(in_file)
            return config_file

    def parse_xml(self):
        with self as in_file:
            root = ET.parse(in_file)
            return root


class Catalog(object):
    """ The Catalog object allows Task() objects to crete and store files long term.  One such example is the
    Upgrade(Task) creates a history file, that's referenced every time the Upgrade(Task) is run.  Internally that
    task will read the file and determine if nodes need to upgrade.
    """

    def __init__(self, task_name=None):
        # Get the name of the class which initialized this.  It'll be the working dir.  If this didn't come from a
        # Task() directly, then it will be under default

        if task_name is None:
            self.name = str(inspect.stack()[1][0].f_locals.get('self', 'default'))
        else:
            self.name = task_name

        self.catalog_dir = os.path.join(cvmanager_defines.CATALOG_DIR, 'task_storage', self.name)
        self.file_lines = []

        self.__init()

    def __init(self):
        self.__create_working_dir()

    def __create_working_dir(self):
        # because many threads can call this.  handle thread race condition.
        if not os.path.exists(self.catalog_dir):
            while True:
                try:
                    os.makedirs(self.catalog_dir)
                    break
                except OSError, ose:
                    if not ose.errno == errno.EEXIST:
                        raise
                    pass

    def read_file(self, file_path):
        c_file = CatalogFile(file_path)
        c_file.read_file()

        return c_file

    def get_temp_file(self, **kwargs):
        # Create the temp file and force it to the catalog dir.
        kwargs['dir'] = self.catalog_dir
        fd, path = tempfile.mkstemp(**kwargs)

        temp_file = self.get_file(os.path.basename(path))
        temp_file.file_type = FileType.TEMPORARY

        return temp_file

    def get_script_file(self, prefix, suffix, name=None):
        temp_file = None
        while True:
            if name is None:
                name = next(tempfile._get_candidate_names())
            temp_file = os.path.join(self.catalog_dir, '{0}_{1}_{2}'.format(prefix, name, suffix))
            if not os.path.exists(temp_file):
                break
        else:
            raise Exception("Could not get a temporary script file!")

        temp_file = self.get_file(os.path.basename(temp_file))
        temp_file.file_type = FileType.SCRIPT
        temp_file.name = name

        return temp_file

    def get_file(self, file_name, new_file=False):
        if os.path.exists(file_name):
            raise Exception("Only provide the file name, don't provide a file path!  Catalog() handles the paths.")

        file_path = os.path.abspath(os.path.join(self.catalog_dir, file_name))
        if os.path.exists(file_path) and not new_file:
            return self.read_file(file_path)
        else:
            return self.create_file(file_path, new_file)

    def create_file(self, file_path, new_file=False):
        if not os.path.exists(file_path) or new_file:
            open(file_path, 'w').close()

        return self.read_file(file_path)
