# python imports
import argparse
import os
import sys
import ast

# local imports
import cvmanager_defines as define
import cvmanager_logging
import cvmanager_utils
import cvmanager_catalog
import cvmanager_task_arg


class Arguments(dict):
    def __init__(self):
        self.command_line_kwargs = {}
        self.parse_arguments()

    def remove_input_file(self, system_generated_only=True):
        # Get the path first.
        file_path = self.get_input_path()
        if file_path:
            if isinstance(self.yaml, cvmanager_catalog.CatalogFile):
                # Always clean up catalog files, they're code generated for single run use.
                os.unlink(file_path)
            elif not system_generated_only:
                # Clean up all files, not only system generated (catalog)
                os.unlink(file_path)

    def set_input_path(self, path):
        # Sets in put path to new location; say its been backed up and we want to use backup location.
        self.yaml = path

    def get_input_path(self):
        if isinstance(self.yaml, file):
            file_path = self.yaml.name
        elif isinstance(self.yaml, cvmanager_catalog.CatalogFile):
            file_path = self.yaml.file_path
        else:
            file_path = self.yaml

        file_path = os.path.abspath(file_path)
        if os.path.exists(file_path):
            return file_path
        else:
            return False

    def parse_arguments(self):
        log = cvmanager_logging.get_log()

        parser = argparse.ArgumentParser(
            description=define.MAIN_PROGRAM_NAME,
            formatter_class=argparse.RawTextHelpFormatter,
            add_help=False
        )

        parser.add_argument(
            '--restart',
            '-so',
            action='store_true',
            help='If specified, ignore any previous task executions and start over from scratch.'
        )

        parser.add_argument(
            '--status',
            '-s',
            action='store_true',
            help='Print the status of all currently running tasks.'
        )

        # This must be the full path to status file.
        parser.add_argument(
            '--status_file',
            '-sf',
            help='Display pretty status of the specified status file.'
        )

        # cvmanager.py -cs /opt/commvault/MediaAgent/task_manager/TEMPtask_input.yaml
        parser.add_argument(
            '--check_status',
            '-cs',
            default=None,
            help='Find the most recent status file for this task input yaml file.'
        )

        task_group = parser.add_argument_group('Task Group')

        task_group.add_argument(
            '--task',
            '-t',
            help='Specify the task to run on the command line.  This must include any arguments.'
        )

        task_group.add_argument(
            '--help',
            '-h',
            help='Display all defined optional inputs to this task.',
            action='store_true'
        )

        # THis allows us to just give a yaml file path on the command line for [arch manage commvault execute]
        if len(sys.argv) > 1:
            # cvmanager.py /path/to/input.yaml is this mode.
            if os.path.exists(sys.argv[1]):
                yaml_input = sys.argv[1]
            else:
                yaml_input = None
        else:
            yaml_input = None

        if yaml_input is None:
            # First check the /tmp location for any 'cvmanager_*.yaml' files and consume JUST the latest.  If we
            # find a file there, it will be used.  This is so we can simply run "#> cvmanager.py" with no inputs\options
            # Usefull for automation, copy a yaml to any machine /tmp, and launch cvmanager.py.  Never have to change
            # the caller
            file_pattern = define.YAML_INPUT_FORMAT
            latest_file = cvmanager_utils.get_latest_file_by_pattern(define.TaskDir.tmp, file_pattern)
            if latest_file is not None:
                log.info("Detected input yaml file [{0}].".format(latest_file))
                in_file = latest_file
            else:
                if os.path.exists(os.path.abspath('task_input.yaml')):
                    in_file = 'task_input.yaml'
                else:
                    # This could be some non-yaml option.  We'll fail later if its not.
                    log.debug("No YAML files found to process.")
                    in_file = None

            yaml_input = in_file

        parser.add_argument(
            '--yaml',
            '-y',
            help='The YAML input task file.',
            metavar="FILE",
            default=yaml_input,         # Kind of hacky but we have to do this for arch command to honor it.
            type=argparse.FileType('r')
        )

        parser.add_argument(
            '--output',
            '-o',
            default=define.TASK_OUTPUT_FILE,
            help='The operation output file.'
        )

        parser.add_argument(
            '--skip_reboot',
            '-sr',
            action='store_true',
            help='If specified, skip the required reboot and run the task normally.'
        )

        args, unknown = parser.parse_known_args(namespace=self)

        # The unknown args could be yaml inputs, so pass them through only if they have an equal sign
        for yaml_arg in unknown:
            if "=" not in yaml_arg:
                continue

            spilt_arg = yaml_arg.split("=")
            if not len(spilt_arg) == 2:
                continue

            try:
                val = ast.literal_eval(spilt_arg[1])
            except Exception, err:
                # This could mean we're dealing with a path /ws/ddb which can't eval...so just pass-through as string.
                log.debug("Unable to evaluate command line input, going with direct copy.")
                val = spilt_arg[1]
            self.command_line_kwargs[spilt_arg[0]] = val

        if self.help:
            if self.task is not None:
                cvmanager_task_arg.display_task_help(self.task)
            else:
                parser.print_help()
            sys.exit(1)

        return
