#! /usr/bin/env python
import sys
import Queue
import os
from threading import Lock

sys.path.insert(0, os.path.dirname(os.path.abspath(os.path.join(__file__, os.pardir))))

# Project imports
import cvmanager_logging
import cvmanager_defines as define
import common
import cvmanager_arg
import cvmanager_queue_population as populate
import cvmanager_queue_processor as process
import cvmanager_task_status
import cvmanager_utils
import task_defines
import cvupgrade_common
import cvmanager_workflow

SKIP_INIT = False
DEFAULT_EXIT_CODE = 999
WORKFLOW_INIT_FAILED = 899


class Manager(object):
    def __init__(self):
        # Upgrade manager should be able to init without a real cluster existing....for future expansion.
        log_dir = common.getregistryentry(define.REG_EVENT_KEY, "dEVLOGDIR")

        self.log = cvmanager_logging.get_new_log(define.LOG_FILE_NAME, log_dir, define.LOG_VERBOSE)  # True debug log
        self.log.info("Initializing HyperScale Task Manager.")
        self.log.info("Progress will be logged at: {0}".format(log_dir))
        self._quit_flag = False
        self.mutex = Lock()
        self.resumed = False

        # In order to prevent reboot loop, clear this registry key as early as possible.
        reg_key = define.REG_REBOOT_VALUE
        if common.getregistryentry(define.REG_MA_KEY, reg_key, False):
            self.resumed = True
            self.log.info("This task is being resumed after reboot; clearing registry key [{0}].".format(reg_key))
            common.deleteregistryentry(define.REG_MA_KEY, reg_key)

        # Load and handle the command line arguments
        self.args = cvmanager_arg.Arguments()

        # Setup some basic properties
        self.__queue = Queue.PriorityQueue(define.QUEUE_SIZE)

        # The default exit code.; set it in registry too.
        self._exit_code = DEFAULT_EXIT_CODE

        # If this is in workflow mode, init the workflow
        # Workflow jobs can be specified on the command line OR input yaml.
        self.workflow_list = cvmanager_workflow.workflow_factory(self.args.yaml, self.args.command_line_kwargs)
        for workflow in self.workflow_list:
            workflow.write_job_status(self.exit_code)

        # Set the installation directory; the project structure has task_manager under /opt/commvault/MediaAgent
        dynamic_defines = sys.modules[define.DYNAMIC_DEFINE_NAME]
        dynamic_defines.python_root_dir = os.path.dirname(os.path.abspath(os.path.join(__file__, os.pardir)))
        self.log.info("Setting python source dir: {0}".format(dynamic_defines.python_root_dir))

    @property
    def exit_code(self):
        return self._exit_code

    @exit_code.setter
    def exit_code(self, value):
        self._exit_code = value

    @property
    def quit_flag(self):
        return self._quit_flag

    @quit_flag.setter
    def quit_flag(self, value):
        self._quit_flag = value

    @property
    def queue(self):
        return self.__queue

    def run(self):
        if self.args.status:
            # just tail the status of any running tasks.
            try:
                cvmanager_task_status.TailStatus(by_task_name=self.args.task)
            except Exception, err:
                print(str(err))
            self.log.debug("TODO: No status files found; it appears like all Tasks have completed.")
            self.log.debug("TODO: Check the archive at [{0}] for historical statuses.".format(define.TaskDir.archive))

        elif self.args.status_file is not None:
            cvmanager_task_status.TailStatus(status_file=self.args.status_file)

        # elif self.args.build_yaml_from_task:
        #     cvmanager_yaml.build_yaml(self.args.task_name, self.args.output_file)
        #     self.log.info("Successfully wrote yaml file [{0}].".format(self.args.output_file))

        elif self.args.check_status is not None:
            cvmanager_task_status.LatestStatus(self.args.check_status, self)

        else:
            # The no option mode is to run the cvmanager.
            q_pop = populate.Populate(name='populator', kwargs={'manager': self})

            for t_id in range(define.WORKER_THREADS):
                q_proc = process.Process(name=t_id, kwargs={'manager': self})
                q_proc.start()

            self.log.info("Commvault HyperScale Task Manager Initialized.")
            q_pop.start()
            q_pop.join()

    def join(self):
        self.queue.join()


def setup_pid():
    clean_pid()
    if not os.path.exists(define.PID_FILE):
        open(define.PID_FILE, 'w').close()


def clean_pid():
    try:
        for pid_file in cvmanager_utils.get_all_file_by_pattern(define.ROOT_FILE_LOC, 'cvmanager_*.pid'):
            os.unlink(pid_file)
    except Exception as err:
        print(str(err))


def adjust_permissions():
    """
    Because some setups can have varying permissions or umasks, we need to ensure we can execute what we need to.
    Therefore, set the umask for this python process, and adjust anything in cvmanager that we might need to be
    readable.
    :return:
    """
    # Change the catalog to our required permissions
    cvmanager_utils.adjust_directory_permissions(recursive=True)

    # Set umask for this process execution; this should handle new files\dir going forward for current process.
    os.umask(define.DEFAULT_UMASK)


if __name__ == "__main__":
    exit_code = DEFAULT_EXIT_CODE

    try:
        # Setup the process ID file for this cvmanager.py instance.
        setup_pid()

        # Ensure the catalog is setup and has correct permissions before starting
        # We've seen some issues during Service Pack upgrade where directories get modified and begin to fail.
        adjust_permissions()

        # Start the task manager; only 1 task manager will run per block!
        manager = Manager()

        # If not resumed, this is first run of the task, so initialize any and all workflow(s)
        if not manager.resumed:
            for workflow in manager.workflow_list:
                if manager.args.command_line_kwargs.get('skip_workflow_init', False):
                    manager.log.warning("Workflow task will NOT suspend the job because flag [skip_workflow_init]="
                                        "True.")
                    continue

                if not workflow.initialize():
                    exit_code = WORKFLOW_INIT_FAILED
                    raise Exception("Failed to initialize workflow control.")
        else:
            # This is a resumed task, if its a workflow just mark it initialized.
            for workflow in manager.workflow_list:
                manager.log.info("Resuming a workflow based task.")
                workflow.initialized = True

        manager.run()

        manager.join()

        exit_code = manager.exit_code

        manager.log.info("Commvault HyperScale Task Manager Exiting.")
        manager.log.info("Process Exit Code [{0}].".format(exit_code))

    except Exception as err:
        print (str(err))
    finally:
        # These are STRICTLY teardown functions; don't care if they fail
        try:
            clean_pid()
        except Exception:
            pass

        # Send the status back to the suspended workflow job; but only if this isn't a reboot scenario.
        try:
            for workflow in manager.workflow_list:
                if workflow.initialized and not exit_code == task_defines.ProcessStatusCode.RESUME_AFTER_REBOOT.value:
                    # This is a workflow job that's not rebooting.
                    # Write the status BEFORE resuming the workflow because it will immediately try to read it.
                    try:
                        manager.log.info("Writing job status file.")
                        workflow.write_job_status(exit_code)
                    except Exception:
                        pass

                    try:
                        if not workflow.tear_down():
                            manager.log.error("Failed to tear down the workflow, job will remain suspended. "
                                              "It's safe to kill.")
                    except Exception:
                        pass
        except Exception:
            pass

        # Reboot the system if required.
        if exit_code == task_defines.ProcessStatusCode.RESUME_AFTER_REBOOT.value:
            # Reboot the system!
            cvupgrade_common.reboot_node()

        sys.exit(exit_code)
