# Python imports
import sys
import uuid
import os

# Project imports
from task_manager import cvmanager_defines as define
import cvcluster


# noinspection DuplicatedCode
class VolumeFile(object):
    def __init__(self, vol_name):
        # It's important to keep track of the bottom, since it's read from bottom up; hence, self.last
        self.name = vol_name
        self.path = define.VOLUME_FILE_PATH.format(vol_name)

        if not os.path.exists(self.path):
            raise IOError("Volume file [{1}] not found for specified storage pool [{0}]. Please ensure the gluster is "
                          "properly configured and the storage pool name is correct.  If not, please re-run using the "
                          "[StoragePoolName] option on command line or input file.".format(vol_name, self.path))

        # self.graph is the FULL graph, do not modify.  self.last is the bottom translator
        self.graph, self.last = self.load()

    def load(self):
        # When the file is loaded, its loaded from top down, so a parent will ALWAYS exist to any child.
        try:
            lines = []
            with open(self.path, 'r') as fp:
                lines = fp.readlines()

            all_translators = {}
            translator = None
            last_translator = None

            for line in lines:
                text = line.split()
                if not len(text):
                    continue

                translator_key = text[0]

                if translator_key == define.TK_VOLUME:
                    if translator:
                        raise RuntimeError("Nested volume.")
                    translator = Translator(text[1])
                    continue
                if not translator:
                    raise RuntimeError("Text outside volume.")
                if translator_key == define.TK_TYPE:
                    translator.type = text[1]
                    continue
                if translator_key == define.TK_OPTION:
                    translator.options[text[1]] = ''.join(text[2:])
                    continue
                if translator_key == define.TK_SUBVOLUMES:
                    for sv in text[1:]:
                        translator.subvolumes[all_translators[sv].name] = all_translators[sv]
                    continue
                if translator_key == define.TK_END_VOL:
                    # Done reading the translator information
                    all_translators[translator.name] = translator
                    last_translator = translator
                    translator = None
                    continue
                raise RuntimeError("Unrecognized keyword {0}".format(translator_key))

            if translator:
                raise RuntimeError("Unclosed volume definition.")

            return all_translators, last_translator
        except Exception, err:
            raise err

    def print_y(self, root_graph=None, last_translator=None, stream=sys.stdout):
        # For printing the output
        if root_graph is None:
            root_graph = self.graph

        if last_translator is None:
            last_translator = self.last

        # Print the graph out to console.
        for uid, sv in last_translator.subvolumes.items():
            if not sv.dumped:
                self.print_y(root_graph, sv, stream)
                print("")
                sv.dumped = True
        print("volume %s" % last_translator.name)
        print("    type %s" % last_translator.type)
        for k, v in last_translator.options.items():
            print("    option %s %s" % (k, v))
        if last_translator.subvolumes.items():
            print("    subvolumes %s" % ''.join(
                [sv.name for uid, sv in last_translator.subvolumes.items()]))
        print("end-volume")


class Translator(object):
    def __init__(self, name):
        self.name = name
        self.type = ""
        self.options = {}
        self.subvolumes = {}
        self.dumped = False  # Used for printing output
        self.uuid = uuid.uuid4()  # Get a unique ID of this translator.

    def __repr__(self):
        return "<Translator %s>" % self.name


def get_cluster_volume_graph(sp_name=None):
    if sp_name is None:
        sp_name = cvcluster.get_storagepool_volname()

    if sp_name == "":
        # There is no storage pool, consider this un-configured upgrade.
        # TODO: Figure out how to collect the remote nodes....avahi?  bonjour?
        return None
    return VolumeFile(sp_name)


if __name__ == "__main__":
    vf = VolumeFile(sys.argv[1])
    graph, last = vf.load()
    vf.print_y(graph, last)
