
from threading import Lock
from subprocess import Popen, PIPE
from timeit import default_timer as timer
import time
import sys
import os
import re
import psutil
import xattr

from task_manager import cvmanager_defines
from task_manager import cvmanager_logging
from task_manager import cvmanager_catalog
import common
import wrappers
import utils
import cvnwconfigmgr
import hs_node
import cvupgrade_common

LOG = cvmanager_logging.get_log(cvmanager_defines.LOG_FILE_NAME)
MUTEX = Lock()
NODE_NW_JSON = None


class HiddenPrints(object):
    def __enter__(self):
        self._original_stdout = sys.stdout
        sys.stdout = open(os.devnull, 'w')

    def __exit__(self, exc_type, exc_val, exc_tb):
        sys.stdout.close()
        sys.stdout = self._original_stdout


def mutex_lock(f):
    global MUTEX

    def inner(*args, **kwargs):
        if not kwargs.get('locked', False):
            # locked not present assume we must take the lock.
            with MUTEX:
                return f(*args, **kwargs)
        else:
            # Ok to proceed without locking, because we already have it and know about it.
            return f(*args, **kwargs)
    return inner


def get_last_sw_cache_sync_time():
    last_sw_cache_sync_time = 0
    value = common.getregistryentry(cvmanager_defines.REG_MA_KEY, "nSWCacheLastSyncTime")
    if value:
        last_sw_cache_sync_time = long(value)

    LOG.debug("nSWCacheLastSyncTime: [{0}]".format(last_sw_cache_sync_time))
    return last_sw_cache_sync_time


def get_block_device_info_from_xml(xml_file):
    """
    Parse the hardware xml file and extract the number of block devices of each type.
    For this discovery, we only care about data drives.  We can ignore SSD os drives or metadata drives.
    """
    # Get all the mount point info.
    mount_points = hs_node.get_block_device_mount_paths_from_xml(xml_file)

    disk_map = {}
    for disk in xml_file.findall('Disk'):
        if not disk.get('cvdrive') == 'data':
            continue
        mount_point_for_device = mount_points.get(disk.get('devname'))
        disk.attrib['MOUNT_POINT'] = mount_point_for_device
        disk_map[disk.get('devalias')] = disk.attrib
    return disk_map


@mutex_lock
def browse_avahi_nodes(config_file_path=None, start_service=True, locked=False):
    # Execute avahi_browse_and_print.py and collect output in temporary config file
    try:
        # Write the node information into the catalog, so that workflow and future tasks can read from it.
        if config_file_path is None:
            task_catalog = cvmanager_catalog.Catalog('avahi')
            config_file_path = task_catalog.get_file("discovered_nodes.config", True)
            # LOG.info("Saving discovered avahi nodes to: {0}".format(config_file_path.file_path))

        if start_service:
            wrappers.start_service('avahi-daemon')

        command = "/var/www/my_flask_prod/scripts/avahi_browse_and_print.py {0}".format(config_file_path)
        process = Popen(command, stdout=PIPE, stderr=PIPE, shell=True)
        output, error = process.communicate()

        if not process.returncode == 0:
            LOG.error("Failed to browse avahi nodes.\n{0}\n{1}".format(output, error))

        return process.returncode, config_file_path
    except Exception, err:
        return -1, None
    finally:
        if start_service:
            wrappers.stop_service('avahi-daemon')


def init_logging():
    # cvnwconfigmgr needs special logging setup.
    nw_logger = utils.Logger("cvnwconfigmgr")
    nw_logger.logger_init()

    cvnwconfigmgr.Logger = nw_logger


@mutex_lock
def wait_for_network_config(remote_node, timeout_in_seconds=300):
    start = timer()
    end = 0
    attempt = 0

    while (end-start) < timeout_in_seconds:
        attempt += 1
        LOG.info("Checking if network for avahi node {0} is configured. Attempt #{1}".format(remote_node, attempt))

        # Let's browse all avahi nodes; since many threads can be accessing this, do not stop\start services.
        ret, available_node_info = browse_avahi_nodes(start_service=False, locked=True)
        if not ret == 0:
            LOG.error("Failed to browse avahi nodes, please make sure avahi-daemon is running on this machine.")
            return False

        # Read the browsed nodes.
        avahi_nodes = available_node_info.read_config_file()
        if not avahi_nodes.has_section(remote_node):
            # Didn't find this node in the browse at all.
            # Sleep and try after 5 seconds
            time.sleep(5)
            continue

        # We found this node; check it's status.
        node_status = avahi_nodes.get(remote_node, 'status')
        if node_status == "NwConfigured":
            LOG.info("Avahi status for node " + remote_node + " is now changed to Nwconfigured.")
            return True

        elif node_status == "NwConfigFailed":
            LOG.error("Network configuration failed for avahi node id " + remote_node)
            return False

        # The node has a section in the avahi file but its not ==NwConfigured.  Loop again.
        time.sleep(5)

        # Set the current time.
        end = timer()
    else:
        if (end-start) > timeout_in_seconds:
            LOG.debug('Timed out waiting for network configuration to complete on node [{0}].'.format(remote_node))
            return False

    return True


def get_cs_host_name():
    cs_host_name = None
    process = utils.Process()
    if not process.execute("commvault status | grep 'CommServe Host Name' | cut -d= -f2-") == 0:
        LOG.error("Unable to get CommServe Host Name for registration.")
        return False

    cs_host_name = process.getout().strip()
    return cs_host_name


def ip_in_use(ip_address):
    # This check is bad; getting a reply means in use, it can also mean the node using it is just powered off.
    cmd = "sudo ping -c 1 {0}".format(ip_address)
    task = utils.Process()
    task.execute(cmd)
    if task.failed == 1:
        LOG.debug("STDOUT: {0}\nSTDERR: {1}".format(task.strout, task.strerr))
        return False
    return True


def validate_network_not_in_use(node, node_info):
    """
    Ensures both addresses with included in the user input for each node.
    Ensures that neither network address are already in use.

    :param node: str - Name of the remote node.
    :param node_info: dict - User input of IP addresses for storage pool and data protection
    { 'data_protection_ip' : 'ip', 'storage_pool_ip': 'ip'}
    :return: bool - True: networks not in use, False: 1 or many networks in use.
    """
    valid = True
    if not node_info.get('data_protection_ip', False) or not node_info.get('storage_pool_ip', False):
        LOG.error("Please ensure all required inputs are provided.")
        return False

    # This information dictionary is collected during discovery.  You can read the Catalog() file generated at:
    # /ws/ddb/cvmanager/catalog/task_storage/Discover_Nodes/Available_Nodes.json
    avahi_ip = node_info.get('AVAHI', {}).get('ip', '')
    node_ips_from_arch = node_info.get('ARCH', {}).get('IP_ADDRESS_INFO', {}).values()

    if ip_in_use(node_info['data_protection_ip']):
        if node_info['data_protection_ip'] == avahi_ip:
            # If the data_protection_ip == the avahi IP, it could already be set.  Do not fail.
            LOG.info("Data protection IP [{0}] for node[{1}] is in use by avahi IP on the same node. "
                     "OK to proceed.".format(node, node_info['data_protection_ip']))
        else:
            LOG.error("Data protection IP [{0}] for node[{1}] is already in use!".format(
                node, node_info['data_protection_ip']))
            valid = False

    if ip_in_use(node_info['storage_pool_ip']):
        if node_info['storage_pool_ip'] in node_ips_from_arch:
            # This IP is in use, but its by the remote node itself.  OK to proceed.
            LOG.info("Storage pool IP [{0}] for node[{1}] is already set on the node.  OK to proceed.".format(
                node, node_info['storage_pool_ip']))

        else:
            LOG.error("Storage pool IP [{0}] for node[{1}] is already in use!".format(
                node, node_info['storage_pool_ip']))
            valid = False

    return valid


def validate_network_for_node(node, node_info, backend_json=NODE_NW_JSON):
    """
    Validates that network information for [node] is valid.
    :param backend_json:
    :param node:
    :param node_info:
    :return:
    """
    # Load this nodes network info to memory.  DO NOT re-read over and over.
    global NODE_NW_JSON
    if backend_json is None:
        local_node = hs_node.HyperScaleNode()
        NODE_NW_JSON = local_node.get_node_network_json()
    else:
        NODE_NW_JSON = backend_json

    # Get the current local node subnet mask for data protection and storage pool
    data_protection_netmask = None
    storage_pool_netmask = None
    local_data_protection_ip = None
    local_storage_pool_ip = None

    interfaces = NODE_NW_JSON['interfaces']
    for interface in interfaces:
        if interface.get('hyperscale_nwtype', False) == 'commServeRegistration':
            data_protection_netmask = interface.get('netmask', None)
            local_data_protection_ip = interface.get('ipaddr', None)
        elif interface.get('hyperscale_nwtype', False) == 'storagePool':
            storage_pool_netmask = interface.get('netmask', None)
            local_storage_pool_ip = interface.get('ipaddr', None)

    if not validate_network_not_in_use(node, node_info):
        # Simple check; check the catalog if we already have a network config json for this node.
        # If we do, it means we already built the json, and configured the node.
        return False

    remote_data_protection_ip = node_info['data_protection_ip']
    remote_storage_pool_ip = node_info['storage_pool_ip']

    # Make sure that DP & SP IP addresses are on different subnets.
    if common.compare_ipv4_networks(remote_data_protection_ip, data_protection_netmask,
                                    remote_storage_pool_ip, storage_pool_netmask):
        LOG.error("Data protection IP [{0}] and storage pool IP [{1}] are on the same subnet.".format(
            remote_data_protection_ip, remote_storage_pool_ip
        ))
        return False

    # Make sure that the DP IP for existing Node, is on same subnet as DP for new remote node.
    if not common.compare_ipv4_networks(local_data_protection_ip, data_protection_netmask,
                                        remote_data_protection_ip, data_protection_netmask):
        LOG.error("Local node data protection IP [{0}] and remote node data protection IP [{1}] are on "
                       "different subnets.".format(local_data_protection_ip, remote_data_protection_ip))
        return False

    # Make sure that the SP IP for existing Node, is on same subnet as SP for new node.
    if not common.compare_ipv4_networks(local_storage_pool_ip, storage_pool_netmask,
                                        remote_storage_pool_ip, storage_pool_netmask):
        LOG.error("Local node storage pool IP [{0}] and remote node storage pool IP [{1}] are on "
                       "different subnets.".format(local_storage_pool_ip, remote_storage_pool_ip))
        return False

    LOG.info("Successfully validated node [{0}] network settings.".format(node))
    return True


def run_it(cmd):
    process = Popen(cmd, stdout=PIPE, stderr=PIPE, shell=True)
    output, error = process.communicate()
    if not process.returncode == 0:
        LOG.error("Failed running: {0}".format(cmd))
        LOG.debug("Output: {0}\nError: {1}".format(output, error))

    return process.returncode


def get_etc_hostnames():
    """
    Parses /etc/hosts file and returns all the hostnames in a list.
    """
    with open('/etc/hosts', 'r') as f:
        hostlines = f.readlines()
    hostlines = [line.strip() for line in hostlines
                 if not line.startswith('#') and line.strip() != '']
    hosts = []
    for line in hostlines:
        hostnames = line.split('#')[0].split()[1:]
        hosts.extend(hostnames)
    return hosts


def valid_password(password, return_password_on_failure=True):
    """

    :param password: str - password to check for validity.
    :param return_password_on_failure: bool - if the password is not valid encrypted format, just return it.
    :return: str - The valid format of the password, or the same password if its in-valid.  If
    return_password_on_failure=False, then return boolean False.
    """

    if not password:
        return password

    try:
        decp = cvupgrade_common.decp(password)
        decp.decode('ascii')
        password = decp
    except UnicodeDecodeError, ue:
        if not return_password_on_failure:
            LOG.debug("Password is in-valid.")
            return False
    except Exception, err:
        if not return_password_on_failure:
            LOG.debug("Invalid password exception.")
            return False
        return password

    return password


def get_rpm_version(rpm_path, rpm_name):
    version = None
    process = utils.Process()
    command = r'find ' + rpm_path + ' -name ' + rpm_name + \
              r' -exec cat {} \; 2>/dev/null | rpm --queryformat "%{VERSION}" -qp -'
    if not process.execute(command) == 0:
        LOG.error("Failed getting version of rpm [{0}] in location [{1}].".format(rpm_name, rpm_path))
        return False

    version = process.getout().strip()
    return version


def check_ip(ip_address):
    regex = '''^(25[0-5]|2[0-4][0-9]|[0-1]?[0-9][0-9]?)\.( 
                25[0-5]|2[0-4][0-9]|[0-1]?[0-9][0-9]?)\.( 
                25[0-5]|2[0-4][0-9]|[0-1]?[0-9][0-9]?)\.( 
                25[0-5]|2[0-4][0-9]|[0-1]?[0-9][0-9]?)$'''

    if re.search(regex, ip_address):
        return True
    else:
        return False


def convert_size_to_bytes(size, unit):
    if unit == 'B':
        return float(size)
    elif unit == 'KB':
        return float(size) * float(1024)
    elif unit == 'MB':
        return float(size) * float(1024 ** 2)
    elif unit == 'GB':
        return float(size) * float(1024 ** 3)
    elif unit == 'TB':
        return float(size) * float(1024 ** 4)
    elif unit == 'PB':
        return float(size) * float(1024 ** 5)
    else:
        raise Exception("Unknown size unit [{0}].".format(unit))


def format_bytes(size, unit=None):
    # 2**10 = 1024
    power = 2**10
    n = 0
    power_labels = {0: 'B', 1: 'KB', 2: 'MB', 3: 'GB', 4: 'TB', 5: 'PB'}
    while size > power:
        size /= power
        n += 1
        if unit is not None:
            if n == power_labels.keys()[power_labels.values().index(unit)]:
                break
    return size, power_labels[n]


def get_xattr_mounts(attribute_name, xfs_only=True):
    """
    Finds all mount points which have extended attribute <attribute_name> set and returns that list of mounts.
    We only set xattr on xfs file systems.
    :param xfs_only: bool - By default Hyperscale only sets its xattr xfs mount points, so this function (by default)
    will only look for <attribute_name> on an xfs mount point.  If <xfs_only=False> then this will look for
    <attribute_name> on ALL mount points.
    :param attribute_name: str - Name of extended attribute to search for.
    :return: list - All mount points which have this xattr set.
    """
    list_of_mounts = []
    mount_points = psutil.disk_partitions(all=not xfs_only)
    for m in mount_points:
        if xfs_only:
            if not m.fstype == "xfs":
                continue
        try:
            xattr.getxattr(m.mountpoint, attribute_name)
            list_of_mounts.append(m.mountpoint)
        except IOError as ex:
            LOG.debug("It looks [{0}] is not set for path [{1}]".format(attribute_name, m.mountpoint))
            continue

    return list_of_mounts
