#! /usr/bin/env python

import traceback
import select
import os
from paramiko import SSHClient, SSHException, MissingHostKeyPolicy, SFTPClient, AutoAddPolicy
from contextlib import contextmanager
from functools import partial
import crypt
import sys
import re

import task_manager.cvmanager_logging
import task_manager.cvmanager_defines
import hs_defines
import common

LOG = task_manager.cvmanager_logging.get_log(task_manager.cvmanager_defines.LOG_FILE_NAME)
SSH_PATH = "~/.ssh/"
SSH_PRIV_KEY = 'id_rsa'
SSH_INPUT_FILE_PATH = SSH_PATH + SSH_PRIV_KEY
SSH_KNOWN_HOSTS_PATH = SSH_PATH + 'known_hosts'
SSH_AUTH_KEYS_PATH = SSH_PATH + 'authorized_keys'
SSH_PUB_FILE_PATH = SSH_INPUT_FILE_PATH + ".pub"
DYNAMIC_DEFINES = sys.modules[task_manager.cvmanager_defines.DYNAMIC_DEFINE_NAME]

# SSH debugging only.
# import logging
# logging.basicConfig()
# logging.getLogger("paramiko").setLevel(logging.DEBUG)  # for example


class MySFTPClient(SFTPClient):
    def put_dir(self, source, target):
        """ Uploads the contents of the source directory to the target path. The
            target directory needs to exists. All subdirectories in source are
            created under target.
        """
        for item in os.listdir(source):
            if os.path.isfile(os.path.join(source, item)):
                self.put(os.path.join(source, item), '%s/%s' % (target, item))
            else:
                self.mkdir('%s/%s' % (target, item), ignore_existing=True)
                self.put_dir(os.path.join(source, item), '%s/%s' % (target, item))

    def mkdir(self, path, mode=511, ignore_existing=False):
        """ Augments mkdir by adding an option to not fail if the folder exists  """
        try:
            super(MySFTPClient, self).mkdir(path, mode)
        except IOError, ioe:
            if ignore_existing:
                pass
            else:
                raise

    def mkdir_p(self, remote_directory):
        """Change to this directory, recursively making new folders if needed.
        Returns True if any folders were created."""
        if remote_directory == '/':
            # absolute path so change directory to root
            self.chdir('/')
            return
        if remote_directory == '':
            # top-level relative directory must exist
            return
        try:
            self.chdir(remote_directory)  # sub-directory exists
        except IOError:
            dirname, basename = os.path.split(remote_directory.rstrip('/'))
            self.mkdir_p(dirname)  # make parent directories
            self.mkdir(basename)  # sub-directory missing, so created it
            self.chdir(basename)
            return True


class _SSHconn(SSHClient):
    def __init__(self, remote_host, *args, **kwargs):
        """ Internal class for opening ssh connection to remote host using paramiko.  This is so people don't have to
        explicitly connect.  We can handle host keys or anything else we need to worry about, in here.

        we should not open or control ssh connection anywhere else in any code.

        :param remote_host: str - Remote hostname to open SSH connection to.
        """
        self.remote_hostname = remote_host
        self.args = args
        self.kwargs = kwargs

        # paramiko SSHClient() requires no variables for init.
        super(_SSHconn, self).__init__()

    def __enter__(self, retry_password_less=True):
        try:
            # Attempt to load configured host keys on this node; we should already have hostkeys for known nodes
            self.load_system_host_keys()
            self.set_missing_host_key_policy(MissingHostKeyPolicy())

            # Get the named_user_only flag.
            named_user_only = self.kwargs.pop('named_user_only', False)
            if named_user_only:
                try:
                    self.connect(hostname=self.remote_hostname, **self.kwargs)
                except SSHException, se:
                    if "not found in known_hosts" in se.message:
                        # Re-try connection with allowing missing host key; but do not auto add these keys.
                        # This issue should resolve once we upgrade paramiko.
                        self.set_missing_host_key_policy(MissingHostKeyPolicy())
                        try:
                            self.connect(hostname=self.remote_hostname, **self.kwargs)
                        except Exception, err:
                            raise
                    else:
                        raise
                return self

            if not retry_password_less:
                # Reset the passwords to None; to attempt password less SSH
                self.kwargs['username'] = None
                self.kwargs['password'] = None
                self.connect(hostname=self.remote_hostname, **self.kwargs)
            else:
                self.connect(hostname=self.remote_hostname, **self.kwargs)
            return self
        except SSHException, se:
            if "not found in known_hosts" in se.message:
                # Re-try connection with allowing missing host key; but do not auto add these keys.
                # This issue should resolve once we upgrade paramiko.
                self.set_missing_host_key_policy(MissingHostKeyPolicy())
                # self.set_missing_host_key_policy(AutoAddPolicy())
                try:
                    if not retry_password_less:
                        # Reset the passwords to None; to attempt password less SSH
                        self.kwargs['username'] = None
                        self.kwargs['password'] = None
                        self.connect(hostname=self.remote_hostname, **self.kwargs)
                    else:
                        self.connect(hostname=self.remote_hostname, **self.kwargs)
                except Exception, err:
                    if retry_password_less:
                        LOG.debug("Re-trying connection with passwordless SSH in case it's been setup already.")
                        return self.__enter__(False)
                    LOG.debug("Failed connecting to remote host [{0}] using ssh.  With error: {1}".format(
                        self.remote_hostname, err))
                    raise
                return self
            else:
                # Even reconnect failed
                LOG.debug("Failed connecting to remote host [{0}] using ssh.".format(self.remote_hostname))
                raise
        except Exception, err:
            if retry_password_less:
                LOG.debug("Re-trying connection with passwordless SSH in case it's been setup already.")
                return self.__enter__(False)
            LOG.debug("Failed connecting to remote host [{0}] using ssh.  With error: {1}".format(
                self.remote_hostname, err))
            raise
        return None

    def __exit__(self, type, value, traceback):
        # Close any active connections.
        if self.get_transport() is not None:
            self.get_transport().is_active()

            self.close()

    def persistent_connect(self):
        return self.__enter__()


@contextmanager
def _hs_ssh_conn(remote_node, *args, **kwargs):
    """ Internal API to open an SSH connection to a known node. The HyperScale install would have already added this
    node into the known_hosts file on the executing node.  Preferred method is to use RemoteSSH() object.

    :param remote_node: str - Hostname of remote node.
    :return: yields new _SSHconn with connection to the node.
    """
    with _SSHconn(remote_node, *args, **kwargs) as ssh:
        yield ssh


def _hs_ssh_persistent_conn(remote_node, *args, **kwargs):
    ssh = _SSHconn(remote_node, *args, **kwargs)
    ssh.persistent_connect()
    return ssh


class RemoteSSH(object):
    def __init__(self, remote_host, *args, **kwargs):
        """ Use this API to open an SSH connection to a known node. The HyperScale install would have already added this
        node into the known_hosts file on the executing node.  By using this class, all calls to on_remotenode, and
        execute_remote, would make an ssh connection, and close afterwards.

        Example:
        ssh_conn = RemoteSSH('remotenode.com')
        if not ssh_conn.on_remotenode('restart_glusterd') == 0:
            sys.exit(100)

        if not ssh_conn.execute_remote('ls -l /opt/commvault') == 0:
            sys.exit(100)

        ssh_conn.ssh_cli extends paramiko.SSHClient(), with all options available.

        :param remote_host: str - Hostname of remote node.
        :param kwargs: dict - Any possible argument needed in paramiko.

        """
        self.ssh_cli = partial(_hs_ssh_conn, remote_host, *args, **kwargs)
        self.persistent_cli = partial(_hs_ssh_persistent_conn, remote_host, *args, **kwargs)
        self.output = None
        self.error = None
        self.status = 999  # default status code.
        self.remote_host = remote_host

        # Maintain a list of all connections this instance spawns; identified by instance ID
        self.__instance_connections = {}

    @classmethod
    def is_ready(cls):
        if not common.getregistryentry(task_manager.cvmanager_defines.REG_MA_KEY, "sUsePasswordlessSSH").lower() == "true":
            LOG.error("Passwordless ssh is not available.")
            return False
        return True

    def test_connection(self, named_user_only=False):
        # When we're testing a connection, do not try different users (like root), only try the named user.
        if not RemoteSSH.is_ready():
            return False
        try:
            with self.ssh_cli(**{'named_user_only': named_user_only}) as ssh:
                pass
        except Exception, err:
            LOG.debug("Failed connecting using ssh; ssh not available to node [{0}].".format(self.remote_host))
            return False
        return True

    def geterr(self):
        return getattr(self, 'strerr', '')

    def getout(self):
        return getattr(self, 'strout', '')

    def getexitcode(self):
        return getattr(self, 'failed', 999)

    def on_remotenode(self, command, timeout=None, log_error=True):
        # All of these commands are for upgrade; using cvupgrade_node.py.
        command = "{1}/cvupgrade_node.py {0}".format(command, DYNAMIC_DEFINES.python_root_dir)
        return self.execute_remote(command, timeout=timeout, log_error=log_error)

    def close_connections(self):
        for conn in self.__instance_connections.values():
            conn.close()

    def get_remote_file(self, remote_file, local_file):
        if not RemoteSSH.is_ready():
            return False
        with self.ssh_cli() as ssh:
            # local_file MUST exist or this will do nothing.
            sftp_client = ssh.open_sftp()
            sftp_client.get(remote_file, local_file)
            sftp_client.close()

        return True

    def remove_remote_file(self, remote_file):
        if not RemoteSSH.is_ready():
            return False
        with self.ssh_cli() as ssh:
            # local_file MUST exist or this will do nothing.
            sftp_client = ssh.open_sftp()
            sftp_client.remove(remote_file)
            sftp_client.close()

        return True

    def get_system_host_keys(self, local_file):
        # Generate known_hosts file
        if not self.execute_remote("cat /etc/ssh/ssh_host_*.pub > /etc/ssh/.known_hosts") == 0:
            return False

        # Get this file, and remove the one we created.
        if not self.get_remote_file("/etc/ssh/.known_hosts", local_file):
            return False

        if not self.remove_remote_file("/etc/ssh/.known_hosts"):
            LOG.debug("Failed to clean generated host key file.")

        return True

    def copy_dir_to_remote(self, local_dir, remote_dir, mode=None, ignore_existing=False):
        if not RemoteSSH.is_ready():
            return False
        with self.ssh_cli() as ssh:
            sftp_client = MySFTPClient.from_transport(ssh._transport)
            # sftp_client.mkdir(remote_dir, ignore_existing=True)
            sftp_client.mkdir_p(remote_dir)
            sftp_client.put_dir(local_dir, remote_dir)
            sftp_client.close()

        return True

    def copy_file_to_remote(self, local_file, remote_file, mode=None):
        if not RemoteSSH.is_ready():
            return False
        with self.ssh_cli() as ssh:
            sftp_client = ssh.open_sftp()
            sftp_client.put(local_file, remote_file)

            if mode is not None:
                sftp_client.chmod(remote_file, mode)

            sftp_client.close()

        return True

    def execute(self, command, timeout=None, log_error=True, debug=False):
        # This is used to mirror utils.Process() class, so we can make ssh calls in the same way we make those calls.
        status = self.execute_remote(command, timeout=timeout, log_error=log_error, debug=debug)

        # Set the Process() like attributes.
        str_out = ''
        str_err = ''
        if self.output is not None:
            for line in self.output:
                str_out += line

        if self.error is not None:
            for line in self.error:
                str_err += line

        setattr(self, 'strout', str_out)
        setattr(self, 'strerr', str_err)
        setattr(self, 'failed', self.status)

        return status

    def __conn_manager(self, ssh, command, timeout=None, log_error=True, debug=False, non_blocking=False, no_log=False):
        """
        Because almost all usage will be considered command line (silent) non-interactive, this will dump any stdin
        and ignore it.  Will prevent command hanging if command is asking a question waiting for user input.

        :param ssh:
        :param command:
        :param timeout:
        :param log_error:
        :param debug:
        :param non_blocking:
        :return:
        """
        try:
            # Default error.
            self.error = traceback.format_exc()

            # Add this new instance to the list on connections.
            self.__instance_connections[id(ssh)] = ssh

            if debug:
                if not no_log:
                    LOG.info("Firing ssh command on [{0}]: {1}".format(ssh.remote_hostname, command))
            else:
                if not no_log:
                    LOG.debug("Firing ssh command on [{0}]: {1}".format(ssh.remote_hostname, command))

            # At this point we're connected, ssh is the connection, so get the transport to send the keep alives on.
            transport = ssh.get_transport()
            if transport is not None and hs_defines.SSH_SEND_KEEPALIVE:
                LOG.debug("Setting ssh keepalive interval to [{0}] seconds for connection > [{1}].".format(
                    hs_defines.SSH_KEEPALIVE_INTERVAL_SECONDS, ssh.remote_hostname
                ))
                transport.set_keepalive(hs_defines.SSH_KEEPALIVE_INTERVAL_SECONDS)

            _stdin, _stdout, _error = ssh.exec_command(command)  # 8/28/20: EF - , get_pty=True, timeout=timeout)

            # get the shared channel for stdout/stderr/stdin
            channel = _stdout.channel

            # we do not need stdin for now, since most calls are scripted non-interactive, treat input as failure
            _stdin.close()

            # indicate that we're not going to write to that channel anymore
            channel.shutdown_write()

            if non_blocking:
                # This is non-blocking, so just return 0, meaning we successfully launched the command.
                LOG.debug("Non-blocking SSH call; not waiting for command completion.")
                if channel.exit_status_ready():
                    # Looks like we got an exit status; maybe bad command, etc.  Get it, return it.
                    self.status = channel.recv_exit_status()
                    self.error = _error.readlines()
                    self.output = _stdout.readlines()
                else:
                    # We launched the command, it's running, set status to 0 for success.
                    self.status = 0
                return self.status

            # read stdout/stderr in order to prevent read block hangs
            stdout_chunks = []
            stderr_chunks = []
            stdout_chunks.append(_stdout.channel.recv(len(_stdout.channel.in_buffer)))

            # chunked read to prevent stalls
            while not channel.closed or channel.recv_ready() or channel.recv_stderr_ready():
                # stop if channel was closed prematurely, and there is no data in the buffers.
                got_chunk = False
                readq, _, _ = select.select([_stdout.channel], [], [], timeout)
                for c in readq:
                    if c.recv_ready():
                        stdout_chunks.append(_stdout.channel.recv(len(c.in_buffer)))
                        got_chunk = True
                    if c.recv_stderr_ready():
                        # make sure to read stderr to prevent stall
                        stderr_chunks.append(_error.channel.recv_stderr(len(c.in_stderr_buffer)))
                        got_chunk = True
                '''
                1) make sure that there are at least 2 cycles with no data in the input buffers in order to not 
                exit too early (i.e. cat on a >200k file).
                2) if no data arrived in the last loop, check if we already received the exit code
                3) check if input buffers are empty
                4) exit the loop
                '''
                if not got_chunk \
                        and _stdout.channel.exit_status_ready() \
                        and not _error.channel.recv_stderr_ready() \
                        and not _stdout.channel.recv_ready():
                    # indicate that we're not going to read from this channel anymore
                    _stdout.channel.shutdown_read()
                    # close the channel
                    _stdout.channel.close()
                    break  # exit as remote side is finished and our bufferes are empty

            self.output = stdout_chunks
            self.error = stderr_chunks

            # close all the pseudofiles
            _stdout.close()
            _error.close()

            # exit code is always ready at this point
            self.status = _stdout.channel.recv_exit_status()

            if not self.status == 0 and log_error:
                if len(self.output) > 0:
                    LOG.error("Output: {0}".format("\n".join(self.output)))
                if len(self.error) > 0:
                    LOG.error("Error: {0}".format("\n".join(self.error)))

        except Exception, err:
            pass

        return self.status

    def execute_remote(self, command, timeout=None, log_error=True, debug=False, non_blocking=False, no_log=False):
        if not RemoteSSH.is_ready():
            return False

        try:
            # Reset the outputs in case the process is re-launched, like post reboot.
            self.output = None
            self.error = None
            self.status = 999   # default status code.

            # If non-blocking, that means someone wants to run SSH command and leave it go, monitor and handle external
            if non_blocking:
                # Open a non-context managed SSH connection.
                ssh = self.persistent_cli()
                LOG.debug("Opened a persistent SSH connection to [{0}].".format(ssh.remote_hostname))
                return self.__conn_manager(ssh, command, timeout, log_error, debug, non_blocking, no_log)
            else:
                with self.ssh_cli() as ssh:
                    # Use the context manager and open a normal SSH blocking call.  Wait for the command to complete.
                    return self.__conn_manager(ssh, command, timeout, log_error, debug, non_blocking, no_log)

        except Exception, err:
            self.error = traceback.format_exc()

        return self.status

    def adjust_ownership_for_user(self, user_name, file_path):
        if not self.execute("id -gn {0}".format(user_name)) == 0:
            return False
        group_name = self.getout().strip()

        command = "chmod 644 {0}; chgrp {1} {0}; chown {2} {0}".format(file_path, group_name, user_name)
        if not self.execute_remote(command) == 0:
            return False
        return True

    def update_user_known_hosts_from_root(self, user_name):
        # Copy the known_hosts to the user; overwriting whatever is there for now.
        known_hosts_file = SSH_KNOWN_HOSTS_PATH
        user_known_hosts_file = '~{0}/.ssh/known_hosts'.format(user_name)

        if not self.execute("id -gn {0}".format(user_name)) == 0:
            return False
        group_name = self.getout().strip()

        command = "cp {0} {1}; chmod 644 {1}; chgrp {2} {1}; chown {3} {1}".format(known_hosts_file,
                                                                                   user_known_hosts_file,
                                                                                   group_name, user_name)

        if not self.execute_remote(command) == 0:
            return False
        return True

    def create_user(self, user_name, password):
        """
        Create the user if it does not exist.
        TODO: update password if it exists?
        TODO: ensure it's not locked.
        :param user_name:
        :param password:
        :return:
        """
        try:
            if password is not None:
                LOG.debug("Password provided, checking and creating user as required.")

                # Create the user if they do not exist.
                enc_pass = crypt.crypt(password, self.ssh_cli.args[0])
                if not self.execute_remote("id -u {1} &>/dev/null || useradd -p {0} {1}".format(
                        enc_pass, user_name)) == 0:
                    return False

                # Ensure the password is OK
                if not self.set_password(user_name, password) == 0:
                    return False

            # Ensure that user is part of 'users' group.
            if not self.execute_remote("usermod -a -G users {0}".format(user_name)) == 0:
                return False

            # Ensure the user account is not fail locked.
            if not self.execute_remote("faillock --user {0} --reset".format(user_name)) == 0:
                return False

            return True
        except Exception, err:
            LOG.error("Failed creating user {0}.".format(user_name))
            return False

    def set_hostname_remote_node(self, hostname):
        command = 'hostnamectl set-hostname {0}'.format(hostname)
        return self.execute_remote(command, timeout=None)

    def set_password(self, user='root', password='cvadmin'):
        command = "echo '" + password + "' | passwd " + user + " --stdin "
        return self.execute_remote(command, timeout=None, no_log=True)

    def config_network(self, config_file, python_path=None):
        """
        On the remote node we're connected to; launch the network configuration, but DO NOT wait for it.
        We can't wait, because during network config, the IP addresses will change and the SSH connection goes stale.

        Non-blocking will return immediately.
        :return:
        """
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        command = '{1}/cvremotenwconfig.py {0}'.format(config_file, python_path)
        return self.execute_remote(command, non_blocking=True)

    def get_node_ip_addresses(self, python_path=None):
        """
        :return: bool - True if file successfully generated and collected; False otherwise.
        """
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        command = 'cd '+python_path+';' \
                  'python -c "from task_manager import cvmanager_utils;' \
                  'print(\',\'.join(cvmanager_utils.ip4_addresses()));"'

        # Only allow 30 seconds to run this command.  Need to test the timeout code.
        if not self.execute(command, timeout=30, log_error=False) == 0:
            LOG.error("Failed getting remote node IP addresses.")
            return False

        return True

    def get_block_device_info(self, output_file, python_path=None):
        """
        Run the cvhyperscale -c blk command to get the device info xml.  File is generated locally on the remote node
        in /tmp location at /tmp/.blk_dev_scan.xml.  If the file is successfully generated, get it to the location
        specified as "output_file".
        :param python_path:
        :param output_file: str - Location where to put the remote file after generating it.
        :return: bool - True if file successfully generated and collected; False otherwise.
        """
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        # File will be written here on the remote node.
        remote_node_output_file = '/tmp/.blk_dev_scan.xml'

        command = "export LD_LIBRARY_PATH=$LDLIBRARY_PATH:/opt/commvault/Base;" \
                  "{1}/cvhyperscale -c blk -o {0}".format(
                  remote_node_output_file, python_path)

        # Only allow 30 seconds to run this command.  Need to test the timeout code.
        if not self.execute_remote(command, timeout=30, log_error=False) == 0:
            LOG.error("Failed executing [cvhyperscale -c blk] on remote node [{1}].".format(
                remote_node_output_file, self.remote_host))
            return False

        # Get the file from the remote node
        if not self.get_remote_file(remote_node_output_file, output_file):
            LOG.error("Failed getting file [{0}] from remote node [{1}].".format(remote_node_output_file,
                                                                                 self.remote_host))
            return False

        return True

    def hs_2dot0(self, python_path=None):
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        command = "cd "+python_path+"; python -c 'import common; print common.is_hs2dot0()'"
        if not self.execute(command, timeout=1, log_error=False) == 0:
            return False

        output = self.getout()
        if 'True' not in output:
            return False

        return True

    def test_connection_remote_node(self, remote_node, user_name):
        if not self.execute_remote('ssh {0}@{1}'.format(user_name, remote_node), no_log=True, log_error=False) == 0:
            return False
        return True

    def generate_ssh_key(self, input_file_path=SSH_INPUT_FILE_PATH, overwrite=False, username=None,
                         switch_to_user=False):
        """
        On the remote node we're connected to; generate the SSH key for that node.

        Generates the key for the user connected.

        :return:
        """
        user_public_key = input_file_path.replace("~", "~{0}".format(username))

        if overwrite:
            overwrite = 'echo y | '
        else:
            overwrite = ''

        if switch_to_user:
            # Run this generation as named user over root ssh connection.
            command = 'sudo -S -u {0} -i /bin/bash -l -c \'ssh-keygen -f {1} -N "" -C "{0}@cvmanager_key_{2}"\''.format(
                username, user_public_key, self.remote_host
            )
        else:
            command = '{1}ssh-keygen -f {0} -N "" -C "{3}@cvmanager_key_{2}"'.format(user_public_key, overwrite,
                                                                                     self.remote_host, username)

        # Generate the key on the remote node.
        if not self.execute_remote(command, log_error=False) == 0:
            if 'already exists' in ''.join(self.output):
                if not overwrite:
                    LOG.debug("SSH public key already exists on node [{0}]; not overwriting.".format(
                        self.remote_host))
                else:
                    LOG.debug("SSH public key already exists on node [{0}]; overwriting with new key pair.".format(
                        self.remote_host))
            else:
                return False
        return True

    def copy_user_key_to_host(self, remote_host, user_name):
        """
        This requires root passwordless ssh already setup.
        Connect to the remote host, get the public key, and authorize it on self.

        self user_name will now be authorized to ssh to the remote_host as ROOT user.

        ssh-copy-id -i ~<user_name>/.ssh/id_rsa.pub -o StrictHostKeyChecking=no root@<remote_host>
        ssh-copy-id -i ~admin/.ssh/id_rsa.pub -o StrictHostKeyChecking=no root@192.168.39.133

        :param remote_host:
        :param user_name:
        :return:
        """

        global SSH_PUB_FILE_PATH

        user_public_key = SSH_PUB_FILE_PATH.replace("~", "~{0}".format(user_name))
        command = 'ssh-copy-id -i {0} -o StrictHostKeyChecking=no root@{1}'.format(user_public_key, remote_host)
        return self.execute_remote(command)

    def copy_ssh_id_to_host(self, remote_host, remote_username, remote_password=None, switch_to_user=False):
        """
        Connect to the remote host, get the public key, and authorize it on self.

        sshpass -p <remote_password> ssh-copy-id -i ~<user_name>/.ssh/id_rsa.pub
                    -o StrictHostKeyChecking=no <remote_username>@<remote_host>
        sshpass -p password ssh-copy-id -i ~admin/.ssh/id_rsa.pub -o StrictHostKeyChecking=no admin@192.168.39.133

        :return:
        """
        global SSH_PUB_FILE_PATH, SSH_AUTH_KEYS_PATH
        user_public_key = SSH_PUB_FILE_PATH.replace("~", "~{0}".format(remote_username))
        authorize_keys = SSH_AUTH_KEYS_PATH.replace("~", "~{0}".format(remote_username))

        if remote_password is not None:
            password_command = r'sshpass -p {0} '.format(re.escape(remote_password))
        else:
            password_command = ''

        if switch_to_user:
            # In this case, we need to take the pubkey from self, and copy to the authorized_keys for the specified user
            # on remote_host.  This again assumes ROOT is setup already.  It will fail without root passwordless ssh.
            command = r"cat {0} | ssh root@{1} 'umask 0077; mkdir -p {2}; cat >> {3}; sort -u {3} -o {3}'".format(
                user_public_key, remote_host, os.path.dirname(authorize_keys), authorize_keys)
        else:
            command = r'{0}ssh-copy-id -i {1} -o StrictHostKeyChecking=no {2}@{3}'.format(password_command,
                                                                                          user_public_key,
                                                                                          remote_username,
                                                                                          remote_host)
        if not self.execute_remote(command, no_log=True) == 0:
            return False

        if switch_to_user:
            if not self.adjust_ownership_for_user(remote_username, authorize_keys):
                return False
        return True
