#!/usr/bin/env python


# Python imports
import six
import abc
import os
from subprocess import Popen, PIPE
from timeit import default_timer as timer
import pwd
import crypt
import time
import sys
import xattr


if __name__ == "__main__":
    # This is critical to the project structure.
    sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

# Project imports
import common
import registertocs
import wrappers
import utils
import cvnwconfigmgr
import cvupgrade_common
import cvupgradeos
import cvnwhelper

import task_manager.cvmanager_catalog as cvmanager_catalog
import task_manager.cvmanager_defines as cvmanager_defines
import task_manager.cvmanager_utils as cvmanager_utils
import task_manager.cvmanager_logging as cvmanager_logging

import hs_ssh
import hs_defines
import hs_utils

LOG = cvmanager_logging.get_log(cvmanager_defines.LOG_FILE_NAME)
DYNAMIC_DEFINES = sys.modules[cvmanager_defines.DYNAMIC_DEFINE_NAME]


class NoExecutorException(Exception):
    def __init__(self, node):
        self.node = node
        self.message = "Unable to run passwordless ssh command on remote node [{0}].".format(node)
        super(NoExecutorException, self).__init__(self.message)


@six.add_metaclass(abc.ABCMeta)
class HyperScaleNode:
    """ This is the common structure for all HyperScale nodes.  The underlying cluster software does not matter.
    These methods are capable of being used on any node, typically commands related to commvault.

    """
    @property
    def log(self):
        if self._log is None:
            self._log = LOG
        return self._log

    @log.setter
    def log(self, value):
        self._log = value

    def __getstate__(self):
        """Return state values to be pickled. Since we can't pickle loggers or files, drop it."""
        self._log = None
        return self

    def __str__(self):
        # We use the hostname as defining characteristic of hyperscale nodes, they must be unique.
        return self.hostname

    def __iter__(self):
        yield self.hostname

    def __init__(self, hostname=None, node_type=None, *args, **kwargs):
        # Set the hostname
        if hostname is None:
            # no hostname given, so assume it's the local host
            hostname = wrappers.get_hostname()
        self.hostname = hostname

        # Set the node type; gluster or hv....can also force stand alone.
        if node_type is None:
            self.__detect_node_type()
        else:
            self.node_type = node_type

        # By default assume the upgrade is required.
        self.upgrade_required = True

        # Check if the node is local node (where cvmanager is running) or remote node.
        self.local_node = self.is_local_node()

        self._log = LOG

        # Many of the cvupgradeos and cvupgrade_common functions have odd logging, so setup the log dir
        local_log_dir = os.path.join(common.get_cvlt_logdir(), cvupgrade_common.debug_logdir)
        if not os.path.exists(local_log_dir):
            os.makedirs(local_log_dir)

        self.init_node_params(*args, **kwargs)

    def __detect_node_type(self):
        # Lets run some commands and check if we can determine the type.
        if common.is_hs2dot0():
            self.node_type = hs_defines.NodeTypes.HEDVIG
        else:
            self.node_type = hs_defines.NodeTypes.GLUSTER

    def __is_gluster_node(self):
        # gluster pool list will always return localhost if this is a glusterfs node.
        process = utils.Process()
        command = "gluster pool list | grep localhost"
        process.execute(command)
        output = process.getout()
        output = output.strip()
        if len(output) > 0:
            return True
        else:
            return False

    def init_node_params(self, *args, **kwargs):
        # Any node parameters that are external properties, get & set them here.
        setattr(self, 'last_upgrade_time', 0)
        setattr(self, 'cluster_name', None)
        self.get_upgrade_timestamp_from_registry()

        # cvnwconfigmgr needs special log handling.
        hs_utils.init_logging()

        if kwargs.get('get_ips', False):
            # Getting the IPs for nodes using passwordless SSH for ROOT user.
            node_ips = self.get_node_ips(True)
            if node_ips:
                node_ips = node_ips.get('IP_ADDRESS_INFO', {})
            else:
                node_ips = {}
            list_of_ips = [ip for ip in node_ips.values() if not ip.startswith('127')]
            list_of_ips.append(self.hostname)
            setattr(self, 'node_ips', list_of_ips)

        if kwargs.get('node_ips', False):
            setattr(self, 'node_ips', kwargs.get('node_ips'))

    def get_upgrade_timestamp_from_registry(self):
        if self.local_node:
            value = common.getregistryentry(cvupgrade_common.maregistry, "nHyperScaleLastUpgradeTime")
            if value:
                setattr(self, 'last_upgrade_time', long(value))

    def is_local_node(self):
        # Get the IP address for this hostname
        addr = wrappers.name2ip(self.hostname)

        # Get all IP addresses of this node
        ip_addresses = cvmanager_utils.ip4_addresses()

        if addr in ip_addresses:
            return True

        return False

    def stop_services(self):
        if self.local_node:
            LOG.info("Stopping services on local node [{0}].".format(self.hostname))
            if not cvupgrade_common.stop_services(False, False):
                LOG.error("Failed stopping services on [{0}].".format(self.hostname))
                return False
        else:
            LOG.info("Stopping services on remote node [{0}].".format(self.hostname))
            ret = cvupgradeos.on_remotenode("stopservices_commvault", wrappers.get_hostname(), self.hostname)
            if ret:
                LOG.error("Failed stopping services on remote node [{0}].".format(self.hostname))
                return False

        return True

    def start_services(self):
        if self.local_node:
            LOG.info("Starting services on local node [{0}].".format(self.hostname))
            if not cvupgrade_common.start_commvault():
                LOG.error("Failed starting services on [{0}].".format(self.hostname))
                return False
        else:
            LOG.info("Starting services on remote node [{0}].".format(self.hostname))
            ret = cvupgradeos.on_remotenode("startservices", wrappers.get_hostname(), self.hostname)
            if ret:
                LOG.error("Failed starting services on remote node [{0}].".format(self.hostname))
                return False

        return True

    def check_lvscan(self):
        if self.local_node:
            LOG.info("Checking for lvscan on local node ..." + self.hostname)
            if not cvupgrade_common.check_for_lvscan():
                return False
        else:
            LOG.info("Checking for lvscan on remote node ..." + self.hostname)
            ret = cvupgradeos.on_remotenode("check_lvscan", wrappers.get_hostname(), self.hostname)
            if ret:
                LOG.error("Failed checking lvscan on remote node [{0}].".format(self.hostname))
                return False
        return True

    def disable_multipath(self):
        if self.local_node:
            self.log.info("Disabling multipath on local node [{0}].".format(self.hostname))
            ret_code = cvupgrade_common.disable_multipath()
            if ret_code == 2:
                # multipath.d not enabled to begin with.  set reg key and continue
                common.setregistryentry(cvmanager_defines.REG_MA_KEY, "sEnableMultipathAfterUpgrade", "No")
                return True
            elif not ret_code == 0:
                return False
        else:
            self.log.error("Disabling multipath not supported for remote nodes.")
            return False
        return True

    def enable_multipath(self):
        if self.local_node:
            enable_multipath = common.getregistryentry(cvmanager_defines.REG_MA_KEY, "sEnableMultipathAfterUpgrade")
            if enable_multipath.lower() == 'yes':
                self.log.info("Enabling multipath on local node [{0}].".format(self.hostname))
                if not cvupgrade_common.enable_multipath() == 0:
                    return False
            else:
                self.log.info("Not enabling multipathd service because it was not running at start of upgrade.")
        else:
            self.log.error("Enabling multipath not supported for remote nodes.")
            return False
        return True

    def set_grub_default(self):
        """ Kernel format is always like:
        '3.10.0-957.10.1.el7.x86_64'
        <major>.
        """
        if self.local_node:
            installed_kernel_list = utils.Platform.get_sorted_kernel_list(reverse=True)

            cur_kernel_version = wrappers.get_kernel_version()
            latest_kernel_version = installed_kernel_list[0]

            self.log.info("Current kernel version [{0}]".format(cur_kernel_version))
            self.log.info("New kernel version [{0}]".format(latest_kernel_version))

            run_mkconfig = False
            if not cur_kernel_version == latest_kernel_version:
                # The current kernel is NOT the latest kernel, run mkconfig
                self.log.info("Current kernel is not the latest kernel available, running mkconfig.")
                run_mkconfig = True

            # check for initramfs files
            if not cvupgrade_common.initramfs_present(latest_kernel_version):
                return False

            # Check kernel version in grubenv
            # if not cvupgrade_common.is_grubenv_ok(latest_kernel_version):
            if run_mkconfig:
                self.log.info("Performing mkconfig ...")
                if not cvupgrade_common.make_grubcfg() == 0:
                    # mkconfig failed.
                    return False
                
                self.log.info("Force set the new kernel in grub env...")
                if not cvupgrade_common.set_saved_entry(latest_kernel_version) == 0:
                    return False
            return True
        else:
            self.log.error("Setting grub default kernel not supported on remote nodes.")
            return False

    def uninstall_conflicting_packages(self):
        if not os.path.exists("/usr/bin/pip"):
            self.log.warning("pip not installed, no packages to uninstall")
            return True

        pkg_list = cvupgrade_common.get_pkgs_to_uninstall()
        installed = False
        for pkg in pkg_list:
            rpm_name = pkg["rpm_name"]
            pip_name = pkg["pip_name"]

            with open(cvmanager_defines.UPGRADE_REPO_MASTER_RPM_LIST, "r") as f:
                found_rpm = any(rpm_name in line for line in f)
                if found_rpm:
                    self.log.info("Found [%s] in rpmlist file ..." % rpm_name)
                    installed = cvupgrade_common.is_pkg_in_pip(pip_name)
            f.close()

            if installed:
                self.log.info("[%s] was installed via pip ... uninstalling " % pip_name)
                if not cvupgrade_common.uninstall_from_pip(pip_name):
                    return False
            else:
                self.log.info("[%s] was not installed via pip .... nothing to be done." % pip_name)
        return True

    def get_node_ips(self, json_format=False, python_path=None):
        """
        Using [arch config-nw] command, perform network config on the remote node.
        :param python_path:
        :param json_format:
        :param output_file:
        :param remote_host:
        :return:
        """
        task_catalog = cvmanager_catalog.Catalog('Node_Configuration')
        node_config = task_catalog.get_file("{0}_arch_hw_info.config".format(self.hostname), True)

        # Connect to the node using default credentials.
        ssh_conn = hs_ssh.RemoteSSH(self.hostname, username=cvmanager_defines.DEFAULT_IMAGE_USER,
                                    password=cvmanager_defines.DEFAULT_IMAGE_PASSWORD,
                                    timeout=hs_defines.SSH_TIMEOUT, banner_timeout=hs_defines.SSH_BANNER_TIMEOUT)

        # Generate the block device info file on the remote node.
        if not ssh_conn.get_node_ip_addresses(python_path=python_path):
            self.log.warning("Failed getting ip addresses in use for node [{0}].".format(self.hostname))
            return False

        node_config = ssh_conn.getout()

        if json_format:
            json_config = {'IP_ADDRESS_INFO': {}}
            for i, ip in enumerate(node_config.strip().split(",")):
                json_config['IP_ADDRESS_INFO']['interface_{0}'.format(i)] = ip
            node_config = json_config

        return node_config

    def get_arch_hw_info(self, json_format=False):
        """
        Using [arch config-nw] command, perform network config on the remote node.
        :param json_format:
        :param output_file:
        :param remote_host:
        :return:
        """
        task_catalog = cvmanager_catalog.Catalog('Node_Configuration')
        node_config = task_catalog.get_file("{0}_arch_hw_info.config".format(self.hostname), True)

        command = r'arch get-hwinfo {0} {1}'.format(self.hostname, str(node_config))
        process = Popen(command, stdout=PIPE, stderr=PIPE, shell=True)
        output, error = process.communicate()

        if not process.returncode == 0:
            raise Exception(
                "Failed getting hardware info from remote node [{0}], with output [{1}] error [{2}].".format(
                    self.hostname, output, error))

        # Successfully read hardware info.
        node_config_info = node_config.read_config_file()

        # Make this JSON
        if json_format:
            json_config = {}
            for section in node_config_info.sections():
                json_config[section] = cvmanager_utils.get_json_from_config_section(node_config_info, section)
            node_config_info = json_config

        return node_config_info

    def check_if_device_mounted(self, device, target, use_default_image_cred=False):
        """
        Uses findmnt to check if the device is mounted at target.
        :param use_default_image_cred: bool - True will use default image credentials. False (default) will use
        passwordless ssh connection.
        :param device:
        :param target:
        :param python_path:
        :return: bool - True if device mounted at target, False otherwise.
        """
        task = self.__get_executor(use_default_image_cred=use_default_image_cred)

        command = 'findmnt -S {0} -n -o TARGET'.format(device)
        if not task.execute(command) == 0:
            # If findmnt does not return 0, that means the device is not mounted.
            self.log.error("findmnt did not find device [{0}] mounted.".format(device))
            return False

        mounted_target = task.getout().strip()

        if not mounted_target == target:
            self.log.error("Device [{0}] is mounted at [{1}] but expected to be mounted on [{2}].".format(
                device, mounted_target, target
            ))
            return False

        return True

    def get_node_block_device_xml(self, python_path=None):
        if self.local_node:
            return self.__get_local_node_block_device_xml(python_path=python_path)
        else:
            return self.__get_remote_node_block_device_xml(python_path=python_path)

    def __get_remote_node_block_device_xml(self, python_path=None):
        """
        Create a file in the task catalog to save the node hardware info.  This file will reside in the catalog, with
        the format '<catalog_path>/<node_name>_hwinfo.config'

        EXAMPLE: /ws/ddb/cvmanager/catalog/task_storage/Discover_Nodes/1.1.1.1_hwinfo.config

        Because this uses arch; it will work locally and remotely.

        """
        task_catalog = cvmanager_catalog.Catalog('Node_Configuration')
        node_config = task_catalog.get_file("{0}_hwinfo.xml".format(self.hostname), True)

        # Connect to the node using default credentials.
        ssh_conn = hs_ssh.RemoteSSH(self.hostname, username=cvmanager_defines.DEFAULT_IMAGE_USER,
                                    password=cvmanager_defines.DEFAULT_IMAGE_PASSWORD,
                                    timeout=hs_defines.SSH_TIMEOUT, banner_timeout=hs_defines.SSH_BANNER_TIMEOUT)

        # Generate the block device info file on the remote node.
        if not ssh_conn.get_block_device_info(str(node_config), python_path=python_path):
            self.log.warning("Failed getting node configuration for node [{0}].".format(self.hostname))
            return False

        node_config_info = node_config.parse_xml()

        return node_config_info

    def __get_local_node_block_device_xml(self, python_path=None):
        """
        Create a file in the task catalog to save the node hardware info.  This file will reside in the catalog, with
        the format '<catalog_path>/<node_name>_hwinfo.xml'
        This is the local node hardware block device information.

        Uses [cvhyperscale -c blk] command to enumerate the devices.

        EXAMPLE: /ws/ddb/cvmanager/catalog/task_storage/Discover_Nodes/1.1.1.1_hwinfo.xml

        """
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        # Create the output file
        task_catalog = cvmanager_catalog.Catalog("Node_Configuration")
        node_config = task_catalog.get_file("{0}_hwinfo.xml".format(self.hostname), True)

        # Use cvhyperscale to get the output.
        command = 'export LD_LIBRARY_PATH=$LDLIBRARY_PATH:/opt/commvault/Base;' \
                  '{1}/cvhyperscale -c blk -o {0}'.format(node_config.file_path, python_path)
        process = Popen(command, stdout=PIPE, stderr=PIPE, shell=True)
        output, error = process.communicate()

        if not process.returncode == 0:
            raise Exception("Failed getting node xml for node [{0}], with output [{1}] error [{2}].".format(
                self.hostname, output, error
            ))

        # cvhyperscale command outputs XML, parse it.
        node_config_info = node_config.parse_xml()

        return node_config_info

    def get_node_network_json(self, front_end_json=False):
        """
        Create a file in the task catalog to save the node network information. This file will reside in the catalog,
        with the format '<catalog_path>/<node_name>_nwinfo.json'
        This is the local node network configuration.

        Uses [cvnwconfigmgr.py get out.json] command to enumerate the devices.

        EXAMPLE: /ws/ddb/cvmanager/catalog/task_storage/Discover_Nodes/1.1.1.1.json

        """
        if self.local_node:
            return self.__get_local_node_network_json(front_end_json)
        else:
            self.log.error("Getting remote node network configuration JSON is not supported.")
            return {}

    @hs_utils.mutex_lock
    def convert_backend_json_to_front_end(self, serial_num, backend_json_data):
        """
        Takes the backend nw config and converts it to front end nw config.  This cvnwhelper code uses globals,
        so need to decorate this with the mutex lock, preventing multiple threads from running this concurrently.
        This decorator will just run sequentially.

        :param serial_num:
        :param backend_json_data:
        :return:
        """

        # Retrieve existing settings
        backend_json_data["update_avahi"] = "true"

        # Add missing fields
        backend_json = {}
        backend_json["advancedNodes"] = {}
        backend_json["advancedNodes"][serial_num] = backend_json_data
        backend_json["basicNodes"] = {}
        backend_json["basicNodes"][serial_num] = {}
        backend_json["basicNodes"][serial_num]["serialno"] = ""
        backend_json["basicNodes"][serial_num]["dataProtectionNetwork"] = ""
        backend_json["basicNodes"][serial_num]["storagePool"] = ""
        backend_json["basicNodes"][serial_num]["tooltips"] = []

        # convert backend json to frontend json
        frontend_json = cvnwhelper.convert_to_frontend_json(backend_json, self.log)

        return frontend_json

    @hs_utils.mutex_lock
    def __get_front_end_json(self, output_file):
        """

        :param output_file: cvmanager_task_catalog.CatalogFile() - Use the Catalog().get_file to get this file object.
        :return: dict - JSON dictionary of the front-end JSON dict.
        """
        # Retrieve existing settings; backend_json.
        with hs_utils.HiddenPrints():
            data = cvnwconfigmgr.cvnwmanager.get_configured_interfaces()

        # Convert this backend JSON to the front-end.
        front_end_json = self.convert_backend_json_to_front_end(data)

        # Write the file to the catalog.
        output_file.write(front_end_json, cvmanager_catalog.FileType.JSON)

        return front_end_json

    @hs_utils.mutex_lock
    def __get_local_node_network_json(self, front_end_json):
        # Create the output file
        task_catalog = cvmanager_catalog.Catalog("Node_Configuration")
        node_config = task_catalog.get_file("{0}_nwinfo.json".format(self.hostname), True)

        if front_end_json:
            # Get the front-end JSON and not the back-end JSON.
            return self.__get_front_end_json(node_config)

        # Retrieve existing settings; backend_json.
        with hs_utils.HiddenPrints():
            data = cvnwconfigmgr.cvnwmanager.get_configured_interfaces()
        node_config.write(data, cvmanager_catalog.FileType.JSON)

        return data

    def create_user(self, user_name, password):
        try:
            if password is not None:
                enc_pass = crypt.crypt(password, self.hostname)
                os.system("useradd -p {0} {1}".format(enc_pass, user_name))
            else:
                os.system("useradd {0}".format(user_name))
            return True
        except Exception, err:
            self.log.error("Failed creating user {0}.".foramt(user_name))
            return False

    def set_node_password(self, user_name, password):
        proc = utils.Process()
        command = "echo '" + password + "' | passwd " + user_name + " --stdin "
        if not proc.execute(command) == 0:
            self.log.error("Failed setting password for user: {0}".foramt(user_name))
            return False
        return True

    def set_user_group(self, user_name, group_name="users"):
        """
        "usermod -a -G {0} users".format(user_name)
        :param user_name:
        :param group_name:
        :return:
        """
        proc = utils.Process()
        command = "usermod -a -G {0} {1}".format(group_name, user_name)
        if not proc.execute(command) == 0:
            self.log.error("Failed adding user {0} to group {1}.".format(user_name, group_name))
            return False
        return True

    def __check_and_create_local_user(self, username, password):
        # Create the user if it does not exist.
        try:
            pwd.getpwnam(username)
            # User exists, make sure local password is OK.
            if password is not None:
                if not self.set_node_password(username, password):
                    return False
            else:
                # We don't have a password, don't modify it.
                pass
        except KeyError, ke:
            # User does not exist, create it.
            if not self.create_user(username, password):
                return False

        # In all cases ensure user is in the gruop:
        if not self.set_user_group(username):
            return False

        return True

    def __check_and_create_remote_user(self, username, password):
        """
        This can only be performed with root already setup and working.
        :param username:
        :param password:
        :return:
        """
        ssh_conn_local_to_new_node = hs_ssh.RemoteSSH(self.hostname)
        if not ssh_conn_local_to_new_node.create_user(username, password):
            self.log.error("Failed checking creating user [{0}] on remote node [{1}].".format(username,self.hostname))
            return False
        return True

    def __test_local_node_connection(self, remote_node, user_name):
        test_conn = hs_ssh.RemoteSSH(remote_node.hostname, username=user_name)
        if test_conn.test_connection(named_user_only=True):
            self.log.info(
                "Passwordless ssh is already configured from [{0}] -> [{1}] for user [{2}].".format(
                    self, remote_node, user_name
                ))
            return True
        return False

    def __test_remote_node_connection(self, remote_node, user_name):
        # Get a root connection to SELF.  Only works when ROOT is setup from localhost to self
        user_ssh_conn = hs_ssh.RemoteSSH(self.hostname)
        if not user_ssh_conn.test_connection_remote_node(remote_node, user_name):
            return False
        return True

    def test_node_connection(self, remote_node, user_name):
        """
        Check named user ssh connection between 2 nodes.

        :param username:
        :param password:
        :return:
        """
        if self.local_node:
            return self.__test_local_node_connection(remote_node, user_name)
        else:
            return self.__test_remote_node_connection(remote_node, user_name)

    def check_and_create_user(self, username, password):
        """
        If this username does not have a password, don't create it, just assume the user already exists.  We can only
        create new users which have passwords.

        :param username:
        :param password:
        :return:
        """
        if self.local_node:
            return self.__check_and_create_local_user(username, password)
        else:
            return self.__check_and_create_remote_user(username, password)

    def generate_new_user_key(self, username, password, over_write=False):
        """
        Generate a new SSH key.  For the root user, we should assume it has been created already and DO NOT overwrite
        for now.

        When we generate a user key, if password is none, this means we should attempt passwordless ssh as ROOT, connect
        to self, and [su username] once connected, before generating the key.

        :param over_write:
        :param username:
        :param password:
        :return:
        """
        switch_to_user = False
        if password is not None:
            user_ssh_conn = hs_ssh.RemoteSSH(self.hostname, username=username, password=password)
        else:
            # No password was given, so connect to the node as root passwordless, and su to user.
            switch_to_user = True
            user_ssh_conn = hs_ssh.RemoteSSH(self.hostname)

        if not user_ssh_conn.generate_ssh_key(overwrite=over_write, username=username, switch_to_user=switch_to_user):
            return False
        return True

    def get_public_key_from_remote_node(self, ssh_conn, remote_node, username, password):
        """
        Connects to a node (self.hostname) over passwordless ssh.  Then gets the public key from the remote_node,
        stores it locally on (self) temp location, then updates authorized_keys on that node.
        :param remote_node:
        :return:
        """
        user_public_key = hs_ssh.SSH_PUB_FILE_PATH.replace("~", "~{0}".format(username))

        # Get the id_rsa.pub from remote_host
        remote_host_pub_key = '{0}_{1}'.format(user_public_key, remote_node)

        # This will get the file from the remote_node onto self.hostname
        # sshpass -p newuser1 scp newuser1@1.1.1.1:~newuser1/.ssh/id_rsa.pub ~newuser1/.ssh/id_rsa.pub_1.1.1.1
        command = 'scp {3}@{0}:{1} {2}'.format(remote_node, user_public_key, remote_host_pub_key, username)

        if not ssh_conn.execute_remote(command) == 0:
            self.log.error("Failed getting public key from [{0}] on to [{1}].".format(remote_node, self))
            return False

        command = 'sudo -S -u {0} -i /bin/bash -l -c \'ssh-keygen -f {1} -N "" -C "{0}@cvmanager_key_{2}"\''.format(
            username, user_public_key, remote_node)

        # if not ssh_conn.execute_remote(command) == 0:
        if not ssh_conn.execute_remote('touch /home/admin/.ssh/id_rsa.pub.new') == 0:
            return False

        if not ssh_conn.get_remote_file('/home/admin/.ssh/id_rsa.pub', '/home/admin/.ssh/id_rsa.pub.new'):
            self.log.error("Failed getting public key from [{0}] on to [{1}].".format(remote_node, self))
            return False

        # TODO: Check if the authorized key is already there; if so, do not add it.
        # Now that the public key is on self.hostname, add it to authorized keys of that node.
        # cat /root/.ssh/id_rsa.pub_1.1.1.1 >> authorized_keys
        command = 'cat {0} >> {1}'.format(remote_host_pub_key, hs_ssh.SSH_PATH + 'authorized_keys')
        if not ssh_conn.execute_remote(command) == 0:
            self.log.error("Failed adding public key [{0}] on to [{1}] authorized_keys.".format(
                remote_host_pub_key, self))
            return False

        if not ssh_conn.execute_remote("chmod 644 {0}".format(hs_ssh.SSH_PATH + 'authorized_keys')) == 0:
            self.log.error("Failed adjusting permissions on [{1}].".format(self, "authorized_keys"))
            return False

        # Remove the temp key.
        # rm /root/.ssh/id_rsa.pub_1.1.1.1
        command = 'rm {0}'.format(remote_host_pub_key)
        if not ssh_conn.execute_remote(command) == 0:
            self.log.error("Failed removing temporary public key [{0}] on to [{1}].".format(remote_host_pub_key, self))
            return False

        return True

    def wait_for_reg_key(self, task, key_file, key_name, key_value=None, timeout=300):
        start = timer()
        end = 0
        complete = False

        # This command will read the key name from the key file and ONLY if the key is found it will exit with code 0
        command = r"out=$(cat " + key_file + " | grep -w ^" + key_name + "*"\
                  ") && echo \"$out\" | awk \"{print $2}\" FS=\" \""

        while not complete and (end - start) < timeout:
            time.sleep(5)
            if not task.execute(command, log_error=False) == 0:
                # Failed reading the key, it might not be there yet.  non-fatal at this point.
                pass
            else:
                # Key Found
                if key_value is None:
                    break

                key_output = task.getout().replace(key_name, '').strip()
                if len(key_output) == 0:
                    # key present, but not populated
                    return False

                if not key_value == key_output:
                    return False

                complete = True
            end = timer()
        else:
            # This won't actually kill it, but the remote processes are daemon threads, so they are just aborted.
            if (end - start) > timeout:
                self.log.error('Timed out waiting for reg key [{0}] to be set to proper value.'.format(key_name))
                return False

        return True

    def wait_for_v_disk_online(self, storage_pool_name=None, v_disk_name=None):
        """
        Wait for wait_time minutes, until the reg keys are populated.  That's enough to confirm the vdisk is mounted,
        and configuration is complete.

        We first wait for sHyperScaleStoragePoolName to be populated & equal to storage_pool_name.  Once that is
        populated, we double check the v_disk_list and wait for that.

        Finally, try to access the v disk.

        :param storage_pool_name:
        :param v_disk_name:
        :return:
        """
        if self.local_node:
            # Read local reg keys.
            task = utils.Process()
        else:
            # Remote node, read reg over ssh.
            ssh_conn = hs_ssh.RemoteSSH(self.hostname)
            if ssh_conn.test_connection():
                # SSH to the node worked.
                task = ssh_conn
            else:
                self.log.debug("SSH connection not available to node [{0}].".format(self))
                return False

        # 10-20-2020: EF - Do not wait for this key, only wait for storage pool name key based on discussion.
        # if not self.wait_for_reg_key(task, cvmanager_defines.REG_MA_KEY, 'sVDiskList', ",".join(v_disk_name)):
        #     self.log.error("sVDiskList was never detected as online on node [{0}].".format(self.hostname))
        #     return False

        if not self.wait_for_reg_key(task, cvmanager_defines.REG_MA_KEY, 'sHyperScaleStoragePoolName',
                                     storage_pool_name):
            self.log.error("sHyperScaleStoragePoolName was never detected on node [{0}].".format(self.hostname))
            return False

        return True

    def copy_id_to_host(self, remote_node_hosts, username, password, authorize_to_root=False):
        """
        This is going to connect to a remote node (self) using passwordless ssh, and copy the public key from that node
        to all remote_node_hosts.

        :param authorize_to_root:
        :param password:
        :param remote_node_hosts:
        :return:
        """
        copied_all_keys = True

        switch_to_user = False
        if password is not None:
            ssh_conn = hs_ssh.RemoteSSH(self.hostname, username=username, password=password)
        else:
            # No password was given, so connect to the node as root passwordless, and su to user.
            switch_to_user = True
            ssh_conn = hs_ssh.RemoteSSH(self.hostname)

        for host in remote_node_hosts:
            if not ssh_conn.copy_ssh_id_to_host(host, username, password, switch_to_user=switch_to_user):
                self.log.error("Failed copying id from [{0}] to [{1}].".format(self, host))
                copied_all_keys = False
                continue

            if authorize_to_root and not username == 'root':
                # For non-root user, we need to inform root of this new key for this specific user.  ROOT passwordless
                # SSH must already be setup for this.  This will allow this named user root ssh access to self.
                root_ssh_conn = hs_ssh.RemoteSSH(self.hostname)
                if not root_ssh_conn.copy_user_key_to_host(host, username) == 0:
                    self.log.error("Failed copying id from [{0}] to [{1}].".format(self, host))
                    copied_all_keys = False

        return copied_all_keys

    def update_known_hosts(self, host_keys, username):
        # TODO: if we need this should update self known_hosts for this user.
        return True

    def configure_node_passwordless_ssh(self, node, user_name=None, password=None, over_write=False, block_config=False,
                                        authorize_to_root=False):
        """
        This will configure passwordless ssh between any 2 nodes.

        :param authorize_to_root: If True, we only generate the KEY for this user, and make it known to root.
        :param block_config:
        :param over_write:
        :param node:
        :param user_name:
        :param password:
        :return:
        """
        # Set the reg key locally so that passwordless SSH methods proceed.  This key indicates that we used this
        # method to configure OR that
        common.setregistryentry(cvmanager_defines.REG_MA_KEY, "sUsePasswordlessSSH", "True")

        self.log.info("Configuring passwordless ssh between [{0}] and [{1}] for user [{2}].".format(self, node,
                                                                                                    user_name))

        if not user_name == 'root':
            # If password less SSH is being setup for any user other than root, we need root ssh already setup
            # in order to perform the configuration.  Check to ensure it's working from SELF -> NODE
            ssh_conn_self_to_node = hs_ssh.RemoteSSH(node.hostname)
            if not ssh_conn_self_to_node.test_connection():
                self.log.error(
                    "Passwordless ssh is not configured for root from [{0}] to [{1}]; unable to configure "
                    "any additional passwordless ssh connections.  Please configure for root first, and "
                    "re-attempt.".format(self.hostname, node.hostname))
                return False

            if self.test_node_connection(node, user_name):
                self.log.info("Passwordless ssh is already configured for [{0}] -> [{1}] user [{2}].".format(
                    self, node, user_name
                ))
                return True

            # Ensure that the user exists on se if not create it.
            if not self.check_and_create_user(user_name, password):
                return False

            # Ensure the specified user being configured exists ON the remote node; if not create it.
            if not ssh_conn_self_to_node.create_user(user_name, password):
                self.log.error("Failed checking creating user [{0}] on remote node [{1}].".format(user_name,
                                                                                                  node.hostname))
                return False

            # Update user known_hosts from root.
            if not ssh_conn_self_to_node.update_user_known_hosts_from_root(user_name):
                self.log.error("Failed updating known_hosts on remote node [{0}].".format(node.hostname))

        else:
            if self.test_node_connection(node, user_name):
                self.log.info("Passwordless ssh is already configured for [{0}] -> [{1}] user [{2}].".format(
                    self, node, user_name
                ))
                return True

        # Setup local key for this new user; connect to self using the user requested information.
        if not self.generate_new_user_key(user_name, password, over_write=over_write):
            self.log.error("Failed generating ssh key on node [{0}].".format(self.hostname))
            return False
        else:
            self.log.debug("Verified ssh public key on node [{0}].".format(self.hostname))

        # Generate the public\private key pair on the NEW node which is being setup.  Connect with standard
        # ssh connection to the node and generate the key\pair
        switch_to_user = False
        if password is not None:
            remote_ssh = hs_ssh.RemoteSSH(node.hostname, username=user_name, password=password)
        else:
            # No password was given, so connect to the node as root passwordless, and su to user.
            switch_to_user = True
            remote_ssh = hs_ssh.RemoteSSH(node.hostname)

        if not remote_ssh.generate_ssh_key(overwrite=over_write, username=user_name, switch_to_user=switch_to_user):
            self.log.error("Failed generating ssh key on node [{0}].".format(node.hostname))
            return False
        else:
            self.log.debug("Verified ssh public key on node [{0}].".format(node.hostname))

        # Connect to the node, and get its host key, update it as known_hosts for the specified user.
        # remote_node_host_keys = cvmanager_catalog.Catalog()
        # remote_host_key_file = remote_node_host_keys.get_file("{0}_host_keys".format(node.hostname), True)
        # if not remote_ssh.get_system_host_keys(str(remote_host_key_file)):
        #     self.log.error("Failed getting host key from [{0}].".format(node.hostname))
        #     return False
        # remote_host_key_file.read_file()
        #
        # # Now that we have the hostkeys for the new node, update it into known_hosts.
        # if not self.update_known_hosts(remote_host_key_file.lines, user_name):
        #     self.log.error("Failed adding remote host [{0}] to known_hosts on [{1}].".format(node.hostname,
        #                                                                                      self.hostname))

        # Users and KEYS exist on BOTH nodes now; self & node.  Now self needs to send pub key to node, and node needs
        # to send pub key to self.
        # if not self.hostname == node.hostname:
        # Copy self pub key to node.
        node_ips = getattr(node, 'node_ips', [node.hostname])
        node_ips.append(node.hostname)
        node_ips = list(set(node_ips))
        if not self.copy_id_to_host(node_ips, user_name, password, authorize_to_root):
            self.log.error("Failed copying ssh id from [{0}] -> [{1}].".format(self, node))
            return False
        self.log.info("Successfully configured passwordless ssh from [{0}] -> [{1}].".format(self, node))

        if block_config:
            # This is a block config, so the entire block will handle the traverse direction, no need to do it here.
            return True

        # Get the public key from remote node, onto node SELF.
        # if not self.copy_id_to_host(node_info['ip_addresses'], user_name, password):
        node_ips = getattr(node, 'node_ips', [self.hostname])
        node_ips.append(self.hostname)
        node_ips = list(set(node_ips))
        if not node.copy_id_to_host(node_ips, user_name, password, authorize_to_root):
            self.log.error("Failed copying ssh id from [{0}] <- [{1}].".format(self, node))
            return False
        self.log.info("Successfully configured passwordless ssh from [{0}] <- [{1}].".format(self, node))

        return True

    def __mark_node_configured(self, task, python_path=None):
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        command = 'cd {0};python -c "import cvavahi;cvavahi.update_avahi_config(\'{1}\',\'{2}\');"'.format(
            python_path, "status", "HyperScaleConfigured")
        if not task.execute(command) == 0:
            return False
        return True

    def set_xattr_cluster_nodes(self, cluster_nodes=None, python_path=None, active_nodes_only=False):
        """
        Gets all the cluster nodes this node is part of, and sets an extended attribute on all data drives.
        :param active_nodes_only:
        :param python_path:
        :type cluster_nodes: list - List of all active nodes in the cluster.
        :return:
        """
        if not self.local_node:
            task = self.__get_executor()
            # For the remote node, call the hs_node.py -set_cluster_nodes_xattr
            if python_path is None:
                python_path = DYNAMIC_DEFINES.python_root_dir

            command = '{0}/HsObject/hs_node.py -set_active_cluster_nodes_xattr'.format(python_path)
            if not task.execute(command) == 0:
                self.log.warning("[{0}] failed on remote node [{1}].".format(command, self.hostname))
                return False

            return True

        if cluster_nodes is None:
            cluster_nodes = get_remote_nodes(active_nodes_only=active_nodes_only)

        if self not in cluster_nodes:
            cluster_nodes.append(self)

        list_of_nodes = ",".join(str(node) for node in cluster_nodes)

        if not self.set_hs_xattr("trusted.hedvigclusternodes", list_of_nodes, python_path=python_path):
            self.log.error("Failed setting extended attribute [{0}] to [{1}].".format("trusted.hedvigclusternodes",
                                                                                      list_of_nodes))
            return False

        self.log.info("Successfully set extended attribute [{0}] to [{1}].".format("trusted.hedvigclusternodes",
                                                                                   list_of_nodes))
        return True

    def set_hs_xattr(self, xattr_name, xattr_value, python_path=None):
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        # Get the block device info for this node
        local_node_xml = self.get_node_block_device_xml(python_path=python_path)

        # Get the block device mount points from the XML
        block_device_info = get_block_device_mount_paths_from_xml(local_node_xml)

        for mount_point, mount_info in block_device_info.items():
            if not mount_info.get("fstype", None) == "xfs":
                continue
            try:
                xattr.set(mount_point, xattr_name, xattr_value)
            except Exception, ex:
                self.log.error(str(ex))
                continue

        return True

    def mark_node_configured(self, python_path=None):
        task = self.__get_executor()

        return self.__mark_node_configured(task, python_path)

    def register_to_cs(self, cs_user, cs_pass, auth_code, cs_host_name=None, python_path=None):
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        if self.local_node:
            # Use local process execution.
            task = utils.Process()
        else:
            # Use remote process execution.
            ssh_conn = hs_ssh.RemoteSSH(self.hostname)
            if ssh_conn.test_connection():
                # SSH to the node worked.
                task = ssh_conn
            else:
                self.log.debug("SSH connection not available to node [{0}].".format(self))
                return False

            # Copy the registertocs.py script from this node to the remote node.  This is temporary until we no longer
            # have SP20 images in the field.  Get the servicepack of the node, if it's less than = SP22, push it.
            try:
                node_sp = 0
                if ssh_conn.execute('commvault status | grep Version') == 0:
                    # Couldn't
                    if 'Version' in ssh_conn.getout():
                        node_sp = ssh_conn.getout().split('.')[1]  # 11.20.0 would be the format.cd

                        if int(node_sp) <= 21:
                            # Take backup of register to cs on node.
                            ssh_conn.execute('mv {0}/registertocs.py {0}/registertocs.py.bak'.format(python_path))

                            # Copy local node version to remote node.
                            ssh_conn.copy_file_to_remote(
                                '{0}/registertocs.py'.format(python_path),
                                '{0}/registertocs.py'.format(python_path), 0775)
            except Exception, err:
                self.log.warning("Could not update registertocs.py with this version.  Will attempt registration "
                                 "anyway.")

        return self.__register_to_cs(task, cs_user, cs_pass, auth_code, cs_host_name, python_path=python_path)

    def __register_to_cs(self, executor, cs_user, cs_pass, auth_code, cs_host_name=None, python_path=None):
        """
        Run cs registration from self over passwordless ssh or locally.

        Get the commserve of this node.
        :param executor:
        :return:
        """
        if not executor:
            self.log.error("Unable to register node with CommServe.")
            return False

        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        if cs_host_name is None:
            process = utils.Process()
            if not process.execute("commvault status | grep 'CommServe Host Name' | cut -d= -f2-") == 0:
                LOG.error("Unable to get CommServe Host Name for registration.")
                return False

            cs_host_name = process.getout().strip()

        if auth_code:
            # auth code form of registration.
            command = '{3}/registertocs.py -n {0} -c {1} -a {2}'.format(
                self.hostname, cs_host_name, auth_code, python_path)
        else:
            command = '{4}/registertocs.py -n {0} -c {1} -u {2} -p {3}'.format(
                self.hostname, cs_host_name, cs_user, cs_pass, python_path)

        if not executor.execute(command) == 0:
            return False

        return True

    def backup_network(self, python_path=None):
        if self.local_node:
            # Use local process execution.
            task = utils.Process()
        else:
            # Use remote process execution.
            ssh_conn = hs_ssh.RemoteSSH(self.hostname)
            if ssh_conn.test_connection():
                # SSH to the node worked.
                task = ssh_conn
            else:
                self.log.debug("SSH connection not available to node [{0}].".format(self))
                return False

        return self.__backup_network(task, python_path)

    def __backup_network(self, executor, python_path=None):
        """
        Hook into cvavahi.py backup_network over passwordless ssh OR local if node is local.
        :param executor:
        :return:
        """
        if not executor:
            self.log.error("Unable to set interface resolution.")
            return False

        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        command = '{0}/cvavahi.py backup_network_to_alldata'.format(python_path)
        if not executor.execute(command) == 0:
            return False

        return True

    def set_private_interface_resolution(self, storage_pool_host, resolve=False):
        """
        For this node, set the storage pool ip as resolvable.  This basically makes entires in /etc/hosts of self.

        Can take a single IP,hostname,or list of either.  If resolve=True, run [arch name2ip] command against each and
        every storage_pool_ip.

        If we're setting up a brand new node, set resolve=False.  Then you pass the IP address directly, and thats
        set for the storage pool IP.  If we're making a new node known to the rest of the cluster, set resolve=True
        which will use arch name2ip to get the IP address for the storage pool.

        :param resolve: bool - If true, attempt to resolve the hostname, if false, use the IP directly.
        :param storage_pool_host: str or list - IP addresses or hostnames which will be set in /etc/hosts
        :return:
        """
        if not isinstance(storage_pool_host, list):
            storage_pool_host = [storage_pool_host]

        if self.local_node:
            # Use local process execution.
            task = utils.Process()
        else:
            # Use remote process execution.
            ssh_conn = hs_ssh.RemoteSSH(self.hostname)
            if ssh_conn.test_connection():
                # SSH to the node worked.
                task = ssh_conn
            else:
                self.log.debug("SSH connection not available to node [{0}].".format(self))
                return False

        return self.__set_private_interface_resolution(task, storage_pool_host, resolve)

    def __set_private_interface_resolution(self, executor, storage_pool_hosts, resolve):
        """
        This is a new remote node.  Check if passwordless ssh is available and use it, if not, use arch.
        :param storage_pool_hosts:
        :return:
        """
        if not executor:
            self.log.error("Unable to set interface resolution.")
            return False

        def exec_flush(exec_obj, cmd):
            if isinstance(exec_obj, utils.Process):
                exec_obj.strout = ''
                exec_obj.strerr = ''
                exec_obj.failed = 0
            return exec_obj.execute(cmd)

        for storage_pool_host in storage_pool_hosts:
            if resolve:
                # With this flag, storage_pool_host is expected to be a hostname and already configured on self.
                # Connect to the storage_pool_host with passwordless ssh, and get its name to ip
                ssh_sp_host = hs_ssh.RemoteSSH(storage_pool_host)
                command = "getent hosts {0}".format(storage_pool_host)
                if not exec_flush(ssh_sp_host, command) == 0:
                    self.log.error("Failed getting storage pool ip on node [{0}] for [{1}].".format(
                        self, storage_pool_host))
                    return False

                # Assume /etc/hosts entry:   <STORAGE POOL IP> <FQDN> <SHORT>
                # Example:   192.168.39.132  smhs23db0102.commvault.com smhs23db0102
                # Example:   192.168.39.132  smhs23db0102sds.commvault.com smhs23db0102sds

                private_ip_addr = ssh_sp_host.getout().split(' ')[0]
                private_ip_addr = private_ip_addr.rstrip()
                self.log.debug("Storage pool {0} resolved to {1} on {2}.".format(
                    storage_pool_host, private_ip_addr, self.hostname))
                private_interface_hostname = storage_pool_host
            else:
                # We don't need to resolve the storage_pool_host (we already know the IP).  So set that.
                private_interface_hostname = self.hostname
                private_ip_addr = storage_pool_host

            # Figure out the private network hostname based on hyperscale version.  1.x uses sds suffix.
            if common.is_hs2dot0():
                private_interface_hostname = private_interface_hostname
            else:
                private_interface_hostname = registertocs.add_suffix_to_hname(private_interface_hostname, "sds")

            private_interface_shortname = private_interface_hostname.split('.', 1)[0]

            # hosts entry
            host_entry = '{0} {1} {2}'.format(private_ip_addr, private_interface_hostname,
                                              private_interface_shortname)

            self.log.info("Checking and adding /etc/host entry if required: {0}".format(host_entry))
            command = "grep -qxF '{0}' /etc/hosts || echo '{0}' >> /etc/hosts".format(host_entry)

            if not exec_flush(executor, command) == 0:
                self.log.error("Failed to set private interface resolvable on remote node [{0}].".node(self))
                return False
            self.log.debug("Successfully set storage pool IP [{0}] resolvable to [{1}] on node [{2}].".format(
                private_ip_addr, private_interface_hostname, self))

        self.log.info("Successfully configured storage pool IP for cluster on [{0}].".format(self))
        return True

    def get_rpm_version(self, package_name):
        if not self.local_node:
            return None

        task = utils.Process()
        task.execute("rpm --queryformat '%{VERSION}' -q " + package_name)
        version = task.getout()
        version = version.rstrip()
        return version

    def login_to_cs(self, cs_username, cs_password, cs_hostname, cs_name=None):
        login_command = '/opt/commvault/Base64/qlogin -cs {0} -u {1} -clp {2}'.format(
            cs_hostname, cs_username, cs_password)

        if cs_name is not None:
            login_command += ' -csn {0}'.format(cs_name)

        login = utils.Process()
        login.execute(login_command)
        if not login.failed == 0:
            self.log.error("Failed logging into CS.\n{0}".format(login.getout()+login.geterr()))
            return False
        return True

    def add_or_update_reg_key(self, key_file, key_name, key_value, python_path=None):
        """
        Set's a reg key either locally or will use passwordless SSH connection to remote node.
        :param key_file: str - Path to .properties file.
        :param key_name: str - Name of the key to set in .properties
        :param key_value: str - Value of key to set.
        :param python_path: str - Root location where we're executing from.
        :return: bool - True or False when successful or failed.
        """
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        task = self.__get_executor()

        command = 'cd {3};python -c "import common;' \
                  'common.setregistryentry(\'{0}\',\'{1}\',\'{2}\');"'.\
            format(key_file, key_name, key_value, python_path)

        if not task.execute(command) == 0:
            self.log.debug("Failed setting [{0} {1}] on [{2}].".format(key_name, key_value, self.hostname))
            return False

        return True

    def delete_reg_key(self, key_file, key_name, python_path=None):
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        task = self.__get_executor()

        command = 'cd {2};python -c "import common;' \
                  'common.deleteregistryentry(\'{0}\',\'{1}\');"'.\
            format(key_file, key_name, python_path)

        if not task.execute(command) == 0:
            self.log.debug("Failed deleting [{0}] from [{1}].".format(key_name, self.hostname))
            return False

        return True

    def get_server_serial_number(self, python_path=None):
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        task = self.__get_executor()

        command = 'cd {0};python -c "import cvavahi;print(cvavahi.get_server_serial_number());"'.format(python_path)
        if not task.execute(command) == 0:
            return False

        serial_num = task.getout().strip()
        return serial_num

    def get_enc_p(self, python_path=None):
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        task = self.__get_executor()

        command = 'cd {0};python -c "import cvupgrade_common;print(cvupgrade_common.get_server_serial_number());"'.\
            format(python_path)
        if not task.execute(command) == 0:
            return False

        serial_num = task.getout().strip()
        return serial_num

    def resize_v_disk(self, python_path=None):
        raise NotImplementedError("This method is not implemented for base node class; please use hs_hv_node or"
                                  " hs_gluster_node only.")

    def prepare_for_addition_to_cluster(self, v_disk_name, hedvig_cluster_pw, python_path=None):
        """
        This is for setting up a new node to be added into an existing cluster.  This takes parameters from the
        existing cluster node, and sets whatever is needed before expanding storage pool.

        :param v_disk_name:
        :param hedvig_cluster_pw:
        :param python_path:
        :return:
        """
        if python_path is None:
            python_path = DYNAMIC_DEFINES.python_root_dir

        # Need to get the serial number of this server
        serial_number = self.get_server_serial_number(python_path=python_path)
        if not serial_number:
            self.log.error("Failed getting serial number of node [{0}].".format(self))
            return False

        # Set the cluster password on the new node
        node_hedvig_cid = cvupgrade_common.encp(hedvig_cluster_pw, serial_number)
        if not self.add_or_update_reg_key(cvmanager_defines.REG_MA_KEY, cvmanager_defines.REG_HEDVIG_CID,
                                          node_hedvig_cid, python_path=python_path):
            self.log.error("Failed setting {0} key on node [{1}].".format(cvmanager_defines.REG_HEDVIG_CID, self))
            return False

        # Set this sVDiskList on the new node(s).  CVMA uses this to mount.
        if not self.add_or_update_reg_key(cvmanager_defines.REG_MA_KEY, cvmanager_defines.REG_V_DISK, v_disk_name,
                                          python_path=python_path):
            self.log.error("Failed setting {0} key on node [{1}].".format(cvmanager_defines.REG_V_DISK, self))
            return False

        # Remote the default password key.
        if not self.delete_reg_key(cvmanager_defines.REG_MA_KEY, cvmanager_defines.REG_HEDVIG_CLUSTER_VER,
                                   python_path=python_path):
            self.log.error("Failed removing {0} key on node [{1}].".format(cvmanager_defines.REG_V_DISK, self))
            return False

        return True

    def __get_executor(self, use_default_image_cred=False):
        """This returns either SSH connection or local utils.Process()"""
        if self.local_node:
            # Use local process execution.
            task = utils.Process()
        else:
            # Use remote process execution.
            if use_default_image_cred:
                ssh_conn = hs_ssh.RemoteSSH(self.hostname, username=cvmanager_defines.DEFAULT_IMAGE_USER,
                                            password=cvmanager_defines.DEFAULT_IMAGE_PASSWORD,
                                            timeout=hs_defines.SSH_TIMEOUT,
                                            banner_timeout=hs_defines.SSH_BANNER_TIMEOUT)
            else:
                ssh_conn = hs_ssh.RemoteSSH(self.hostname)

            if ssh_conn.test_connection():
                # SSH to the node worked.
                task = ssh_conn
            else:
                self.log.debug("SSH connection not available to node [{0}].".format(self))
                raise NoExecutorException
        return task


def get_local_node():
    import hs_gluster_node
    import hs_hv_node

    if common.is_hs2dot0():
        node_obj = hs_hv_node.HVNode()
    else:
        node_obj = hs_gluster_node.GlusterNode()

    return node_obj


def get_remote_nodes(active_nodes_only=False):
    """

    :param active_nodes_only: bool - By default return all nodes which have been attempted to be added into the cluster,
    not just the active nodes.  If this is True, we clear out non-active nodes (adds --preclean to hv_deploy command)
    :return:
    """
    # This is a factory style method, only import them when called to avoid circular import.
    import hs_gluster_node
    import hs_hv_node
    import hs_hv_commands

    if not common.is_hs2dot0():
        node_obj = hs_gluster_node.get_nodes()
    else:
        cluster_name = hs_hv_commands.get_cluster_name_hedvig()
        node_obj = hs_hv_node.get_nodes(cluster_name, active_only=active_nodes_only)

    return node_obj


def get_block_device_mount_paths_from_xml(xml_file):
    """
    Parse the hardware xml file and extract the number of block devices of each type.
    For this discovery, we only care about data drives.  We can ignore SSD os drives or metadata drives.
    """
    disk_map = {}
    for disk in xml_file.findall('MntPath'):
        if not disk.get('type') == 'data':
            continue
        disk_map[disk.get('blkdev')] = disk.attrib
    return disk_map


def collect_remote_nodes(task, active_nodes_only=False):
    from HsObject import hs_stand_alone_node
    """
    Here we will check cluster nodes, the YAML input, and the registry, and concatenate that list of nodes
    for various operations.

    This should never include the current node!!!!
    If the current node hostname OR IP address is found, it will be removed.  Here we're concerned with REMOTE NODES
    only.

    Also, the type of Node is determined by order its detected.  Meaning if the node is a storage pool node, that
    takes precedence over it being in the registry OR from yaml\\command line.  This prevents us from ever treating a
    storage pool node as a standalone node , like command line would.

    Args:
        task: cvmanager_task.Task() - Task() object for the running operation; Query_RPM, Upgrade, Update_node, etc.

    Returns: list - List of hs_node.Node() objects.

    """
    # The returned list of HSObject.hs_node
    node_list = []

    import cvmanager_task
    if not isinstance(task, cvmanager_task.TaskObject):
        raise Exception("collect_remote_nodes() can ONLY operate on proper task objects.  Please see the documentation.")

    task.log.info("Getting the list of storage pool nodes.")
    remote_nodes = get_remote_nodes(active_nodes_only)  # Nodes part of the cluster storage pool; gluster or HV
    node_list.extend(remote_nodes)

    # These are any nodes in the YAML input file.
    task.log.info("Adding nodes specified in the input file or from command line.")
    stand_alone_nodes = list(task.kwargs.get('nodes', []))

    # Get any registry supplied nodes.  Can we remove this and just use YAML?
    task.log.info("Checking registry [sAdditionalNodesForUpgrade] for any nodes added by user.")
    stand_alone_nodes.extend(cvupgrade_common.get_additional_nodes([node.hostname for node in remote_nodes]))

    for node in stand_alone_nodes:
        if str(node) not in [x.hostname for x in node_list]:
            # Assume all nodes from yaml or registry are stand alone, non-clustered nodes; gluster OR 2.0
            node_list.append(hs_stand_alone_node.StandAloneNode(node))

    # These are any nodes to specifically exclude; remove them from the list
    # Make sure this node where the task is running is NOT included.  If you want to run the task on the same node,
    # then explicitly handle it in your task.  This method is getting REMOTE NODES FOR UPGRADE only!!
    task.log.info("Removing any nodes specified to be excluded.")
    excluded_nodes = list(task.kwargs.get('excluded_nodes', []))
    excluded_nodes.extend(cvmanager_utils.ip4_addresses())
    excluded_nodes.append(wrappers.get_hostname())
    node_list = list(x for x in node_list if x.hostname not in excluded_nodes)

    # Final list of nodes to operate on: [m4hcadevb0303.commvault.com(GLUSTER), m4hcadevb0302.commvault.com(GLUSTER)]
    task.log.info("Final list of remote nodes to operate on: [{0}]".format(', '.join(['{0}({1})'.format(
        x.hostname, x.node_type.name) for x in node_list])))

    return node_list


def valid_input():
    if len(sys.argv) == 1:
        return False
    return True


if __name__ == "__main__":
    return_code = 1

    if not valid_input():
        print("In-valid command specified.")
        sys.exit(return_code)

    local_node = HyperScaleNode()

    if sys.argv[1] == '-set_cluster_nodes_xattr':
        if not local_node.set_xattr_cluster_nodes():
            return_code = 0
    elif sys.argv[1] == '-set_active_cluster_nodes_xattr':
        if not local_node.set_xattr_cluster_nodes(active_nodes_only=True):
            return_code = 0
    else:
        print("Unknown command.")

    sys.exit(return_code)
