import hs_node
import hs_defines
import cvupgrade_common
import hs_hv_commands
import hs_ssh
import hs_utils
import cvhedvig
import common

import task_manager.cvmanager_defines as cvmanager_defines


class HVNode(hs_node.HyperScaleNode):
    def __iter__(self):
        yield self.hostname

    def __init__(self, node_hostname=None, *args, **kwargs):
        # Form arguments for node creation.
        kwargs.update({'hostname': node_hostname, 'node_type': hs_defines.NodeTypes.HEDVIG})
        super(HVNode, self).__init__(**kwargs)

        self.cluster_version = self.get_rpm_version('hedvig-cluster')

    def start_cluster(self):
        if self.local_node:
            self.log.info("Starting hedvig cluster on local node [{0}].".format(self.hostname))
            if not cvupgrade_common.start_glusterd():
                self.log.error("Failed starting hedvig cluster on [{0}].".format(self.hostname))
                return False
        else:
            self.log.info("Starting hedvig cluster on remote node [{0}].".format(self.hostname))
            ssh_conn = hs_ssh.RemoteSSH(self.hostname)
            if not ssh_conn.on_remotenode('') == 0:
                self.log.error("Failed starting hedvig cluster on [{0}].".format(self.hostname))
                return False

        return True

    def stop_cluster(self):
        if self.local_node:
            self.log.info("Stopping hedvig cluster on local node [{0}].".format(self.hostname))
            if not cvupgrade_common.stop_hedvig_services():
                self.log.error("Failed hedvig cluster glusterd on [{0}].".format(self.hostname))
                return False
        else:
            self.log.info("Stopping hedvig cluster on remote node [{0}].".format(self.hostname))
            ssh_conn = hs_ssh.RemoteSSH(self.hostname)
            if not ssh_conn.on_remotenode('') == 0:
                self.log.error("Failed stopping hedvig cluster on [{0}].".format(self.hostname))
                return False

        return True

    def restart_cluster(self):
        if self.local_node:
            self.log.info("Restarting hedvig cluster on local node [{0}].".format(self.hostname))
            if not cvupgrade_common.restart_glusterd():
                self.LOG.error("Failed restarting hedvig cluster on [{0}].".format(self.hostname))
                return False
        else:
            self.log.info("Restarting hedvig cluster on remote node [{0}].".format(self.hostname))
            ssh_conn = hs_ssh.RemoteSSH(self.hostname)
            if not ssh_conn.on_remotenode('') == 0:
                self.log.error("Failed restarting hedvig cluster on [{0}].".format(self.hostname))
                return False

        return True

    def resize_v_disk(self, ignore_tenant=False):
        """
        For HyperScale 2.x, at this point in time, we are hard coding erasure code factor of 4+2 and 90% of total size
        <total_cluster_size> * 0.9 * (4 / (4 + 2))

        Because the existing sizes can come back in different measures (GB, TB, PB?), first read those existing sizes
        and convert to BYTES, then perform the calculation, and convert back to GB, TB, or PB? based on the size.

        :return: bool - True if vDisk successfully resized.
        """

        # Get the vDisk.  For HyperScale at this time we will only support single vDisk
        v_disk_name = common.getregistryentry(cvmanager_defines.REG_MA_KEY, cvmanager_defines.REG_V_DISK)

        # Get session to the cluster & collect basic information about cluster, like total size.
        cluster_obj = cvhedvig.CVHedvig()
        cluster_info = cluster_obj.get_cluster_info()

        # Get all cluster virtual disks.
        disks = cluster_obj.hedvig_get_vdisks_desc()

        # Make sure we only have 1 vDisk or fail.  At this point this is a manual config if cluster > 1 vdisk.
        if not len(disks) == 1:
            self.log.error("Detected more than 1 vDisk for cluster [{0}].  Unable to automatically resize".format(
                cluster_obj.cluster
            ))
            return False

        # Make sure this node vDisk read from registry is returned from the cluster.  We're on the right cluster.
        if v_disk_name not in [disk.get('vDiskName', None) for disk in disks]:
            # Didn't find this vDisk in the cluster config.
            self.log.error("Unable to find vDisk [{0}] in cluster [{1}].  Failed re-sizing.".format(
                v_disk_name, cluster_obj.cluster
            ))
            return False

        # Get existing vDisk size.  Since we've confirmed only 1 vDisk present, use index [0]
        existing_v_disk = disks[0]
        self.log.info("Current vDisk [{0}] has size [{1}].".format(v_disk_name, existing_v_disk.get('size', {})))
        existing_v_disk_size = existing_v_disk.get('size', {}).get('value', {})
        existing_v_disk_units = existing_v_disk.get('size', {}).get('units', {})

        # Convert the existing vDisk size down to BYTES based on incoming unit (TB\GB\etc)
        existing_v_disk_size_bytes = hs_utils.convert_size_to_bytes(existing_v_disk_size, existing_v_disk_units)

        # Calculate new vDisk size.
        total_cluster_size = cluster_info.get('capacity', {}).get('total', {})
        if len(total_cluster_size) == 0:
            self.log.error("Failed reading total size of existing cluster, unable to re-size vDisk.")
            return False

        total_cluster_size_value = total_cluster_size.get('value', None)
        if not isinstance(total_cluster_size_value, float) and not isinstance(total_cluster_size_value, int):
            self.log.error("Invalid total cluster size [{0}] received from rest API, unable to re-size vDisk.".format(
                total_cluster_size_value
            ))
            return False

        # Convert the existing cluster size down to BYTES based on incoming unit (TB\GB\etc)
        total_cluster_size_bytes = hs_utils.convert_size_to_bytes(total_cluster_size.get('value'),
                                                                  total_cluster_size.get('units'))

        # Calculate the new vDisk size.
        new_v_disk_size_bytes = 0.0
        try:
            # new_v_disk_size_bytes2 = total_cluster_size_bytes * hs_defines.VDISK_RESERVE_SPACE * (4.0 / (4 + 2))
            new_v_disk_size_bytes = total_cluster_size_bytes * hs_defines.VDISK_RESERVE_SPACE * (1.0 / 1.62)

            if (new_v_disk_size_bytes == 0.0 or new_v_disk_size_bytes == 0) \
                    or not new_v_disk_size_bytes > existing_v_disk_size_bytes:
                # Some arithmetic error occurred, do not set the vDisk size
                if new_v_disk_size_bytes == 0.0 or new_v_disk_size_bytes == 0:
                    self.log.error("Calculated new vDisk size of [{0}] is in-valid.  Unable to set new size.".format(
                        hs_utils.format_bytes(new_v_disk_size_bytes, 'GB')
                    ))
                else:
                    self.log.error("Calculated new vDisk size of [{0}] is not larger than current vDisk size [{1}]."
                                   "Unable to set new size.".format(
                                    hs_utils.format_bytes(new_v_disk_size_bytes),
                                    hs_utils.format_bytes(existing_v_disk_size_bytes)))
                return False
        except Exception, err:
            self.log.error("Exception calculating vDisk size.")
            return False

        # All checks passed, setting the disk size.
        new_v_disk_size_formatted, new_v_disk_size_unit = hs_utils.format_bytes(new_v_disk_size_bytes, 'GB')
        new_v_disk_size_formatted = round(new_v_disk_size_formatted)
        new_v_disk_params = {'units': new_v_disk_size_unit, 'value': new_v_disk_size_formatted}

        self.log.info("Resizing existing vDisk [{0}] of size [{1}] to new size [{2}].".format(
            v_disk_name, existing_v_disk.get('size', {}), new_v_disk_params
        ))

        # Call the REST API to perform the tenant.
        if not cluster_obj.resize_tenant(existing_v_disk.get('tenant', ''), new_v_disk_size_formatted,
                                         new_v_disk_size_unit):
            self.log.error("Failed resizing the tenant.")
            if not ignore_tenant:
                return False

        # Call the REST API to perform the resize.
        if not cluster_obj.resize_v_disk(v_disk_name, new_v_disk_size_formatted, new_v_disk_size_unit):
            self.log.error("Failed resizing vDisk.")
            return False

        # Re-load and confirm the resize happened.  Make sure to convert to the units we changed to for the check.
        reloaded_disks = cluster_obj.hedvig_get_vdisks_desc()
        # Format the expected size into what hedvig does and with their precision.
        expected_new_size, expected_new_units = hs_utils.format_bytes(new_v_disk_size_bytes)
        expected_new_size = round(expected_new_size, hs_defines.REST_API_PRECISION)

        if not reloaded_disks[0].get('size', {}) == {'units': expected_new_units, 'value': expected_new_size}:
            self.log.error("Resized vDisk actual value [{0}] not equal to expected value [{1}].".format(
                reloaded_disks[0].get('size', {}), {'units': expected_new_units, 'value': expected_new_size}
            ))
            return False

        self.log.info("Successfully resized vDisk [{0}].".format(v_disk_name))
        return True


def get_nodes(cluster_name=None, active_only=False):
    node_obj_list = []

    for node in hs_hv_commands.get_remote_nodes_hedvig(cluster_name, active_only=active_only):
        # Nodes which are part of the storage pool
        node_obj = HVNode(node)
        node_obj.cluster_name = cluster_name
        node_obj_list.append(node_obj)

    return node_obj_list


def add_nodes_to_cluster(nodes):
    return hs_hv_commands.add_nodes_to_cluster(nodes)