import os
from enum import Enum
import re

from task_manager import cvmanager_catalog
from task_manager import cvmanager_defines
from task_manager import cvmanager_logging
import wrappers
import common
import cvupgrade_common


LOG = cvmanager_logging.get_log(cvmanager_defines.LOG_FILE_NAME)
DEBUG_SCRIPT = False


class CommandType(Enum):
    BASIC = 1
    SERVICE = 2
    PASSWORD_REQ = 3


class Command(object):
    def __str__(self):
        return self.hedvig_command

    def __init__(self, command, command_type):
        self.hedvig_command = command
        self.type = command_type


class Commands(object):
    # All needed hedvig commands defined here.
    __hv_deploy = r'/opt/hedvig/bin/hv_deploy'
    
    SHOW_ALL_CLUSTERS = Command(__hv_deploy + " --show_all_clusters",
                                CommandType.BASIC)

    GET_CLUSTER_NODES = Command(__hv_deploy + " --cluster_name {0} --show_nodes | grep hostname | cut -d ' ' -f2",
                                CommandType.BASIC)

    GET_ACTIVE_CLUSTER_NODES = Command(__hv_deploy + " --cluster_name {0} --show_nodes --preclean | grep hostname "
                                                     "| cut -d ' ' -f2", CommandType.BASIC)

    ADD_NODE_TO_CLUSTER = Command(__hv_deploy + " --add_allinone_node {0} --cluster_name {1} --preclean",
                                  CommandType.BASIC)


class HedvigCommand(object):
    def __str__(self):
        return 'HedvigCommand'

    def __del__(self):
        # clean up the script and output files.
        if DEBUG_SCRIPT:
            # leave the generated files behind
            return

        if os.path.exists(self.script_file.file_path):
            os.unlink(self.script_file.file_path)

        if os.path.exists(self.output_file.file_path):
            os.unlink(self.output_file.file_path)

    def __init__(self, command, *args, **kwargs):
        self.command = command
        self.sysdig = False

        # If we have arguments, pass into the command
        if len(args) > 0:
            self.command.hedvig_command = self.command.hedvig_command.format(*args)

        # Load the catalog handler
        catalog = cvmanager_catalog.Catalog()

        self.script_file = catalog.get_script_file(cvmanager_defines.HV_CMD_PREFIX, cvmanager_defines.HV_CMD_SUFFIX)
        self.output_file = catalog.get_script_file(cvmanager_defines.HV_CMD_PREFIX, cvmanager_defines.HV_LOG_SUFFIX,
                                                   self.script_file.name)

        # Prepare the command
        if self.command.type == CommandType.BASIC:
            prepared_cmd = self.prepare_basic()
        elif self.command.type == CommandType.SERVICE:
            prepared_cmd = self.prepare_service()
        elif self.command.type == CommandType.PASSWORD_REQ:
            prepared_cmd = self.prepare_password()
        else:
            # Unknown command type, raise exception
            raise Exception("Unknown command type for command: {0}".format(self.command.hedvig_command))

        # Create the script
        self.create_command_script(prepared_cmd)

        self.output = None

    def prepare_basic(self):
        cmd = 'sudo -u admin  bash -cx \"export HV_PUBKEY=1; {1} &>>{0}\" &>> {0}'.format(
            self.output_file.file_path, self.command.hedvig_command)
        return cmd

    def prepare_service(self):
        cmd = 'sudo -u admin  bash -cx \"export HV_PUBKEY=1; export HV_UPGRADE_HPOD=1; ' \
              'export HV_UPGRADE_PAGES=1; {1} &>>{0}\" &>> {0}'.format(
               self.output_file.file_path, self.command.hedvig_command)
        return cmd

    def prepare_password(self):
        cmd = 'sudo -u admin  bash -cx \"export HV_PUBKEY=1; export HV_ROOTPASS={2}; ' \
              'export HV_UPGRADE_PAGES=1; {1} &>>{0}\" &>>{0}'.format(
              self.output_file.file_path, self.command.hedvig_command, self.root_password)
        return cmd

    def create_command_script(self, prepare_command):
        LOG.debug("Executing cmd: " + prepare_command)
        script_data = [
            '#!/bin/bash\n',
            str(prepare_command) + str('\n'),
            'ret=$?\n',
            'exit ${ret}\n'
        ]
        self.script_file.write(script_data, cvmanager_catalog.FileType.MULTILINE)
        wrappers.chmod(self.script_file.file_path, 0755)
        wrappers.chmod(self.output_file.file_path, 0777)

    def run(self):
        if self.sysdig:
            common.start_sysdig()
        ret = wrappers.waitsystem(self.script_file.file_path)
        if DEBUG_SCRIPT:
            LOG.debug("script rc [{0}]".format(ret))

        dest_logdir = os.path.join(common.get_cvlt_logdir(), cvupgrade_common.debug_logdir)
        dest_logpath = os.path.join(dest_logdir, "hvcmd.log")
        cmd = "cat " + self.output_file.file_path + " >> " + dest_logpath
        wrappers.waitsystem(cmd)

        if self.sysdig:
            common.stop_sysdig()

        if not ret == 0:
            LOG.error("Script [{0}] failed with return code [{1}], Please check {2}".format(self.script_file.file_path,
                                                                                            ret,
                                                                                            dest_logpath))
            return False

        self.output_file.read_file()
        self.output = self.output_file.lines[1:]  # [0] contains date

        return True


# Exported APIs
def get_cluster_name_hedvig():
    hv_cmd = HedvigCommand(Commands.SHOW_ALL_CLUSTERS)
    hv_cmd.run()

    # Only care about the first file....
    cluster_info = filter(lambda x: x.startswith('Cluster name: '), hv_cmd.output)
    if not len(cluster_info) == 1:
        raise Exception("Unable to determine cluster info, please check log file: {0}".format(
            hv_cmd.output_file.file_path))
    cluster_name = (re.split(':|-', cluster_info[0])[1].strip()).split('\n')[0]

    del hv_cmd
    return cluster_name


def get_cluster_nodes_hedvig(cluster_name, active_only=False):
    if active_only:
        # includes the --preclean option.  If this is specified, only active node are returned. Non-active are removed.
        cmd = Commands.GET_ACTIVE_CLUSTER_NODES
    else:
        # Gets all cluster nodes, active or not.
        cmd = Commands.GET_CLUSTER_NODES

    hv_cmd = HedvigCommand(cmd, (cluster_name))
    hv_cmd.run()

    node_list = []
    for line in hv_cmd.output:
        line = (line.strip()).split('\n')[0]
        if not line == '' and not line.startswith("+"):
            node_list.append(line)

    del hv_cmd
    return node_list


def get_remote_nodes_hedvig(cluster_name=None, active_only=False):
    if cluster_name is None:
        cluster_name = get_cluster_name_hedvig()
    LOG.info("Cluster name: " + cluster_name)
    cluster_nodes = get_cluster_nodes_hedvig(cluster_name, active_only=active_only)

    hostname = wrappers.get_hostname()
    remote_nodes = []
    for node in cluster_nodes:
        if node != hostname:
            remote_nodes.append(node)
    return remote_nodes


def add_node_to_cluster(node_to_add, cluster_name=None):
    if cluster_name is None:
        cluster_name = get_cluster_name_hedvig()

    hv_cmd = HedvigCommand(Commands.ADD_NODE_TO_CLUSTER, *(node_to_add, cluster_name))
    hv_cmd.sysdig = True
    if not hv_cmd.run():
        LOG.error("Failed adding node(s) [{0}] to cluster.".format(node_to_add))
        return False
    del hv_cmd
    return True


def add_nodes_to_cluster(list_of_nodes, cluster_name=None):
    nodes_to_add = " ".join(list_of_nodes)

    return add_node_to_cluster(nodes_to_add, cluster_name=cluster_name)
