# Python imports
import psutil
import xattr

# Project imports
import hs_block
import hs_hv_node
import hs_node
import hs_defines
import hs_utils
from task_manager import cvmanager_logging
from task_manager import cvmanager_defines

LOG = cvmanager_logging.get_log(cvmanager_defines.LOG_FILE_NAME)


class HVBlock(hs_block.HyperScaleBlock):
    def __init__(self, block_node_list, **kwargs):
        # Form arguments for block creation if needed.
        if kwargs.get('from_xattr', False):
            block_node_list = self.get_cluster_from_xattr()

        super(HVBlock, self).__init__(block_node_list, **kwargs)

    def get_cluster_from_xattr(self):
        """
        Gets a list of nodes using the local xattr.
        :return:
        """
        str_of_nodes = ''
        list_of_nodes = []

        # Find all the mount points which have our specific xattr set 'trusted.hedvigclusternodes'
        mount_points_with_xattr = hs_utils.get_xattr_mounts(hs_defines.XATTR_CLUSTER_NODES)
        if len(mount_points_with_xattr) == 0:
            # No mount points with the xattr mean its not set and we can't create a cluster this way.
            raise Exception("Missing [{0}] xattr.".format(hs_defines.XATTR_CLUSTER_NODES))

        # Read the value of the xattr; try all the mounts in case the value is missing from one of them.
        try:
            for m in mount_points_with_xattr:
                str_of_nodes = xattr.getxattr(m, hs_defines.XATTR_CLUSTER_NODES)
                if str_of_nodes.strip() == '':
                    continue
                LOG.debug("Extended attribute at path [{0}] : name [{1}], value [{2}] ".format(
                    m, hs_defines.XATTR_CLUSTER_NODES, str_of_nodes))
                break
            else:
                # Did not find any value on any of the mounts.
                raise Exception("Missing [{0}] xattr.".format(hs_defines.XATTR_CLUSTER_NODES))
        except Exception as ex:
            LOG.error(str(ex))
            raise

        # Create the node objects for each node.
        for node in str_of_nodes.split(","):
            list_of_nodes.append(hs_hv_node.HVNode(node))

        if len(list_of_nodes) == 0:
            # something went very wrong getting the nodes.
            raise Exception("Unable to get cluster nodes using xattr [{0}].".format(hs_defines.XATTR_CLUSTER_NODES))

        return list_of_nodes

    def add_nodes(self, nodes, *args, **kwargs):
        for node in nodes:
            if isinstance(node, hs_hv_node.HVNode) or isinstance(node, hs_node.HyperScaleNode):
                self.nodes.append(node)
            else:
                self.nodes.append(hs_hv_node.HVNode(node, *args, **kwargs))
