import hs_node
import hs_defines
import cvupgrade_common
import wrappers
import cvovirtsdk
import cvcluster


class GlusterNode(hs_node.HyperScaleNode):
    """ This is a gluster node part of a gluster installation.
    These methods are the gluster specific operations that occur on a node.

    """
    def __init__(self, node_hostname):
        # Form arguments for node creation.
        kwargs = {'hostname': node_hostname, 'node_type': hs_defines.NodeTypes.GLUSTER}
        super(GlusterNode, self).__init__(**kwargs)

    def start_cluster(self):
        if self.local_node:
            self.log.info("Starting glusterd on local node [{0}].".format(self.hostname))
            if not cvupgrade_common.start_glusterd():
                self.log.error("Failed starting glusterd on [{0}].".format(self.hostname))
                return False
        else:
            self.log.info("Starting glusterd on remote node [{0}].".format(self.hostname))
            ret = cvupgrade_common.on_remotenode("start_glusterd", wrappers.get_hostname(), self.hostname)
            if not ret == 0:
                self.log.error("Failed starting glusterd on [{0}].".format(self.hostname))
                return False

        return True

    def stop_cluster(self):
        if self.local_node:
            self.log.info("Stopping glusterd on local node [{0}].".format(self.hostname))
            if not cvupgrade_common.stop_gluster_processes():
                self.log.error("Failed stopping glusterd on [{0}].".format(self.hostname))
                return False
        else:
            self.log.info("Stopping glusterd on remote node [{0}].".format(self.hostname))
            ret = cvupgrade_common.on_remotenode("stop_glusterd", wrappers.get_hostname(), self.hostname)
            if not ret == 0:
                self.log.error("Failed stopping glusterd on [{0}].".format(self.hostname))
                return False

        return True

    def restart_cluster(self):
        if self.local_node:
            self.log.info("Restarting glusterd on local node [{0}].".format(self.hostname))
            if not cvupgrade_common.restart_glusterd():
                self.LOG.error("Failed restarting glusterd on [{0}].".format(self.hostname))
                return False
        else:
            self.log.info("Restarting glusterd on remote node [{0}].".format(self.hostname))
            ret = cvupgrade_common.on_remotenode("restart_glusterd", wrappers.get_hostname(), self.hostname)
            if not ret == 0:
                self.log.error("Failed restarting glusterd on [{0}].".format(self.hostname))
                return False

        return True

    def is_running_he(self, ovirt_he):
        # Check if vm is running on node
        ret = cvovirtsdk.is_vm_on_host(ovirt_he.host, ovirt_he.user, cvupgrade_common.decp(ovirt_he.ename),
                                       "HostedEngine", self.hostname)
        if ret == 0:
            return True

        return False

    def stop_he_vm(self):
        if self.local_node:
            if not cvupgrade_common.stop_he_vm():
                self.log.error("There was an error while stopping Hosted Engine VM. Try stopping manually and rerun "
                               "upgrade.")
                return False
        else:
            self.log.info("Stopping hosted engine vm on remote node [{0}].".format(self.hostname))
            ret = cvupgrade_common.on_remotenode("stop_hosted_engine", wrappers.get_hostname(), self.hostname)
            if not ret == 0:
                return False
        return True

    def fix_gluster_vol_size(self):
        if self.local_node:
            if not cvupgrade_common.fix_gluster_vol_size() == 0:
                self.log.error("There was an error while checking and adjusting gluster volume size.")
                return False
        else:
            self.log.info("Fixing gluster volume size on remote node [{0}].".format(self.hostname))
            ret = cvupgrade_common.on_remotenode("fix_gluster_volsize", wrappers.get_hostname(), self.hostname)
            if not ret == 0:
                return False
        return True

    def reflect_vol_size(self):
        if self.local_node:
            ret = cvupgrade_common.reflect_volsize()
            if not ret == 0:
                return False
        return True

    def start_services(self):
        # start_services method for gluster based nodes only.
        if self.local_node:
            if not cvupgrade_common.start_services():
                self.log.error("Failed starting services on [{0}].".format(self.hostname))
                return False
        else:
            self.log.info("Starting services on remote node [{0}].".format(self.hostname))
            ret = cvupgrade_common.on_remotenode("startservices", wrappers.get_hostname(), self.hostname)
            if ret:
                self.log.error("Failed starting services on remote node [{0}].".format(self.hostname))
                return False

        return True


def get_nodes():
    """
    get_nodes may not get all nodes without glusterd on driving node. Wait for glusterd to start
    completely, other wise, post upgrade steps like service start etc will not happen on remote nodes
    """
    node_obj = []
    cvupgrade_common.start_glusterd()

    # TODO - Wait for glusterd to start.

    # Get\Check for gluster nodes first.  This command is removing the SDS; FYI!
    for node in cvcluster.get_remote_nodes():
        # Nodes which are part of the storage pool
        node_obj.append(GlusterNode(node))

    return node_obj
