# Python imports

# Project imports
import hs_block
import hs_gluster_node
import volfile
import cvmanager_logging
import cvarchhelper
from task_manager import cvmanager_defines

LOG = cvmanager_logging.get_log(cvmanager_defines.LOG_FILE_NAME)


class GlusterBlock(hs_block.HyperScaleBlock):
    def __init__(self, block_node_list, **kwargs):
        # Form arguments for block creation if needed.
        super(GlusterBlock, self).__init__(block_node_list, **kwargs)

    def add_nodes(self, nodes):
        # Create and add the node objects for this block.
        for node in nodes:
            # All gluster blocks are derived from the storage pool volume files, get the real hostname here using arch.
            gluster_node = cvarchhelper.get_remote_hostname(node)
            self.nodes.append(hs_gluster_node.GlusterNode(gluster_node))

    def restart_cluster_all_nodes(self):
        for node in self.nodes:
            if not node.restart_cluster():
                return False
        return True

    def start_cluster_all_nodes(self):
        for node in self.nodes:
            if not node.start_cluster():
                return False
        return True


def get_gluster_blocks(storage_pool_name):
    vol_file_obj = volfile.get_cluster_volume_graph(storage_pool_name)

    blocks = get_hs_blocks_from_volume_file(vol_file_obj)

    block_obj = []
    for block in blocks:
        block_obj.append(GlusterBlock(block))

    return block_obj


def get_hs_blocks_from_volume_file(volume_file):
    # start with the last translator in the file as it's the root.  this is not true recursion as we have process
    # translators differently to build our blocks.
    all_blocks = []
    for sv_name, translator in volume_file.graph.items():
        if not translator.type == r'cluster/disperse':
            continue
        '''<translator> object is of volfile.Translator()
        translator = {Translator} <Translator HyperScaleStoragePool-disperse-0>
             dumped = {bool} False
             name = {str} 'HyperScaleStoragePool-disperse-0'
             options = {dict} <type 'dict'>: {'shd-max-threads': '1', 'redundancy': '2'}
             subvolumes = {dict} <type 'dict'>: {'HyperScaleStoragePool-client-4': 
             <Translator HyperScaleStoragePool-client-4>, 'HyperScaleStoragePool-client-5': 
             <Translator HyperScaleStoragePool-client-5>, 'HyperScaleStoragePool-client-2': 
             <Translator HyperScaleStoragePool-client-2>, 'HyperScaleStoragePool-client-3': 
             <Translator HyperScaleStoragePool-client-3>, 'HyperScaleStoragePool-client-0': 
             <Translator HyperScaleStoragePool-client-0>, 'HyperScaleStoragePool-client-1': 
             <Translator HyperScaleStoragePool-client-1>}
             type = {str} 'cluster/disperse'
             uuid = {UUID} 8944796d-0a18-410b-a0e3-9d70fc7e15cd

        <translator.subvolumes> is a dictionary of Translator objects for each of the subvolumes.  Iterate through all
        subvolumes.

        Example:
        translator.subvolumes[0] = Translator(HyperScaleStoragePool-client-4)
        'HyperScaleStoragePool-client-4' (140673529753296) = {Translator} <Translator HyperScaleStoragePool-client-4>
         dumped = {bool} False
         name = {str} 'HyperScaleStoragePool-client-4'
         options = {dict} <type 'dict'>: {'username': '29c31382-0910-43fc-ba16-4bd73f6b4272', 
         'transport.socket.keepalive-interval': '2', 'transport.tcp-user-timeout': '0', 'send-gids': 'true', 
         'transport.socket.keepalive-time': '20', 'remote-host': 'm4hcadevb0102sds.commvault.com', 
         'transport-type': 'tcp', 'remote-subvolume': '/ws/disk2/ws_brick', 'transport.socket.keepalive-count': '9', 
         'password': '6cd97433-bf27-47a2-8036-0186379cf278', 'ping-timeout': '42', 'transport.address-family': 'inet'}
         subvolumes = {dict} <type 'dict'>: {}
         type = {str} 'protocol/client'
         uuid = {UUID} 25f70c63-10fe-435d-9ce2-c9653d6deaa0

        '''

        # Get all the remote host options from each of the Translator subvolumes.
        node_list = [client_translator.options['remote-host']
                     for sv_name, client_translator in translator.subvolumes.items()]

        ''' node_list = list containing all remote-host options for this specific translator:
                ['m4hcadevb0101sds.commvault.com', 'm4hcadevb0101sds.commvault.com', 'm4hcadevb0102sds.commvault.com', 
                'm4hcadevb0102sds.commvault.com', 'm4hcadevb0103sds.commvault.com', 'm4hcadevb0103sds.commvault.com']
        '''

        # Converting to a set removes any duplicates.
        node_set = set(node_list)

        # Save this as a potential block.
        all_blocks.append(list(node_set))

    '''
    This is a list of lists [[]], each sub-list is all subvolumes per disperse group.
    all_blocks = {list} <type 'list'>: [
        ['m4hcadevb0103sds.commvault.com', 'm4hcadevb0102sds.commvault.com', 'm4hcadevb0101sds.commvault.com'], 
        ['m4hcadevb0103sds.commvault.com', 'm4hcadevb0102sds.commvault.com', 'm4hcadevb0101sds.commvault.com'] ]
            0 = {list} <type 'list'>: ['m4hcadevb0103sds.commvault.com', 'm4hcadevb0102sds.commvault.com', 
                                        'm4hcadevb0101sds.commvault.com']
            1 = {list} <type 'list'>: ['m4hcadevb0103sds.commvault.com', 'm4hcadevb0102sds.commvault.com', 
                                        'm4hcadevb0101sds.commvault.com']
        __len__ = {int} 2
    '''

    # Do some magic to figure out the blocks.  Essentially remove any duplicates found across the disperse groups.
    blocks = [list(x) for x in set(tuple(x) for x in all_blocks)]

    return blocks
