# Python imports
import hashlib
import six
import abc

# Project imports
from HsObject import hs_hv_node
from HsObject import hs_node


@six.add_metaclass(abc.ABCMeta)
class HyperScaleBlock(object):
    def __str__(self):
        return "[{0}]-[UID: {1}]".format(",".join([str(node) for node in self.nodes]), self.block_id)

    def __init__(self, block_node_list, *args, **kwargs):
        """HSBlock is a Commvault style HyperScale block, wherein nodes are added in groups of 3 for redundancy
        purposes.
        Args:
            block_list (list): List of node hostname(s) that form a block.
            *args:
            **kwargs:
        """
        self.nodes = []

        self.block_id = hashlib.md5(','.join([str(node) for node in block_node_list])).hexdigest()

        self.add_nodes(block_node_list, *args, **kwargs)

    @abc.abstractmethod
    def add_nodes(self, nodes, *args, **kwargs):
        # TODO: Default node implementation?
        pass

    @abc.abstractmethod
    def get_cluster_from_xattr(self):
        """
        Creates a cluster object using the trusted.hedvigclusternodes extended attribute set on XFS mounts during
        initial HyperScale deployment.
        :return:
        """
        raise NotImplementedError("This method is not implemented for hs_block base class; please use hs_hv_block only")
    
    def stop_services_on_all_nodes(self):
        """Stops Commvault services on all nodes."""
        for node in self.nodes:
            if not node.stop_services():
                return False
        return True

    def stop_gluster_on_all_nodes(self):
        """Stops Commvault services on all nodes."""
        for node in self.nodes:
            if not node.stop_cluster():
                return False
        return True

    def start_services_on_all_nodes(self):
        """Starts Commvault services on all nodes."""
        for node in self.nodes:
            if not node.start_services():
                return False
        return True

    def add_or_update_reg_key(self, key_file, key_name, key_value, python_path=None, include_local_node=True):
        """
        Adds or updates a reg key on all nodes in the cluster, local OR remote.

        :param include_local_node: bool - True to include local where executing, False to be remote only.
        :param key_file: str - Path to .properties file.
        :param key_name: str - Name of property
        :param key_value: str - Value of property
        :param python_path: str - Path for python executable in case running in debug mode.
        :return: bool - True if successful, False any failure.
        """
        if include_local_node:
            local_node = hs_node.HyperScaleNode()
            if local_node.hostname not in [node.hostname for node in self.nodes]:
                self.nodes.append(local_node)

        for node in self.nodes:
            if not node.add_or_update_reg_key(key_file, key_name, key_value, python_path):
                return False
        return True

    def delete_reg_key(self, key_file, key_name, python_path=None, include_local_node=True):
        """
        Deletes a reg key on all nodes in the cluster.

        :param include_local_node: bool - True to include local where executing, False to be remote only.
        :param key_file: str - Path to .properties file.
        :param key_name: str - Name of property
        :param key_value: str - Value of property
        :param python_path: str - Path for python executable in case running in debug modes.
        :return: bool - True if successful, False any failure.
        """
        if include_local_node:
            local_node = hs_node.HyperScaleNode()
            if local_node.hostname not in [node.hostname for node in self.nodes]:
                self.nodes.append(local_node)

        for node in self.nodes:
            if not node.delete_reg_key(key_file, key_name, python_path):
                return False
        return True

    def configure_passwordless_ssh(self, user_name=None, password=None, over_write=False):
        """
        This will configure passwordless ssh in the cluster block.
        Assuming 3 node block, start on first node, configure bi-directional: 1 -> 2, 1 -> 3,
        :param gen_key_only_to_root:
        :param user_name:
        :param password:
        :param over_write:
        :return:
        """
        # local_node is wherever this script is running; get the SSH conn working FROM this node out.  This is a
        # pre-requisite.
        local_node = hs_hv_node.HVNode()
        if local_node.hostname not in [node.hostname for node in self.nodes]:
            self.nodes.append(local_node)

        def __ssh_config(n, u, p):
            for current_node in self.nodes:
                if not n.configure_node_passwordless_ssh(current_node, u, p, over_write, block_config=True,
                                                         authorize_to_root=True):
                    return False
            return True

        for node in self.nodes:
            if not __ssh_config(node, user_name, password):
                return False

        return True
