# Python imports
import sys
import uuid


# noinspection DuplicatedCode
class VolumeFile(object):
    def __init__(self, vol_name):
        # It's important to keep track of the bottom, since it's read from bottom up; hence, self.last
        self.name = vol_name
        self.path = r'/var/lib/glusterd/vols/{0}/trusted-{0}.tcp-fuse.vol'.format(vol_name)

        # self.graph is the FULL graph, do not modify.  self.last is the bottom translator
        self.graph, self.last = self.load()

    def load(self):
        # When the file is loaded, its loaded from top down, so a parent will ALWAYS exist to any child.
        try:
            lines = []
            with open(self.path, 'r') as fp:
                lines = fp.readlines()

            all_translators = {}
            translator = None
            last_translator = None

            for line in lines:
                text = line.split()
                if not len(text):
                    continue

                translator_key = text[0]

                if translator_key == 'volume':
                    if translator:
                        raise RuntimeError("Nested volume.")
                    translator = Translator(text[1])
                    continue
                if not translator:
                    raise RuntimeError("Text outside volume.")
                if translator_key == 'type':
                    translator.type = text[1]
                    continue
                if translator_key == 'option':
                    translator.options[text[1]] = ''.join(text[2:])
                    continue
                if translator_key == 'subvolumes':
                    for sv in text[1:]:
                        translator.subvolumes[all_translators[sv].name] = all_translators[sv]
                    continue
                if translator_key == 'end-volume':
                    # Done reading the translator information
                    all_translators[translator.name] = translator
                    last_translator = translator
                    translator = None
                    continue
                raise RuntimeError("Unrecognized keyword {0}".format(translator_key))

            if translator:
                raise RuntimeError("Unclosed volume definition.")

            return all_translators, last_translator
        except Exception, err:
            raise err

    def print_y(self, root_graph=None, last_translator=None, stream=sys.stdout):
        # For printing the output
        if root_graph is None:
            root_graph = self.graph

        if last_translator is None:
            last_translator = self.last

        # Print the graph out to console.
        for uid, sv in last_translator.subvolumes.items():
            if not sv.dumped:
                self.print_y(root_graph, sv, stream)
                print("")
                sv.dumped = True
        print("volume %s" % last_translator.name)
        print("    type %s" % last_translator.type)
        for k, v in last_translator.options.items():
            print("    option %s %s" % (k, v))
        if last_translator.subvolumes.items():
            print("    subvolumes %s" % ''.join(
                [sv.name for uid, sv in last_translator.subvolumes.items()]))
        print("end-volume")


class Translator(object):
    def __init__(self, name):
        self.name = name
        self.type = ""
        self.options = {}
        self.subvolumes = {}
        self.dumped = False  # Used for printing output
        self.uuid = uuid.uuid4()  # Get a unique ID of this translator.

    def __repr__(self):
        return "<Translator %s>" % self.name


def get_hs_blocks_from_volume_file(volume_file):
    # start with the last translator in the file as it's the root.  this is not true recursion as we have process
    # translators differently to build our blocks.
    all_blocks = []
    for sv_name, translator in volume_file.graph.items():
        if not translator.type == r'cluster/disperse':
            continue
        '''<translator> object is of volfile.Translator()
        translator = {Translator} <Translator HyperScaleStoragePool-disperse-0>
             dumped = {bool} False
             name = {str} 'HyperScaleStoragePool-disperse-0'
             options = {dict} <type 'dict'>: {'shd-max-threads': '1', 'redundancy': '2'}
             subvolumes = {dict} <type 'dict'>: {'HyperScaleStoragePool-client-4': 
             <Translator HyperScaleStoragePool-client-4>, 'HyperScaleStoragePool-client-5': 
             <Translator HyperScaleStoragePool-client-5>, 'HyperScaleStoragePool-client-2': 
             <Translator HyperScaleStoragePool-client-2>, 'HyperScaleStoragePool-client-3': 
             <Translator HyperScaleStoragePool-client-3>, 'HyperScaleStoragePool-client-0': 
             <Translator HyperScaleStoragePool-client-0>, 'HyperScaleStoragePool-client-1': 
             <Translator HyperScaleStoragePool-client-1>}
             type = {str} 'cluster/disperse'
             uuid = {UUID} 8944796d-0a18-410b-a0e3-9d70fc7e15cd

        <translator.subvolumes> is a dictionary of Translator objects for each of the subvolumes.  Iterate through all
        subvolumes.

        Example:
        translator.subvolumes[0] = Translator(HyperScaleStoragePool-client-4)
        'HyperScaleStoragePool-client-4' (140673529753296) = {Translator} <Translator HyperScaleStoragePool-client-4>
         dumped = {bool} False
         name = {str} 'HyperScaleStoragePool-client-4'
         options = {dict} <type 'dict'>: {'username': '29c31382-0910-43fc-ba16-4bd73f6b4272', 
         'transport.socket.keepalive-interval': '2', 'transport.tcp-user-timeout': '0', 'send-gids': 'true', 
         'transport.socket.keepalive-time': '20', 'remote-host': 'm4hcadevb0102sds.commvault.com', 
         'transport-type': 'tcp', 'remote-subvolume': '/ws/disk2/ws_brick', 'transport.socket.keepalive-count': '9', 
         'password': '6cd97433-bf27-47a2-8036-0186379cf278', 'ping-timeout': '42', 'transport.address-family': 'inet'}
         subvolumes = {dict} <type 'dict'>: {}
         type = {str} 'protocol/client'
         uuid = {UUID} 25f70c63-10fe-435d-9ce2-c9653d6deaa0

        '''

        # Get all the remote host options from each of the Translator subvolumes.
        node_list = [client_translator.options['remote-host']
                     for sv_name, client_translator in translator.subvolumes.items()]

        ''' node_list = list containing all remote-host options for this specific translator:
                ['m4hcadevb0101sds.commvault.com', 'm4hcadevb0101sds.commvault.com', 'm4hcadevb0102sds.commvault.com', 
                'm4hcadevb0102sds.commvault.com', 'm4hcadevb0103sds.commvault.com', 'm4hcadevb0103sds.commvault.com']
        '''

        # Converting to a set removes any duplicates.
        node_set = set(node_list)

        # Save this as a potential block.
        all_blocks.append(list(node_set))

    '''
    This is a list of lists [[]], each sub-list is all subvolumes per disperse group.
    all_blocks = {list} <type 'list'>: [
        ['m4hcadevb0103sds.commvault.com', 'm4hcadevb0102sds.commvault.com', 'm4hcadevb0101sds.commvault.com'], 
        ['m4hcadevb0103sds.commvault.com', 'm4hcadevb0102sds.commvault.com', 'm4hcadevb0101sds.commvault.com'] ]
            0 = {list} <type 'list'>: ['m4hcadevb0103sds.commvault.com', 'm4hcadevb0102sds.commvault.com', 
                                        'm4hcadevb0101sds.commvault.com']
            1 = {list} <type 'list'>: ['m4hcadevb0103sds.commvault.com', 'm4hcadevb0102sds.commvault.com', 
                                        'm4hcadevb0101sds.commvault.com']
        __len__ = {int} 2
    '''

    # Do some magic to figure out the blocks.  Essentially remove any duplicates found across the disperse groups.
    blocks = [list(x) for x in set(tuple(x) for x in all_blocks)]

    return blocks


if __name__ == "__main__":
    vf = VolumeFile(sys.argv[1])
    graph, last = vf.load()
    vf.print_y(graph, last)

    print("\nDetected HyperScale Blocks:")
    from pprint import pprint
    blocks = get_hs_blocks_from_volume_file(vf)
    for idx, block in enumerate(blocks, 1):
        print("Block {0}: ".format(idx))
        pprint(block)

