#!/bin/bash

###################### Linux 1-Touch recovery rev3 ##############################
# 
#
#
. /storage_comparison_funcs.sh



#Recreate partitions for a disk
#   Name: recreatePartitionsForDisk
#   API DEF:

recreatePartitionsForDisk()
{
    local diskname=$1
    Log "Invoked recreatePartitionsForDisk with [$1][$diskname]"
    cat $SYS_STATE_LOC/crypt_metadata/devicenames|grep -wq `basename $diskname`
    if [ $? -eq 0 ]
    then
        Log "device is encrypt[$diskname]"
        rm -f /tmp/recreate_info_specific
        tdev=`basename $diskname`
        while read line
        do
            Log "Checking for encryption[$line]"
            is_enc=`echo $line|grep "^LUKS##"`
            if [ -n "$is_enc" ]
            then
                name=`echo $line|awk '{print $2}'|cut -f2 -d'='`
                if [ "x$name" == "x$tdev" ]
                then
                    echo $line > /tmp/recreate_info_specific
                    break
                fi
            fi
        done < /tmp/storage_tree/recreate_info_all
        recreateCryptDevice /tmp/recreate_info_specific
        ret=$?
        echo "$diskname" >> $STORAGE_TREE/processed_disks
        return $ret
    fi
    resdiskNode=""
    echo $diskname|grep -Eq "^/dev/md"
    if [ $? -eq 0 ]
    then
        Log "Given disk is RAID disk[$diskname]"
        rm -f /tmp/recreate_info_specific
        while read line
        do
            Log "Checking[$line]"
            is_md=`echo $line|grep "^MD##"`
            if [ -n "$is_md" ]
            then
                name=`echo $line|awk '{print $2}'|cut -f2 -d'='`
                if [ "x$name" == "x$diskname" ]
                then
                    echo $line > /tmp/recreate_info_specific
                    break
                fi
            fi
        done < /tmp/storage_tree/recreate_info_all
        recreateMD /tmp/recreate_info_specific
        ret=$?
        if [ $ret -ne 0 ]
        then
            return $ret
        fi
        resdiskNode=`DiskNodeForDiskName "backup" "$diskname"`
    else
        resdiskNode=`DiskNodeForDiskName "restore" "$diskname"` 
    fi
    if [ -z "$resdiskNode" ]; then
        Log "recreatePartitions: Error: Restore disk node for [$diskname] not found"
        return 1
    fi
    ptype=`cat $resdiskNode/ptype`
    partinfo=`cat $resdiskNode/partinfo`
    if [ -z "$ptype" ]; then
        Log "recreatePartitions: Debug: Partition type of [$diskname] not found! No partition table to create"
        return 0
    fi

    if [ "$ptype" == "loop" ]; then
        Log "recreatePartitions: Debug: Partition type of [$diskname] detected as loop - nothing to recreate"
        return 0
    fi
 
    if [ -z "$partinfo" ]; then
        Log "recreatePartitions: Debug: Partition table information of [$diskname] not found! No partition table to create"
        return 0
    fi
    SCRIPT=$resdiskNode/parted_script
    createPartInfoScript "$resdiskNode" "true" "true"
    ret=$?
    if [ $ret == "1" ]; then
        Log "recreatePartitions: Error: Failed to create & run Partinfo script with [$diskname] [$resdiskNode]"
        return 1
    fi
    echo "$diskname" >> $STORAGE_TREE/processed_disks
    # check to see if device nodes have been created. For older systems like RH 4.x we might need to recreate the devices manually
    # this won't work for cciss as they are enumerated as /sys/block/cciss!c0d0 and the dev node is /dev/cciss/c0d0
    basedevname=`basename $diskname`
    partnames=`ls /sys/block/$basedevname | grep $basedevname`
    for partname in $partnames;
    do
        Log "Check to see if device for [$partname] exists"
        if [ -e /dev/$partname -o -e /dev/mapper/$partname ]; then
            Log "device for [$partname] exists"
        else
            majmin=`cat /sys/block/$basedevname/$partname/dev`
            if [ -n "$majmin" ]; then
                maj=`echo $majmin | cut -d':' -f1`
                min=`echo $majmin | cut -d':' -f2`
                Log "Extracted maj:min [$maj:$min] for /sys/block/$basedevname/$partname"
                mknodcmd=""
                if [ -e /dev/$basedevname ]; then
                    mknodcmd="mknod /dev/$partname b $maj $min"
                elif [ -e /dev/mapper/$basedevname ]; then
                    mknodcmd="mknod /dev/mapper/$partname b $maj $min"
                fi
                $mknodcmd
                ret=$?
                Log "mknod cmd [$mknodcmd] returns [$ret]"
            else
                Log "Could not find maj:min for [$partname] in /sys/block/$basedevname/$partname"
            fi
        fi
    done
    return 0
}


#Recreate all partitions using restore tree and recreation info file
#   Name: recreatePartitions
#   API DEF:
#       IN:             arg1=recreation info file, arg2=AUTOLAYOUTFILE
#       OUT:            none. return 0 on success, 1 on failure
#       EXIT:           no
recreatePartitions()
{
    RECFILE=$1
    AUTOLAYOUTFILE=$2
    Log "Debug: RECFILE [$1] AUTOLAYOUTFILE [$2]"

    #For VirtualizeME, if auto disk layout is selected, use an alternate code flow
    if [ -n "$AUTOLAYOUTFILE" -a -e "$AUTOLAYOUTFILE" ]; then
        Log "Invoked CreateAutoPartitions as mode=AUTOLAYOUT"
        ls $SYS_CONF/partition_info/*.sfdisk | while read sfile
        do
            Log "About to execute command $sfile"
            ExecuteRet "$sfile"
            curdev=`basename $sfile | cut -d'.' -f1`
            specfile=$SYS_CONF/partition_info/$curdev.out
            Log "curdev=[$curdev] specfile=[$specfile]"
            sfdisk -d /dev/$curdev > $specfile
            outfile=`cat $specfile`
            Log "partition info saved: $outfile"
            CreatePartNodes $curdev $specfile
        done    
    
        return 0
    fi

    if [ -z "$RECFILE" ]; then
        Log "recreatePartitions: Error: Invoked without recreation file name"
        return 1
    fi
    if [ ! -f $RECFILE ]; then
        Log "recreatePartitions: Error: Recreation information file [$RECFILE] not found"
        return 1
    fi
    Execute "rm -f $STORAGE_TREE/processed_disks"

    OLDLD=$LD_LIBRARY_PATH
    export LD_LIBRARY_PATH="$SRDIR:$LD_LIBRARY_PATH"
    Log "\n\n"
    Log "New LD_LIBRARY_PATH: [$LD_LIBRARY_PATH]"
    #cat $RECFILE | while read line
    while read line # $RECFILE
    do
        Log "\nrecreatePartitions: Debug: Read [$line]"
        is_disk=`echo $line | grep "^DISK##"`
        if [ -n "$is_disk" ]; then
            Log "\nrecreatePartitions: Debug: Process disk from [$line]"
            diskinfo=`echo $line | awk '{print $2}'`
            diskname=`echo $diskinfo | cut -d':' -f1`
            Log "recreatePartitions: Debug: Process disk [$diskname]"
            partnum=`echo $diskinfo | cut -f2 -d':'`
            if [ $partnum -eq 0 ]
            then
                echo $diskname > $STORAGE_TREE/processed_disks
            fi

            #Check if already processed
            is_processed=`grep -w $diskname $STORAGE_TREE/processed_disks`
            if [ -z "$is_processed" ]; then
                recreatePartitionsForDisk $diskname
                ret=$?
                if [ $ret -ne 0 ]; then
                    Log "recreatePartitionsForDisk Failed!"
                    return 1
                fi
            else
                Log "recreatePartitions: Debug: Ignore already processed disk [$diskname]"
            fi
        else
            Log "recreatePartitions: Debug: Ignore non-disk info on line [$line]"
        fi
    done < $RECFILE
    export LD_LIBRARY_PATH=$OLDLD
    return 0
}
 
#Recreate SoftRaid devices
#   Name: recreateMD
#   API DEF:
#       IN:             arg1=recreation info file, arg2=AUTOLAYOUTFILE
#       OUT:            none. return 0 on success, 1 on failure
#       EXIT:           no

recreateMD(){
    RECFILE=$1
    AUTOLAYOUTFILE=$2

    #Pre checks
    if [ "x`whereis mdadm`" == "x" ]
    then
        Log "MDADM command not found! Verify that whether we even have to create the device or not!"
        return 0
    fi

    if [ -n "$AUTOLAYOUTFILE" -a -e "$AUTOLAYOUTFILE" ]
    then
        Log "ADL recreation of MD devices is not supported yet!"
        return 0
    fi

    
    while read line # $RECFILE 
    do
        Log "\nrecreateMD: Debug: Read [$line]"
        is_md=`echo $line | grep "^MD##"`
        if [ -n "$is_md" ]; then
            touch /tmp/md_found
            Log "recreateMD: Debug: process [$line] from [$RECFILE]"
            name=`echo $line | awk '{print $2}' | cut -d'=' -f2`
            cat $STORAGE_TREE/processed_disks|grep "$name"
            if [ $? -eq 0 ]
            then
                Log "recreateMD:Debug:[$name] is already processed"
                continue
            fi
            devlist=`echo $line | awk '{print $3}' | cut -d'=' -f2- | awk -F"," '{for(i=1;i<NF;i++) print $i}'`
            if [ -z "$name" ]; then
                Log "recreateMD: Error: Cannot parse MD name from [$line]"
                return 1
            fi
            if [ -z "$devlist" ]; then
                Log "recreateMD: Error: Cannot parse devices list from [$line]"
                return 1
            fi
            Log "\nrecreateMD: Debug: process Name=[$name] with devlist=[$devlist]"
            newlist=""
            num=0
            for pv in $devlist;
            do
                pvdiskname=`echo $pv | cut -d':' -f1`
                pvpartno=`echo $pv | cut -d':' -f2`

                Log "Extraced device's disk name [$pvdiskname] partnum [$pvpartno]. check if it is processed"
                if [ -n "$pvpartno" -a "$pvpartno" != "0" ]; then
                    is_processed=`grep -w $pvdiskname $STORAGE_TREE/processed_disks`
                    if [ -z "$is_processed" ]; then
                        Log "...not previously processed. invoke 'recreatePartitionsForDisk $pvdiskname'"
                        recreatePartitionsForDisk $pvdiskname
                        ret=$?
                        if [ $ret -ne 0 ]; then
                            Log "recreatePartitionsForDisk Failed!"
                            return 1
                        fi
                    fi
                else
                    #PV might be a full disk, but it might also have some partitions on it. we need to wipe off the PT first
                    cmd="parted $pv --script mklabel msdos"
                    ExecuteRet "$cmd"
                    ret=$?
                    cmd="/sbin/blockdev --rereadpt $pv"
                    ExecuteRet "$cmd"
                    Log "Wipe previous partition table on $pv - retcode [$ret]" #ignore errors
                fi
                pvdevice=`recreateDeviceName "restore" "$pvdiskname" "$pvpartno"`
                Log "Constructed device=[$pvdevice] from [$pv] of VG=[$name] from [$RECFILE]"
                if [ ! -e "$pvdevice" ]; then
                    Log "recreateMD: Error: Device not found [$pvdevice]"
                    return 1
                fi

                Log "Remove PV if exist on it.[$pvdevice] of [$name] from [$RECFILE]"
                cmd="pvremove -y -ff $pvdevice"
                ExecuteRet "$cmd"
                num=$((num+1))
                if [ "x$newlist" == "x"]
                then
                    newlist="$pvdevice"
                else
                    newlist="$newlist $pvdevice"
                fi
            done
            Log "MD device[$name] is made up of devices[$newlist]"
            
            path=`find $SYS_STATE_LOC/md_metadata/disk/*/name -type f -exec grep -wH \`basename $name\` {} \;|cut -f1 -d':'`
            if [ "x$path" == "x" ]
            then
                Log "Could not find required Device in backup. Check Logs."
                return 1
            fi


            path=`dirname $path`

            Log "MD device[$name] found in [$path]"

            Log "Extract information"

            type=`cat $path/type`

            bnum=`ls $path/device/|wc -l`

            if [ "x$bnum" != "x$num" ]
            then
                Log "num of Backup disks[$bnum] and restore disks[$num] did not match for device [$name]!"
                return 1
            fi

            #Clean it. Just to be sure
            mdadm --remove $name
            mdadm --stop $name

            # Gather information
            metadata=$( grep "Version" $path/detail | tr -d " " | cut -d ":" -f "2")
            level=$( grep "Raid Level" $path/detail | tr -d " " | cut -d ":" -f "2")
            uuid=$( grep "UUID" $path/detail | tr -d " " | cut -d "(" -f "1" | cut -d ":" -f "2-")
            layout=$( grep "Layout" $path/detail | tr -d " " | cut -d ":" -f "2")
            chunksize=$( grep "Chunk Size" $path/detail | tr -d " " | cut -d ":" -f "2" | sed -r 's/^([0-9]+).+/\1/')
            nraiddevice=$( grep "Raid Devices" $path/detail | tr -d " " | cut -d ":" -f "2")
            nsparedevice=$((nraiddevice-bnum))
            lopt="--metadata=$metadata --level=$level --uuid=$uuid"

            if [ -n "$layout" ]
            then
                lopt="$lopt --layout=$layout"
            fi

            if [ -n "$chunksize" ]
            then
                lopt="$lopt --chunk=$chunksize"
            fi

            if [ "$nsparedevice" -ne 0 ]
            then
                lopt="$lopt --spare-device=$nsparedevice"
            fi
            cmd="echo \"Y\"|mdadm --create --force $name $lopt --raid-devices=$nraiddevice $newlist"

            ExecuteRet "$cmd"
            ret=$?
            if [ $ret -ne 0 ]; then
                Log "recreateMD: Error: Cannot mdadm [$name]"
                resetLibs
                return 1
            else
                Log "recreateMD: Debug: MD [$name] recreated successfully"
            fi

        else
            Log "recreateMD: Debug: Ignore non MD line from $RECFILE"
        fi
    done < $RECFILE
    Log "recreateMD: MD Creation End here!"
    Log "\n"
}

#Recreate LVM using restore tree and recreation info file
#   Name: recreateLVM
#   API DEF:
#       IN:             arg1=recreation info file, arg2=AUTOLAYOUTFILE
#       OUT:            none. return 0 on success, 1 on failure
#       EXIT:           no
recreateLVM()
{
    RECFILE=$1
    AUTOLAYOUTFILE=$2

    #For VirtualizeME, if auto disk layout is selected, use an alternate code flow
    if [ -n "$AUTOLAYOUTFILE" -a -e "$AUTOLAYOUTFILE" ]; then
        #If we are restarting the process, let's clean up all auto-disk layout LVM first (ignoring errors)
        moveLibs
        ExecuteRet "vgchange -an"
        ExecuteRet "vgremove -f VolGroup00 VolGroup01"
        if [ -e "${YASTGUI}/VMeHyperVHookFile" ]; then
            if [ ! -e "/dev/hda" -a -e "/dev/sda" ]; then
                ExecuteRet "pvremove -ff -y /dev/sda2 /dev/sdb"
            else
                ExecuteRet "pvremove -ff -y /dev/hda2 /dev/hdb"
            fi
        else
            ExecuteRet "pvremove -ff -y /dev/sda2 /dev/sdb"
        fi

        #...and recreate the lvm
        chmod +x $SYS_CONF/lvm_metadata/lvm.sh
        Execute "$SYS_CONF/lvm_metadata/lvm.sh"
        resetLibs
        return 0
    fi

    if [ -z "$RECFILE" ]; then
        Log "recreateLVM: Error: Invoked without recreation file name"
        return 1
    fi
    if [ ! -f $RECFILE ]; then
        Log "recreateLVM: Error: Recreation information file [$RECFILE] not found"
        return 1
    fi

    if [ -f $STORAGE_TREE/AUTOSCALESTORAGE ]; then
        
        moveLibs
        Log "recreateLVM: Debug: AUTOSCALESTORAGE"
        
        while read line
        do
            is_lvm=`echo $line | grep "^LVM##"`
            pvlist=""
            vgname=""

            if [ -n "$is_lvm" ]; then
                Log "recreateLVM: Debug: process [$line] from [$RECFILE]"
                vgname=`echo $line | awk '{print $2}' | cut -d'=' -f2`
                pvlist=`echo $line | awk '{print $3}' | cut -d'=' -f2- | awk -F"," '{for(i=1;i<NF;i++) print $i}'`
            fi

            if [ -z "$pvlist" -o -z "$vgname" ]; then
                continue
            fi

            if [ `vgs $vgname | wc -l` -gt 0 ]; then
                cmd="vgremove -f $vgname"
                ExecuteRet "$cmd"
                ret=$?
                if [ $ret -ne 0 ]; then
                    Log "recreateLVM: Error: VG [$vgname] removal failed. Cannot restore VG [$vgname]"
                    resetLibs
                    return 1
                fi
            fi

            pvdevices=""
            for pv in $pvlist
            do
                pvdiskname=`echo $pv | cut -d':' -f1`
                pvpartno=`echo $pv | cut -d':' -f2`
               
                Log "Extraced PV device's disk name [$pvdiskname] partnum [$pvpartno]. check if it is processed"
                
                if [ -n "$pvpartno" ]; then
                    is_processed=`grep -w $pvdiskname $STORAGE_TREE/processed_disks`
                    if [ -z "$is_processed" ]; then
                        Log "...not previously processed. invoke 'recreatePartitionsForDisk $pvdiskname'"
                        recreatePartitionsForDisk $pvdiskname
                        ret=$?
                        if [ $ret -ne 0 ]; then
                            Log "recreatePartitionsForDisk Failed!"
                            return 1
                        fi
                    fi
                else
                    #PV might be a full disk, but it might also have some partitions on it. we need to wipe off the PT first
                    cmd="parted $pv --script mklabel msdos"
                    ExecuteRet "$cmd"
                    ret=$?
                    cmd="/sbin/blockdev --rereadpt $pv"
                    ExecuteRet "$cmd"
                    Log "Wipe previous partition table on $pv - retcode [$ret]" #ignore errors
                fi
                
                pvdevice=`recreateDeviceName "restore" "$pvdiskname" "$pvpartno"`
                Log "Constructed pvdevice=[$pvdevice] from [$pv] of VG=[$vgname] from [$RECFILE]"
                if [ ! -e "$pvdevice" ]; then
                    Log "recreateLVM: Error: Device not found [$pvdevice]"
                    return 1
                fi

                pvdevices="$pvdevices $pvdevice"
 
                cmd="pvcreate -y -ff $pvdevice"
                ExecuteRet "$cmd"
                ret=$?
                if [ $ret -ne 0 ]; then
                    Log "recreateLVM: Error: PV [$pvdevice] creation failed. Cannot restore VG [$vgname]"
                    resetLibs
                    return 1
                fi
            done

            vg_meta_dir=$STORAGE_TREE/autoScale2/LVM/$vgname
            pe_size=`cat $vg_meta_dir/pe_size`

            cmd="vgcreate -y -ff -s ${pe_size}B $vgname $pvdevices"
     
            ExecuteRet "$cmd"
            ret=$?
            if [ $ret -ne 0 ]; then
                Log "recreateLVM: Error: Cannot restore VG [$vgname]"
                resetLibs
                return 1
            else
                Log "recreateLVM: Debug: VG [$vgname] recreated successfully"
            fi

            for lv_meta_dir in `find $vg_meta_dir/ -maxdepth 1 -mindepth 1 -type d`
            do 
                lv_name=`basename $lv_meta_dir`
                lv_size=`cat $lv_meta_dir/used_size`

                # Usually lvcreate option does not ask for y/n confirmation. However it might ask 
                # If some previous fs signature is present on it. On older systems lvcreate -y option 
                # is not supported, so i am sending 'y' via pipe to make it compatible on both newer 
                # and older systems.
                cmd="echo 'y' | lvcreate -n $lv_name $vgname -L ${lv_size}B"

                ExecuteRet "$cmd"
                ret=$?
                if [ $ret -ne 0 ]; then
                    Log "recreateLVM: Error: Cannot restore LV [$lv_name], VG[$vgname]"
                    resetLibs
                    return 1
                else
                    Log "recreateLVM: Debug: LV [$lv_name], VG[$vgname] recreated successfully"
                fi
            done
        done < $RECFILE

        resetLibs
        return 0
    fi

    Execute "rm -f $STORAGE_TREE/processed_vgs"
    cp $STORAGE_TREE/backup/lvm_metadata/pvs.list $STORAGE_TREE/backup/lvm_metadata/pvs.list.org
    #Intentionaly running outside cloud_flag as to make sure does not be used by normal 1-touch process.
    rm -rf $STORAGE_TREE/VOL_MAP
    if [ -f $STORAGE_TREE/backup/lvm_metadata/pvs.list ] && [ -f /cloud_flag ]
    then
         rm -rf $STORAGE_TREE/VOL_MAP
#        while read line
#        do
#            id=`echo $line|awk '{print $2}'`
#            uuid=`genuuid`
#            sed -i "s/$id/$uuid/g" $STORAGE_TREE/backup/lvm_metadata/pvs.list
#            find $STORAGE_TREE/backup/lvm_metadata/ -name \*.vgcfg -type f -exec sed -i "s/$id/$uuid/g" {} ;
#        done < $STORAGE_TREE/backup/lvm_metadata/pvs.list.org
        Log "vgcfg files are cmd:[find $STORAGE_TREE/backup/lvm_metadata/ -name \*.vgcfg -type f] result[`find $STORAGE_TREE/backup/lvm_metadata/ -name \*.vgcfg -type f`]"
        for vgfile in `find $STORAGE_TREE/backup/lvm_metadata/ -name \*.vgcfg -type f`
        do
            cp $vgfile $vgfile.orig # for debug purpose
            vgname=`basename $vgfile|sed -e "s/\.vgcfg//g"`
            vgname1=$vgname
            while read line
            do
                if [ x"$line" == x"$vgname" ]
                then
                    vgname1=vg$RANDOM
                fi
            done < /proxyvg
#            uuid=`genuuid`
#            sed -i "0,/id =.*$/s/id = .*$/id = \"$uuid\"/" $vgfile
            Log "vg name changed [ $vgname ] to [ $vgname1 ] taken from file [$vgfile]"
            echo "$vgname1 $vgname" >> $STORAGE_TREE/VOL_MAP
            sed -i "s/$vgname/$vgname1/g" $vgfile
        done
    fi
    while read line # $RECFILE 
    do
        Log "\nrecreateLVM: Debug: Read [$line]"
        is_lvm=`echo $line | grep "^LVM##"`
        if [ -n "$is_lvm" ]; then
            Log "recreateLVM: Debug: process [$line] from [$RECFILE]"
            vgname=`echo $line | awk '{print $2}' | cut -d'=' -f2`
            pvlist=`echo $line | awk '{print $3}' | cut -d'=' -f2- | awk -F"," '{for(i=1;i<NF;i++) print $i}'`
            is_vg_resized=0

            if [ -z "$vgname" ]; then
                Log "recreateLVM: Error: Cannot parse VG name from [$line]"
                return 1
            fi
            if [ -z "$pvlist" ]; then
                Log "recreateLVM: Error: Cannot parse PV list from [$line]"
                return 1
            fi

            # If this VG has a PV which is present on a disk mapped by autoScaleStorage1.
            # We need to handle this carefully. since PV and LV might have been resized.
            # We can not use the volumegroup.cfg collected from source rather we have to 
            # recreate VG and LV again.
            for pv in $pvlist
            do  
                pvdiskname=`echo $pv | cut -d':' -f1`
                pvpartno=`echo $pv | cut -d':' -f2`

                #getting backup device node
                resdisknode=`DiskNodeForDiskName "restore" "$pvdiskname"`
            
                if [ -f "$resdisknode/resized" ]; then           
                    is_vg_resized=1
                    break
                fi
            done

            if [ $is_vg_resized -eq 0 ]; then
                Log "\nrecreateLVM: Debug: process VG=[$vgname] with PVs=[$pvlist]"
                for pv in $pvlist;
                do
                    pvdiskname=`echo $pv | cut -d':' -f1`
                    pvpartno=`echo $pv | cut -d':' -f2`

                    Log "Extraced PV device's disk name [$pvdiskname] partnum [$pvpartno]. check if it is processed"
                    if [ -n "$pvpartno" ]; then
                        is_processed=`grep -w $pvdiskname $STORAGE_TREE/processed_disks`
                        if [ -z "$is_processed" ]; then
                            Log "...not previously processed. invoke 'recreatePartitionsForDisk $pvdiskname'"
                            recreatePartitionsForDisk $pvdiskname
                            ret=$?
                            if [ $ret -ne 0 ]; then
                                Log "recreatePartitionsForDisk Failed!"
                                return 1
                            fi
                        fi
                    else
                        #PV might be a full disk, but it might also have some partitions on it. we need to wipe off the PT first
                        cmd="parted $pv --script mklabel msdos"
                        ExecuteRet "$cmd"
                        ret=$?
                        cmd="/sbin/blockdev --rereadpt $pv"
                        ExecuteRet "$cmd"
                        Log "Wipe previous partition table on $pv - retcode [$ret]" #ignore errors
                    fi
                    pvdevice=`recreateDeviceName "restore" "$pvdiskname" "$pvpartno"`
                    Log "Constructed pvdevice=[$pvdevice] from [$pv] of VG=[$vgname] from [$RECFILE]"
                    if [ ! -e "$pvdevice" ]; then
                        Log "recreateLVM: Error: Device not found [$pvdevice]"
                        return 1
                    fi
                    bckpvdiskname=`lookupDiskMap "restore" "$pvdiskname"`
                    if [ -z "$bckpvdiskname" ]; then
                        Log "recreateLVM: Error: Cannot find backup PV device from DISK MAP for [$pvdiskname]"
                        cat $SYS_STATE_LOC/crypt_metadata/devicenames|grep -wq `basename $pvdiskname`
                        if [ $? -eq 0 ]; then
                            Log "It is encrypted disk[$pvdiskname]"
                            bckpvdiskname=$pvdiskname
                        else
                            echo $pvdiskname|grep -q "^/dev/md"
                            if [ $? -eq 0 ]; then
                                Log "It is RAID disk"
                                bckpvdiskname=$pvdiskname
                            else
                                return 1
                            fi
                        fi
                    fi
                    echo $bckpvdiskname | grep "^/dev/md"
                    if [ $? -eq 0 ]; then
                        bckpvdevice=$pvdevice
                    else
                        bckpvdevice=`recreateDeviceName "backup_pv" "$bckpvdiskname" "$pvpartno"`
                    fi
                    Log "Reconstructed backup pvdevice=[$bckpvdevice] of VG=[$vgname] from [$RECFILE]"
                    pvuuid=`cat $STORAGE_TREE/backup/lvm_metadata/pvs.list | grep -w $vgname | grep -w $bckpvdevice | awk '{print $2}'`
                    Log "Extracted PV UUID=[$pvuuid] for PV[$bckpvdevice] VG=[$vgname]. Now recreate it"
                    if [ "x$pvuuid" == "x" ]
                    then
                        Log "Unable to find correct pv device from backup list,check and correct /system_state/sysconf/kpartx_delim and retry."
                        return 1
                    fi

                    #cmd="pvremove -y -ff $bckpvdevice"
                    #ExecuteRet "$cmd"
                    
                    #FIXME: move the libdevmapper libraries used for partinfo out of '/' as they take precedence over system libdevmapper
                    #libraries causing pvremove/pvcreate to fail. once done, we'll move them back so any subsequent function that makes use 
                    #of partinfo will continue to work
                    moveLibs

                    cmd="pvremove -y -ff $pvdevice"
                    ExecuteRet "$cmd"

                    #there might be a PV with the same uuid, but on a different device - this can happen when disks got swapped.
                    #there was also an issue with disk mapping when srcompare was restarted (ycp back/forth) that might cause this
                    otherpv=`pvs -o pv_name,pv_uuid | grep $pvuuid | awk '{print $1}'`
                    if [ -n "$otherpv" ]; then
                        ct=`echo $otherpv | wc -w`
                        if [ $ct -gt 1 ]; then
                            Log "recreateLVM: Found multiple PVs [$otherpv] with same uuid as intended pv [$pvdevice] - [$pvuuid] - (hint: enable mpath?)"
                            return 1
                        fi
                        Log "recreateLVM: Found PV [$otherpv] with same uuid as intended pv [$pvdevice] - [$pvuuid] - remove $otherpv"
                        cmd="pvremove -y -ff $otherpv"
                        ExecuteRet "$cmd"
                    fi

                    cmd="pvcreate -y -ff -u $pvuuid --restorefile $STORAGE_TREE/backup/lvm_metadata/$vgname.vgcfg $pvdevice"
                    ExecuteRet "$cmd"
                    ret=$?
                    if [ $ret -ne 0 ]; then
                        Log "recreateLVM: Error: PV [$pvdevice] creation failed. Cannot restore VG [$vgname]"
                        resetLibs
                        return 1
                    fi
                done
                cmd="vgcfgrestore -f $STORAGE_TREE/backup/lvm_metadata/$vgname.vgcfg $vgname"
                if [ -f /cloud_flag ]
                then
                    vgname1=`cat $STORAGE_TREE/VOL_MAP|grep -w $vgname|awk '{print $1}'`
                    cmd="vgcfgrestore -f $STORAGE_TREE/backup/lvm_metadata/$vgname.vgcfg $vgname1"
                fi
                ExecuteRet "$cmd"
                ret=$?
                if [ $ret -ne 0 ]; then
                    Log "recreateLVM: Error: Cannot restore VG [$vgname]"
                    resetLibs
                    return 1
                else
                    Log "recreateLVM: Debug: VG [$vgname] recreated successfully"
                fi
            else
                unset N_LV_SIZE
                declare -A N_LV_SIZE
                #resize logical volumes and then create them manually
                pvdevices=""
              
                #vgname="$vgname"_"new"

                if [ `vgs $vgname | wc -l` -gt 0 ]; then
                    cmd="vgremove -f $vgname"
                    ExecuteRet "$cmd"
                    ret=$?
                    if [ $ret -ne 0 ]; then
                        Log "recreateLVM: Error: VG [$vgname] removal failed. Cannot restore VG [$vgname]"
                        resetLibs
                        return 1
                    fi
                fi

                pe_size=0

                for pv in $pvlist;
                do
                    pvdiskname=`echo $pv | cut -d':' -f1`
                    pvpartno=`echo $pv | cut -d':' -f2`
               
                    Log "Extraced PV device's disk name [$pvdiskname] partnum [$pvpartno]. check if it is processed"
                
                    if [ -n "$pvpartno" ]; then
                        is_processed=`grep -w $pvdiskname $STORAGE_TREE/processed_disks`
                        if [ -z "$is_processed" ]; then
                            Log "...not previously processed. invoke 'recreatePartitionsForDisk $pvdiskname'"
                            recreatePartitionsForDisk $pvdiskname
                            ret=$?
                            if [ $ret -ne 0 ]; then
                                Log "recreatePartitionsForDisk Failed!"
                                return 1
                            fi
                        fi
                    else
                        #PV might be a full disk, but it might also have some partitions on it. we need to wipe off the PT first
                        cmd="parted $pv --script mklabel msdos"
                        ExecuteRet "$cmd"
                        ret=$?
                        cmd="/sbin/blockdev --rereadpt $pv"
                        ExecuteRet "$cmd"
                        Log "Wipe previous partition table on $pv - retcode [$ret]" #ignore errors
                    fi
                    pvdevice=`recreateDeviceName "restore" "$pvdiskname" "$pvpartno"`
                    Log "Constructed pvdevice=[$pvdevice] from [$pv] of VG=[$vgname] from [$RECFILE]"
                    if [ ! -e "$pvdevice" ]; then
                        Log "recreateLVM: Error: Device not found [$pvdevice]"
                        return 1
                    fi

                    #getting backup device node
                    resdisknode=`DiskNodeForDiskName "restore" "$pvdiskname"`

                    for lv_name in `ls $resdisknode/$pvpartno/LV`
                    do
                        res_part_dir="$resdisknode/$pvpartno"
                        res_lv_dir="$res_part_dir/LV/$lv_name"

                        if [ $pe_size -eq 0 ]; then
                            pe_size=`cat $res_part_dir/pv_info | awk -F"," '{ if (NR > 1) { print $7}}'` 
                        fi
                      
                        lv_total_pe=${N_LV_SIZE[$lv_name]}
                      
                        if [ -z "$lv_total_pe" ]; then
                            lv_total_pe=0
                        fi

                        lv_used_pe_on_cur_pv=`cat $res_lv_dir/total_used_pe_on_cur_pv`
                        lv_total_pe=$((lv_total_pe + lv_used_pe_on_cur_pv))
                        N_LV_SIZE[$lv_name]=$lv_total_pe
                    done                

                    pvdevices="$pvdevices $pvdevice" 
                    
                    cmd="pvcreate -y -ff $pvdevice"
                    ExecuteRet "$cmd"
                    ret=$?
                    if [ $ret -ne 0 ]; then
                        Log "recreateLVM: Error: PV [$pvdevice] creation failed. Cannot restore VG [$vgname]"
                        resetLibs
                        return 1
                    fi

                done

                cmd="vgcreate -y -ff -s ${pe_size}B $vgname $pvdevices"
     
                ExecuteRet "$cmd"
                ret=$?
                if [ $ret -ne 0 ]; then
                    Log "recreateLVM: Error: Cannot restore VG [$vgname]"
                    resetLibs
                    return 1
                else
                    Log "recreateLVM: Debug: VG [$vgname] recreated successfully"
                fi

                for lv_name in "${!N_LV_SIZE[@]}"
                do 
                    lv_total_pe=${N_LV_SIZE[$lv_name]}

                    # Usually lvcreate option does not ask for y/n confirmation. However it might ask 
                    # If some previous fs signature is present on it. On older systems lvcreate -y option 
                    # is not supported, so i am sending 'y' via pipe to make it compatible on both newer 
                    # and older systems.
                    cmd="echo 'y' | lvcreate -n $lv_name $vgname -l $lv_total_pe"

                    ExecuteRet "$cmd"
                    ret=$?
                    if [ $ret -ne 0 ]; then
                        Log "recreateLVM: Error: Cannot restore LV [$lv_name], VG[$vgname]"
                        resetLibs
                        return 1
                    else
                        Log "recreateLVM: Debug: LV [$lv_name], VG[$vgname] recreated successfully"
                    fi
                done
            fi
        else
            Log "recreateLVM: Debug: Ignore non LVM line from $RECFILE"
        fi
    done < $RECFILE
    ExecuteRet "vgchange -ay"
    ExecuteRet "vgscan --mknodes"
    resetLibs
    return 0
}

# check make filesytem option
# of type [-<option> <value>]
#    Name: checkFSOption
#    API DEF:
#        IN:         arg1=mkfs_cmd    args2=option
#        OUT:        0=success 1=error
#        EXIT:       no
checkFSOption()
{
    local lmkfs_cmd lopt
    lmkfs_cmd=$1
    lopt=$2
    $lmkfs_cmd 2>&1 | grep --quiet "\-${lopt} "
    ret=$?
    if [ $ret -eq 0 ]; then
        echo 0
    else
        echo 1
    fi
}

# Parse FS options
#    Name: parseFSOptions
#    API DEF:
#        IN:            arg1=fstype   arg2=raw_fs_options   arg3=dev_label
#        OUT:           mkfs command with fs options
#        EXIT:          no
parseFSOptions()
{
    local lfstype lrawinfo llabel lcmd lopt
    lfstype=$1
    lrawinfo=$2
    llabel=$3
    lcmd=""
    case $fstype in
        "ext2")
                lcmd="mkfs -t ext2 -F"
                lopt=`grep "Block size" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext2" "b"` == "x0" ]; then
                    lcmd="$lcmd -b $lopt"
                fi
                lopt=`grep "Blocks per group" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext2" "g"` == "x0" ]; then
                    lcmd="$lcmd -g $lopt"
                fi
                lopt=`grep "Filesystem revision #" $lrawinfo | cut -d":" -f2 | awk '{print $1}'`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext2" "r"` == "x0" ]; then
                    lcmd="$lcmd -r $lopt"
                fi
                lopt=`grep "Filesystem UUID" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext2" "U"` == "x0" ]; then
                    lcmd="$lcmd -U $lopt"
                fi
                lopt=`grep "^Reserved block count" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                lopt1=`grep "^Block count" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a -n "$lopt1" -a x`checkFSOption "mkfs.ext2" "m"` == "x0" ]; then
                    hund=100
                    lopt=`expr $lopt \* $hund`
                    lopt1=`expr $lopt / $lopt1`
                    lcmd="$lcmd -m $lopt1"
                fi
                if [ "x$llabel" != "x" -a x`checkFSOption "mkfs.ext2" "L"` == "x0" ]; then
                    lcmd="$lcmd -L $llabel"
                fi
                lcmd="$lcmd $dev"
                ;;
        "ext3")
                lcmd="mkfs -t ext3 -F"
                lopt=`grep "Block size" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext3" "b"` == "x0" ]; then
                    lcmd="$lcmd -b $lopt"
                fi
                lopt=`grep "Blocks per group" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext3" "g"` == "x0" ]; then
                    lcmd="$lcmd -g $lopt"
                fi
                lopt=`grep "Filesystem revision #" $lrawinfo | cut -d":" -f2 | awk '{print $1}'`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext3" "r"` == "x0" ]; then
                    lcmd="$lcmd -r $lopt"
                fi
                lopt=`grep "Filesystem UUID" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext3" "U"` == "x0" ]; then
                    lcmd="$lcmd -U $lopt"
                fi
                lopt=`grep "^Reserved block count" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                lopt1=`grep "^Block count" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a -n "$lopt1" -a x`checkFSOption "mkfs.ext3" "m"` == "x0" ]; then
                    hund=100
                    lopt=`expr $lopt \* $hund`
                    lopt1=`expr $lopt / $lopt1`
                    lcmd="$lcmd -m $lopt1"
                fi
                if [ "x$llabel" != "x" -a x`checkFSOption "mkfs.ext3" "L"` == "x0" ]; then
                    lcmd="$lcmd -L $llabel"
                fi
                lcmd="$lcmd $dev"
                ;;
        "ext4")
                lcmd="mkfs -t ext4 -F -E lazy_itable_init"
                lopt=`grep "Block size" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext4" "b"` == "x0" ]; then
                    lcmd="$lcmd -b $lopt"
                fi
                lopt=`grep "Blocks per group" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext4" "g"` == "x0" ]; then
                    lcmd="$lcmd -g $lopt"
                fi
                lopt=`grep "Filesystem revision #" $lrawinfo | cut -d":" -f2 | awk '{print $1}'`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext4" "r"` == "x0" ]; then
                    lcmd="$lcmd -r $lopt"
                fi
                lopt=`grep "Flex block group size" $lrawinfo | cut -d":" -f2 | awk '{print $1}'`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext4" "G"` == "x0" ]; then
                    lcmd="$lcmd -G $lopt"
                fi
                lopt=`grep "Filesystem UUID" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a x`checkFSOption "mkfs.ext4" "U"` == "x0" ]; then
                    lcmd="$lcmd -U $lopt"
                fi
                lopt=`grep "^Reserved block count" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                lopt1=`grep "^Block count" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" -a -n "$lopt1" -a x`checkFSOption "mkfs.ext4" "m"` == "x0" ]; then
                    hund=100
                    lopt=`expr $lopt \* $hund`
                    lopt1=`expr $lopt / $lopt1`
                    lcmd="$lcmd -m $lopt1"
                fi
                if [ "x$llabel" != "x" -a x`checkFSOption "mkfs.ext4" "L"` == "x0" ]; then
                    lcmd="$lcmd -L $llabel"
                fi
                lcmd="$lcmd $dev"
                ;;
        "vfat")
                lcmd="mkfs.vfat $dev"
                ;;
        "reiserfs")
                lcmd="mkreiserfs -f"
                lopt=`grep "Blocksize" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd --block-size $lopt"
                fi
                lopt=`grep "Hash" $lrawinfo | cut -d":" -f2 | tr -d [:blank:]`
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd --hash $lopt"
                fi
                lopt=`grep "Device" $lrawinfo | awk '{print $2}' | tr -d '[]'`
                if [ -n "$lopt" -a "m$lopt" != "m0x0" ]; then
                    lcmd="$lcmd -j $lopt"
                fi
                lopt=`grep "UUID" $lrawinfo | awk '{print $2}' | tr -d ' '`
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd --uuid $lopt"
                fi
                if [ "x$llabel" != "x" ]; then
                    lcmd="$lcmd -l $llabel"
                fi
                lcmd="$lcmd $dev"
                ;;
        "xfs")
                lcmd="mkfs.xfs -f"
                lopt=`grep -w "^data " $lrawinfo | grep bsize | awk '{print $3}' | cut -d"=" -f2`
                bsize=""
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd -b size=$lopt"
                    bsize=$lopt
                fi
                lopt=`grep -w "^data " $lrawinfo | grep imaxpct | awk '{print $5}' | cut -d"=" -f2`
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd -i maxpct=$lopt"
                fi
                lopt=`grep -w "^data " -A1 $lrawinfo | grep sunit | awk '{print $2}' | cut -d"=" -f2`
                lopt1=`grep -w "^data " -A1 $lrawinfo | grep swidth | awk '{print $3}' | cut -d"=" -f2`
                if [ -n "$lopt" -a -n "$lopt1" ]; then
                    if [ -n "$bsize" ]; then
                        mult=$((bsize/512))
                        loptm=$((lopt * mult))
                        lopt1m=$((lopt1 * mult))
                    else
                        loptm=$lopt
                        lopt1m=$lopt1
                    fi
                    lcmd="$lcmd -d sunit=$loptm -d swidth=$lopt1m"
                fi

                lopt=`grep -w "^meta-data" -A2 $lrawinfo | grep sectsz | awk '{print $2}' | cut -d"=" -f2`
                lopt=`grep -w "^meta-data" $lrawinfo | awk '{print $1}' | cut -d"=" -f2`
                lopt=`grep -w "^meta-data" $lrawinfo | grep isize | awk '{print $2}' | cut -d"=" -f2`
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd -i size=$lopt"
                fi
                lopt=`grep -w "^meta-data" $lrawinfo | grep agcount | awk '{print $3}' | cut -d"=" -f2 | tr -d ','`
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd -d agcount=$lopt"
                fi
                lopt=`grep -w "^meta-data" -A3 $lrawinfo | grep crc | awk '{print $2}' | cut -d"=" -f2 | tr -d ','`
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd -m crc=$lopt"
                fi

                lopt=`grep -w "^naming" $lrawinfo | grep version | awk '{print $3}'` 
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd -n version=$lopt"
                fi
                lopt=`grep -w "^naming" $lrawinfo | grep bsize | awk '{print $4}' | cut -d"=" -f2`

                lopt=`grep -w "^log " -A1 $lrawinfo | grep lazy-count | awk '{print $5}' | cut -d"=" -f2`
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd -l lazy-count=$lopt"
                fi
                lopt=`grep -w "^log " $lrawinfo | awk '{print $2}' | cut -d"=" -f2`
                if [ -n "$lopt" -a "x$lopt" != "xinternal" ]; then
                    lcmd="$lcmd -l logdev=$lopt"
                fi
                lopt=`grep -w "^log " $lrawinfo | grep bsize | awk '{print $3}' | cut -d"=" -f2`
                lopt=`grep -w "^log " $lrawinfo | grep version | sed 's/.*version=//g'`
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd -l version=$lopt"
                fi

                lopt=`grep -w "^realtime " $lrawinfo | grep extsz | awk '{print $3}' | cut -d"=" -f2`
                if [ -n "$lopt" ]; then
                    lcmd="$lcmd -r extsize=$lopt"
                fi
                lcmd="$lcmd $dev"
                ;;
        "btrfs")
                lcmd="Btrfs Disk Already Created"
                ;;
        "*")
                Log "generateMKFScmd: Unsupported fstype!"
                return
                ;;
    esac
    echo "$lcmd"
}


#Generate mkfs command
#   Name: generateMKFScmd
#   API DEF:
#       IN:             arg1=dev  arg2=mntpt  arg3=fstype   arg4=raw_fsoptions   arg5=label
#       OUT:            mkfs command
#       EXIT:           no
generateMKFScmd()
{
    dev=$1
    mntpoint=$2
    fstype=$3
    raw_fsoptions=$4
    label=$5
    cmd=""

    if [ -z "$dev" -o -z "$mntpoint" -o -z "$fstype" ]; then
        Log "\ngenerateMKFScmd: Error: Invalid args to generateMKFScmd [$dev][$mntpoint][$fstype][$raw_fsoptions][$label]"
        return
    fi

    cmd=`parseFSOptions "$fstype" "$raw_fsoptions" "$label"`
    echo "$cmd" | grep -q "Btrfs Disk Already Created"
    ret=$?
    if [ x"$ret" == "x0" ]
    then
        echo $cmd
    else
        Log "\ngenerateMKFScmd: Debug: Add mount command to [$cmd]"
        tunecmd="sleep 1"
        if [ "$fstype" == "ext2" -o "$fstype" == "ext3" -o "$fstype" == "ext4" ]; then
            default_mnt_opts=`grep "Default mount options" "$raw_fsoptions" | cut -d ':' -f2`
            echo ${default_mnt_opts} | grep -q "(none)"
            if [ $? -ne 0 -a -n "${default_mnt_opts}" ]; then
                new_attr_list=""
                for attr in ${default_mnt_opts}
                do
                    if [ -z "$new_attr_list" ]; then
                        new_attr_list=${attr}
                    else
                        new_attr_list="${new_attr_list},${attr}"
                    fi
                done
                tunecmd="/sbin/tune2fs -o ${new_attr_list} $dev"
            fi
        fi
        cmd="${cmd} && { $tunecmd ; sleep 4 ; mkdir -p /tmp/MOUNTDIR/$mntpoint ; mount $dev /tmp/MOUNTDIR/$mntpoint; echo \"$dev $mntpoint $fstype defaults 0 0\" >> $STORAGE_TREE/fstab; } || exit 1"
        if [ "$fstype" == "xfs" ]; then
            rblks=`grep "^reserved blocks" "$raw_fsoptions" | cut -d"=" -f2 | tr -d [:blank:]`
            if [ -n "$rblks" ]; then
                cmd="${cmd} ; xfs_io -x -c \"resblks $rblks\" /tmp/MOUNTDIR/$mntpoint  2>/dev/null"
            fi
        fi
    fi

    Log "generateMKFScmd: Debug: return command=[$cmd]"
    echo $cmd
    return
}

#Generate mkswap command
#   Name: generateMKSWAPcmd
#   API DEF:
#       IN:             arg1=dev  arg2=swap_type  arg3=swap_priority 
#       OUT:            mkfs command
#       EXIT:           no
generateMKSWAPcmd()
{
    dev=$1
    swap_type=$2
    swap_priority=$3

    if [ -z "$dev" ]; then
        Log "generateMKSWAPcmd: Error: Invalid args to generateMKSWAPcmd [$dev][$swap_type][$swap_priority]"
        return
    fi
    cmd="mkswap $dev"

    swap_mount_cmd=""
    Log "\ngenerateMKSWAPcmd: Debug: Add mount command to [$cmd]"
    if [ -n "$swap_type" -a "$swap_type" == "partition" ]; then
        if [ -n "$swap_priority" ]; then
            swap_mount_cmd="swapon -p $swap_priority $dev"
        else
            swap_mount_cmd="swapon $dev"
        fi
    fi
    cmd="${cmd} ; sleep 4 ; ${swap_mount_cmd}; echo \"$dev swap swap defaults 0 0\" >> $STORAGE_TREE/fstab"

    if [ -f /cloud_flag ]
    then
        Log "Let's add [$dev] into destswap"
        echo $dev >> /destswap
    fi
    Log "\ngenerateMKSWAPcmd: Debug: return command=[$cmd]"
    echo $cmd
    return
}

#Prepare disk before mounting btrfs on it.
#   Name: prepareBtrfsDisk
#   API DEF:
#       IN:         no args
#       OUT:        none, return 0 on sucess, 1 on failure
#       EXIT:       yes
prepareBtrfsDisk()
{
    Log "prepareBtrfsDisk:Debug:Function Called! "
    tree="$STORAGE_TREE/backup/mount_tree"
    rm -rf /tmp/btrfs_disk
    while read mntpoint
    do
        mntmatch=`find $tree -maxdepth 2 -type f -name mntpoint | xargs egrep -wH "${mntpoint}$"`
        fmntmatch=`echo $mntmatch | cut -d':' -f1` 
        dmntmatch=`dirname $fmntmatch`
        mdir=`basename $dmntmatch`

        to_restore=`cat $tree/$mdir/to_restore`
        
        device=`cat $tree/$mdir/device`
        fstype=`cat $tree/$mdir/fstype`
        label=`cat $tree/$mdir/label`
        raw_fsoptions=`cat $tree/$mdir/raw_fsoptions`
        recreation_info=`cat $tree/$mdir/recreation_info`
        swap_type=`cat $tree/$mdir/swap_type`
        swap_priority=`cat $tree/$mdir/swap_priority`

        btrfs_par=`echo $device |sed "s/\//_/g"`
        btrfs_check=`grep "$btrfs_par" /tmp/btrfs_disk`
        Log "prepareBtrfsDisk:Debug:parse mntpoint[$mntpoint],mdir[$mdir],fstype[$fstype],to_restore[$to_restore],btrfs_check[$btrfs_check]"
        if [ "$to_restore" != "1" ] || [ x"$fstype" != "xbtrfs" ] || [ x"$btrfs_check" != "x" ]; then
            continue
        fi

        Log "prepareBtrfsDisk: Debug: Extract device=[$device] mntpoint=[$mntpoint] fstype=[$fstype] label=[$label] recreation_info=[$recreation_info] FSOptions=[$tree/$mdir/raw_fsoptions]"

        isdisk=`echo $recreation_info | grep -E "MD##|LUKS##|LVM##"`
        if [ -z "$isdisk" ]; then
            isdisk=`echo $recreation_info | grep "DISK##"`
            Log "prepareBtrfsDisk: Debug: device is a disk. Rewrite and validate it"
            
            if [ -f $STORAGE_TREE/AUTOSCALESTORAGE ]; then
                bkpdevice=`echo $isdisk | awk '{print $2}'`
                resdevice=`cat $STORAGE_TREE/autoScale2/PART_MAP | grep ^$bkpdevice | awk '{print $2}'`
                res_disk=`echo $resdevice | cut -d ":" -f1`
                partnum=`echo $resdevice | cut -d ":" -f2`
            else
                bck_disk=`echo $isdisk | awk '{print $2}' | cut -d':' -f1`
                partnum=`echo $isdisk | awk '{print $2}' | cut -d':' -f2`
                res_disk=`lookupDiskMap "backup" "$bck_disk"`
            fi
            resdevice=`recreateDeviceName "restore" "$res_disk" "$partnum"`
        else
            if [ -f $STORAGE_TREE/AUTOSCALESTORAGE ]; then
                #Mountpoints residing on MD devices are not supported in AUTOSCALE approach 2   
                is_md=`echo $recreation_info | grep 'MD##' | wc -l`
                if [ $is_md -eq 1 ]; then
                    Log "prepareBtrfsDisk: Mountpoint [$mntpoint] is present of MD device which is not supported when autoScale2 method has been used for disk/part mapping"
                    continue
                fi 
            fi

            Log "prepareBtrfsDisk: Debug: device is a LV. validate it"
            resdevice="$device"
        fi

        if [ ! -e "$resdevice" -a ! -L "$resdevice" ]; then
            Log "prepareBtrfsDisk: Error: mapped restore device [$resdevice] not found!"
            return 1
        fi

        if [ -z "$raw_fsoptions" ]; then
            Log "prepareBtrfsDisk: Debug: raw_fsoptions empty..."
            raw_fsoptions="~"
        fi
        btrfs_info=/system_state/sysconf/btrfs_info/$btrfs_par
        btrfile=$btrfs_info/subvolume
        if [ "a`cat /system_state/sysconf/distribution`" == "aSUSE" ]; then
            #these mkfs options are incompatible on some kernel versions
            ExecuteRet "{ mkfs.btrfs -f -O ^extref,^skinny-metadata $resdevice; } || exit 1"
            ret=$?
            if [ $ret -ne 0 ]
            then
                Log "Failed to create btrfs on Disk[$resdevice]"
                exit 1
            fi
        else
            ExecuteRet "{ mkfs.btrfs -f $resdevice; } || exit 1"
            ret=$?
            if [ $ret -ne 0 ]
            then
                Log"Failed to create btrfs on disk[$resdevice]"
                exit 1
            fi
        fi
        ExecuteRet "{ mount $resdevice /tmp/MOUNTDIR; } || exit 1"
        ret=$?
        if [ $ret -ne 0 ]
        then
            Log"Failed to create btrfs on disk[$resdevice]"
            exit 1
        fi
        while read bmnt
        do
            btrfs_name=`echo $bmnt | cut -f2 -d'='|cut -f1 -d','`
            if [ x"$btrfs_name" == x"root" ]
            then
                Log "Root Subvolume Skip it!Already Done!!"
                continue
            fi
            btrfs_sub=`echo $btrfs_name|sed "s/^root\///g"`
            btrfs_subdir=`dirname $btrfs_sub`
            Log "prepareBtrfsDisk: Debug: btrfs_sub[$btrfs_sub], btrfs_dir[$btrfs_subdir]"
            ExecuteRet "mkdir -p /tmp/MOUNTDIR/$btrfs_subdir"
            ExecuteRet "btrfs subvolume create /tmp/MOUNTDIR/$btrfs_sub"
        done < $btrfile
        echo "$btrfs_par" > /tmp/btrfs_disk
        Log "Restoring btrfs default value"
        
        btrfs_name=`cat $btrfs_info/default | cut -f2 -d'='|cut -f1 -d','`
        if [ x"$btrfs_name" == x"root" ]
        then
            ExecuteRet "{ btrfs subvolume set-default 5 /tmp/MOUNTDIR; } || exit 1"
            ret=$?
            if [ $ret -ne 0 ]
            then
                Log"Failed to set default value on btrfs disk[$resdevice]"
                exit 1
            fi
        else
            btrfs_sub=`echo $btrfs_name|sed "s/^root\///g"`
            def_id=`btrfs subvolume list /tmp/MOUNTDIR | grep "path $btrfs_sub$"|awk '{print $2}'`
            ExecuteRet "{ btrfs subvolume set-default $def_id /tmp/MOUNTDIR; } || exit 1"
            ret=$?
            if [ $ret -ne 0 ]
            then
                Log"Failed to set default value on btrfs disk."
                exit 1
            fi
        fi
        ExecuteRet "umount /tmp/MOUNTDIR"
    done < $STORAGE_TREE/sortedmntpoint.lst
    #find largest string and assign it as root.For other children create subvolumes.
}

#Recreate Encrypted Device(LUKS Supported)
# Name: recreateCryptDevice
# API DEF:
#      IN:              arg1=recreation_info_all file location
#      OUT:             none. return 0 on success, 1 on failure
#      EXIT:            no
# AUTHOR:    SHIVAM GARG
recreateCryptDevice()
{
    RECFILE=$1
    if  [ -z "$RECFILE" ]
    then
        Log "recreateFile Location is empty"
        return 1
    fi
    while read line
    do
        Log "recreateCryptDevice: Debug: READ [$line]"
        is_enc=`echo $line|grep "^LUKS##"`
        if [ -n "$is_enc" ]
        then
            Log "Debug: Process line [$line]"
            name=`echo $line|awk '{print $2}'|cut -f2 -d'='`
            cat $STORAGE_TREE/processed_disks | grep -wq "/dev/mapper/$name"
            if [ $? -eq 0 ]
            then
                Log "Already Processed [/dev/mapper/$name]"
                continue
            fi
           disk=`echo $line|awk '{print $3}'|cut -f2 -d'='`
           islv=`echo $disk|cut -f2 -d':'|grep "lv"`
           if [ "x$islv" == "x" ]
           then
               diskname=`echo $disk|cut -f1 -d':'`
               partnum=`echo $disk|cut -f2 -d':'`
               disk=`recreateDeviceName "restore" $diskname $partnum`
            else
                disk=`echo $disk|cut -f1 -d':'`
            fi
            if [ -e "$disk" ]
            then
                Log "Found disk[$disk]! start encrypting"
                folder=$SYS_STATE_LOC/crypt_metadata/$name
                if [ -d $folder ]
                then
                    echo $disk > $folder/restoreDisk
                    echo YES|cryptsetup luksHeaderRestore $disk --header-backup-file $folder/diskHeader.backup
                    if [ $? -ne 0 ]
                    then
                       Log "Debug: Unable to create encrypted devices!"
                       return 1
                    fi
                   cryptsetup luksClose $name
                   val=`cat /tmp/override/cryptFile.key|grep "^/dev/mapper/$name"|awk '{print $2}'`
                   if [ "x$val" == "xf" ]
                   then
                       file=`cat /tmp/override/cryptFile.key|grep "^/dev/mapper/$name"|awk '{print $3}'`
                       cp -f $file $folder/keycontent
                       echo /root/`basename $file` > $folder/keyfile
                   fi
                   if [ -f $folder/keycontent ]
                   then
                       cryptsetup luksOpen $disk $name --key-file=$folder/keycontent
                       if [ $? -ne 0 ]
                       then
                           Log "Encryption Failed!Please check key provided is right!"
                       fi
                   else
                       if [ "x$val" != "xp" ]
                       then
                           Log "Please check format of file cryptFile.key"
                           return 1
                       fi
                       pass=`cat /tmp/override/cryptFile.key|grep "^/dev/mapper/$name"|awk '{print $3}'`
                       if [ -z "$pass" ]
                       then
                           Log "Password not found!Please manually put password in this file: /tmp/override/cryptFile.key"
                           return 1
                       fi
                       echo "$pass" | cryptsetup luksOpen $disk $name
                       if [ $? -eq 0 ]
                       then
                           Log "Successfully crypted and opened the device!"
                       else
                           Log "Look like password is wrong or disk is already open"
                           return 1
                       fi
                  fi
               else
                   Log "Folder [$folder] not found! returning error!"
                   return 0
               fi
           else
               Log "Could not found node for disk [$disk]"
               return 1
           fi
       fi
   done < $RECFILE 
   return 0
}


#Recreate Mount points using restore tree and recreation info file
#   Name: recreateMountPoints
#   API DEF:
#       IN:             arg1=AUTOLAYOUTFILE $STORAGE_TREE/backup/mount_tree assumed to be present. recreatePartitions/recreateLVM should have been called
#       OUT:            none. return 0 on success, 1 on failure
#       EXIT:           no
recreateMountPoints()
{
    AUTOLAYOUTFILE=$1

    #For VirtualizeME, if auto disk layout is selected, use an alternate code flow
    if [ -n "$AUTOLAYOUTFILE" -a -e "$AUTOLAYOUTFILE" ]; then
        cat $SYS_CONF/mountpoints.lst | while read line
        do
            dev=`echo $line | awk '{print $1}'`
            mpt=`echo $line | awk '{print $2}'`
            fstype=`echo $line | awk '{print $3}'`
            if [ "x$mpt" == "xswap" ]; then
                fmt_cmd="mkswap $dev"
            else
                fmt_cmd="mkfs -t $fstype $dev"
            fi
            Log "\nfmt_cmd: $fmt_cmd"
            ExecuteRet "$fmt_cmd"
            if [ "$fstype" != "swap" -a "$fstype" != "raid" -a "$fstype" != "lvm" ]; then
                cmd="mkdir -p /tmp/MOUNTDIR/$mpt ; mount $dev /tmp/MOUNTDIR/$mpt"
                Log "Mount command [$cmd]"
                ExecuteRet "$cmd"
            else
                Log "Skip MountDrives since fstype=[$fstype]" 
            fi
        done
        ExecuteRet "cp -f $SYS_CONF/fstab $STORAGE_TREE/"
        return 0
    fi

    Execute "rm -rf $STORAGE_TREE/fstab"
    Execute "rm -rf $SYS_CONF/mountpoints.lst"
    Log "Setting mountpoints_to_exclude : proc, dev/shm"
	echo "/proc" > $SYS_CONF/mountpoints_to_exclude.lst
	echo "/dev/shm" >> $SYS_CONF/mountpoints_to_exclude.lst
	cat /system_state/sysconf/skipped_mountpoint_list >> $SYS_CONF/mountpoints_to_exclude.lst
    skipped=`cat /tmp/restore_skip_list`
    Log "Setting mountpoints_to_exclude : user defined excludes if any from /tmp/restore_skip_list  [$skipped]"
    cat /tmp/restore_skip_list >> $SYS_CONF/mountpoints_to_exclude.lst

    tree="$STORAGE_TREE/backup/mount_tree"

    # If we run same script again,we need to remark all mount points again.
    rm -rf $tree/*/recreatedMount

    if [ -f $STORAGE_TREE/VOL_MAP ]
    then
        while read line
        do
            Log "changing each vgname[$vgname] with [$vgname1] in mount_tree."
            cp -a $tree $tree.orig
            vgname=`echo $line|cut -f2 -d' '`
            vgname1=`echo $line|cut -f1 -d' '`
            Log "files with $vgname:`find $tree -type f -exec grep -wH $vgname {} \;`"
            find $tree -type f -exec sed -i "s/\b$vgname\b/$vgname1/g" {} \;
            Log "files with $vgname1:`find $tree -type f -exec grep -wH $vgname1 {} \;`"
        done < $STORAGE_TREE/VOL_MAP
    fi
    tree="$STORAGE_TREE/backup/mount_tree"
    Execute "mkdir -p /tmp/MOUNTDIR"

    Execute "rm -rf $STORAGE_TREE/mntpoint.lst"
    Execute "rm -rf $STORAGE_TREE/sortedmntpoint.lst"
    for mdir in `ls -v $tree`
    do
        curr_mntpnt=`cat $tree/$mdir/mntpoint`
        echo $curr_mntpnt | grep -q ".snapshots"
        if [ $? -eq 0 ]; then
            Log "recreateMountPoints: Debug: skipping snapshot mount[$curr_mntpnt]"
            continue
        fi
        echo "$curr_mntpnt"  >> $STORAGE_TREE/mntpoint.lst
    done
    cat $STORAGE_TREE/mntpoint.lst | sort > $STORAGE_TREE/sortedmntpoint.lst
    prepareBtrfsDisk
    while read mntpoint
    do
        mntmatch=`find $tree -maxdepth 2 -type f -name mntpoint | xargs egrep -wH "${mntpoint}$"`
        #We have same mountpoint name for all swap mount points.
        for i in `echo -e "$mntmatch"`
        do
            fmntmatch=`echo $i|cut -d':' -f1` 
            dmntmatch=`dirname $fmntmatch`
            mdir=`basename $dmntmatch`
            Log "\n Debug: Process [$i]. Check it is already created or not."
            if [ ! -f $tree/$mdir/recreatedMount ]
            then
                touch $tree/$mdir/recreatedMount
                mntmatch=$i
                break
            fi
            Log "$i is already recreated!"
        done
        fmntmatch=`echo $mntmatch | cut -d':' -f1` 
        dmntmatch=`dirname $fmntmatch`
        mdir=`basename $dmntmatch`
        Log "\nrecreateMountPoints: Debug: Process $tree/$mdir"
        to_restore=`cat $tree/$mdir/to_restore`
        #mntpoint=`cat $tree/$mdir/mntpoint`
        if [ "$to_restore" != "1" ]; then
            Log "recreateMountPoints: Debug: to_restore not set. Skip it."
            echo "$mntpoint" >> $SYS_CONF/mountpoints_to_exclude.lst
            continue
        fi
        device=`cat $tree/$mdir/device`
        fstype=`cat $tree/$mdir/fstype`
        label=`cat $tree/$mdir/label`
        raw_fsoptions=`cat $tree/$mdir/raw_fsoptions`
        recreation_info=`cat $tree/$mdir/recreation_info`
        swap_type=`cat $tree/$mdir/swap_type`
        swap_priority=`cat $tree/$mdir/swap_priority`

        if [ x"$fstype" == "xbtrfs" ]; then
            Log "Adding .snapshots directory under $mntpoint to exclude list"
            if [ "$mntpoint" == "/" ]; then
                echo "$mntpoint.snapshots" >> $SYS_CONF/mountpoints_to_exclude.lst
            else
                echo "$mntpoint/.snapshots" >> $SYS_CONF/mountpoints_to_exclude.lst
            fi
        fi

        Log "recreateMountPoints: Debug: Extract device=[$device] mntpoint=[$mntpoint] fstype=[$fstype] label=[$label] recreation_info=[$recreation_info] FSOptions=[$tree/$mdir/raw_fsoptions]"

        isdisk=`echo $recreation_info | grep -E "MD##|LUKS##|LVM##"`
        if [ -z "$isdisk" ]; then
            isdisk=`echo $recreation_info | grep "DISK##"`
            Log "recreateMountPoints: Debug: device is a disk. Rewrite and validate it"
    
            if [ -f $STORAGE_TREE/AUTOSCALESTORAGE ]; then
                bkpdevice=`echo $isdisk | awk '{print $2}'`
                resdevice=`cat $STORAGE_TREE/autoScale2/PART_MAP | grep ^$bkpdevice | awk '{print $2}'`
                res_disk=`echo $resdevice | cut -d ":" -f1`
                partnum=`echo $resdevice | cut -d ":" -f2`
            else
                bck_disk=`echo $isdisk | awk '{print $2}' | cut -d':' -f1`
                partnum=`echo $isdisk | awk '{print $2}' | cut -d':' -f2`
                res_disk=`lookupDiskMap "backup" "$bck_disk"`
            fi
            resdevice=`recreateDeviceName "restore" "$res_disk" "$partnum"`
        else
            if [ -f $STORAGE_TREE/AUTOSCALESTORAGE ]; then
                #Mountpoints residing on MD devices are not supported in AUTOSCALE approach 2   
                is_md=`echo $recreation_info | grep 'MD##' | wc -l`
                if [ $is_md -eq 1 ]; then
                    Log "recreateMountPoints: Mountpoint [$mntpoint] is present of MD device which is not supported when autoScale2 method has been used for disk/part mapping"
                    echo "$mntpoint" >> $SYS_CONF/mountpoints_to_exclude.lst
                    continue
                fi 
            fi
            Log "recreateMountPoints: Debug: device is a LV. validate it"
            resdevice="$device"
        fi

        if [ ! -e "$resdevice" -a ! -L "$resdevice" ]; then
            Log "recreateMountPoints: Error: mapped restore device [$resdevice] not found!"
            return 1
        fi

        if [ -z "$raw_fsoptions" ]; then
            Log "recreateMountPoints: Debug: raw_fsoptions empty..."
            raw_fsoptions="~"
        fi

        if [ "$fstype" != "swap" ]; then
            mkfs_cmd=`generateMKFScmd "$resdevice" "$mntpoint" "$fstype" "$tree/$mdir/raw_fsoptions" "$label"`            
            listentry="$resdevice   $mntpoint   $fstype $label"
        else
            mkfs_cmd=`generateMKSWAPcmd "$resdevice" "$swap_type" "$swap_priority"`
            listentry="$resdevice   swap    swap"
        fi

        if [ x"$fstype" == "xbtrfs" ]
        then
            if [ x"`cat $tree/$mdir/subvolume_path`" == "xroot" ]
            then
                mkfs_cmd="{ mkdir -p /tmp/MOUNTDIR/$mntpoint ;echo \"$resdevice $mntpoint $fstype subvolid=5 0 0\" >> $STORAGE_TREE/fstab; mount -o subvolid=5 $resdevice /tmp/MOUNTDIR/$mntpoint; } || exit 1"
            else
                btrfs_name=`cat $tree/$mdir/subvolume_path|sed "s/^root\///g"`
                mkfs_cmd="{ mkdir -p /tmp/MOUNTDIR/$mntpoint ; echo \"$resdevice $mntpoint $fstype subvol=\"$btrfs_name\" 0 0\" >> $STORAGE_TREE/fstab; mount -o subvol=\"$btrfs_name\" $resdevice /tmp/MOUNTDIR/$mntpoint; } || exit 1"
            fi
        fi
        ExecuteRet "$mkfs_cmd"
        ret=$?
        if [ $ret -ne 0 ]; then
            Log "recreateMountPoints: Error: Failed to create FS with command [$mkfs_cmd]"
            return 1
        else
            Log "recreateMountPoints: Debug: FS creation success with command [$mkfs_cmd]"
            #at this point, generate a list of mountpoints for installbootloader to pick up
            echo $listentry >> $SYS_CONF/mountpoints.lst
            if [ "$mntpoint" == "/" ]; then
                Log "recreateMountPoints: Debug: set root_mnt = [$resdevice]"
                echo "$resdevice" > $SYS_CONF/root_mnt
            fi
        fi
        if [ -f $STORAGE_TREE/retain_diskSerials ]
        then
            while read adisk
            do
                aresdisk=`lookupDiskMap "backup" "$adisk"`
                adiskNode=`DiskNodeForDiskName "backup" "$adisk"`
                adiskserial=`cat $adiskNode/diskserial`
                Log "/partinfo setDiskSerial $aresdisk $adiskserial"
                /partinfo setDiskSerial $aresdisk $adiskserial
            done < $tree/$mdir/disks_for_mntpoint
        fi
    done < $STORAGE_TREE/sortedmntpoint.lst

    #Lets add all bind mounts to exclusion list 
    while read fsentry 
    do
        echo ${fsentry} | grep -w none | grep -w bind 
        ret=$?
        if [ $ret -eq 0 ]; then
            bindmount=`echo ${fsentry} | awk '{print $2}'` 
            echo ${bindmount} >> $SYS_CONF/mountpoints_to_exclude.lst
            Log "recreateMountPoints: Debug: Added bind mount [${bindmount}] to filter list"
        fi
    done < /system_state/sysconf/fstab

    return 0
}

modprobe dm-snapshot
#Assumption: srcompare.sh must have been called and $STORAGE_TREE/recreate_info_all should be present!
#
#For each disk in $STORAGE_TREE/recreate_info_all look up $STORAGE_TREE/restore/disks/ and find the partition type/table details

RESTART_PONT=$1
if [ -z "$RESTART_PONT" ]; then
    RESTART_PONT=0
fi

# Is auto disk layout selected?
is_mtpt=`cat $YASTGUI/mtpt_main | grep OPTIMIZE_STORAGE | cut -d'=' -f2`
if [ "x$is_mtpt" == "x1" ]; then
    Log "Create $AUTOLAYOUTFILE as OPTIMIZE_STORAGE is set" | tee -a $LOG_FILE
    touch $AUTOLAYOUTFILE
else
    Log "Remove $AUTOLAYOUTFILE as OPTIMIZE_STORAGE is unset" | tee -a $LOG_FILE
    rm -f $AUTOLAYOUTFILE
fi

# Is this post-webadm ?
# SEQUENCE
# Following will be required prior
#
#rm -rf $SYS_CONFIG_DIR/fstab
#rm -rf $SYS_CONFIG_DIR/mountpoints.lst
#rm -rf $SYS_CONFIG_DIR/grub_cmdline_args

#    1. IF /tmp/storage_tree/webadm_verified, then it is..
#    2. for each [disk] in the list saved during srcompare recheck, create and run the parted scripts (in commit mode)
#    3. for each /tmp/storage_tree/postadm/lvm_metadata/ (VG), remove old and create new PV with its name/size, create VG and the LVs for it.
#    4. for each mountpoint, mount with provided options (ignore but log errors, in case of errors try to mount w/o options).  Append to $SYS_CONF/mountpoints.lst and $SYS_CONF/fstab each mntpt
#
if [ -f "$STORAGE_TREE/webadm_verified" ]; then
    rm -rf $SYS_CONFIG_DIR/fstab
    rm -rf $SYS_CONFIG_DIR/mountpoints.lst
    rm -rf $SYS_CONFIG_DIR/grub_cmdline_args

    Log "\nDebug: Unmount any previously mounted devices"
    mounted=`mount | grep MOUNTDIR | awk '{print $1}' | sort -r`
    if [ -n "$mounted" ]; then
        Log "Debug: found [$mounted] already mounted"
        ExecuteRet "umount $mounted"
    else
        Log "Debug: no device mounted under MOUNTDIR"
    fi

    #disks & partitions
    CreatePartinfoScriptForDiskList "$STORAGE_TREE/postadm/disks.lst" "postadm" "true" "true"
    ret=$?
    if [ $ret == "1" ]; then
        Log "Error: CreatePartinfoScriptForDiskList failed for list[`cat $STORAGE_TREE/postadm/disks.lst`]" 
        exit 1
    fi

    #cleanup LVM
    moveLibs
    for vg in `ls $STORAGE_TREE/postadm/lvm_metadata/`;
    do
         for lv in `ls $STORAGE_TREE/postadm/lvm_metadata/$vg/lvs/`;
         do
             ExecuteRet "lvm lvremove --force /dev/$vg/$lv"
             ret=$?
             if [ $ret -ne 0 ]; then
                 Log "Debug: failed to remove LV [/dev/$vg/$lv] rc[$ret]"
             fi
         done
         ExecuteRet "vgchange -an $vg"
         ExecuteRet "lvm vgremove --force /dev/$vg"
         ret=$?
         if [ $ret -ne 0 ]; then
             Log "Debug: failed to remove VG [/dev/$vg] rc[$ret]"
         fi
         for pv in `ls $STORAGE_TREE/postadm/lvm_metadata/$vg/pvs/`;
         do
             ExecuteRet "pvremove --force /dev/$pv"
             ret=$?
             if [ $ret -ne 0 ]; then
                 Log "Debug: failed to remove PV [/dev/$pv] rc[$ret]"
             fi
             #As we are removing all pv, it does not matter even if it mpath
             ExecuteRet "pvremove --force /dev/mapper/$pv"
         done
    done

    #create LVM
    for vg in `ls $STORAGE_TREE/postadm/lvm_metadata/`;
    do
        pvsforvg=""
        for pv in `ls $STORAGE_TREE/postadm/lvm_metadata/$vg/pvs/`;
        do
            cat $STORAGE_TREE/postadm/lvm_metadata/$vg/pvs/$pv|grep -q "mapper"
            if [ $? -eq 0 ]
            then
                pvname="/dev/mapper/$pv"
				#FIXME: Will work only for /dev/mapper/mpath devices. Will fail for SLES MPATH.
                pvdisk=`echo $pvname|sed 's/[0-9]*$//'`
                pvnum=`echo $pvname|sed 's/[^0-9]//g'`
                for i in ${DELIM_OPTIONS[*]}
                do
                    Log "Debug: delim[$i].Check device[$pvdisk$i$pvnum] exist or not"
                    pvname=$pvdisk$i$pvnum
                    if [ -e $pvname ]
                    then
                        break
                    fi
                done
                if [ ! -e $pvname ]
                then
                    pvname=$pvdisk$pvnum
                fi
            else
                pvname="/dev/$pv"
            fi
            pvsforvg="${pvsforvg} $pvname"
            ExecuteRet "pvcreate -ff -y $pvname"
            ret=$?
            if [ $ret -ne 0 ]; then
                Log "Error: failed to create PV [$pv] rc[$ret]"
                resetLibs
                exit 1
            fi
        done
        ExecuteRet "vgcreate $vg $pvsforvg"
        ret=$?
        if [ $ret -ne 0 ]; then
            Log "Error: failed to create VG [$vg] PVlist[$pvsforvg] rc[$ret]"
            resetLibs
            exit 1
        fi
        var=0
        lv_count=`ls $STORAGE_TREE/postadm/lvm_metadata/$vg/lvs/ | wc -l`
        for lv in `ls $STORAGE_TREE/postadm/lvm_metadata/$vg/lvs/`;
        do
            var=$((var+1))
            lvsize=`grep SIZE $STORAGE_TREE/postadm/lvm_metadata/$vg/lvs/$lv | awk '{print $2}'`
            lvsize=`expr $lvsize / 1024`
            ExecuteRet "yes | lvcreate -L ${lvsize}K -n $lv /dev/$vg"
            ret=$?
            if [ $ret -ne 0 -a $var -eq $lv_count ]; then
                Log "Debug: using remaining free size of VG[$vg] for last LV[$lv]"
                ExecuteRet "yes | lvcreate -l 100%FREE -n $lv /dev/$vg"
                ret=$?
            fi
            if [ $ret -ne 0 ]; then
                Log "Error: failed to create LV [/dev/$vg/$lv] size [$lvsize] rc[$ret]"
                resetLibs
                exit 1
            fi
        done
    done
    resetLibs

    #mountTree
    for mntnum in `ls $STORAGE_TREE/postadm/mount_tree/`;
    do
        mntdev=`cat $STORAGE_TREE/postadm/mount_tree/$mntnum/device`
        mntdisk=`echo $mntdev| sed 's/[0-9]*$//'`
        partnum=`echo $mntdev| sed 's/[^0-9]//g'`
        mntfstype=`cat $STORAGE_TREE/postadm/mount_tree/$mntnum/fstype`
        mntdir=`cat $STORAGE_TREE/postadm/mount_tree/$mntnum/mntpoint`
        mntopt=`cat $STORAGE_TREE/postadm/mount_tree/$mntnum/mntoptions`   
        if [ -f $STORAGE_TREE/postadm/mount_tree/$mntnum/kpartx_delim ]
        then
            for i in ${DELIM_OPTIONS[*]}
            do
                Log "Debug: delim[$i].Check device[$mdisk$i$partnum] exist or not"
                mntdev=$mntdisk$i$partnum
                if [ -e $mntdev ]
                then
                    break
                fi
            done
            if [ ! -e $mntdev ]
            then
                mntdev=$mntdisk$partnum
            fi
        fi
        if [ x"$mntdev" == "x" -o x"$mntfstype" == "x" ]; then
            Log "Error: Invalid mount entry -- dev[$mntdev] dir[$mntdir] fstype[$mntfstype]"
            exit 1
        fi
        if [ x"$mntdir" == "x/" ]; then
            echo $mntdev > "$SYS_CONF/root_mnt"
        fi
        case $mntfstype in
            "ext2"|"ext3"|"ext4")
                mkfs_cmd="mkfs -t $mntfstype -F $mntdev"
                ;;
            "vfat")
                mkfs_cmd="mkfs.vfat $mntdev"
                ;;
            "reiserfs")
                mkfs_cmd="mkreiserfs -f $mntdev"
                ;;
            "xfs")
                mkfs_cmd="mkfs.xfs -f $mntdev"
                ;;
            "swap")
                mkfs_cmd="mkswap $mntdev"
                ;;
            "btrfs")
                mkfs_cmd="mkfs.btrfs -f $mntdev"
                ;;
            "*")
                Log "Error: Unsupported FStype [$mntfstype] !!"
                exit 1 
                ;;
        esac
        ExecuteRet "$mkfs_cmd"
        ret=$?
        if [ $ret -ne 0 ]; then
            Log "Error: failed to create FS with cmd [$mkfs_cmd]"
            exit 1
        fi
        if [ "$mntfstype" != "swap" ]; then
            mntdir_orig=${mntdir}
            mntdir="/tmp/MOUNTDIR/${mntdir}"
            ExecuteRet "mkdir -p $mntdir"
        else
            echo "$mntdev   swap  swap"  >> "$SYS_CONF/mountpoints.lst"
            echo "$mntdev   swap  swap  defaults 0 0"  >> "$SYS_CONF/fstab"
            continue
        fi
        if [ -z "$mntopt" ]; then
            ExecuteRet "/bin/mount -t $mntfstype $mntdev $mntdir"
            ret=$?
            if [ $ret -ne 0 ]; then
                Log "Error: Failed to mount [$mntdev] on [$mntdir] having fstype [$mntfstype] rc[$ret]"
            else
                Log "Debug: successfully mounted [$mntdev] on [$mntdir] having fstype [$mntfstype]"
            fi
        else
            ExecuteRet "/bin/mount -t $mntfstype -o $mntopt $mntdev $mntdir"
            ret=$?
            if [ $ret -ne 0 ]; then
                Log "Error: Failed to mount [$mntdev] on [$mntdir] having fstype [$mntfstype] with options [$mntopt] rc[$ret], trying w/o options"
                ExecuteRet "/bin/mount -t $mntfstype $mntdev $mntdir"
                ret=$?
                if [ $ret -ne 0 ]; then
                    Log "Error: Failed to mount [$mntdev] on [$mntdir] having fstype [$mntfstype] rc[$ret]"
                else
                    Log "Debug: successfully mounted [$mntdev] on [$mntdir] having fstype [$mntfstype]"
                fi
            else
                Log "Debug: successfully mounted [$mntdev] on [$mntdir] having fstype [$mntfstype] with options [$mntopt]"
            fi
        fi
        echo "$mntdev $mntdir_orig $mntfstype" >> "$SYS_CONF/mountpoints.lst"
        if [ -z "$mntopt" ]; then        
            echo "$mntdev $mntdir_orig $mntfstype defaults 0 0" >> "$SYS_CONF/fstab"
        else
            echo "$mntdev $mntdir_orig $mntfstype $mntopt 0 0" >> "$SYS_CONF/fstab"
        fi
    done

    grep -v rootfs /proc/mounts > /etc/mtab
    ExecuteRet "cp -f $SYS_CONF/fstab $STORAGE_TREE/fstab"
    #Set appropriate flags required for initrd recreation and grub config modification
    touch /tmp/adm_invoked
    #adding mountpoint which we are excluded in /tmp/restore_skip_list
    if [ -f /tmp/restore_skip_list ]
    then
        cat /tmp/restore_skip_list >> $SYS_CONF/mountpoints_to_exclude.lst
        Log " following folders are excluded for restore [ `cat $SYS_CONF/mountpoints_to_exclude.lst` ]"
    fi
    exit 0
fi


if [ $RESTART_PONT -lt 1 ]; then
    Log "\nDebug: Unmount any previously mounted devices"
    mounted=`mount | grep MOUNTDIR | awk '{print $1}' | sort -r`
    if [ -n "$mounted" ]; then
        Log "Debug: found [$mounted] already mounted"
        ExecuteRet "umount $mounted"
    else
        Log "Debug: no device mounted under MOUNTDIR"
    fi

    moveLibs
    Log "\nDebug: deactivate all volume groups"
    if [ ! -f /cloud_flag ]
    then
        #Clean up Area! Clean anything unneccasary metadata here!
        ExecuteRet "vgchange -an"
        devices=`dmsetup ls --target crypt|awk '{print $1}'`
        for i in `echo -e "$devices"`
        do
            crypsetup luksClose $i
            Log "Crypt device [$i] closed"
        done
        for i in `mdadm --detail --scan|awk '{print $2}'`
        do
            mdadm --remove $i
            mdadm --stop $i
        done
    fi
    resetLibs

    Log "\nDebug: Begin recreatePartitions\n"
    recreatePartitions $STORAGE_TREE/recreate_info_all $AUTOLAYOUTFILE
    ret=$?
    if [ $ret -ne 0 ]; then
        Log "Error: Failed to recreate partitions!"
        exit 1
    else
        Log "Debug: Recreated all partitions"
    fi
    RESTART_PONT=$((RESTART_PONT+1))
fi

if [ $RESTART_PONT -lt 2 ]; then
    Log "\nDebug: Begin recreateLVM\n"
    recreateLVM $STORAGE_TREE/recreate_info_all $AUTOLAYOUTFILE
    ret=$?
    if [ $ret -ne 0 ]; then
        Log "Error: Failed to recreate LVM!"
        exit 1
    else
        Log "Debug: Recreated all LVM storage"
    fi
    RESTART_PONT=$((RESTART_PONT+1))
fi

if [ $RESTART_PONT -lt 3 ]; then
    Log "\nDebug: Begin recreateMD\n"
    recreateMD $STORAGE_TREE/recreate_info_all $AUTOLAYOUTFILE
    ret=$?
    if [ $ret -ne 0 ]; then
        Log "Error: Failed to recreate MD device!"
        exit 1
    else
        Log "Debug: Recreated all MD storage"
    fi
    RESTART_PONT=$((RESTART_PONT+1))
fi

if [ $RESTART_PONT -lt 4 ]; then
    if [ -d /$SYS_STATE_LOC/crypt_metadata ]
    then
        Log "\nDebug: encrypt file is found!"
        recreateCryptDevice $STORAGE_TREE/recreate_info_all 
        ret=$?
        if [ $ret -ne 0 ]
        then
            Log "Error: Failed to encrypted Devices. Please confirm password or key file."
            exit 1
        else
            Log "Debug: recreated all encrypted devices!"
        fi
    fi
    Log "\nDebug: Begin recreateMountPoints\n"
    recreateMountPoints $AUTOLAYOUTFILE
    ret=$?
    if [ $ret -ne 0 ]; then
        Log "Error: Failed to recreate mount points!"
        exit 1
    else
        Log "Debug: Recreated all mountpoints"
    fi
    if [ ! -f $STORAGE_TREE/fstab ]; then
        Log "postCompare: Error: Failed to generate new fstab!"
        exit 1
    fi
    df -Ph /tmp/MOUNTDIR/* | tail -n+2 | grep -vE 'boot|var|lib|usr|root|opt|home|mnt|^/dev/sd|MOUNTDIR$' | awk '{print $1}' > $STORAGE_TREE/non_sys_lvs
    sed -i 's/ /\t/g' $STORAGE_TREE/fstab
    grep -v rootfs /proc/mounts > /etc/mtab
    RESTART_PONT=$((RESTART_PONT+1))
fi

if [ -f /tmp/override/cryptFile.key ]
then
    while read line
    do
        ispass=`echo $line| awk '{print $2}'`
        if [ "x$ispass" != "xp" ]
        then
            continue
        fi
        dev=`echo $line|awk '{print $1}'`
        grep -v "^/dev/mapper/`basename $dev`" $STORAGE_TREE/fstab > $STORAGE_TREE/fstab.new
        mv $STORAGE_TREE/fstab.new $STORAGE_TREE/fstab
    done < /tmp/override/cryptFile.key
fi

if [ ! -f $SYS_STATE_LOC/use_FSTAB_name ]
then
	mv $STORAGE_TREE/fstab $STORAGE_TREE/fstab.cloud
	fstabDiskToUUID $STORAGE_TREE/fstab.cloud $STORAGE_TREE/fstab
fi

. /common.sh #doing this here to avoid conflicts with common_funcs.sh - this is needed only for vm_percent
vm_percent 70
exit 0
