#!/bin/bash

. /common_funcs.sh
STORAGE_TREE="/tmp/storage_tree"


#Reset Libs
#    Name: resetLibs
#    API DEF:
resetLibs()
{
    Log "Move devmapper libraries from /libs to /"
    mv /libs/* /
}

#Move Libs
#   Name: moveLibs
#   API DEF:
moveLibs()
{
    Log "Move devmapper libraries from / to /libs"
    mkdir -p /libs
    mv /libdevmapper* /libs
    mv /libparted* /libs
}

#Prepare storage trees
#   Name: PrepareTree
#   API DEF:
#       IN:             arg1=system_state_loc(backup) arg2=sysconfig_loc(recovery)
#       OUT:            Copy of tree in /tmp/storage_tree - return 0(success) or 1(failure)
#       EXIT:           no
PrepareTree()
{
    bckdir=$1
    resdir=$2

    if [ ! -d $bckdir ]; then
        Log "PrepareTree: Error: Backed up system state [$bckdir] not found"
        return 1
    fi
    if [ ! -d $resdir ]; then
        Log "PrepareTree: Error: Current system sysconfig [$resdir] not found"
        return 1
    fi
    Execute "rm -rf $STORAGE_TREE"
    Execute "mkdir -p $STORAGE_TREE"
    Execute "mkdir -p $STORAGE_TREE/backup"
    Execute "mkdir -p $STORAGE_TREE/restore"

    Execute "cp -a $bckdir/disklist $STORAGE_TREE/backup"
    Execute "cp -a $bckdir/disks $STORAGE_TREE/backup"
    Execute "cp -a $bckdir/mount_tree $STORAGE_TREE/backup"
    Execute "cp -a $bckdir/lvm_metadata $STORAGE_TREE/backup"
    Execute "cp -a $bckdir/SIMPANA_MOUNT_PATH $STORAGE_TREE/backup"
    
    Execute "cp -a $resdir/disklist $STORAGE_TREE/restore"
    Execute "cp -a $resdir/disks $STORAGE_TREE/restore"
    Execute "cp -a $resdir/mount_tree $STORAGE_TREE/restore"
    Execute "cp -a $resdir/lvm_metadata $STORAGE_TREE/restore"
   
    #for older backups, we might need to massage system state based on fixes made eventually. do this here.

    #diskserial for full disk PVs used to be enumerated incorrectly. we now use PVUUID as disk serial for full disk PVs where no partition table is present.
    while read pvline;
    do
        pvname=`echo $pvline | awk '{print $1}'`
        pvuuid=`echo $pvline | awk '{print $2}'`
        grep --quiet $pvname $STORAGE_TREE/backup/disklist
        ret=$?
        if [ $ret -eq 0 ]; then
            Log "PV [$pvname][$pvuuid] is a full disk. update it's diskserial in backup"
            basediskname=`basename $pvname`
            nodedir=`ExecFn "find $STORAGE_TREE/backup/disks -type d -name $basediskname"`
            if [ -z "$nodedir" ]; then
                Log "Error: Could not find disk node for [$basename] in [$STORAGE_TREE/backup/disks] !"
                continue
            fi
            oldserial=`cat $nodedir/diskserial`
            if [ -z "$oldserial" ]; then
                Log "Error: Could not read [$nodedir/diskserial]!"
                continue
            fi
            echo $pvuuid > $nodedir/diskserial
            Log "Debug: Update [$nodedir/diskserial] with [$pvuuid]"
        else
            Log "ignore PV [$pvname][$pvuuid] as it is not a full disk"
        fi

    done < $STORAGE_TREE/backup/lvm_metadata/pvs.list

    #kparx delimiter might have been incorrectly parsed during backup. Update it
    if [ ! -f /tmp/kpartx_delim ]; then
        SaveKpartxDelimiter
    fi
    delim=`cat /system_state/sysconf/kpartx_delim`
    if [ -z "$delim" ]; then
        Log "kpartx delimiter in system state backup is empty. Update it" 
        touch /tmp/kpartx_delim
        cp -f /tmp/kpartx_delim /system_state/sysconf/kpartx_delim
    fi
}

#Disks for mount point
#   Name: DisksForMountPoint
#   API DEF:
#       IN:             arg1=from_tree(backup|restore) arg2=mount_point#(within arg1)
#       OUT:            Validate mount point subdirectory and return string 'WS' separated list of disks; empty string on error
#       EXIT:           no
DisksForMountPoint()
{
    tree=$1
    mntnum=$2
    if [ ! -d $STORAGE_TREE/$tree ]; then
        Log "DisksForMountPoint: Error: Tree not found [$STORAGE_TREE/$tree]"
        return
    fi
    if [ ! -d $STORAGE_TREE/$tree/mount_tree/$mntnum ]; then
        Log "DisksForMountPoint: Error: Mount point subdirectory not found [$STORAGE_TREE/$tree/$mntnum]"
        return
    fi
    DIR="$STORAGE_TREE/$tree/mount_tree/$mntnum"
    cur_mnt=`cat $DIR/mntpoint`
    if [ -z "$cur_mnt" ]; then
        Log "DisksForMountPoint: Error: Cannot read mntpoint from [$DIR]"
        return
    fi
    cur_dev=`cat $DIR/device`
    if [ -z "$cur_dev" ]; then
        Log "DisksForMountPoint: Error: Cannot read device from [$DIR]"
        return
    fi
    cur_disks=`cat $DIR/disks_for_mntpoint`
    if [ -z "$cur_disks" ]; then
        Log "DisksForMountPoint: Error: Cannot read disks_for_mntpoint from [$DIR]"
        return
    fi

    Log "DisksForMountPoint: Debug: Return disks [$cur_disks] from [$DIR]"
    echo -e "$cur_disks"
}

#Disk node in storage tree for disk name
#   Name: DiskNodeForDiskName
#   API DEF:
#       IN:             arg1=from_tree(backup|restore) arg2=disk_name
#       OUT:            Disk node within STORAGE_TREE
#       EXIT:           no
DiskNodeForDiskName()
{
    tree=$1
    diskname=$2

    if [ ! -d $STORAGE_TREE/$tree ]; then
        Log "DiskNodeForDiskName: Error: Tree not found [$STORAGE_TREE/$tree]"
        return
    fi

    if [ -z $diskname ]; then
        Log "DiskNodeForDiskName: Error: Argument#2 (disk name) empty"
        return
    fi
    shortdiskname=`basename $2`
    Log "DiskNodeForDiskName: Debug: Looking for [$diskname] in [$STORAGE_TREE/$tree] using basename [$shortdiskname]"
    
    disknode=`ExecFn "find $STORAGE_TREE/$tree/disks -maxdepth 2 -name $shortdiskname"`
    
    if [ -n "$disknode" ]; then
        Log "DiskNodeForDiskName: Debug: Found node [$disknode] for [$diskname] in [$STORAGE_TREE/$tree]"
        echo $disknode
    else
        Log "DiskNodeForDiskName: Error: Could not find node for [$diskname] in [$STORAGE_TREE/$tree]"
        echo ""
    fi
}

#Disk node in storage tree for diskserial and size(disks size should be less than given size in second parameter)
#   Name: DiskNodeForDiskSerialAndSize
#   API DEF:
#       IN:             arg1=from_tree(backup|restore) arg2=diskserial arg3=size
#       OUT:            Disk node within STORAGE_TREE
#       EXIT:           no
DiskNodeForDiskSerialAndSize()
{
    tree=$1
    diskserial=$2
    disksize=$3
    if [ ! -d $STORAGE_TREE/$tree ]; then
        Log "DiskNodeForDiskSerialAndSize: Error: Tree not found [$STORAGE_TREE/$tree]"
        return
    fi

    if [ -z $diskserial ]; then
        Log "DiskNodeForDiskSerialAndSize: Error: Argument#2 (diskserial) empty"
        return
    fi
    Log "DiskNodeForDiskSerialAndSize: Debug: Looking for [$diskserial] in [$STORAGE_TREE/$tree]"
    
    #First check all DM devices
    if [ -d "$STORAGE_TREE/$tree/disks/DM" ]; then
        list=`ExecFn "find $STORAGE_TREE/$tree/disks/DM -name diskserial"`
        for file in $list;
        do
            serial=`cat $file`
            if [ "$serial" == "$diskserial" ]; then
                retdisk=`dirname $file`
                cat $STORAGE_TREE/storage_used | grep -wq `cat $retdisk/dev_path`
                if [ $? -eq 0 ]
                then
                    continue
                fi
                retdisksize=`cat $retdisk/size`
                if [ "$retdisksize" -lt "$disksize" ]
                then
                    continue
                fi
                Log "Debug: Found match for diskserial [$diskserial] in [$file] disknode=[$retdisk]"
                echo $retdisk
                return
            fi
        done
    fi
    Log "Execute find $STORAGE_TREE/$tree/disks -maxdepth 3 -name diskserial | grep -v \"$STORAGE_TREE/$tree/disks/DM\""
    list=`find $STORAGE_TREE/$tree/disks -maxdepth 3 -name diskserial | grep -v \"$STORAGE_TREE/$tree/disks/DM\"`
    for file in $list;
    do
        serial=`cat $file`
        if [ "$serial" == "$diskserial" ]; then
            retdisk=`dirname $file`
            cat $STORAGE_TREE/storage_used | grep -wq `cat $retdisk/dev_path`
            if [ $? -eq 0 ]
            then
                continue
            fi
            retdisksize=`cat $retdisk/size`
            if [ "$retdisksize" -lt "$disksize" ]
            then
                continue
            fi
            Log "Debug: Found match for diskserial [$diskserial] in [$file] disknode=[$retdisk]"
            echo $retdisk
            return
        fi
    done

    echo ""
    return
}

#set to_restore flag on mountpoints
#   Name: setToRestore
#   API DEF:
#       IN:             arg1=from_tree(backup|restore) arg2=mount_point_num (0=all)
#       OUT:            none (touch to_restore for each selected mount_point)
#       EXIT:           no
setToRestore()
{
    tree=$1
    mountnum=$2
    if [ ! -d $STORAGE_TREE/$tree ]; then
        Log "setToRestore: Error: Tree not found [$STORAGE_TREE/$tree]"
        return 1
    fi
    if [ "$mountnum" == "0" ]; then
        for dir in `ls $STORAGE_TREE/$tree/mount_tree`;
        do
            Log "setToRestore: Debug: set to_restore for [$dir]"
            echo "1" > $STORAGE_TREE/$tree/mount_tree/$dir/to_restore
        done
    else
        dir="$STORAGE_TREE/$tree/mount_tree/$mountnum"
        if [ -d "$dir" ]; then
            Log "setToRestore: Debug: set to_restore for [$dir]"
            echo "1" > $tree/mount_tree/$dir/to_restore
        else
            Log "setToRestore: Error: [$dir] not found"
            return 1
        fi
    fi
    return 0
}

#Check if storage is identical
#   Name: checkIdenticalStorage
#   API DEF:
#       IN:             none <backup mount_tree with to_restore set to 1 for each mount_point required to be restored>
#       OUT:            0=true 1=false 2=error. $STORAGE_TREE/storage_missing_all to list all missing storage device paths
#       EXIT:           no
#
#   Format of DISK_MAP FILE:
#       <backup_disk_path>      <current_disk_path>     <0=IDENTICAL|1=SIMILAR|2=MANUAL>
checkIdenticalStorage()
{
    mdir=$STORAGE_TREE/backup/mount_tree
    Execute "rm -f $STORAGE_TREE/storage_missing_all"
    Execute "rm -f $STORAGE_TREE/DISK_MAP"
    Execute "rm -f $STORAGE_TREE/retain_diskSerials"
    Execute "rm -f $STORAGE_TREE/storage_used"
    rm -f $STORAGE_TREE/retain_diskSerials
    if [ -f /tmp/override/DISK_MAP ]
    then
        Log "DISK_MAP is Overrided!! "
        cp /tmp/override/DISK_MAP $STORAGE_TREE/DISK_MAP
    fi
    for mountnum in `ls $mdir`;
    do
        Log "\nDebug: Process mountnum=[$mdir/$mountnum]"
        DIR=$mdir/$mountnum
        if [ ! -d $DIR ]; then
            Log "Error: mount dir [$DIR] does not exist!"
            return 2
        fi
        if [ ! -f $DIR/to_restore ]; then
            Log "Debug: $DIR/to_restore does not exist. Skip it"
            continue
        fi
        restore_flag=`cat $DIR/to_restore`
        if [ "x$restore_flag" != "x1" ]; then
            Log "Debug: $DIR/to_restore [$restore_flag] not set. Skip it"
            continue
        fi
        mntpoint=`cat $DIR/mntpoint`
        reqdisks=`cat $DIR/disks_for_mntpoint`

        Execute "rm -f $DIR/storage_missing"

        for disk in $reqdisks;
        do
            check_disk=`cat $STORAGE_TREE/DISK_MAP|grep -w "^$disk"`
            Log "Debug checkIdenticalStorage:[$disk],[$check_disk]"
            if [ "x$check_disk" != "x" ]
            then
                Log "For backup disk[$disk],entry is already there in DISK_MAP.Skipping!!"
                resdisk=`cat $STORAGE_TREE/DISK_MAP|grep -w "^$disk"|head -n 1|awk '{print $2}'`
                checkdisk=`cat $STORAGE_TREE/storage_used |grep -w "^$resdisk"`
                if [ "x$checkdisk" == "x" ]
                then
                    echo $resdisk >> $STORAGE_TREE/storage_used
                fi
                continue
            fi
            if [ -f /tmp/override/mapDiskName ]
            then
                echo "$disk     $disk    0" >> $STORAGE_TREE/DISK_MAP
                echo "$disk" >> $STORAGE_TREE/storage_used
                continue
            fi
            bckdiskNode=`DiskNodeForDiskName "backup" "$disk"`
            if [ -z "$bckdiskNode" ]; then
                Log "Error: Backup disk [$disk] for mount point [$mntpoint][$DIR] not found! Check system state backup!"
                return 2
            fi
            bckdiskSerial=`cat $bckdiskNode/diskserial`
            bcksize=`cat $bckdiskNode/size` 
            Log "Debug: Extract diskserial#=[$bckdiskSerial],size[$bcksize] for [$disk] required for mount point [$mntpoint]"

            resdiskNode=`DiskNodeForDiskSerialAndSize "restore" "$bckdiskSerial" "$bcksize"`
            if [ -z "$resdiskNode" ]; then
                Log "Debug: Could not find a disk with serial [$bckdiskSerial] in [$STORAGE_TREE/restore/disks]. This would need IDM to map disks!"
                Log "Mark mount_point with flag storage_missing [$DIR]"
                echo "$disk" >> $DIR/storage_missing
                echo "$disk" >> $STORAGE_TREE/storage_missing_all
            else
                resdiskname=`cat $resdiskNode/dev_path`
                Log "Debug: Found same disk [$disk] on restored machine [$resdiskname]. This mountpoint can be recreated, if no other storage is missing"
                echo "$resdiskname" >> $STORAGE_TREE/storage_used
                echo "$disk     $resdiskname    0" >> $STORAGE_TREE/DISK_MAP
            fi
        done
        if [ ! -f $DIR/storage_missing ]; then
            Log "Debug: All storage devices for [$mntpoint] found on current system."
            touch $DIR/storage_present
        fi
    done
    if [ ! -f $STORAGE_TREE/storage_missing_all ]; then
        #Nothing else to do!
        Log "\nDebug: All disks found identical to backed up machine. This is a recovery to the same machine/disks"
        touch $STORAGE_TREE/retain_diskSerials
        return 0
    else
        Log "\nDebug: Not all disks found identical to backed up machine. This could be a recovery to a machine with dis-similar storage or replaced storage"
        cat $STORAGE_TREE/storage_missing_all | sort -u > /tmp/tmpfile.$$
        mv /tmp/tmpfile.$$ $STORAGE_TREE/storage_missing_all
        return 1
    fi
}

#List new disks (previously unmapped disks on current system)
#   Name: listNewDisks
#   API DEF:
#       IN:             arg1=<full_disklist> arg2=<used_disks>  arg3=<new_disks>   (all arguments are filenames)
#       OUT:            0=success 1=failures
#       EXIT:           no
listNewDisks()
{
    f_disklist=$1
    f_st_used=$2
    f_st_new=$3

    Execute "rm -f $f_st_new"

    if [ ! -s $f_disklist ]; then
        Log "listNewDisks: disklist file empty [$f_disklist]"
        return 1
    fi

    rm -rf /tmp/dockerDisks
    
    if [ -f $SYS_STATE_LOC/dockerPVuuid ]
    then
        while read pvuuid
        do
            deviceName=`pvs -o pv_name,pv_uuid | grep "$pvuuid" | awk '{print $1}'`
            diskName=`getDiskForDev $deviceName`
            echo $diskName >> /tmp/dockerDisks
        done < $SYS_STATE_LOC/dockerPVuuid
    fi

    cat $f_disklist | while read disk
    do
        Log "listNewDisks: Debug: Remove slave disks"
        slavefol="`dirname $f_disklist`/disks/DM/*/slaves"
        is_slave=""
        for i in `ls $slavefol`
        do
            if [ "x$i" == "x`basename $disk`" ]
            then
                Log "listNewDisks: Debug: disk [$disk] is slave disk."
                is_slave=1
				rdiskNode=`DiskNodeForDiskName "restore" "$diskname"`
				echo 1 > $rdiskNode/is_slave
                break
            fi
        done
        if [ -n "$is_slave" ]
        then
            Log "listNewDisks: Debug: Ignoring slave disk[$disk]"
            continue
        fi
        Log "listNewDisks: Debug: look for [$disk] in [$f_st_used]"
        if [ -f "$f_st_used" ]; then
            is_used=`grep -w $disk $f_st_used`
        else
            is_used=""
        fi

        if [ -f "/tmp/dockerDisks" ]; then
            is_docker=`grep -w $disk /tmp/dockerDisks`
        else
            is_docker=""
        fi

        if [ -n "$is_docker" ] || [ -n "$is_used" ]; then
            Log "listNewDisks: Debug : disk [$is_docker][$is_used] is used"
        else
            echo $disk >> $f_st_new
            Log "listNewDisks: Debug: disk [$disk] not used. Add to [$f_st_new]"
        fi
    done
    return 0
}

#Sort list of disk names by size
#   Name: sortDisksbySize
#   API DEF:
#       IN:             arg1=from_tree(backup|restore) arg2=disklist_file
#       OUT:            WS separated disk list in ascending order of size, empty string on errors
#       EXIT:           no
sortDisksbySize()
{
    tree=$1
    disklist=$2

    if [ ! -d $STORAGE_TREE/$tree ]; then
        Log "sortDisksbySize: Error: Tree not found [$STORAGE_TREE/$tree]"
        echo ""
        return
    fi

    Execute "rm -f $STORAGE_TREE/restore/disklist_bysize"

    cat $disklist | while read disk
    do
        Log "\nsortDisksbySize: Debug: Process disk [$disk]"
        disknode=`DiskNodeForDiskName "restore" $disk`
        if [ -z "$disknode" ]; then
            Log "sortDisksbySize: Error: Disk node not found for [$disk]"
            echo ""
            return
        fi
        removable=`cat $disknode/removable`
        path=`cat $disknode/dev_path`
        size=`cat $disknode/size`
        Log "sortDisksbySize: Debug: path=[$path] size=[$size] removable=[$removable] for [$disk]"
        if [ "$removable" == "1" ]; then
            Log "sortDisksbySize: Debug: Skip removable device [$disk]"
            continue
        fi
        echo "$size    $path" >> $STORAGE_TREE/restore/disklist_bysize
    done
    #Sort the file
    cat $STORAGE_TREE/restore/disklist_bysize | sort -n -k 1 > $STORAGE_TREE/restore/disklist_bysize.new
    mv $STORAGE_TREE/restore/disklist_bysize.new $STORAGE_TREE/restore/disklist_bysize
    disklist=`cat $STORAGE_TREE/restore/disklist_bysize | awk '{print $2}'`
    Log "sortDisksbySize: Return [$disklist]"
    echo $disklist
}

#Sort list of disk names by PCI address
#   Name: sortDisksbyPCI
#   API DEF:
#       IN:             arg1=from_tree(backup|restore) arg2=disklist_file
#       OUT:            WS separated disk list in ascending order of PCI addresses, empty string on errors
#       EXIT:           no
sortDisksbyPCI()
{
    tree=$1
    disklist=$2

    if [ ! -d $STORAGE_TREE/$tree ]; then
        Log "sortDisksbyPCI: Error: Tree not found [$STORAGE_TREE/$tree]"
        echo ""
        return
    fi

    Execute "rm -f $STORAGE_TREE/restore/disklist_byPCI"

    cat $disklist | while read disk
    do
        Log "\nsortDisksbyPCI: Debug: Process disk [$disk]"
        disknode=`DiskNodeForDiskName "restore" $disk`
        if [ -z "$disknode" ]; then
            Log "sortDisksbyPCI: Error: Disk node not found for [$disk]"
            echo ""
            return
        fi
        removable=`cat $disknode/removable`
        path=`cat $disknode/dev_path`
        size=`cat $disknode/size`
        pciId=`getDevPCIId $path`
        Log "sortDisksbyPCI: Debug: path=[$path] size=[$size] removable=[$removable] pciId=[$pciId] for [$disk]"
        if [ "$removable" == "1" ]; then
            Log "sortDisksbyPCI: Debug: Skip removable device [$disk]"
            continue
        fi
        echo "$pciId    $size  $path" >> $STORAGE_TREE/restore/disklist_byPCI
    done
    #Sort the file
    cat $STORAGE_TREE/restore/disklist_byPCI | sort -n -k 1 > $STORAGE_TREE/restore/disklist_byPCI.new
    mv $STORAGE_TREE/restore/disklist_byPCI.new $STORAGE_TREE/restore/disklist_byPCI
    disklist=`cat $STORAGE_TREE/restore/disklist_byPCI | awk '{print $2}'`
    Log "sortDisksbyPCI: Return [$disklist]"
    echo $disklist
}

#Find the smallest disk >= size in list
#   Name: findDiskbySize
#   API DEF:
#       IN:             arg1=size   arg2=disklist_bysize containing new disks arg3=file containing all used(mapped) disks
#       OUT:            string:diskname or empty string if not found
#       EXIT:           no
findDiskbySize()
{
    insize=$1
    disklistfile=$2
    st_usedfile=$3
    Log "findDiskbySize: Debug: invoked with disksize=[$insize] file [$disklistfile] contents[`cat $disklistfile`]"
    Execute "rm -f /tmp/found.$$"

    cat $disklistfile | while read line
    do
        size=`echo $line | awk '{print $1}'`
        name=`echo $line | awk '{print $2}'`
        Log "findDiskbySize: Debug: Process [$name]:[$size] in $disklistfile"

        if [ "$insize" -le "$size" ]; then
            Log "findDiskbySize: Debug: Found match [$name] with size[$size] for insize=[$insize]. Check if used or available"
            grep --quiet $name $st_usedfile
            if [ $? -ne 0 ]; then
                echo $name
                touch "/tmp/found.$$"
                break
            else
                Log "findDiskbySize: Debug : Disk [$name] is already mapped, so cannot be re-used!"
            fi
        fi
    done
    if [ ! -f "/tmp/found.$$" ]; then
        Log "findDiskbySize: Error: Cannot find a new Disk with size >= [$insize] in $disklistfile"
        echo ""
    fi
    Execute "rm -f /tmp/found.$$"
    return
}

#Move from new to used
#   Name: moveNewtoUsed
#   API DEF:
#       IN:             arg1=diskName arg2=newFile arg3=newDiskName arg4=usedFile
#       OUT:            none
#       EXIT:           no
moveNewtoUsed()
{
    name=$1
    f_st_new=$2
    newname=$3
    f_st_used=$4

    cat $f_st_new | grep -vw "$name" > /tmp/tmpfile.$$
    mv /tmp/tmpfile.$$ $f_st_new

    echo "$newname" >> $f_st_used

}


#Find the disk node by mountpoint name, assuming only 1 disk is required for it
#   Name: findDiskbyMntPoint
#   API DEF:
#       IN:             arg1=tree arg2=mountpoint-name
#       OUT:            disk node
#       EXIT:           no
findDiskbyMntPoint()
{
    tree=$1
    mntpointname=$2

    if [ ! -d $STORAGE_TREE/$tree ]; then
        Log "findDiskbyMntPoint: Error: Tree not found [$STORAGE_TREE/$tree]"
        return
    fi

    if [ -z $mntpointname ]; then
        Log "findDiskbyMntPoint: Error: Argument#2 (mountpoint name) empty"
        return
    fi

    Log "findDiskbyMntPoint: Debug: Looking for [$mntpointname] in [$STORAGE_TREE/$tree]"
    node=`ExecFn "grep -lw $mntpointname $STORAGE_TREE/$tree/mount_tree/*/mntpoint"`
    
    if [ -n "$node" ]; then
        Log "findDiskbyMntPoint: Debug: Found node [$node] for [$mntpoint] in [$STORAGE_TREE/$tree]"
        Log "findDiskbyMntPoint: Debug: Looking for disks needed for disk node [$node]"
    dnode=`dirname $node`
        disks=`ExecFn "cat $dnode/disks_for_mntpoint"`
        if [ -n "$disks" ]; then
            Log "findDiskbyMntPoint: Debug: Found disks required for mountpoint [$mntpointname] => [$disks]"
            echo $disks
        else
            Log "findDiskbyMntPoint: Error: Could not find disks needed for mountpoint [$mntpointname] from node [$node]"
            echo ""
        fi
    else
        Log "findDiskbyMntPoint: Error: Could not find node for [$mntpoint] in [$STORAGE_TREE/$tree]"
        echo ""
    fi
}

#Check if storage can be automatically mapped
#   Assumption: checkIdenticalStorage must be called prior to making this call
#   Name: autoMapStorage
#   API DEF:
#       IN:             arg1=mount_tree with to_restore set to 1 for each mount_point required to be restored
#       OUT:            0=true 1=false 2=error.
#       EXIT:           no
#   LOGIC:
#       $STORAGE_TREE/storage_missing_all contains all missing device paths.  --> [A]
#       $STORAGE_TREE/storage_used contains all USED disks. Filter them from $STORAGE_TREE/restore/disks and save as $STORAGE_TREE/storage_new  --> [B]
#       For each disk in [A] map to [B] and if found, add to $STORAGE_TREE/DISK_MAP. If all could be mapped, return success, else fail
#       Update mountpoint directory that contained storage_missing. Recompute dependency in the end
autoMapStorage()
{
    mdir=$STORAGE_TREE/backup/mount_tree
    f_st_missing=$STORAGE_TREE/storage_missing_all
    f_st_missing_b4sort=$STORAGE_TREE/storage_missing_all_b4sort
    f_st_missing_sortfile=$STORAGE_TREE/storage_missing_all_sortfile
    f_st_used=$STORAGE_TREE/storage_used
    f_st_new=$STORAGE_TREE/storage_new
    f_diskmap=$STORAGE_TREE/DISK_MAP

    Log "Debug : autoMapStorage"

    Execute "rm -f $f_st_missing_b4sort"

    if [ ! -s $f_st_missing ]; then
       Log "autoMapStorage: Debug: No missing disks in [$f_st_missing]. return success"
       return 0
    else
        #Prepare new disks [Set of current disks - set of required and present disks, previously mapped]
        listNewDisks $STORAGE_TREE/restore/disklist $f_st_used $f_st_new
        ret=$?
        if [ $ret != "0" ]; then
            Log "autoMapStorage: Error: Failed to list new disks"
            return 1
        fi
        #There are some missing disks. Check if there are any new disks...
        if [ ! -s $f_st_new ]; then
            Log "autoMapStorage: Error: No new disks in [$f_st_new]."
            return 1
        fi

        #For each disk in storage_missing_all, find the smallest disk in storage_new whose size >= the missing disk. If found, add to map
        listNewDisksBySize=`sortDisksbySize "restore" $f_st_new`
        if [ -z "$listNewDisksBySize" ]; then
            Log "autoMapStorage: Cannot sort $f_st_new"
            return 1
        fi

        listNewDisksByPCI=`sortDisksbyPCI "restore" $f_st_new`
        if [ -z "$listNewDisksByPCI" ]; then
          Log "autoMapStorage: Cannot sort disks by PCI addresses"
          return 1
        fi


        Execute "rm -f $STORAGE_TREE/storage_missing_after_automap"
        Execute "rm -f $f_st_missing_sortfile"
        #Sort storage missing by size descending
        cat $f_st_missing | while read missingDiskName
        do
            Log "\nautoMapStorage: Debug: Process(sort) missing disk [$missingDiskName]"
            missingDiskNode=`DiskNodeForDiskName "backup" $missingDiskName`
            if [ -z "$missingDiskNode" -o ! -d "$missingDiskNode" ]; then
                Log "autoMapStorage: Error: Cannot find disk node for [$missingDiskName]"
                return 2
            fi
            missingDiskSize=`cat $missingDiskNode/size`
            Log "\nautoMapStorage: Debug: (sort) $missingDiskSize $missingDiskNode"
            echo "$missingDiskSize $missingDiskName" >> $f_st_missing_sortfile
        done
        mv $f_st_missing $f_st_missing_b4sort
        sort -nr $f_st_missing_sortfile > /tmp/storage_tree/storage_missing_sorted.$$
        cat /tmp/storage_tree/storage_missing_sorted.$$ | awk '{print $2}' > $f_st_missing

        #if disks for /boot is missing, we will try to map it to the 'first' disk on the system.
        #this is required especially on HyerV which is capable of booting via the first disk.
        #the problem is given a mix of disk types/names, we may not know easily which is the first disk!

        bootDiskName=`findDiskbyMntPoint "backup" "/boot"`
        if [ -n "$bootDiskName" ]; then
            #Now, check if this disk is missing
            cat $f_st_missing | while read missingDiskName
            do
                if [ "$missingDiskName" == "$bootDiskName" ]; then
                    Log "\nautoMapStorage: Debug: Found boot disk [$bootDiskName] in missing list"

                    missingDiskNode=`DiskNodeForDiskName "backup" $missingDiskName`
                    if [ -z "$missingDiskNode" -o ! -d "$missingDiskNode" ]; then
                        Log "autoMapStorage: Error: Cannot find disk node for [$missingDiskName]"
                        return 2
                    fi
                    missingDiskSize=`cat $missingDiskNode/size`
                    Log "autoMapStorage: Debug: **missing boot disk detected as [$missingDiskName] with size [$missingDiskSize]"

                    #If machine is identical, Try to map with same diskname and disksize
                    firstDiskNode=`DiskNodeForDiskName "restore" $missingDiskName`
                    grep -qw $missingDiskName $f_st_used
                    ret=$?
                    if  [ ! -z "$firstDiskNode" ] && [ -d "$firstDiskNode" ] && [ $ret -ne 0 ]; then
                        Log "if size matches, consider it identical and map the disk"
                        firstDiskSize=`cat $firstDiskNode/size`
                        if [ "x$firstDiskSize" == "x$missingDiskSize" ]; then
                            Log "Disk mapped [$missingDiskName] with [$missingDiskName] on base of size."
                            echo "$missingDiskName    $missingDiskName    1" >> $STORAGE_TREE/DISK_MAP
                            moveNewtoUsed $missingDiskName $f_st_new $missingDiskName $f_st_used
                            break
                        fi
                    fi

                    #Try to find the 'first' disk attached to this machine.
                    firstDiskName=`cat $STORAGE_TREE/restore/disklist_byPCI | head -n 1 | awk '{print $3}'`
                    firstDiskSize=`cat $STORAGE_TREE/restore/disklist_byPCI | head -n 1 | awk '{print $2}'`
                    Log "autoMapStorage: Debug: **First disk detected as [$firstDiskName] with size [$firstDiskSize]"
                    if [ $missingDiskSize -le $firstDiskSize ]; then
                        Log "autoMapStorage: Debug: Found match for [$missingDiskName] [$missingDiskSize] --> [$firstDiskName] [$firstDiskSize]"
                        echo "$missingDiskName    $firstDiskName    1" >> $STORAGE_TREE/DISK_MAP
                        moveNewtoUsed $missingDiskName $f_st_new $firstDiskName $f_st_used
                    else
                        Log "autoMapStorage: Error: *** First disk's size is less than the size needed for the boot disk. should another suitable disk be found, BIOS order change will be needed to boot properly"
                    fi
                    break
                fi
            done
        fi

        #Now iterate missing disks
        cat $f_st_missing | while read missingDiskName
        do
            Log "\nautoMapStorage: Debug: Process missing disk [$missingDiskName]"

            is_already_mapped=`cat $f_diskmap | awk '{print $1}' | grep -w $missingDiskName | wc -l` 
            if [ $is_already_mapped -eq 1 ]; then
                Log "\nautoMapStorage: Debug: Skip already mapped disk [$missingDiskName]"
                continue
            fi

            missingDiskNode=`DiskNodeForDiskName "backup" $missingDiskName`
            if [ -z "$missingDiskNode" -o ! -d "$missingDiskNode" ]; then
                Log "autoMapStorage: Error: Cannot find disk node for [$missingDiskName]"
                return 2
            fi
            missingDiskSize=`cat $missingDiskNode/size`

            matchedDiskName=`findDiskbySize $missingDiskSize $STORAGE_TREE/restore/disklist_bysize $f_st_used`
            if [ -z "$matchedDiskName" ]; then
                Log "\nautoMapStorage: Error: Cannot find a match for missing disk [$missingDiskName] size[$missingDiskSize] in [$STORAGE_TREE/restore/disklist_bysize]"
                echo "$missingDiskName" >> $STORAGE_TREE/storage_missing_after_automap
            else
                matchedDiskSize=`grep -w $matchedDiskName $STORAGE_TREE/restore/disklist_bysize | awk '{print $1}'`
                Log "autoMapStorage: Debug: Found match for [$missingDiskName] [$missingDiskSize] --> [$matchedDiskName] [$matchedDiskSize]"
                echo "$missingDiskName    $matchedDiskName    1" >> $STORAGE_TREE/DISK_MAP
                moveNewtoUsed $missingDiskName $f_st_new $matchedDiskName $f_st_used
            fi
        done
        if [ -s $STORAGE_TREE/storage_missing_after_automap ]; then
            Log "autoMapStorage: Error: Missing disks after trying to autoMap = [`cat $STORAGE_TREE/storage_missing_after_automap`]"
            return 1
        fi

        #Validate each mountpoint's missing storage to the DISK_MAP
        mdir=$STORAGE_TREE/backup/mount_tree
        for mountnum in `ls $mdir`;
        do
            Log "Debug: autoMapStorage: Validate: Process mountnum=[$mdir/$mountnum]"
            DIR=$mdir/$mountnum
            if [ ! -d $DIR ]; then
                Log "Error: autoMapStorage: Validate: mount dir [$DIR] does not exist!"
                return 2
            fi
            if [ ! -f $DIR/to_restore ]; then
                Log "Debug: autoMapStorage: Validate: $DIR/to_restore does not exist. Skip it"
                Execute "rm -f $DIR/storage_missing"
                continue
            fi
            restore_flag=`cat $DIR/to_restore`
            if [ "x$restore_flag" != "x1" ]; then
                Log "Debug: autoMapStorage: Validate: $DIR/to_restore [$restore_flag] not set. Skip it"
                Execute "rm -f $DIR/storage_missing"
                continue
            fi
            Execute "rm -f $DIR/storage_missing.new"
            cat $DIR/storage_missing | while read missingDiskName
            do
                Log "Debug: autoMapStorage: Validate: Read missing disk [$missingDiskName] from [$DIR/storage_missing]"
                is_found=`cat $STORAGE_TREE/DISK_MAP | awk '{print $1}' | grep -w $missingDiskName`
                if [ -n "$is_found" ]; then
                    Log "Debug: autoMapStorage: Validate: Found [$missingDiskName] in DISK_MAP. Remove from missing list"
                else
                    Log "Debug: autoMapStorage: Validate: Did not find [$missingDiskName] in DISK_MAP. Retain in missing list"
                    echo "$missingDiskName" >> $DIR/storage_missing.new
                fi
            done
            if [ ! -s $DIR/storage_missing.new ]; then
                Log "Debug: autoMapStorage: Validate: All missing devices for mountnum=[$mountnum] mapped."
                Execute "rm -f $DIR/storage_missing.new"
                Execute "rm -f $DIR/storage_missing"
                touch $DIR/storage_present
            else
                Log "Deug: autoMapStorage: Validate: One or more devices for mountnum=[$mountnum] still missing"
                Execute "mv $DIR/storage_missing.new $DIR/storage_missing"
            fi
        done
        any_missing=`find $STORAGE_TREE/backup/mount_tree -name "storage_missing"`
        if [ -n "$any_missing" ]; then
            Log "Debug: autoMapStorage: Validate Failed! Storage still missing for one or more mountpoints selected for restore"
            return 1
        else
            Log "Debug: autoMapStorage: Validate SUCCESS!"
            return 0
        fi
    fi
}

######
#   Called for each partition if it is a PV by resizePartitionForDisk
#   Name: resizePVPartition
#   API DEF:
#       IN:             arg1=backup disknode path, arg2=restore disk node path,arg3=partition number,arg4=delta_size
#       OUT:            0=true 1=false 2=error.
#       EXIT:           no
#   LOGIC:#   
#   This function will be called for each PV partition on resizePartitionForDisk.
#   This function resizes the all LV present on this PV.
########
resizePVPartition()
{
    bkpdisknode=$1
    resdisknode=$2
    partno=$3
    part_delta_size=$4

    pvname=`basename $bkpdisknode`":$partno"

    Execute "mkdir -p $resdisknode/$partno/LV"

    total_free_pe_on_cur_pv=0

    Execute "cp -f $bkpdisknode/$partno/pv_info $resdisknode/$partno/"

    pe_size=`cat $resdisknode/$partno/pv_info | awk -F"," '{ if (NR > 1) {print $7}}'` 
    total_pe_left_on_cur_pv=$((part_delta_size / pe_size)) 
    new_total_used_pe_on_pv=$((`cat $bkpdisknode/$partno/used_size` / pe_size))
  
    Log "PV[$pvname] : Total Used PE [$new_total_used_pe_on_pv], Total PE Left[$total_pe_left_on_cur_pv]"
  
    for lv_name in `ls  $bkpdisknode/$partno/LV`
    do
        bkp_lv_dir=$bkpdisknode/$partno/LV/$lv_name
        lv_total_pe_on_cur_pv=`cat $bkp_lv_dir/total_pe_on_cur_pv`
        lv_used_pe_on_cur_pv=`cat $bkp_lv_dir/total_used_pe_on_cur_pv`
        lv_free_pe_on_cur_pv=$((lv_total_pe_on_cur_pv - lv_used_pe_on_cur_pv))
        total_free_pe_on_cur_pv=$((total_free_pe_on_cur_pv + lv_free_pe_on_cur_pv))
        Log "LV[$lv_name] : Total PE [$lv_total_pe_on_cur_pv], Used PE[$lv_used_pe_on_cur_pv] , Free PE [$lv_free_pe_on_cur_pv]"
    done
  
    Log "PV[$pvname] : Total Free PE [$total_free_pe_on_cur_pv]"

  
    for lv_name in `ls  $bkpdisknode/$partno/LV`
    do
        Log "PV[$pvname] : Total Left PE [$total_pe_left_on_cur_pv]"
        
        bkp_lv_dir=$bkpdisknode/$partno/LV/$lv_name
        res_lv_dir=$resdisknode/$partno/LV/$lv_name
        Execute "mkdir -p $res_lv_dir"
      
        lv_total_pe_on_cur_pv=`cat $bkp_lv_dir/total_pe_on_cur_pv`
        lv_used_pe_on_cur_pv=`cat $bkp_lv_dir/total_used_pe_on_cur_pv`
        lv_free_pe_on_cur_pv=$((lv_total_pe_on_cur_pv - lv_used_pe_on_cur_pv))
        lv_total_used_pe_on_cur_pv=`cat $bkp_lv_dir/total_used_pe_on_cur_pv`

        lv_delta_pe=`echo $total_pe_left_on_cur_pv $lv_free_pe_on_cur_pv $total_free_pe_on_cur_pv | awk '{ printf "%d", int($1*($2/$3))}'`
        lv_new_used_pe_on_cur_pv=$((lv_total_used_pe_on_cur_pv + lv_delta_pe))
      
        Log "LV[$lv_name] : Delta PE [$lv_delta_pe], New Used PE [$lv_new_used_pe_on_cur_pv]"

        if [ $lv_new_used_pe_on_cur_pv -gt $lv_total_pe_on_cur_pv ]
        then
            Log "LV[$lv_name] : New Used PE[$lv_new_used_pe_on_cur_pv] is greater than Total PE [ $lv_total_pe_on_cur_pv] "
            lv_new_used_pe_on_cur_pv=$lv_total_pe_on_cur_pv
            lv_delta_pe=$((lv_total_pe_on_cur_pv - lv_total_used_pe_on_cur_pv))
            Log "LV[$lv_name] : New used PE[$lv_new_used_pe_on_cur_pv, New Delta PE[$lv_delta_pe]"
        fi 

        new_total_used_pe_on_pv=$((new_total_used_pe_on_pv + lv_delta_pe))
        
        Log "PV[$pvname] : Delta PE[$lv_delta_pe], Total Used PE[$new_total_used_pe_on_pv]"
        
        echo $lv_new_used_pe_on_cur_pv > $res_lv_dir/total_used_pe_on_cur_pv
    done
  
    echo "success"
}

#Check if storage can be automatically mapped
#   Name: resizePartitionForDisk
#   API DEF:
#       IN:             arg1=src disk name, which partiton is going to be resized, arg2=dst disk which is mapped to src disk
#       OUT:            0=true 1=false 2=error.
#       EXIT:           no
#   LOGIC:
#       This function will be called for each disk which was mapped by autoScaleStorage1 function. 
#       For each disk mapping if backup disk's used size is smaller than restore disk's total size, then remaining size(restore disk's total size - backup disk's used size) shall be distributed among the all partition and new used size shall be calculated. 
#       If a partition on backup disk is PV then all LV residing on that PV should also be resized, hence resizePVPartition shall be called for each PV partition.

resizePartitionForDisk()
{
    srcDisk=$1
    dstDisk=$2

    bkpdisknode=`DiskNodeForDiskName "backup" $srcDisk`
    src_disk_used_size=`cat $bkpdisknode/used_size`
    src_disk_size=`cat $bkpdisknode/size`
 
    ##claculate the offset 
    resdisknode=`DiskNodeForDiskName "restore" $dstDisk`
    dst_disk_size=`cat $resdisknode/size`

    #keeping 1MB reserved
    total_space_left=$((dst_disk_size - src_disk_used_size - 1048576))

    new_disk_used_size=$src_disk_used_size

    Log "BkpDisk[$srcDisk] : Total Size[$src_disk_size], Used Size[$src_disk_used_size]"
    Log "ResDisk[$dstDisk] : Total Size[$dst_disk_size], Used Size[$new_disk_used_size], Space Left[$total_space_left]"
  
    if [ `cat $bkpdisknode/partinfo | wc -l` -gt 0 ]; then
        part_total_free_size=0
        for partno in `cat $bkpdisknode/partinfo | awk -F"partno=" '{ print $2}'  | awk '{print $1}'`
        do
            Execute "mkdir -p $resdisknode/$partno"
            Execute "rm -rf $resdisknode/$partno/*"
      
            part_total_size=`cat $bkpdisknode/$partno/total_size`
            part_used_size=`cat $bkpdisknode/$partno/used_size`
            part_free_size=$((part_total_size - part_used_size))
            Log "bkpDisk[$srcDisk] Part[$partno]: Total Size[$dst_disk_size], Used Size[$new_disk_used_size], Free Size[$part_free_size]"
            part_total_free_size=$((part_total_free_size + part_free_size))
        done
    
        Log "bkpDisk[$srcDisk]: Total Free Size[$part_free_size]"
        extended_part_size=0
        extended_part_no=0
        
        while read pline
        do
            parttype=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "ptyp="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
            partno=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "partno="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
            blk_sz=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "blk_sz="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
            
            part_offset=`cat $bkpdisknode/$partno/offset` 
            part_total_size=`cat $bkpdisknode/$partno/total_size`
            part_used_size=`cat $bkpdisknode/$partno/used_size`
            part_free_size=$((part_total_size - part_used_size))
            part_delta_size=`echo "$total_space_left $part_free_size $part_total_free_size $blk_sz" | awk '{ printf "%d", (int(int($1*($2/$3))/$4)*$4) }'`
            new_part_used_size=$((part_used_size + part_delta_size))

            Log "part[$partno]: Used Size[$new_part_used_size], Delta Size[$part_delta_size]"
  
            if [ $new_part_used_size -gt $part_total_size ]; then
                Log "part[$partno]: Used Size[$new_part_used_size] is greater than Total Size[$part_total_size]"
                new_part_used_size=$part_total_size
                part_delta_size=$((part_total_size - part_used_size)) 
                Log "part[$partno]: New Used Size[$new_part_used_size], New Delta Size[$part_delta_size]"
            fi

            if [ -f $bkpdisknode/$partno/is_pv ]; then
                Log "part[$partno]: This is a PV, Resizing PV with Delta Size[$part_delta_size]"
                ret=`ExecFn "resizePVPartition $bkpdisknode $resdisknode $partno $part_delta_size"`
                if [ -z "$ret" ]; then
                    echo ""
                    return 1
                fi
            fi 

            if [ "$parttype" == "extended" ]; then
                extended_part_no=$partno
                extended_part_size=0
            fi

            if [ "$parttype" == "logical" ]; then
                Log "part[$partno] : Is a logical partition, adding it size to extended partition"
                extended_part_size=$((extended_part_size + part_offset + new_part_used_size))
                Log "resDisk[$dstDisk] : extended_part[$extended_part_no] new size [$extended_part_size]"
                echo $extended_part_size > $resdisknode/$extended_part_no/extended_part_size   
            fi

            echo $new_part_used_size > $resdisknode/$partno/used_size 
            echo $part_offset > $resdisknode/$partno/offset
 
            new_disk_used_size=$((new_disk_used_size + part_delta_size))
            Log "resDisk[$dstDisk]: Total Used Size[$new_disk_used_size]"

        done < $bkpdisknode/partinfo

        offset_of_last_part=0
        offset_of_last_logical_part=0        
        cat $bkpdisknode/partinfo | while read pline
        do
            name=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "name="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
            parttype=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "ptyp="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
            partno=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "partno="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
            blk_sz=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "blk_sz="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`

            disk_part_size=`cat $resdisknode/$partno/used_size`
            offset=`cat $resdisknode/$partno/offset`

            if [ "$parttype" == "logical" ]; then
                offset_of_last_logical_part=$(( offset_of_last_logical_part + offset))  
                startblk=$((offset_of_last_logical_part / blk_sz))
                offset_of_last_logical_part=$((offset_of_last_logical_part + disk_part_size))
            else
                offset_of_last_part=$((offset_of_last_part + offset))
                
                if [ "$parttype" == "extended" ]; then
                    offset_of_last_logical_part=$offset_of_last_part
                    disk_part_size=`cat $resdisknode/$partno/extended_part_size`
                fi
                startblk=$((offset_of_last_part / blk_sz))
                offset_of_last_part=$((offset_of_last_part + disk_part_size))
            fi

            sizeblk=$((disk_part_size/blk_sz))

            fstype=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "fstype="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
            flags_all=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "flags="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
            echo "name=$name ptyp=$parttype partno=$partno start=$startblk blk_sz=$blk_sz size_in_blks=$sizeblk fstype=$fstype flags=$flags_all"  >> $resdisknode/partinfo

            startblk=$((startblk+sizeblk))
        done
    else  
        part_delta_size=$total_space_left
        new_part_used_size=$dst_disk_size
        new_disk_used_size=$new_part_used_size
        partno=0

        Log "bkpDisk[$srcDisk]: This is a whole disk"
        

        Execute "mkdir -p $resdisknode/$partno/"
        rm -rf $resdisknode/$partno/*
        
        if [ -f $bkpdisknode/$partno/is_pv ]; then
            Log "bkpDisk[$srcDisk]: This is a PV, Resizing PV DeltaSize [$part_delta_size]"
            ret=`ExecFn "resizePVPartition $bkpdisknode $resdisknode $partno $part_delta_size"`
            if [ -z "$ret" ]; then
                echo ""
                return 1
            fi
        fi
        
        Log "dstDisk[$dstDisk][0]: Total Used Size[$new_part_used_size]"
        Log "dstDisk[$dstDisk]: Total Used Size[$new_disk_used_size]"
        echo $new_part_used_size >  $resdisknode/$partno/used_size
    fi

    echo "$new_disk_used_size" > $resdisknode/used_size
    echo "1" > $resdisknode/resized
    echo "done"
    return 0
}

#Check if storage can be automatically mapped
#   Name: autoScaleStorage1
#   API DEF:
#       OUT:            0=true 1=false 2=error.
#       EXIT:           no
#   LOGIC:
#       If there is still a src disk in $STORAGE_TREE/storage_missing_all which could not be mapped during even after checkIdenticalStorage, autoMapStorage,
#        This function will try to map that disk with remaining dst disk in $STORAGE_TREE/storage_new if src disk's used_size is less or equal to that dst disk total size. 
#        If no unmapped disk is left in $STORAGE_TREE/storage_missing_all, function will return 0 otherwise 1.
#       For each disk mapped by autoScaleStoare, it will resize the original partitition present on src disk.

autoScaleStorage1()
{
    mdir=$STORAGE_TREE/backup/mount_tree
    f_st_missing=$STORAGE_TREE/storage_missing_all
    f_st_missing_list_b4autoScale1=$STORAGE_TREE/storage_missing_all_b4autoScale1
    f_st_missing_b4sort=$STORAGE_TREE/storage_missing_all_b4sort
    f_st_missing_sortfile=$STORAGE_TREE/storage_missing_all_sortfile
    f_st_used=$STORAGE_TREE/storage_used
    f_st_new=$STORAGE_TREE/storage_new
    f_diskmap=$STORAGE_TREE/DISK_MAP


    Log "Debug: autoScaleStorage1"

    Execute "rm -f $f_st_missing_b4sort"

    if [ ! -s $f_st_missing ]; then
        Log "autoScaleStorage1: Debug: No missing disks in [$f_st_missing]. return success"
        return 0
    else
        #Prepare new disks [Set of current disks - set of required and present disks, previously mapped]
        listNewDisks $STORAGE_TREE/restore/disklist $f_st_used $f_st_new
        ret=$?
        if [ $ret != "0" ]; then
            Log "autoScaleStorage1: Error: Failed to list new disks"
            return 1
        fi

        #There are some missing disks. Check if there are any new disks...
        if [ ! -s $f_st_new ]; then
            Log "autoScaleStorage1: Error: No new disks in [$f_st_new]."
            return 1
        fi

        #For each disk in storage_missing_all, find the smallest disk in storage_new whose size >= the missing disk. If found, add to map
        listNewDisksBySize=`sortDisksbySize "restore" $f_st_new`
        if [ -z "$listNewDisksBySize" ]; then
            Log "autoScaleStorage1: Cannot sort $f_st_new"
            return 1
        fi

        listNewDisksByPCI=`sortDisksbyPCI "restore" $f_st_new`
        if [ -z "$listNewDisksByPCI" ]; then
            Log "autoScaleStorage1: Cannot sort disks by PCI addresses"
            return 1
        fi

        Execute "rm -f $STORAGE_TREE/storage_missing_after_automap"
        Execute "rm -f $f_st_missing_sortfile"
        Execute "rm -f $f_st_missing_list_b4autoScale1"
        Execute "cp -f $f_st_missing $f_st_missing_list_b4autoScale1"

        #Sort storage missing by size descending

        cat $f_st_missing | while read missingDiskName
        do
            Log "\nautoScaleStorage1: Debug: Process(sort) missing disk [$missingDiskName]"

            #filtering out already mapped disks from missing list
            is_already_mapped=`cat $f_diskmap | grep ^$missingDiskName | wc -l`
            if [ $is_already_mapped -eq 1 ]; then
                Log "\nautoScaleStorage1: Debug: (sort) missing disk [$missingDiskName] is already mapped by autoMapStorage, removing it from new missing list"
                continue
            fi 

            missingDiskNode=`DiskNodeForDiskName "backup" $missingDiskName`
            if [ -z "$missingDiskNode" -o ! -d "$missingDiskNode" ]; then
                Log "autoScaleStorage1: Error: Cannot find disk node for [$missingDiskName]"
                return 2
            fi
            missingDiskUsedSize=`cat $missingDiskNode/used_size`
            Log "\nautoScaleStorage1: Debug: (sort) $missingDiskUsedSize $missingDiskNode"
            echo "$missingDiskUsedSize $missingDiskName" >> $f_st_missing_sortfile
        done

        mv $f_st_missing $f_st_missing_b4sort
        sort -nr $f_st_missing_sortfile > /tmp/storage_tree/storage_missing_sorted.$$
        cat /tmp/storage_tree/storage_missing_sorted.$$ | awk '{print $2}' > $f_st_missing

        #if disks for /boot is missing, we will try to map it to the 'first' disk on the system.
        #this is required especially on HyerV which is capable of booting via the first disk.
        #the problem is given a mix of disk types/names, we may not know easily which is the first disk!

        bootDiskName=`findDiskbyMntPoint "backup" "/boot"`

        if [ -n "$bootDiskName" ]; then
            #Now, check if this disk is missing
            cat $f_st_missing | while read missingDiskName
            do
                if [ "$missingDiskName" == "$bootDiskName" ]; then
                    Log "\nautoScaleStorage1: Debug: Found boot disk [$bootDiskName] in missing list"

                    missingDiskNode=`DiskNodeForDiskName "backup" $missingDiskName`
                    if [ -z "$missingDiskNode" -o ! -d "$missingDiskNode" ]; then
                        Log "autoScaleStorage1: Error: Cannot find disk node for [$missingDiskName]"
                        return 2
                    fi
                    missingDiskUsedSize=`cat $missingDiskNode/used_size`
                    Log "autoScaleStorage1: Debug: **missing boot disk detected as [$missingDiskName] with size [$missingDiskUsedSize]"

                    #If machine is identical, Try to map with same diskname and disksize
                    firstDiskNode=`DiskNodeForDiskName "restore" $missingDiskName`
                    grep -qw $missingDiskName $f_st_used
                    ret=$?
                    if  [ ! -z "$firstDiskNode" ] && [ -d "$firstDiskNode" ] && [ $ret -ne 0 ]; then

                        Log "if size matches, consider it identical and map the disk"
                        firstDiskSize=`cat $firstDiskNode/size`
                        if [ "x$firstDiskSize" == "x$missingDiskUsedSize" ]; then
                            Log "Disk mapped [$missingDiskName] with [$missingDiskName] on base of size."
                            echo "$missingDiskName    $missingDiskName    1" >> $STORAGE_TREE/DISK_MAP
                            moveNewtoUsed $missingDiskName $f_st_new $missingDiskName $f_st_used

                            #Here need to resize the partititon here
                            is_done=`ExecFn "resizePartitionForDisk $missingDiskName  $missingDiskName"`

                            if [ -z "$is_done" -o "$is_done"  != "done" ]; then
                                Log "Failed to resize partition for disk=$missingDiskName"
                                return 1
                            fi

                            break
                        fi
                    fi

                    #Try to find the 'first' disk attached to this machine.
                    firstDiskName=`cat $STORAGE_TREE/restore/disklist_byPCI | head -n 1 | awk '{print $3}'`
                    firstDiskSize=`cat $STORAGE_TREE/restore/disklist_byPCI | head -n 1 | awk '{print $2}'`
                    Log "autoScaleStorage1: Debug: **First disk detected as [$firstDiskName] with size [$firstDiskSize]"

                    if [ $missingDiskUsedSize -le $firstDiskSize ]; then

                        Log "autoScaleStorage1: Debug: Found match for [$missingDiskName] [$missingDiskUsedSize] --> [$firstDiskName] [$firstDiskSize]"
                        echo "$missingDiskName    $firstDiskName    1" >> $STORAGE_TREE/DISK_MAP
                        moveNewtoUsed $missingDiskName $f_st_new $firstDiskName $f_st_used

                        #Here need to resize the partititon here
                        is_done=`ExecFn "resizePartitionForDisk $missingDiskName  $firstDiskName"`

                        if [ -z "$is_done" -o "$is_done"  != "done" ]; then
                            Log "Failed to resize partition for disk=$missingDiskName"
                            return 1
                        fi

                    else
                        Log "autoScaleStorage1: Error: *** First disk's size is less than the size needed for the boot disk. should another suitable disk be found, BIOS order change will be needed to boot properly"
                    fi
                    break
                fi
            done
        fi

        #Now iterate missing disks
        cat $f_st_missing | while read missingDiskName
        do
            Log "\nautoScaleStorage1: Debug: Process missing disk [$missingDiskName]"

            is_already_mapped=`cat $f_diskmap | awk '{print $1}' | grep -w $missingDiskName | wc -l` 
            if [ $is_already_mapped -eq 1 ]; then
                Log "\nautoMapStorage: Debug: Skip already mapped disk [$missingDiskName]"
                continue
            fi

            missingDiskNode=`DiskNodeForDiskName "backup" $missingDiskName`
            if [ -z "$missingDiskNode" -o ! -d "$missingDiskNode" ]; then
                Log "autoScaleStorage1: Error: Cannot find disk node for [$missingDiskName]"
                return 2
            fi
            missingDiskUsedSize=`cat $missingDiskNode/used_size`

            matchedDiskName=`findDiskbySize $missingDiskUsedSize $STORAGE_TREE/restore/disklist_bysize $f_st_used`
            if [ -z "$matchedDiskName" ]; then
                Log "\nautoScaleStorage1: Error: Cannot find a match for missing disk [$missingDiskName] used size[$missingDiskUsedSize] in [$STORAGE_TREE/restore/disklist_bysize]"
          
                echo "$missingDiskName" >> $STORAGE_TREE/storage_missing_after_automap
            else
                matchedDiskSize=`grep -w $matchedDiskName $STORAGE_TREE/restore/disklist_bysize | awk '{print $1}'`
                Log "autoScaleStorage1: Debug: Found match for disk [$missingDiskName] used size[$missingDiskUsedSize] --> [$matchedDiskName] [$matchedDiskSize]"
                echo "$missingDiskName    $matchedDiskName    1" >> $STORAGE_TREE/DISK_MAP
                moveNewtoUsed $missingDiskName $f_st_new $matchedDiskName $f_st_used

                #Here need to resize the partititon here
                is_done=`ExecFn "resizePartitionForDisk $missingDiskName  $matchedDiskName"`

                if [ -z "$is_done" -o "$is_done"  != "done" ]; then
                    Log "Failed to resize partition for disk=$missingDiskName"
                    return 1
                fi
            fi
        done

        if [ -s $STORAGE_TREE/storage_missing_after_automap ]; then
            Log "autoScaleStorage1: Error: Missing disks after trying to autoMap = [`cat $STORAGE_TREE/storage_missing_after_automap`]"
            return 1
        fi

        #Validate each mountpoint's missing storage to the DISK_MAP
        mdir=$STORAGE_TREE/backup/mount_tree
        for mountnum in `ls $mdir`;
        do
            Log "Debug: autoScaleStorage1: Validate: Process mountnum=[$mdir/$mountnum]"
            DIR=$mdir/$mountnum
            if [ ! -d $DIR ]; then
                Log "Error: autoScaleStorage1: Validate: mount dir [$DIR] does not exist!"
                return 2
            fi

            if [ ! -f $DIR/to_restore ]; then
                Log "Debug: autoScaleStorage1: Validate: $DIR/to_restore does not exist. Skip it"
                Execute "rm -f $DIR/storage_missing"
                continue
            fi

            restore_flag=`cat $DIR/to_restore`
            if [ "x$restore_flag" != "x1" ]; then
                Log "Debug: autoScaleStorage1: Validate: $DIR/to_restore [$restore_flag] not set. Skip it"
                Execute "rm -f $DIR/storage_missing"
                continue
            fi

            Execute "rm -f $DIR/storage_missing.new"

            cat $DIR/storage_missing | while read missingDiskName
            do
                Log "Debug: autoScaleStorage1: Validate: Read missing disk [$missingDiskName] from [$DIR/storage_missing]"
                is_found=`cat $STORAGE_TREE/DISK_MAP | awk '{print $1}' | grep -w $missingDiskName`
                if [ -n "$is_found" ]; then
                    Log "Debug: autoScaleStorage1: Validate: Found [$missingDiskName] in DISK_MAP. Remove from missing list"
                else
                    Log "Debug: autoScaleStorage1: Validate: Did not find [$missingDiskName] in DISK_MAP. Retain in missing list"
                    echo "$missingDiskName" >> $DIR/storage_missing.new
                fi
            done

            if [ ! -s $DIR/storage_missing.new ]; then
                Log "Debug: autoScaleStorage1: Validate: All missing devices for mountnum=[$mountnum] mapped."
                Execute "rm -f $DIR/storage_missing.new"
                Execute "rm -f $DIR/storage_missing"
                touch $DIR/storage_present
            else
                Log "Deug: autoScaleStorage1: Validate: One or more devices for mountnum=[$mountnum] still missing"
                Execute "mv $DIR/storage_missing.new $DIR/storage_missing"
            fi

        done
        any_missing=`find $STORAGE_TREE/backup/mount_tree -name "storage_missing"`
        if [ -n "$any_missing" ]; then
            Log "Debug: autoScaleStorage1: Validate Failed! Storage still missing for one or more mountpoints selected for restore"
            return 1
        fi

        Log "Debug: autoScaleStorage1: Validate SUCCESS!"
        return 0
    fi
}

#Returns the mount point directory for a partition
#   Name: FindMntPointDirForDevId
#   API DEF:
#       IN :            arg1=dev_id made up of maj and minor number
#       OUT:            0=true 1=false
#       EXIT:           no
#   LOGIC:

FindMntPointDirForDevId()
{
    dev_id=$1
    for f_dev_id in `ls $STORAGE_TREE/backup/mount_tree/*/dev_id`
    do
        dev_id2=`cat $f_dev_id`
        if [ "$dev_id2" == "$dev_id" ]; then
            mntdir=`dirname $f_dev_id`
            Log "Found mountdir[$mntdir] for dev id[$dev_id]"
            echo $mntdir
            return 0
        fi
    done

    Log "Could not find mountdir for dev id[$dev_id]"
    echo ""
    return 1
}

#Create a sorted list of partitions with flag indicated system/non-system and other associated details
#   Name: CreateSortInfo
#   API DEF:
#       IN :            arg1=output file name.
#       OUT:            0=true 1=false 2=error.
#       EXIT:           no
#   LOGIC:
#       This function will be called by autoScaleStorage2.
#       This function will iterate over each disk present in $STORAGE_TREE/backup/disks and generate a generate a sort info list
#       That list will be used to sort and segregate sys/non system partititon  and basic partititon or physcial volume.
#       Each line in the list would be as following.  
#         <partition or disk type DISK or LUKS or LVM>#< recreate_info for non pv partition or disk >#<used size of volume | used size of non pv partition>#<0-If this is a non pv partition, 1- if this is a volume group>#<0-if entry belongs to non system partition, 1-if entry belongs to system partition>#<1-if this is boot partition, otherwise 0>

CreateSortInfo()
{
    mdir=$STORAGE_TREE/backup/mount_tree/
    sbkpdir=$STORAGE_TREE/backup
    sresdir=$STORAGE_TREE/restore
    auto_scale_dir=$STORAGE_TREE/autoScale2/

    disk_meta_dir=$auto_scale_dir/DISK
    md_meta_dir=$auto_scale_dir/MD
    lvm_meta_dir=$auto_scale_dir/LVM
    luk_meta_dir=$auto_scale_dir/LUKS

    Execute "mkdir -p $auto_scale_dir"
    Execute "rm -rf $auto_scale_dir/*"
    
    for disktype in `ls $sbkpdir/disks`
    do
        disks_type_dir="$sbkpdir/disks/$disktype"

        for disk in `ls $disks_type_dir`
        do
            disk_dir="$disks_type_dir/$disk"
            dev_path=`cat $disk_dir/dev_path`

            Log "Enumerating Disk [$disk]"

            for part_dir in `ls -d $disk_dir/[0-9]*`
            do
                is_sys=0
                is_boot=0
                part=`basename $part_dir`
                
                Log "Disk[$disk] : Found Partition[$part]"
                
                if [ -f $part_dir/is_extended ]; then
                    Log "Disk[$disk] Part[$part] : Is an exteneded partition hence skipping it"
                    continue
                fi

                if [ -f $part_dir/is_boot ]; then
                    Log "Disk[$disk] Part[$part] : Is a boot partition"
                    is_boot=1
                fi

                if [ -f $part_dir/is_disk ]; then
                    
                    part_meta_dir=$disk_meta_dir/$disk/$part
                    part_used_size=`cat $part_dir/used_size`
                    device="$dev_path:$part"
                    dev_id=`cat $part_dir/dev_id`
                    priority=0

                    Log "Disk[$disk] Part[$part] : Is a Disk, Device[$device], Used Size[$part_used_size]"
                    Execute "mkdir -p $part_meta_dir/"
                    
                    mntptdir=`ExecFn "FindMntPointDirForDevId $dev_id"`
                    if [ -z "$mntptdir" ]; then
                        Log "Disk[$disk] Part[$part] : Could not get mount dir for device[$device] in mount_tree, hence skipping this device"
                        continue
                    fi

                    mntpoint=`cat $mntptdir/mntpoint`
                    Log "Disk[$disk] Part[$part] : Device[$device] is mounted at mounpoint[$mntpoint]"
                    
                    if [ ! -f $mntptdir/to_restore ]; then
                        Log "Disk[$disk] Part[$part] : Mount point [$mntpoint] is not selected for restore, hence skipping this device"
                        continue
                    fi

                    is_sys=`ExecFn "is_system_mntpt $mntpoint"`

                    echo $is_boot > $part_meta_dir/is_boot
                    echo $is_sys > $part_meta_dir/is_sys
                    echo $part_used_size > $part_meta_dir/used_size
                    
                    Log "Disk[$disk] Part[$part] : SortInfo[DISK#$device#$part_used_size#$priority#$is_sys#$is_boot] created"
                    echo "DISK#$device#$part_used_size#$priority#$is_sys#$is_boot" > $part_meta_dir/sort_info 
                elif [ -f $part_dir/is_luks ]; then
                    part_meta_dir=$luk_meta_dir/$disk/$part
                    part_used_size=`cat $part_dir/used_size`
                    dev_id=`cat $part_dir/dev_id`
                    device="$dev_path:$part"
                    priority=0

                    Log "Disk[$disk] Part[$part] : Is a LUK, Device[$device], Used Size[$part_used_size]"
                    Execute "mkdir -p $part_meta_dir/"
                    
                    mntptdir=`ExecFn "FindMntPointDirForDevId $dev_id"`
                    if [ -z "$mntptdir" ]; then
                        Log "Disk[$disk] Part[$part] : Could not get mount dir for Luk device[$device] in mount_tree, hence skipping this device"
                        continue
                    fi

                    mntpoint=`cat $mntptdir/mntpoint`
                    cryptdevname=`cat $mntptdir/recreation_info | grep 'LUKS##' | awk '{ print $2 }' | cut -d"=" -f2`
                    
                    Log "Disk[$disk] Part[$part] : Device[$device], cryptDevice[$device], is mounted at mounpoint[$mntpoint]"
                    
                    if [ ! -f $mntptdir/to_restore ]; then
                        Log "Disk[$disk] Part[$part] : Mount point [$mntpoint] is not selected for restore, hence skipping this device"
                        continue
                    fi

                    is_sys=`ExecFn "is_system_mntpt $mntpoint"`

                    echo $is_boot > $part_meta_dir/is_boot
                    echo $is_sys > $part_meta_dir/is_sys
                    echo $part_used_size > $part_meta_dir/used_size
                    
                    Log "Disk[$disk] Part[$part] : SortInfo[LUKS#$cryptdevname:$device#$part_used_size#$priority#$is_sys#$is_boot] created"
                    echo "LUKS#$cryptdevname:$device#$part_used_size#$priority#$is_sys#$is_boot" > $part_meta_dir/sort_info 
                elif [ -f $part_dir/is_pv ]; then
                    part_used_size=`cat $part_dir/used_size`
                    device="$dev_path:$part"
                    priority=1
                 
                    Log "Disk[$disk] Part[$part] : Is a Physical volume, Device[$device], Used Size[$part_used_size]"
                    Execute "mkdir -p $lvm_meta_dir/"
                    
                    vg_name=`cat $part_dir/pv_info | tail -n+2 | awk -F"," '{ print $NF }'`
                    pe_size=`cat $part_dir/pv_info | tail -n+2 | awk -F"," '{ print $7 }'`
                       
                    if [ -z "$vg_name" ]; then
                        Log "Disk[$disk] Part[$part] : Failed to parse VG name for this PV, skipping this"
                        continue
                    fi
 
                    Log "Disk[$disk] Part[$part] : VG[$vg_name], PESIZE[$pe_size]"
                    
                    vg_meta_dir="$lvm_meta_dir/$vg_name"
                    Execute "mkdir -p $vg_meta_dir/"
    
                    if [ -f $vg_meta_dir/is_sys ]; then
                        is_sys=`cat $vg_meta_dir/is_sys`
                    fi    
                     
                    vg_total_size=0
                    if [ -f $vg_meta_dir/used_size ]; then
                        vg_total_size=`cat $vg_meta_dir/used_size`
                    fi

                    Log "Disk[$disk] Part[$part] : VG[$vg_name], PESIZE[$pe_size], VGTotalSize[$vg_total_size]"
                    
                    for lv_name in `ls $part_dir/LV`
                    do
                        Log "Disk[$disk] Part[$part] : VG[$vg_name], Found LV[$lv_name]"

                        lv_dir=$part_dir/LV/$lv_name
                        lv_total_used_pe_on_cur_pv=`cat $lv_dir/total_used_pe_on_cur_pv`
                        dev_id=`cat $lv_dir/dev_id`

                        if [ -f $lv_dir/is_block ]; then
                            Log "Disk[$disk] Part[$part] : This LV wa not in use on src machine hence skipping it"
                            continue
                        fi

                        if [ $lv_total_used_pe_on_cur_pv -eq 0 ]; then
                            Log "Disk[$disk] Part[$part] : This LV does not does not have any used PE on this PV hence skipping this LV"
                            continue
                        fi
                     
                        mntptdir=`ExecFn "FindMntPointDirForDevId $dev_id"`
                        if [ -z "$mntptdir" ]; then
                            Log "Disk[$disk] Part[$part] : Could not get mount point for this LV[$lv_name] in mount_tree, hence skipping this LV"
                            continue
                        fi

                        mntpoint=`cat $mntptdir/mntpoint`
                        Log "Disk[$disk] Part[$part] : Device[${vg_name}-${lv_name}] is mounted at mounpoint[$mntpoint]"
                        
                        if [ ! -f $mntptdir/to_restore ]; then
                            Log "Disk[$disk] Part[$part] : Mountpoint[$mntpoint] is not selected for restore, hence skipping this LV"
                            continue
                        fi

                        lv_total_size_on_cur_pv=`echo $lv_total_used_pe_on_cur_pv $pe_size | awk '{ printf "%d", int($1*$2)}'`

                        Log "Disk[$disk] Part[$part] : VG[$vg_name] LV[$lv_name] has size[$lv_total_size_on_cur_pv] on this PV"

                        if [ $is_sys -eq 0 ]; then
                            is_sys=`ExecFn "is_system_mntpt $mntpoint"`
                        fi

                        lv_total_used_size=0
                        lv_meta_dir=$vg_meta_dir/$lv_name
                        Execute "mkdir -p $lv_meta_dir/"
                        
                        if [ -f $lv_meta_dir/used_size ]; then
                            lv_total_used_size=`cat $lv_meta_dir/used_size`
                        fi

                        lv_total_used_size=$((lv_total_used_size + lv_total_size_on_cur_pv))
                        vg_total_size=$((vg_total_size + lv_total_size_on_cur_pv))

                        echo $lv_total_used_size > $lv_meta_dir/used_size

                        Log "Disk[$disk] Part[$part] : LV[$lv_name], LV Total Size[$lv_total_used_size], VG[$vg_name], VG Total Size[$vg_total_size]"
                    done

                    Log "Disk[$disk] Part[$part] : VG[$vg_name], VG Total Size[$vg_total_size]"

                    if [ $vg_total_size -eq 0 ]; then
                        Log "Disk[$disk] Part[$part] : VG[$vg_name], Total size is zero, hence removing VG meta dir"
                        Execute "rm -rf $vg_meta_dir/"
                        continue
                    fi

                    echo $is_sys > $vg_meta_dir/is_sys
                    echo $is_boot > $vg_meta_dir/is_boot
                    echo $vg_total_size > $vg_meta_dir/used_size
                    echo $pe_size > $vg_meta_dir/pe_size

                    Log "VG[$vg_name] : Last Sort Info [LVM#$vg_name#$vg_total_size#$priority#$is_sys#$is_boot]"
                    echo "LVM#$vg_name#$vg_total_size#$priority#$is_sys#$is_boot" > $vg_meta_dir/sort_info 
                elif [ -f $part_dir/is_md ]; then
                    #to be implemented later for MD.
                    Log "Disk[$disk] Part[$part] : Is part of a MD device, MD is not supported yet under autoScaleStorage2, hence skipping it"
                else
                    Log "Disk[$disk] Part[$part] : Is a Block device, Block device is not supported yet under autoScale2, hence skipping it"
                fi               
            done 
        done
    done
 
    find $auto_scale_dir/ -name 'sort_info' | xargs cat > $auto_scale_dir/sort_info_all 
    cat $auto_scale_dir/sort_info_all > $auto_scale_dir/sort_info_all_b4sort
    cat $auto_scale_dir/sort_info_all_b4sort | sort -s -n -t "#" -k3 > $auto_scale_dir/sort_info_all_afsize_sort
    cat $auto_scale_dir/sort_info_all_afsize_sort | sort -s -n -t "#" -k4 > $auto_scale_dir/sort_info_all_aftype_sort
    cat $auto_scale_dir/sort_info_all_aftype_sort | sort -s -r -n -t "#" -k5 > $auto_scale_dir/sort_info_all_afsys_sort
    cat $auto_scale_dir/sort_info_all_afsys_sort | sort -s -r -n -t "#" -k6 > $auto_scale_dir/sort_info_all
} 

#
#   Name: GeneratePartInfo
#   API DEF:
#       IN :            arg1=partition type, arg2=new partition number on destination, arg3=offset of a partition on destination disk, arg4=size of the parttion,
#                       arg5=flag for partition.
#       OUT:            0=true 1=false 2=error.
#       EXIT:           no
#   LOGIC:
#       This function will be called by autoScaleStorage2.
#       This function will be generate a new part info for a given partition on destination.

GeneratePartInfo()
{
    ptype=$1
    new_part_no=$2
    new_start_size=$3
    new_size=$4
    flags=$5
    
    new_blk_size=512
    bstart=0
    bsize=0
    fs_type=""

    bstart=$((new_start_size / 512))
    bsize=$((new_size / 512))
    echo "name= ptyp=$ptype partno=$new_part_no start=$bstart blk_sz=512 size_in_blks=$bsize fstype=$fs_type flags=$flags"
}


#Check if storage can be automatically mapped
#   Assumption: checkIdenticalStorage must be called prior to making this call
#   Name: autoScaleStorage2
#   API DEF:
#       IN:             arg1=mount_tree with to_restore set to 1 for each mount_point required to be restored
#       OUT:            0=true 1=false 2=error.
#       EXIT:           no
#   LOGIC:
#       If there still exist a disk $STORAGE_TREE/storage_missing yet to mapped even after checkIdenticalStorage, autoMapStorage, autoScaleStorage1.
#       This function will retry to map all disks in STORAGE_TREE/storage_missing based on approach 2.
autoScaleStorage2()
{
    mdir=$STORAGE_TREE/backup/mount_tree
    f_st_used=$STORAGE_TREE/storage_used_do_not_use
    f_st_new=$STORAGE_TREE/storage_new
    d_st_autoScale2=$STORAGE_TREE/autoScale2
    f_st_sort_info_all=$d_st_autoScale2/sort_info_all
    f_st_part_map=$STORAGE_TREE/autoScale2/PART_MAP

    Log "Debug : autoScaleStorage2 Entering"

    Execute "rm -f $STORAGE_TREE/DISK_MAP"
    Execute "rm -f $STORAGE_TREE/recreate_info_all_*"
    Execute "rm -f $STORAGE_TREE/recreate_info_all"
    Execute "rm -f $f_st_part_map"

    for disk in `cat $STORAGE_TREE/restore/disklist`
    do
        DISKDIR=`DiskNodeForDiskName "restore" $disk`
        Log "Cleaning Disk[$disk]"
        Execute "rm -f $DISKDIR/resized"
        Execute "rm -f $DISKDIR/partinfo"
        Execute "rm -f $DISKDIR/used_size"
        Execute "rm -f $DISKDIR/last_part"
        Execute "rm -f $DISKDIR/ptype"
        Execute "rm -f $DISKDIR/has_extended"
        Execute "rm -rf $DISKDIR/[0-9]*"
    done

    ExecFn "CreateSortInfo"
    ret=$?
    if [ $ret -ne 0 ]; then
        Log "autoScaleStorage2: CreateSortInfo failed!"
        return 1
    fi
    
    #Prepare new disks [Set of current disks - set of required and present disks, previously mapped]
    listNewDisks $STORAGE_TREE/restore/disklist $f_st_used $f_st_new
    ret=$?
    if [ $ret != "0" ]; then
        Log "autoScaleStorage2: Error: Failed to list new disks"
        return 1
    fi
    #There are some missing disks. Check if there are any new disks...
    if [ ! -s $f_st_new ]; then
        Log "autoScaleStorage2: Error: No new disks in [$f_st_new]."
        return 1
    fi

    #For each disk in storage_missing_all, find the smallest disk in storage_new whose size >= the missing disk. If found, add to map
    listNewDisksBySize=`sortDisksbySize "restore" $f_st_new`
    if [ -z "$listNewDisksBySize" ]; then
        Log "autoScaleStorage2: Cannot sort $f_st_new"
        return 1
    fi

    listNewDisksByPCI=`sortDisksbyPCI "restore" $f_st_new`
    if [ -z "$listNewDisksByPCI" ]; then
        Log "autoScaleStorage2: Cannot sort disks by PCI addresses"
        return 1
    fi

    for line in `cat $f_st_sort_info_all`
    do
        pvlist=""
        vg_name=""
        device=""
        cryptdevname=""
        is_mapped=0
        
        is_vg=`echo $line | grep ^LVM | wc -l`
        is_disk=`echo $line | grep ^DISK | wc -l`
        is_md=`echo $line | grep ^MD | wc -l`
        is_luks=`echo $line | grep ^LUKS | wc -l`
        is_boot=`echo $line | cut -d"#" -f6`
        flags=""

        Log "Processing $line"

        if [ $is_vg -eq 1 ]; then
            vg_name=`echo $line | cut -d "#" -f2`
            pe_size=`cat $d_st_autoScale2/LVM/$vg_name/pe_size` 
        else
            device=`echo $line | cut -d "#" -f2`
        fi

        size_required=`echo $line | cut -d "#" -f3`

        #TODO fix this boot size issue
        if [ $is_boot -eq 1 -a $size_required -lt $MIN_SIZE_BOOT_PART ]
        then
            size_required=$MIN_SIZE_BOOT_PART
        fi
        
        for disk in `cat $STORAGE_TREE/restore/disklist_byPCI | awk '{print $3}'`
        do
            disk_node=`DiskNodeForDiskName "restore" $disk`
            disk_total_size=`cat $disk_node/size`
            disk_used_size=1048576    #Offset of first partition
            disk_free_size=0          
            size_on_cur_disk=0
            disk_last_part=0
            ptype="primary"
            
            if [ -f  $disk_node/last_part ]; then
                disk_last_part=`cat $disk_node/last_part`
            fi

            if [ $is_boot -eq 1 ]; then
                flags="boot;"
            fi

            if [ -f  $disk_node/used_size ]; then
                disk_used_size=`cat $disk_node/used_size`
            fi

            disk_available_size=$((disk_total_size - 1048576)) #Reducing 1 MB from total disk space so fix the disk alignment
            disk_free_size=$((disk_available_size - disk_used_size)) 
            
            Log "autoScaleStorage2: Disk[$disk], free_size[$disk_free_size], used_size[$disk_used_size], available_size[$disk_available_size], total_size[$disk_total_size], disk_last_part[$disk_last_part]"

            if [ ! -f $disk_node/has_extended ]; then
                if [ $disk_last_part -eq 3 ]; then
                    Log "autoScaleStorage2: Disk[$disk], no of partition is already 3, now creating extended partition"
                    ptype="extended"
                    disk_last_part=$((disk_last_part + 1))
                    size_on_cur_disk=$disk_free_size 
                    part_info=`GeneratePartInfo $ptype $disk_last_part $disk_used_size $size_on_cur_disk ""`
 
                    echo $disk_used_size > $disk_node/used_size
                    echo $disk_last_part > $disk_node/last_part
                    echo "$part_info" >> $disk_node/partinfo
                    echo "msdos" > $disk_node/ptype
                    touch $disk_node/has_extended
                fi
            fi

            if [ -f $disk_node/has_extended ]; then
                ptype="logical"
                disk_used_size=$((disk_used_size + 1048576))  #Each subsequent logical partition would be at least 1MB farther from previously existed partitions.
                disk_free_size=$((disk_available_size - disk_used_size))

                Log "autoScaleStorage2: Primary partition Disk[$disk], free_size[$disk_free_size], used_size[$disk_used_size], available_size[$disk_available_size], total_size[$disk_total_size], disk_last_part[$disk_last_part]"
            fi

            disk_last_part=$((disk_last_part + 1))

            if [ $is_vg -eq 1 ]; then
                no_of_pe_to_be_mapped=$(((size_required - 1) / pe_size + 1)) #no of pe required or still left to mapped for this VG to disks
                no_of_actual_pe_required=$((no_of_pe_to_be_mapped + 1))  #adding 1 pe required for storing meta of PV
                no_of_pe_count=$((disk_free_size / pe_size))  #total number of pe space available on this disk

                Log "autoScaleStorage2: VG[$vg_name], pe_size[$pe_size]" 
                Log "autoScaleStorage2: VG[$vg_name], size_to_be_mapped[$size_required], no_pe_to_be_mapped[$no_of_pe_to_be_mapped] + 1 pe for PV meta, disk_free_size[$disk_free_size], total_free_pe_on_disk[$no_of_pe_count]"
    
                if [ $no_of_actual_pe_required -le $no_of_pe_count ]; then
                    no_of_actual_pe_mapped_on_disk=$no_of_pe_to_be_mapped
                    size_on_cur_disk=$((no_of_actual_pe_required * pe_size)) 
                    pvlist=$pvlist"$disk"":""$disk_last_part"","
                    
                    echo "LVM## VG=$vg_name PVs=$pvlist" >> $STORAGE_TREE/recreate_info_all
                    Log "autoScaleStorage2: VG[$vg_name], no_pe_mapped_on_this_disk[$no_of_actual_pe_mapped_on_disk], total_pe_yet_to_be_maped[0], mapping finished"
                    is_mapped=1
                elif [ $no_of_pe_count -ge 2 ]; then          #disk must have space available for at least one pe plus one meta pe
                    size_on_cur_disk=$((no_of_pe_count * pe_size))
                    no_of_actual_pe_mapped_on_disk=$((no_of_pe_count - 1)) #substracting 1 pe for meta
                    no_of_pe_required=$((no_of_pe_to_be_mapped - no_of_actual_pe_mapped_on_disk))
                    size_required=$((no_of_pe_required * pe_size))
                    pvlist=$pvlist"$disk"":""$disk_last_part"","

                    Log "autoScaleStorage2: VG[$vg_name], no_pe_mapped_on_this_disk[$no_of_actual_pe_mapped_on_disk], total_pe_yet_to_be_mapped[$no_of_pe_required], mapping pending"
                else
                    Log "autoScaleStorage2: VG[$vg_name], no_pe_mapped_on_this_disk[0], total_pe_yet_to_be_mapped[$no_of_pe_to_be_mapped], Could not mapped any pe on this disk, mapping pending"
                    continue #try next disk in this destination disk list
                fi
                Log "autoScaleStorage2: VG[$vg_name], PartInfo[$part_info]"
                part_info=`GeneratePartInfo $ptype $disk_last_part $disk_used_size $size_on_cur_disk "lvm;"`
            else

                Log "autoScaleStorage2: Device[$device], size_to_be_mapped[$size_required], disk_free_size[$disk_free_size]"

                if [ $size_required -lt $disk_free_size ]; then
                    size_on_cur_disk=$size_required
                    resdisk=$disk
                    resdevice="$resdisk:$disk_last_part"

                    if [ $is_luks -eq 1 ]; then
                        cryptdevname=`echo $device | cut -d ":" -f1`
                        bkpdevice=`echo $device | cut -d ":" -f2-`
                        echo "LUKS## name="$cryptdevname" DISK=$disk:$disk_last_part" >> $STORAGE_TREE/recreate_info_all
                    else
                        bkpdevice=$device
                        echo "DISK## $disk:$disk_last_part" >> $STORAGE_TREE/recreate_info_all
                    fi
                    
                    echo "$bkpdevice $resdevice" >> $f_st_part_map
                    Log "autoScaleStorage2: Device[$device], size_mapped_on_disk[$size_required], size_yet_to_be_mapped[0], mapping finished"
                    is_mapped=1
                else
                    Log "autoScaleStorage2: Device[$device], size_mapped_on_disk[0], size_yet_to_be_mapped[$size_required], mapping pending"
                    continue #try next disk in this destination disk list
                fi
                part_info=`GeneratePartInfo $ptype $disk_last_part $disk_used_size $size_on_cur_disk $flags`
                Log "autoScaleStorage2: Device[$device], PartInfo[$part_info]"
            fi

            disk_used_size=$((disk_used_size + size_on_cur_disk))
            Log "autoScaleStorage2: Disk[$disk], size_mapped[$size_on_cur_disk], new used size[$disk_used_size], new_last_part[$disk_last_part]"

            echo $disk_used_size > $disk_node/used_size
            echo $disk_last_part > $disk_node/last_part
            echo "$part_info" >> $disk_node/partinfo
            echo "msdos" > $disk_node/ptype

            if [ $is_mapped -eq 1 ]; then  #if src partition or volume group has been mapped successfully then move to next src partition
                break
            fi
        done

        if [ $is_mapped -eq 0 ]; then
            Log "autoScaleStorage2: This entry could not be processed"
            return 1
        fi
    done

    cp $STORAGE_TREE/recreate_info_all $STORAGE_TREE/recreate_info_all_display
    sed -i 's/DISK##/Partition and Format <disk:partiton#> -/g' $STORAGE_TREE/recreate_info_all_display
    sed -i 's/LVM##/Re-create LVM -/g' $STORAGE_TREE/recreate_info_all_display

    touch $STORAGE_TREE/AUTOSCALESTORAGE
    cat $STORAGE_TREE/recreate_info_all
    cat $f_st_part_map

    Log "Debug : autoScaleStorage2 Exiting"
    return 0
}



#dump recreation_info for each mount point to be restored to a file and unique sort it
#   Name: dumpRecreationInfo
#   API DEF:
#       IN:             arg1=mount_tree directory  arg2=outfile
#       OUT:            none
#       EXIT:           no
dumpRecreationInfo()
{
    mdir=$1
    outfile=$2

    Execute "rm -f $outfile"

    for mountnum in `ls $mdir`;
    do
        Log "dumpRecreationInfo: Debug: Process mountnum=[$mdir/$mountnum]"
        DIR=$mdir/$mountnum
        if [ ! -d $DIR ]; then
            Log "dumpRecreationInfo: Error: mount dir [$DIR] does not exist!"
            return 2
        fi
        if [ ! -f $DIR/to_restore ]; then
            Log "dumpRecreationInfo: Debug: $DIR/to_restore does not exist. Skip it"
            continue
        fi
        restore_flag=`cat $DIR/to_restore`
        if [ "x$restore_flag" != "x1" ]; then
            Log "dumpRecreationInfo: Debug: $DIR/to_restore [$restore_flag] not set. Skip it"
            continue
        fi
        if [ ! -f $DIR/recreation_info ]; then
            Log "dumpRecreationInfo: Error: $DIR/recreation_info not found!"
            return 1
        else
            Log "dumpRecreationInfo: Debug: dump [$DIR/recreation_info] [`cat $DIR/recreation_info`] to [$outfile]"
            cat $DIR/recreation_info >> $outfile
        fi
    done
    if [ -f $outfile ]; then
        Log "dumpRecreationInfo: Debug: Dumped all recreation_info to [$outfile] ==[`cat $outfile`]"
        cat $outfile | sort -u > /tmp/sortfile.$$
        mv /tmp/sortfile.$$ $outfile
        Log "dumpRecreationInfo: Debug: Sorted recreation_info to [$outfile] == [`cat $outfile`]"
        return 0
    else
        Log "dumpRecreationInfo: Error: dumped recreation_info file [$outfile] not found!"
        return 1
    fi
}

#rewrite disk names from DISK_MAP
#   Name: rewriteDiskNamesFromMap
#   API DEF:
#       IN:             arg1=infile arg2=outfile  (Asuumption: DISK_MAP file present at $STORAGE_TREE/DISK_MAP)
#       OUT:            none
#       EXIT:           no
rewriteDiskNamesFromMap()
{
    infile=$1
    outfile=$2
    mapfile=$STORAGE_TREE/DISK_MAP

    if [ ! -f $mapfile ]; then
        Log "rewriteDiskNamesFromMap: Debug: MAP file [$mapfile] not found"
        return 1
    fi
    if [ ! -f $infile ]; then
        Log "rewriteDiskNamesFromMap: Error: infile file [$infile] not found"
        return 1
    fi
    if [ -f $outfile ]; then
        Log "rewriteDiskNamesFromMap: Debug: outfile file [$outfile] present.Delete it"
        Execute "rm -f $outfile"
    fi
#    Execute "cp -f $infile $outfile"
    Execute "rm -rf $outfile"

    workingInfile=$STORAGE_TREE/recreate_workingfile
    Execute "rm -rf $workingInfile"
    Execute "cp -f $infile $workingInfile"

    cat $mapfile | while read line
    do
        Log "rewriteDiskNamesFromMap: Debug: Read line [$line] from map file"

        bckdisk=`echo $line | awk '{print $1}'`
        resdisk=`echo $line | awk '{print $2}'`
        Log "rewriteDiskNamesFromMap: Debug: bckdisk=[$bckdisk] resdisk=[$resdisk]"

        if [ "$bckdisk" == "$resdisk" ]; then
            Log "rewriteDiskNamesFromMap: Nothing to rewrite.. skip"
            continue
        else
            Execute "rm -rf /tmp/out"
            echo $bckdisk | sed -e 's/[\/&]/\\&/g' > /tmp/out
            bckdisk_e=`cat /tmp/out`
            
            echo $resdisk | sed -e 's/[\/&]/\\&/g' > /tmp/out
            resdisk_e=`cat /tmp/out`"@"
            Execute "rm -rf /tmp/out"

            Log "rewriteDiskNamesFromMap: Debug: extract all lines with [$bckdisk] from [$workingInfile]"
            cat $workingInfile | sed 's/'$bckdisk_e:'/'$resdisk_e:'/g'> $outfile
            Execute "cp $outfile $workingInfile"
        fi
    done
    cat $workingInfile | sed 's/'@'/''/g' > $outfile

}

#Generate list of skipped mountpoints
#   Name: genSkippedMountsList
#   API DEF:
#       IN:             tree, output file name
#       OUT:            none
#       EXIT:           no
genSkippedMountsList()
{
    tree=$1
    OFILE=$2

    for dir in `ls $STORAGE_TREE/$tree/mount_tree`;
    do
        to_restore=`cat $STORAGE_TREE/$tree/mount_tree/$dir/to_restore`
        Log "$STORAGE_TREE/$tree/mount_tree/$dir --> to_restore [$to_restore]"
        if [ -z "$to_restore" -o "$to_restore" == "0" ]; then
            mntpt=`cat $STORAGE_TREE/$tree/mount_tree/$dir/mntpoint`
            Log "...add [$mntpt] to skip list"
            echo "$mntpt" >> $OFILE
        else
            Log "...not skipping [$mntpt]"
        fi
    done
}

#Copy partition type/table from backed up to current disk tree
#   Name: copyPartitions
#   API DEF:
#       IN:             none
#       OUT:            none (DISK_MAP and disk node for backup/restore tree must be present)
#       EXIT:           no
copyPartitions()
{
    mapfile=$STORAGE_TREE/DISK_MAP
    if [ ! -f $mapfile ]; then
        Log "copyPartitions: Error: MAP file [$mapfile] not found"
        return 1
    fi
    BTREE=$STORAGE_TREE/backup/disks
    RTREE=$STORAGE_TREE/restore/disks
    if [ ! -d $BTREE ]; then
        Log "copyPartitions: Backup disks tree not found at [$BTREE]"
        return 1
    fi
    if [ ! -d $RTREE ]; then
        Log "copyPartitions: Restore disks tree not found at [$BTREE]"
        return 1
    fi

    Execute "rm -rf $STORAGE_TREE/restore/disks_before_edit"
    Execute "cp -a $RTREE $STORAGE_TREE/restore/disks_before_edit"

    cat $mapfile | while read line
    do
        Log "copyPartitions: Debug: Read line [$line] from map file"
        bckdisk=`echo $line | awk '{print $1}'`
        resdisk=`echo $line | awk '{print $2}'`

        bckdiskNode=`DiskNodeForDiskName "backup" "$bckdisk"`
        if [ -z "$bckdiskNode" ]; then
            Log "copyPartitions: Error: Backup disk node for  [$bckdisk] not found"
            return 1
        fi
        resdiskNode=`DiskNodeForDiskName "restore" "$resdisk"`
        if [ -z "$resdiskNode" ]; then
            Log "copyPartitions: Error: Restore disk node for  [$resdisk] not found"
            return 1
        fi
        Log "copyPartitions: Debug: Found Backup disk node [$bckdiskNode] for [$bckdisk] and Restore disk node [$resdiskNode] for [$resdisk]"

        if [ ! -f "$bckdiskNode/ptype" ]; then
            Log "copyPartitions: Error: ptype not found in [$bckdiskNode]"
            return 1
        else
            Execute "cp -f $bckdiskNode/ptype $resdiskNode"
            ExecuteRet "cp -f $bckdiskNode/ptype_disp $resdiskNode"
            Log "copyPartitions: Debug: Copied [$bckdiskNode/ptype -> $resdiskNode]"
        fi

        if [ -f "$resdiskNode/resized" ]
        then
            Log "copyPartitions: resized found in [$bckdiskNode], partition info has already been copied"
            continue
        else
            #Copying partition dirs from backup disk to restore disk if they are not mapped by autoScale  
            for partdir in `ls -d $bckdiskNode/[0-9]*`
            do
                Execute "cp -r $partdir $resdiskNode/"               
            done
        fi

        if [ ! -f "$bckdiskNode/partinfo" ]; then
            Log "copyPartitions: Error: partinfo not found in [$bckdiskNode]"
            return 1
        else
            Execute "cp -f $bckdiskNode/partinfo $resdiskNode"
            Log "copyPartitions: Debug: Copied [$bckdiskNode/partinfo -> $resdiskNode]"
        fi
    done
    return 0
}

#Look up DISK MAP
#   Name: lookupDiskMap
#   API DEF:
#       IN:             arg1=<backup|restore> arg2=<name>
#       OUT:            name from the other column, empty string if not found
#       EXIT:           no
lookupDiskMap()
{
    from=$1
    name=$2
    if [ -z "$from" ]; then
        Log "lookupDiskMap: Error: arg1=backup|restore"
        return
    fi
    if [ -z "$name" ]; then
        Log "lookupDiskMap: Error: arg2=<name>"
        return
    fi
    if [ "$from" != "backup" -a "$from" != "restore" ]; then
        Log "lookupDiskMap: Error: arg1=backup|restore"
        return
    fi

    if [ ! -f $STORAGE_TREE/DISK_MAP ]; then
        Log "lookupDiskMap: Error: $STORAGE_TREE/DISK_MAP not found"
        return
    fi

    cat $STORAGE_TREE/DISK_MAP | while read line
    do
        bck=`echo $line | awk '{print $1}'`
        res=`echo $line | awk '{print $2}'`
        
        if [ $from == "restore" ]; then
            if [ "$res" == "$name" ]; then
                Log "lookupDiskMap: Debug: Found match on line [$line]. Return bck[$res]"
                echo "$bck"
                return
            fi
        else
            if [ "$bck" == "$name" ]; then
                Log "lookupDiskMap: Debug: Found match on line [$line]. Return res[$bck]"
                echo "$res"
                return
            fi
        fi
    done
}

#recreate device name from diskname and partnum
#   Name: recreateDeviceName
#   API DEF:
#       IN:             arg1=<current|backup> arg2=<diskname> arg3=<partnum>
#       OUT:            device name
#       EXIT:           no
recreateDeviceName()
{
    from=$1
    diskname=$2
    partnum=$3

    if [ "$partnum" == "0" ]; then
        Log "recreateDeviceName: since partnum=0, return $diskname"
        echo $diskname
        return
    fi

    delim=`cat /system_state/sysconf/kpartx_delim`
    Log "recreateDeviceName: Debug: Read kpartx_delim [$delim] from system state"

    if [ "$from" == "restore" ]; then
        Log "recreateDeviceName: Debug: Locate in current system"
        devmajmin=`getDevMajMin $diskname`
        if [ -z "$devmajmin" ]; then
            Log "recreateDeviceName: Error: Cannot find maj:min for disk [$diskname]"
            return
        fi
        maj=`echo $devmajmin | cut -d':' -f1`
        if [ "$maj" == "$DEV_DM" -o "$maj" == "$DEV_CPQARRAY" ]; then
            #partitiondevice = <disk><kpartx_delim><partno>
            device="${diskname}${delim}${partnum}"
        else
            device="${diskname}${partnum}"
        fi
    elif [ "$from" == "backup" -o "$from" == "backup_pv" ]; then
        Log "recreateDeviceName: Debug: Locate from backup system"
        bdiskname=`basename $diskname`
        isdm=`find $STORAGE_TREE/backup/disks/DM $STORAGE_TREE/backup/disks/CPQARRAY -name $bdiskname`
        if [ -n "$isdm" ]; then
            Log "recreateDeviceName: Debug: Found dm device for [$bdiskname]"
            device="${diskname}${delim}${partnum}"
        else
            Log "recreateDeviceName: Debug: dm device not found."
            device="${diskname}${partnum}"
        fi

    else
        Log "recreateDeviceName: Error: Invalid arg1. should be <backup|restore>"
        return
    fi
    Log "recreateDeviceName: Debug: reconstructed device=[$device] from [$diskname]:[$partnum]"
  Log "recreateDeviceName: Debug: Check whether node[$device] even exist or not"
    if [ $from == "backup" ]
    then
        echo $device
        return
    fi
    if [ $from == "backup_pv" ]
    then
        #Verify it with saved pv devices present in pvs.list
        cat $STORAGE_TREE/backup/lvm_metadata/pvs.list|grep -wqE "^$device"
        initDevice=$device
        if [ $? -eq 0 ]
        then
            echo $device
            return
        fi
        for i in ${DELIM_OPTIONS[*]}
        do
            Log "Debug: delim[$i].Check device[$diskname""$i""$partnum""] exist or not"
            device="$diskname""$i""$partnum"
            cat $STORAGE_TREE/backup/lvm_metadata/pvs.list|grep -wqE "^$device"
            if [ $? -eq 0 ]
            then
                echo $device
                echo "$i" > /system_state/sysconf/kpartx_delim
                return
            fi
        done
        #Last try. Add nothing for delimiter.
        device="$diskname""$partnum"
        cat $STORAGE_TREE/backup/lvm_metadata/pvs.list|grep -wqE "^$device"
        if [ $? -eq 0 ]
        then
            echo $device
            return
        fi
        # if we reach here it might be issue with pvs.list. Anyway pass the default device.
        echo $initDevice
        return
    fi
    if [ -e $device ]
    then
        echo $device
        return
    fi
    for i in ${DELIM_OPTIONS[*]}
    do
        Log "Debug: delim[$i].Check device[$diskname""$i""$partnum""] exist or not"
        device="$diskname""$i""$partnum"
        if [ -e $device ]
        then
            echo $device
            echo "$i" > /system_state/sysconf/kpartx_delim
            return
        fi
    done
    #Last try. Add nothing for delimiter
    device="$diskname""$partnum"
    echo $device
    return
}


# Create Partinfo script for input disk
#   Name: createPartInfoScript
#   API DEF:
#       IN:           arg1=diskNode  arg2=runscript{true|false} arg3=commit{true|false}
#       OUT:          0: success 1: error
#       EXIT:         no
createPartInfoScript()
{
    local diskNode=$1
    local execscript=$2
    local commit_mode=$3
    local diskname=`basename $diskNode`
    diskname="/dev/$diskname"

    if [[ $diskNode == *"CPQARRAY"* ]]; then
        diskname="/dev/cciss/`basename $diskNode`"
    elif [[ $diskNode == *"DM"* ]]; then
        diskname="/dev/mapper/`basename $diskNode`"
    fi

    Log "Invoked createPartInfoScript with diskName[$diskname] diskNode[$diskNode] runopt[$execscript] commitMode[$commit_mode]"

    #sort the partinfo file by the partition numbers
    mv $diskNode/partinfo $diskNode/partinfo.before_sort
    cat $diskNode/partinfo.before_sort | sed 's/partno=/partno=@/' | sort -n -t\@ -k2,2 | tr -d '@' > $diskNode/partinfo

    local ptype=`cat $diskNode/ptype`
    local ptype_disp=`cat $diskNode/ptype_disp` #exact partition table type, e.g. 'gpt_sync_mbr' when ptype might have just 'gpt' - partition labels are different for each distro; our parted is from git
    local ptype_lbl=$ptype
    if [ "$ptype" == "gpt" ]; then
        if [[ "$ptype_disp" == *"gpt"* ]]; then #currently, lets handle only PTs that are slight variants of gpt, e.g. gpt_sync_mbr
            ptype_lbl=$ptype_disp
        fi
    fi
    SCRIPT=$diskNode/parted_script
    echo "#!/bin/bash" > $SCRIPT
    echo "parted $diskname --script mklabel $ptype_lbl || exit 1" >> $SCRIPT

    local disk_for_boot="$STORAGE_TREE/postadm/disk_for_boot"
    local boot_diskName=`cat $disk_for_boot`
    local source_had_uefi=`cat $SYS_STATE_LOC/efi_loc | wc -c`
    local append_bbp=0
    grep -q "flags=bios_grub" $diskNode/partinfo
    if [ $? -ne 0 -a "$ptype" == "gpt" -a -e "$disk_for_boot" -a "$commit_mode" == "true" -a "x$boot_diskName" == "x$diskname" -a ! -e "/tmp/yastgui/unified_mode" ]; then
        append_bbp=1
    fi
    if [ -e "/tmp/recovery_jobid" -a "$ptype" == "gpt" -a "$source_had_uefi" -gt 0 ]; then #for vME of a UEFI machine, destination will not have UEFI - need to append BBP
        append_bbp=1
    fi

    if [ $append_bbp -eq 1 ]; then
        #check that we haven't already added a bbp- either because it was present in the source OR this is a restart
        ExecuteRet "grep -q bios_grub $diskNode/partinfo"
        ret=$?
        if [ $ret -eq 0 ]; then
            append_bbp=0
        fi
    fi

    if [ $append_bbp -eq 1 ]; then
        Log "Adding a BIOS boot partition for GPT labelled disk [$diskname]"
        bpartno=`tail -1 $diskNode/partinfo | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "partno="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        bpartno=$((bpartno+1))
        bstart=`tail -1 $diskNode/partinfo | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "start="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        bsize=`tail -1 $diskNode/partinfo | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "size_in_blks="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        bstart=$((bstart+bsize))
        bsize=4096   # 2MB
        echo "name= ptyp=primary partno=$bpartno start=$bstart blk_sz=512 size_in_blks=$bsize fstype= flags=bios_grub;" >> $diskNode/partinfo
        Log "Appended [name= ptyp=primary partno=$bpartno start=$bstart blk_sz=512 size_in_blks=$bsize fstype= flags=bios_grub;] to [$diskNode/partinfo]"
    fi

    cat $diskNode/partinfo | while read pline
    do
        name=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "name="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        if [ "$ptype" == "gpt" -a "$ptype" == "$ptype_lbl" ]; then
            parttype="p" #for gpt we need to specify a name
        else
            #for msdos we need a type. primary|extended|logical
            parttype=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "ptyp="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        fi
        partno=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "partno="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        start=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "start="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        sizeblk=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "size_in_blks="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        blksz=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "blk_sz="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        cur_blksz=`blockdev --getss $diskname`
        if [ $cur_blksz -eq 0 ]; then
            # safety check, will default to old behavior of ignoring block size difference between backed up and current machine
            size=$sizeblk
        else
            size=$(($sizeblk*$blksz/$cur_blksz))
        fi
        Log "for [$diskname]: sizeblk [$sizeblk] blksz [$blksz] cur_blksz [$cur_blksz] new size [$size]"
        end=$((start+size-1))
        fstype=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "fstype="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        fstype=`echo $fstype | cut -d'(' -f1`
        flags_all=`echo $pline | awk '{for(i=1;i<=NF;i++) { mtok = match($i, "flags="); if (mtok) print $i; } }' | awk -F"=" '{print $2}'`
        flaglist=""
    if [ -n "$flags_all" -a "$flags_all" != ";" ]; then
          flaglist=`echo $flags_all | awk -F";" -v diskname=$diskname -v partno=$partno '{for(i=1;i<NF;i++) print "parted " diskname " --script set " partno " " $i " on";}'`
    fi
        #Now that we have information to re-create this partition, add the statements to the script
        case $fstype in
            "swap")
                    fstype="linux-swap"
                    ;;
            "vfat")
                    fstype="fat32"
                    ;;
            "Physical")
                    fstype=""
                    ;;
            "ext4")
                    uname -r|grep -q "el5"
                    if [ $? -eq 0 ]
                    then
                        fstype="ext3"  #parted command does not support ext4 fstype for redhat 5.x.
                    fi
                    ;;
            "*")
                    ;;
        esac

        # if we are working with legacy parted (RH 4.x), it does not accept sectors in the sizes for mkpart by only MBs
        parted_majorversion=`parted --version | head -n 1 | awk '{print $NF}' | cut -f1 -d'.'`
        parted_minorversion=`parted --version | head -n 1 | awk '{print $NF}' | cut -f2 -d'.'`
        usePar="s"
        if [ -z "$parted_minorversion" ]
        then
            # We might never come to this condition.Just putting to play safe!!!
            parted_minorversion=0
        fi
        if [ -n "$parted_majorversion" -a "$parted_majorversion" == "1" ]
        then
            if [ -n "$parted_minorversion" -a $parted_minorversion -lt 6 ] || [ -z "$parted_minorversion" ]
            then
                usePar="m"
            fi
       fi
       if [ -e /tmp/parted_m_flag ]
       then
           usePar="m"
       fi
       #Sector flag have more preference than other mb unit
       if [ -e /tmp/parted_s_flag ]
       then
           usePar="s"
       fi

        if [ x"$usePar" == "xm" ]
        then
            startMB=$(( ( (1024*1024 - 1) + (start * 512))/(1024*1024) ))
            endMB=$(( ( (1024*1024 - 1) + (end * 512))/(1024*1024) ))
            echo -e "parted $diskname --script mkpart $parttype $fstype ${startMB} ${endMB} || exit 1" >> $SCRIPT
        else
            echo -e "parted $diskname --script mkpart $parttype $fstype ${start}s ${end}s || exit 1" >> $SCRIPT
        fi
        if [ -n "$flaglist" ]; then
            echo -e "$flaglist" >> $SCRIPT
        fi
        devmajmin=`getDevMajMin $diskname`
        if [ -z "$devmajmin" ]; then
            Log "createPartInfoScript: Debug: Cannot find device maj:min for [$diskname]"
            return 1
        fi
        maj=`echo $devmajmin | cut -d':' -f1`
        echo "/sbin/blockdev --rereadpt $diskname" >> $SCRIPT
        echo "sleep 4" >> $SCRIPT
        if [ "$maj" == "$DEV_DM" ]; then
            Log "recreatePartitions: Debug: Detected DM device. Add kpartx delete/add to ensure discovered partitions are sane and their sysfs uuids are updated"
            echo "kpartx -d $diskname" >> $SCRIPT
            echo "sleep 4" >> $SCRIPT
            echo "kpartx -a $diskname" >> $SCRIPT
            echo "sleep 4" >> $SCRIPT
        fi

        Log "recreatePartitions: Debug: Extracted partition info for [$diskname] as [$name]:[$parttype]:[$partno]:[$start]:[$end]:[$fstype]:[$flags_all]:[$flaglist]"
    done
    if [ x"$execscript" == "xtrue" -a -e "$diskNode/partinfo" ]; then
        Log "createPartInfoScript: Debug: Created [$SCRIPT] with contents [`cat $SCRIPT`]"
        Log "createPartInfoScript: Debug: Execute [$SCRIPT]"
        chmod a+x $SCRIPT
        moveLibs #move /libdevmapper* and /libparted* out of the way to use the system parted libs
        /bin/bash $SCRIPT 2>&1 | tee -a $LOGFILE
        ret=${PIPESTATUS[0]}
        resetLibs
        if [ $ret -ne 0 ]; then
            Log "createPartInfoScript: Error: Execution of $SCRIPT failed with code [$ret]"
            return 1
        else
            Log "createPartInfoScript: Debug: Execution of $SCRIPT returned SUCCESS"
        fi
    fi
    return 0
}

# Create partinfo script for input disks list
#   Name: CreatePartinfoScriptForDiskList
#   API DEF:
#       IN:    arg1=disklist  arg2=tree{backup|restore|webadm} arg3=runscript{true|false} arg4=commit{true|false}
#       OUT:   0: success 1: error
#       EXIT:  no
CreatePartinfoScriptForDiskList()
{
    local disklist=$1
    local tree=$2
    Log "Invoked createPartInfoScriptForDiskList with disklist[$disklist] tree[$tree] runopt[$3] commitMode[$4]"
    cat $disklist | while read disk
    do
        diskNode=`DiskNodeForDiskName "$tree" "$disk"`
        if [ -z "$diskNode" ]; then
            Log "CreatePartinfoScriptForDiskList: Error: $tree node for [$disk] not found"
            return 1
        fi
        createPartInfoScript  "$diskNode" "$3" "$4"
        ret=$?
        if [ $ret == "1" ]; then
            Log "CreatePartinfoScriptForDiskList: Error: createPartInfoScript failed for disk[$disk] diskNode[$diskNode]"
            return 1
        fi
    done
    return 0
}

# Verify PVs for list of VGs and try dry pvcreate for them
#    Name: checkPVsForVGList
#    API DEF:
#        IN:      arg1=tree
#        OUT:     0: success 1: error
#        EXIT:    no
checkPVsForVGList()
{
    local tree=$1
    moveLibs
    cat "$tree/vgs.lst" | while read vgpath
    do
        Log "checkPVsForVGList: Debug: Processing VG [$vgpath]"
        vgdev=`echo $vgpath | cut -d"/" -f3`
        for pv in `ls $tree/lvm_metadata/$vgdev/pvs/`;
        do
            pv="/dev/$pv"
            ExecuteRet "pvcreate -ff -y --test $pv"
            ret=$?
            if [ $ret -ne 0 ]; then
                echo $pv | grep -qe ".*[0-9]$"
                ret=$?
                if [ $ret -ne 0 ]; then
                    Log "checkPVsForVGList: Error: pvcreate test failed for PV[$pv] VG[$vgpath] rc[$ret]"
                    atLeastOneFailure="yes"
                fi
            fi
        done
    done
    resetLibs
    if [ -n "$atLeastOneFailure" ]; then
        return 1
    fi
    return 0
}

# Verify if all selected mountpoints have been mapped via manual intervention
#   Name: recheckManualMapping
#   API DEF:
#       IN:             tree
#       OUT:            0:success 1: error. Set YAST error too
#       EXIT:           no
recheckManualMapping()
{
    tree="$STORAGE_TREE/$1"
    Log "recheckManualMapping: Debug: treePath[$tree]"
#SEQUENCE
#  -- find a way to save disk name for each device in PVs and mountpoints -- use that below instead of device
#  1. for each subdirectory [M] of $tree/mount_tree/, check if [M]/device (remove trailing numbers as workaround) is found under $tree/postadm/disks/*/*/dev_path  (e.g. grep sda $tree/disks/*/*/dev_path)
#  2. if found, add disk to list [disks]. if not found try step 3
#  3. check if the device is of form /dev/VG/LV, and see if that exists : $tree/lvm_metadata/VG/lvs/LV. If yes, add to list of [VGs] and each of its pvs to [disks] -- here again if you have disk names use them else rmeove trailing numbers as a workaround. Error out if not found.
#  4. --- reorganize recreatePartitionsForDisk() from storage_recreate.sh to move code that creates partinfo script to a common function.
#  5. For each [disk] in the list created in step 2/3, create partinfo script in no-commit mode (dry run) and return errors if any
#  6. For each [VG] in list above, find all PVs it requires and try pvcreate in no-commit mode (dry run) and return errors if any
#  -- Functions for [5] and [6] above need to be placed in a common file that can ben used in storage_recreate as well --
#  7. Mark flag in /tmp/storage_tree/webadm_verified
#  done (refer to storage_recreate.sh : line ~720

    if [ -f "${STORAGE_TREE}/webadm_verified" ]; then
        rm -f "${STORAGE_TREE}/webadm_verified"
    fi
    if [ ! -d "$tree" ]; then
        Log "recheckManualMapping: Error: postADM tree not found"
        return 1
    fi

    for mountnum in `ls $tree/mount_tree/`;
    do
        Log "recheckManualMapping: Debug: Process mount#[$mountnum] underlyingDevice[`cat $tree/mount_tree/$mountnum/device`]"
        mdev=`cat "$tree/mount_tree/$mountnum/device" | sed 's/[0-9]*$//'`
        Log "recheckManualMapping: Debug: truncated device [$mdev]"
        grep --quiet  $mdev  $tree/disks/*/*/dev_path
        if [ $? -eq 0 ]; then
            # is disk?
            Log "recheckManualMapping: Debug: appending [$mdev] to the Disk list"
            echo $mdev >>  "$tree/disks.lst"
            # if GPT partitioning for /boot mountpoint disk, save the diskname
            if [ `cat $tree/mount_tree/$mountnum/mntpoint` == "/boot" ]; then
               for devpath in `find $tree/disks -maxdepth 2 -mindepth 2`;
               do
                   if [ `cat $devpath/ptype` == "gpt" -a `cat $devpath/dev_path` == "$mdev" ]; then
                       cat $devpath/dev_path >  "$tree/disk_for_boot"
                       Log "recheckManualMapping: Debug: saved device [$devpath/dev_path] in $tree/disk_for_boot"
                       break
                   fi
               done
            fi 
            echo $mdev | grep -q "mapper"
            if [ $? -eq 0 ]
            then
                touch $tree/mount_tree/$mountnum/kpartx_delim
            fi
        else
            # is LVM device?
            lv=`cat "$tree/mount_tree/$mountnum/device" | cut -d"/" -f4`
            if [ -n "$lv" ]; then
                vg=`cat "$tree/mount_tree/$mountnum/device" | cut -d"/" -f3`
                if [ -f "$tree/lvm_metadata/$vg/lvs/$lv" ]; then
                    Log "recheckManualMapping: Debug: appending LV [/dev/$vg/$lv] to LV list"
                    cat "$tree/mount_tree/$mountnum/device" >> "$tree/lvs.lst"
                    Log "recheckManualMapping: Debug: appending [/dev/$vg] to the VG list"
                    echo "/dev/$vg" >>  "$tree/vgs.lst"
                    for pv in `ls $tree/lvm_metadata/$vg/pvs/`;
                    do
                         Log "recheckManualMapping: Debug: appending [/dev/$pv] to the PV list"
                         #Fix Me for SLES MPATH. Currently works on only /dev/mapper/mpath* devices.
                         echo $pv|grep -qE "^m.*$"
                         if [ $? -ne 0 ]
                         then
                             echo "/dev/mapper/$pv" >> "$tree/pvs.lst"
                         else
                             echo "/dev/$pv"  >> "$tree/pvs.lst"
                         fi
                         pvdev=`grep DEVICE "$tree/lvm_metadata/$vg/pvs/$pv" | awk '{print $2}' | sed 's/[0-9]*$//'`
                         Log "recheckManualMapping: Debug: appending [$pvdev] to the Disk list" 
                         echo $pvdev >>  "$tree/disks.lst"
                    done
                else
                    Log "recheckManualMapping: Error: LV device[/dev/$vg/$lv] not found under lvm_metadata tree"
                    return 1
                fi
            fi
        fi
    done

    cp "$tree/disks.lst" "$tree/disks.lst.bkp"
    cat "$tree/disks.lst.bkp" | sort | uniq >  "$tree/disks.lst"
    CreatePartinfoScriptForDiskList "$tree/disks.lst" "$1" "false" "false"
    ret=$?
    if [ "$ret" == "1" ]; then
        Log "recheckManualMapping: Error: CreatePartinfoScriptForDiskList failed for list[`cat $tree/disks.lst`]"
        return 1
    fi
    cp "$tree/vgs.lst" "$tree/vgs.lst.bkp"
    cat "$tree/vgs.lst.bkp" | sort | uniq >  "$tree/vgs.lst"
    checkPVsForVGList "$tree"
    ret=$?
    if [ "$ret" == "1" ]; then
        Log "recheckManualMapping: Error: checkPVsForVGList failed for list[`cat $tree/vgs.lst`]"
        return 1
    fi
    touch "${STORAGE_TREE}/webadm_verified"
    return 0
}


# Set up web ADM environment
#  Name: setup_webadm
#  API DEF:
#    IN:    none
#    OUT:  none
#    EXIT:  no
setup_webadm()
{
# prepare list of supported filesystems
    modprobe ext3
    modprobe ext4
    modprobe xfs
    modprobe btrfs
    modprobe reiserfs
    modprobe vfat
    cat /proc/filesystems | grep -v nodev > /tmp/kern_fs_list
    Log "setup_webadm: Debug: List of FS supported by kernel [`cat /tmp/kern_fs_list`]"

    Execute "rm -f /tmp/fs_list"
    while read line
    do
        which mkfs.${line}
        ret=$?
        if [ $ret -eq 0 ]; then
            Log "setup_webadm: Debug: Add $line as a supported FS"
            echo $line >> /tmp/fs_list
        else
            Log "setup_webadm: Debug: Not ading $line as a supported FS as userspace utils are missing"
        fi
    done < /tmp/kern_fs_list
    Log "setup_webadm: Debug: List of FS supported by kernel and userspace [`cat /tmp/fs_list`]"

# extract webadm from CDROM into chroot
    rm -rf /chroot/webadm*
    cp -f /cdrom/webadm.tgz /chroot
    ret=$?
    if [ $? -ne 0 ]; then
        Log "Error: Cannot extract webadm from DVD"
        return
    else
        Log "Extracted webadm from DVD"
    fi

# prepare directories and www and run the ADM
    prepmount_cmd
    OLDDIR=`pwd`
    cd /chroot
    cp -a /cdrom/webadm.tgz .
    rm -rf webadm
    tar zxf ./webadm.tgz
    cd ./webadm
    mkdir www
    chmod +x ./prepare_www.sh
    Execute "./prepare_www.sh"

    cd backend
    chmod +x ./webadm.py
    Log "About to start webadm"
    chroot /chroot /bin/bash -c "cd /webadm/backend; ./webadm.py >/dev/null 2>&1 &"
    ret=$?
    Log "Launched webadm : retcode [$ret] - ps dump >"
    ps -efHww | grep webadm | tee -a $LOGFILE
    
    
    cd $OLDDIR
}
