#!/bin/bash
#This script contains code used by first stage as well as second stage of 1-Touch recovery
#Use this function for logging in all the scripts

LOG_FILE=/tmp/cvrestore.log
DMESG_OUTPUT=/var/log/messages
DEBUGMODETEST="/tmp/enable_kexec_debug"

YASTGUI="/tmp/yastgui"
CLIENTDETAILS="$YASTGUI/client"
CSDETAILS="$YASTGUI/csdetails"
FWDETAILS="$YASTGUI/fwdetails"
DNSDETAILS="$YASTGUI/dns"
LASTERROR="$YASTGUI/failure_reason"
YASTINPUT="$YASTGUI/input"
YASTOUTPUT="$YASTGUI/output"
YAST_FSDATA_JOBLIST="$YASTGUI/fsdata_joblist"
YAST_SYSTEMSTATE_JOBLIST="$YASTGUI/systemstate_joblist"
YAST_FSDATA_JOBID="$YASTGUI/fsdata_jobid"
YAST_SYSTEMSTATE_JOBID="$YASTGUI/systemstate_jobid"
COPYPRECFILE="$YASTGUI/copyprecedence"
MANAMEFILE="$YASTGUI/maname"
BACKUPSETS_FILE="$YASTGUI/backupsetnames"
DEFAULTBACKUPSET="$YASTGUI/defaultbackupsetname"

YAST_NFS_MOUNT="$YASTGUI/nfsmount"

RESTORE_STAGE=1
CFG_FILE="/tmp/NETDETAILS"

RES_DISK_TREE=/python_dir/disk_tree
SYS_CONFIG_DIR=/tmp/sysconfig
SYS_STATE_LOC=/system_state/sysconf/
RES_NTWK_TREE=/tmp/config_res/networking/tree
RES_NTWK_TREE_STG1=/tmp/config_res/networking/tree_stg1
BAK_NTWK_TREE=$SYS_STATE_LOC/networkconfig/tree
FS_NTWK_DIR=/tmp/MOUNTDIR/etc/sysconfig
#Extract the required value from the registry
get_registry_value()
{
    cat $1 | while read line
    do
        key=`echo $line | cut -d' ' -f1`
        if [ "$key" = "$2" ]; then
            echo $line | awk '{print $2}'
            break
        fi
    done
}

file="/etc/CommVaultRegistry/Galaxy/Instance001/Base/.properties"
if [ -f $file ]; then
    GALAXY=`get_registry_value $file "dGALAXYHOME"`
    BASEDIR=`get_registry_value $file "dBASEHOME"`
    export LD_LIBRARY_PATH=/lib64:/usr/lib64:/lib/x86_64-linux-gnu:/usr/lib/x86_64-linux-gnu:/lib:/usr/lib:${GALAXY}/Base:/usr/X11R6/lib/:/ylibs
else
    export LD_LIBRARY_PATH=/lib64:/usr/lib64:/lib/x86_64-linux-gnu:/usr/lib/x86_64-linux-gnu:/lib:/usr/lib:/opt/simpana/Base:/usr/X11R6/lib/:/ylibs
fi
if [ -f /extra_ld_library_paths ]; then
    export LD_LIBRARY_PATH=$LD_LIBRARY_PATH:`cat /extra_ld_library_paths`
fi
PATH=/bin:/usr/bin:/sbin:/usr/sbin:/usr/local/bin:/usr/local/sbin
USER=root
export PATH USER
VERSION=`uname -r`
if [ -f /cloud_flag ]
then
    VERSION=`ls /boot|grep vmlinuz|sed 's/vmlinuz-//g'`
fi
MODULES_PATH=/lib/modules/${VERSION}

RESPONSE_FILE=onetouchresponse.xml
RESPONSE_FILE_PATH=${YASTGUI}/onetouchresponse.xml
RESPONSE_LOCATION=${YASTGUI}/resp
include_mpath_initrd="/tmp/include_mpath_initrd"
include_lvm_initrd="/tmp/include_lvm_initrd"

AUTOLAYOUTFILE=/tmp/autolayout

#The following are the files that will control the drivers that will be included in initrd image
initrd_control_directory="/tmp/1tch_initrd_control"
minimal_initrd="${initrd_control_directory}/create_minimal_initrd"
add_network_initrd="${initrd_control_directory}/add_nic_initrd"
add_removable_scsi_disks="${initrd_control_directory}/add_removable_scsi"
add_removable_ide_disks="${initrd_control_directory}/add_removable_ide"

#The following files are global variables that will be set by the GRUB installation code
grub_control_directory="/tmp/1tch_grub_control"
grub_boot_device="${grub_control_directory}/boot_device"
grub_boot_mnt_point="${grub_control_directory}/boot_mnt"

Log()
{
        DATE=`date +%D' '%H:%M:%S`
        echo "${DATE} ${FUNCNAME[1]}@${BASH_SOURCE[0]} ${BASH_LINENO[0]} $*"
}

LogFile()
{
    DATE=`date +%D' '%H:%M:%S`
    echo "${DATE} ${FUNCNAME[1]}@${BASH_SOURCE[0]} ${BASH_LINENO[0]} $*" >> $LOG_FILE
}
log_last_error()
{
    echo -e "$@" >> $LASTERROR
}
reset_last_error()
{
    rm -f $LASTERROR
    touch $LASTERROR
}

determine_kernel_cmdline_arg()
{
    key=$1  
    cmd_line=`cat /proc/cmdline`
    set $cmd_line
    until [ -z "$1" ]  # Until all parameters used up . . .
    do
        parm=`echo $1 | cut -d '=' -f 1`
        value=`echo $1 | cut -d '=' -f 2`
        if [ "x$parm" == "x$key" ]; then
             echo "$value"
             return              
        fi
        shift
    done
    echo "invalid"  
    return               
}

#Determine whether the 1-Touch restore was started through CD or network boot
determine_boot_mode()
{
    [[ -e $DEBUGMODETEST ]] && rm -f $DEBUGMODETEST 

    kern_parm=`determine_kernel_cmdline_arg "method"`
    mode=`echo $kern_parm | cut -d ':' -f 1`
    if [ "x$mode" == "xnfs" ]; then
         boot_mode=NETWORK
         nfs_ip=`echo $kern_parm | cut -d ':' -f 2`
         nfs_mnt=`echo $kern_parm | cut -d ':' -f3` 
    fi      

    kern_parm=`determine_kernel_cmdline_arg "debug"`
    if [ "x$kern_parm" != "xinvalid" ]; then
         touch $DEBUGMODETEST
    fi
}

determine_boot_label()
{
    kern_parm=`determine_kernel_cmdline_arg "label"`
    if [ "x$kern_parm" != "xinvalid" ]; then
         boot_label=$kern_parm
    fi
}


#Determine the restore stage
determine_restore_stage()
{
    if [ -e /proc/cmdline ]; then   
         kern_parm=`determine_kernel_cmdline_arg "stage"`
         if [ "x$kern_parm" != "xinvalid" ]; then
              RESTORE_STAGE=$kern_parm
         fi
    fi
}

#Determine unified mode 
determine_restore_mode()
{
    if [ -e /proc/cmdline ]; then   
         kern_parm=`determine_kernel_cmdline_arg "unified"`
         if [ "x$kern_parm" != "xinvalid" ]; then
              RESTORE_MODE=$kern_parm
         fi
    fi
}

determine_restore_mode
if [ -n "$RESTORE_MODE" -o -f "$SYS_STATE_LOC/SRV11used" ]; then
    SYS_CONFIG_DIR="/tmp/sysconf"
fi

log_dmesg()
{
    echo "--------------------------------" >> $LOG_FILE
    echo "dmesg:" >> $LOG_FILE
    dmesg >> $LOG_FILE
    echo "--------------------------------" >> $LOG_FILE
}              

# Execute the command and check the return code
Execute()
{
    $*
    if [ $? -ne 0 ]
    then
        Log "Error: execution of '$*' failed" >> $LOG_FILE
        log_last_error "Error: execution of '$*' failed"
        log_dmesg
        exit 1 
    fi
}

job_pending_reason()
{
    job_id=$1
    ${GALAXY}/Base/qlist job -co r -j $job_id > /tmp/jpr
    if [ $? -ne 0 ]; then
        return 1
    fi
    jpr=$(tail -1 /tmp/jpr | cut -d" "  -f3-)
    echo "$jpr" 
}

get_job_status()
{
    job_id=$1
    job_description=$2

    job_status=" "
    ${GALAXY}/Base/qlist job -j $job_id > /tmp/JOBRUNNINGDETAILS 
    if [ $? -ne 0 ]; then
         return 1
    fi
    LINE=1
    cat /tmp/JOBRUNNINGDETAILS | while read line
    do
        if [ $LINE -eq 3 ]; then
             job_status=`echo $line` 
             echo "status=$job_status" > /tmp/job_status             
        fi 
        LINE=$((LINE+1))
    done

    job_status=`cat /tmp/job_status | awk '{print $3}'`    
    echo "status=$job_status" > $YASTOUTPUT
    if [ x"$job_status" = "xRunning" ]; then
         #The job is still running 
         #Get the percentage of job completion 
         percent=`cat /tmp/job_status | awk '{print $NF}'`
         echo "percent=$percent" >> $YASTOUTPUT          
    elif [ x"$job_status" = "xSuspended" ]; then
         jpr=`job_pending_reason $job_id`
         echo "jpr=$jpr" >> $YASTOUTPUT
    elif [ x"$job_status" = "xPending" ]; then
         jpr=`job_pending_reason $job_id`
         echo "jpr=$jpr" >> $YASTOUTPUT
     fi 
     return 0    
}

wait_for_job_completion()
{
     job_id=$1

     JOB_FLAG=1
     job_status=" "
     pending_ctr=0
     while [ $JOB_FLAG -eq 1 ]
     do
          sleep 10
          ${GALAXY}/Base/qlist job -j $job_id > /tmp/JOBRUNNINGDETAILS 
          LINE=1
          cat /tmp/JOBRUNNINGDETAILS | while read line
          do
              if [ $LINE -eq 3 ]; then
                  job_status=`echo $line` 
                  echo "$job_status" > /tmp/job_status             
              fi 
              LINE=$((LINE+1))
          done
          if [ ! -e /tmp/job_status ]; then
               continue
          fi              
          job_status=`cat /tmp/job_status | awk '{print $3}'`    
          if [ x"$job_status" = "xRunning" ]; then
              continue
          elif [ x"$job_status" = "xCompleted" ]; then
              rm /tmp/extra_restore_job_err
              break
          elif [ x"$job_status" = "xFailed" ]; then
              touch /tmp/extra_restore_job_err
              break
          elif [ x"$job_status" = "xSuspended" ]; then
              touch /tmp/extra_restore_job_err
              continue
          elif [ x"$job_status" = "xPending" ]; then
              #The job is in pending state. for the use case here, we are not polling. TBR
              pending_ctr=$((pending_ctr+1))
              if [ $pending_ctr -ge 3 ]; then
                  ${GALAXY}/Base/qoperation jobcontrol -o kill -j $job_id
                  touch /tmp/extra_restore_job_err
                  break
              else
                  continue
              fi
          elif [ x"$job_status" = "xKilled" ]; then
              touch /tmp/extra_restore_job_err
              break 
          fi 
     done
}

get_nth_line()
{
    file_name=$1
    line_num=$2
    count=1
    while read 
    do
        if [ $count -eq $line_num ]; then
             echo "$REPLY"
             return 0
        fi
        count=$((count+1))
    done < $file_name
    return 1
}

wait_with_timeout()
{
    clear
    delay=$1
    description=$2
    while [ $delay -gt 0 ]; do
        clear
        echo "Press enter key in $delay seconds to $description"
        read -t 1
        if [ $? -eq 0 ]; then
             break 
        fi
        delay=$((delay-1))
    done

    if [ $delay -gt 0 ]; then
         return 0
    else
         return 1
    fi
}            

clean_simpana()
{   Log "clean_simpana called" >> $LOG_FILE

    # Following code is `lifted` from cvsrrestore.sh
    # 1-Touch will always be installed as first instance
    file="/etc/CommVaultRegistry/Galaxy/Instance001/Base/.properties"
    if [ -f $file ]; then
        GALAXY=`get_registry_value $file "dGALAXYHOME"`

        # try stopping simpana services
        galaxy_stopper="${GALAXY}/Base/`basename ${GALAXY}`"
        if [ -f $galaxy_stopper ]; then
            $galaxy_stopper stop 2> /dev/null
            if [ $? -ne 0 ]; then
                Log "Failed to stop galaxy services. Cannot continue." >> $LOG_FILE
                reset_last_error
                log_last_error "Failed to stop galaxy services. Cannot continue."
            fi
        fi
    fi

    # delete files
    rm -rf /opt/* &> /dev/null
    rm -rf /var/log/* &> /dev/null
    rm -rf /etc/CommVaultRegistry/ &> /dev/null

    #unmount the tmpfs file systems
    umount /opt 2> /dev/null
    umount /var/log 2>/dev/null
}

load_usb()
{
    test `cat /proc/modules | awk '{print $1}' | grep ehci-hcd` && modprobe ehci-hcd
    test `cat /proc/modules | awk '{print $1}' | grep ohci-hcd` && modprobe ohci-hcd
    test `cat /proc/modules | awk '{print $1}' | grep uhci-hcd` && modprobe uhci-hcd
    test `cat /proc/modules | awk '{print $1}' | grep usbhid` && modprobe usbhid
    test `cat /proc/modules | awk '{print $1}' | grep serio_raw` && modprobe serio_raw
    test `cat /proc/modules | awk '{print $1}' | grep hpilo` && modprobe hpilo
}

_XLIST="binfmt_misc xfs libcrc32c fcoe libfcoe libfc ebtable_nat ebtable_broute bridge stp llc ebtable_filter ebtables xprtrdma ib_iser ib_srp scsi_transport_srp ib_ipoib rdma_ucm ib_ucm ib_uverbs ib_umad rdma_cm ib_cm iw_cm ocrdma mlx4_ib ib_sa ib_mad ib_core ib_addr i2c_i801 i2c_core sg mei_me pcspkr mei lpc_ich shpchp mfd_core ext4 mbcache jbd2 loop nls_utf8 isofs mlx4_en sr_mod cdrom sd_mod crc32_pclmul crc32c_intel ghash_clmulni_intel aesni_intel glue_helper ablk_helper cryptd lpfc crc_t10dif crct10dif_generic be2net crct10dif_pclmul mlx4_core scsi_transport_fc vxlan scsi_tgt ip6_udp_tunnel crct10dif_common udp_tunnel ahci libahci libata tg3 megaraid_sas ptp pps_core wmi usb_storage sunrpc xts lrw gf128mul dm_crypt dm_round_robin dm_snapshot dm_bufio dm_mirror dm_region_hash dm_log dm_zero linear raid10 raid456 async_raid6_recov async_memcpy async_pq raid6_pq async_xor xor async_tx raid1 raid0 iscsi_ibft iscsi_boot_sysfs iscsi_tcp libiscsi_tcp libiscsi squashfs cramfs scsi_transport_iscsi edd dm_multipath dm_mod"

load_additional_modules()
{
    for m in $_XLIST;
    do
        modprobe $m
    done
}

create_virtual_terminals()
{
    for(( i=0; i<9; i++ ))
    do
          if [ ! -e /dev/tty${i} ]; then     
               mknod /dev/tty${i} c 4 ${i} 
          fi
    done
}

mount_proc()
{
    /bin/echo "mounting proc...."
    Log "mounting proc...." >> $LOG_FILE
    /bin/mount -t proc none /proc 
    if [ ! -d /sys ]; then  
         /bin/echo "mounting sys...."
         Log "mounting sys...." >> $LOG_FILE
         mkdir /sys
         mount /sys/ /sys -v -t sysfs 
    fi       
    umount /dev/pts
    determine_restore_stage
    if [ $RESTORE_STAGE -eq 2 ]; then
        rm -rf /dev/pts/*
        rm -rf /dev/ptmx
    fi
    mknod /dev/ptmx c 5 2
    mount -t devpts devpts /dev/pts -o rw,gid=5,mode=666,ptmxmode=666
    if [ -e /dev/pts/ptmx ]; then
        rm /dev/ptmx
        chmod 666 /dev/pts/ptmx
        cd /dev
        ln -s pts/ptmx /dev/ptmx
    fi
    chmod a+rw /dev/pts/*
    cd -
}

open_virtual_terminals()
{
    #open the terminals.
    if [ $RESTART_LEVEL  -lt 30 ]; then
         openvt -c 2 /bin/bash
         RESTART_LEVEL=30
         echo $RESTART_LEVEL>/tmp/restart_level
         Log "RESTART_LEVEL:$RESTART_LEVEL Opened first VT succesfully" >> $LOG_FILE
    fi
    
    if [ $RESTART_LEVEL  -lt 40 ]; then
         openvt -c 3 /bin/bash
         RESTART_LEVEL=40
         echo $RESTART_LEVEL>/tmp/restart_level
         Log "RESTART_LEVEL:$RESTART_LEVEL Opened second VT succesfully" >> $LOG_FILE
    fi   
    
    if [ $RESTART_LEVEL  -lt 50 ]; then
         openvt -c 4 /bin/bash
         RESTART_LEVEL=50
         echo $RESTART_LEVEL>/tmp/restart_level
         Log "RESTART_LEVEL:$RESTART_LEVEL Opened third VT succesfully" >> $LOG_FILE
    fi   
    
    if [ $RESTART_LEVEL  -lt 60 ]; then
         openvt -c 5 /bin/bash
         RESTART_LEVEL=60
         echo $RESTART_LEVEL>/tmp/restart_level
         Log "RESTART_LEVEL:$RESTART_LEVEL Opened fourth VT succesfully" >> $LOG_FILE
    fi   
    
    if [ $RESTART_LEVEL  -lt 70 ]; then
         openvt -c 6 -- tail -f $LOG_FILE
         RESTART_LEVEL=70
         echo $RESTART_LEVEL>/tmp/restart_level
         Log "RESTART_LEVEL:$RESTART_LEVEL Opened fifth VT succesfully" >> $LOG_FILE
    fi
    
    if [ $RESTART_LEVEL  -lt 80 ]; then
         openvt -c 7 -- tail -f $DMESG_OUTPUT 
         RESTART_LEVEL=80
         echo $RESTART_LEVEL>/tmp/restart_level
         Log "RESTART_LEVEL:$RESTART_LEVEL Opened sixth VT succesfully" >> $LOG_FILE
    fi   
}

create_disk_nodes()
{
    DISK_TREE=$1
    ide_list=`ls $DISK_TREE/ide`
    scsi_list=`ls $DISK_TREE/scsi`
    cciss_list=`ls $DISK_TREE/cciss`
    if [ x"$ide_list" != x ]; then
         for ide_disk in $ide_list
         do
             MAJOR_NO=`cat $DISK_TREE/ide/$ide_disk/major`
             MINOR_NO=`cat $DISK_TREE/ide/$ide_disk/minor` 
             if [ ! -e /dev/$ide_disk  ]; then  
                  /bin/mknod /dev/$ide_disk b $MAJOR_NO $MINOR_NO
             fi       
         done
    fi

    if [ x"$scsi_list" != x ]; then
         for scsi_disk in $scsi_list
         do
             MAJOR_NO=`cat $DISK_TREE/scsi/$scsi_disk/major`
             MINOR_NO=`cat $DISK_TREE/scsi/$scsi_disk/minor` 
             if [ ! -e /dev/$scsi_disk ]; then
                  /bin/mknod /dev/$scsi_disk b $MAJOR_NO $MINOR_NO
             fi       
         done
    fi

    if [ x"$cciss_list" != x ]; then
         mkdir /dev/cciss
         for cciss_disk in $cciss_list
         do
             MAJOR_NO=`cat $DISK_TREE/cciss/$cciss_disk/major`
             MINOR_NO=`cat $DISK_TREE/cciss/$cciss_disk/minor` 
             if [ ! -e /dev/cciss/$cciss_disk ]; then
                  /bin/mknod /dev/cciss/$cciss_disk b $MAJOR_NO $MINOR_NO
             fi       
         done
    fi
}

check_cd_exists()
{
    device_list=$1  
    for cd in $device_list
    do
        for (( i=0; i<3; i++ ))
        do
            dd if=$cd bs=1 skip=32808 count=32
            if [ $? -eq 0 ]; then
                 break
            fi
        done

        label=`dd if=$cd bs=1 skip=32808 count=32 | grep ONETOUCH`
        if [ x"$label" = x ]; then
             Log "$cd is not a CD/DVD with label ONETOUCH" >> $LOG_FILE
        else 
             #The CV/DVD is the correct 1-Touch restore CD. 
             mount -t iso9660 $cd /cdrom
             if [ $? -eq 0 ]; then
                  Log "Succesfully mounted $cd under /cdrom" >> $LOG_FILE
                  return 0
             else 
                  Log "The disk $cd contained the ONETOUCH label. But mount on it failed" >> $LOG_FILE 
             fi
        fi     
    done
    return 1
}

get_field_value()
{
    cat $1 | while read line
    do
        key=`echo $line | cut -d'=' -f1`
        value=`echo $line | cut -d'=' -f2-`
        if [ "$key" = "$2" ]; then
             echo $value
             Log "Extracted [$2] = [$value] from [$1]" >> $LOG_FILE
             break
        fi
    done
    echo""
}

get_ping()
{
    echo "ping"
}

check_connectivity()
{
    PING=`get_ping`
    $PING -c 5 $1  > /dev/null
    if [ $? -ne 0 ]; then
         echo "Unable to $PING $1 " > $LASTERROR
         echo "Unable to $PING $1 " >> $LOG_FILE
         return 1
    fi
    return 0
}

get_interface_list()
{
    retlist=""
    prelist=`ls -I "lo" /sys/class/net/ | egrep '(^eth[0-9]*)' | sort -k1.4 -n`
    prelist+=" "
    prelist+=`ls -I "lo" /sys/class/net/ | egrep '(^em[0-9]*)' | sort -k1.3 -n`
    prelist+=" "
    prelist+=`ls -I "lo" /sys/class/net/ | egrep '(^p[0-9]p[0-9])' | sort -k1.1 -n`
    prelist+=" "
    prelist+=`ls -I "lo" /sys/class/net/ | egrep '(^eno[0-9]*)' | sort -k1.4 -n`
    prelist+=" "
    prelist+=`ls -I "lo" /sys/class/net/ | egrep '(^enp[0-9]*)' | sort -k1.4 -n`
    prelist+=" "
    prelist+=`ls -I "lo" /sys/class/net/ | egrep '(^ens[0-9]*[a-z]*[0-9]*)' | sort -k1.4 -n`
    prelist+=" "
    prelist+=`ls -I "lo" /sys/class/net/ | egrep '(^enp[0-9]*[a-z]*[0-9]*)' | sort -k1.4 -n`
    #FIXME: currently we aren't filtering out various virtual networks - we need to filter prelist here
    for intf in $prelist;
    do
        if [[ $intf != *"vnet"* ]]
        then
            retlist="${retlist} ${intf}"
        fi
    done
    echo $retlist
}

create_netconfig_template()
{
    eth_list=`get_interface_list`
    for eth in $eth_list; do
        if [ ! -e ${YASTGUI}/${eth} ]; then
             mac=`cat /sys/class/net/$eth/address`
             [[ ! -z "$mac" ]] && echo "MAC=$mac" > ${YASTGUI}/${eth}                
        fi
    done
}

get_os_type()
{
    ans=`uname -a | grep "x86_64"`
    if [ -n "$ans" ]; then
         echo "64"
    else
         echo "32"
    fi      
}

#call this function after the restore to unmount the mounted file systems.
UnmountRestoredDevs ()
{
    cd /
    if [ -e /tmp/unmountlist ]; then
        rm -f /tmp/unmountlist
    fi
    touch /tmp/unmountlist
    for mnt_dir in `mount -l | grep "MOUNTDIR" | awk '{print $3}'`
    do
        echo     $mnt_dir >> /tmp/unmountlist
    done
    tac /tmp/unmountlist | while read mnt_point
    do
        Log "Unmounting  $mnt_point" >> $LOG_FILE
        umount $mnt_point
    done

    #stop volume groups
    vgchange -an 2> /dev/null
    
}

get_job_state()
{
    job_id=$1

    JOB_FLAG=1
    job_status=" "
    ${GALAXY}/Base/qlist job -j $job_id > /tmp/job_details.$$
    if [ $? -ne 0 ]; then
         echo "unknown"
         return
    fi

    LINE=1
    cat /tmp/job_details.$$ | while read line
    do
        if [ $LINE -eq 3 ]; then
             job_status=`echo $line`
             echo "$job_status" > /tmp/job_status.$$
             break
        fi
        LINE=$((LINE+1))
    done
        
    if [ -e /tmp/job_status.$$ ]; then
        job_status=`cat /tmp/job_status | awk '{print $3}'`
        echo "$job_status"
    else
        echo "unknown"
    fi
    return
}

#Search for the mount point in the mount point
#Configuration file.
SearchMountPoint()
{
    local column1 column2 column3 column4
    local mnt_to_search 

    mnt_to_search=$1
    echo "$mnt_to_search" > /tmp/mnt.$$
    if [ -e /tmp/found.$$ ]; then
         rm -f /tmp/found.$$
    fi
    cat $SYS_CONFIG_DIR/mountpoints.lst | while read line
    do
        column1=`echo $line | awk '{print $1}'`         
        column2=`echo $line | awk '{print $2}'`         
        column3=`echo $line | awk '{print $3}'`         
        column4=`echo $line | awk '{print $4}'`         
        mnt_to_search=`cat /tmp/mnt.$$`
        if [ "$mnt_to_search" == "$column2" ]; then
             touch  /tmp/found.$$
             echo "$column1 $column2 $column3 $column4" 
             break
        fi 
    done
    if [ -e /tmp/found.$$ ]; then
         return 0;
    else 
         return 1;
    fi
}

setup_ycp_chroot()
{
    mkdir -p /chroot
    mkdir -p /chroot/bin
    mkdir -p /chroot/usr
    mkdir -p /chroot/sbin
    mkdir -p /chroot/usr
    mkdir -p /chroot/var /chroot/var/lib /chroot /var/run
    mkdir -p /chroot/ylibs
    mkdir -p /chroot/proc
    mkdir -p /chroot/sys
    mkdir -p /chroot/tmp
    mkdir -p /chroot/python_dir
    mkdir -p /chroot/system_state

    mv /tmp /tmp_tmp
    mv /system_state /system_state_tmp

    mkdir /tmp
    mkdir /python_dir
    mkdir /system_state

    mount -t tmpfs tmpfs /tmp
    mount -t tmpfs tmpfs /python_dir
    mount -t tmpfs tmpfs /system_state

    cp -r /tmp_tmp/* /tmp/
    cp -r /system_state_tmp/* /system_state/
    rm -rf /tmp_tmp
    rm -rf /system_state_tmp

    cp /cdrom/ycp/*.ycp /chroot/
    cp -a /cdrom/stage1scripts/* /chroot/
    cp /launchyast.sh /chroot/
    cp /cdrom/chrootcmd /chroot/
    cp /common.sh /chroot

	cp /unified_yast_autofill /chroot

    ver=`ldd --version | head -n 1 | awk '{print $4}' | cut -d'-' -f1`
    maj=`echo $ver | cut -d'.' -f1`
    min=`echo $ver | cut -d'.' -f2`
    if [ $maj -le 2 -a $min -lt 6 ]; then
        #use legacy mode
        echo "Using legacy mode chroot environment" >> $LOG_FILE
        cd /chroot
        rm -rf bin usr lib sbin ylibs
        echo "Uncompress /cdrom/simpana_dump/legacy_chroot.tgz" >> $LOG_FILE
        tar zxf /cdrom/simpana_dump/legacy_chroot.tgz
        echo "done - Uncompress /cdrom/simpana_dump/legacy_chroot.tgz" >> $LOG_FILE
        mv tmp/cdimage/* .
        rm -rf tmp
        mkdir tmp proc sys python_dir system_state
        # Copy python from cdrom
        if [ -e /cdrom/simpana_dump/python2.7_legacy.tgz ]; then
            cd usr
            tar zxf /cdrom/simpana_dump/python2.7_legacy.tgz
            echo "done - Uncompress /cdrom/simpana_dump/python2.7_legacy.tgz" >> $LOG_FILE
        else
            echo "Not Found! : /cdrom/simpana_dump/python2.7_legacy.tgz" >> $LOG_FILE
        fi
        touch /tmp/legacy_chroot_flag
        cd /
    else
        echo "Using regular chroot environment" >> $LOG_FILE
        RESTORE_MODE=""
        determine_restore_mode
        if [ -n "$RESTORE_MODE" ]; then
            mount --bind /cdrom/bin   /chroot/bin
            echo "Copying /usr from boot media..."
            cp -a /cdrom/usr   /chroot/
            echo "Copying /lib from boot media..."
            cp -a /cdrom/lib   /chroot/
            mount --bind /cdrom/sbin  /chroot/sbin
            mount --bind /cdrom/ylibs /chroot/ylibs
            cp -a /dev /chroot
            rm -f /chroot/dev/log
            mkdir -p /chroot/var/log /chroot/var/run /chroot/root
            touch /chroot/var/log/wtmp /chroot/var/log/utmp /chroot/var/log/lastlog
            cp -a /cdrom/stage1scripts/base64 /
            if [ -e /cdrom/stage1scripts/jm_notify64 ]; then
                cp -a /cdrom/stage1scripts/jm_notify64 /jm_notify
            fi
            if [ -e /cdrom/stage1scripts/send_logs64 ]; then
                cp -a /cdrom/stage1scripts/send_logs64 /send_logs
            fi
            if [ -e "/cdrom/python2.7.tgz" ]; then
                cp -a /cdrom/python2.7.tgz /chroot/
                cd /chroot
                mv usr/lib/python usr/lib/python.orig
                tar zxf python2.7.tgz
                mv usr/lib/python.orig/* usr/lib/python2.7
                rm -rf usr/lib/python.orig
                cd -
            fi
            cp -a /cdrom/etc /chroot
        else
            mount --bind /cdrom/bin   /chroot/bin
            mount --bind /cdrom/usr   /chroot/usr
            mkdir -p /chroot/lib; mount --bind /cdrom/lib   /chroot/lib
            mount --bind /cdrom/sbin  /chroot/sbin
            mount --bind /cdrom/ylibs /chroot/ylibs
            cp -a /etc /chroot/
        fi
    fi
    mount --bind /proc        /chroot/proc
    mount --bind /sys         /chroot/sys
    mount --bind /tmp         /chroot/tmp
    mount --bind /python_dir  /chroot/python_dir/
    mount --bind /system_state  /chroot/system_state/

    mkdir -p /chroot/dev
    cp -a /dev/console /chroot/dev/
    cp -a /dev/null /chroot/dev/
    cp -a /dev/urandom /chroot/dev
    cp -a /dev/pt* /chroot/dev
    cp -a /dev/pt* /chroot/tmp
}

#Search for the mount point in the mount point
#Configuration file.
SearchMountPoint()
{
    local column1 column2 column3 column4
    local mnt_to_search 

    mnt_to_search=$1
    echo "$mnt_to_search" > /tmp/mnt.$$
    if [ -e /tmp/found.$$ ]; then
         rm -f /tmp/found.$$
    fi
    cat $SYS_CONFIG_DIR/mountpoints.lst | while read line
    do
        column1=`echo $line | awk '{print $1}'`         
        column2=`echo $line | awk '{print $2}'`         
        column3=`echo $line | awk '{print $3}'`         
        column4=`echo $line | awk '{print $4}'`         
        mnt_to_search=`cat /tmp/mnt.$$`
        if [ "$mnt_to_search" == "$column2" ]; then
             touch  /tmp/found.$$
             echo "$column1 $column2 $column3 $column4" 
             break
        fi 
    done
    if [ -e /tmp/found.$$ ]; then
         return 0;
    else 
         return 1;
    fi
}

deinitialize_drivers()
{
    killall -9 hypervkvpd
    sleep 3

    EXCLUDE_LIST="hpilo
                rdma
                rpc
                wdt
                lockd
                serio
                hid
                hcd"
    
    FILE=/tmp/drivers
    rm -rf $FILE
    lsmod | awk '{print $1}' > $FILE
    sed -i '1d' $FILE
    cat $FILE | while read line
    do
        #consult exclusion list, but the exclusion list driver names could be substrings of actual driver names
        found=0
        for excld in $EXCLUDE_LIST
        do
            ret=`echo $line | grep $excld`
            if [ "x$ret" != "x" ]; then
                found=1
                break    
            fi
            if [ $found -eq 0 ]; then
                echo "unload $line" >> $LOG_FILE
                rmmod $line 2>/dev/null
            else
                echo "Not unloading $line, as it is present in exclude list" >> $LOG_FILE
            fi
        done
    done
}

vm_set_env_vmware()
{
    if [ ! -f /sbin/vmware-guestd ]; then
        if [ -f /usr/bin/vmware-guestd ]; then
            cp /usr/bin/vmware-guestd /sbin
        fi
        if [ -f /bin/vmware-guestd ]; then
            cp /bin/vmware-guestd /sbin
        fi
    fi
    if [ -x /sbin/vmware-guestd ]; then
        encresp=`/sbin/vmware-guestd --cmd 'machine.id.get' 2>/dev/null`
        if [ "x$encresp" != "x" -a "x$encresp" != "xNo machine id" ]; then
            echo "$encresp" > /tmp/encfile
            /base64 -d /tmp/encfile /tmp/respfile
            cp -f /tmp/respfile ${RESPONSE_FILE_PATH}
            echo "${RESPONSE_FILE_PATH}" > ${RESPONSE_LOCATION}
            Log "(vmware guest) Copied 1-Touch response file to ${RESPONSE_FILE_PATH}" >> $LOG_FILE
            val=`/sbin/vmware-guestd --cmd 'info-get guestinfo.provisioningJobId' 2>/dev/null`
            if [ "x$val" != "x" -a "x$val" != "xNo value found" -a "x$val" != "x0" ]; then
                echo $val > /tmp/recovery_jobid
                Log "(vmware guest) Read recovery job id : `cat /tmp/recovery_jobid`" >> $LOG_FILE
            else    
                Log "NO recovery job id found [$val]" >> $LOG_FILE
            fi
            return 0
        fi    
    fi
    return 1
}

vm_set_env_hyperv()
{
    if [ ! -d /var/lib/hyperv ]; then
        mkdir -p /var/lib/hyperv
    fi
    if [ -x /sbin/hypervkvpd ]; then
        /sbin/hypervkvpd 
        if [ -x /getKVP ]; then
            sleep 5
            cd /
            OLD_LD=$LD_LIBRARY_PATH
            export LD_LIBRARY_PATH=.:$LD_LIBRARY_PATH
            ./getKVP
            retcode=$?
            export LD_LIBRARY_PATH=$OLD_LD
            cd -
            if [ "${retcode}" == "0" ]; then
                Log "(HyperV guest) successfully got the response file data" >> $LOG_FILE
            else
                Log "(HyperV guest) Failed to get the enc response file data" >> $LOG_FILE
                return 1
            fi
        else
            Log "(HyperV guest) getKVP either not found or no execute perm" >> $LOG_FILE
            return 1
        fi
        touch ${YASTGUI}/VMeHyperVHookFile
        /bin/mount | grep "/cdrom" | awk '{print $1}' > ${YASTGUI}/VMeHyperVHookFile
        Log "(HyperV guest) CDROM device : `cat ${YASTGUI}/VMeHyperVHookFile`" >> $LOG_FILE
        /base64 -d /tmp/OneTouchEncRespFile /tmp/respfile
        cp -f /tmp/respfile ${RESPONSE_FILE_PATH}
        echo "${RESPONSE_FILE_PATH}" > ${RESPONSE_LOCATION}
        Log "(HyperV guest) Copied 1-Touch response file to ${RESPONSE_FILE_PATH}" >> $LOG_FILE
        if [ -f /tmp/OneTouchGuestinfoJobId ]; then
            cp -f /tmp/OneTouchGuestinfoJobId /tmp/recovery_jobid
            Log "(HyperV guest) Read recovery job id : `cat /tmp/recovery_jobid`" >> $LOG_FILE
        else
            Log "(HyperV guest) NO recovery job id found [$val]" >> $LOG_FILE
        fi
        return 0
    else
        Log "(HyperV guest) HyperV KVP daemon not found or no execute perm" >> $LOG_FILE
    fi
    return 1
}

vm_percent()
{
    parent=`cat /tmp/recovery_jobid`
    if [ "a$parent" != "a" ]; then
        cnorg=`cat /tmp/NETDETAILS | grep -w CLIENTNAME | awk '{print $2}'`
        cn="${cnorg}_recovery"
        vm=`cat ${GALAXY}/galaxy_vm | grep GALAXY_VM | cut -d'"' -f2`
        cs=`cat /tmp/NETDETAILS | grep CSNAME | awk '{print $2}'`
        if [ "a$2" != "a" ]; then
            determine_restore_stage
            determine_restore_mode
            if [ $RESTORE_STAGE -eq 2 -o -n "$RESTORE_MODE" ]; then
    			percent=$2
            else    
                percent=$1
            fi
        else
            percent=$1
        fi    

        if [ ! -e ${BASEDIR}/jm_notify ]; then
            cp /jm_notify ${BASEDIR}/
        fi
        OLDPWD=`pwd`
        cd $BASEDIR
        ./jm_notify -c $cn -v $vm -s $cs -j $parent -p $percent
        cd $OLDPWD
    fi
}

Log_sysconf() {
    fail=0
    echo " "   >> /tmp/systemconf.log
    echo " "   >> /tmp/systemconf.log
    echo "$1:" >> /tmp/systemconf.log
    echo "-------------------------------" >> /tmp/systemconf.log
    $* 2>&1 >> /tmp/systemconf.log
    if [ $? -ne 0 ]; then
        Log "Error: execution of '$*' failed"
    fi
    echo "-------------------------------" >> /tmp/systemconf.log
    echo " "   >> /tmp/systemconf.log
    echo " "   >> /tmp/systemconf.log
}

#Prepare the log to contain the system configuration.
SystemConfigLog()
{
    Log_sysconf "mount -l"
    Log_sysconf "sfdisk -df"
    Log_sysconf "ifconfig"
    Log_sysconf "route -n"
    Log_sysconf "cat /etc/fstab"
    if [ -f /etc/raidtab ]; then
        Log_sysconf "cat /etc/raidtab" 
    fi
    Log_sysconf "vgs"
    Log_sysconf "pvs"
    Log_sysconf "vgdisplay"
    Log_sysconf "pvdisplay"
    Log_sysconf "vgs -o vg_name,vg_attr"
    vg_unexported_list=`vgs --noheadings -o vg_name,vg_attr | awk '{if (substr($2, 3,1)!="x") print $1}'`
    Log_sysconf "lvdisplay $vg_unexported_list" 
    Log_sysconf "mount -t nfs"
    if [ -d "/proc/lvm" ]; then
        Log_sysconf "ls -l /proc/lvm" 
    fi
    if [ -d "/dev/mapper" ]; then
        Log_sysconf "ls -l /dev/mapper" 
    fi
    if [ -d "/dev/md*" ]; then
        Log_sysconf "ls -l /dev/md*" 
    fi
    if [ -d "/proc/lvm/VGs/$volume_group/PVs" ]; then
        Log_sysconf "ls -l /proc/lvm/VGs/$volume_group/PVs" 
    fi
    Log_sysconf "lsmod" 

    if [[ "$isRH" != "" ]] || [[ "$isSUSE" != "" ]]; then
        Log_sysconf "rpm -qa" 
    fi

    Galaxy_Bin=`find /etc/ | grep 'rc..d/S[0-9]\+Galaxy' | tail -1`
    Log_sysconf "$Galaxy_Bin status" 
 
    Log_sysconf "lspci -v" 
}

send_logs_wrapper()
{
    mkdir /collectedlogs
    cd /collectedlogs
    rm -rf /tmp/systemconf.log
    SystemConfigLog
    tar --exclude "/tmp/MOUNTDIR" --exclude "/tmp/yastgui/nfsmount" -zcvf tmp.tgz /tmp
    if [ -d /system_state ]; then
        tar -zcvf system_state.tgz /system_state
    fi        
    if [ -d /etc/CommVaultRegistry ]; then
        tar -zcvf registry.tgz /etc/CommVaultRegistry
    fi
    if [ -d /python_dir ]; then
        tar -zcvf python_dir.tgz /python_dir
    fi        
    file="/etc/CommVaultRegistry/Galaxy/Instance001/EventManager/.properties"
    ldir=`get_registry_value $file "dEVLOGDIR"`
    if [ -d $ldir ]; then
        tar -zcvf simpanalogs.tgz $ldir
    fi   
    if [ -d $BASEDIR/../iDataAgent/jobResults/CV_JobResults ]; then
        tar -zcvf jobResults.tgz $BASEDIR/../iDataAgent/jobResults/CV_JobResults
    fi
    filelist=""    
    if [ -f /var/log/messages ]; then
        filelist="$filelist /var/log/messages"
    fi
    if [ -f /var/log/YaST2/y2log ]; then
        filelist="$filelist /var/log/YaST2"
    fi
    if [ -f /chroot/var/log/YaST2/y2log ]; then
        filelist="$filelist /chroot/var/log"
    fi
    if [ -n "$filelist" ]; then
        tar -zcvf var.tgz $filelist
    fi
    cd -
    logfile="1touchclientlogs-`date +%H-%M_%m-%d-%y`.tgz"
    tar zcvf /tmp/$logfile /collectedlogs
    if [ $? -ne 0 ]; then
        Log "Error while creating log tar file, file may be corrupted!!!" >> $LOG_FILE
    else
        echo "" >> $LOG_FILE
        echo "Collected logs and saved to [/tmp/$logfile]" >> $LOG_FILE
        echo "" >> $LOG_FILE

    fi    
    csname=`grep CSDISPLAYNAME $CFG_FILE 2>/dev/null | awk '{print $2}'`
    cshost=`grep CSNAME $CFG_FILE 2>/dev/null | awk '{print $2}'`
    clientnam_src=`grep -w CLIENTNAME $CFG_FILE 2>/dev/null | awk '{print $2}'`
    clientname="${clientnam_src}_recovery"
    if [ "a`which pidof 2>/dev/null`" != "a" ]; then
        PIDOF="pidof"
    else
        PIDOF="pidofproc"
    fi
    cvdpid=`$PIDOF cvd`
    if [ -n "$csname" -a -n "$cshost" -a -n "$clientname" -a -n "$cvdpid" ]; then
        cplogfile="1touchclientlogs-$clientname.tgz"
        rm -f /tmp/$cplogfile
        cp /tmp/$logfile /tmp/$cplogfile
        cd ${BASEDIR}
        if [ ! -e ./send_logs ]; then
            cp /send_logs . 
        fi
        ./send_logs -c $clientname -s $csname -h $cshost -p /tmp -f $cplogfile
        if [ $? -ne 0 ]; then
            Log "Operation failed! : /send_logs -c $clientname -s $csname -h $cshost -p /tmp -f $cplogfile" >> $LOG_FILE
        fi
        cd -
        Log "Operation succeeded : /send_logs -c $clientname -s $csname -h $cshost -p /tmp -f $cplogfile" >> $LOG_FILE
        rm -f /tmp/$cplogfile
    else    
        Log "one of the fields empty. skip send logs :[$csname][$cshost][$clientname][$cvdport]" >> $LOG_FILE
    fi    

    rm -rf /collectedlogs
#    echo "...sending logs to CS currently disabled. Please copy [/tmp/$logfile] manually!!!" >>$LOG_FILE
    setup_weburl
    mkdir -p /var/www/older-logs
    mv /var/www/* /var/www/older-logs
    [ $(ls -A /var/www/older-logs) ] || rm -rf /var/www/older-logs              
    cp -f /tmp/$logfile /var/www
    echo "<a href=\"$logfile\" download>Download Logs</a>" > /var/www/index.html
}

process_fwconfigblob()
{
    fwfile=$1
    orig_client=$2
    new_client=$3

    determine_restore_stage
    if [ $RESTORE_STAGE  -eq 1 ]; then
        if [ "x$orig_client" != "x$new_client" ]; then
            echo "Modifying [$orig_client] to [$new_client] in file [$fwfile]" >> $LOG_FILE
            if [ ! -f $fwfile.orig ]
            then
                #if someone is creating this file first time. This will help us to preserver original file if someone else use it second time by mistake.
                cp $fwfile $fwfile.orig
            fi
            sed -i 's/$orig_client/$new_client/g' $fwfile
            echo "Updated file [`cat $fwfile`]" >> $LOG_FILE
        else
            echo "Backup and restore client name is same, skip process_fwconfigblob [$orig_client]" >> $LOG_FILE
          
        fi
        
    else
        echo "Stage2, skip process_fwconfigblob" >> $LOG_FILE
    fi
}

get_valid_ip_from_nics()
{
   def_if=`ip route show default | awk '{ if ($1 == "default") { print $5} }'`
   if [ -n "$def_if" ]; then
       ipaddr=`ifconfig $def_if | grep "inet " | awk '{ print $2 }'`
   else
       ipaddr=`ifconfig | grep "inet " | awk '{ print $2 }'| head -n 1`
   fi
   if [ -n "$ipaddr" ]; then
       echo $ipaddr
   fi
}

setup_weburl()
{
    HNAME=$1
    #do we have pidof
    if [ "a`which pidof 2>/dev/null`" != "a" ]; then
        PIDOF="pidof"
    else
        PIDOF="pidofproc"
    fi
    $PIDOF boa
    #try to start once more
    if [ $? -eq 0 ]; then
        sed -i 's/Group nogroup/Group root/g' /boa.conf
        boa -c /
        sleep 3
    fi
    $PIDOF boa
    if [ $? -eq 0 ]; then
        echo "set http://$HNAME/ as weburl" >> $LOG_FILE
        echo "http://$HNAME/"> $YASTGUI/weburl
        echo "http://$HNAME:8888/webadm"> $YASTGUI/webadmurl
    else
        echo "boa is not running. remove weburl" >> $LOG_FILE
        rm -f $YASTGUI/weburl
        rm -f $YASTGUI/webadmurl
    fi
}

append_hosts()
{
    # Expects one string as input (mind-- only one string).
    grep -xFq "$@" /etc/hosts
    [ $? -ne 0 ] && echo "$@" >> /etc/hosts
}

REDHAT="redhat"
SUSE="suse"
UBUNTU="ubuntu"
get_distribution()
{
    distribution=""
    if [ -e /tmp/MOUNTDIR/etc/ ]; then
        str=`cat /tmp/MOUNTDIR/etc/*release | grep -i "SUSE"`
        if [ "x$str" != "x" ]; then
            distribution=$SUSE
        else
            str=`cat /tmp/MOUNTDIR/etc/*release | egrep -i 'ubuntu|debian'`
            if [ "x$str" != "x" ]; then
                distribution=$UBUNTU
            else    
                distribution=$REDHAT
            fi
        fi
    else
        distribution=`cat $SYS_STATE_LOC/distribution`
    fi
}

prepmount_cmd()
{
    if [ -d /tmp/MOUNTDIR ]; then
        mkdir -p /tmp/MOUNTDIR/proc /tmp/MOUNTDIR/sys /tmp/MOUNTDIR/dev
        mount --bind /proc /tmp/MOUNTDIR/proc
        mount --bind /sys /tmp/MOUNTDIR/sys
        mount --bind /dev /tmp/MOUNTDIR/dev
        if [ $? -ne 0 ]; then
            mount -t devtmpfs devtmpfs /tmp/MOUNTDIR/dev
            if [ $? -ne 0 ]; then
                cp -a /dev /tmp/MOUNTDIR
            fi
        fi
    fi
}
prepumount_cmd()
{
    if [ -d /tmp/MOUNTDIR ]; then
        umount /tmp/MOUNTDIR/sys
        umount /tmp/MOUNTDIR/proc
        umount /tmp/MOUNTDIR/dev
    fi
}
get_grub_version()
{
    prepmount_cmd
    grubprg=`chroot /tmp/MOUNTDIR which grub-probe`
    if [ $? -ne 0 ]; then
        grubprg=`chroot /tmp/MOUNTDIR which grub2-probe`
    fi
    if [ -n "$grubprg" -a -x "/tmp/MOUNTDIR/$grubprg" ]; then
        bin=`echo $grubprg | awk -F"/" '{print $NF}'`
        ver=`chroot /tmp/MOUNTDIR $bin --version`
    else
        ver=`chroot /tmp/MOUNTDIR grub --version`
    fi
    echo $ver | grep "0.9"
    if [ $? -ne 0 ]; then
        ver="2" ##LATER, check for exact versions
    else
        ver="1"
    fi
    prepumount_cmd
}

start_udev()
{
     olddir=`pwd`
     cd /
     [[ -f /udev.tgz ]] && tar zxvf /udev.tgz
     grep --quiet tmpfs /proc/filesystems
     if [ $? -eq 0 ]; then
          Log "mounting tmpfs on /run"
          mkdir /run
          /bin/mount -t tmpfs tmpfs /run
          if [ $? -eq 0 ]; then
               Log "mount succeeded for /run"
          fi
     fi
     killall -9 systemd-udevd      #required for stage-2 but harmless for stage-1; udev might have been started w/o proper environemnt
     killall -9 udevd
     sleep 5

     echo "Will create udev entries"
     Log "Will create udev entries" >> $LOG_FILE
     if [ -x /lib/systemd/systemd-udevd ]; then
          /lib/systemd/systemd-udevd --daemon&
     elif [ -x /usr/lib/systemd/systemd-udevd ]; then
          /usr/lib/systemd/systemd-udevd --daemon&
     elif [ -x /sbin/udevd ]; then
          /sbin/udevd --daemon&
     fi
     sleep 10

     grep --quiet devtmpfs /proc/filesystems
     if [ $? -eq 0 ]; then
          Log "mounting devtmpfs on /dev"
          /bin/mount -t devtmpfs devtmpfs /dev
          if [ $? -eq 0 ]; then
               mkdir /dev/pts
          fi
     fi

	 echo "Creating udev devices"
	 Log "Creating udev devices" >> $LOG_FILE

     if [ -x /sbin/udevtrigger ]; then
          /sbin/udevtrigger
          sleep 10
     elif [ -x /sbin/udevadm ]; then
          /sbin/udevadm trigger
          sleep 10
     fi     
     Log "Started the udevtrigger" >> $LOG_FILE

     if [ -x /usr/lib/systemd/systemd-journald ]; then
          echo "Will start the journal daemon"
          Log "Will start the journal daemon"
          /usr/lib/systemd/systemd-journald &
          sleep 10
     fi

     echo "Will call udevsettle with timeout=30 seconds for the udev entries to be populated"
     Log "Will call udevsettle with timeout=30 seconds for the udev entries to be populated" >> $LOG_FILE
     if [ -x /sbin/udevsettle ]; then
          /sbin/udevsettle --timeout=30
     elif [ -x /sbin/udevadm ]; then
          /sbin/udevadm settle --timeout=30
     fi

     cd $olddir
}

#during backup, some libs may be present in simpana Base directory; move them to /lib
#but before that, if we have saved OS specific libraries on the DVD, let's use those instead. these are more likely the ones we really need. - this change is deemed temporary, 
copy_missing_system_libraries()
{
    cdir=`cat /system_state/sysconf/CLIENTSRDIR`
    mkdir -p $cdir
    bdir="$cdir/../../Base"
    if [ -d "$bdir" ]; then
        if [ "a`uname -i`" == "ax86_64" ]; then
            suffix="64"
        else
            suffix=""
        fi

        files=`find /lib$suffix /usr/lib$suffix -name "libgcc_*so*"`
        if [ -z "$files" ]; then
            cp -avf $bdir/libgcc_* /lib$suffix/ /usr/lib$suffix/*
        fi
        files=`find /lib$suffix /usr/lib$suffix -name "libstdc*so*"`
        if [ -z "$files" ]; then
            cp -avf $bdir/libstdc* /lib$suffix/ /usr/lib$suffix/*
        fi
   fi
}


generate_restore_xml()
{
    FILES_PATH=$1
    EXCLUDE_FILE=$2 #file containing list of exclude entries
    DESTDIR=$3
    RESTOREXML_FILENAME=$4
    EXTRARESTORE=$5

    Log "generate_restore_xml invoked with Files[$FILES_PATH] exclude [$EXCLUDE_FILE] destdir [$DESTDIR] output xml[$RESTOREXML_FILENAME]" >> $LOG_FILE

    CLIENT_NAME=$(cat $CFG_FILE | grep -w "CLIENTNAME" | awk '{print $2}')
    #NEWCLIENT_NAME=`cat $CFG_FILE | grep -w "NEWCLIENTNAME" | awk '{print $2}'`
    NEWCLIENT_NAME="${CLIENT_NAME}_recovery"
    MA_NAME=`cat $CFG_FILE | grep -w "MA" | awk '{print $2}'`
    if [ "x$MA_NAME" != "x" ]
    then
        MA_NAME="-m $MA_NAME"
    fi
    COPYPREC=`cat $COPYPRECFILE 2>/dev/null`
    if [ "x$COPYPREC" == "x" ]; then
        COPYPREC=-1
    fi

    if [ -e $YASTGUI/backupsetname ]; then
        bsn=`cat $YASTGUI/backupsetname`
        echo "Extract backupsetname[$bsn] from [$YASTGUI/backupsetname]" >> $LOG_FILE
    fi
    if [ -e "$bsn" ]; then
        bsn="defaultBackupSet"
    fi    

    if [ "$EXCLUDE_FILE" == "-" ]; then
        EXARG=""
    else
        EXARG="-x $EXCLUDE_FILE"
    fi

    determine_restore_stage
    if [ $RESTORE_STAGE -eq 2 ]; then
        STG="-n 1"
    else
        STG="-n 0"
    fi

    # this restore is out of band with the VirtualizeME child jobs.
    if [ -f /tmp/recovery_jobid -a $EXTRARESTORE != "-" ]; then
        mv /tmp/recovery_jobid /tmp/recovery_jobid.saved
    fi
    if [ ! -e $BASEDIR/createqcmd_xml ]; then
        cp /createqcmd_xml $BASEDIR/
    fi
    OLDPWD=`pwd`
    cd $BASEDIR
    echo "Execute command : ./createqcmd_xml $STG -s $CLIENT_NAME -d $NEWCLIENT_NAME $MA_NAME -p $COPYPREC -i $FILES_PATH $EXARG -o $DESTDIR -b $bsn -t `cat /tmp/1touch_totime | tail -n 1`" >> $LOG_FILE
    ./createqcmd_xml $STG -s $CLIENT_NAME -d $NEWCLIENT_NAME $MA_NAME -p $COPYPREC -i $FILES_PATH $EXARG -o "$DESTDIR" -b "$bsn" -t "`cat /tmp/1touch_totime | tail -n 1`"
    if [ $? -eq 0 ]; then
        echo "created xml for sys state"
        mv /qcmd_xml $RESTOREXML_FILENAME
        cp -f $RESTOREXML_FILENAME /tmp
    else	
        echo "could not create xml! " >> $LOG_FILE
    fi	
    cd $OLDPWD

    if [ -f /tmp/recovery_jobid.saved -a $EXTRARESTORE != "-" ]; then
        mv /tmp/recovery_jobid.saved /tmp/recovery_jobid
    fi
}

#this method will restore a list of files specified in $1 to a directory specified in $2
#$1 can contain a space separated list of files, enclosed in ""
restore_files()
{
    FILES_PATH=$1
    DESTDIR=$2
    Log "restore_files: invoked with [`cat $FILES_PATH`] [$DESTDIR]" >> $LOG_FILE

    rm -f /tmp/restore_files_xml_error /tmp/restore_files_job_error

    if [ ! -d "$DESTDIR" ]; then
        mkdir -p $DESTDIR
    fi
    generate_restore_xml $FILES_PATH "-" "$DESTDIR" "/extra_restores.xml" "OOB"
    if [ ! -f "/extra_restores.xml" ]; then
        Log "Failed to generate xml for additional system file restores!" >> $LOG_FILE
        touch /tmp/restore_files_xml_error
        return 
    fi

    echo "Using XML extra_restores.xml to invoke restore" >> $LOG_FILE
    ${GALAXY}/Base/qoperation execute -af "/extra_restores.xml" > /tmp/qcmd_resp
    cat /tmp/qcmd_resp | grep "jobIds val" | head -n 1 | cut -d'"' -f2 > /tmp/EXTRA_RESTORE_JOBID

    JOBID=`cat /tmp/EXTRA_RESTORE_JOBID`
    if [ -z "$JOBID" ]; then
        touch /tmp/restore_files_job_error
        return
    fi

    rm /tmp/extra_restore_job_err
    wait_for_job_completion $JOBID
    if [ -f /tmp/extra_restore_job_err ]; then
        Log "additional system file restore job failed!" >> $LOG_FILE
        touch /tmp/restore_files_job_error
        return
    else
        Log "additional system file restore job completed" >> $LOG_FILE
    fi
}

#since this method also identifies and restores any softlink references to the restored files, all files in $1 need to be in the same directory e.g.  "/lib64/a.so and /lib64/b.so" is fine but "/lib64/a.so and /lib/b.so" is not
restore_files_and_links()
{
    FILES_PATH=$1
    DESTDIR=$2
    Log "restore_files_and_links: invoked with [`cat $FILES_PATH`] [$DESTDIR]" >> $LOG_FILE

    restore_files $FILES_PATH $DESTDIR
    if [ -f /tmp/restore_files_xml_error -o -f /tmp/restore_files_job_error ]; then
        return
    fi

    fname=`cat $FILES_PATH | head -n 1`
    bdir=`dirname $fname`
    #find softlinks if any    
    rm -f /tmp/links_needed
    find $DESTDIR -type l | xargs -n 1 readlink -f  >> /tmp/links_needed
    links=`cat /tmp/links_needed`
    if [ -n "$links" ]; then
        Log "Found symlinks [`cat /tmp/links_needed`] - processing them" >> $LOG_FILE
        rm -f /tmp/srcpaths
        for file in `cat /tmp/links_needed`;
        do
            sfile=${file#$DESTDIR} #filename without the destination path (prefix)
            fsfile="$bdir/$sfile"
            echo $fsfile >> /tmp/srcpaths

        done

        restore_files /tmp/srcpaths $DESTDIR
    fi
}

# Method: on SLES12, we might (incorrectly, but by default) have subdirectories mounted within /boot/grub2. these prevent restoring files to the actual /boot as is essential for methods
#         like grub2-install or grub2-mkconig
#         we'll move these files within to /boot
# ARGS : 1: fstab file, 2: flag to indicate {0: fix and unmount, 1: remount}
function fix_boot_multimount()
{
    FSTAB=$1
    MODE=$2

    Log "fix_boot_multimount: invoked with [$1][$2]" >> $LOG_FILE

    cat $FSTAB |grep /boot/grub > /tmp/bootcheck
    if [ x"cat /tmp/bootcheck" != "x" ]
    then
        while read line
        do
            bfolder=`echo $line|awk '{print $2}'`
            boption=`echo $line|awk '{print $4}'`
            if [ $MODE == "0" ]; then
                bdevice=`mount | grep -w "$bfolder" | awk '{print $1}'` #UUID/device on current system may be different than on backed up fstab
                mkdir -p /tmp/BOOTCORRECT
                rm -rf /tmp/BOOTCORRECT/*
                cp -rf /tmp/MOUNTDIR/$bfolder/* /tmp/BOOTCORRECT/.
                umount /tmp/MOUNTDIR/$bfolder
                cp -rf /tmp/BOOTCORRECT/* /tmp/MOUNTDIR/$bfolder/.
                echo "$bdevice" > /tmp/fix_boot_multimount_bootdevice
                echo "1" > /tmp/boot_multimount_fixed
            else
                bdevice=`cat /tmp/fix_boot_multimount_bootdevice`
                mount -o $boption $bdevice /tmp/MOUNTDIR/$bfolder
            fi
        done < /tmp/bootcheck
        rm -rf /tmp/BOOTCORRECT
    fi
}



#1-Touch will always be installed as first instance
REGISTRY=/etc/CommVaultRegistry/Galaxy/Instance001/Base/.properties
if [ -e $REGISTRY  ]; then
     GALAXY=`get_registry_value $REGISTRY "dGALAXYHOME"`
fi

#Determine the restore stage
determine_restore_stage

# OS_ARCH type
OS_ARCH=`uname -m`
UNAME=`uname -a`

#Determine platform type.
platform=32
str=`uname -a | grep "x86_64"`
if [ "x$str" != "x" ]; then
     platform=64
fi

#this has been removed to handle cases where diffs for PCI slots with no disk attached are ignored        
##Create minimal initrd image
#mkdir -p ${initrd_control_directory}
#touch ${minimal_initrd}

#Create the GRUB control directory
mkdir -p ${grub_control_directory}

if [ -e /tmp/exportfw_params ]; then
    . /tmp/exportfw_params
fi
