var Uploader, UploadBatchInfo, UploadFileInfo, FileProgressItem, FileError, UploadProgressItem;

window.Uploader = Uploader = (function() {
	function Uploader(options) {
		// Constants
		this.PARAM_ISFILE = "isFile";
		this.PARAM_FILENAME = 'fileName';
		this.PARAM_FILE_SIZE = "fileSize";
		this.PARAM_FILE_LAST_MODIFIED = "lastModifiedTime";
		this.PARAM_DEBUG = "debug";
		this.PARAM_STREAMING = "streamingProtocol";
		this.PARAM_RESUME_UPLOAD = "resumeUpload";

		// Event Constants
		this.EventEnum = {
			ALL_EVENTS : "All Events",
			UPLOAD_START : "Start",
			FILE_ADDED : "Add File",
			FILE_ADD_ERROR : "Error while adding file",
			READY_FOR_UPLOAD : "Ready for Upload",
			FILE_UPLOAD_STARTING : "File Upload Starting",
			FILE_UPLOAD_STARTED : "File Upload Started",
			FILE_UPLOAD_INPROGRESS : "File Upload In Progress",
			FILE_UPLOAD_PAUSED : "File Upload Paused",
			FILE_UPLOAD_PAUSE_ERROR : "Error while Pausing file upload",
			FILE_UPLOAD_RESUMED : "File Upload Resumed",
			FILE_UPLOAD_RESUME_ERROR : "Error while Resuming file upload",
			FILE_UPLOAD_CANCELED : "File Upload Canceled",
			FILE_UPLOAD_CANCEL_ERROR : "Error while canceling file upload",
			FILE_UPLOAD_ERROR : "File Upload Error",
			FILE_UPLOAD_COMPLETED : "File Upload Completed",
			FILE_REMOVED : "Remove File",
			FILE_REMOVE_ERROR : "Error while removing file",
			UPLOAD_END : "End",
			UPLOAD_PROGRESS : "Upload overall Progress"
		};

		this.options = options;
		this.defaults = {
			// Chunk size in bytes (MB is preffered)
			chunkSize : 1 * 1048576,
			// Max Simultaneous upload is 5, consider network if you want more than 3
			simultaneousUploads : 3,
			// Map of data that need to be sent to the handler
			query : {},
			// HTTP optional headers
			headers : {},
			// Handler URL
			target : '/',
			// True to test the upload, which will write all data to local disk
			debug : void false,
			// Maximum number of file allowed for upload
			maxFiles : void 0,
			// Minimum file size in bytes, 0K files will error out (PS: not folders)
			minFileSize : void 0,
			// Maximum file size in bytes
			maxFileSize : void 0,
			// Only matched file name will be allowed, this is not a Regular Expression
			matchFileName : void "",
			// Only matched file extension will be allowed, this is not a Regular Expression
			matchFileExtension : void "",
			// Destination folder, if you want upload to know about the destination folder
			destinationFolder : void "",
			// To resume upload from it was left out, True to start from previous end position
			resumeUpload : true,
			// Ask user about the action, either continue from where it was left or restart from start
			resumeOrRestartUpload : false,
			// True to check for duplicate files
			duplicateFile : false,
			// True to include folder during upload
			allowFolder : false,
			// True to include empty folder during upload
			allowEmptyFolder : false,
			
			encodeFileName : false,
			
			sendFileIdAsName : false
		};
		if (this.options == null) {
			this.options = {};
		}
		this.eventListeners = new HashMap();
		this.currentUploadMap = new HashMap();
		this.fileUploadMap = new HashMap();
		this.currentUploadCount = 0;
		this.uploadProgress = null;
		this.uploadBatch = new Array();
	}

	Uploader.prototype.isSupported = function() {
		var isStreamingEnabled = ((typeof File !== "undefined" && File !== null) &&
				(typeof Blob !== "undefined" && Blob !== null) &&
				(typeof FileList !== "undefined" && FileList !== null) && ((Blob.prototype.webkitSlice != null) ||
				(Blob.prototype.mozSlice != null) || (Blob.prototype.slice != null)));
		return isStreamingEnabled;
	};

	Uploader.prototype.getOptions = function(optionKey) {
		var opts, index, key;
		if (optionKey instanceof Array) {
			opts = new HashMap();
			for (index = 0; index < optionKey.length; index++) {
				key = optionKey[index];
				opts.put(key, this.getOptions(key));
			}
			return opts;
		} else {
			if (this.options != null && this.options[optionKey] != null) {
				return this.options[optionKey];
			} else if (this.defaults != null) {
				return this.defaults[optionKey];
			}
		}
	};

	Uploader.prototype.setOptions = function(optionKey, optionValue) {
		if (this.options != null) {
			this.options[optionKey] = optionValue;
			return true;
		}
		return false;
	};

	Uploader.prototype.getBatchOptions = function(optionKey, currentBatch) {
		if (typeof currentBatch !== "undefined" && currentBatch !== null) {
			if (currentBatch < this.uploadBatch.length) {
				var uploadBatchInfo = this.uploadBatch[currentBatch];
				if (typeof uploadBatchInfo !== "undefined" && uploadBatchInfo !== null) {
					return uploadBatchInfo.getBatchOptions(optionKey);
				}
			}
		}
		// If the index does not match, get the default value
		return this.defaults[optionKey];
	};

	Uploader.prototype.fireEvent = function(firedEvent, eventData) {
		if (this.eventListeners != null && this.eventListeners.size() > 0) {
			// Find if there are any specific event listeners, if so call them
			var specificCallBack = this.eventListeners.get(firedEvent);
			if (typeof specificCallBack !== "undefined" && specificCallBack !== null) {
				specificCallBack.call(this, eventData);
			}
			// Find if there are any all event listeners, if so call them
			var allEventCallBack = this.eventListeners.get(this.EventEnum.ALL_EVENTS);
			if (typeof allEventCallBack !== "undefined" && allEventCallBack !== null) {
				allEventCallBack.call(this, firedEvent, eventData);
			}
			// Find if there are any Progress event listeners, if so call them
			var progressEventCallBack = this.eventListeners.get(this.EventEnum.UPLOAD_PROGRESS);
			if (typeof progressEventCallBack !== "undefined" && progressEventCallBack !== null &&
					this.uploadProgress != null) {
				progressEventCallBack.call(this, this.EventEnum.UPLOAD_PROGRESS, this.uploadProgress);
			}
		}
	};

	Uploader.prototype.onEvent = function(eventEnum, callback) {
		if (typeof eventEnum !== "undefined" && eventEnum != null) {
			if (eventEnum instanceof Array) {
				for (var index = 0; index < eventEnum.length; index++) {
					this.eventListeners.put(eventEnum[index], callback);
				}
			} else {
				this.eventListeners.put(eventEnum, callback);
			}
		}
	};

	Uploader.prototype.isFolderUploadSupported = function(inputControl) {
		var isFolderUploadSupported = false;
		if (typeof inputControl === "undefined" || inputControl === null) {
			inputControl = document.createElement('input');
		}
		if ('webkitdirectory' in inputControl || 'mozdirectory' in inputControl || 'odirectory' in inputControl ||
				'msdirectory' in inputControl || 'directory' in inputControl) {
			isFolderUploadSupported = true;
		}
		return isFolderUploadSupported;
	};
	Uploader.prototype.assignBrowse = function(domNodes, isDirectory) {
		if (domNodes.length == null) {
			domNodes = [ domNodes ];
		}
		var inputControl = null;
		for (var index = 0; index < domNodes.length; index++) {
			var node = domNodes[index];
			if (node.tagName === 'INPUT' && node.type === 'file') {
				inputControl = node;
			} else {
				inputControl = document.createElement('input');
				inputControl.setAttribute('type', 'file');
				node.style.overflow = 'hidden';
				node.style.cursor = 'pointer';
				inputControl.style.position = 'absolute';
				inputControl.style.top = 0;
				inputControl.style.left = 0;
				inputControl.style.bottom = 0;
				inputControl.style.right = 0;
				inputControl.style.opacity = 0;
				inputControl.style.cursor = 'pointer';
				inputControl.style.width = '100%';
				inputControl.style.height = '30px';
				inputControl.style.fontSize = '0px';
				node.appendChild(inputControl);
			}
		}
		if (inputControl !== null) {
			var maxFiles = this.getOptions('maxFiles');
			var allowFolder = this.getOptions('allowFolder');
			if ((maxFiles != null) || maxFiles !== 1) {
				inputControl.setAttribute('multiple', 'multiple');
			} else {
				inputControl.removeAttribute('multiple');
			}
			if (isDirectory != null && isDirectory && allowFolder != null && allowFolder &&
					this.isFolderUploadSupported()) {
				inputControl.setAttribute('webkitdirectory', 'webkitdirectory');
			} else {
				if (isDirectory) {
					domNodes.hide();
				}
				inputControl.removeAttribute('webkitdirectory');
			}
			var self = this;
			var changeHandler = function(changeEvent) {
				var fileList = changeEvent.target.files;
				if (typeof fileList !== "undefined" && fileList !== null && fileList.length > 0) {
					var isFolderUpload = isDirectory != null && isDirectory && allowFolder != null && allowFolder &&
							self.isFolderUploadSupported(inputControl);
					self.prepareFileForUpload(fileList, null, !isFolderUpload, isFolderUpload);
				}
				changeEvent.stopPropagation();
				changeEvent.target.value = null;
				return;
			};
			if ($.browser.msie) {
				return inputControl.addEventListener('input', changeHandler, false);
			} else {
				return inputControl.addEventListener('change', changeHandler, false);
			}
		}
	};

	Uploader.prototype.addFileEntry = function(uploader, fileEntry, currentBatch) {
		fileEntry.file(function(file) {
			if (uploader != null) {
				file.relativePath = fileEntry.fullPath;
				file.isFile = true;
				file.folderUpload = true;
				uploader.addFile(file, currentBatch);
			}
		});
	};

	Uploader.prototype.addFileEntries = function(uploader, itemEntry, currentBatch) {
		var allowFolder = this.getOptions('allowFolder');
		var allowEmptyFolder = this.getOptions('allowEmptyFolder');
		var self = this;
		var lastModifiedDate = null;
		itemEntry.getMetadata(function(metadata) {
			lastModifiedDate = metadata.modificationTime;
		});
		if (itemEntry.isFile) {
			this.addFileEntry(uploader, itemEntry, currentBatch);
		} else if (itemEntry.isDirectory) {
			if (allowFolder != null && allowFolder) {
				var dirReader = itemEntry.createReader();
				dirReader.readEntries(function(entries) {
					var idx = entries.length;
					if (idx <= 0 && allowEmptyFolder != null && allowEmptyFolder) {
						var file = {};
						file.name = itemEntry.name;
						file.relativePath = itemEntry.fullPath;
						file.lastModifiedDate = lastModifiedDate;
						file.isFile = false;
						file.folderUpload = true;
						file.size = 0;
						self.addFile(file, currentBatch);
					} else {
						// Create the directory before uploading the file
						var dirBatch = self.getBatch(true);
						var file = {};
						file.name = itemEntry.name;
						file.relativePath = itemEntry.fullPath;
						file.lastModifiedDate = lastModifiedDate;
						file.isFile = false;
						file.folderUpload = true;
						file.size = 0;
						self.addFile(file, dirBatch);
						while (idx--) {
							self.addFileEntries(uploader, entries[idx], dirBatch);
						}
					}
				});
			} else {
				// Folders are not allow for upload
				var file = {};
				file.name = itemEntry.name;
				file.relativePath = itemEntry.fullPath;
				file.lastModifiedDate = lastModifiedDate;
				file.isFile = false;
				file.folderUpload = true;
				file.size = 0;
				this.addFile(file, currentBatch);
			}
		}
	};

	Uploader.prototype.onDropPrepareUploadFilesOrFolders = function(dataTransferItems) {
		// Call this function only if browser supports Folder uploads
		var currentBatch = this.getBatch(true);
		var files = [];
		var uploadsContainDirectory = false;
		for (var index = 0; index < dataTransferItems.length; index++) {
			var dataTransferItem = dataTransferItems[index];
			var entry = null;
			if (dataTransferItem.getAsEntry) { //Standard HTML5 API
				entry = dataTransferItem.getAsEntry();
			} else if (dataTransferItem.webkitGetAsEntry) { //WebKit implementation of HTML5 API.
				entry = dataTransferItem.webkitGetAsEntry();
			}
			if (entry.isFile) {
				files.push(dataTransferItem.getAsFile());
			} else if (entry.isDirectory) {
				uploadsContainDirectory = true;
				break;
				this.addFileEntries(this, entry);
			}
		}
		if (uploadsContainDirectory) {
			cvUtil.errorToast('Folder upload is not supported yet!');
			return;
		}
		if (files.length > 0) {
			this.prepareFileForUpload(files, currentBatch);
		}
	};

	Uploader.prototype.onDropPrepareUploadFiles = function(dataTransferFiles) {
		// Call this function only if browser does not supports Folder uploads
		return this.prepareFileForUpload(dataTransferFiles, null, true);
	};

	Uploader.prototype.prepareFileForUpload = function(fileList, currentBatch, doNotAllowFolder, isFolderUpload) {
		if (typeof fileList !== "undefined" && fileList !== null && fileList.length > 0) {
			if (typeof this.uploadProgress === "undefined" || this.uploadProgress == null) {
				this.uploadProgress = new UploadProgressItem();
			}
			this.uploadProgress.init();
			this.fireEvent(this.EventEnum.UPLOAD_START);
		} else {
			cvUtil.errorToast(localMsg.folderDropUploadError);
			return false;
		}

		if (typeof currentBatch === "undefined" || currentBatch === null) {
			currentBatch = this.getBatch(true);
		}
		if (typeof doNotAllowFolder === "undefined" || doNotAllowFolder === null) {
			doNotAllowFolder = false;
		}
		if (typeof isFolderUpload === "undefined" || isFolderUpload === null) {
			isFolderUpload = false;
		}

		for (var index = 0; index < fileList.length; index++) {
			file = fileList[index];
			if (typeof file.isFile === "undefined" || file.isFile === null) {
				file.isFile = true;
			}

			file.name = file.fileName = file.name || file.fileName;
			if (typeof file.folderUpload === "undefined" || file.folderUpload === null) {
				file.folderUpload = isFolderUpload;
			}
			if (doNotAllowFolder != null && doNotAllowFolder && isFolderUpload != null && isFolderUpload) {
				// Upload via Input, get the folder and relative path from file
			}
			var uploadFileInfo = new UploadFileInfo(file, currentBatch);

			if (!this.validateFileAndAdd(uploadFileInfo, doNotAllowFolder, fileList.length)) {
				break;
			}
		}
		// TODO: When folder upload is made then we have to make change to this event!!
		this.fireEvent(this.EventEnum.READY_FOR_UPLOAD);
		return true;
	};

	Uploader.prototype.validateFileAndAdd = function(uploadFileInfo, doNotAllowFolder, fileListLength) {
		var maxFiles = this.getOptions('maxFiles');
		var minFileSize = this.getOptions('minFileSize');
		var maxFileSize = this.getOptions('maxFileSize');
		var matchFileName = this.getOptions('matchFileName');
		var matchFileExtension = this.getOptions('matchFileExtension');
		var checkDuplicateFile = this.getOptions('duplicateFile');
		var allowFolder = this.getOptions('allowFolder');

		var errorMessage = null;
		var returnVal = true;
		var file = uploadFileInfo.fileForUpload;

		if ((maxFiles != null) && maxFiles < (fileListLength + this.fileUploadMap.size())) {
			errorMessage = localMsg.maxFilesError + " " + maxFiles;
			returnVal = false;
		} else if (!cvUtil.isValidWindowFileName(file.name)) {
			errorMessage = localMsg.notValidName;
		} else if (file.isFile && (minFileSize != null) && (file.size <= 0 || file.size < minFileSize)) {
			errorMessage = localMsg.minFileSizeError + " " + cvUtil.formatSize(minFileSize);
		} else if (file.isFile && (maxFileSize != null) && file.size > maxFileSize) {
			errorMessage = localMsg.maxFileSizeError + " " + cvUtil.formatSize(maxFileSize);
		} else if (matchFileName != null && matchFileName.trim().toLowerCase() != file.name.trim().toLowerCase()) {
			errorMessage = localMsg.matchFileNameError + " " + matchFileName;
		} else if (matchFileExtension != null && matchFileExtension.length > 0 &&
				matchFileExtension.indexOf(file.name.substring(file.name.lastIndexOf(".") + 1)) == -1) {
			errorMessage = localMsg.matchFileExtensionError + " " + matchFileExtension;
		} else if (checkDuplicateFile != null && checkDuplicateFile && this.isDuplicateFile(uploadFileInfo)) {
			errorMessage = localMsg.fileDuplicateError;
		} else if (doNotAllowFolder || (allowFolder != null && !allowFolder)) {
			// Check if drag & drop already figured its a directory
			if (file.isFile !== null && !file.isFile) {
				errorMessage = localMsg.folderUploadError;
			} else if (!file.type && file.size > 0 && file.size % 4096 == 0 && file.size <= 102400) {
				// Checks if the file is folder, if so then fail.
				var self = this;
				var reader = new FileReader();
				reader.onerror = function() {
					self.sendError(uploadFileInfo, localMsg.folderUploadError);
				};
				reader.onload = function() {
					//On Error reading which means file of unknown type
					self.addValidFile(uploadFileInfo);
				};
				reader.readAsDataURL(file);
			} else {
				doNotAllowFolder = false;
			}
		}

		if (typeof errorMessage !== "undefined" && errorMessage !== null && errorMessage !== "") {
			this.sendError(uploadFileInfo, errorMessage);
			return returnVal;
		}
		if (!doNotAllowFolder && returnVal) {
			this.addValidFile(uploadFileInfo);
		}
		return returnVal;
	};

	Uploader.prototype.sendError = function(uploadFileInfo, errorMessage) {
		uploadFileInfo.fileProgressItem.error.display = "error";
		uploadFileInfo.fileProgressItem.error.message = errorMessage;
		uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.ERROR);
		this.fireEvent(this.EventEnum.FILE_ADD_ERROR, uploadFileInfo.fileProgressItem);
	};

	Uploader.prototype.addValidFile = function(uploadFileInfo) {
		if (typeof this.uploadProgress !== "undefined" && this.uploadProgress !== null) {
			this.uploadProgress.addFileSize(uploadFileInfo.fileProgressItem.fileSize, true);
		}
		this.fileUploadMap.put(uploadFileInfo.fileProgressItem.fileID, uploadFileInfo);
		this.fireEvent(this.EventEnum.FILE_ADDED, uploadFileInfo.fileProgressItem);
	};

	Uploader.prototype.addFile = function(file, currentBatch) {
		if (typeof file !== "undefined" && file !== null) {
			return this.prepareFileForUpload([ file ], currentBatch);
		}
		return false;
	};

	Uploader.prototype.removeFile = function(fileId) {
		if (typeof fileId !== "undefined" && fileId !== null) {
			var uploadFileInfo = this.fileUploadMap.get(fileId);
			if (uploadFileInfo != null && uploadFileInfo.fileProgressItem.canFileBeRemoved()) {
				this.fileUploadMap.remove(fileId);
				this.fireEvent(this.EventEnum.FILE_REMOVED, uploadFileInfo.fileProgressItem);
				return true;
			} else {
				uploadFileInfo.fileProgressItem.error.display = "error";
				uploadFileInfo.fileProgressItem.error.message = localMsg.removeError;
				this.fireEvent(this.EventEnum.FILE_REMOVE_ERROR, uploadFileInfo.fileProgressItem);
			}
		}
		return false;
	};

	Uploader.prototype.getBatch = function(create) {
		if (create || this.uploadBatch.length == 0) {
			var uploadBatchInfo = new UploadBatchInfo();
			uploadBatchInfo.setBatchOptions(this, 'query');
			uploadBatchInfo.setBatchOptions(this, 'headers');
			uploadBatchInfo.setBatchOptions(this, 'destinationFolder');
			this.uploadBatch.push(uploadBatchInfo);
		}
		return (this.uploadBatch.length - 1);
	};

	Uploader.prototype.isDuplicateFile = function(uploadFileInfo) {
		if (typeof uploadFileInfo !== "undefined" && uploadFileInfo !== null) {
			var keys = this.fileUploadMap.getKeys();
			for (var index = 0; index < keys.length; index++) {
				if (uploadFileInfo.equals(this, this.fileUploadMap.get(keys[index]))) {
					return true;
				}
			}
		}
		return false;
	};

	Uploader.prototype.canUploadNextFile = function(uploadFileInfo) {
		this.currentUploadCount--;
		if (typeof uploadFileInfo !== "undefined" && uploadFileInfo !== null) {
			this.currentUploadMap.remove(uploadFileInfo.fileProgressItem.fileID);
			delete uploadFileInfo;
		}
		this.startUpload();
	};

	Uploader.prototype.startUpload = function() {
		var simultaneousUploads = this.getOptions('simultaneousUploads');
		if (simultaneousUploads == null || simultaneousUploads < 1) {
			simultaneousUploads = 1;
		}
		while (this.currentUploadCount < simultaneousUploads && this.hasNext()) {
			var uploadFileInfo = this.next();
			this.currentUploadMap.put(uploadFileInfo.fileProgressItem.fileID, "");
			this.sendHeaderData(uploadFileInfo);
		}
		var keys = this.fileUploadMap.getKeys();
		var uploadCompleted = null;
		for (var index = 0; index < keys.length; index++) {
			var uploadFileInfo = this.fileUploadMap.get(keys[index]);
			if (uploadFileInfo.canFileMarkedUploadComplete()) {
				if (uploadCompleted === null) {
					uploadCompleted = true;
				}
			} else {
				uploadCompleted = false;
			}
		}
		if (uploadCompleted == null || uploadCompleted) {
			this.onUploadDone();
		}
	};

	Uploader.prototype.hasNext = function() {
		var keys = this.fileUploadMap.getKeys();
		for (var index = 0; index < keys.length; index++) {
			var uploadFileInfo = this.fileUploadMap.get(keys[index]);
			if (uploadFileInfo != null &&
					uploadFileInfo.fileProgressItem.fileStatus == uploadFileInfo.fileProgressItem.Status.NOT_PICKED) {
				delete keys;
				delete uploadFileInfo;
				delete index;
				return true;
			}
		}
		delete keys;
		return false;
	};

	Uploader.prototype.next = function() {
		var keys = this.fileUploadMap.getKeys();
		for (var index = 0; index < keys.length; index++) {
			var uploadFileInfo = this.fileUploadMap.get(keys[index]);
			if (uploadFileInfo != null &&
					uploadFileInfo.fileProgressItem.fileStatus == uploadFileInfo.fileProgressItem.Status.NOT_PICKED) {
				//If there are files to upload increment the upload count
				uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.PICKED);
				this.currentUploadCount++;
				delete keys;
				delete index;
				return uploadFileInfo;
			}
		}
		delete keys;
		return null;
	};

	Uploader.prototype.sendHeaderData = function(uploadFileInfo) {
		var params = [];
		var self = this;
		var customQuery = this.getBatchOptions('query', uploadFileInfo.uploadBatchIndex);
		if (typeof customQuery === 'function') {
			customQuery = customQuery(uploadFileInfo);
		}
		if (customQuery != null) {
			for ( var key in customQuery) {
				var value = customQuery[key];
				if (value != null) {
					params.push(cvUtil.createParamString(key, value));
				}
			}
		}
		delete customQuery;
		uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.WAITING);
		// Update file info and then Push file information
		if (typeof this.uploadProgress !== "undefined" && this.uploadProgress !== null) {
			this.uploadProgress.addFileSize(uploadFileInfo.getFileSizeDiff());
		}
		params.push(cvUtil.createParamString(this.PARAM_STREAMING, true));
		params.push(cvUtil.createParamString(this.PARAM_ISFILE, uploadFileInfo.fileProgressItem.isFile));
		
		var encodeFileName = this.getOptions('encodeFileName');

		var sendFileIdAsName = this.getOptions('sendFileIdAsName');
		
		if (sendFileIdAsName != null && sendFileIdAsName === true) {
			params.push(cvUtil.createParamString(this.PARAM_FILENAME,uploadFileInfo.fileProgressItem.fileID));
		}
		else if (encodeFileName != null && encodeFileName === true) {
			params.push(cvUtil.createParamString(this.PARAM_FILENAME,encodeURIComponent(uploadFileInfo.fileProgressItem.fileName)));
		}
		else
		params.push(cvUtil.createParamString(this.PARAM_FILENAME, uploadFileInfo.fileProgressItem.fileName));
		
		params.push(cvUtil.createParamString(this.PARAM_FILE_SIZE, uploadFileInfo.fileProgressItem.fileSize));
		params.push(cvUtil.createParamString(this.PARAM_FILE_LAST_MODIFIED,
				uploadFileInfo.fileProgressItem.lastModifiedTime));
		var debug = this.getOptions('debug');
		if (debug != null && debug) {
			params.push(cvUtil.createParamString(this.PARAM_DEBUG, debug));
		}
		
		var headers = this.getBatchOptions('headers', uploadFileInfo.uploadBatchIndex);
		var target = this.getOptions('target');
		if (headers == null) {
			headers = {};
		}
		if (!uploadFileInfo.fileProgressItem.canFileContinue()) {
			return;
		}
		var headerAjaxId = $.ajax({
			contentType : "application/x-www-form-urlencoded; charset=UTF-8",
			async : true,
			cache : false,
			type : 'POST',
			url : target,
			headers : headers,
			addCsrf : true,
			data : cvUtil.createParamsString(params),
			beforeSend : function(xhr,s) {
				if(cvUtil.isAdminConsole())
					cvUtil.addCsrf(xhr,s);
				
				uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.STARTED);
				self.fireEvent(self.EventEnum.UPLOAD_PROGRESS);
				return true;
			},
			success : function(serverResponse) {
				delete headers;
				delete params;
				var returnedItems = JSON.parse(serverResponse);
				var returnCode = returnedItems.returnCode;
				if (returnCode == "Success") {
					uploadFileInfo.fileProgressItem.updateProgress(returnedItems.chunkOffset);
					uploadFileInfo.fileProgressItem.serverRequestId = returnedItems.requestId;
					uploadFileInfo.fileProgressItem.currentOffset = returnedItems.chunkOffset;
					self.fireEvent(self.EventEnum.FILE_UPLOAD_STARTING, uploadFileInfo.fileProgressItem);
					delete serverResponse;
					if (uploadFileInfo.fileProgressItem.canFileContinue()) {
						self.sendData(uploadFileInfo);
					}
				} else if (returnCode == "FileExists") {
					var resumeUpload = self.getOptions('resumeUpload');
					var resumeOrRestartUpload = self.getOptions('resumeOrRestartUpload');
					if (!resumeUpload && !resumeOrRestartUpload) {
						// TODO:
						// Do not support Resume feature, resend the header with PARAM_RESUME_UPLOAD as true
					} else if (!resumeUpload && resumeOrRestartUpload) {
						// TODO:
						// Support Resume feature, but ask the user to continue or restart
						// if continue use default implentation
						// if restart, resend the header with PARAM_RESUME_UPLOAD as true
					} else if (resumeUpload && !resumeOrRestartUpload) {
						// Default Implementation
						uploadFileInfo.fileProgressItem.updateProgress(returnedItems.chunkOffset, self.uploadProgress);
						uploadFileInfo.fileProgressItem.serverRequestId = returnedItems.requestId;
						uploadFileInfo.fileProgressItem.currentOffset = returnedItems.chunkOffset;
						self.fireEvent(self.EventEnum.FILE_UPLOAD_STARTING, uploadFileInfo.fileProgressItem);
						delete serverResponse;
						if (uploadFileInfo.fileProgressItem.canFileContinue()) {
							self.sendData(uploadFileInfo);
						}
					} else if (resumeUpload && resumeOrRestartUpload) {
						// TODO:
						// Always ask the user to continue or restart
						// if continue use default implentation
						// if restart, resend the header with PARAM_RESUME_UPLOAD as true
					}
				} else if (returnCode == "Error") {
					var errorCode = 0;
					if (returnedItems.errorCode) {
						errorCode = returnedItems.errorCode;
					}
					self.onUploadError(returnedItems.returnErrorMessage, uploadFileInfo, errorCode);
					delete serverResponse;
				}
			},
			error : function(jqXHR, textStatus, errorThrown) {
				delete headers;
				delete params;
				var errMsg = jqXHR.responseText || textStatus || errorThrown;
				if (errMsg === "error") {
					errMsg = localMsg.fileLockError;
				}
				self.onUploadError(errMsg, uploadFileInfo);
			}
		});
		window.uploadAjaxRequests[window.uploadAjaxRequests.length] = headerAjaxId;
		return true;
	};

	Uploader.prototype.sendData = function(uploadFileInfo) {
		if (uploadFileInfo.fileProgressItem.canFileContinue()) {
			this.uploadFile(uploadFileInfo);
		}
	};

	Uploader.prototype.uploadFile = function(uploadFileInfo) {
		var self = this;
		var range_from = uploadFileInfo.fileProgressItem.currentOffset;
		if (range_from === uploadFileInfo.fileProgressItem.fileSize) {
			// Cool, we already completely uploaded this.
			// Update progress to 100%.
			this.onUploadComplete(range_from, uploadFileInfo);
			return true;
		}
		this.fireEvent(this.EventEnum.FILE_UPLOAD_INPROGRESS, uploadFileInfo.fileProgressItem);
		var range_to = uploadFileInfo.fileProgressItem.fileSize;
		var chunkSize = this.getOptions('chunkSize');
		if (chunkSize != null && chunkSize > 0) {
			range_to = Math.min(range_to, range_from + chunkSize);
		}
		delete chunkSize;
		var slice = uploadFileInfo.fileForUpload.slice || uploadFileInfo.fileForUpload.webkitSlice ||
				uploadFileInfo.fileForUpload.mozSlice;
		var blob = slice.call(uploadFileInfo.fileForUpload, range_from, range_to, uploadFileInfo.fileForUpload.type);
		var xhr = $.ajaxSettings.xhr();

		var headers = this.getBatchOptions('headers', uploadFileInfo.uploadBatchIndex);
		var target = this.getOptions('target');
		if (headers == null) {
			headers = {};
		}
		target += "?requestId=" + uploadFileInfo.fileProgressItem.serverRequestId;
		if (!uploadFileInfo.fileProgressItem.canFileContinue()) {
			return;
		}

		var options = {
			type : 'POST',
			processData : false,
			url : target,
			headers : headers,
			contentType : "application/offset+octet-stream;",
			data : blob,
			addCsrf : true,
			xhr : function() {
				return xhr;
			},
			async : true,
			cache : false,
			beforeSend : function(xhr,s) {
				if(cvUtil.isAdminConsole())
					cvUtil.addCsrf(xhr,s);
				
				if (range_to === uploadFileInfo.fileProgressItem.fileSize) {
					uploadFileInfo.fileProgressItem.updateProgress(range_to, self.uploadProgress, false);
					uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.POST_PROCESS);
					self.fireEvent(self.EventEnum.FILE_UPLOAD_INPROGRESS, uploadFileInfo.fileProgressItem);
				}
			}
		};

		var dataAjaxId = $.ajax(options).fail(function(jqXHR, textStatus, errorThrown) {
			delete slice;
			delete blob;
			delete xhr;
			delete headers;
			delete target;
			delete options;

			self.onUploadFailure(uploadFileInfo, jqXHR, textStatus, errorThrown);
		}).done(function(serverResponse) {
			delete slice;
			delete blob;
			delete xhr;
			delete headers;
			delete target;
			delete options;

			self.onUploadSuccess(uploadFileInfo, serverResponse, range_from, range_to);
		});
		window.uploadAjaxRequests[window.uploadAjaxRequests.length] = dataAjaxId;
		return true;
	};

	Uploader.prototype.onUploadSuccess = function(uploadFileInfo, serverResponse, range_from, range_to) {
		var returnedItems = JSON.parse(serverResponse);
		var returnCode = returnedItems.returnCode;
		if (returnCode == "Success") {
			if (range_to === uploadFileInfo.fileProgressItem.fileSize) {
				this.onUploadComplete(range_to, uploadFileInfo);
			} else {
				// still have more to upload
				uploadFileInfo.fileProgressItem.updateProgress(returnedItems.chunkOffset, this.uploadProgress);
				this.fireEvent(this.EventEnum.FILE_UPLOAD_INPROGRESS, uploadFileInfo.fileProgressItem);
				delete serverResponse;
				if (uploadFileInfo.fileProgressItem.canFileContinue()) {
					this.uploadFile(uploadFileInfo);
				}
			}
		} else if (returnCode == "Error") {
			var errorMessage = returnedItems.returnErrorMessage;
			delete serverResponse;
			this.onUploadError(errorMessage, uploadFileInfo);
		} else if (returnCode == "Cancel") {
			uploadFileInfo.fileProgressItem.updateProgress(range_from, this.uploadProgress);
			uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.CANCELED);
			this.fireEvent(this.EventEnum.FILE_UPLOAD_CANCELED, uploadFileInfo.fileProgressItem);
			this.fileUploadMap.remove(uploadFileInfo.fileProgressItem.fileID);

			delete serverResponse;
			delete range_from;
			delete range_to;

			this.canUploadNextFile(uploadFileInfo);
		}
	};

	Uploader.prototype.onUploadComplete = function(range, uploadFileInfo) {
		uploadFileInfo.fileProgressItem.updateProgress(range, this.uploadProgress);
		uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.COMPLETED);
		if (typeof this.uploadProgress !== "undefined" && this.uploadProgress !== null) {
			this.uploadProgress.updateUploadFileCount(this.uploadProgress.Status.COMPLETED);
		}
		this.fireEvent(this.EventEnum.FILE_UPLOAD_COMPLETED, uploadFileInfo.fileProgressItem);
		this.fileUploadMap.remove(uploadFileInfo.fileProgressItem.fileID);

		delete range;

		this.canUploadNextFile(uploadFileInfo);
	};

	Uploader.prototype.onUploadFailure = function(uploadFileInfo, jqXHR, textStatus, errorThrown) {
		var errMsg = jqXHR.responseText || textStatus || errorThrown;
		if (errMsg === "error") {
			errMsg = localMsg.fileLockError;
		}
		this.onUploadError(errMsg, uploadFileInfo);
	};

	Uploader.prototype.onUploadError = function(errorMessage, uploadFileInfo, errorCode) {
		uploadFileInfo.fileProgressItem.error.display = "error";
		uploadFileInfo.fileProgressItem.error.message = errorMessage;
		uploadFileInfo.fileProgressItem.error.errorCode = errorCode;
		uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.ERROR);
		this.fireEvent(this.EventEnum.FILE_UPLOAD_ERROR, uploadFileInfo.fileProgressItem);
		if (typeof this.uploadProgress !== "undefined" && this.uploadProgress !== null) {
			this.uploadProgress.cancelFileUpload(uploadFileInfo.fileProgressItem.fileSize,
					uploadFileInfo.fileProgressItem.currentOffset);
			this.uploadProgress.updateUploadFileCount(this.uploadProgress.Status.ERROR);
		}
		this.fileUploadMap.remove(uploadFileInfo.fileProgressItem.fileID);
		delete errorMessage;

		this.canUploadNextFile(uploadFileInfo);
	};

	Uploader.prototype.onUploadDone = function() {
		// Once the upload is complete remove all the AJAX id from the array
		if (typeof this.uploadProgress !== "undefined" && this.uploadProgress !== null) {
			this.uploadProgress.updateUploadStatus();
		}
		this.fireEvent(this.EventEnum.UPLOAD_END);
		this.doCleanUp();
		console.debug("Upload Completed.");
	};

	Uploader.prototype.pause = function(fileId) {
		if (typeof fileId !== "undefined" && fileId !== null) {
			var uploadFileInfo = this.fileUploadMap.get(fileId);
			if (uploadFileInfo != null && uploadFileInfo.fileProgressItem.canFileBePaused()) {
				uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.PAUSED);
				this.fireEvent(this.EventEnum.FILE_UPLOAD_PAUSED, uploadFileInfo.fileProgressItem);
				this.canUploadNextFile(uploadFileInfo);
				return true;
			} else {
				uploadFileInfo.fileProgressItem.error.display = "error";
				uploadFileInfo.fileProgressItem.error.message = localMsg.pauseError;
				this.fireEvent(this.EventEnum.FILE_UPLOAD_PAUSE_ERROR, uploadFileInfo.fileProgressItem);
				return false;
			}
		}
		return false;
	};

	Uploader.prototype.resume = function(fileId) {
		if (typeof fileId !== "undefined" && fileId !== null) {
			var uploadFileInfo = this.fileUploadMap.get(fileId);
			if (uploadFileInfo != null && uploadFileInfo.fileProgressItem.canFileBeResumed()) {
				uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.NOT_PICKED);
				this.fireEvent(this.EventEnum.FILE_UPLOAD_RESUMED, uploadFileInfo.fileProgressItem);
				this.canUploadNextFile(uploadFileInfo);
				return true;
			} else {
				uploadFileInfo.fileProgressItem.error.display = "error";
				uploadFileInfo.fileProgressItem.error.message = localMsg.resumeError;
				this.fireEvent(this.EventEnum.FILE_UPLOAD_RESUME_ERROR, uploadFileInfo.fileProgressItem);
				return false;
			}
		}
		return false;
	};

	Uploader.prototype.cancel = function(fileId) {
		if (typeof fileId !== "undefined" && fileId !== null) {
			var uploadFileInfo = this.fileUploadMap.get(fileId);
			if (uploadFileInfo != null && uploadFileInfo.fileProgressItem.canFileBeCanceled()) {
				uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.CANCELED);
				if (typeof this.uploadProgress !== "undefined" && this.uploadProgress !== null) {
					this.uploadProgress.cancelFileUpload(uploadFileInfo.fileProgressItem.fileSize,
							uploadFileInfo.fileProgressItem.currentOffset);
					this.uploadProgress.updateUploadFileCount(this.uploadProgress.Status.CANCELED);
				}
				this.fireEvent(this.EventEnum.FILE_UPLOAD_CANCELED, uploadFileInfo.fileProgressItem);
				this.canUploadNextFile(uploadFileInfo);
				return true;
			} else {
				uploadFileInfo.fileProgressItem.error.display = "error";
				uploadFileInfo.fileProgressItem.error.message = localMsg.cancelError;
				this.fireEvent(this.EventEnum.FILE_UPLOAD_CANCEL_ERROR, uploadFileInfo.fileProgressItem);
				return false;
			}
		}
		return false;
	};

	Uploader.prototype.cancelAll = function() {
		// To stop the all batched upload, set the cancel property
		// Mark current uploads as cancelled
		var keys = this.currentUploadMap.getKeys();
		for (var index = 0; index < keys.length; index++) {
			var uploadFileInfo = this.fileUploadMap.get(keys[index]);
			if (uploadFileInfo != null) {
				uploadFileInfo.fileProgressItem.updateStatus(uploadFileInfo.fileProgressItem.Status.CANCELED);
				this.uploadProgress.updateUploadFileCount(this.uploadProgress.Status.CANCELED);
				this.fireEvent(this.EventEnum.FILE_UPLOAD_CANCELED, uploadFileInfo.fileProgressItem);
			}
		}
		this.fileUploadMap.clear();
		this.currentUploadMap.clear();
		this.canUploadNextFile();
		return true;
	};

	Uploader.prototype.isSameParentPath = function(fileId, browsePath) {
		if (typeof fileId !== "undefined" && fileId !== null && typeof browsePath !== "undefined" &&
				browsePath !== null) {
			var uploadFileInfo = this.fileUploadMap.get(fileId);
			if (uploadFileInfo != null && uploadFileInfo.uploadBatchIndex != null) {
				var destinationFolder = uploader.getBatchOptions('destinationFolder', uploadFileInfo.uploadBatchIndex);
				return !uploadFileInfo.fileProgressItem.folderUpload && destinationFolder === browsePath;
			}
		}
		return false;
	};

	Uploader.prototype.doCleanUp = function() {
		this.fileUploadMap.clear();
		this.currentUploadMap.clear();
		this.currentUploadCount = 0;
		this.uploadProgress = null;
		this.uploadBatch = new Array();
		while (window.uploadAjaxRequests.length > 0) {
			window.uploadAjaxRequests.pop();
		}
	};

	return Uploader;
})();

window.UploadBatchInfo = UploadBatchInfo = (function() {
	function UploadBatchInfo() {
		this.batchOptions = {};
	}

	UploadBatchInfo.prototype.getBatchOptions = function(optionKey) {
		if (this.batchOptions != null && this.batchOptions[optionKey] != null) {
			return this.batchOptions[optionKey];
		} else {
			return null;
		}
	};

	UploadBatchInfo.prototype.setBatchOptions = function(uploader, optionKey) {
		if (this.batchOptions != null) {
			this.batchOptions[optionKey] = uploader.getOptions(optionKey);
			return true;
		}
		return false;
	};

	UploadBatchInfo.prototype.setBatchOption = function(optionKey, optionValue) {
		if (this.batchOptions != null) {
			this.batchOptions[optionKey] = optionValue;
			return true;
		}
		return false;
	};

	return UploadBatchInfo;
})();

window.UploadFileInfo = UploadFileInfo = (function() {
	function UploadFileInfo(fileForUpload, uploadBatchIndex) {
		this.fileForUpload = fileForUpload;
		this.fileProgressItem = new FileProgressItem(this.fileForUpload);
		this.uploadBatchIndex = uploadBatchIndex;
	}

	UploadFileInfo.prototype.getFileSizeDiff = function() {
		if (this.fileProgressItem.fileSize !== this.fileForUpload.size) {
			var diffSize = (this.fileForUpload.size - this.fileProgressItem.fileSize);
			this.fileProgressItem.fileSize = this.fileForUpload.size;
			return diffSize;
		} else {
			return 0;
		}
	};

	UploadFileInfo.prototype.canFileMarkedUploadComplete = function() {
		if (this.fileProgressItem.isFileUploadCompleted() || this.fileProgressItem.isFileUploadCanceled() ||
				this.fileProgressItem.isFileUploadError()) {
			return true;
		}

		return false;
	};

	UploadFileInfo.prototype.equals = function(uploader, uploadFileInfo) {
		if (this.fileProgressItem === null || uploadFileInfo === null || uploadFileInfo.fileProgressItem === null) {
			return false;
		}
		if (this.uploadBatchIndex != null && uploadFileInfo.uploadBatchIndex != null &&
				this.uploadBatchIndex === uploadFileInfo.uploadBatchIndex) {
			// If the files are same index which means same folder
			return this.fileProgressItem.equals(uploadFileInfo.fileProgressItem);
		} else {
			var thisDestinationFolder = uploader.getBatchOptions('destinationFolder', this.uploadBatchIndex);
			var thatDestinationFolder = uploader.getBatchOptions('destinationFolder', uploadFileInfo.uploadBatchIndex);
			if (thisDestinationFolder !== null && thatDestinationFolder != null &&
					thisDestinationFolder === thatDestinationFolder) {
				// If the files are for the same destination folder then compare, else we can allow upload of same
				// file to different folder
				return this.fileProgressItem.equals(uploadFileInfo.fileProgressItem);
			} else {
				return false;
			}
		}
	};

	return UploadFileInfo;
})();

window.FileProgressItem = FileProgressItem = (function() {
	function FileProgressItem(file) {
		// File Status Constants
		this.Status = {
			NOT_PICKED : "notpicked",
			PICKED : "picked",
			WAITING : "waiting",
			STARTED : "started",
			INPROGRESS : "inProgress",
			PAUSED : "paused",
			CANCELED : "cancelled",
			POST_PROCESS : "postprocess",
			COMPLETED : "complete",
			ERROR : "error"
		};

		this.fileID = cvUtil.generateUUID();
		this.fileStatus = this.Status.NOT_PICKED;
		this.isFile = (typeof file.isFile === "undefined") ? true : file.isFile;
		this.folderUpload = (typeof file.folderUpload === "undefined") ? false : file.folderUpload;
		this.fileName = file.fileName || file.name;
		this.fileSize = file.size;
		this.lastModifiedDate = file.lastModifiedDate;
		this.lastModified = file.lastModified;
		this.lastModifiedDateFormatted = this.getFormatedDateTime(cvUtil.isAdminConsole()?"MMM DD, YYYY h:mm:ss a":"NNN dd, yyyy h:mm:ss a");
		this.lastModifiedTime = this.getTimeInSec();
		this.currentOffset = -1;
		this.relativePath = file.relativePath || file.webkitRelativePath;
		this.filePercent = 0;
		this.fileProgress = localMsg.fileProgress;
		this.fileDirection = "up";
		this.fileSizeFormatted = cvUtil.formatSize(this.fileSize);
		this.error = new FileError("no-error", "");
		this.serverRequestId = "";
	}

	FileProgressItem.prototype.equals = function(fileProgressItem) {
		if (fileProgressItem === null) {
			return false;
		}
		if (this.isFile == fileProgressItem.isFile && this.fileName == fileProgressItem.fileName &&
				this.relativePath == fileProgressItem.relativePath && this.fileSize == fileProgressItem.fileSize &&
				this.lastModifiedTime == fileProgressItem.lastModifiedTime) {
			return true;
		}
		return false;
	};

	FileProgressItem.prototype.updateProgress = function(currentOffset, uploadProgress, isStatusUpdateRequired) {
		var noOfBytes = 0;
		var updateStatus = true;
		if (typeof isStatusUpdateRequired !== "undefined" && isStatusUpdateRequired !== null) {
			updateStatus = isStatusUpdateRequired;
		}
		if (this.fileStatus == this.Status.CANCELED || this.fileStatus == this.Status.ERROR) {
			return;
		}
		if (typeof currentOffset !== "undefined" && currentOffset !== null) {
			noOfBytes = (currentOffset - this.currentOffset);
			this.currentOffset = currentOffset;
			if (updateStatus) {
				if (this.currentOffset <= 0) {
					this.updateStatus(this.Status.STARTED);
				} else if (this.currentOffset > 0 && this.currentOffset < this.fileSize) {
					this.updateStatus(this.Status.INPROGRESS);
				} else if (this.currentOffset > 0 && this.currentOffset == this.fileSize) {
					this.updateStatus(this.Status.COMPLETED);
				}
			}
			this.filePercent = (this.currentOffset > 0 && this.fileSize > 0) ? (this.currentOffset / this.fileSize * 100)
					.toFixed(0)
					: 0;
			this.fileProgress = cvUtil.formatSize(this.currentOffset) + localMsg.fileOf +
					cvUtil.formatSize(this.fileSize);
		}
		if (typeof uploadProgress !== "undefined" && uploadProgress !== null) {
			uploadProgress.updateUploadProgress(noOfBytes);
		}
	};

	FileProgressItem.prototype.updateStatus = function(newStatus) {
		if (typeof newStatus !== "undefined" && newStatus !== null) {
			// DO NOT update the status if the request is canceled
			if (this.fileStatus != this.Status.CANCELED) {
				if (this.fileStatus != this.Status.COMPLETED || this.fileStatus != this.Status.POST_PROCESS) {
					this.fileStatus = newStatus;
				}
				if (this.fileStatus != this.Status.ERROR) {
					this.error.display = "no-error";
					this.error.message = "";
				}
			} else if (this.fileStatus === this.Status.CANCELED) {
				// For now do nothing
			} else {
				// For now do nothing
			}
		}
	};

	FileProgressItem.prototype.canFileContinue = function() {
		var status = this.fileStatus;
		var returnVal = false;
		if (typeof status != "undefined" && status != null) {
			if (status == this.Status.PAUSED) {
				returnVal = false;
			} else if (status == this.Status.CANCELED) {
				returnVal = false;
			} else {
				returnVal = true;
			}
		}
		return returnVal;
	};

	FileProgressItem.prototype.canFileBePaused = function() {
		var status = this.fileStatus;
		var returnVal = false;
		if (typeof status != "undefined" && status != null) {
			if (status == this.Status.NOT_PICKED) {
				returnVal = true;
			} else if (status == this.Status.INPROGRESS) {
				returnVal = true;
			} else {
				returnVal = false;
			}
		}
		return returnVal;
	};

	FileProgressItem.prototype.canFileBeResumed = function() {
		var status = this.fileStatus;
		var returnVal = false;
		if (typeof status != "undefined" && status != null) {
			if (status == this.Status.PAUSED) {
				returnVal = true;
			} else if (status == this.Status.CANCELED) {
				returnVal = true;
			} else {
				returnVal = false;
			}
		}
		return returnVal;
	};

	FileProgressItem.prototype.canFileBeCanceled = function() {
		if (this.fileSize == this.currentOffset) {
			return false;
		}
		var status = this.fileStatus;
		var returnVal = false;
		if (typeof status != "undefined" && status != null) {
			if (status != this.Status.CANCELED) {
				returnVal = true;
			} else if (status != this.Status.COMPLETED) {
				returnVal = true;
			} else if (status != this.Status.POST_PROCESS) {
				returnVal = true;
			} else if (status != this.Status.ERROR) {
				returnVal = true;
			} else {
				returnVal = false;
			}
		}
		return returnVal;
	};

	FileProgressItem.prototype.canFileBeRemoved = function() {
		var status = this.fileStatus;
		var returnVal = false;
		if (typeof status != "undefined" && status != null) {
			if (status == this.Status.NOT_PICKED) {
				returnVal = true;
			} else if (status == this.Status.ERROR) {
				returnVal = true;
			} else if (status == this.Status.CANCELED) {
				returnVal = true;
			} else {
				returnVal = false;
			}
		}
		return returnVal;
	};

	FileProgressItem.prototype.isFileUploadCanceled = function() {
		var status = this.fileStatus;
		if (typeof status != "undefined" && status != null) {
			return status === this.Status.CANCELED;
		}
		return false;
	};

	FileProgressItem.prototype.isFileUploadCompleted = function() {
		var status = this.fileStatus;
		if (typeof status != "undefined" && status != null) {
			return status === this.Status.COMPLETED;
		}
		return false;
	};

	FileProgressItem.prototype.isFileUploadError = function() {
		var status = this.fileStatus;
		if (typeof status != "undefined" && status != null) {
			return status === this.Status.ERROR;
		}
		return false;
	};

	FileProgressItem.prototype.getTimeInSec = function() {
		var itemModifiedDateTime = "";
		if (typeof this.lastModifiedDate != "undefined" && this.lastModifiedDate != null) {
			itemModifiedDateTime = this.lastModifiedDate;
		} else if (typeof this.lastModified != "undefined" && this.lastModified != null) {
			itemModifiedDateTime = this.lastModified;
		}
		var dateObj = new Date(itemModifiedDateTime);
		return (dateObj.getTime() / 1000).toFixed(0);
	};

	FileProgressItem.prototype.getFormatedDateTime = function(dateTimeFormat) {
		var itemModifiedDateTime = "";
		if (typeof this.lastModifiedDate != "undefined" && this.lastModifiedDate != null) {
			itemModifiedDateTime = this.lastModifiedDate;
		} else if (typeof this.lastModified != "undefined" && this.lastModified != null) {
			itemModifiedDateTime = this.lastModified;
		}
		var dateObj = new Date(itemModifiedDateTime);
		return cvUtil.isAdminConsole()?moment(dateObj).format(dateTimeFormat):dateObj.format(dateTimeFormat);
	};

	FileProgressItem.prototype.toString = function() {
		var fileProgressItemStr = "Name: " + this.fileName;
		fileProgressItemStr += " Size: " + this.fileSizeFormatted;
		fileProgressItemStr += " Status: " + this.fileStatus;
		fileProgressItemStr += " Progress: " + this.fileProgress;
		fileProgressItemStr += " Percent: " + this.filePercent;

		return fileProgressItemStr;
	};

	return FileProgressItem;
})();

window.FileError = FileError = (function() {
	function FileError(display, message) {
		this.display = display;
		this.message = message;
	}

	return FileError;
})();

window.UploadProgressItem = UploadProgressItem = (function() {
	function UploadProgressItem() {
		// Upload Status Constants
		this.Status = {
			NONE : "none",
			CANCELED : "canceled",
			COMPLETED_WITH_ERROR : "partial",
			COMPLETED : "success",
			ERROR : "error"
		};
		this.uploadCompleted = false;
		this.uploadFileSuccessCount = 0;
		this.uploadFileFailedCount = 0;
		this.uploadFileCanceledCount = 0;
		this.uploadTotalFileCount = -1;
		this.formattedUploadCount = "";
		this.uploadFileSize = 0;
		this.uploadTotalFileSize = -1;
		this.formattedUploadSize = "";
		this.uploadPercent = 0;
		this.formattedUploadStatus = "";
		this.formattedUploadPercentage = "";
	}

	UploadProgressItem.prototype.init = function() {
		this.uploadCompleted = false;
		this.formattedUploadStatus = localMsg.progressHeader;
	};

	UploadProgressItem.prototype.updateUploadStatus = function() {
		this.uploadCompleted = true;
		if (this.uploadFileSuccessCount === 0 && this.uploadFileFailedCount === 0) {
			if (this.uploadFileCanceledCount > 0) {
				this.formattedUploadStatus = localMsg.cancelHeader;
			} else {
				this.formattedUploadStatus = localMsg.completeHeader;
			}
		} else if (this.uploadFileSuccessCount === 0 && this.uploadFileFailedCount > 0) {
			this.formattedUploadStatus = localMsg.errorHeader;
		} else if (this.uploadFileSuccessCount > 0 && this.uploadFileFailedCount === 0) {
			this.formattedUploadStatus = localMsg.completeHeader;
		} else if (this.uploadFileSuccessCount > 0 && this.uploadFileFailedCount > 0) {
			this.formattedUploadStatus = localMsg.partialHeader;
		}
	};

	UploadProgressItem.prototype.addFileSize = function(fileSize, updateFileCount) {
		if (typeof fileSize != "undefined" && fileSize != null && fileSize >= 0) {
			if (this.uploadTotalFileSize < 0) {
				this.uploadTotalFileSize = 0;
			}
			if (typeof updateFileCount != "undefined" && updateFileCount != null && updateFileCount) {
				if (this.uploadTotalFileCount < 0) {
					this.uploadTotalFileCount = 0;
				}
				this.uploadTotalFileCount++;
			}

			this.uploadTotalFileSize += fileSize;
			this.updateUploadProgress(0);
		}
	};

	UploadProgressItem.prototype.updateUploadFileCount = function(uploadState) {
		if (typeof uploadState === "undefined" || uploadState === null || uploadState === "") {
			return;
		}
		if (uploadState === this.Status.COMPLETED) {
			this.uploadFileSuccessCount++;
		} else if (uploadState === this.Status.ERROR) {
			this.uploadFileFailedCount++;
		} else if (uploadState === this.Status.CANCELED) {
			this.uploadFileCanceledCount++;
		}
		this.updateUploadProgress(0);
	};

	UploadProgressItem.prototype.cancelFileUpload = function(fileSize, currentOffset) {
		if (this.uploadTotalFileCount > 0) {
			this.uploadTotalFileCount--;
		} else {
			this.uploadTotalFileCount = 0;
		}
		if (typeof fileSize != "undefined" && fileSize != null && fileSize >= 0) {
			if (this.uploadTotalFileSize > 0) {
				this.uploadTotalFileSize -= fileSize;
			} else {
				this.uploadTotalFileSize = 0;
			}
		}
		if (typeof currentOffset != "undefined" && currentOffset != null && currentOffset >= 0) {
			if (this.uploadFileSize > 0) {
				this.uploadFileSize -= currentOffset;
			} else {
				this.uploadFileSize = 0;
			}
		}
		this.updateUploadProgress(0);
	};

	UploadProgressItem.prototype.updateUploadProgress = function(noOfBytes) {
		if (typeof noOfBytes != "undefined" && noOfBytes != null && noOfBytes >= 0) {
			this.uploadFileSize += noOfBytes;
		}
		if (this.uploadFileSize > 0 && this.uploadTotalFileSize > 0) {
			this.uploadPercent = (this.uploadFileSize / this.uploadTotalFileSize * 100).toFixed(0);
		} else {
			this.uploadPercent = 0;
		}
		if (this.uploadTotalFileCount >= 0 && this.uploadFileSuccessCount >= 0) {
			this.formattedUploadCount = this.uploadFileSuccessCount + localMsg.fileOf + this.uploadTotalFileCount +
					" " + localMsg.filedone;
		} else {
			this.formattedUploadCount = "";
		}
		if (this.uploadFileSize > 0 && this.uploadTotalFileSize > 0) {
			this.formattedUploadSize = cvUtil.formatSize(this.uploadFileSize) + localMsg.fileOf +
					cvUtil.formatSize(this.uploadTotalFileSize) + " " + localMsg.filedone;
			;
		}
		if (this.uploadPercent >= 0) {
			this.formattedUploadPercentage = this.uploadPercent + "%";
		} else {
			this.formattedUploadPercentage = "";
		}
	};

	return UploadProgressItem;
})();
