/*!
 * jQuery plugin
 * Component for each facet
 * Responsible for creating widget for each facet config
 */
;
(function(factory) {
	if (!jQuery) {
		throw 'cvFacetWidget depends on jQuery and seems to be missing...';
	} else {
		factory(jQuery);
	}
}(function($) {
	"use strict";

	$.fn.cvFacetWidget = function(queryBuilder, facetsConfig) {
		return this.each(function() {
			var $this = $(this), data = $this.data('cvFacetWidget');
			if (!data) {
				$this.data('cvFacetWidget', new FacetWidget(this, queryBuilder, facetsConfig));
			}
		});
	};
	$.fn.cvFacetWidget.id = 0;

}));

function FacetWidget(element, qbId, config) {
	this._qbId = qbId;
	this.init(element, config);
	this._config = $.extend({}, this.defaults, this._config);
};

FacetWidget.prototype.defaults = {
	allowMultiple : true,
	noOfValuesToShow : 4,
	sortValuesBy : "count",
	listType : "infinite",
	facetType : "simple"
};
FacetWidget.prototype.Constants = {
	ID_PREFIX : "cvFacetWidgetHolder_",
	HOLDER_CLASS : "cvFacetWidgetHolder",
	VALUES_HOLDER_CLASS : "cvFacetWidgetValuesHolder",
	CUSTOM_HOLDER_CLASS : "cvFacetWidgetCustomHolder",
	FACET_SUGGEST_BOX_CLASS : "cvFacetSuggestBox",
	FACET_DATERANGE_CLASS : "cvFacetDateRangePicker",
	FACET_CUSTOM_INPUT_CLASS : "cvCustomFacetInput",
	FACET_CUSTOM_RANGE_CLASS : "cvCustomFacetRangeDiv",
	FACET_CUSTOM_RANGE_WITH_UNIT_CLASS : "cvCustomFacetRangeInputWithUnit",

	DATA_ATTR_VALUE : "cv-facet-value",
	DATA_ATTR_RANGE : "cv-range-ctrl-for",
	DATA_ATTR_RANGE_START : "from",
	DATA_ATTR_RANGE_END : "to",
};

FacetWidget.prototype.templates = {
	holderTmpl : function(id, className) {
		return "<div class='" + className + "' id='" + id + "'></div>";
	},
	radioOrCheckBoxTmpl : function(type, name, dispStr, isSelected) {
		var chkStr = "";
		if (isSelected)
			chkStr = "checked";
		return "<div><label><input type='" + type + "' name='" + name + "' " + chkStr + ">" + dispStr +
				"</label></div>";
	},
	comboBoxItemTmpl : function(value, dispStr) {
		return "<option value='" + value + "'>" + dispStr + "</option>";
	},
	suggestBoxTmpl : function(name) {
		return "<input type='search' name='" + name + "' placeholder='" + cvSearchMessages.SuggestBoxFilterText +
				"' class='typeahead form-control " + cvSearchConstants.CssConstants.SUGGEST_BOX_CLASS + " " +
				FacetWidget.prototype.Constants.FACET_SUGGEST_BOX_CLASS + "' " + cvSearchConstants.suggestBoxProps +
				"/>";
	},
	customTextBoxTmpl : function(name) {
		return "<input type='search' name='" + name + "' placeholder='" + cvSearchMessages.SuggestBoxFilterText +
				"' class='form-control " + FacetWidget.prototype.Constants.FACET_CUSTOM_INPUT_CLASS + "'/>";
	},
	customRangeTextBoxTmpl : function(name, unitObj) {
		var inputClass = "";
		if (unitObj)
			inputClass = FacetWidget.prototype.Constants.FACET_CUSTOM_RANGE_WITH_UNIT_CLASS;
		var htm = "<div class='" + FacetWidget.prototype.Constants.FACET_CUSTOM_RANGE_CLASS + "'>" +
				"<input type='search' name='" + name + "' placeholder='" + cvSearchMessages.RangeFacetStartText +
				"' class='form-control " + FacetWidget.prototype.Constants.FACET_CUSTOM_INPUT_CLASS + " " + inputClass +
				"' data-" + FacetWidget.prototype.Constants.DATA_ATTR_RANGE + "='" +
				FacetWidget.prototype.Constants.DATA_ATTR_RANGE_START + "'/>";
		htm += "<span>";
		if (unitObj)
			htm += " " + unitObj.unit;
		htm += " - </span>";
		htm += "<input type='search' name='" + name + "' placeholder='" + cvSearchMessages.RangeFacetEndText +
				"' class='form-control " + FacetWidget.prototype.Constants.FACET_CUSTOM_INPUT_CLASS + " " + inputClass +
				"' data-" + FacetWidget.prototype.Constants.DATA_ATTR_RANGE + "='" +
				FacetWidget.prototype.Constants.DATA_ATTR_RANGE_END + "'/>";
		if (unitObj)
			htm += "<span> " + unitObj.unit + "</span>";
		htm += "</div>";
		return htm;
	},
	customDateRangeTmpl : function(name) {
		return "<div class='input-group'><input type='daterange' name='" + name + "' placeholder='" +
				cvSearchMessages.SuggestBoxFilterText + "' class='form-control " +
				FacetWidget.prototype.Constants.FACET_CUSTOM_INPUT_CLASS + " " +
				FacetWidget.prototype.Constants.FACET_DATERANGE_CLASS +
				"'/><span class='input-group-addon glyphicon glyphicon-calendar'></span></div>";
	},

};

FacetWidget.prototype.init = function(element, config) {
	this._id = $.fn.cvFacetWidget.id++;
	this.$element = $(element);
	this._origConfig = config;
	this._config = $.extend({}, FacetWidget.prototype.defaults, config);

	this.state = {};
	this.state.isToBeHidden = false;
	this.state.hasSelection = false;
	this.suggestReqUrl = (this._config.listType && this._config.listType !== "finite") ? qb.sv.getFacetSuggestReqStr(this._qbId, this._config) : "";
	this.render();
	this.isDateFacet = qb.sv.isDateRangeAllowed(this._config.schema.type);
	this.isDecimalAllowed = qb.sv.isDecimalRangeAllowed(this._config.schema.type);
};
FacetWidget.prototype.render = function() {
	try {
		this.clear();
		var facetsHolderDiv = $(FacetWidget.prototype.templates.holderTmpl(this.Constants.ID_PREFIX + this._id,
				this.Constants.HOLDER_CLASS));
		facetsHolderDiv.append(this.renderContent());
		this.$element.append(facetsHolderDiv);
		this.onPostRender();
		this.attachListeners(facetsHolderDiv);
	} catch (e) {
		console.error(e);
	}
};
FacetWidget.prototype.clear = function() {
	$("#" + this.getIdStr()).remove();
	this.state.isToBeHidden = false;
	this.state.hasSelection = false;
};
FacetWidget.prototype.getIdStr = function() {
	return this.Constants.ID_PREFIX + this._id;
};
FacetWidget.prototype.mask = function() {
	$("#" + this.getIdStr()).mask("");
};
FacetWidget.prototype.unmask = function() {
	$("#" + this.getIdStr()).unmask();
};
FacetWidget.prototype.getConfig = function() {
	return $.extend({}, this._config);
};
FacetWidget.prototype.getConfigToSave = function() {
	//return only the config modified by the user; not by the system
	return cvSearchUtil.getDiffObjects(this.defaults, this._config);
};
FacetWidget.prototype.setFacetResp = function(facetValues) {
	var self = this;
	this.facetValues = facetValues;
	this.facetMoreValues = [];
	/*
	 * if only one/two extra facet value than the noOfValuesToShow then do not add combo/suggest box as it
	 * will look odd
	 */
	if (!$.isEmptyObject(this.facetValues) && facetValues.length > this._config.noOfValuesToShow + 2) {
		this.facetMoreValues = this.facetValues.slice(this._config.noOfValuesToShow, this.facetValues.length);
		this.facetValues = this.facetValues.slice(0, this._config.noOfValuesToShow);
	}
	/*
	 * If user has selected a facet from combo/suggest, we need to show even if the count is more than the
	 * noOfValuesToShow. So take out the facets with selected = true and push it to facetValues others will be
	 * in facetMoreValues
	 */
	if (!$.isEmptyObject(this.facetMoreValues) && $.isArray(this.facetMoreValues)) {
		this.filteredMoreValues = this.facetMoreValues.filter(function(eachValueObj) {
			if (eachValueObj.selected)
				self.facetValues.push(eachValueObj);
			return !eachValueObj.selected;
		});
		this.facetMoreValues = this.filteredMoreValues;
	}
	this.render();
};
FacetWidget.prototype.fnIsToBeHidden = function() {
	return this.state.isToBeHidden;
};
FacetWidget.prototype.fnHasSelection = function() {
	return this.state.hasSelection;
};
FacetWidget.prototype.getState = function() {
	return this.state;
};
FacetWidget.prototype.setHasSelection = function(facetReqObj) {
	if ($.isEmptyObject(facetReqObj) || $.isEmptyObject(facetReqObj.params))
		this.state.hasSelection = false;
	else
		this.state.hasSelection = true;
};

FacetWidget.prototype.getBaseRequest = function() {
	var req = {
		fieldName : this._config.fieldName,
		facetType : this._config.facetType,
		params : []
	};
	return req;
};
FacetWidget.prototype.attachListeners = function(elm) {
	var self = this;
	elm.on("click", "input[type=checkbox],input[type=radio]", function(e) {
		var selectedFacet = self.onCheckBoxClicked(elm, e);
		self.setHasSelection(selectedFacet);
		$(this).trigger(cvSearchConstants.EventConstants.ON_FACET_SELECTED, selectedFacet);
	});
	var submitCustomSelect = function(e, start, end) {
		var selectedFacet = self.onCustomValueSelected(elm, e, start, end);
		self.setHasSelection(selectedFacet);
		$(e.target).trigger(cvSearchConstants.EventConstants.ON_FACET_SELECTED, selectedFacet);
	};
	elm.on("change", "select", function(e) {
		submitCustomSelect(e);
	});
	elm.on("keypress", "input[type=search]", function(e) {
		if (e.which === 13) {
			// check if range facet value
			var elm = $(e.target);
			var rangeFor = elm.data(self.Constants.DATA_ATTR_RANGE);
			if (rangeFor) {
				var start, end;
				if (rangeFor === self.Constants.DATA_ATTR_RANGE_START) {
					start = elm.val();
					var rangeEndElm = elm.parent().find("input[name=" + elm.attr("name") + "][data-" +
							self.Constants.DATA_ATTR_RANGE + "=" + self.Constants.DATA_ATTR_RANGE_END + "]");
					if (rangeEndElm)
						end = $(rangeEndElm).val();
				} else if (rangeFor === self.Constants.DATA_ATTR_RANGE_END) {
					end = elm.val();
					var rangeStartElm = elm.parent().find("input[name=" + elm.attr("name") + "][data-" +
							self.Constants.DATA_ATTR_RANGE + "=" + self.Constants.DATA_ATTR_RANGE_START + "]");
					if (rangeStartElm)
						start = $(rangeStartElm).val();
				}
				if (start && end)
					submitCustomSelect(e, start, end);
				else
					submitCustomSelect(e);
			} else
				submitCustomSelect(e);
		}
	});
	elm.on("typeahead:selected", "input[type=search].typeahead", function(e) {
		submitCustomSelect(e);
	});
	elm.on("apply.daterangepicker", "input[type=daterange]", function(e, dt) {
		submitCustomSelect(e, dt.startDate.startOf("day"), dt.endDate.endOf("day"));
	});
	elm.on("click", "input[type=daterange]+span", function(e) {
		$(this).prev().data("daterangepicker").show();
	});

};

FacetWidget.prototype.renderContent = function() {
	var self = this;
	var elmToReturn = $("<div><div class='" + self.Constants.VALUES_HOLDER_CLASS + "'></div><div class='" +
			self.Constants.CUSTOM_HOLDER_CLASS + "'></div></div>");
	var valuesDiv = elmToReturn.find("." + self.Constants.VALUES_HOLDER_CLASS);
	var customDiv = elmToReturn.find("." + self.Constants.CUSTOM_HOLDER_CLASS);
	self.state.hasSelection = false;
	if ($.isEmptyObject(this.facetValues))
		valuesDiv.append(cvSearchMessages.EmptyMsg);
	else {
		var type = "checkbox";
		if (!self._config.allowMultiple)
			type = "radio";
		$.each(this.facetValues, function(i, obj) {
			if (!$.isEmptyObject(obj)) {
				if (obj.selected)
					self.state.hasSelection = true;
				obj.dispValue = cvFormatters.getFacetDisplayString(obj, self._config.renderer);
				var el = $(self.templates.radioOrCheckBoxTmpl(type, self.getIdStr(), obj.dispValue, obj.selected));
				el.find("input").data(self.Constants.DATA_ATTR_VALUE, obj);
				valuesDiv.append(el);
			}
		});
		var el = "";
		if (this._config.listType && this._config.listType === "finite") {
			if (!$.isEmptyObject(this.facetMoreValues))
				el = $(this.getComboHtml());
		} else {
			if (this.isDateFacet)
				el = $(this.templates.customDateRangeTmpl(this.getIdStr()));
			else if (qb.sv.isFacetSuggestionAllowed(this._config.schema.type))
				el = $(this.getSuggestBoxHtml());
			else {
				if (qb.sv.isNumberRangeAllowed(this._config.schema.type))
					el = $(this.templates.customRangeTextBoxTmpl(this.getIdStr(),
							this._config.renderer ? this._config.renderer.unitObj : undefined));
				else
					el = $(this.templates.customTextBoxTmpl(this.getIdStr()));
			}
		}
		customDiv.append(el);
	}
	return elmToReturn;
};
FacetWidget.prototype.onPostRender = function() {
	// initialize the suggestbox after attaching to the dom
	var sBoxElm = this.$element.find("." + FacetWidget.prototype.Constants.FACET_SUGGEST_BOX_CLASS);
	if (sBoxElm && sBoxElm.length > 0)
		this.initSuggestBox(sBoxElm[0]);
	var dtElm = this.$element.find("." + FacetWidget.prototype.Constants.FACET_DATERANGE_CLASS);
	if (dtElm && dtElm.length > 0)
		$(dtElm[0]).daterangepicker({
			/*
			 * ranges : { 'Today' : [ new Date(), new Date() ], 'Yesterday' : [ moment().subtract('days', 1),
			 * moment().subtract('days', 1) ], 'Last 7 Days' : [ moment().subtract('days', 6), new Date() ],
			 * 'Last 30 Days' : [ moment().subtract('days', 29), new Date() ], 'This Month' : [
			 * moment().startOf('month'), moment().endOf('month') ], 'Last Month' : [
			 * moment().subtract('month', 1).startOf('month'), moment().subtract('month', 1).endOf('month') ] },
			 */
			showDropdowns : true,
			format : this._config.renderer.fmt
		});
};
FacetWidget.prototype.initSuggestBox = function(sBoxElm) {
	var self = this;
	if (sBoxElm) {
		var sBoxBh = new Bloodhound({
			datumTokenizer : Bloodhound.tokenizers.obj.whitespace('value'),
			queryTokenizer : Bloodhound.tokenizers.whitespace,
			remote : {
				//TODO: facet suggest request has to be submitted by qb so that fq and q will be honored
				url : this.suggestReqUrl,				
				filter : function(resp) {
					return qb.sv.getFacetRespFromSuggestionResp(resp, self._config.fieldName);
				}
			}
		});
		sBoxBh.initialize();

		$(sBoxElm).typeahead({
			highlight : true,
			autoselect : true,
		}, {
			displayKey : "value",
			source : sBoxBh.ttAdapter(),
			templates : {
				empty : cvSearchConstants.templates.emptySuggestionsTmpl(),
				suggestion : function(obj) {
					obj.dispValue = cvFormatters.getFacetDisplayString(obj, self._config.renderer);
					return obj.dispValue;
				}
			}
		});
	}
};

FacetWidget.prototype.getComboHtml = function() {
	var self = this;
	var htm = "<div><select name='" + this.getIdStr() + "' class='form-control'><option>" +
			cvSearchMessages.ComboBoxSelectText + "</option>";
	$.each(this.facetMoreValues, function(i, obj) {
		obj.dispValue = cvFormatters.getFacetDisplayString(obj, self._config.renderer);
		htm += self.templates.comboBoxItemTmpl(obj.value, obj.dispValue);
	});
	htm += "</select></div>";
	return htm;
};

FacetWidget.prototype.getSuggestBoxHtml = function() {
	var sBoxElmHolder = $("<div></div>");
	var sBoxElm = $(this.templates.suggestBoxTmpl(this.getIdStr()));
	sBoxElmHolder.append(sBoxElm);
	return sBoxElmHolder[0].outerHTML;
};
FacetWidget.prototype.onCheckBoxClicked = function(parentElm, event) {
	var self = this;
	var facetReqObj = this.getBaseRequest();
	var groupName = $(event.target).attr("name");
	var selectedElms = parentElm.find("input[type='checkbox'][name='" + groupName +
			"']:checked,input[type='radio'][name='" + groupName + "']:checked");
	var values = [];
	if (selectedElms && selectedElms.length > 0) {
		$.each(selectedElms, function(i, elm) {
			values.push($(elm).data(self.Constants.DATA_ATTR_VALUE));
		});
	}
	facetReqObj.params = values;
	return facetReqObj;
};
FacetWidget.prototype.onCustomValueSelected = function(parentElm, event, start, end) {
	var facetReqObj = this.getBaseRequest();
	if (this._config.allowMultiple)
		facetReqObj = this.onCheckBoxClicked(parentElm, event);
	var param = {
		isCustom : true
	};
	var unitObj = this._config.renderer ? this._config.renderer.unitObj : undefined;
	if (start && end) {
		param.start = typeof (start) !== "object" && unitObj ? start * unitObj.value : start;
		param.end = typeof (end) !== "object" && unitObj ? end * unitObj.value : end;
		if (!this.isDecimalAllowed && !this.isDateFacet) {
			param.start = Math.round(param.start, 0);
			param.end = Math.round(param.end, 0);
		}
		param.isRange = true;
		param.value = (typeof (start) === "object" && moment.isMoment(start)) ? (start.utc().unix() + "-" + end.utc()
				.unix()) : (start + "-" + end);
	} else
		param.value = $(event.target).val();
	facetReqObj.params.push(param);
	return facetReqObj;
};
