/**
 * Controller for data set manager page.
 */

(function() {
	"use strict";

	angular.module('reports').controller('dataSetManagerController', dataSetManagerController);
	dataSetManagerController.$inject = ['$scope', '$uibModal', 'reportService', 'customReportSvc' ];
	function dataSetManagerController($scope, $uibModal, reportService, customReportSvc) {
		var dataSetManager = this,

		// Serves as fulfillment handler to HTTP requests.
		onFulfilled = function(message) {
			return function(response) {
				cvUtil.toast(message, null, null, true);
				reloadTable();
			}
		},

		// Serves as rejection handler to HTTP requests.
		onRejected = function(errorResponse) {
			// This Function is also called when the data set dialog is closed, but it results into a no-op since
			// errorResponse.data is undefined at that time.
			customReportSvc.errorToast(errorResponse.data);
		},

		// Reloads the table component.
		reloadTable = function() {
			rpt.clearCache();
			rpt.reload('dataSetManagerTable');
		},

		// Fetches names of all existing data sets.
		getAllDataSetNames = function() {
			return reportService.getAllDataSets(true)
			.then(function(response) {
				var dataSets = response.data.dataSet || [];
				return dataSets.map(function(dataSet) { return dataSet.dataSet.dataSetName; });
			});
		},

		// Opens the Add/Edit Data Set modal dialog.
		openDataSetDialog = function(dataSet) {

			$scope.dsScope = angular.element("#dataSetManagerTable").scope();
			// The 'Add data set' modal controller requires its arguments present in the input scope.
			//$scope.dsScope.clearAndResetCacheId = function() {};
			$scope.dsScope.dataSet = angular.copy(dataSet);
			$scope.dsScope.mode = (typeof dataSet === "object" ? "Edit" : "Add");
			$scope.dsScope.includeBootStrap = true;
			$scope.dsScope.page = { inputs : [] };
			$scope.dsScope.requestParms = [];
			$scope.dsScope.isPreview = true;
			$scope.dsScope.sharedMode = true;
			$scope.dsScope.autoDeploy = false;
			// It is required to prefetch all data set names since they should be available when modal dialog
			// controller tries to resolve them.
			return getAllDataSetNames()
			.then(function(dataSetNames) {
				$scope.dsScope.dataSetNames = dataSetNames;
				var modalInstance = $uibModal.open({
					templateUrl : '../reportsplus/addDataSetModal/modal.jsp',
					controller : 'addDataSetModalCtrl',
					size : 'lg',
					backdrop : 'static',
					scope: $scope.dsScope,
				});

				return modalInstance.result;
			})
		};

		// Opens 'Add data set' dialog and requests server to create a new data set.
		dataSetManager.createDataSet = function() {
			openDataSetDialog()
			.then(reportService.saveDataSet)
			.then(onFulfilled('Data Set was created successfully.'))
			.catch(onRejected);
		};

		// Opens 'Edit data set' dialog and requests server to update an existing data set.
		dataSetManager.editDataSet = function(dataSetName) {
			reportService.getDataSet(dataSetName, false)
			.then(function(response) {
				return openDataSetDialog(response.data)
			})
			.then(reportService.saveDataSet)
			.then(function(response) {
				if($scope.dsScope.autoDeploy){
					$scope.dsScope.autoDeploy = false;
					dataSetManager.deployDataSet(dataSetName);
				}else{
					cvUtil.toast('Data Set '+dataSetName+' updated successfully.', null, null, true);
					reloadTable();
				}
			})
			.catch(onRejected);
		};

		// Requests server to deploy the data set.
		dataSetManager.deployDataSet = function(dataSetName) {
			reportService.deployDataSet(dataSetName)
			.then(onFulfilled('Data Set '+dataSetName+' deployed successfully.'))
			.catch(onRejected);
		};

		// Requests server to delete the data set.
		dataSetManager.deleteDataSet = function(dataSetName) {
			if(confirm("Delete the data set '" + dataSetName + "'?") == true) {
				reportService.deleteDataSet(dataSetName)
				.then(onFulfilled('Data Set '+dataSetName+' deleted successfully.'))
				.catch(onRejected);
			}
		}

		// Enables user to create a new data set from existing data set.
		dataSetManager.cloneDataSet = function(dataSetName) {
			reportService.getDataSet(dataSetName, false)
			.then(function(response) {
				var dataset = response.data;

				// Custom reports engine will treat the data set as new one, if data set GUID is changed.

				var modalInstance = $uibModal.open({
					templateUrl : '../reportsplus/manager/dataSetClone.jsp',
					controller : [
						"$scope","dSet",
						function($scope,dSet) {
							$scope.dataSetName = dSet.dataSetName;
							dSet.dataSetName = dSet.dataSetName + " - Copy";
							$scope.dSet = dSet;
							$scope.cloneDataset = function(){
								$scope.$close($scope.dSet);
							}
						}],
					size : 'md',
					backdrop : 'static',
					resolve:{
						dSet : function(){
							return {
								dataSetName : dataset.dataSet.dataSetName,
								description : dataset.description
							}
						}
					}

				});
				modalInstance.result.then(function(dSet) {
					if(dSet){
						dataset.dataSet.dataSetName = dSet.dataSetName;
						dataset.guid = customReportSvc.generateUUID();
						dataset.description = dSet.description;
						reportService.saveDataSet(dataset).then(onFulfilled('Data Set ' + dataset.dataSet.dataSetName+'  cloned  successfully.'))
						.catch(onRejected);
					}
				});

			}).then().catch(onRejected);
		};
	}
})();
