(function() {
	"use strict";

	angular.module("reports").service("dataCubeService", dataCubeService);

	function dataCubeService() {
		/**
		 * get handler id
		 * 
		 * @param {object}
		 *            dataSet - data cube dataset information
		 * @return {handlerId} Returns handlerId for given data set
		 */
		function _getId(dataSet) {
			if (dataSet.hasOwnProperty("dCubeDataSet")) {
				return dataSet.dCubeDataSet.dsHandler['handlerId']; // if needed
				// change
				// the logic
				// later to
				// make sure
				// uniqueness
			} else {
				return dataSet.dataSet.dataSetName;
			}
		}

		function _getSearchUrl(dataSet, cacheId, isPOSTApi, reportInputs) {
			if (dataSet.hasOwnProperty("dCubeDataSet")) {
				const dataApi = dataSet.dCubeDataSet.dataApi;
				if(dataApi) {
					const {dsId,dsHandler,...remaining} = dataSet.dCubeDataSet;
					let inputs = {dsId,...dsHandler};
					if(reportInputs && reportInputs.length) {
						reportInputs.forEach(input => {
							if(!Object.keys(inputs).includes(input.id))
								inputs[input.id] = input.value; 
						});
					}

					const separator = dataApi.trim().startsWith("/") ? "" : "/";
					const formattedApi = cvFormatters.formatString(dataApi,inputs);
					return `${cvUtil.getContextPath()}/api${separator}${formattedApi}`;
				} else {
					var api = cvUtil.getContextPath() + "/api/dcube/handler/{0}/{1}";
					if (dataSet.dCubeDataSet.dsType === "federated" && dataSet.dCubeDataSet.dsId == -1) {
						api = cvUtil.getContextPath() + "/api/dcube/dynamicfederated/{0}/{1}/{2}";
						return cvFormatters.formatString(api, [
								dataSet.dCubeDataSet.dsHandler['entityTypeId'],
								dataSet.dCubeDataSet.dsHandler['entityId'],
								dataSet.dCubeDataSet.dsHandler['handlerName'] ]);
					} else if (dataSet.dCubeDataSet.dsType === "federated") {
						api = cvUtil.getContextPath() + "/api/dcube/federated/{0}/{1}";
					}
					return cvFormatters.formatString(api, [
							dataSet.dCubeDataSet.dsHandler['handlerId'],
							dataSet.dCubeDataSet.dsHandler['handlerName'] ]);
				}
				
			} else {
				if (isPOSTApi) {
					return "retrieveDataSetByPost.do";
				} else {
					return "retrieveDataSetByGet.do";
				}
			}
		}

		function _getAdditionParamsObj(dataSet, cacheId, isPOSTApi) {
			var paramObj = {};
			if (!dataSet.hasOwnProperty("dCubeDataSet")) {
				paramObj.cacheId = cacheId;
				if (isPOSTApi) {
					paramObj.dataSet = JSON.stringify(dataSet);
				} else {
					paramObj.dataSetName = dataSet.dataSet.dataSetId;
				}
			} else {
				// temp change
				paramObj.useDCubeReq = true;
				var dCubeSet = dataSet.dCubeDataSet;
				if (dCubeSet && dCubeSet.dsType === "federated" && dCubeSet.dsId == -1 && dCubeSet.dsHandler.entityId == -1) {
					paramObj.invalidSearchUrl = true;
				}
			}

			return paramObj;
		}

		/*
		 * options = { measureDataField, dimensionDataField, sortOptions} returns tableParams object
		 */
		function convertToTableParams(measureDataField, dimensionDataField, sortOptions, customFilterQuery, offset) {
			var sort = [];
			// limit taken from cvMapBox.js
			var limit = 999999;
			// Only sort on the basis of the first field in x-axis
			// in case of multiple fields in Y-axis, client side disables the
			// sort option for y-axis
			if (sortOptions) {
				if (sortOptions.sortAxis == "XAxis") {
					sortOptions.columnId = dimensionDataField[0].column;
					sort.push(sortOptions);
				} else if (sortOptions.sortAxis == "YAxis") {
					sortOptions.columnId = measureDataField[0].column;
					sort.push(sortOptions);
				}
			}
			angular.forEach(measureDataField, function(measure) {
				measure.dataField = measure.column;
			});
			angular.forEach(dimensionDataField, function(dimention) {
				dimention.dataField = dimention.column;
			});
			var limit = dimensionDataField[0].numPointsToDisplay.includeAll ? limit
					: dimensionDataField[0].numPointsToDisplay.maxPoints;
			var columns = dimensionDataField;
			columns = columns.concat(measureDataField);
			var tableParams = {
				columns : columns,
				offset : offset ? offset : 0,
				sort : sort,
				limit : limit,
				customFilterQuery
			};
			return tableParams;
		}

		/*
		 * Converts a single record object to an array of member elements This is required for chart input
		 */
		function convertRecordsToArray(record, columns) {
			var returnRecord = [];
			for ( var column in columns) {
				returnRecord.push(record[columns[column].dataField]);
			}
			return returnRecord;
		}

		/*
		 * Converts the search response to chart input format. This function iterates through all the records
		 * and convert the json object to an array
		 */
		function convertResponseToChartFormat(result) {
			var records = result.records, columns = result.columns, counter = 0;
			while (counter < records.length) {
				var record = records[counter];
				record = convertRecordsToArray(record, columns);
				records[counter] = record;
				counter++
			}
			result.records = records;
			return result;
		}

		this.init = function(dataSet, cacheId, isPOSTApi, inputs) {

			qb.rv.init(_getId(dataSet), {
				searchUrl : _getSearchUrl(dataSet, cacheId, isPOSTApi,inputs),
				schema : dataSet.fields,
				fieldMapping : dataSet.fieldMapping,
				additionalParams : _getAdditionParamsObj(dataSet, cacheId, isPOSTApi)
			});
		}

		/*
		 * options = { dataSet, measureDataField, dimensionDataField, sortOptions, filters }
		 */
		this.getChartData = function(options, callback) {
			var isObjFormat = false;
			if (options.componentType === "PIVOT_TABLE" ||
					(options.chartType === "heatmap" || options.chartType === "treemap")) {
				isObjFormat = true;
			}
			var isAggr = false;
			angular.forEach(options.measureDataField, function(measure) {
				if (measure.aggrType != 'None') {
					isAggr = true;
					return;
				}
			});
			if (isAggr) {
				const otherOptions = {
					sortOpts: getDeepCopy(options.sortOptions),
					isRespInObjFmt: isObjFormat,
					filtersMap: getDeepCopy(options.filters),
					constructGroupStructure: options.constructGroupStructure,
					groups: getDeepCopy(options.groups),
					isFlat: false,
					isPageBuilder: false,
					isAVoidFQ: false,
					doNotExclude: options.doNotExclude,
					inputParams: options.inputParams,
					customFilterQuery: options.customFilterQuery,
					skipAllFilters: false
				};
				qb.rv.doFacet(_getId(options.dataSet),
						getDeepCopy(options.measureDataField),
						getDeepCopy(options.dimensionDataField),
						otherOptions,
						callback);
			} else {
				qb.rv.doSearch(_getId(options.dataSet), convertToTableParams(getDeepCopy(options.measureDataField),
						getDeepCopy(options.dimensionDataField),
						getDeepCopy(options.sortOptions), options.customFilterQuery, options.offset), getDeepCopy(options.filters), options.inputParams, function(
						resp, reqObj) {
					callback(convertResponseToChartFormat(resp), reqObj);
				});
			}
		}

		this.getChartDataWithoutFacetReq = (options, callback) => {
			const otherOptions = {
				sortOpts: getDeepCopy(options.sortOptions),
				filtersMap: getDeepCopy(options.filters),
				reqParams: options.reqParams
				
			};
			qb.rv.getChartData(_getId(options.dataSet),
			getDeepCopy(options.measureDataField),
			getDeepCopy(options.dimensionDataField),
			otherOptions,
			callback);
		}

		/*
		 * options = { dataSet, tableParams }
		 */

		this.getTableData = function(options, callback) {
			qb.rv
					.doSearch(_getId(options.dataSet),
							options.tableParams,
							options.filters,
							options.inputParams,
							callback);
		};

		/*
		 * options = { isAllColumns | measureDataField }
		 */

		this.getHitsPanelData = function(options, callback) {
			// in case of allColumns and there are no other fields dropped.
			if (options.isAllColumns && $.isEmptyObject(options.measureDataField)) {
				qb.rv.doSearch(_getId(options.dataSet), {
					columns : [],
					limit : 0,
				}, options.filters, options.inputParams, function(resp) {
					callback([ resp.totalRecordCount ]);
				});
			} else if (!$.isEmptyObject(options.measureDataField)) {
				var measureDataField = getDeepCopy(options.measureDataField);
				var dimensionDataField = getDeepCopy(options.measureDataField);
				angular.forEach(measureDataField, function(measure, index) {
					if (measure.aggrType === "Top" || measure.aggrType === "Bottom") {
						var dimension = getDeepCopy(measure);
						dimension = {
							column : measure.column,
							numPointsToDisplay : {
								maxPoints : 1
							}
						};
						var direction = measure.aggrType === "Top" ? "desc" : "asc";
						measure.aggrType = "Count";
						var sortField = measure.aggrType + "_" + measure.column;
						dimension.sort = {
							"sortField" : sortField,
							"sortDirection" : direction
						};
						measure.aggrTypes = [ "Count" ];
						dimension.measureDataFields = [ measure ];
						dimensionDataField[index] = dimension;
					}
				});
				const otherOptions = {
					sortOpts: undefined,
					isRespInObjFmt: false,
					filtersMap: options.filters,
					constructGroupStructure: false,
					groups: [],
					isFlat:true,
					isPageBuilder: false,
					isAVoidFQ: false,
					doNotExclude: options.doNotExclude,
					inputParams: options.inputParams,
					customFilterQuery: {},
					skipAllFilters: options.skipAllFilters
				};
				qb.rv.doFacet(_getId(options.dataSet),
						measureDataField,
						dimensionDataField,
						otherOptions,
						function(resp) {
							var response = {};
							if (resp && !$.isEmptyObject(resp)) {
								response["totalRecords"] = resp.count;
								angular.forEach(resp, function(respObj, key) {
									if (key !== 'count') {
										response[key] = respObj.count !== undefined ? respObj.count : respObj;
									}
								});
							}
							callback(response);
						});
			}
		};

		/*
		 * options = { dataSet, isAllColumns, field, fieldType, searchquery }
		 */
		this.doSearch = function(options, callback) {
			const searchFields = options.searchFields ? options.searchFields : [];
			qb.rv.setKeyword(_getId(options.dataSet), options.searchquery,searchFields);
			
			// TODO: check for error handling in query builder
			callback(true);
		};

		/*
		 * options = { dataSet, measureDataField, dimensionDataField, sortOptions }
		 */
		this.getMapData = function(options, callback) {
			var isAggr = false;
			angular.forEach(options.measureDataField, function(measure) {
				if (measure.aggrType != 'None') {
					isAggr = true;
					return;
				}
			});
			if (isAggr) {
				const otherOptions = {
					sortOpts: getDeepCopy(options.sortOptions),
					isRespInObjFmt: options.isObjFormat,
					filtersMap: getDeepCopy(options.filters),
					constructGroupStructure: options.constructGroupStructure,
					groups: getDeepCopy(options.groups),
					isFlat: false,
					isPageBuilder: false,
					isAVoidFQ: false,
					doNotExclude: options.doNotExclude,
					inputParams: options.inputParams,
					customFilterQuery: {},
					skipAllFilters: false
				};
				qb.rv.doFacet(_getId(options.dataSet),
						getDeepCopy(options.measureDataField),
						getDeepCopy(options.dimensionDataField),
						otherOptions,
						callback);
			} else {
				qb.rv.doSearch(_getId(options.dataSet),
						options.tableParams,
						options.filters,
						options.inputParams,
						callback);
			}
		}

		/**
		 * to get data for facet component
		 * 
		 * @param {Object}
		 *            options options : dataSet, filterDataField, sortOptions, filters
		 * @param {Function}
		 *            callback
		 */
		this.getFacetData = function(options, callback) {
			var sortOptions = {
				direction : "Desc",
				sortAxis : "XAxis"
			};
			const otherOptions = {
				sortOpts: undefined,
				isRespInObjFmt: false,
				filtersMap: getDeepCopy(options.filters),
				constructGroupStructure: false,
				groups: getDeepCopy(options.groups),
				isFlat: options.isFlat,
				isPageBuilder: options.isPageBuilder,
				isAVoidFQ: options.isAvoidFQ,
				doNotExclude: options.doNotExclude,
				inputParams: options.inputParams,
				customFilterQuery: options.customFilterQuery,
				skipAllFilters: false
			};
			qb.rv.doFacet(_getId(options.dataSet),
					getDeepCopy(options.filterDataField),
					getDeepCopy(options.filterDataField),
					otherOptions,
					callback);
		}
		/*
		 * options = { dataSet, field, rangeType (absolute| relative), range: (for absolute { from, to}) (for
		 * relative directly) }
		 */

		this.applyDateRange = function(options, callback) {
			var dtFilterObj = {};
			dtFilterObj[options.field] = {
				"include" : options.range
			};
			qb.rv.setFilterMap(_getId(options.dataSet), dtFilterObj, options.fqAlias);
			callback(true);
		};
		/*
		 * options = { dataSet, filters, isExact }
		 */
		this.applyFilters = function(options, callback) {
			qb.rv.setFilterMap(_getId(options.dataSet),
					options.filters,
					options.fqAlias,
					options.isExact,
					options.filterType);
			callback(true);
		};

		/*
		 * options = { dataSet, fieldName}
		 */
		this.getHierarchicalField = function(options) {
			return qb.rv.getHierarchicalField(_getId(options.dataSet), options.fieldName);
		};
		/*
		 * options = { dataSet, fieldName}
		 */
		this.getSLevelField = function(options) {
			return qb.rv.getSLevelField(_getId(options.dataSet), options.fieldName);
		};
		this.getActualFieldName = function(options) {
			return qb.base.getFieldName(_getId(options.dataSet), options.fieldName);
		};
		/*
		 * options = { operationType: 'get' / 'search' initialValues: array of integer (in case of get),
		 * searchQuery (in case of search otherwise empty string), start: (for pagination), offset: (for
		 * pagination) }
		 */
		this.getXaxisCategories = function(options, callback) {

			var tempDataForTesting = [];
			var tempDataForPagination = [];
			var dataForSearching = [];
			angular.forEach(options.initialValues, function(values, index) {
				tempDataForTesting.push({
					'value' : values,
					'count' : index + 1
				});
			});
			tempDataForPagination = $.extend(true, [], tempDataForTesting);
			callback(tempDataForTesting);
		}

		/**
		 * get custom query results for custom html component
		 */
		this.getCustomData = function(options, callback) {
			qb.rv.doQuery(_getId(options.dataSet), options.customQuery, options.inputParams, function(data) {
				callback(data);
			});
		};

		/**
		 * to get data by using some facet functions without doing any bucketing
		 * 
		 * @param {Object}
		 *            options measureList
		 * @param {Function}
		 *            callback
		 * @return
		 */
		this.getFacetFunction = function(options, callback) {
			qb.rv.doFacetFunction(_getId(options.dataSet), options.measureDataFields, options.filters, callback);
		}

		this.getDateFormatAndTimeZone = (cellExpression) => {			
			let DateFormat = {};
			let timeZone = {};
			DateFormat["key"] = "clientDateTime";
			let fmt = cellExpression.fmt;
			if (fmt === 'AutoDateTime') {
				fmt = "MMM dd, YYYY hh:mm:ss a";
			} else if (fmt === 'Auto') {
				fmt = "MMM dd, YYYY";
			} else if (fmt === 'Custom') {
				fmt = cellExpression.customDateFormat;
			}
			DateFormat["value"] = fmt;
			timeZone["key"] = "clientTimezone";
			timeZone["value"] = moment.tz.guess(); //destination time zone will be always browser's timezone
			return {
				format: DateFormat,
				timeZone 
			}
		}

		function getDeepCopy(obj) {
			if (!obj) {
				return obj;
			}
			var isArray = false;
			if (Array.isArray(obj)) {
				isArray = true;
			}
			var copyObj = $.extend(true, isArray ? [] : {}, obj);
			return copyObj;
		}
	}
})();
