// Closure to isolate scope, as per best practices
(function() {
	"use strict";

	// Get a reference to the reportsBuilder module
	var app = angular.module("reports");

	// Directive to trigger function on press of 'enter' key
	// Currently in use for table filters and some of the property elements
	app.directive('ngEnter', function() {
		return function($scope, element, attrs) {
			element.on("keydown keypress", function(event) {
				if (event.which === 13) {
					$scope.$apply(function() {
						$scope.$eval(attrs.ngEnter);
					});

					event.preventDefault();
				}
			});
		};
	});

	// Directive to focus when showing an input
	app.directive('focusOnShow', function($timeout) {
		return {
			restrict : 'A',
			link : function($scope, $element, $attr) {
				if ($attr.ngShow) {
					$scope.$watch($attr.ngShow, function(newValue) {
						if (newValue) {
							$timeout(function() {
								$element[0].focus();
							}, 0);
						}
					})
				}
				if ($attr.ngHide) {
					$scope.$watch($attr.ngHide, function(newValue) {
						if (!newValue) {
							$timeout(function() {
								$element[0].focus();
							}, 0);
						}
					})
				}

			}
		};
	})

	// Directive to make elements stick to a certain point when scrolling
	// Note: The stickToTop function could be implemented directly in the link
	// function of the directive, however this makes it cleaner and easier to
	// isolate. Do not remove function wrapper.
	app.directive("stickyScroll", function() {
		return {
			restrict : "A",
			scope : {
				stickyScrollLimit : "="
			},
			link : function($scope, element, attrs) {
				/**
				 * Force a div to stick at a certain point when scrolling
				 * 
				 * @param {String}
				 *            element CSS selector for desired element; element must be position:
				 *            fixed|absolute|static
				 * @param {Number}
				 *            limit Point at which element should stick during scrolling
				 */
				$scope.stickToTop = function(limit) {
					// Cache the initial scroll state in case script is called
					// when page is already partially scrolled
					var initScroll = angular.element(window).scrollTop();
					angular.element(window).scrollTop(0);

					// Get the element and its distance from the top
					var $el = angular.element(element);
					var target = $el.offset().top;

					// Reset scroll state
					angular.element(window).scrollTop(initScroll);

					angular.element(window).on("scroll",function(e) {
						var scrollTop = angular.element(this).scrollTop();

						// Position element between limit and target depending on scroll
						if (scrollTop > target) {
							$el.css("top", limit + "px");
						}
						if (scrollTop < target) {
							$el.css("top", Math.max(limit, Math.min(target, target - scrollTop)) + "px");
						}
					});
				};

				$scope.stickToTop($scope.stickyScrollLimit);
			}
		};
	}); // end directive stickyScroll

	// Directive for add the HTML in the cell and do JavaScript expressions
	app.directive('cvReportCustomization', [
			'customReportSvc',
			function(customReportSvc) {
				return {
					restrict : 'AEC',
					replace : false,
					scope : {
						elementData : '@cvElementData',
						expression : '@cvExpression',
						allCellExpression : '@cvAllCellExpression',
						tagAttributes : '@cvTagAttributes',
						showTitleTooltip : '@cvShowTitleTooltip',
						textOnly : '@cvTextOnly',
						localize : '@cvLocalize',
						// For Table Component only
						isTableCustomization : '@cvTableCustomization',
					},
					controller : [
							"$scope",
							"customReportSvc",
							"$element",
							function($scope, customReportSvc, $element) {
								$scope.isTableCustomization = $scope.isTableCustomization !== undefined &&
										$scope.isTableCustomization != null && $scope.isTableCustomization
								if ($scope.isTableCustomization) {
									$scope.rowIndex = $scope.$parent.$parent.$index;
									$scope.columnIndex = $scope.$parent.$index;
									$scope.row = $scope.$parent.$data[$scope.rowIndex];
									$scope.cellData = $scope.row[$scope.$parent.column.id];
									$scope.$watch('cellData', function() {
										customReportSvc.cvReportCustomization($scope, $element);
									});
								} else {
									$scope.$watch('expression', function() {
										customReportSvc.cvReportCustomization($scope, $element);
									});
								}
							} ]
				};
			} ]); // End of cvReportCustomization directive

	app.directive('resizeColumn', [ '$timeout', function($timeout) {
		var MIN_COL_WIDTH = 40;

		/**
		 * Ensure that columns are at least MIN_COL_WIDTH pixels wide; if the table allows, let them expand
		 * accordingly
		 * 
		 * @param {jQuery}
		 *            $cols collection of columns in need of minimum width
		 */
		function setColMinWidths($cols) {
			$cols = $cols.filter(":visible");

			/// Remove any previously applied minimum widths to allow the table
			/// to try and set its own layout
			$cols.removeClass("column-min-width")

			/// Once the table has attempted rendering, expand columns as
			/// needed to accommodate the minimum width requirement, marking
			/// which columns have been modified
			$cols.each(function() {
				if ($(this).outerWidth() < MIN_COL_WIDTH) {
					$(this).addClass("column-min-width");
				}
			});
		} /// end method setColMinWidths

		return {
			restrict : "A",
			link : function($scope, elem) {
				/// When the user clicks on the resize handle, begin resizing
				/// the column
				$(elem).on("mousedown", function(e) {
					/// Stop event propagation to prevent text selection
					e.preventDefault();
					e.originalEvent.preventDefault();

					/// Save initial mouse position to compute change in
					/// width
					var initial_x = e.pageX,

					/// Get a reference to the table header element
					$col = $(this).parent(),

					/// Left-most allowed point
					min_x = $col.offset().left + MIN_COL_WIDTH,

					/// Save initial width of column
					initial_width = $col.width(),

					/// Column resize marquee element
					$marquee = $("<div class='column-resize-marquee'></div>").appendTo("body").css({
						height : $col.outerHeight()
					}),

					/**
					 * Position the resize marque
					 * 
					 * @param {Number}
					 *            left pixels from the left to position the marquee
					 */
					positionMarquee = function(left) {
						$marquee.css({
							top : $col.offset().top - $(window).scrollTop(),
							left : Math.max(min_x, left)
						});

					}, /// end method positionMarquee

					/**
					 * Update column width while dragging the resize handle
					 * 
					 * @param {Event}
					 *            e mousemove event
					 */
					mousemove = function(e) {
						/// Stop event from bubbling
						e.preventDefault();
						e.stopPropagation();

						positionMarquee(e.pageX);
					}; /// end handler mousemove

					/// Position the marquee initially
					positionMarquee(e.pageX);

					/// Bind mouse movement event handler
					$(window).on("mousemove", mousemove);

					/// On mouse release, unbind column resize event handler
					/// and set the new width
					$(window).one("mouseup", function(e) {
						$(window).off("mousemove", mousemove);

						/// Set the new width on the column element
						$col.removeClass("column-min-width").width(
						/// Compute new width based on initial width and the
						/// change in mouse location; only use this new
						/// width if it is at least the minimum
						Math.max(initial_width + e.pageX - initial_x, MIN_COL_WIDTH));

						/// Make sure every column is at least MIN_COL_WIDTH;
						/// don't reset $col's width, since it was just set
						/// manually
						setColMinWidths($col /// the current th being resized
						.parent() /// the containing thead
						.children() /// the set of all th in the thead
						);

						$marquee.remove();
					});
				}); /// end handler mousedown

				/// On initial page load, apply the width limit to all columns
				/// each time any column is added/rendered
				setColMinWidths($(elem) /// the resize handle
				.parent() /// the current th being resized
				.parent() /// the containing thead
				.children() /// the set of all th in the thead
				);
			} /// end link method
		}; /// end returned directive object
	} ]); /// end directive resizeColumn

	app.directive('cvBindHtml', [ '$sce', function($sce) {
		return {
			restrict : 'A',
			link : function($scope, element, attr) {
				$scope.$watch(attr.cvBindHtml, function(value) {
					if (value) {
						value = $sce.trustAsHtml(value);
					}
					element.html(value);
				});
			}
		}
	} ]);

	app.directive('cvModalDraggable', [ '$sce', function($sce) {
		return {
			restrict : 'A',
			link : function($scope, element, attr) {
				if (customReports.appName === "webconsole") {
					element.closest('.modal-dialog').draggable();
				}
			}
		}
	} ]);

	// Allows dragging modal dialog only when the cursor is on the modal dialog's header.
	app.directive('cvModalHeaderDraggable', function() {
		return {
			restrict : 'A',
			link : function(scope, element) {
				if (customReports.appName === "webconsole") {
					element.closest('.modal-dialog').draggable({
						handle : '.modal-header'
					});
				}
			}
		};
	});

	app.directive('cvModalResizable', [ '$sce', function($sce) {
		return {
			restrict : 'A',
			link : function($scope, element, attr) {
				if (customReports.appName === "webconsole") {
					element.closest('.modal-content').resizable()
				}
			}
		}
	} ]);

	app.directive('cvDraggable', function() {
		return {
			restrict : 'A',
			link : function(scope, element, attrs) {
				if(customReports.appName === "webconsole"){
					if (!attrs.dragParent) {
						element.draggable({
							handle : attrs.dragHandler
						});
					} else {
						element.closest(attrs.dragParent).draggable({
							handle : attrs.dragHandler
						});
					}
				}
			}
		}
	});
	app
			.directive('inplace',
					function() {
						return {
							restrict : 'AE',
							replace : true,
							scope : {
								model : '=',
								placeholder : '@placeholder',
								removeFn : '=onDelete'
							},
							template : '<span>'
									+ '<span class="c1" ng-hide="editorEnabled"'
									+ 'ng-click="enableEditor();" cv-report-customization cv-show-title-tooltip="true" cv-expression="{{model}}" cv-localize="true"></span>'
									+ '<input ng-show="editorEnabled" placeholder="{{placeholder}}" ng-model="editModel"'
									+ 'ng-required ng-blur="finishedEdit()"'
									+ 'ui-event="{\'blur\': \'finishedEdit()\'}"/>' + '</span>',
							// The linking function will add behavior to the template
							link : function(scope, element, attrs) {
								scope.editorEnabled = false;

								if ((typeof scope.model === 'undefined' || scope.model == "") &&
										scope.$parent.reportMode == 'builder') {
									scope.editorEnabled = true;
								}

								scope.$watch('model', function() {
									scope.editModel = angular.copy(scope.model);
								});
								scope.$watch('editModel', function() {
									scope.model = angular.copy(scope.editModel);
								});
								scope.finishedEdit = function() {
									scope.model = angular.copy(scope.editModel);
									if (!scope.model) {
										scope.editorEnabled = true;
									} else {
										scope.editorEnabled = false;
									}
								};

								scope.enableEditor = function() {
									if (scope.$parent.reportMode != 'builder') {
										return;
									}

									//scope.clearDefaultTitle();
									scope.editModel = angular.copy(scope.model);
									scope.editorEnabled = true;
									setTimeout(function() {
										element.find('input')[0].focus();
										//element.find('input').focus().select(); // w/ jQuery
									});
								};

								scope.clearDefaultTitle = function() {
									var component = scope.$parent.component;
									if (component.defaultTitle && component.title.text === component.defaultTitle) {
										component.title.text = "";
										scope.editModel = "";
										scope.model = angular.copy(scope.editModel);
										// $(element.find('input')[0]).val("");
									}
								};
							}

						}
					});

	app.filter('excludeFrom', [ function() {
		return function(array, currentOption, sorting) {
			return array.filter(function(item) {
				var retValue = true;
				angular.forEach(sorting, function(value, key) {
					if (value.columnId == item.id && value.columnId != currentOption) {
						retValue = false;
					}
				});

				return retValue;
			});
		};
	} ]);

	app.directive('cvColDraggable', function() {
		return {
			link : function(scope, elem, attr) {

				elem.attr("draggable", true);
				var dragDataVal = '';
				var draggedGhostImgElemId = '';
				attr.$observe('dragdata', function(newVal) {
					dragDataVal = newVal;

				});

				attr.$observe('dragimage', function(newVal) {
					draggedGhostImgElemId = newVal;
				});

				elem.on("dragstart", function(e) {
					var sendData = dragDataVal;
					e.originalEvent.dataTransfer.setData("Text", sendData);
					e.originalEvent.dataTransfer.setData("Type", "col-reorder"); // column - reorder

					/*
					 * if (attr.dragimage !== 'undefined') { e.originalEvent.dataTransfer.setDragImage(
					 * document.getElementById(draggedGhostImgElemId), 0, 0 ); }
					 */

					/*
					 * var dragFn = attr.drag; if (dragFn !== 'undefined') { scope.$apply(function() {
					 * scope[dragFn](sendData); }) }
					 */

				});
			}
		}
	});

	app.directive('cvRptSortable', function() {
		return {
			controller : [ '$scope', '$attrs',function($scope, $attrs) {
				var listModel = null;
				$scope.$watch($attrs.cvRptSortable, function(sortable) {
					listModel = sortable;
				});
				this.move = function(fromIndex, toIndex) {
					listModel.splice(toIndex, 0, listModel.splice(fromIndex, 1)[0]);
				};
			}]
		};
	});

	app.directive('cvRptSortableItem', function() {
		return {
			restrict : 'A',
			require : '^cvRptSortable',
			link : function(scope, element, attrs, controller) {
				var index = scope.$index;
				scope.$watch(attrs.cvRptSortableItem, function(newIndex) {
					index = newIndex;
				});
				attrs.$set('draggable', true);

				var wrappedListeners = {
					dragstart : function(e) {
						e.originalEvent.dataTransfer.effectAllowed = 'move';
						e.originalEvent.dataTransfer.dropEffect = 'move';
						e.originalEvent.dataTransfer.setData('application/json', index);
						element.addClass('dragging');
					},
					dragend : function(e) {
						element.removeClass('dragging');
					},
					dragenter : function(e) {
					},
					dragleave : function(e) {
						element.removeClass('hover');
					},
					drop : function(e) {
						e.preventDefault();
						e.stopPropagation();
						var sourceIndex = e.originalEvent.dataTransfer.getData('application/json');
						controller.move(sourceIndex, index);
						element.removeClass('hover');
					}
				};

				var unwrappedListeners = {
					dragover : function(e) {
						e.preventDefault();
						element.addClass('hover');
					},
					mouseenter : function() {
						element.addClass('hover');
					},
					mouseleave : function() {
						element.removeClass('hover');
					}
				};

				angular.forEach(wrappedListeners, function(listener, event) {
					element.on(event, wrap(listener));
				});

				angular.forEach(unwrappedListeners, function(listener, event) {
					element.on(event, listener);
				});

				function wrap(fn) {
					return function(e) {
						scope.$apply(function() {
							fn(e);
						});
					};
				}

			}
		}
	});

	app.directive('cvColDroppable', [ '$parse', function($parse) {
		return {
			link : function(scope, element, attr) {
				function onDragOver(e) {

					if (e.preventDefault) {
						e.preventDefault();
					}

					if (e.stopPropagation) {
						e.stopPropagation();
					}
					e.originalEvent.dataTransfer.dropEffect = 'move';
					return false;
				}

				function onDrop(e) {
					var type = e.originalEvent.dataTransfer.getData("Type");
					if (!type) {
						return;
					}
					if (e.preventDefault) {
						e.preventDefault();
					}
					if (e.stopPropagation) {
						e.stopPropagation();
					}
					var data = e.originalEvent.dataTransfer.getData("Text");

					data = angular.fromJson(data);

					var dropfn = attr.drop;
					var fn = $parse(attr.drop);
					scope.$apply(function() {

						scope[dropfn](data, e.target);
					});

				}

				element.on("dragover", onDragOver);
				element.on("drop", onDrop);

			}
		};

	} ]);

	app
			.directive('cvMultiCheckBox',
					function() {
						return {
							restrict : 'AE',
							scope : {
								options : '=',
								model : '=',
								hideAllCb : '=',
								isMulti : '=',
								allSelected : '=?'
							},
							template : '<ul class="multiCheckBox">' +
									'<li class="hideAllCb" data-ng-click="selectAll()" data-ng-hide="hideAllCb || !isMulti"><div class="custom-cb"  data-ng-class="{\'checked\':allSelected , \'unchecked\':!allSelected}" ></div><span class="custom-cb-label">' +
									localMsg['customReport']['All'] +
									'</span></li>' +
									'<li data-ng-repeat="option in options" data-ng-click="setSelectedItem()"><div class="custom-cb "data-ng-class="isChecked(option.value)"></div><span class="custom-cb-label">{{option.label}}</span></li>' +
									'</ul>',
							controller : [ '$scope', function($scope) {

								$scope.selectedLabel = localMsg['customReport']['Select'];
								$scope.allSelected = false;

								$scope.selectAll = function() {
									if ($scope.allSelected) {
										$scope.model = [];
										$scope.allSelected = false;
									} else {
										$scope.model = $.map($scope.options, function(o) {
											return o["value"];
										});
										$scope.allSelected = true;
									}
									if (Object.keys($scope.$parent.dependentInputs).length > 1) {
										$scope.$parent.reloadDependendentInputs($scope.$parent.input.id);
									}
								};

								$scope.setSelectedItem = function() {
									var id = this.option.value;
									if ($scope.isMulti) {
										var idx = $scope.model.indexOf(id);
										if (idx !== -1) {
											$scope.model.splice(idx, 1);
										} else {
											$scope.model.push(id);
											$scope.checkAllSelected();
										}
									} else {
										$scope.model = [];
										$scope.model.push(id);
									}

									if (Object.keys($scope.$parent.dependentInputs).length > 1) {
										$scope.$parent.reloadDependendentInputs($scope.$parent.input.id);
									}
								}

								$scope.checkAllSelected = function() {
									if ($scope.model) {
										if ($scope.model.length == $scope.options.length) {
											$scope.allSelected = true;
										} else {
											$scope.allSelected = false;
										}
									}
								}

								$scope.$watch('model', function() {
									$scope.checkAllSelected();
								});

								$scope.isChecked = function(id) {
									if ($scope.model.indexOf(id) !== -1) {
										return 'checked';
									}
									return 'unchecked';
								};

							} ]
						}
					});

	app
			.directive('cvMultiSelect',
					function() {
						return {
							restrict : 'AE',
							scope : {
								options : '=',
								model : '=',
								reference : '=',
								isMulti : '=',
								allSelected : '=?'
							},
							template : '<div data-ng-click="open=!open;openDropdown()" class="btn-group dropdown-trigger select-label" data-ng-class="{open: open}">' +
									'<span class="cv-multi-select-items" title="{{selectedLabel}}">{{selectedLabel}}' +
									'<span class="sprite icon-arrow-dropdown hideOnExportFriendly" alt="down" ></span></span>' +
									'<span class="dropdown-menu">' +
									'<div> <input type="text" class="searchTextBox" placeholder="Search..." data-ng-click="handleSearchText()" data-ng-model="search" data-ng-change="handleSearchText()" autofocus="autofocus" /></div>' +
									'<div data-ng-click="selectAll()" data-ng-hide="hideAllCheckBox || !isMulti" class="allCheckbox"><div class="custom-cb"  data-ng-class="{\'checked\':allSelected , \'unchecked\':!allSelected}" ></div><span class="custom-cb-label">' +
									localMsg['customReport']['SelectAll'] +
									'</span></div>' +
									'<ul id="customSelectOptions" class="customSelectOptions">' +
									'<li data-ng-repeat="option in options" data-ng-click="setSelectedItem()"><div class="custom-cb" data-ng-class="isChecked(option.value)" data-ng-hide="!isMulti"></div><span class="custom-cb-label">{{option.label}}</span><span alt="down" data-ng-class="{\'icon-selected\':option.selected}" ></span></li>' +
									'</ul>' + '</span></div>',
							link : function(scope, element, attr) {
								scope.open = false;
								scope.toggleSelect = function() {
									scope.open = !scope.open;
								}

								$(document).on('click', function(event) {
									var isClickedElementChildOfPopup = element.find(event.target).length > 0;

									if (isClickedElementChildOfPopup) {
										return;
									}

									if (scope.open) {
										if (Object.keys(scope.$parent.dependentInputs).length > 1) {
											scope.$parent.reloadDependendentInputs(scope.$parent.input.id);
										}
									}

									scope.$apply(function() {
										scope.open = false;
									});
								});
							},
							controller : [
									'$scope',
									function($scope) {
										$scope.selectedLabel = localMsg['customReport']['Select'];
										$scope.allSelected = false;
										$scope.originalOptions = [];

										$scope.openDropdown = function() {

										};

										$scope.handleSearchText = function() {
											if ($scope.originalOptions.length === 0) {
												angular.copy($scope.options, $scope.originalOptions); // keep a copy of original options
											}

											$scope.options = [];
											if ($scope.search) {
												$scope.hideAllCheckBox = true;
												angular.forEach($scope.originalOptions, function(val, i) {
													if (val.label.toString().toLowerCase().indexOf($scope.search
															.toLowerCase()) !== -1) {
														$scope.options.push(val);
													}
												});
											} else {
												$scope.hideAllCheckBox = false;
												angular.copy($scope.originalOptions, $scope.options); // keep a copy of original options
											}

											event.stopPropagation();
											return false;
										};

										$scope.selectAll = function() {

											if ($scope.allSelected) {
												$scope.model = [];
												$scope.allSelected = false;
											} else {
												$scope.model = $.map($scope.options, function(o) {
													return o["value"];
												});
												$scope.allSelected = true;
											}
											event.stopPropagation();
											return false;
										};

										$scope.setSelectedItem = function() {
											var id = this.option.value;
											if ($scope.isMulti) {
												var idx = $scope.model.indexOf(id);
												if (idx !== -1) {
													$scope.model.splice(idx, 1);
												} else {
													$scope.model.push(id);
												}
											} else {
												if ($scope.selectedOption) {
													$scope.selectedOption.selected = undefined;
												}
												$scope.model = [];
												this.option.selected = true;
												$scope.selectedOption = this.option;
												$scope.model.push(id);
												$scope.open = false;
											}
											$scope.setLabel();
											event.stopPropagation();
											return false;
										};

										$scope.setLabel = function() {
											if ($scope.originalOptions.length === 0) {
												angular.copy($scope.options, $scope.originalOptions); // keep a copy of original options
											}

											if ($scope.model) {
												if ($scope.model.length > 3 ||
														($scope.model.length > 0 && $scope.options.length === 0)) {
													$scope.selectedLabel = $scope.model.length + " Selected";
												} else {
													$scope.selectedLabel = ""; //$scope.model.join(",");
													angular
															.forEach($scope.originalOptions,
																	function(val, i) {
																		if ($scope.model
																				.indexOf($scope.originalOptions[i].value) !== -1) {
																			$scope.selectedLabel = $scope.selectedLabel +
																					$scope.originalOptions[i].label +
																					", ";
																		}
																	});
													$scope.selectedLabel = $scope.selectedLabel.substring(0,
															$scope.selectedLabel.lastIndexOf(","));
												}
												if ($scope.model.length == 0) {
													$scope.selectedLabel = localMsg['customReport']['Select'];
												}
												if (!$scope.hideAllCheckBox &&
														$scope.model.length == $scope.options.length) {
													$scope.selectedLabel = localMsg['customReport']['All'];
													$scope.allSelected = true;
												} else {
													$scope.allSelected = false;
												}
											}

										}

										$scope.$watch('model', function() {
											$scope.setLabel();
											$scope.originalOptions = [];
											angular.copy($scope.options, $scope.originalOptions); // keep a copy of original options
										})

										$scope.isChecked = function(id) {
											if ($scope.model.indexOf(id) !== -1) {
												return 'checked';
											}
											return 'unchecked';
										};
									} ]
						}
					});

	// toggle switch for checkbox
	app
			.directive('cvToggleSwitch',
					function() {
						return {
							restrict : 'A',
							link : function(scope, elem, attrs, controller) {
								elem.addClass('on-off-switch-checkbox');
								var outerWrapper = '<div class="on-off-switch"></div>';
								var parentElem = elem.parent();
								var toggleSwithLabel = '<label class="on-off-switch-label" for="' + attrs.id +
										'"><span class="on-off-switch-inner"></span><span class="on-off-switch-switch"></span></label>';
								var switchDom = angular.element(outerWrapper).append(elem);
								parentElem.append(switchDom.append(angular.element(toggleSwithLabel)));
							}
						};
					});

	// for label title
	app.directive('cvLabelTitle', function() {
		return {
			restrict : 'A',
			link : function(scope, elem, attrs) {
				var title = attrs.title;
				if (!title) {
					elem.attr('title', elem.text());
				}
			}
		}
	});

	/**
	 * minimal datetime range picker wrapper for bootstrap daterange picker
	 */
	app.directive('cvDateTimeRangePicker', [
			'customReportSvc',
			function(customReportSvc) {
				return {
					restrict : 'A',
					scope : {
						component : "="
					},
					link : function(scope, elem, attrs) {
						var dateTimeFormatString = "MMM DD YYYY";
						var start = "", end = "";
						var ranges = {};
						var options = {
							start : moment(start).format(dateTimeFormatString),
							end : moment(end).format(dateTimeFormatString),
							format : dateTimeFormatString,
							ranges : ranges,
							timePicker : true,
							showDropdowns : true
						};
						var abbrMap = {
							'm' : [ 'minute', 'minutes' ],
							'h' : [ 'hour', 'hours' ],
							'd' : [ 'day', 'days' ],
							'M' : [ 'month', 'months' ],
							'y' : [ 'year', 'years' ],
						}
						// default relative dates for component level filters
						var defaultRelativeTimes = [ {
							'label' : 'Last 15 Minutes',
							'value' : ">-15m"
						}, {
							'label' : 'Last 1 hour',
							'value' : ">-1h"
						}, {
							'label' : 'Last 12 hours',
							'value' : ">-12h"
						}, {
							'label' : 'Last 24 hours',
							'value' : ">-24h"
						}, {
							'label' : 'Last 7 days',
							'value' : ">-7d"
						}, {
							'label' : 'Last 1 month',
							'value' : ">-1M"
						}, {
							'label' : 'Last 1 year',
							'value' : ">-1y"
						}, ];
						function getDateRange(relative) {
							var length = relative.length;
							var range = {
								'start' : {},
								'end' : {}
							};
							var unit = relative[length - 1];
							var value = relative.substring(1, length - 1);
							if (relative[0] === '<') { // less than
								range.end = moment();
								range.start = moment().subtract(parseInt(value), abbrMap[unit][1]);
							} else if (relative[0] === '>') { // greater than
								range.end = moment();
								range.start = moment().add(parseInt(value), abbrMap[unit][1]);
							}
							return range;
						}
						function updateRelativeRange(isRedraw) {
							if (attrs.showRelative === "true") {
								if (scope.component.relativeTimes) {
									var relatives = scope.component.relativeTimes;
									var newRanges = {};
									angular.forEach(relatives, function(relative, index) {
										if (relative.label) {
											var range = getDateRange(relative.value);
											newRanges[relative.label] = [ range.start, range.end, relative.value ];
											return;
										}
									});
									options.ranges = newRanges;
									var rangeLabel = 'Custom Range';
									if (scope.component.rangeType === 'absolute') {
										options.start = moment(scope.component.dateRange.absolute.from);
										options.end = moment(scope.component.dateRange.absolute.to);
									} else {
										var range = getDateRange(scope.component.dateRange.relative);
										options.start = range.start;
										options.end = range.end;
										angular.forEach(options.ranges, function(range, label) {
											if (range[2] === scope.component.dateRange.relative) {
												rangeLabel = label;
												return;
											}
										});
									}
									$($(elem).children()[0]).daterangepicker(options, cb);
									if (isRedraw) {
										cb(options.start, options.end, rangeLabel);
									}
								}
							} else {
								var relatives = defaultRelativeTimes;
								var newRanges = {};
								angular.forEach(relatives, function(relative, index) {
									if (relative.label) {
										var range = getDateRange(relative.value);
										newRanges[relative.label] = [ range.start, range.end, relative.value ];
										return;
									}
								});
								options.ranges = newRanges;
								if (scope.component && scope.component.dateRange) {
									if (scope.component.dateRange.absolute && scope.component.dateRange.absolute.from &&
											scope.component.dateRange.absolute.to) {
										options.start = moment(scope.component.dateRange.absolute.from || undefined);
										options.end = moment(scope.component.dateRange.absolute.to || undefined);
										var start1 = moment(options.start).format(dateTimeFormatString);
										var end1 = moment(options.end).format(dateTimeFormatString);
							            $($(elem).children()[0]).val(start1 + " - " + end1);
							        } else if (scope.component.dateRange.relative) {
							        	var rangeLabel = undefined;
							        	angular.forEach(options.ranges, function(range, label) {
							        		if (range[2] === scope.component.dateRange.relative) {
							        			rangeLabel = label;
							        			return;
							        		}
							        	});
							        	$($(elem).children()[0]).val(rangeLabel);
							        } else {
							        	$($(elem).children()[0]).val("");
							        }
						        }
							$($(elem).children()[0]).daterangepicker(options, cb);
								// cb(options.start, options.end, "Custom Range");
							}
						}
						var cb = function(start, end, type, inputElmChanged) {
							if(inputElmChanged) {
								return
							}
							if ($.isEmptyObject(start) || $.isEmptyObject(end) || !start.isValid() || !end.isValid()) {
								$($(elem).children()[0]).val("");
								return;
							}
							var dateRangePicker = $($(elem).children()[0]).data("daterangepicker");
							dateRangePicker.setStartDate(start);
							dateRangePicker.setEndDate(end);
							if (type === 'Custom Range') {
								var start1 = moment(start).format(dateTimeFormatString);
								var end1 = moment(end).format(dateTimeFormatString);
								$($(elem).children()[0]).val(start1 + " - " + end1);
								if (typeof scope.component.dateRange.absolute != 'object') {
									scope.component.dateRange.absolute = {};
								}
								scope.component.dateRange.absolute.from = start.toDate();
								scope.component.dateRange.absolute.to = end.toDate();
								scope.component.rangeType = 'absolute';
								scope.component.dateRange.relative = "";
							} else {
								var rangeValue = "";
								angular.forEach(options.ranges, function(range, label) {
									if (label === type) {
										rangeValue = range[2];
									}
								});
								scope.component.dateRange.relative = rangeValue;
								dateRangePicker.hideCalendars();
								$($(elem).children()[0]).val(type);
								scope.component.rangeType = 'relative';
								scope.component.dateRange.absolute = {
									'from' : '',
									'to' : ''
								};
							}
							scope.component.dateRange.dispVal = $($(elem).children()[0]).val();
							if (attrs.showRelative === "true") {
								customReportSvc.triggerCallback('dateRangeUpdated', {}, scope.component.id);
							} else {
								customReportSvc.triggerCallback('dateRangeUpdated',
										scope.component.dateRange,
										scope.component.id + "_" + scope.component.column);
							}
							// console.log(start,end);
						}
						customReportSvc.registerCallback('relativeRangeUpdated', function(componentId) {
							if (scope.component && scope.component.id === componentId) {
								updateRelativeRange(false);
							}
						});
						updateRelativeRange(true);
					}
				}
			} ]);

	// directive to show elements with fixed positioning
	app.directive('cvShowOnTop', function() {
		return {
			restrict : 'A',
			link : function(scope, elem, attrs) {
				var dropDownElem = elem.children()[1];
				var buttonElem = elem.children()[0];
				angular.element(buttonElem).on('click', function() {
					var dropDownTop = $(this).offset().top + $(this).outerHeight() - $(window).scrollTop();
					$(dropDownElem).css('top', dropDownTop + "px");
					$(dropDownElem).css('position', "fixed");
				})
			}
		}
	});

	// filter to add underscore for spaces in id
	app.filter('formatId', function() {
		return function(componentId) {
			return (componentId || "").replace(/\s/g, "_");
		};
	});

})();
