import { DM2OwnerType, CapabilityType } from 'modules/tags/js/tags.constants.js';

export default class TagManagerUtility {
	// check if the passed capability type (int) is allowed for the given capabilities (long)
	static checkCapability(capabilityType, capabilities) {
		return TagManagerUtility.bitwiseAnd(capabilities, Math.pow(2, capabilityType - 1)) > 0;
	}

	// To support 64 bit AND operation
	static bitwiseAnd(v1, v2) {
		const hi = 0x80000000;
		const low = 0x7fffffff;
		const hi1 = ~~(v1 / hi);
		const hi2 = ~~(v2 / hi);
		const low1 = v1 & low;
		const low2 = v2 & low;
		const h = hi1 & hi2;
		const l = low1 & low2;
		return h * hi + l;
	}

	static formatTagsetResponse(data) {
		if (_.isUndefined(data) || !Array.isArray(data.listOftagSetList)) {
			return;
		}
		const tagsets = [];
		data.listOftagSetList.forEach(({ tagSetsAndItems }) => {
			const { container, tags } = tagSetsAndItems[0];
			const containerTags = [];
			const tagNames = [];
			if (Array.isArray(tags)) {
				tags.forEach(tag => {
					containerTags.push({
						tagId: tag.tagId,
						tagGuid: tag.id,
						name: tag.name
					});
					tagNames.push(tag.name);
				});
			}
			if (!_.isUndefined(container)) {
				const tagset = {
					containerId: container.containerId,
					containerGuid: container.containerGuid,
					name: container.containerName,
					fullName: container.containerFullName,
					description: container.comment,
					createTime: container.createTime,
					isMine: container.containerOwnerType === DM2OwnerType.Owner,
					ownerAlias: _.get(container, 'ownerInfo.aliasName'),
					ownerUserName: _.get(container, 'ownerInfo.userName'),
					tags: containerTags,
					tagsCsv: tagNames.join(','),
					isSystem: (container.attribute & 2) === 2,
					allowEdit: TagManagerUtility.checkCapability(CapabilityType.Edit, container.capabilities)
				};
				tagsets.push(tagset);
			}
		});
		return tagsets;
	}

	// Get an object of tag id and name mapping
	static getTagMapping(data) {
		const tagMapping = {};
		const tagsets = TagManagerUtility.formatTagsetResponse(data);
		if (Array.isArray(tagsets)) {
			tagsets.forEach(tagset => {
				if (Array.isArray(tagset.tags)) {
					tagset.tags.forEach(tag => {
						tagMapping[tag.tagGuid] = tag.name;
					});
				}
			});
		}
		return tagMapping;
	}
}
