import { ediscoveryAppModule } from 'common/js/modules';
import 'adminConsole/js/directives/acDirectives.js';
import getTagsGridColumns from 'modules/tags/js/columnTemplates/tags.column.template.js';
import 'modules/tags/js/services/tagManager.svc.js';
import 'modules/tags/js/controllers/addTag.ctrl.js';
import { DM2OwnerType, DM2EntityType, CapabilityType } from 'modules/tags/js/tags.constants.js';
import 'modules/tags/js/controllers/addTagset.ctrl.js';
import 'adminConsole/js/services/security.svc.js';
import 'ediscovery/common/js/directives/dataSourceSecurity.js';
import TagManagerUtility from 'modules/tags/js/services/tagManager.util.js';
import { Url } from 'ediscovery/common/js/constants';

class TagsetDetailsController {
	constructor(
		$state,
		$stateParams,
		$uibModal,
		cvLoc,
		cvBreadcrumbsTabsFactory,
		tagManagerService,
		securityService,
		dsSecurityFactory,
		settingsService,
		PERMISSIONS
	) {
		this.$state = $state;
		this.$stateParams = $stateParams;
		this.$uibModal = $uibModal;
		this.cvLoc = cvLoc;
		this.cvBreadcrumbsTabsFactory = cvBreadcrumbsTabsFactory;
		this.tagManagerService = tagManagerService;
		this.securityService = securityService;
		this.dsSecurityFactory = dsSecurityFactory;
		this.settingsService = settingsService;
		this.PERMISSIONS = PERMISSIONS;

		this.init();
	}

	async init() {
		this.addBreadcrumbs();
		await this.initPermissions();
		this.getTagsetDetails();
	}

	addBreadcrumbs() {
		let breadcrumbs;
		if (this.$state.is('activateTagsetDetails')) {
			breadcrumbs = [
				{
					title: this.cvLoc('label.ediscovery'),
					link: Url.Activate
				},
				{
					title: this.cvLoc('label.tagManager'),
					link: '#activate/tagManager'
				}
			];
		} else {
			breadcrumbs = [
				{
					title: this.cvLoc('label.tagManager'),
					link: '#tagManager'
				}
			];
		}
		this.cvBreadcrumbsTabsFactory.addBreadCrumbs(breadcrumbs);
	}

	async initPermissions() {
		await this.settingsService.getUserHasCapability(this.PERMISSIONS.TAG_MANAGEMENT).then(resp => {
			const { data } = resp;
			this.tagManagementPermission = !!data;
		});
	}

	getTagsetDetails() {
		this.tagManagerService
			.getTagset(this.$stateParams.tagsetId)
			.then(resp => {
				this.tagset = this.formatTagsetResponse(resp.data);
				this.allowEdit = TagManagerUtility.checkCapability(CapabilityType.Edit, this.tagset.capabilities);
				this.setupTagsGrid();
				this.setupSecurityComponent();
			})
			.catch(err => {
				console.error(err);
			});
	}

	formatTagsetResponse(data) {
		if (_.isUndefined(data) || !Array.isArray(data.containers)) {
			return;
		}
		const container = data.containers[0];
		const tagset = {
			containerId: container.containerId,
			name: container.containerName,
			description: container.comment,
			createTime: container.createTime,
			ownerAlias: _.get(container, 'ownerInfo.aliasName'),
			ownerUserName: _.get(container, 'ownerInfo.userName'),
			isSystem: (container.attribute & 2) === 2,
			capabilities: container.capabilities
		};
		tagset.type = tagset.isSystem ? this.cvLoc('label.tagset.system') : this.cvLoc('label.tagset.custom');
		return tagset;
	}

	setupSecurityComponent() {
		this.entityType = 'TAG_ENTITY';
		this.genericEntity = {
			tagId: this.$stateParams.tagsetId,
			_type_: DM2EntityType.Tags
		};
		this.securityService.getSecurityAssociation(this.entityType, this.$stateParams.tagsetId).then(resp => {
			const { data } = resp;
			this.securityAssociations = this.dsSecurityFactory.formatSecurityAssociationsObject(data.securityAssociations);
			this.associationsLoaded = true;
		});
	}

	editTagset() {
		const self = this;
		const modalInstance = this.$uibModal.open({
			templateUrl: `${appUtil.appRoot}modules/tags/partials/addTagset.jsp`,
			controller: 'addTagsetController',
			controllerAs: 'addTagsetCtrl',
			backdrop: 'static',
			windowClass: 'small-size',
			resolve: {
				isEdit() {
					return true;
				},
				tagset() {
					return self.tagset;
				}
			}
		});

		modalInstance.result.then(res => {
			if (!_.isUndefined(res.container)) {
				this.tagset.name = res.container.containerName;
				this.tagset.description = res.container.comment;
			}
		});
	}

	setupTagsGrid() {
		this.tagsGridOptions = {};
		this.tagsGridOptions.gridTitle = this.cvLoc('label.tags');
		this.tagsGridOptions.tableName = 'TagsTable';
		this.tagsGridOptions.pageSize = 10;
		this.tagsGridOptions.gridEmptyMessage = this.cvLoc('info.noTags');
		this.tagsGridOptions.columns = getTagsGridColumns(this.cvLoc);
		this.tagsGridOptions.url = this.getTagsForTagset.bind(this);
		this.tagsGridOptions.beforeGridInitialize = ({ grid }) => {
			this.tagsGrid = grid;
		};

		this.tagsGridOptions.gridToolbarMenu = [
			{
				id: 'ADD_TAG',
				label: this.cvLoc('label.tag.add'),
				onSelect: () => {
					this.onAddTag();
				},
				disableOnDeselect: false,
				hidden: !this.allowEdit || !this.tagManagementPermission
			}
		];

		this.tagsGridOptions.actionMenu = [
			{
				id: 'EDIT_TAG',
				label: this.cvLoc('label.edit'),
				contextMenuOnly: true,
				onSelect: ({ selectedRowValues }) => {
					if (selectedRowValues.length === 1) {
						this.onEditTag(selectedRowValues[0]);
					}
				},
				hidden: !this.tagManagementPermission
			}
		];
		this.tagsGridOptions.onActionMenuOpen = this.actionMenuOpen.bind(this);
	}

	actionMenuOpen(event) {
		['EDIT_TAG'].forEach(option => {
			if (!this.allowEdit || _.get(event, 'selectedRowValues[0].attribute.system')) {
				event.grid.disableActionMenuOption(option);
			} else {
				event.grid.enableActionMenuOption(option);
			}
		});
	}

	getTagsForTagset(options) {
		this.tagManagerService.getTagsForTagset(this.$stateParams.tagsetId).then(resp => {
			const { data } = resp;
			if (Array.isArray(_.get(data, 'listOftagSetList[0].tagSetsAndItems[0].tags'))) {
				options.success(data.listOftagSetList[0].tagSetsAndItems[0].tags);
			} else {
				options.success([]);
			}
		});
	}

	onAddTag() {
		const self = this;
		const modalInstance = this.$uibModal.open({
			templateUrl: `${appUtil.appRoot}modules/tags/partials/addTag.jsp`,
			controller: 'addTagController',
			controllerAs: 'addTagCtrl',
			resolve: {
				isEdit() {
					return false;
				},
				tagset() {
					return self.tagset;
				},
				tag() {
					return {};
				}
			},
			backdrop: 'static',
			windowClass: 'small-size'
		});

		modalInstance.result.then(res => {
			this.tagsGrid.refreshData();
		});
	}

	onEditTag(row) {
		const self = this;
		const modalInstance = this.$uibModal.open({
			templateUrl: `${appUtil.appRoot}modules/tags/partials/addTag.jsp`,
			controller: 'addTagController',
			controllerAs: 'addTagCtrl',
			resolve: {
				isEdit() {
					return true;
				},
				tagset() {
					return self.tagset;
				},
				tag() {
					return {
						tagId: row.tagId,
						name: row.name
					};
				}
			},
			backdrop: 'static',
			windowClass: 'small-size'
		});

		modalInstance.result.then(res => {
			this.tagsGrid.refreshData();
		});
	}
}

TagsetDetailsController.$inject = [
	'$state',
	'$stateParams',
	'$uibModal',
	'cvLoc',
	'cvBreadcrumbsTabsFactory',
	'tagManagerService',
	'securityService',
	'dsSecurityFactory',
	'settingsService',
	'PERMISSIONS'
];

ediscoveryAppModule.controller('tagsetDetailsController', TagsetDetailsController);
export default ediscoveryAppModule;
